#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>
#include "qconfirm.h"
#include "qconfirm_id.h"
#include "qconfirm_fmt_time.h"
#include "qconfirm_conf_get.h"
#include "sgetopt.h"
#include "strerr.h"
#include "stralloc.h"
#include "lock.h"
#include "env.h"
#include "direntry.h"
#include "buffer.h"
#include "fmt.h"
#include "str.h"
#include "byte.h"
#include "error.h"
#include "open.h"
#include "cdb.h"
#include "cdb_make.h"
#include "tai.h"
#include "seek.h"

#define FATAL "qconfirm: fatal: "
#define WARNING "qconfirm: warning: "

#define USAGE1 " [-d dir] [-iCD] [-c cdb] ch|ac|dr|bo|ba|pe|re|sn id|address ..."
#define USAGE2 " [-d dir] [-iC] [-c cdb] list [pending|ok|return|bad]"

#define VERSION "$Id: qconfirm.c,v 1.16 2004/01/18 14:07:34 pape Exp $"

#define NOOP 0
#define BOUNCE 1
#define DROP 2
#define PEND 3
#define ACCEPT 4
#define BAD 5
#define SNEAK 6

const char *progname;

stralloc qconfirm_dir ={0};
unsigned int forceid =0;
unsigned int wildhost =1;
stralloc id ={0};
stralloc sa ={0};
stralloc cdbfn ={0};
int fdcdb =-1;
int origdir =-1;
struct stat st;
struct cdb cdb;
buffer bcdb;
char bcdbspace[4096];
char uintbuf[4];
uint32 pos =0;
uint32 eod;
uint32 klen;
uint32 dlen;

void usage() {
  strerr_warn3("usage: ", progname, USAGE1, 0);
  strerr_die4x(1, "usage: ", progname, USAGE2, "\n");
}
void die_nomem() { strerr_die2x(111, FATAL, "out of memory."); }
void fatal(char *m0, char *m1) { strerr_die4sys(111, FATAL, m0, m1, ": "); };
void fatalx(char *m0, char *m1) { strerr_die3x(111, FATAL, m0, m1); };
void fatal_qdir(char *m0, char *m1) {
  strerr_die6sys(111, FATAL, m0, qconfirm_dir.s, "/", m1, ": ");
}
void warn(char *m0, char *m1) {
  strerr_warn4(WARNING, m0, m1, ": ", &strerr_sys);
}
void warnx(char *m0, char *m1) { strerr_warn3(WARNING, m0, m1, 0); };
void warn_qdir(char *m0, char*m1) {
  strerr_warn6(WARNING, m0, qconfirm_dir.s, "/", m1, ": ", &strerr_sys);
}
void out(char *m) { buffer_puts(buffer_1, m); };
void flush(char *m) { buffer_putsflush(buffer_1, m); };

void pget(buffer *b, char *buf, unsigned int l) {
  int r;
  while (l > 0) {
    r =buffer_get(b, buf, l);
    if (r == -1) fatal("unable to read: ", cdbfn.s);
    if (r == 0)
      strerr_die4x(111, FATAL, "unable to read: ", cdbfn.s,
		   ": truncated file");
    buf +=r; l -=r;
  }
}
void list(char *arg) {
  DIR *dir;
  direntry *d;
  char *state;
  char buf[FMT_ULONG];
  char c;
  struct tai ts;
  char timebuf[QCONFIRM_FMT_TIME];

  if (arg) c =*arg;
  else c ='p';
  switch (c) {
  case 'o': state ="ok"; break;
  case 'r': state ="return"; break;
  case 'b': state ="bad"; break;
  default: state ="pending"; break;
  }

  if (chdir(state) == -1)
    fatal_qdir("unable to change directory: ", state);
  if (! (dir =opendir(".")))
    fatal_qdir("unable to read directory: ", state);

  errno =0;
  while ((d =readdir(dir))) {
    if (d->d_name[0] == '.') continue;
    if (stat(d->d_name, &st) == -1) {
      warn_qdir("unable to stat: ", d->d_name);
      errno =0;
      continue;
    }
    timebuf[qconfirm_fmt_time(timebuf, st.st_mtime)] =0;
    out(timebuf); out(" ");
    if (! forceid) {
      if (id2address(&sa, d->d_name) == -1) die_nomem();
      buffer_put(buffer_1, sa.s, sa.len);
    }      
    else
      out(d->d_name);
    if (! arg)
      switch (st.st_mode & S_IRWXU) {
      case S_IWUSR: out(" bounce"); break;
      case 0: out(" drop"); break;
      case S_IRWXU: out(" confirm"); break;
      case S_IRUSR: out(" sneak"); break;
      }
    else
      if (*arg == 'r')
	if (st.st_size > 0) {
	  out(" ");
	  buffer_put(buffer_1, buf, fmt_ulong(buf, st.st_size));
	}
    flush("\n");
  }
  if (errno) {
    closedir(dir);
    fatal_qdir("unable to read directory: ", state);
  }
  closedir(dir);

  if ((c == 'o') && cdbfn.s) {
    /* open cdb */
    if ((fdcdb =open_read(cdbfn.s)) == -1) fatal("unable to open ", cdbfn.s);
    buffer_init(&bcdb, buffer_unixread, fdcdb, bcdbspace, sizeof bcdbspace);

    /* size of database */
    pget(&bcdb, uintbuf, 4);
    pos +=4;
    uint32_unpack(uintbuf, &eod);

    while (pos < 2048) {
      pget(&bcdb, uintbuf, 4);
      pos +=4;
    }
    while (pos < eod) {
      pget(&bcdb, uintbuf, 4);
      pos +=4;
      uint32_unpack(uintbuf, &klen);
      pget(&bcdb, uintbuf, 4);
      pos +=4;
      uint32_unpack(uintbuf, &dlen);
      if (! stralloc_ready(&id, klen +dlen)) die_nomem();
      pget(&bcdb, id.s, klen +dlen);
      pos +=klen +dlen;
      if (dlen != 8)
	strerr_die4x(111, FATAL, "unable to read: ", cdbfn.s, ": format error");
      tai_unpack(id.s +klen, &ts);
      timebuf[qconfirm_fmt_time(timebuf, ts.x)] =0;
      out(timebuf); out(" ");
      id.s[klen] =0;
      if (! forceid) {
	if (id2address(&sa, id.s) == -1) die_nomem();
	buffer_put(buffer_1, sa.s, sa.len);
      }      
      else
	out(id.s);
      flush("\n");
    }
    close(fdcdb);
  }
}

int state(char *arg, int what) {
  mode_t m =S_IRUSR | S_IWUSR;

  if (! arg) return(-1);

  switch (what) {
  case BOUNCE: m =S_IWUSR; break;
  case DROP: m =0; break;
  case PEND: m =S_IRUSR | S_IWUSR; break;
  case SNEAK: m =S_IRUSR; break;
  }

  if (forceid) {
    if (! stralloc_copys(&id, arg)) die_nomem();
    if (! stralloc_0(&id)) die_nomem();
  }
  else
    if (address2id(&id, arg) == -1) die_nomem();
  if (! stralloc_copys(&sa, "pending/")) die_nomem();
  if (! stralloc_cat(&sa, &id)) die_nomem();
  if (chmod(sa.s, m) == -1) {
    if (errno == error_noent) {
      out(arg); flush(": not pending.\n");
      return(0);
    }
    else
      warn_qdir("unable to chmod: ", sa.s);
    return(-1);
  }
  out(arg); out(": ");
  switch (what) {
  case BOUNCE: flush("bounce.\n"); break;
  case DROP: flush("drop.\n"); break;
  case PEND: flush("pending.\n"); break;
  case SNEAK: flush("sneak.\n"); break;
  }
  return(1);
}
int check(char *arg, int what) {
  stralloc fn ={0};
  stralloc fn_bad ={0};
  stralloc host ={0};
  struct stat st;
  int i;

  if (! arg) return(-1);
  if (forceid) {
    if (! stralloc_copys(&id, arg)) die_nomem();
    if (! stralloc_0(&id)) die_nomem();
  }
  else
    if (address2id(&id, arg) == -1) die_nomem();

  /* check ok */
  if (! stralloc_copys(&fn, "ok/")) die_nomem();
  if (! stralloc_cat(&fn, &id)) die_nomem();
  if (stat(fn.s, &st) != -1) {
    out(arg); out(": known: "); out(qconfirm_dir.s); out("/"); out(fn.s);
    flush("\n");
    return(1);
  }
  if (errno != error_noent) {
    warn_qdir("unable to stat: ", fn.s);
    return(-1);
  }
  /* check -default */
  for (i =id.len; i >= 0; --i)
    if (!i || (id.s[i -1] == '-')) {
      if (! stralloc_copys(&sa, "ok/")) die_nomem();
      if (! stralloc_catb(&sa, id.s, i)) die_nomem();
      if (! stralloc_cats(&sa, "default")) die_nomem();
      if (! stralloc_0(&sa)) die_nomem();
      if (stat(sa.s, &st) != -1) {
	out(arg); out(": known: "); out(qconfirm_dir.s); out("/"); out(sa.s);
	flush("\n");
	return(1);
      }
      if (errno != error_noent) {
	warn_qdir("unable to stat: ", sa.s);
	return(-1);
      }
    }
  /* check host: a:b:c:d, :b:c:d, :c:d, :d. */
  if (wildhost && ((i =byte_chr(id.s, id.len, '=')) != id.len)) {
    if (! stralloc_copyb(&host, id.s, i)) die_nomem();
    for (i =0; i < host.len; ++i) {
      if ((host.s[i] == ':') || ! i) {
	if (! stralloc_copys(&sa, "ok/")) die_nomem();
	if (! stralloc_catb(&sa, host.s +i, host.len -i)) die_nomem();
	if (! stralloc_0(&sa)) die_nomem();
	if (stat(sa.s, &st) != -1) {
	  out(arg); out(": known: "); out(qconfirm_dir.s); out("/"); out(sa.s);
	  flush("\n");
	  return(1);
	}
	if (errno != error_noent) {
	  warn_qdir("unable to stat: ", sa.s);
	  return(-1);
	}
      }
    }
  }
  if (fdcdb != -1) {
    /* check cdb */
    if (cdb_find(&cdb, id.s, str_len(id.s))) {
      out(arg); out(": known: "); out(cdbfn.s); out(": "); out(id.s);
      flush("\n");
      return(1);
    }
    for (i =id.len; i >= 0; --i)
      if (!i || (id.s[i -1] == '-')) {
	if (! stralloc_copyb(&sa, id.s, i)) die_nomem();
	if (! stralloc_cats(&sa, "default")) die_nomem();
	if (cdb_find(&cdb, sa.s, sa.len)) {
	  out(arg); out(": known: "); out(cdbfn.s); out(": ");
	  buffer_put(buffer_1, sa.s, sa.len);
	  flush("\n");
	  return(1);
	}
      }
    if (wildhost && ((i =byte_chr(id.s, id.len, '=')) != id.len)) {
      if (! stralloc_copyb(&host, id.s, i)) die_nomem();
      for (i =0; i < host.len; ++i)
	if ((host.s[i] == ':') || ! i) {
	  if (cdb_find(&cdb, host.s +i, host.len -i)) {
	    out(arg); out(": known: "); out(cdbfn.s); out(": ");
	    buffer_put(buffer_1, host.s +i, host.len -i);
	    flush("\n");
	    return(1);
	  }
	}
    }
  }
  /* check pending */
  if (! stralloc_copys(&sa, "pending/")) die_nomem();
  if (! stralloc_cat(&sa, &id)) die_nomem();
  if (stat(sa.s, &st) != -1) {
    switch (what) {
    case ACCEPT:
      if (chmod(sa.s, S_IRUSR | S_IWUSR | S_IXUSR) == -1) {
	warn_qdir("unable to chmod: ", sa.s);
	return(-1);
      }
      out(arg); flush(": confirm.\n");
      return(1);
      break;
    case BAD:
      if (chmod(sa.s, S_IWUSR) == -1) {
	warn_qdir("unable to chmod: ", sa.s);
	return(-1);
      }
      out(arg); flush(": bounce.\n");
      break;
    default:
      out(arg); out(": pending: "); out(qconfirm_dir.s); out("/"); out(sa.s);
      flush("\n");
      return(1);
      break;
    }
  }
  if (errno != error_noent) {
    warn_qdir("unable to stat: ", sa.s);
    return(-1);
  }

  /* check bad */
  if (! stralloc_copys(&fn_bad, "bad/")) die_nomem();
  if (! stralloc_cat(&fn_bad, &id)) die_nomem();
  if (stat(fn_bad.s, &st) != -1) {
    switch (what) {
    case ACCEPT:
      if (unlink(fn_bad.s) == -1) warn_qdir("unable to unlink: ", fn_bad.s);
      break;
    case NOOP: case BAD:
      out(arg); out(": bad: "); out(qconfirm_dir.s); out("/"); out(sa.s);
      flush("\n");
      return(1);
      break;
    }
  }
  if (errno != error_noent) {
    warn_qdir("unable to stat: ", fn_bad.s);
    return(-1);
  }
  switch (what) {
  case ACCEPT:
    if ((i =open_trunc(fn.s)) == -1) {
      warn_qdir("unable to create: ", fn.s);
      return(-1);
    }
    close(i);
    break;
  case BAD:
    if ((i =open_trunc(fn_bad.s)) == -1) {
      warn_qdir("unable to create: ", fn_bad.s);
      return(-1);
    }
    close(i);
    break;
  }
  return(0);
}

int remove_ids(char **arg) {
  stralloc todo ={0};
  stralloc tmp ={0};
  int fdtmp;
  struct cdb_make c;
  char *s;
  mode_t mode =S_IRUSR | S_IWUSR;;

  if (! arg || ! *arg) return(-1);

  if (cdbfn.s) {
    if (stat(cdbfn.s, &st) == -1) fatal("unable to stat: ", cdbfn.s);
    mode =st.st_mode;

    /* open cdb */
    if ((fdcdb =open_read(cdbfn.s)) == -1) fatal("unable to open ", cdbfn.s);
    cdb_init(&cdb, fdcdb);
  }
  if (! stralloc_copys(&todo, "")) die_nomem();

  for (; arg && *arg; arg++) {
    if (forceid) {
      if (! stralloc_copys(&id, *arg)) die_nomem();
      if (! stralloc_0(&id)) die_nomem();
    }
    else
      if (address2id(&id, *arg) == -1) die_nomem();
    if (! stralloc_copys(&sa, "ok/")) die_nomem();
    if (! stralloc_cat(&sa, &id)) die_nomem();
    if (unlink(sa.s) == -1) {
      if (errno != error_noent)
	warn_qdir("unable to unlink: ", sa.s);
      else {
	if (fdcdb != -1)
	  if (cdb_find(&cdb, id.s, str_len(id.s))) {
	    if (! stralloc_catb(&todo, id.s, id.len)) die_nomem();
	    if (! stralloc_cats(&todo, *arg)) die_nomem();
	    if (! stralloc_0(&todo)) die_nomem();
	    continue;
	  }
	out(*arg); out(": "); flush("unknown.\n");
      }
      continue;
    }
    else {
      out(*arg); out(": "); flush("remove.\n");
    }
  }
  /* remove todo from cdb */
  if ((fdcdb != -1) && todo.s) {
    /* lock data.cdb */
    if (lock_exnb(fdcdb) == -1) warn("lock is busy, waiting...", cdbfn.s);
    if (lock_ex(fdcdb) == -1) fatal("unable to lock", cdbfn.s);
    if (seek_begin(fdcdb) == -1) fatal("unable to seek: ", cdbfn.s);

    buffer_init(&bcdb, buffer_unixread, fdcdb, bcdbspace, sizeof bcdbspace);

    /* open data.tmp */
    if (! stralloc_copys(&tmp, cdbfn.s)) die_nomem();
    if (! stralloc_cats(&tmp, ".tmp")) die_nomem();
    if (! stralloc_0(&tmp)) die_nomem();
    if ((fdtmp =open_trunc(tmp.s)) == -1) fatal("unable to create", tmp.s);
    if (cdb_make_start(&c, fdtmp) == -1) fatal("unable to create", tmp.s);
    
    /* size of database */
    pget(&bcdb, uintbuf, 4);
    pos +=4;
    uint32_unpack(uintbuf, &eod);

    while (pos < 2048) {
      pget(&bcdb, uintbuf, 4);
      pos +=4;
    }
    while (pos < eod) {
      int found =0;

      pget(&bcdb, uintbuf, 4);
      pos +=4;
      uint32_unpack(uintbuf, &klen);
      pget(&bcdb, uintbuf, 4);
      pos +=4;
      uint32_unpack(uintbuf, &dlen);
      
      if (! stralloc_ready(&sa, klen +dlen)) die_nomem();
      pget(&bcdb, sa.s, klen +dlen);
      pos +=klen +dlen;
      for (s =todo.s; s < todo.s +todo.len;) {
	if (byte_equal(sa.s, klen, s)) {
	  found =1;
	  break;
	}
	while (*s) ++s; ++s;
	while (*s) ++s; ++s;
      }
      if (found) {
	while (*s) ++s;	++s;
	out(s); out(": "); flush("remove (cdb).\n");
      }
      else
	if (cdb_make_add(&c, sa.s, klen, sa.s +klen, dlen) == -1)
	  fatal("unable to write data.tmp", 0);
    }
    if (cdb_make_finish(&c) == -1) fatal("unable to write cdb", tmp.s);
    if (fsync(fdtmp) == -1) fatal("unable to write cdb", tmp.s);
    close(fdtmp);
    if (rename(tmp.s, cdbfn.s) == -1) fatal("unable to replace", cdbfn.s);
    if (chmod(cdbfn.s, mode) == -1) fatal("unable to chmod", cdbfn.s);
  }
  return(1);
}

void ctrl(char **arg) {
  const char *c;

  if (! arg || ! *arg) usage();
  c =*arg++;
  if (! arg || ! *arg) usage();
  switch (*c) {
  case 'b': /* bounce or bad */
    if (*(c +1) && (*(c +1) == 'a')) { /* bad */
      for (; arg && *arg; arg++)
	if (check(*arg, BAD) == 0) {
	  out(*arg); flush(": bad.\n");
	}
    }
    else /* bounce */
      for (; arg && *arg; arg++)
	state(*arg, BOUNCE);
    break;
  case 'd': /* drop */
    for (; arg && *arg; arg++)
      state(*arg, DROP);
    break;
  case 'p':
    for (; arg && *arg; arg++)
      state(*arg, PEND);
    break;
  case 'c': /* check */
    if (cdbfn.s) {
      /* open cdb */
      if ((fdcdb =open_read(cdbfn.s)) == -1) fatal("unable to open ", cdbfn.s);
      cdb_init(&cdb, fdcdb);
    }
    for (; arg && *arg; arg++)
      if (check(*arg, NOOP) == 0) {
	out(*arg); flush(": unknown.\n");
      }
    close(fdcdb);
    break;
  case 'a': /* accept */
    if (cdbfn.s) {
      /* open cdb */
      if ((fdcdb =open_read(cdbfn.s)) == -1) fatal("unable to open ", cdbfn.s);
      cdb_init(&cdb, fdcdb);
    }
    for (; arg && *arg; arg++)
      if (check(*arg, ACCEPT) == 0) {
	out(*arg); flush(": ok.\n");
      }
    close(fdcdb);
    break;
  case 'r': /* remove */
    remove_ids(arg);
    break;
  case 's': /* show or sneak */
    if (*(c +1) && (*(c +1) == 'n')) { /* sneak */
      for (; arg && *arg; arg++)
	state(*arg, SNEAK);
    }
    else /* show id */
      for (; arg && *arg; arg++) {
	if (address2id(&id, *arg) == -1) die_nomem();
	out(*arg); out(": "); out(id.s); flush("\n");
      }
    break;
  default:
    usage();
  }
}

int main(int argc, char **argv) {
  int opt;
  char *s;
  const char *cs =0;

  progname =*argv;

  if ((s =env_get("QCONFIRM_DIR")))
    if (! stralloc_copys(&qconfirm_dir, s)) die_nomem();
  if (! qconfirm_dir.s) {
    if (! (s =env_get("HOME")))
      strerr_die2x(100, FATAL, "environment variable HOME not set.");
    if (! stralloc_copys(&qconfirm_dir, s)) die_nomem();
    if (! stralloc_cats(&qconfirm_dir, "/" QCONFIRMDIR)) die_nomem();
  }
  if (! stralloc_0(&qconfirm_dir)) die_nomem();
  if (! (cs =conf_get_dflt(qconfirm_dir.s, "QCONFIRM_OKCDB", "")))
    fatal("unable to read config: ", "QCONFIRM_OKCDB");

  while ((opt =getopt(argc, (const char **)argv, "d:iDCc:V")) != opteof) {
    switch(opt) {
    case 'd':
      if (! stralloc_copys(&qconfirm_dir, optarg)) die_nomem();
      if (! stralloc_0(&qconfirm_dir)) die_nomem();
      break;
    case 'i':
      forceid =1;
      break;
    case 'D':
      wildhost =0;
      break;
    case 'C':
      if (! *cs) cs ="ok.cdb";
      break;
    case 'c':
      cs =optarg;
      break;
    case 'V':
      strerr_warn1(VERSION, 0);
    case '?':
      usage();
    }
  }
  argv +=optind;
  umask(077);

  if (! argv || ! *argv) usage();
  if (*cs) {
    if (cs[0] != '/') {
      if (! stralloc_copys(&cdbfn, qconfirm_dir.s)) die_nomem();
      if (! stralloc_catb(&cdbfn, "/", 1)) die_nomem();
      if (! stralloc_cats(&cdbfn, cs)) die_nomem();
    }
    else
      if (! stralloc_copys(&cdbfn, cs)) die_nomem();
    if (! stralloc_0(&cdbfn)) die_nomem();
  }

  if ((origdir =open_read(".")) == -1)
    fatal("unable to open", "current directory");
  if (chdir(qconfirm_dir.s) == -1)
    fatal("unable to change directory: ", qconfirm_dir.s);

  if (**argv == 'l')
    list(*++argv);
  else
    ctrl(argv);

  if (fchdir(origdir) == -1)
    fatal("unable to change to original directory", 0);
  _exit(0);
}
