#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "sgetopt.h"
#include "strerr.h"
#include "error.h"
#include "stralloc.h"
#include "buffer.h"
#include "sig.h"
#include "env.h"
#include "open.h"
#include "byte.h"
#include "scan.h"
#include "str.h"
#include "openreadclose.h"
#include "qconfirm.h"
#include "qconfirm_id.h"
#include "next_paragraph.h"
#include "getline.h"

#define USAGE " [-n limit ] [dir]"
#define VERSION "$Id: qconfirm-return.c,v 1.11 2003/11/21 13:45:53 pape Exp $"
#define FATAL "qconfirm-return: fatal: "
#define WARNING "qconfirm-return: warning: "
#define INFO "qconfirm-return: info: "

const char *progname;
char *sender;
const char *qconfirm_dir;
stralloc sa ={0};
stralloc pending ={0};
stralloc id ={0};
stralloc k ={0};
struct stat s;
unsigned long limit =0;

void usage() { strerr_die4x(111, "usage: ", progname, USAGE, "\n"); }
void die_nomem() { strerr_die2x(111, FATAL, "out of memory."); }
void die_noqsbmf() { strerr_die2x(0, WARNING, "no QSBMF."); }
void die_badheader() {
  strerr_die2x(0, WARNING, "bad header in original message.");
}
void info(char *m0, char *m1) { strerr_die3x(99, INFO, m0, m1); }
void warn(char *m0, char *m1) { strerr_die3x(0, WARNING, m0, m1); }
void fatal(char *m0, char *m1) { strerr_die4sys(111, FATAL, m0, m1, ": "); }

int main(int argc, const char **argv) {
  int r;
  int fd;
  int opt;

  progname =*argv;

  while ((opt =getopt(argc, argv, "Vn:")) != opteof) {
    switch(opt) {
    case 'n': scan_ulong(optarg, &limit); break;
    case 'V': strerr_warn1(VERSION, 0);
    case '?': usage();
    }
  }
  argv +=optind;

  umask(0177);
  sig_ignore(sig_pipe);

  qconfirm_dir =*argv;
  if (! qconfirm_dir) qconfirm_dir =env_get("QCONFIRM_DIR");
  if (! qconfirm_dir) qconfirm_dir =QCONFIRMDIR;

  /* check for empty sender */
  sender =env_get("SENDER");
  if (sender && *sender) warn("sender not empty: ", sender);
  
  /* skip header */
  if (next_paragraph(buffer_0) == -1) strerr_die2x(0, WARNING, "no body.");

  /* 'Hi. This is the' */
  if ((r =getline(buffer_0, &sa)) == -1)
    strerr_die2x(0, WARNING, "empty body.");
  if ((sa.len < 15) || ! str_start(sa.s, "Hi. This is the")) die_noqsbmf();
  if (next_paragraph(buffer_0) == -1) die_noqsbmf();

  /* failure paragraph */
  if (getline(buffer_0, &sa) == -1) die_noqsbmf();
  /* create id */
  if ((sa.len < 4) || (sa.s[0] != '<') ||
      (sa.s[sa.len -2] != ':') || (sa.s[sa.len -3] != '>')) die_noqsbmf();
  sa.s[sa.len -3] =0;
  if (address2id(&id, sa.s +1) == -1) die_nomem();

  /* check pending/id */
  if (! stralloc_copys(&pending, qconfirm_dir)) die_nomem();
  if (! stralloc_cats(&pending, "/pending/")) die_nomem();
  if (! stralloc_cat(&pending, &id)) die_nomem();
  if (stat(pending.s, &s) == -1) {
    if (errno != error_noent) fatal("unable to stat: ", pending.s);
    warn("not pending: ", id.s);
  }
  if ((s.st_mode & S_IRWXU) == 0) info("already drop: ", pending.s);
  if (next_paragraph(buffer_0) == -1) die_noqsbmf();

  /* break paragraph */
  if (getline(buffer_0, &sa) == -1) die_noqsbmf();
  if (! sa.len || (sa.s[0] != '-')) die_noqsbmf();
  if (next_paragraph(buffer_0) == -1) die_noqsbmf();

  /* original message */
  while ((r =getline(buffer_0, &sa)) > 0) {
    int at, dash;

    if ((sa.len > 6) && str_start(sa.s, "From: ")) {
      /* From: header */
      if (sa.s[sa.len -2] != '>') die_badheader();
      if ((at =byte_rchr(sa.s, sa.len, '@')) == sa.len) die_badheader();
      if ((dash =byte_rchr(sa.s, at, '-')) == at) die_badheader();
      if (at -dash != 33) die_badheader();

      /* read key */
      if (openreadclose(pending.s, &k, 74) == -1)
	fatal("unable to read: ", pending.s);
      if ((k.len != 32) && (k.len < 34)) /* backward compatibility 0.8.0 */
	strerr_die4x(111, FATAL,
		     "unable to read: ", pending.s, ": bad format");
      if (byte_diff(sa.s +dash +1, 32, k.s) != 0)
        warn("keys do not match: ", id.s);
      break;
    }
  }
  if (r <= 0) die_badheader();

  /* check return/id */
  if (! stralloc_copys(&sa, qconfirm_dir)) die_nomem();
  if (! stralloc_cats(&sa, "/return/")) die_nomem();
  if (! stralloc_cat(&sa, &id)) die_nomem();
  if (stat(sa.s, &s) == -1) {
    if (errno != error_noent) fatal("unable to stat: ", sa.s);
    if ((fd =open_trunc(sa.s)) == -1) fatal("unable to create: ", sa.s);
    close(fd);
    info("create: ", sa.s);
  }
  if (s.st_size >= limit) {
    /* drop */
    if (chmod(pending.s, 0) == -1) fatal("unable to chmod: ", pending.s);
    info("drop: ", pending.s);
  }
  if ((fd =open_append(sa.s)) == -1) fatal("unable to open: ", sa.s);
  if (write(fd, "-", 1) != 1) fatal("unable to write: ", sa.s);
  close(fd);
  info("touch: ", sa.s);
  _exit(0);
}
