#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "qconfirm.h"
#include "qconfirm_key.h"
#include "strerr.h"
#include "open.h"
#include "buffer.h"
#include "error.h"
#include "stralloc.h"
#include "sgetopt.h"

#define USAGE1 " dir"
#define USAGE2 " -i dir dot local host [mailbox]"
#define VERSION "$Id: qconfirm-conf.c,v 1.7 2004/01/18 14:07:34 pape Exp $"
#define FATAL "qconfirm-conf: fatal: "
#define WARNING "qconfirm-conf: warning: "
#define INFO "qconfirm-conf: info: "

#define MSG_REQUEST "\
Subject: Please confirm your message\n\
Return-Path: <>\n\
\n\
Hi. This is the qconfirm mail-handling program.  One or more messages\n\
from you are being held because your address was not recognized.\n\
\n\
To release your pending message(s) for delivery, please reply to this\n\
request.  Your reply will not be read, so an empty message is fine.\n\
\n\
If you do not reply to this request, your message(s) will eventually be\n\
returned to you, and will never be delivered to the envelope recipient.\n\
\n\
This confirmation verifies that your message(s) are legitimate and not\n\
junk-mail.\n\
\n\
Regards, the qconfirm program, http://smarden.org/qconfirm/.\n\
\n\
--- Below this line is the top of a message from you.\n\
"
#define MSG_ACK "\
Subject: Confirmation accepted\n\
Return-Path: <>\n\
\n\
Hi. This is the qconfirm mail-handling program.  Your confirmation was\n\
accepted, and so your pending message or messages have been delivered.\n\
\n\
Thanks for your cooperation.\n\
\n\
Regards, the qconfirm program, http://smarden.org/qconfirm/.\n\
\n\
--- Below this line is the top of the confirmation message\n\
"
#define MSG_CONFIRM "\
Subject: Confirming delivery\n\
\n\
Hi. This is the qconfirm mail-handling program.  I've received a delivery\n\
notification that I identified as a delivery confirmation request.\n\
\n\
This message is sent to confirm delivery of the corresponding message.\n\
\n\
--- Below this line is the top of the confirmation request message\n\
"

const char *progname;

void usage() {
  strerr_warn3("usage: ", progname, USAGE1, 0);
  strerr_die4x(1, "usage: ", progname, USAGE2, "\n");
}

const char *dir;
const char *dot;
const char *local;
const char *host;
const char *mailbox;
const char *fn;
unsigned int verbose =0;
unsigned int force =0;
unsigned int inject =0;
char *key;
char buf[1024];
int fd;
buffer b;
stralloc sa ={0};

void die_nomem() { strerr_die2x(111, FATAL, "out of memory."); }
void fail() {
  if (fn[0] == '/') 
    strerr_die4sys(111, FATAL, "unable to create ", fn, ": ");
  else
    strerr_die6sys(111, FATAL, "unable to create ", dir, "/", fn, ": ");
}
unsigned int exists(const char *s) {
  if ((fd =open_read(s)) != -1) {
    close(fd); return(1);
  }
  if (errno != error_noent)
    strerr_die4sys(111, FATAL, "unable to open: ", s, ": ");
  return(0);
}
void start(const char *s) {
  if (exists(s))
    if (! force)
      strerr_die4x(111, FATAL, "unable to create: ", s, ": file exists");
  fn = s;
  fd = open_trunc(fn);
  if (fd == -1) fail();
  buffer_init(&b, write, fd, buf, sizeof buf);
}
void outs(const char *s) { if (buffer_puts(&b, s) == -1) fail(); };
void finish(void) {
  if (buffer_flush(&b) == -1) fail();
  if (fsync(fd) == -1) fail();
  close(fd);
}
void perm(int mode) { if (chmod(fn, mode) == -1) fail(); };
void makedir(const char *s) {
  fn =s;
  if (mkdir(fn, 0750) != -1) return;
  if (force && (errno == error_exist)) return;
  fail();
}

int main_inject(const char **argv) {
  if (! (dot =*++argv)) usage();
  if (! (local =*++argv)) usage();
  if (! (host =*++argv)) usage();
  if (! (mailbox =*++argv)) mailbox ="./Maildir/";
  if (*dot != '/') strerr_die2x(111, FATAL, "dot must start with slash.");
  if (chdir(dir) == -1)
    strerr_die4sys(111, FATAL, "unable to switch to ", dir, ": ");

  if (! stralloc_copys(&sa, dir)) die_nomem();
  if (! stralloc_cats(&sa, dot)) die_nomem();

  start("msg/confirm");
  outs(MSG_CONFIRM);
  finish();
  perm(0644);

  makedir("notice");

  if (! (key =qconfirm_key14(&sa))) die_nomem();
  start("conf/QCONFIRM_MID");
  if (buffer_put(&b, key, 14) == -1) fail();
  outs(".");
  outs(host);
  outs("\n");
  finish();
  perm(0644);

  start("conf/QNOTICE_USER");
  outs(local);
  outs("\n");
  finish();
  perm(0644);

  start("conf/QNOTICE_HOST");
  outs(host);
  outs("\n");
  finish();
  perm(0644);

  if (! (key =qconfirm_key7(&sa))) die_nomem();
  start(QNOTICEKEY);
  if (buffer_put(&b, key, 7) == -1) fail();
  finish();
  perm(0600);

  if (! stralloc_copys(&sa, dot)) die_nomem();
  if (! stralloc_cats(&sa, "-" QNOTICEEXT)) die_nomem();
  if (! stralloc_catb(&sa, key, 7)) die_nomem();
  if (! stralloc_cats(&sa, "-default")) die_nomem();
  if (! stralloc_0(&sa)) die_nomem();
  start(sa.s);
  outs("|qconfirm-notice -d");
  outs(dir);
  outs("\n");
  outs(mailbox);
  outs("\n");
  finish();
  perm(0644);
  /*  strerr_warn3(INFO, "create: ", sa.s, 0); */

  buffer_puts(buffer_1, "\nDirectory ");
  buffer_puts(buffer_1, dir);
  buffer_puts(buffer_1, " modified.\nFile ");
  buffer_puts(buffer_1, sa.s);
  buffer_putsflush(buffer_1, " created.\n\n\
Now use qconfirm-inject instead of qmail-inject to inject mail.\n\n\
");
  return(0);
}

int main(int argc, const char **argv) {
  int opt;

  progname =*argv;
  umask(022);

  while ((opt =getopt(argc, argv, "Vvif")) != opteof) {
    switch(opt) {
    case 'v': verbose =1; break;
    case 'i': inject =1; break;
    case 'f': force =1; break;
    case 'V': strerr_warn1(VERSION , 0);
    case '?': usage();
    }
  }
  argv +=optind;

  if (! *argv) usage();
  dir =*argv;
  if (dir[0] != '/') strerr_die2x(111, FATAL, "dir must start with slash.");

  if (inject) return(main_inject(argv));

  if (chdir(dir) == -1) {
    if (errno != error_noent)
      strerr_die4sys(111, FATAL, "unable to switch to ", dir, ": ");
    if (mkdir(dir, 0700) == -1)
      strerr_die4sys(111, FATAL, "unable to create ", dir, ": ");
    if (chmod(dir, 0750) == -1)
      strerr_die4sys(111, FATAL, "unable to set mode of ", dir, ": ");
    if (chdir(dir) == -1)
      strerr_die4sys(111, FATAL, "unable to switch to ", dir, ": ");
  }
  makedir("ok");
  perm(0750);
  makedir("pending");
  perm(0750);
  makedir("conf");
  perm(0750);
  start("conf/QCONFIRM_DIR");
  outs(dir);
  outs("\n");
  finish();
  perm(0640);
  makedir("msg");
  start("msg/request");
  outs(MSG_REQUEST);
  finish();
  perm(0640);
  start("msg/acknowledge");
  outs(MSG_ACK);
  finish();
  perm(0640);

  buffer_puts(buffer_1, "\nDirectory ");
  buffer_puts(buffer_1, dir);
  buffer_puts(buffer_1, " created.\n\n");
  buffer_puts(buffer_1, "\
Add the following line to the top of your .qmail file and make sure to\n\
also have a line specifying delivery to your normal mailbox.\n\n\
");
  buffer_puts(buffer_1, "|qconfirm-check -d");
  buffer_puts(buffer_1, dir);
  buffer_putsflush(buffer_1, "\n\n");
  return(0);
}
