#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include "qconfirm.h"
#include "qconfirm_id.h"
#include "qconfirm_conf_get.h"
#include "sgetopt.h"
#include "strerr.h"
#include "error.h"
#include "stralloc.h"
#include "buffer.h"
#include "env.h"
#include "open.h"
#include "lock.h"
#include "byte.h"
#include "str.h"
#include "tai.h"
#include "cdb.h"
#include "scan.h"
#include "seek.h"

#define USAGE " [-v] [-D] [-c cdb] [-t sec]"
#define VERSION "$Id: qconfirm-cdb-check.c,v 1.7 2003/11/21 13:45:53 pape Exp $"
#define FATAL "qconfirm-cdb-check: fatal: "
#define WARNING "qconfirm-cdb-check: warning: "
#define INFO "qconfirm-cdb-check: info: "

const char *progname;

void usage() { strerr_die4x(111, "usage: ", progname, USAGE, "\n"); }
void die_nomem() { strerr_die2x(111, FATAL, "out of memory."); }

void info(char *m0, char *m1) {
  buffer_puts(buffer_1, INFO);
  buffer_puts(buffer_1, m0);
  buffer_puts(buffer_1, m1);
  buffer_putsflush(buffer_1, "\n");
}
void warn(char *m0, char *m1) {
  strerr_warn4(WARNING, m0, m1, ": ", &strerr_sys);
}
void fatal(char *m0, char *m1) { strerr_die4sys(111, FATAL, m0, m1, ": "); }

char *sender;
char *qconfirm_dir;
char *cdbfn =0;
unsigned long timeout =2419200;
unsigned int wildhost =1;
unsigned int verbose =0;
stralloc tmp ={0};
stralloc cdb ={0};
stralloc id ={0};
stralloc sa ={0};
static struct cdb c;
int fdcdb;
int fdtmp;

int main(int argc, const char **argv) {
  int opt;
  int i;
  struct stat st;
  mode_t mode;
  char tspack[8];
  struct tai now;
  struct tai ts;
  struct tai max;
  unsigned int dlen;
  uint32 dpos;
  buffer bcdb;
  char bcdbspace[4096];
  buffer btmp;
  char btmpspace[4096];

  progname =*argv;

  qconfirm_dir =env_get("QCONFIRM_DIR");
  cdbfn =env_get("QCONFIRM_OKCDB");

  while ((opt =getopt(argc, argv, "Vc:t:Ddv")) != opteof) {
    switch(opt) {
    case 'c': cdbfn =(char*) optarg; break;
    case 't': scan_ulong(optarg, &timeout); break;
    case 'D': wildhost =0; break;
    case 'd':
      strerr_warn2(WARNING, "the -d option is now default, see qconfirm-cdb-check(1)", 0);
      break;
    case 'v': verbose =1; break;
    case 'V': strerr_warn1(VERSION, 0);
    case '?': usage();
    }
  }
  argv +=optind;
  umask(077);

  if (! qconfirm_dir) qconfirm_dir =QCONFIRMDIR;
  if (! cdbfn)
    if (! (cdbfn =conf_get_dflt(qconfirm_dir, "QCONFIRM_OKCDB", "ok.cdb")))
      fatal("unable to read config: ", "QCONFIRM_OKCDB");
  sender =env_get("SENDER");
  if (! sender)
    strerr_die2x(100, FATAL, "environment variable SENDER not set.\n");
  if (! *sender) { if (verbose) info("ok: ", "empty sender."); _exit(0); }
  if (str_equal(sender, "#@[]")) {
    if (verbose) info("ok: ", "double bounce.");
    _exit(0);
  }

  if (cdbfn[0] != '/') {
    if (! stralloc_copys(&cdb, qconfirm_dir)) die_nomem();
    if (! stralloc_catb(&cdb, "/", 1)) die_nomem();
    if (! stralloc_cats(&cdb, cdbfn)) die_nomem();
  }
  else
    if (! stralloc_copys(&cdb, cdbfn)) die_nomem();
  if (! stralloc_0(&cdb)) die_nomem();

  if (stat(cdb.s, &st) == -1) {
    if (errno == error_noent) {
      warn("unable to stat: ", cdb.s);
      _exit(100);
    }
    fatal("unable to stat: ", cdb.s);
  }
  mode =st.st_mode;

  /* open cdb */
  if ((fdcdb =open_read(cdb.s)) == -1) fatal("unable to open cdb: ", cdb.s);
  cdb_init(&c, fdcdb);
  /* create id */
  if (address2id(&id, sender) == -1) die_nomem();
  /* look up id */
  if(! cdb_find(&c, id.s, id.len -1)) {
    int found =0;
    
    /* check -default */
    for (i =id.len; i >= 0; --i) {
      if (! i || (id.s[i -1] == '-')) {
        if (! stralloc_copyb(&sa, id.s, i)) die_nomem();
        if (! stralloc_cats(&sa, "default")) die_nomem();
	if(cdb_find(&c, sa.s, sa.len)) {
	  if (! stralloc_copyb(&id, sa.s, sa.len)) die_nomem();
	  if (! stralloc_0(&id)) die_nomem();
	  found =1;
	  break;
	}
      }
    }
    if (! found && wildhost) {
      stralloc host ={0};
      
      /* check host: a:b:c:d, :b:c:d, :c:d, :d. */
      if ((i =byte_chr(id.s, id.len, '=')) != id.len) {
	if (! stralloc_copyb(&host, id.s, i)) die_nomem();
	for (i =0; i < host.len; ++i)
	  if ((host.s[i] == ':') || ! i)
	    if(cdb_find(&c, host.s +i, host.len -i)) {
	      if (! stralloc_copyb(&id, host.s +i, host.len -i)) die_nomem();
	      if (! stralloc_0(&id)) die_nomem();
	      found =1;
	      break;
	    }
      }
    }
    if (! found) {
      cdb_free(&c);
      close(fdcdb);
      if (verbose) info("unknown: ", id.s);
      _exit(100);
    }
  }

  /* found */
  dpos =cdb_datapos(&c);
  dlen =cdb_datalen(&c);
  if (timeout) {
    /* read timestamp */
    if (cdb_datalen(&c) != 8) fatal("cdb format error: ", cdb.s);
    if (cdb_read(&c, tspack, 8, dpos) == -1)
      fatal("unable to read cdb: ", cdb.s);
    tai_unpack(tspack, &ts);
    tai_now(&now);
    max.x =ts.x +timeout;
    if (! tai_less(&max, &now)) { info("ok: ", id.s); _exit(0); }

    /* update cdb */
    tai_pack(tspack, &now);
    /* lock cdb */
    if (lock_exnb(fdcdb) == -1)
      if (verbose) warn("lock is busy, waiting...: ", cdbfn);
    if (lock_ex(fdcdb) == -1) fatal("unable to lock: ", cdbfn);
    /* copy cdb cdb.tmp */
    if (seek_begin(fdcdb) == -1) fatal("unable to seek: ", cdb.s);
    if (! stralloc_copys(&tmp, cdb.s)) die_nomem();
    if (! stralloc_cats(&tmp, ".tmp")) die_nomem();
    if (! stralloc_0(&tmp)) die_nomem();
    if ((fdtmp =open_trunc(tmp.s)) == -1) fatal("unable to create: ", tmp.s);
    buffer_init(&bcdb, buffer_unixread, fdcdb, bcdbspace, sizeof bcdbspace);
    buffer_init(&btmp, buffer_unixwrite, fdtmp, btmpspace, sizeof btmpspace);
    while ((i =buffer_feed(&bcdb)) > 0) {
      if (buffer_put(&btmp, buffer_peek(&bcdb), i) == -1) 
	fatal("unable to write: ", tmp.s);
      buffer_seek(&bcdb, i);
    }
    if (i == -1) fatal("unable to read: ", cdb.s);
    if (buffer_flush(&btmp) == -1) fatal("unable to write: ", tmp.s);
    
    /* manipulate cdb.tmp */
    if (seek_set(fdtmp, dpos) == -1) fatal("unable to seek: ", tmp.s);
    if (write(fdtmp, tspack, 8) != 8) fatal("unable to write: ", tmp.s);
    
    if (fsync(fdtmp) == -1) fatal("unable to write: ", tmp.s);
    if (close(fdtmp) == -1) fatal("unable to write: ", tmp.s);
    close(fdcdb);
    
    /* move cdb.tmp cdb */
    if (rename(tmp.s, cdb.s) == -1) fatal("unable to replace: ", cdb.s);
    if (chmod(cdb.s, mode) == -1) warn("unable to chmod: ", cdb.s);
    info("update: ", id.s);
    _exit(0);
  }

  info("ok: ", id.s);
  return(0);
}
