#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "qconfirm.h"
#include "env.h"
#include "strerr.h"
#include "stralloc.h"
#include "open.h"
#include "buffer.h"
#include "error.h"
#include "pathexec.h"
#include "wait.h"

#define USAGE " id dir [prog]"
#define FATAL "qconfirm-accept: fatal: "
#define WARNING "qconfirm-accept: warning: "
#define INFO "qconfirm-accept: info: "

const char *progname;

void usage() { strerr_die4x(111, "usage: ", progname, USAGE, "\n"); }
void die_nomem() { strerr_die2x(111, FATAL, "out of memory."); }
void warn_unlink(char *fn) {
  strerr_warn4(WARNING, "unable to unlink: ", fn, ": ", &strerr_sys);
}

stralloc fn ={0};
int fd;
buffer b;
char b_space[BUFFER_OUTSIZE];
const char *id;
const char *qconfirm_dir;
char *extension;
const char **prog;

int main(int argc, const char **argv) {
  int r;
  char *s;
  unsigned char x;

  progname =*argv++;
  umask(077);

  if (! (id =*argv++)) usage();
  if (! (qconfirm_dir =*argv++)) usage();
  prog =argv;

  extension =env_get("EXT");
  if (! extension)
    strerr_die2x(111, FATAL, "environment variable EXT not set.");

  /* lowercase ext, replace dot with colon */
  for (s =extension; *s; ++s) {
    x =*s -'A';
    if (x <= 'Z' -'A') *s =x +'a';
    if (*s == '.') *s =':';
  }

  /* setlock? */

  /* create ok/<id> */
  if (! stralloc_copys(&fn, qconfirm_dir)) die_nomem();
  if (! stralloc_cats(&fn, "/ok/")) die_nomem();
  if (! stralloc_cats(&fn, id)) die_nomem();
  if (! stralloc_0(&fn)) die_nomem();
  if ((fd =open_trunc(fn.s)) == -1)
    strerr_die4sys(111, FATAL, "unable to create: ", fn.s, ": ");
  buffer_init(&b, buffer_unixwrite, fd, b_space, sizeof b_space);
  while ((r =buffer_feed(buffer_0)) > 0) {
    buffer_put(&b, buffer_peek(buffer_0), r);
    buffer_seek(buffer_0, r);
  }
  buffer_flush(&b);
  close(fd);

  /* remove pending/<id> */
  if (! stralloc_copys(&fn, qconfirm_dir)) die_nomem();
  if (! stralloc_cats(&fn, "/pending/")) die_nomem();
  if (! stralloc_cats(&fn, id)) die_nomem();
  if (! stralloc_0(&fn)) die_nomem();
  if (unlink(fn.s) == -1) warn_unlink(fn.s);

  /* remove .qmail-qconfirm-<key> */
  if (! stralloc_copys(&fn, ".qmail-")) die_nomem();
  if (! stralloc_cats(&fn, extension)) die_nomem();
  if (! stralloc_0(&fn)) die_nomem();
  if (unlink(fn.s) == -1) warn_unlink(fn.s);

  /* remove .qconfirm/return/<id> if it exists */
  if (! stralloc_copys(&fn, qconfirm_dir)) die_nomem();
  if (! stralloc_cats(&fn, "/return/")) die_nomem();
  if (! stralloc_cats(&fn, id)) die_nomem();
  if (! stralloc_0(&fn)) die_nomem();
  unlink(fn.s); /* don't care if this fails */

  strerr_warn2(INFO, id, 0);

  if (*prog) {
    /* run prog */
    int pid;
    int wstat;

    if (! pathexec_env("QCONFIRM_DIR", qconfirm_dir)) die_nomem();
    if ((pid =fork()) == -1) strerr_die2sys(0, FATAL, "unable to fork: ");
    if (! pid) {
      /* child */
      pathexec(prog);
      strerr_die4sys(111, FATAL, "unable to start child: ", *prog, ": ");
    }
    if (wait_pid(&wstat, pid) != pid) strerr_die2sys(0, FATAL, "wait_pid: ");
    if (wait_exitcode(wstat) != 0)
      strerr_warn3(WARNING, "child exits non-zero: ", *prog, 0);
  }
  return(0);
}
