/*
   Copyright (c) 1999 MicroSource

   This source file and all accompanying files are the exclusive property
   of the author Paul Carapetis and MicroSource, and no modifications are
   to be performed on said source without expressed permission from the
   author or an authorised representative of MicroSource, unless the
   purpose of such modifications is to effect a successful compile on your
   platform.

   This source is only to be distributed in its complete form and only with
   all of the following files included:

        INSTALL
        README
        ibmchars.uu
        license.doc
        newargs.txt
        pcps.man
        register.doc

        makefile
        define.h
        pcps.c
        pcps.h
        struct.h
        unix.c
        unix.h
        valargs.h

   The utility "PCPS" is a SHAREWARE product and you, as the user, agree
   that the source will be removed from your filesystem once the utility
   has been successfully compiled and is found to be functional.  Any
   changes to the source that were required for a successful compile must
   be returned to the author for inclusion in the main source control.

   The utility may be trialed for the period of time mentioned in the file
   license.doc, but must either be registered or destroyed within said
   trial time.  The cost and method of registration is detailed in the file
   license.doc.

Author:  Paul Carapetis  Internet: Paul_Carapetis@iname.com
Company: MicroSource     Snail:    10 Denise Court
                                   Narre Warren
                                   Australia  3805

*/

/* unix.c - routines applicable only to the Unix version */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "define.h"
#include "struct.h"

struct stat FileStatus;
struct tm *Timeptr;                /* for curr time when using stdin */

/*****************************************************************************/
/*        External variables                                                 */
/*****************************************************************************/

extern char Host[TMSZ];            /* Machine name for TM */
extern char UserID[TMSZ];          /* Userid or machine name for TM in corner */
extern char Origin[MAX_PATH];      /* Origin path of PCPS.EXE (argv[0]) */
extern BOOL bNoDisplay;            /* fprintf's will not work */
extern FILE *Display;              /* where to send the fancy info */
extern char VerString[30];         /* string to store version (PCPS V##) */
extern file_dtl FndFile;           /* destination structure for information */
extern char VerString[30];         /* string to store version (PCPS V##) */
extern char *HelpMessage[];
extern font FontList[];
extern unsigned Stdout_Tty;        /* non zero if stdout is attached to a tty */
extern FILE *Pipe;                 /* keeps track of open pipe */
/*****************************************************************************/
/*                Function prototypes                                        */
/*****************************************************************************/

void  BadSys(int);              /* routine called when bad system call */
void  BadPipe(int);             /* routine called for bad pipe */
void  GetName(void);            /* get userid or mach name for TM in corner */
void  GetOrigin(char *);        /* find the origin directory of pcps */
BOOL  GetStat(char *);          /* get file stats for unix */
void  Interrupt(int);           /* routine called when int or break detected */
FILE *OpenMore(void);           /* open a pipe to more or to stdout */
void  ShowUsage(void);          /* show how to use */
char *strlwr(char *);           /* convert string to lower case */
char *strupr(char *);           /* convert string to upper case */
extern MyExit(unsigned,unsigned); /* my exit routine */


/*****************************************************************************/
/*  Function BadPipe()                                                       */
/*                                                                           */
/*    Called when bad pipe detected - call MyExit to cleanup                 */
/*****************************************************************************/
void BadPipe(int i)
{
  SHOW(stderr,"\nBad pipe - cleaning up...\n");
  MyExit(2,NOHELP);
}

/*****************************************************************************/
/*  Function BadSys()                                                        */
/*                                                                           */
/*    Called when bad system call detected - call MyExit to cleanup          */
/*****************************************************************************/
void BadSys(int i)
{
  SHOW(stderr,"\nBad system call detected - cleaning up...\n");
  MyExit(2,NOHELP);
}

/*****************************************************************************/
/*  Function GetName()                                                       */
/*                                                                           */
/*    Get the userid and hostname under unix                                 */
/*****************************************************************************/
void GetName(void)
{
  char *cp;

  if ((cp = getenv("LOGNAME")) != NULL)
  {
    strncpy(UserID,cp,TMSZ-1); /* env var USER */
  }
  else
  {                             /* no LOGNAME so execute "whoami" */
    Pipe = (FILE *)popen("whoami","r");
    fgets(UserID,TMSZ-1,Pipe);
    pclose(Pipe);
    Pipe=NULL;
    UserID[strlen(UserID)-1] = '\0'; /* throw away the \n */
  }
  Pipe = (FILE *)popen("hostname","r");   /* execute command and read response */
  fgets(Host,TMSZ-1,Pipe);
  pclose(Pipe);
  Pipe=NULL;
  Host[strlen(Host)-1] = '\0';       /* throw away the \n */
}

/*****************************************************************************/
/*  Function GetOrigin()                                                     */
/*                                                                           */
/*    Copy the origin directory of PCPS into the variable Origin.            */
/*****************************************************************************/
void GetOrigin(char *Orig)
{
  char *pp;
  static char Path[MAX_PATH], EnvPath[MAX_PATH];
  static char *cp, *ep;

  if (strchr(Orig,PD) == NULL)     /* no path, so go find which one */
  {
    Origin[0] = '\0';              /* start as null */
    if ( (cp = getenv("PATH")) != NULL)
    {
      /* I was going to use "which", but this is slow, relies on csh being
         present and is fiddly dealing with output of .cshrc.

         OK, so I have to go look for it.  If no explicit path entered,
         then it was the first occurence in the path.  We now have the path
         from the environment, so tack argv[0] onto the end of each portion
         and try to open the file for a read.  The first one we find is the
         origin path.
         The path is in the form:     path:path:path
         so we find the next ':', replace it with '\0' try the path
         preceding it, if unsuccessful, inc pointer past '\0' and repeat
         until successful or run out of path
       */
      strcpy(EnvPath,cp);
      ep = &EnvPath[0];
      cp = strchr(ep,':');                 /* find first ':' */
      *cp = '\0';                          /* make it end of string */
      while (cp)
      {
        sprintf(Path,"%s%c%s",ep,PD,Orig);     /* create path/argv[0] */
        if (access(Path,X_OK) == 0)
        {                          /* found it */
          strcpy(Origin,Path);     /* remember origin */
          break;
        }
        ep = ++cp;                       /* go to next part of string */
        if ( (cp = strchr(ep,':')) == NULL)  /* no more ':' */
        {
          if (strlen(ep) == 0)
          {
            break;    /* if at end then break, else there is still path after last ':' */
          }
          cp = ep + strlen(ep) -1;  /* so reset of EnvPath works */
        }
        else                                  /* still more ':', so continue */
        {
          *cp = '\0';                         /* make it end of string */
        }
      }
    }
  }
  else             /* there were slashes, so resolve path */
  {
#if defined(RS6K) || defined(ULTRIX)/* no realpath so just hack argv[0] */

    strcpy(Origin,Orig);       /* and copy it into Origin */

#else

    realpath(Orig,Origin);

#endif
  }



  if (*Origin == '\0')
  {
    SHOW(Display,"%sCould not find origin - support files may not be found.\n",
                    VerString);
  }

  if (Origin[strlen(Origin)-1] != PD)
  {
    cp = strrchr(Origin,PD);     /* find last slash */
    *++cp = '\0';                /* cut down to dirname */
  }
}

/*****************************************************************************/
/*  Function GetStat()                                                       */
/*                                                                           */
/*    Get file statistics under unix                                         */
/*****************************************************************************/
BOOL GetStat(char *filename)
{

  if (stat(filename, &FileStatus))
  {
    return(FALSE);
  }
  else
  {
    strcpy(FndFile.fname,filename);
    Timeptr = localtime(&FileStatus.st_mtime);
    FndFile.fsecs = Timeptr->tm_sec;
    FndFile.fmins = Timeptr->tm_min;
    FndFile.fhour = Timeptr->tm_hour;
    FndFile.fday  = Timeptr->tm_mday;
    FndFile.fmonth = Timeptr->tm_mon + 1;
    FndFile.fyear = Timeptr->tm_year + 1900;
    FndFile.fsize = FileStatus.st_size;
    return(TRUE);
  }
}

/*****************************************************************************/
/*  Function Interrupt()                                                     */
/*                                                                           */
/*    Called when break or int pressed - call MyExit                         */
/*****************************************************************************/
void Interrupt(int i)
{
  fprintf(stderr,"\nInterrupt detected - cleaning up...\n");
  MyExit(2,NOHELP);
}


/*****************************************************************************/
/*  Function  ListFonts()                                                    */
/*    List fonts, and fontcodes to console...                                */
/*****************************************************************************/
void ListFonts(void)
{
  register unsigned i;

  Pipe = OpenMore();

  fprintf(Pipe
          ,"\nFontCode\tFont\n"
           "--------\t----\n\n"
         );
  for (i=0; i < NUMBFONTS; i++)
  {
    fprintf(Pipe,"%s\t\t%s\n",FontList[i].FontIndex,FontList[i].FontName);
  }
  pclose(Pipe);

  MyExit(1,NOHELP);
}

/*****************************************************************************/
/*  Function OpenMore()                                                      */
/*                                                                           */
/*    Open a pipe to a more command if output going to tty, or set it to     */
/*    stdout if not                                                          */
/*****************************************************************************/
FILE *OpenMore(void)
{
  if (Stdout_Tty)
  {
    return((FILE *)popen(MORECMD,"w"));
  }
  else
  {
    return(stdout);
  }
}

/*****************************************************************************/
/*  Function ShowUsage()                                                     */
/*                                                                           */
/*     Print Usage message with [more] capability.                           */
/*****************************************************************************/
void  ShowUsage(void)
{
  char **printline;

#ifdef FUNCS
fprintf(Display,"->ShowUsage\n");
#endif

  Pipe = OpenMore();

  fprintf(Pipe,"\n%s %s\t\t%s\n",VerString,MODE,COPYRIGHT);
  for (
       printline = HelpMessage;                           /* start of message */
       *printline != (char *)NULL;                        /* stop when at EOM */
       printline++
      )
  {
    fprintf(Pipe,"%s\n",*printline);
  }
  pclose(Pipe);
  Pipe=NULL;

  MyExit(1,NOHELP);
}

/*****************************************************************************/
/*  Function strlwr()                                                        */
/*                                                                           */
/*    Convert string to lower case                                           */
/*****************************************************************************/
char *strlwr(char *sp)
{
  register int i;
  register char *str;

  str = sp;
  for (i=0; str[i] != 0; i++)
  {
    if (str[i] >= 'A' && str[i] <= 'Z')
    {
      str[i] |= 0x20;
    }
  }
  return(sp);
}

/*****************************************************************************/
/*  Function strupr()                                                        */
/*                                                                           */
/*    Convert string to upper case                                           */
/*****************************************************************************/
char *strupr(char *sp)
{
  register int i;
  register char *str;

  str = sp;
  for (i=0; str[i] != 0; i++)
  {
    if (str[i] >= 'a' && str[i] <= 'z')
    {
      str[i] &= 0xdf;
    }
  }
  return(sp);
}


