const std = @import("std");
const fs = std.fs;
const io = std.io;
const os = std.os;
const mem = std.mem;
const testing = std.testing;
const expect = testing.expect;
const expectEqual = testing.expectEqual;
const expectEqualSlices = testing.expectEqualSlices;
const expectError = testing.expectError;
const ziglyph = @import("ziglyph");
const pp = @import("main.zig");

const words =
    \\sitjestillingi
    \\pubertetsalder
    \\knivstikkar
    \\profesjonist
    \\matreiddum
    \\randomisera
    \\aksleleddi
    \\lunarisk
    \\hostemikstur
    \\folklorist
    \\familiefest
    \\velkometale
    \\putleteare
    \\samletal
    \\bankgruppe
    \\hypokonder
    \\terskelfjord
    \\illiberalitet
    \\slagbord
    \\utklass
;

const uniwords =
    \\åååå
    \\ääää
    \\öööö
    \\ææææ
    \\þþþþ
;

const uniwords_nofold =
    \\안녕하세요
    \\
;

fn Ctx(comptime buf_size: usize) type {
    return struct {
        out_buf: [buf_size]u8 = undefined,
        stream: io.FixedBufferStream([]u8),
        prng: std.rand.Random,

        pub fn init() @This() {
            var xoshiro = std.rand.DefaultPrng.init(0);
            var ctx: @This() = undefined;
            ctx.stream = io.fixedBufferStream(&ctx.out_buf);
            ctx.prng = xoshiro.random();
            return ctx;
        }
    };
}

test "number of words" {
    var ctx = Ctx(0x1000).init();
    try pp.writeRandomWords(
        ctx.stream.writer(),
        &ctx.prng,
        words,
        .{
            .num_words = 20,
            .capitalize = false,
            .separator = " ",
        },
    );
    var wc: u8 = 1;
    for (ctx.stream.getWritten()) |b| {
        if (' ' == b)
            wc += 1;
    }
    try expectEqual(@as(u8, 20), wc);
}

test "add symbol" {
    var ctx = Ctx(0x1000).init();
    const words2 = "apa\nvad\nlol\n";
    try pp.writePassphrase(
        ctx.stream.writer(),
        &ctx.prng,
        words2,
        .{
            .add_symbol = true,
            .add_digit = false,
            .capitalize = false,
            .separator = " ",
        },
    );
    const phrase = ctx.stream.getWritten();
    var syms_found: usize = 0;
    for (phrase) |b|
        if (isSymbol(b)) {
            syms_found += 1;
        };
    try expectEqual(@as(usize, 1), syms_found);
}

fn isSymbol(char: u8) bool {
    // zig fmt: off
    return ('!' <= char and char <= '/')
        or (':' <= char and char <= '@')
        or ('[' <= char and char <= '`')
        or ('{' <= char and char <= '~');
    // zig fmt: on
}

test "add digit" {
    var ctx = Ctx(0x1000).init();
    const words2 = "apa\nvad\nlol\n";
    try pp.writePassphrase(
        ctx.stream.writer(),
        &ctx.prng,
        words2,
        .{
            .add_symbol = false,
            .add_digit = true,
            .capitalize = false,
            .separator = " ",
        },
    );
    const phrase = ctx.stream.getWritten();
    var digits_found: usize = 0;
    for (phrase) |b|
        if ('0' <= b and b <= '9') {
            digits_found += 1;
        };
    try expectEqual(@as(usize, 1), digits_found);
}

test "capitalize" {
    var ctx = Ctx(0x1000).init();
    try pp.writePassphrase(
        ctx.stream.writer(),
        &ctx.prng,
        words,
        .{
            .num_words = 3,
            .add_symbol = false,
            .add_digit = false,
            .capitalize = true,
            .separator = " ",
        },
    );
    var cc: u8 = 0;
    for (ctx.stream.getWritten()) |b| {
        if (ziglyph.general_category.isUppercaseLetter(b))
            cc += 1;
    }
    try expectEqual(@as(u8, 3), cc);
}

const fulani =
    \\𞤥𞤵𞤤𞤵𞤺𞤮𞤤
    \\
;

test "capitalize fulani" {
    var ctx = Ctx(0x1000).init();
    try pp.writePassphrase(
        ctx.stream.writer(),
        &ctx.prng,
        fulani,
        .{
            .num_words = 1,
            .add_symbol = false,
            .add_digit = false,
            .capitalize = true,
            .separator = "",
        },
    );
    var cc: u8 = 0;
    var it = ziglyph.CodePointIterator{ .bytes = ctx.stream.getWritten(), .i = 0 };

    while (it.next()) |cp| {
        if (ziglyph.general_category.isUppercaseLetter(cp.code)) {
            try expectEqual(@as(usize, 0), cp.offset);
            cc += 1;
        }
    }
    try expectEqual(@as(u8, 1), cc);
}

test "case-fold unicode" {
    var ctx = Ctx(0x1000).init();
    try pp.writePassphrase(
        ctx.stream.writer(),
        &ctx.prng,
        uniwords,
        .{
            .num_words = 3,
            .add_symbol = false,
            .add_digit = false,
            .capitalize = true,
            .separator = "",
        },
    );
    var it = (try std.unicode.Utf8View.init(ctx.stream.getWritten())).iterator();
    var cc: u8 = 0;
    while (it.nextCodepoint()) |cp| {
        if (ziglyph.general_category.isUppercaseLetter(cp))
            cc += 1;
    }
    try expectEqual(@as(u8, 3), cc);
}

test "case-fold unfoldable unicode" {
    var ctx = Ctx(0x1000).init();
    try pp.writePassphrase(
        ctx.stream.writer(),
        &ctx.prng,
        uniwords_nofold,
        .{
            .capitalize = true,
        },
    );
    const writ = ctx.stream.getWritten();
    const l = uniwords_nofold.len - 1;

    try expectEqualSlices(u8, uniwords_nofold[0..l], writ[0..l]);
}

test "empty wordlist" {
    var ctx = Ctx(0x1000).init();
    // A completely empty file is rejected when mmapping.
    const words2 = "\n";
    const res = pp.writePassphrase(
        ctx.stream.writer(),
        &ctx.prng,
        words2,
        .{
            .num_words = 1,
            .add_symbol = false,
            .add_digit = false,
            .capitalize = false,
            .separator = "",
        },
    );
    try expectError(error.InvalidFormat, res);
}

test "password" {
    var ctx = Ctx(0x1000).init();
    const len: usize = 24;
    try pp.writePassword(ctx.stream.writer(), &ctx.prng, len);
    try expectEqual(len, ctx.stream.getWritten().len);
}
