/*
 * palm-db-tools: Read/write JFile v3 databases
 * Copyright (C) 2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string>
#include <vector>
#include <strstream>
#include <cstring>

#include "JFile3.h"
#include "libsupport/strop.h"

using namespace PalmLib;
using namespace PalmLib::FlatFile;

bool PalmLib::FlatFile::JFile3::classify(const PalmLib::Database& pdb)
{
    return (pdb.creator() == PalmLib::mktag('J','B','a','s'))
        && (pdb.type()    == PalmLib::mktag('J','b','D','b'));
}

bool PalmLib::FlatFile::JFile3::match_name(const std::string& name)
{
    return (name == "JFile3") || (name == "jfile3")
        || (name == "JF3") || (name == "jf3");
}

PalmLib::Record
PalmLib::FlatFile::JFile3::build_record(const std::vector<std::string>& fields) const
{
    unsigned i;

    // Calculate the size of the record.
    unsigned size = 0;
    for (i = 0; i < fields.size(); ++i) {
        size += 1 + fields[i].length();
    }

    // Allocate a record of the calculated size.
    PalmLib::Record record(0, 0, size);

    // Pack the strings into the record.
    PalmLib::Record::pointer p = record.data();
    for (i = 0; i < fields.size(); ++i) {
        strcpy((char *) p, fields[i].c_str());
        p += 1 + fields[i].length();
    }

    // The caller can now have fun with the packed record.
    return record;
}

std::vector<std::string>
PalmLib::FlatFile::JFile3::parse_record(const PalmLib::Record& rec) const
{
    std::vector<std::string> result;

    // Point at the start of the record data.
    PalmLib::Record::const_pointer p = rec.data();

    // Loop while we can extract more information.
    while (p != rec.end()) {
        PalmLib::Record::const_pointer null_ptr =
            reinterpret_cast<PalmLib::Record::const_pointer>
            (memchr(p, 0, rec.size() - (p - rec.data())));

        if (!null_ptr)
            throw PalmLib::error("corrupt record: unterminated string");

        result.push_back(std::string((char *) p, null_ptr - p));
        p = null_ptr + 1;
    }

    return result;
}

PalmLib::FlatFile::JFile3::JFile3(const PalmLib::Database& pdb)
    : Database("jf3", pdb), m_flags(0)
{
    unsigned i, j;
    PalmLib::FlatFile::ListView lv;

    // Unpack the application information block.
    JFileAppInfoType hdr;
    hdr.unpack(pdb.getAppInfoBlock());
    m_password = hdr.password;
    m_flags = hdr.flags;

    // Extract the schema from the header.
    for (i = 0; i < hdr.numFields; ++i) {
        PalmLib::FlatFile::Field::FieldType type;

        switch (hdr.fieldTypes[i]) {
        case 0x0001:
            type = PalmLib::FlatFile::Field::STRING;
            break;

        case 0x0002:
            type = PalmLib::FlatFile::Field::BOOLEAN;
            break;

        case 0x0004:
            type = PalmLib::FlatFile::Field::DATE;
            break;

        case 0x0008:
            type = PalmLib::FlatFile::Field::INTEGER;
            break;

        case 0x0010:
            type = PalmLib::FlatFile::Field::FLOAT;
            break;

        case 0x0020:
            type = PalmLib::FlatFile::Field::TIME;
            break;

         case 0x0040: // LIST - fake it as a string
            type = PalmLib::FlatFile::Field::STRING;
            break;

        default:
            throw PalmLib::error("corrupt header: unknown field type");
        }

        // Add this field to the single list view.
        lv.push_back(ListViewColumn(i, hdr.showDBColumnWidths[i]));

        // Add this field to the schema.
        appendField(hdr.fieldNames[i], type);
    }

    // Add the constructed list view as the only list view.
    appendListView(lv);

    // Extract the records.
    for (i = 0; i < pdb.getNumRecords(); ++i) {
        PalmLib::FlatFile::Record rec;
        PalmLib::Record pdb_record = pdb.getRecord(i);

        // Skip over all deleted records.
        if (pdb_record.deleted())
            continue;

        // Parse out the fields of the record.
        std::vector<std::string> fields = parse_record(pdb_record);
        if (fields.size() != getNumOfFields())
            throw PalmLib::error("corrupt record: field number mismatch");
        
        for (j = 0; j < getNumOfFields(); ++j) {
            PalmLib::FlatFile::Field f;

            f.type = field_type(j);
            switch (f.type) {
            case PalmLib::FlatFile::Field::STRING:
                f.v_string = fields[j];
                break;

            case PalmLib::FlatFile::Field::BOOLEAN:
                f.v_boolean = StrOps::string2boolean(fields[j]);
                break;

            case PalmLib::FlatFile::Field::INTEGER:
                StrOps::convert_string(fields[j], f.v_integer);
                break;

            case PalmLib::FlatFile::Field::FLOAT:
                StrOps::convert_string(fields[j], f.v_float);
                break;

            case PalmLib::FlatFile::Field::DATE:
                {
                    StrOps::string_list_t array;

                    try {
                        array = StrOps::str_to_array(fields[j], "/",
                                                     false, false);
                    } catch (...) {
                        f.no_value = true;
                    }

                    if (! (f.no_value || array.size() != 3)) {
                        StrOps::convert_string(array[0], f.v_date.day);
                        StrOps::convert_string(array[1], f.v_date.month);
                        StrOps::convert_string(array[2], f.v_date.year);
                    } else {
                        f.no_value = true;
                    }
                }
                break;

            case PalmLib::FlatFile::Field::TIME:
                {
                    StrOps::string_list_t array;

                    try {
                        array = StrOps::str_to_array(fields[j], ":",
                                                     false, false);
                    } catch (...) {
                        f.no_value = true;
                    }

                    if (! (f.no_value || array.size() != 2)) {
                        StrOps::convert_string(array[0], f.v_time.hour);
                        StrOps::convert_string(array[1], f.v_time.minute);
                    } else {
                        f.no_value = true;
                    }
                }
                break;

            default:
                throw PalmLib::error("corrupt record: unsupported field type");
                break;
            }

            rec.appendField(f);
        }

        appendRecord(rec);
    }
}

void PalmLib::FlatFile::JFile3::outputPDB(PalmLib::Database& pdb) const
{
    unsigned i;

    // Let the superclass have a chance.
    SUPERCLASS(PalmLib::FlatFile,Database,outputPDB,(pdb));

    // Set the type and creator for this database.
    pdb.creator(PalmLib::mktag('J','B','a','s'));
    pdb.type(PalmLib::mktag('J','b','D','b'));

    // Setup the application information block.
    JFileAppInfoType hdr;
    hdr.numFields = getNumOfFields();
    hdr.version = 452;
    hdr.showDataWidth = 80;
    hdr.sortFields[0] = 0;
    hdr.sortFields[1] = 0;
    hdr.sortFields[2] = 0;
    hdr.findField = 0;
    hdr.filterField = 0;
    hdr.findString = "";
    hdr.filterString = "";
    hdr.flags = 0;
    hdr.firstColumnToShow = 0;
    hdr.password = m_password;

    // Place the schema into the application information block.
    for (i = 0; i < getNumOfFields(); ++i) {
        hdr.fieldNames[i] = field_name(i);
        switch (field_type(i)) {
        case PalmLib::FlatFile::Field::STRING:
            hdr.fieldTypes[i] = 0x0001;
            break;

        case PalmLib::FlatFile::Field::BOOLEAN:
            hdr.fieldTypes[i] = 0x0002;
            break;

        case PalmLib::FlatFile::Field::DATE:
            hdr.fieldTypes[i] = 0x0004;
            break;

        case PalmLib::FlatFile::Field::INTEGER:
            hdr.fieldTypes[i] = 0x0008;
            break;

        case PalmLib::FlatFile::Field::FLOAT:
            hdr.fieldTypes[i] = 0x0010;
            break;

        case PalmLib::FlatFile::Field::TIME:
            hdr.fieldTypes[i] = 0x0020;
            break;

        default:
            throw PalmLib::error("unsupported field type");
        }
                               
    }

    // Extract the single list view and place the widths into the block.
    PalmLib::FlatFile::ListView lv = getListView(0);
    PalmLib::FlatFile::ListView::const_iterator iter = lv.begin();
    for (i = 0; iter != lv.end(); ++iter, ++i) {
        const PalmLib::FlatFile::ListViewColumn& col = (*iter);
        hdr.showDBColumnWidths[i] = col.width;
    }

    // Set the application information block.
    pdb.setAppInfoBlock(hdr.pack());

    // Pack the records into the PDB.
    for (i = 0; i < getNumRecords(); ++i) {
        PalmLib::FlatFile::Record rec = getRecord(i);
        std::vector<std::string> fields;

        for (unsigned int j = 0; j < getNumOfFields(); j++) {
#ifdef __LIBSTDCPP_PARTIAL_SUPPORT__
			const Field field = rec.fields()[i];
#else
			const Field field = rec.fields().at(i);
#endif
            switch (field.type) {
            case PalmLib::FlatFile::Field::STRING:
                fields.push_back(field.v_string);
                break;

            case PalmLib::FlatFile::Field::BOOLEAN:
                if (field.v_boolean)
                    fields.push_back("1");
                else
                    fields.push_back("0");
                break;

            case PalmLib::FlatFile::Field::INTEGER:
                {
                    std::ostrstream stream;
                    stream << field.v_integer << std::ends;
                    fields.push_back(stream.str());
                    break;
                }

            case PalmLib::FlatFile::Field::FLOAT:
                {
                    std::ostrstream stream;
                    stream << field.v_float << std::ends;
                    fields.push_back(stream.str());
                    break;
                }

            case PalmLib::FlatFile::Field::DATE:
                {
                    std::ostrstream stream;
                    stream << field.v_date.day << '/';
                    stream << field.v_date.month << '/';
                    stream << field.v_date.year << std::ends;
                    fields.push_back(stream.str());
                    break;
                }

            case PalmLib::FlatFile::Field::TIME:
                {
                    std::ostrstream stream;
                    stream.width(2);
                    stream << field.v_time.hour << ':'
                           << field.v_time.minute << std::ends;
                    fields.push_back(stream.str());
                    break;
                }

            default:
                throw PalmLib::error("unsupported field type");
            }
        }

        // Add the packed data to the PDB.
        pdb.appendRecord(build_record(fields));
    }
}

unsigned PalmLib::FlatFile::JFile3::getMaxNumOfFields() const
{
    return 20;
}

bool PalmLib::FlatFile::JFile3::supportsFieldType(const Field::FieldType& type) const
{
    switch (type) {
    case PalmLib::FlatFile::Field::STRING:
    case PalmLib::FlatFile::Field::BOOLEAN:
    case PalmLib::FlatFile::Field::INTEGER:
    case PalmLib::FlatFile::Field::FLOAT:
    case PalmLib::FlatFile::Field::DATE:
    case PalmLib::FlatFile::Field::TIME:
        return true;

    default:
        return false;
    }
}

unsigned PalmLib::FlatFile::JFile3::getMaxNumOfListViews() const
{
    return 1;
}

void PalmLib::FlatFile::JFile3::doneWithSchema()
{
    // Let the superclass have a chance.
    SUPERCLASS(PalmLib::FlatFile, Database, doneWithSchema, ());

    if (getNumOfListViews() < 1)
        throw PalmLib::error("a list view must be specified");

    PalmLib::FlatFile::ListView lv = getListView(0);

    if (lv.size() != getNumOfFields())
        throw PalmLib::error("the list view must have the same number of columns as fields");

    PalmLib::FlatFile::ListView::const_iterator p = lv.begin();
    unsigned field = 0;

    for (; p != lv.end(); ++p, ++field) {
        const PalmLib::FlatFile::ListViewColumn& col = (*p);
        if (field != col.field) {
            throw PalmLib::error("the list view columns must be in the same order as the fields");
        }
    }
}

void PalmLib::FlatFile::JFile3::setOption(const std::string& name,
                                          const std::string& value)
{
    if (name == "password") {
        m_password = value;
    } else {
        SUPERCLASS(PalmLib::FlatFile,Database,setOption,(name, value));
    }
}

PalmLib::FlatFile::Database::options_list_t
PalmLib::FlatFile::JFile3::getOptions(void) const
{
    typedef PalmLib::FlatFile::Database::options_list_t::value_type value;
    PalmLib::FlatFile::Database::options_list_t result =
        SUPERCLASS(PalmLib::FlatFile,Database,getOptions,());

    if (! m_password.empty()) {
        result.push_back(value("password", m_password));
    }

    return result;
}

void PalmLib::FlatFile::JFile3::JFileAppInfoType::unpack(const PalmLib::Block& block)
{
    unsigned i;
    pi_char_t* null_ptr;

    // Ensure that we have enough space to extract information from.
    if (block.size() < ( (20 * (20+1)) + 20*2 + 2 + 2 + 20*2 + 2
                         + 3*2 + 2 + 2 + 16 + 16 + 2 + 2 + 12 ))
        throw PalmLib::error("header is corrupt");

    // Point at the start of the data block.
    const pi_char_t* p = block.data();

    // Extract the field names.
    for (i = 0; i < 20; ++i) {
        /* Find the trailing null byte and extract the string. */
        null_ptr = reinterpret_cast<pi_char_t*> (memchr(p, 0, 21));
        if (null_ptr)
            fieldNames[i] = std::string((char *) p, null_ptr - p);
        else
            fieldNames[i] = "";

        /* Advance the pointer to the next field name. */
        p += 21;
    }

    // Extract the field types.
    for (i = 0; i < 20; ++i) {
        fieldTypes[i] = PalmLib::get_short(p);
        p += sizeof(pi_uint16_t);
    }

    // Extract the number of fields.
    numFields = PalmLib::get_short(p);
    p += sizeof(pi_uint16_t);

    // Extract the version number and check against the supported versions.
    version = PalmLib::get_short(p);
    p += sizeof(pi_uint16_t);
    if (version != 452)
        throw PalmLib::error("unsupported header version");

    // Extract the list view column widths.
    for (i = 0; i < 20; ++i) {
        showDBColumnWidths[i] = PalmLib::get_short(p);
        p += sizeof(pi_uint16_t);
    }

    // Extract the width of the field labels in the edit view.
    showDataWidth = PalmLib::get_short(p);
    p += sizeof(pi_uint16_t);

    // Extract the fields used for sorting.
    for (i = 0; i < 3; ++i) {
        sortFields[i] = PalmLib::get_short(p);
        p += sizeof(pi_uint16_t);
    }

    // Extract the field last searched by Find.
    findField = PalmLib::get_short(p);
    p += sizeof(pi_uint16_t);

    // Extract the field last used for Filter.
    filterField = PalmLib::get_short(p);
    p += sizeof(pi_uint16_t);

    // Extract the string used last by Find.
    null_ptr = reinterpret_cast<pi_char_t*> (memchr(p, 0, 16));
    if (null_ptr)
        findString = std::string((char *) p, null_ptr - p);
    else
        findString = "";
    p += 16;

    // Extract the string used last by Filter.
    null_ptr = reinterpret_cast<pi_char_t*> (memchr(p, 0, 16));
    if (null_ptr)
        filterString = std::string((char *) p, null_ptr - p);
    else
        filterString = "";
    p += 16;

    // Extract the flags.
    flags = PalmLib::get_short(p);
    p += sizeof(pi_uint16_t);

    // Extract the first column displayed in the list view.
    firstColumnToShow = PalmLib::get_short(p);
    p += sizeof(pi_uint16_t);

    // Extract the password (if any).
    null_ptr = reinterpret_cast<pi_char_t*> (memchr(p, 0, 12));
    if (null_ptr)
        password = std::string((char *) p, null_ptr - p);
    else
        password = "";
    p += 12;

    // XXX: Extract the popup strings.
}

PalmLib::Block
PalmLib::FlatFile::JFile3::JFileAppInfoType::pack() const
{
    unsigned i;

    PalmLib::Block block(4096);

    // Clear out the entire block.
    memset(block.data(), 0, block.size());
    
    // Point at the start of the data block.
    pi_char_t* p = block.data();

    // Pack the field names.
    for (i = 0; i < 20; ++i) {
        /* Find the trailing null byte and extract the string. */
        strncpy((char *) p, fieldNames[i].c_str(), 21);
        p[21] = '\0';

        /* Advance the pointer to the position. */
        p += 21;
    }

    // Pack the field types.
    for (i = 0; i < 20; ++i) {
        PalmLib::set_short(p, fieldTypes[i]);
        p += sizeof(pi_uint16_t);
    }

    // Pack the number of fields.
    PalmLib::set_short(p, numFields);
    p += sizeof(pi_uint16_t);

    // Pack the version number.
    PalmLib::set_short(p, version);
    p += sizeof(pi_uint16_t);

    // Pack the list view column widths.
    for (i = 0; i < 20; ++i) {
        PalmLib::set_short(p, showDBColumnWidths[i]);
        p += sizeof(pi_uint16_t);
    }

    // Pack the width of the field labels in the edit view.
    PalmLib::set_short(p, showDataWidth);
    p += sizeof(pi_uint16_t);

    // Pack the fields used for sorting.
    for (i = 0; i < 3; ++i) {
        PalmLib::set_short(p, sortFields[i]);
        p += sizeof(pi_uint16_t);
    }

    // Pack the field last searched by Find.
    PalmLib::set_short(p, findField);
    p += sizeof(pi_uint16_t);

    // Pack the field last used for Filter.
    PalmLib::set_short(p, filterField);
    p += sizeof(pi_uint16_t);

    // Pack the string used last by Find.
    strncpy((char *) p, findString.c_str(), 16);
    p[16] = '\0';
    p += 16;

    // Pack the string used last by Filter.
    strncpy((char *) p, filterString.c_str(), 16);
    p[16] = '\0';
    p += 16;

    // Pack the flags.
    PalmLib::set_short(p, flags);
    p += sizeof(pi_uint16_t);

    // Pack the first column displayed in the list view.
    PalmLib::set_short(p, firstColumnToShow);
    p += sizeof(pi_uint16_t);

    // Pack the password (if any).
    strncpy((char *) p, password.c_str(), 12);
    p[12] = '\0';
    p += 12;

    // Pack the 4 null terminators marking the end of the popup lists.
    *p++ = '\0';
    *p++ = '\0';
    *p++ = '\0';
    *p++ = '\0';

    // Resize the block to the actual space used.
    block.resize(p - block.data());
    return block;
}
