/*
 * u2c.c - Create convertion table from GB2312.TXT or GB12345p.TXT.
 *
    Copyright (C) 2001  Yao Zhang

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * yaoz@users.sourceforge.net
 */
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>

#include "u2c.h"

static int compare(const void *x, const void *y)
{
    U2CEntry *a, *b;

    a = (U2CEntry *)x;
    b = (U2CEntry *)y;

    return a->u - b->u;
}

U2C *u2c_open(char *file)
{
    FILE *fp;
    int count;
    char line[BUFSIZ];
    U2C *u2c;

    fp = fopen(file, "r");
    if (fp == NULL) {
        char path[MAXPATHLEN];

        strcpy(path, PKGDATADIR);
        strcat(path, "/");
        strcat(path, file);
        fp = fopen(path, "r");
        if (fp == NULL) {
            return NULL;
        }
    }

    count = 0;
    while (fgets(line, BUFSIZ, fp) != NULL) {
        uint32_t u, c;

        u = c = 0;
        sscanf(line, "%X%X", &c, &u);
        if ((u > 0) && (c > 0)) {
            count++;
        }
    }

    count += 128;

    u2c = (U2C *)calloc(1, sizeof(U2C));
    u2c->length = count;
    u2c->array = (U2CEntry *)calloc(count, sizeof(U2CEntry));

    /* ASCII portion */
    for (count = 0; count < 128; count++) {
        u2c->array[count].u = count;
        u2c->array[count].c = count;
    }

    /* GB/BIG5 portion: in EUC */
    fseek(fp, 0, SEEK_SET);
    while (fgets(line, BUFSIZ, fp) != NULL) {
        uint32_t u, c;

        u = c = 0;
        sscanf(line, "%X%X", &c, &u);
        if ((u > 0) && (c > 0)) {
            u2c->array[count].u = u;
            if (strcmp(file, BIG5) == 0)
                u2c->array[count].c = c;
            else
                u2c->array[count].c = c + 0x8080;
            count++;
        }
    }
    qsort(u2c->array, u2c->length, sizeof(U2CEntry), compare);

    fclose(fp);

    return u2c;
}

void u2c_close(U2C *u2c)
{
    free(u2c->array);
    free(u2c);
}

uint32_t u2c_convert(U2C *u2c, uint32_t u)
{
    U2CEntry key, *found;

    key.u = u;
    found = bsearch(&key, u2c->array, u2c->length, sizeof(U2CEntry), compare);
    if (found == NULL) {
        return 0;
    }

    return found->c;
}

#ifdef TEST_U2C
int main(int argc, char *argv[])
{
    int i;
    U2C *u2c;
    uint32_t u, c;

    u2c = u2c_open(argv[1]);
    for (i = 0; i < u2c->length; i++) {
        printf("0x%X\t0x%X\n", u2c->array[i].u, u2c->array[i].c);
    }
    u = 0x345;
    c = u2c_convert(u2c, u);
    printf("0x%X: 0x%X\n", u, c);
    u = 0x64;
    c = u2c_convert(u2c, u);
    printf("0x%X: 0x%X\n", u, c);
    u = 0x5927;			/* BIG */
    c = u2c_convert(u2c, u);	/* 0xB4F3 */
    printf("0x%X: 0x%X\n", u, c);
    u2c_close(u2c);

    return 0;
}
#endif
