/* Copyright (C) 2004 Manuel Bouyer

dual_led is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

dual_led is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with dual_led; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <string>
#include <iostream>

#include <gtk/gtk.h>

#include "oscilloscope.h"

gboolean oscw_realize(GtkWidget *Widget, gpointer data);
gboolean oscw_expose(GtkWidget *Widget, GdkEvent  *event, gpointer data);
gboolean osc_realize(GtkWidget *Widget, gpointer data);
gboolean osc_expose(GtkWidget *Widget, GdkEvent  *event, gpointer data);

gboolean osc_gspin_min(GtkAdjustment* adj, gpointer data);
gboolean osc_gspin_max(GtkAdjustment* adj, gpointer data);

#define GRAPH_MARGIN 4

#define MAX_OSC_INPUT 16


typedef struct event {
	guint64 cycle;
	gdouble value;
} event_t;

/*
 * oscilloscope_graph is a GtkBox containing one graph in a GtkDrawingArea,
 * and one rulers on the right.
 * It is derived from IO_input, and intercepts put_node_state().
 */

class oscilloscope_graph : public IOPIN {
public:
	virtual void set_nodeVoltage(double v);
	guint64 *max_cycle;
	guint64 *view_min_cycle;
	guint64 *view_max_cycle;
	oscilloscope_graph(char *opt_name=NULL);
	~oscilloscope_graph(void);
	virtual void realize();
	virtual void expose(bool clear);
	// the main widgets
	GtkWidget *OscGraph;
	GtkWidget *OscRuler;
private:
	// our widgets
	GtkWidget *Graph;
	// an array of events, dynamically allocated
	event_t *events;
	int nevents; // size of *events
	int current_event; // the last record in *events
	GdkColor GraphColor;
};

oscilloscope_graph::oscilloscope_graph(char *opt_name) :
    IOPIN(NULL, 0, opt_name, (Register **)0)
{
	GtkRcStyle *rc_style;
	GdkColor color;

	gdk_color_parse("black", &color);
	rc_style = gtk_rc_style_new ();
	rc_style->bg[GTK_STATE_NORMAL] = color;
	rc_style->color_flags[GTK_STATE_NORMAL] = GTK_RC_BG;

	OscGraph = gtk_event_box_new();
	gtk_widget_set_usize(OscGraph, 500, 50);
	OscRuler = gtk_vruler_new();
	gtk_widget_set_usize(OscRuler, 30, 50);
	Graph = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(OscGraph), Graph);

	gtk_ruler_set_range(GTK_RULER(OscRuler), 1, -1, -1, 1);

	gtk_signal_connect(GTK_OBJECT(Graph), "realize",
		GTK_SIGNAL_FUNC(oscw_realize), this);
	gtk_signal_connect(GTK_OBJECT(Graph), "expose_event",
		GTK_SIGNAL_FUNC(oscw_expose), this);
	// connect motion events to the ruler
	gtk_widget_add_events(OscGraph, GDK_POINTER_MOTION_MASK);
	gtk_signal_connect_object(GTK_OBJECT(OscGraph), "motion_notify_event",
	    (GtkSignalFunc)GTK_WIDGET_CLASS(GTK_WIDGET_GET_CLASS(OscRuler))->motion_notify_event,
	    GTK_OBJECT(OscRuler));
	gtk_widget_modify_style (Graph, rc_style);
	gtk_rc_style_unref (rc_style);
	events = (event_t *)malloc(sizeof(event_t) * 256); // 24576 -> 3 or 6 pages
	nevents = 256;
	current_event = 0;
}

oscilloscope_graph::~oscilloscope_graph(void)
{
	gtk_widget_destroy(Graph);
	gtk_widget_destroy(OscRuler);
	gtk_widget_destroy(OscGraph);
	free(events);
}

void
oscilloscope_graph::set_nodeVoltage( double v )
{
	if (current_event == nevents) {
		event_t *new_events;
		new_events = (event_t *)realloc(events, sizeof(event_t) * (nevents + 256));
		if (new_events == NULL)
			return;
		events = new_events;
		nevents = nevents + 256;
	}
	events[current_event].cycle = get_cycles().get();
	events[current_event].value = v;
	*max_cycle = get_cycles().get();
	current_event++;
	expose(FALSE);
}

void
oscilloscope_graph::realize(void)
{
	GdkColormap *colormap;
	colormap = gdk_window_get_colormap(OscGraph->window);
	GraphColor.red = 0xffff;
	GraphColor.green = 0xffff;
	GraphColor.blue = 0xffff;
	gdk_color_alloc(colormap, &GraphColor);
	expose(FALSE);
}

void
oscilloscope_graph::expose(bool clear)
{
	double ymin = events[0].value, ymax = events[0].value;
	int x0, y0, x1, y1;
	double xscale, yscale;
	int i;

	if (current_event == 0)
		return;
	if (clear)
		gdk_window_clear(Graph->window);

	for (i = 1; i < current_event; i++) {
		if (ymin > events[i].value)
			ymin = events[i].value;
		if (ymax < events[i].value)
			ymax = events[i].value;
	}
	GdkGC *gc = gdk_gc_new(Graph->window);
	gdk_gc_set_foreground(gc, &GraphColor);
	xscale = (double)(Graph->allocation.width - GRAPH_MARGIN * 2) /
	    (double)(*view_max_cycle - *view_min_cycle);
	yscale = (double)(Graph->allocation.height - GRAPH_MARGIN * 2) /
	    (ymax - ymin);

	gtk_ruler_set_range(GTK_RULER(OscRuler),
	    (((double)(ymax) * yscale) - GRAPH_MARGIN) / yscale,
	    (((double)(ymin) * yscale) + GRAPH_MARGIN) / yscale,
	    -1, ymax);

	x0 = GRAPH_MARGIN;
	y0 = Graph->allocation.height - GRAPH_MARGIN -
	    (int)((double)(events[0].value - ymin) * yscale);
	for (i = 1; i < current_event; i++) {
		y1 = Graph->allocation.height - GRAPH_MARGIN -
		    (int)((double)(events[i].value - ymin) * yscale);
		if (events[i].cycle >= *view_min_cycle) {
			if (events[i].cycle < *view_max_cycle) 
				x1 = (int)((double)(events[i].cycle -
				    *view_min_cycle) * xscale) + GRAPH_MARGIN;
			else
				x1 = Graph->allocation.width - GRAPH_MARGIN;
			gdk_draw_line(Graph->window, gc, x0, y0, x1, y0);
			if (events[i].cycle > *view_max_cycle)
				break;
			gdk_draw_line(Graph->window, gc, x1, y0, x1, y1);
			x0 = x1;
		}
		y0 = y1;
	}
	gdk_gc_destroy(gc);
}

gboolean
oscw_realize(GtkWidget *Widget, gpointer data)
{
	oscilloscope_graph *osc = (oscilloscope_graph *)data;

	osc->realize();
	return TRUE;
}

gboolean
oscw_expose(GtkWidget *Widget, GdkEvent *event, gpointer data)
{
	oscilloscope_graph *osc = (oscilloscope_graph *)data;

	osc->expose(FALSE);
	return TRUE;
}

/*
 * oscilloscope_window is a window containing the labels and SpinButtons,
 * the horizontal ruller, and one or more oscilloscope_graph
 */
class oscilloscope_window {
public:
	oscilloscope_window(int ng = 1, const char *opt_name=NULL);
	~oscilloscope_window(void);
        virtual void realize();
	virtual void expose();
        virtual void adj_min();
	virtual void adj_max();
	int nb_osc;
	oscilloscope_graph *osc[MAX_OSC_INPUT];

private:
	GtkWidget *Window;
	GtkWidget *w_table;
	GtkWidget *hruler;
	GtkWidget *spin_table;
	GtkWidget *min_label, *max_label;
	GtkWidget *min_spin, *max_spin;
	GtkObject *min_adj, *max_adj;
	guint64 max_cycle;
	guint64 view_min_cycle, view_max_cycle;
};

oscilloscope_window::oscilloscope_window(int ng, const char *opt_name)
{
	int i;

	view_min_cycle = 0;
	max_cycle = view_max_cycle = 1;
	if (ng < MAX_OSC_INPUT)
		nb_osc = ng;
	else
		nb_osc = MAX_OSC_INPUT;

	Window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(Window), opt_name);

	w_table = gtk_table_new(2, 2 + nb_osc, FALSE);

	hruler = gtk_hruler_new();
	gtk_widget_set_usize(hruler, 500, 30);
	gtk_table_attach(GTK_TABLE(w_table), hruler, 0, 1, 1, 2,
	    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
	    (GtkAttachOptions) (0),
	    5, 2);

	spin_table = gtk_table_new(4, 1, FALSE);
	min_label = gtk_label_new("min cycle: ");
	max_label = gtk_label_new("max cycle: ");
	gtk_table_attach_defaults(GTK_TABLE(spin_table), min_label, 0, 1, 0, 1);
	gtk_table_attach_defaults(GTK_TABLE(spin_table), max_label, 2, 3, 0, 1);


	min_adj = gtk_adjustment_new(0, 0, 0, 1, 1000, 1000);
	max_adj = gtk_adjustment_new(0, 0, 0, 1, 1000, 1000);
	min_spin = gtk_spin_button_new(GTK_ADJUSTMENT(min_adj), 1.0, 0);
	max_spin = gtk_spin_button_new(GTK_ADJUSTMENT(max_adj), 1.0, 0);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(min_spin), TRUE);
	gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(max_spin), TRUE);
	gtk_table_attach_defaults(GTK_TABLE(spin_table), min_spin, 1, 2, 0, 1);
	gtk_table_attach_defaults(GTK_TABLE(spin_table), max_spin, 3, 4, 0, 1);

	gtk_signal_connect(min_adj, "value_changed",
	    GTK_SIGNAL_FUNC(osc_gspin_min), this);
	gtk_signal_connect(max_adj, "value_changed",
	    GTK_SIGNAL_FUNC(osc_gspin_max), this);

	gtk_table_attach(GTK_TABLE(w_table), spin_table, 0, 2, 0, 1,
	    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
	    (GtkAttachOptions) (0),
	    5, 2);

	gtk_signal_connect(GTK_OBJECT(Window), "realize",
		GTK_SIGNAL_FUNC(osc_realize), this);
	gtk_signal_connect(GTK_OBJECT(Window), "expose_event",
		GTK_SIGNAL_FUNC(osc_expose), this);

	char pin_name[100];
	for (i = 0; i < nb_osc; i++) {
		snprintf(pin_name, sizeof(pin_name), "%s.%d", opt_name, i);
		osc[i] = new oscilloscope_graph(pin_name);
		osc[i]->max_cycle = &max_cycle;
		osc[i]->view_min_cycle = &view_min_cycle;
		osc[i]->view_max_cycle = &view_max_cycle;
		gtk_table_attach(GTK_TABLE(w_table), osc[i]->OscGraph,
		    0, 1, i + 2, i + 3,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    5, 5);
		gtk_table_attach(GTK_TABLE(w_table), osc[i]->OscRuler,
		    1, 2, i + 2, i + 3,
		    (GtkAttachOptions) (0),
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    5, 5);
		gtk_signal_connect_object(GTK_OBJECT(osc[i]->OscGraph),
		    "motion_notify_event",
		    (GtkSignalFunc)GTK_WIDGET_CLASS(GTK_WIDGET_GET_CLASS(hruler))->motion_notify_event, GTK_OBJECT(hruler));
	}
	gtk_container_add(GTK_CONTAINER(Window), w_table);
	gtk_widget_show_all(Window);

	view_min_cycle = 0;
	view_max_cycle = max_cycle;
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(max_spin), max_cycle); // XXX


}

oscilloscope_window::~oscilloscope_window()
{
	int i;
	for (i = 0; i < nb_osc; i++) {
		gtk_widget_unparent(osc[i]->OscGraph);
		gtk_widget_unparent(osc[i]->OscRuler);
		delete osc[i];
	}
	gtk_widget_destroy(min_spin);
	gtk_widget_destroy(max_spin);
	gtk_object_destroy(min_adj);
	gtk_object_destroy(max_adj);
	gtk_widget_destroy(min_label);
	gtk_widget_destroy(max_label);
	gtk_widget_destroy(spin_table);
	gtk_widget_destroy(hruler);
	gtk_widget_destroy(w_table);
	gtk_widget_destroy(Window);
}

void
oscilloscope_window::realize()
{
	expose();
}

void
oscilloscope_window::expose()
{
	double unit, xscale;
	
	GTK_ADJUSTMENT(min_adj)->upper = max_cycle;
	GTK_ADJUSTMENT(max_adj)->upper = max_cycle;
	gtk_adjustment_changed(GTK_ADJUSTMENT(min_adj));
	gtk_adjustment_changed(GTK_ADJUSTMENT(max_adj));

	if (view_max_cycle > 999999)
		unit = 1000000;
	else if (view_max_cycle > 999)
		unit = 1000;
	else
		unit = 1;

	xscale = (double)(hruler->allocation.width - GRAPH_MARGIN * 2) /
	    ((double)(view_max_cycle - view_min_cycle) / unit);
	gtk_ruler_set_range(GTK_RULER(hruler),
	    (double)view_min_cycle / unit - (GRAPH_MARGIN) / xscale,
	    (double)view_max_cycle / unit + (GRAPH_MARGIN) / xscale,
	    -1, 1);
}

void
oscilloscope_window::adj_min()
{
	int i;

	view_min_cycle = gtk_spin_button_get_value_as_int(
	    GTK_SPIN_BUTTON(min_spin));
	expose();
	for (i = 0; i < nb_osc; i++)
		osc[i]->expose(TRUE);
}

void
oscilloscope_window::adj_max()
{
	int i;

	view_max_cycle = gtk_spin_button_get_value_as_int(
	    GTK_SPIN_BUTTON(max_spin));
	expose();
	for (i = 0; i < nb_osc; i++)
		osc[i]->expose(TRUE);
}

gboolean
osc_realize(GtkWidget *Widget, gpointer data)
{
	oscilloscope_window *osc = (oscilloscope_window *)data;
	osc->realize();
	return TRUE;
}

gboolean
osc_expose(GtkWidget *Widget, GdkEvent *event, gpointer data)
{
	oscilloscope_window *osc = (oscilloscope_window *)data;
	osc->expose();
	return TRUE;
}

gboolean
osc_gspin_min(GtkAdjustment* adj, gpointer data) {
	oscilloscope_window *osc = (oscilloscope_window *)data;

	osc->adj_min();
	return TRUE;
}

gboolean
osc_gspin_max(GtkAdjustment* adj, gpointer data)
{
	oscilloscope_window *osc = (oscilloscope_window *)data;

	osc->adj_max();
	return TRUE;
}

void
oscilloscope_Interface::SimulationHasStopped(gpointer object)
{
}

oscilloscope::oscilloscope(void)
{
	name_str = strdup("oscilloscope");
	interface = new oscilloscope_Interface(this);
	get_interface().add_interface(interface);
}

oscilloscope::~oscilloscope(void)
{
	delete window;
}

void
oscilloscope::build_window(int ngr, const char *new_name)
{
	window = new oscilloscope_window(ngr, new_name);
}

void
oscilloscope::create_iopin_map(void)
{
	int i;

	create_pkg(window->nb_osc);
	for (i = 0; i < window->nb_osc; i++) {
		assign_pin(i + 1, window->osc[i]);
		get_symbol_table().add_stimulus(get_pin(i + 1));
	}
}

Module *
oscilloscope::construct(const char *_new_name=0)
{
	oscilloscope *osc = new oscilloscope;
	osc->new_name((char*)_new_name);

	osc->build_window(8, _new_name);
	osc->create_iopin_map();
	return osc;
}
