/*
 * Copyright (c) 2013 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <pthread.h>
#include <openssl/x509.h>

#define ENABLE_SSL
#include "openvpn-plugin.h"

#define DEFAULT_TIMEOUT	20
#define DEFAULT_DEPTH	2
#define DEFAULT_WARNING	5
#define DEFAULT_CRITICAL 3

#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3

#define TLS_ID_ "tls_id_"
#define TLS_NOTBEFORE_ "tls_notbefore_"
#define TLS_NOTAFTER_ "tls_notafter_"

struct ctx {
	int cur;
	int timeout;
	int depth;
	int warning;
	int critical;
	time_t *expire;
};

static void *
timeout_thread(arg)
	void *arg;
{
	struct ctx *ctx;

	ctx = (struct ctx *)arg;
	sleep(ctx->timeout);

	printf("CRITICAL timeout after %d seconds\n", ctx->timeout);

	exit(STATE_CRITICAL);
}

#if notanymore
static int
strmatch(s, match)
	char *s;
	char *match;
{
	return strncmp(s, match, strlen(match));
}
#endif

static time_t
date_strtots(strdate)
	char *strdate;
{
	struct tm tm;

	(void)strptime(strdate, "%y%m%d%H%M%SZ", &tm);
	return mktime(&tm);
}

static int
date_tstodays(ts)
	time_t ts;
{
	return (ts / 86400);
}

OPENVPN_EXPORT int
openvpn_plugin_open_v3(vers, in, out)
	const int vers;
	struct openvpn_plugin_args_open_in const *in;
	struct openvpn_plugin_args_open_return *out;
{
	struct ctx *ctx;
	pthread_t thread;
	int ac = 0;
	char **av = __UNCONST(in->argv);
	int i, ch;
	time_t now;

	if (vers != OPENVPN_PLUGINv3_STRUCTVER)
		return OPENVPN_PLUGIN_FUNC_ERROR;

	(void)time(&now);

	if ((ctx = malloc(sizeof(*ctx))) == NULL) {
		printf("UNKNOWN cannot allocate context: %s\n",
			strerror(errno));
		exit(STATE_UNKNOWN);
	}

	ctx->timeout = DEFAULT_TIMEOUT;
	ctx->depth = DEFAULT_DEPTH;
	ctx->warning = DEFAULT_WARNING;
	ctx->critical = DEFAULT_CRITICAL;

	for (i = 0; in->argv[i]; i++)
		ac++;

	while ((ch = getopt(ac, av, "w:c:d:t:")) != -1) {
		switch(ch) {
		case 't': 
			ctx->timeout = atoi(optarg);
			break;
		case 'd': 
			ctx->depth = atoi(optarg);
			break;
		case 'w': 
			ctx->warning = atoi(optarg);
			break;
		case 'c': 
			ctx->critical = atoi(optarg);
			break;
		default:
			printf("UNKNOWN usage: "
			       "[-t timeout] [-d depth] [-w days] [-c days]\n");
			exit(STATE_UNKNOWN);
			/* NOTREACHED */
		}
	}

	/* convert into seconds from Epoch */
	ctx->warning = now + (ctx->warning * 86400);
	ctx->critical = now + (ctx->critical * 86400);

	ctx->cur = ctx->depth;

	if ((ctx->expire = calloc(sizeof(*ctx->expire), ctx->depth)) == NULL) {
		printf("UNKNOWN cannot allocate output: %s\n",
			strerror(errno));
		exit(STATE_UNKNOWN);
	}

	/* Setup timeout thread */
	if (pthread_create(&thread, NULL, timeout_thread, ctx) != 0) {
		printf("UNKNOWN: cannot create timeout thread: %s\n",
		       strerror(errno));
		exit(STATE_UNKNOWN);
	}

	out->type_mask = OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_TLS_VERIFY);
	out->handle = (void *)ctx;

	return OPENVPN_PLUGIN_FUNC_SUCCESS;
}



OPENVPN_EXPORT int
openvpn_plugin_func_v3(vers, in, out)
	const int vers;
	struct openvpn_plugin_args_func_in const *in;
	struct openvpn_plugin_args_func_return *out;
{
	struct ctx *ctx = in->handle;
	time_t notbefore, notafter, now;
	int i;

	if (in->type != OPENVPN_PLUGIN_TLS_VERIFY)
		return OPENVPN_PLUGIN_FUNC_SUCCESS;

	ctx->cur--;

	if (in->current_cert_depth < ctx->cur) {
		printf("CRITICAL certificate chain too long "
		       "(%d, expected %d), adjust -d\n",
		       in->current_cert_depth + 1, ctx->depth);
		exit(STATE_CRITICAL);
	}

	if (in->current_cert_depth >= ctx->depth) {
		printf("CRITICAL certificate chain too short "
		       "(%d, expected %d), adjust -d\n",
		       in->current_cert_depth + 1, ctx->depth);
		exit(STATE_CRITICAL);
	}

	/* A loop mean we have an expired certificate */
	if (ctx->cur != in->current_cert_depth) {
		printf("CRITICAL certificate %d is expired\n",
			in->current_cert_depth);
		exit(STATE_CRITICAL);
	}

	ctx->cur = in->current_cert_depth;

	(void)time(&now);

	notbefore = date_strtots(X509_get_notBefore(in->current_cert)->data);
	if (notbefore == -1) {
		printf("UNKNOWN unexpected date \"%s\"\n", 
		       X509_get_notBefore(in->current_cert)->data);
		exit(STATE_UNKNOWN);
	}

	if (now < notbefore) {
		printf("CRITICAL cert %d will be valid in %d days\n",
		       ctx->cur, date_tstodays(notbefore - now));
		exit(STATE_CRITICAL);
	}


	notafter = date_strtots(X509_get_notAfter(in->current_cert)->data);
	if (notafter == -1) {
		printf("UNKNOWN unexpected date \"%s\"\n",
		       X509_get_notAfter(in->current_cert)->data);
		exit(STATE_UNKNOWN);
	}

	if (now > notafter) {
		printf("CRITICAL cert %d expired %d days ago\n", 
		       ctx->cur, date_tstodays(now - notafter));
		exit(STATE_CRITICAL);
	}

	if (ctx->critical > notafter) {
		printf("CRITICAL cert %d expires in %d days\n",
		       ctx->cur, date_tstodays(notafter - now));
		exit(STATE_CRITICAL);
	}
		
	if (ctx->warning > notafter) {
		printf("WARNING cert %d expires in %d days\n", 
		       ctx->cur, date_tstodays(notafter - now));
		exit(STATE_WARNING);
	}

	ctx->expire[ctx->cur] = notafter - now;

	if (ctx->cur != 0)
		return OPENVPN_PLUGIN_FUNC_SUCCESS;

	printf("OK Expires in");
	for (i = 0; i < ctx->depth; i++)
		printf("%s%d", (i == 0) ? " " : "/",
			date_tstodays(ctx->expire[i])); 
	printf(" days\n");

	exit(STATE_OK);

	/* NOTREACHED */
	return OPENVPN_PLUGIN_FUNC_SUCCESS;
}

/* ARGSUSED0 */
OPENVPN_EXPORT void
openvpn_plugin_close_v1(oph)
	openvpn_plugin_handle_t oph;
{
	printf("CRITICAL: cannot open connexion\n");
	exit(STATE_CRITICAL);	

	/* NOTREACHED */
	return;
}
