/*
 * Copyright (C) 1997-2004, R3vis Corporation.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA, or visit http://www.gnu.org/copyleft/lgpl.html.
 *
 * Original Contributor:
 *   Wes Bethel, R3vis Corporation, Marin County, California
 * Additional Contributor(s):
 *
 * The OpenRM project is located at http://openrm.sourceforge.net/.
 */
/*
 * $Id: rmomesh.c,v 1.4 2004/01/17 00:59:59 wes Exp $
 * Version: $Name: OpenRM-1-5-2-RC3 $
 * $Revision: 1.4 $
 * $Log: rmomesh.c,v $
 * Revision 1.4  2004/01/17 00:59:59  wes
 * Updated copyright line for 2004.
 * Modified rendering code to include a small, texture-size-dependent offset
 * to the texture coords to fix rendering bugs visible when using
 * GL_LINEAR texturing modes on multiple octmeshes located adjacent
 * to one another in space.
 *
 * Revision 1.3  2003/10/03 19:22:05  wes
 * Minor tweakage to automatically generated texture coordinates to avoid
 * rendering artifacts that appear on some platforms in conjunction with
 * using the GL_LINEAR texture environment setting.
 *
 * Revision 1.2  2003/02/02 02:07:15  wes
 * Updated copyright to 2003.
 *
 * Revision 1.1.1.1  2003/01/28 02:15:23  wes
 * Manual rebuild of rm150 repository.
 *
 * Revision 1.8  2003/01/24 17:11:20  wes
 * Test - remove OGL state manip from within rmOctmesh() draw routine in order
 * to avoid some bugs induced when used with CR. Long term - the use of state
 * manip needs to be addressed inside this draw routine.
 *
 * Revision 1.7  2003/01/16 22:21:17  wes
 * Updated all source files to reflect new organization of header files:
 * all header files formerly located in include/rmaux, include/rmi, include/rmv
 * are now located in include/rm.
 *
 * Revision 1.6  2002/06/02 15:16:27  wes
 * Added RMstateCache code to help eliminate the number of state changes
 * made during the render traversal. The RMstateCache tracks
 * the actual OpenGL rendering state w/o the need for querying OpenGL
 * directly, and  is queried by draw code that then decides if any
 * real state changes are required given the configuration of data
* within an RMprimitive.
 *
 * Revision 1.5  2002/05/12 15:25:17  wes
 *
 * Changed how number of slices is computed. Previously, the number of
 * slices was computed to be the minimum of grid dimensions in each
 * axis, divided by the model flag. Changed so nslices is now axis-
 * dependent, divided by the model flag. The result is more visually pleasing.
 *
 * Revision 1.4  2002/04/30 19:33:05  wes
 * Updated copyright dates.
 *
 * Revision 1.3  2001/03/31 17:12:39  wes
 * v1.4.0-alpha-2 checkin.
 *
 * Revision 1.2  2000/04/20 16:29:47  wes
 * Documentation additions/enhancements, some code rearragement.
 *
 * Revision 1.1.1.1  2000/02/28 21:29:40  wes
 * OpenRM 1.2 Checkin
 *
 * Revision 1.1.1.1  2000/02/28 17:18:48  wes
 * Initial entry - pre-RM120 release, source base for OpenRM 1.2.
 *
 */

#include <rm/rm.h>
#include "rmprivat.h"

/*
 * this file contains the "draw routine" for the octmesh primitive.
 * for all practical purposes, there are no routines in this file
 * callable from the application level.
 *
 * rmOctmesh() is the entry point for the octmesh draw routine.
 */

#define X_AXIS 0
#define Y_AXIS 1
#define Z_AXIS 2


/* PRIVATE */
static void
scan_xaxis (RMprimitive *p,
	    float dot,
	    int nslices,
	    float *v,
	    int vstride,
	    int *dims,
	    int *tdims)
{
    int    i;
    int    stepsize = 2;
    int    ix, ixincr;
    int    kmin, kmax, jmin, jmax;
    float *x, *y, *z;
    float  txstart, txincr;

    /* tc's are in (u,w) order (1,0), (0,0), (0,1), (1,1) */
    RMvertex3D tc[4] = {{0.0, 0.0, 1.0}, {0.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 1.0, 1.0}}; 
    RMvertex3D lv[4];
    RMvertex3D n;
    float yEps, zEps;
    
    /*
     * Jan 2004 - compute a slight offset added to texture coordinates
     * in the plane facing the viewer. This is needed to fix some "cracks"
     * that are visible when rendering multiple octmeshes that are
     * located next to one another and when using the GL_LINEAR texture
     * filter mode.
     */

    yEps = -0.5/(float)(tdims[1]);
    zEps = -0.5/(float)(tdims[2]);
    
    tc[0].y -= yEps;
    tc[0].z += zEps;
    tc[1].y -= yEps;
    tc[1].z -= zEps;
    tc[2].y += yEps;
    tc[2].z -= zEps;
    tc[3].y += yEps;
    tc[3].z += zEps;
    
    kmin = jmin = 0;
    kmax = dims[2] - 1;
    jmax = dims[1] - 1;

    stepsize = dims[0] / nslices;

    if (dot < 0)
    {
	ix = dims[0] - 1;
	ixincr = -stepsize;
	txstart = 1.0;
	txincr = -(float)(stepsize) / (float)(dims[0] - 1);
    }
    else
    {
	ix = 0;
	ixincr = stepsize;
	txstart = 0.0;
	txincr = (float)(stepsize) / (float)(dims[0] - 1);
    }
	
    x = v;
    y = v + dims[0];
    z = y + dims[1];

    lv[0].y = y[jmin];
    lv[0].z = z[kmax];
    
    lv[1].y = y[jmin];
    lv[1].z = z[kmin];
    
    lv[2].y = y[jmax];
    lv[2].z = z[kmin];
    
    lv[3].y = y[jmax];
    lv[3].z = z[kmax];

    n.x = 1.0;
    n.y = n.z = 0.0;

    for (i = 0; i < dims[0]; i+= stepsize, ix+= ixincr, txstart += txincr) 
    {
	lv[0].x = x[ix];
	lv[1].x = x[ix];
	lv[2].x = x[ix];
	lv[3].x = x[ix];

	tc[0].x = txstart;
	tc[1].x = txstart;
	tc[2].x = txstart;
	tc[3].x = txstart;
    
	glBegin(GL_QUADS);

	glNormal3fv((float *)&n);
	glTexCoord3fv((float *)tc);
	glVertex3fv((float *)lv);
    
	glTexCoord3fv((float *)(tc + 1));
	glVertex3fv((float *)(lv + 1));
    
	glTexCoord3fv((float *)(tc + 2));
	glVertex3fv((float *)(lv + 2));
    
	glTexCoord3fv((float *)(tc + 3));
	glVertex3fv((float *)(lv + 3));

	glEnd();
    }
}


/* PRIVATE */
static void
scan_yaxis (RMprimitive *p,
	    float dot,
	    int nslices,
	    float *v,
	    int vstride,
	    int *dims,
	    int *tdims)
{
    int    i;
    int    stepsize;
    int    iy, iyincr;
    int    kmin, kmax, imin, imax;
    float *x, *y, *z;
    float  tystart, tyincr;
    float xEps, zEps;
    
    /* tc's are in (u,w) order (0,0), (1,0), (1,1), (0,1) */
    RMvertex3D tc[4] = {{0.0, 0.0, 1.0}, {1.0, 0.0, 1.0}, {1.0, 0.0, 0.0}, {0.0, 0.0, 0.0}};
    RMvertex3D lv[4];
    RMvertex3D n;
    
    /*
     * Jan 2004 - compute a slight offset added to texture coordinates
     * in the plane facing the viewer. This is needed to fix some "cracks"
     * that are visible when rendering multiple octmeshes that are
     * located next to one another and when using the GL_LINEAR texture
     * filter mode.
     */
    xEps = -0.5/(float)(tdims[0]);
    zEps = -0.5/(float)(tdims[2]);
    
    tc[0].x -= xEps;
    tc[0].z += zEps; 

    tc[1].x += xEps;
    tc[1].z += zEps; 
    
    tc[2].x += xEps;
    tc[2].z -= zEps; 
    
    tc[3].x -= xEps;
    tc[3].z -= zEps; 

    imin = kmin = 0;
    kmax = dims[2] - 1;
    imax = dims[0] - 1;

    stepsize = dims[1] / nslices;
    if (stepsize == 0)
        stepsize = 1;

    if (dot > 0)
    {
	iy = dims[1] - 1;
	iyincr = -stepsize;
	tystart = 1.0;
	tyincr = -(float)(stepsize) / (float)(dims[1] - 1);
    }
    else
    {
	iy = 0;
	iyincr = stepsize;
	tystart = 0.0;
	tyincr = (float)(stepsize) / (float)(dims[1] - 1);
    }
	
    x = v;
    y = v + dims[0];
    z = y + dims[1];

    lv[0].x = x[imin];
    lv[0].z = z[kmax];
    
    lv[1].x = x[imax];
    lv[1].z = z[kmax];
    
    lv[2].x = x[imax];
    lv[2].z = z[kmin];
    
    lv[3].x = x[imin];
    lv[3].z = z[kmin];

    n.y = 1.0;
    n.x = n.z = 0.0;

    for (i = 0; i < dims[1]; i+= stepsize, iy += iyincr, tystart += tyincr)
    {
	lv[0].y = y[iy];
	lv[1].y = y[iy];
	lv[2].y = y[iy];
	lv[3].y = y[iy];

	tc[0].y = tystart;
	tc[1].y = tystart;
	tc[2].y = tystart;
	tc[3].y = tystart;
#if 0	
	tc[0].y = (float)i / (float)(p->vsize - 1);
	tc[1].y = (float)i / (float) (p->vsize - 1);
	tc[2].y = (float)i / (float)(p->vsize - 1);
	tc[3].y = (float)i / (float)(p->vsize - 1);
#endif    
	glBegin(GL_QUADS);

	glNormal3fv((float *)&n);
	glTexCoord3fv((float *)tc);
	glVertex3fv((float *)lv);
    
	glTexCoord3fv((float *)(tc + 1));
	glVertex3fv((float *)(lv + 1));
    
	glTexCoord3fv((float *)(tc + 2));
	glVertex3fv((float *)(lv + 2));
    
	glTexCoord3fv((float *)(tc + 3));
	glVertex3fv((float *)(lv + 3));

	glEnd();
    }
}


/* PRIVATE */
static void
scan_zaxis (RMprimitive *p,
	    float dot,
	    int nslices,
	    float *v,
	    int vstride,
	    int *dims,
	    int *tdims)
{
    int    i;
    int    stepsize;
    int    iz, izincr;
    int    imin, imax, jmin, jmax;
    float *x, *y, *z;
    float  tzstart, tzincr;

    /* tc's are in (u,w) order (0,0), (1,0), (1,1), (0,1) */
    RMvertex3D tc[4] = {{0.0, 0.0, 0.5}, {1.0, 0.0, 0.5}, {1.0, 1.0, 0.5}, {0.0, 1.0, 0.5}}; 
    RMvertex3D lv[4];
    RMvertex3D n;
    float xEps, yEps;
    
    imin = jmin = 0;
    imax = dims[0];
    jmax = dims[1];
    stepsize = dims[2] / nslices;

    /*
     * Jan 2004 - compute a slight offset added to texture coordinates
     * in the plane facing the viewer. This is needed to fix some "cracks"
     * that are visible when rendering multiple octmeshes that are
     * located next to one another and when using the GL_LINEAR texture
     * filter mode.
     */
    xEps = -0.5/(float)(tdims[0]);
    yEps = -0.5/(float)(tdims[1]);
    tc[0].x -= xEps;
    tc[0].y -= yEps;
    tc[1].x += xEps;
    tc[1].y -= yEps;
    tc[2].x += xEps;
    tc[2].y += yEps;
    tc[3].x -= xEps;
    tc[3].y += yEps;

    if (dot < 0)
    {
	iz = dims[2] - 1;
	izincr = -stepsize;
	tzstart = 1.0;
	tzincr = -(float)stepsize / (float)(dims[2] - 1); 
	
    }
    else
    {
	iz = 0;
	izincr = stepsize;
	tzstart = 0.;
	tzincr = (float)stepsize / (float)(dims[2] - 1); 
	
    }
	
    x = v;
    y = v + dims[0];
    z = y + dims[1];

    lv[0].x = x[imin];
    lv[0].y = y[jmin];
    
    lv[1].x = x[imax - 1];
    lv[1].y = y[jmin];
    
    lv[2].x = x[imax - 1];
    lv[2].y = y[jmax - 1];
    
    lv[3].x = x[imin];
    lv[3].y = y[jmax - 1];

    n.x = n.y = 0.0;
    n.z = 1.0;

    for (i = 0; i < dims[2]; i+= stepsize, iz += izincr, tzstart += tzincr)
    {
	lv[0].z = z[iz];
	lv[1].z = z[iz];
	lv[2].z = z[iz];
	lv[3].z = z[iz];
#if 0
	tc[0].z = (float)i / (float)(p->wsize - 1);
	tc[1].z = (float)i / (float)(p->wsize - 1);
	tc[2].z = (float)i / (float)(p->wsize - 1);
	tc[3].z = (float)i / (float)(p->wsize - 1);
#endif    
	tc[0].z = tzstart;
	tc[1].z = tzstart;
	tc[2].z = tzstart;
	tc[3].z = tzstart;
    
	glBegin(GL_QUADS);

	glNormal3fv((float *)&n);
	glTexCoord3fv((float *)tc);
	glVertex3fv((float *)lv);
    
	glTexCoord3fv((float *)(tc + 1));
	glVertex3fv((float *)(lv + 1));
    
	glTexCoord3fv((float *)(tc + 2));
	glVertex3fv((float *)(lv + 2));
    
	glTexCoord3fv((float *)(tc + 3));
	glVertex3fv((float *)(lv + 3));

	glEnd();
    }
}


/* PRIVATE */
void
rmOctmesh  OGLPRIMPARMLIST() 
{
    int                  i, best_axis;
    int                  nslices;
    int                  divisor;
    int                  nverts, vstride; 	/* for verts */
    int                  sstride;		/* for config */
    int                 *dims;
    float                dot;
    float               *v;
    RMprimitiveDataBlob *vblob;			/* for verts */
    RMprimitiveDataBlob *sblob; 		/* for config */
    int                  textureDims[3];

    /* figure out which axis is most perpindicular to the line-of-sight */
    {
	RMmatrix tmp;
	float    tv[3][4] = {{1.0, 0.0, 0.0, 0.0}, {0.0, 1.0, 0.0, 0.0},{0.0, 0.0, 1.0, 0.0}};
	double   x, y, z;
		       
	rmMatrixMultiply(&(s->modelView), &(s->projection), &tmp);
	
	for (i = 0; i < 3; i++)
	{
	    rmPoint4MatrixTransform(tv[i], &tmp, tv[i]);
	    rmVertex3DNormalize((RMvertex3D *)tv[i]);
	}

	x = fabs(tv[0][2]);
	y = fabs(tv[1][2]);
	z = fabs(tv[2][2]);

	if (x >= y)
	{
	    if (x >= z)
	    {
#if (DEBUG_LEVEL & DEBUG_TRACE)
		fprintf(stderr, " X axis %g \n", tv[0][2]);
#endif
		best_axis = X_AXIS;
		dot = x;
		dot = -tv[0][2];
	    }
	    else
	    {
#if (DEBUG_LEVEL & DEBUG_TRACE)
		fprintf(stderr, " Z axis %g \n", tv[2][2]);
#endif
		best_axis = Z_AXIS;
		dot = z;
		dot = -tv[2][2];
	    }
	}
	else
	{
	    if (y >= z)
	    {
#if (DEBUG_LEVEL & DEBUG_TRACE)
		fprintf(stderr, " Y axis %g \n", tv[1][2]);
#endif
		best_axis = Y_AXIS;
		dot = y;
		dot = tv[1][2];
	    }
	    else
	    {
#if (DEBUG_LEVEL & DEBUG_TRACE)
		fprintf(stderr, " Z axis %g \n", tv[2][2]);
#endif
		best_axis = Z_AXIS;
		dot = z;
		dot = -tv[2][2];
	    }
	}
    }

    /* how big is the texture? */
    {
	if (s->texture->images[0] == NULL)
	    rmError(" rmOctmesh - big trouble, the texture doesn't have any image data!! Expect a segfault soon.");
	rmImageGetImageSize(s->texture->images[0], NULL, textureDims+0, textureDims+1, textureDims+2, NULL, NULL);
    }

    /* get the size of the octmesh */
    sblob = private_rmBlobFromIndex(p, BLOB_OMESHDIMS_INDEX);
    sstride = private_rmBlobGetStride(sblob) / sizeof(int);
    dims = (int *)private_rmBlobGetData(sblob);

    /* get the grid for the octmesh - we're assuming always a min/max grid (10/11/98) */
    vblob = private_rmBlobFromIndex(p, BLOB_OMESH_RECTGRID_INDEX);
    vstride = private_rmBlobGetStride(vblob) / sizeof(float);
    nverts = private_rmBlobGetNthings(vblob);
    v = (float *)private_rmBlobGetData(vblob);

    divisor = private_rmPrimitiveGetModelFlag(p);
    if (divisor == 0)		/* an RM error of some type */
    {
	rmError("rmOctmesh error: the model flag for the octmesh returned a zero.");
	divisor = 2;		/* pick something */
    }

    switch (best_axis)
       {
       case X_AXIS:
	   nslices = dims[0]/divisor;
	   if (nslices == 0)
	       nslices = 1;
	   scan_xaxis(p, dot, nslices, v, vstride, dims, textureDims);
	   break;
	  
       case Y_AXIS:
	   nslices = dims[1]/divisor;
	   if (nslices == 0)
	       nslices = 1;
	   scan_yaxis(p, dot, nslices, v, vstride, dims, textureDims);
	  break;
	  
       case Z_AXIS:
	   nslices = dims[2]/divisor;
	   if (nslices == 0)
	       nslices = 1;
	   scan_zaxis(p, dot, nslices, v, vstride, dims, textureDims);
	   break;
	  
       default: /* bogus axis enum */
	  break;
       }
#if 0
    private_rmEndOglSurfPropFunc();
#endif
}
/* EOF */
