/*
 * Speex Plugin codec for OpenH323/OPAL
 *
 * Copyright (C) 2004 Post Increment, All Rights Reserved
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Post Increment
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: speexcodec.cxx,v $
 * Revision 1.12  2004/06/16 05:32:52  csoutheren
 * Removed non-working codecs
 *
 * Revision 1.11  2004/06/16 04:35:50  csoutheren
 * Cleanup and add extra tables
 *
 * Revision 1.10  2004/06/16 04:21:05  csoutheren
 * Added rest of supported Speex codecs
 *
 * Revision 1.9  2004/06/16 03:57:58  csoutheren
 * Fixed strings
 *
 * Revision 1.8  2004/06/16 03:33:03  csoutheren
 * Initial support for ACM Speex
 *
 * Revision 1.7  2004/05/13 10:35:09  csoutheren
 * Fixed bit rates and bytes per frame for each codec
 *
 * Revision 1.6  2004/05/12 12:32:44  csoutheren
 * More name fixes
 *
 * Revision 1.5  2004/05/12 11:27:04  csoutheren
 * Changed codec name to match embedded codec
 *
 * Revision 1.4  2004/04/09 12:24:18  csoutheren
 * Renamed h323plugin.h to opalplugin.h, and modified everything else
 * as required
 *
 * Revision 1.3  2004/04/04 12:57:49  csoutheren
 * Updated Makefiles and fixed Linux problems
 *
 * Revision 1.2  2004/04/04 12:44:18  csoutheren
 * Added file headers
 *
 */

#include <opalplugin.h>

extern "C" {
PLUGIN_CODEC_IMPLEMENT("Speex")
};

#include <malloc.h>
#include <string.h>

#include "libspeex/speex.h" 

#define SAMPLES_PER_FRAME        160

const float MaxSampleValue   = 32767.0;
const float MinSampleValue   = -32767.0;

struct PluginSpeexContext {
  struct SpeexBits * bits;
  void      * coderState;
  unsigned  encoderFrameSize;
};

/////////////////////////////////////////////////////////////////////////////

static void * create_encoder(const struct PluginCodec_Definition * codec)
{
  int mode = (int)(codec->userData);

  struct PluginSpeexContext * context = new PluginSpeexContext;
  context->bits = new SpeexBits;
  speex_bits_init(context->bits);

  context->coderState = speex_encoder_init(&speex_nb_mode);
  speex_encoder_ctl(context->coderState, SPEEX_GET_FRAME_SIZE, &context->encoderFrameSize);
  speex_encoder_ctl(context->coderState, SPEEX_SET_QUALITY,    &mode);

  return context;
}

static int codec_encoder(const struct PluginCodec_Definition * codec, 
                                           void * _context,
                                     const void * from, 
                                       unsigned * fromLen,
                                           void * to,         
                                       unsigned * toLen,
                                   unsigned int * flag)
{
  PluginSpeexContext * context = (PluginSpeexContext *)_context;

  if (*fromLen != SAMPLES_PER_FRAME*2)
    return 0;

  short * sampleBuffer = (short *)from;

  // convert PCM to float
  float floatData[SAMPLES_PER_FRAME];
  int i;
  for (i = 0; i < SAMPLES_PER_FRAME; i++)
    floatData[i] = sampleBuffer[i];

  // encode PCM data in sampleBuffer to buffer
  speex_bits_reset(context->bits); 
  speex_encode(context->coderState, floatData, context->bits); 

  *toLen = speex_bits_write(context->bits, (char *)to, context->encoderFrameSize); 

  return 1; 
}

static void destroy_encoder(const struct PluginCodec_Definition * codec, void * _context)
{
  PluginSpeexContext * context = (PluginSpeexContext *)_context;

  speex_bits_destroy(context->bits);
  free(context->bits);

  speex_encoder_destroy(context->coderState); 
  free(context);
}

static void * create_decoder(const struct PluginCodec_Definition * codec)
{
  int tmp = 1;

  PluginSpeexContext * context = new PluginSpeexContext;
  context->bits = new SpeexBits;
  speex_bits_init(context->bits);

  context->coderState = speex_decoder_init(&speex_nb_mode);
  speex_decoder_ctl(context->coderState, SPEEX_SET_ENH, &tmp);

  return context;
}

static int codec_decoder(const struct PluginCodec_Definition * codec, 
                                           void * _context,
                                     const void * from, 
                                       unsigned * fromLen,
                                           void * to,         
                                       unsigned * toLen,
                                   unsigned int * flag)
{
  struct PluginSpeexContext * context = (struct PluginSpeexContext *)_context;

  float floatData[SAMPLES_PER_FRAME];
  short * sampleBuffer = (short *)to;

  // decode Speex data to floats
  speex_bits_read_from(context->bits, (char *)from, *fromLen); 
  speex_decode(context->coderState, context->bits, floatData); 

  // convert float to PCM
  int i;
  for (i = 0; i < SAMPLES_PER_FRAME; i++) {
    float sample = floatData[i];
    if (sample < MinSampleValue)
      sample = MinSampleValue;
    else if (sample > MaxSampleValue)
      sample = MaxSampleValue;
    sampleBuffer[i] = (short)sample;
  }


  return 1;
}

static void destroy_decoder(const struct PluginCodec_Definition * codec, void * _context)
{
  struct PluginSpeexContext * context = (struct PluginSpeexContext *)_context;

  speex_bits_destroy(context->bits);
  free(context->bits);

  speex_decoder_destroy(context->coderState); 
  free(context);
}

/////////////////////////////////////////////////////////////////////////////

static struct PluginCodec_information licenseInfo = {
  //1081075346,                          // Sun 04 Apr 2004 10:42:26 AM UTC
  1087351735,                            // Wed 16 Jun 2004 02:08:55 AM UTC

  "Craig Southeren, Post Increment",                           // source code author
  "2.0",                                                       // source code version
  "craigs@postincrement.com",                                  // source code email
  "http://www.postincrement.com",                              // source code URL
  "Copyright (C) 2004 by Post Increment, All Rights Reserved", // source code copyright
  "MPL 1.0",                                                   // source code license
  PluginCodec_License_MPL,                                     // source code license

  "Speex",                                                     // codec description
  "Jean-Marc Valin, Xiph Foundation.",                         // codec author
  "1.0.3",                                                     // codec version
  "jean-marc.valin@hermes.usherb.ca",                          // codec email
  "http://www.speex.org",                                      // codec URL
  "(C) 2003 Xiph.Org Foundation, All Rights Reserved",         // codec copyright information
  "Xiph BSD license",                                          // codec license
  PluginCodec_License_BSD                                      // codec license code
};

static const char L16Desc[]  = { "L16" };

static const char sdpSpeex[]  = { "speex" };

#define	EQUIVALENCE_COUNTRY_CODE            9
#define	EQUIVALENCE_EXTENSION_CODE          0
#define	EQUIVALENCE_MANUFACTURER_CODE       61

#define	MICROSOFT_COUNTRY_CODE	            181
#define	MICROSOFT_T35EXTENSION	            0
#define	MICROSOFT_MANUFACTURER	            21324

static const unsigned char speexWNarrow5k95Hdr[] =  { 
 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x05, 0x00, 0x00, 0x01, 0x05, 0x01, 0x00, 0x04, 0x00,
 0x00, 0x00, 0x00, 0x00, 0x09, 0xa1, 0x01, 0x00, 0x40, 0x1f, 0x00, 0x00, 0xe7, 0x02, 0x00, 0x00,
 0x77, 0x00, 0x10, 0x00, 0x0e, 0x00, 0x00, 0x01, 0x08, 0x00, 0x02, 0x00, 0x00, 0x00, 0xf4, 0x01,
 0x03, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char speexWNarrow8kHdr[] = 
{
  0x02, 0x00, 0x00, 0x00, 
  0x00, 0x00, 0xa0, 0x00,  
  0x00, 0x00, 0xa0, 0x00, 
  0x04, 0x00, 0x10, 0x00,   

  0x00, 0x00, 0x00, 0x00, 

  // WAVEFORMATEX
  0x09, 0xa1,                //    WORD    wFormatTag;        format type 
  0x01, 0x00,                //    WORD    nChannels;         number of channels (i.e. mono, stereo...)  
  0x40, 0x1f, 0x00, 0x00,    //    DWORD   nSamplesPerSec;    sample rate   
  0xe8, 0x03, 0x00, 0x00,    //    DWORD   nAvgBytesPerSec;   for buffer estimation 

  0x14, 0x00,                //    WORD    nBlockAlign;       block size of data  
  0x10, 0x00,                //    WORD    wBitsPerSample;    Number of bits per sample of mono data 
  0x0e, 0x00,                //    WORD    cbSize;            The count in bytes of the size of extra information  

  // Speex additional information
  0x00, 0x01,                // SPEEXWAVEFORMAT_VERSION
  0x01, 0x00,                // nFramesPerBlock
  0x03, 0x00,                // nQuality
  0x00, 0x00,                // nMode
  0xf4, 0x01,                // nVbrQuality

  0x03, 0x00,                // nComplexity
  0x00, 0x00,                // nFlags

  // unknown
  0x00, 0x00
};

static const unsigned char speexWNarrow11kHdr[] =   { 
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x40, 0x01, 0x02, 0x00, 0x08, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x09, 0xa1, 0x01, 0x00, 0x40, 0x1f, 0x00, 0x00, 0x5f, 0x05, 0x00, 0x00,
  0x37, 0x00, 0x10, 0x00, 0x0e, 0x00, 0x00, 0x01, 0x02, 0x00, 0x05, 0x00, 0x00, 0x00, 0xf4, 0x01,
  0x03, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char speexWNarrow15kHdr[] =   {
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x40, 0x01, 0x02, 0x00, 0x08, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x09, 0xa1, 0x01, 0x00, 0x40, 0x1f, 0x00, 0x00, 0x53, 0x07, 0x00, 0x00,
  0x4b, 0x00, 0x10, 0x00, 0x0e, 0x00, 0x00, 0x01, 0x02, 0x00, 0x07, 0x00, 0x00, 0x00, 0xf4, 0x01,
  0x03, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char speexWNarrow18k2Hdr[] =  {
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x40, 0x01, 0x02, 0x00, 0x08, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x09, 0xa1, 0x01, 0x00, 0x40, 0x1f, 0x00, 0x00, 0xe3, 0x08, 0x00, 0x00,
  0x5b, 0x00, 0x10, 0x00, 0x0e, 0x00, 0x00, 0x01, 0x02, 0x00, 0x09, 0x00, 0x00, 0x00, 0xf4, 0x01,
  0x03, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char speexWNarrow24k6Hdr[] =  {
  0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x40, 0x01, 0x02, 0x00, 0x08, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x09, 0xa1, 0x01, 0x00, 0x40, 0x1f, 0x00, 0x00, 0x03, 0x0c, 0x00, 0x00,
  0x7b, 0x00, 0x10, 0x00, 0x0e, 0x00, 0x00, 0x01, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00, 0xf4, 0x01,
  0x03, 0x00, 0x00, 0x00, 0x00, 0x00
};

#define CREATE_SPEEX_CAP_DATA(desc, suffix, ordinal) \
static const char speex##suffix[] = "Speex" #desc; \
static const char speex##suffix##Str[] = "Speex bs4 Narrow" #ordinal; \
static struct PluginCodec_H323NonStandardCodecData speex##suffix##Cap = \
{ \
  NULL, \
  EQUIVALENCE_COUNTRY_CODE, EQUIVALENCE_EXTENSION_CODE, EQUIVALENCE_MANUFACTURER_CODE, \
  (const unsigned char *)speex##suffix##Str, sizeof(speex##suffix##Str)-1, \
  NULL \
}; \
static const char speexW##suffix[] = "SpeexW" #desc; \
static struct PluginCodec_H323NonStandardCodecData speexW##suffix##Cap = \
{ \
  NULL, \
  MICROSOFT_COUNTRY_CODE, MICROSOFT_T35EXTENSION, MICROSOFT_MANUFACTURER, \
  speexW##suffix##Hdr, sizeof(speexW##suffix##Hdr), \
  NULL \
}; \

#define DECLARE_SPEEX_CODEC(suffix, ordinal) \
/* Original OpenH323 capability */ \
{ \
  /* encoder */ \
  PLUGIN_CODEC_VERSION,               /* codec API version */ \
  &licenseInfo,                       /* license information */ \
  PluginCodec_MediaTypeAudio |        /* audio codec */ \
  PluginCodec_InputTypeRaw |          /* raw input data */ \
  PluginCodec_OutputTypeRaw |         /* raw output data */ \
  PluginCodec_RTPTypeDynamic,         /* dynamic RTP type */ \
  speex##suffix,                      /* text decription */ \
  L16Desc,                            /* source format */ \
  speex##suffix,                      /* destination format */ \
  (void *)ordinal,                    /* user data */ \
  8000,                               /* samples per second */ \
  speexInfo[ordinal-2][0],            /* raw bits per second */ \
  20000,                              /* nanoseconds per frame */ \
  SAMPLES_PER_FRAME,                  /* samples per frame */ \
  speexInfo[ordinal-2][1],            /* bytes per frame */ \
  1,                                  /* recommended number of frames per packet */ \
  1,                                  /* maximum number of frames per packet  */ \
  0,                                  /* IANA RTP payload code */ \
  sdpSpeex,                           /* RTP payload name */ \
  create_encoder,                     /* create codec function */ \
  destroy_encoder,                    /* destroy codec */ \
  codec_encoder,                      /* encode/decode */ \
  NULL,                               /* codec controls */ \
  PluginCodec_H323Codec_nonStandard,  /* h323CapabilityType */ \
  &speex##suffix##Cap                 /* h323CapabilityData */ \
}, \
{  \
  /* decoder */ \
  PLUGIN_CODEC_VERSION,               /* codec API version */ \
  &licenseInfo,                       /* license information */ \
  PluginCodec_MediaTypeAudio |        /* audio codec */ \
  PluginCodec_InputTypeRaw |          /* raw input data */ \
  PluginCodec_OutputTypeRaw |         /* raw output data */ \
  PluginCodec_RTPTypeDynamic,         /* dynamic RTP type */ \
  speex##suffix,                      /* text decription */ \
  speex##suffix,                      /* source format */ \
  L16Desc,                            /* destination format */ \
  (void *)ordinal,                    /* user data */ \
  8000,                               /* samples per second */ \
  speexInfo[ordinal-2][0],            /* raw bits per second */ \
  20000,                              /* nanoseconds per frame */ \
  SAMPLES_PER_FRAME,                  /* samples per frame */ \
  speexInfo[ordinal-2][1],            /* bytes per frame */ \
  1,                                  /* recommended number of frames per packet */ \
  1,                                  /* maximum number of frames per packet */ \
  0,                                  /* IANA RTP payload code */ \
  sdpSpeex,                           /* RTP payload name */ \
  create_decoder,                     /* create codec function */ \
  destroy_decoder,                    /* destroy codec */ \
  codec_decoder,                      /* encode/decode */ \
  NULL,                               /* codec controls */ \
  PluginCodec_H323Codec_nonStandard,  /* h323CapabilityType */ \
  &speex##suffix##Cap                 /* h323CapabilityData */ \
} \

#define DECLARE_SPEEXW_CODEC(suffix, ordinal) \
/* SpeexW OpenH323 capability */ \
{ \
  /* encoder */ \
  PLUGIN_CODEC_VERSION,               /* codec API version */ \
  &licenseInfo,                       /* license information */ \
  PluginCodec_MediaTypeAudio |        /* audio codec */ \
  PluginCodec_InputTypeRaw |          /* raw input data */ \
  PluginCodec_OutputTypeRaw |         /* raw output data */ \
  PluginCodec_RTPTypeDynamic,         /* dynamic RTP type */ \
  speexW##suffix,                      /* text decription */ \
  L16Desc,                            /* source format */ \
  speexW##suffix,                      /* destination format */ \
  (void *)ordinal,                    /* user data */ \
  8000,                               /* samples per second */ \
  speexInfo[ordinal-2][0],            /* raw bits per second */ \
  20000,                              /* nanoseconds per frame */ \
  SAMPLES_PER_FRAME,                  /* samples per frame */ \
  speexInfo[ordinal-2][1],            /* bytes per frame */ \
  1,                                  /* recommended number of frames per packet */ \
  1,                                  /* maximum number of frames per packet  */ \
  0,                                  /* IANA RTP payload code */ \
  NULL,                               /* RTP payload name */ \
  create_encoder,                     /* create codec function */ \
  destroy_encoder,                    /* destroy codec */ \
  codec_encoder,                      /* encode/decode */ \
  NULL,                               /* codec controls */ \
  PluginCodec_H323Codec_nonStandard,  /* h323CapabilityType */ \
  &speexW##suffix##Cap                 /* h323CapabilityData */ \
}, \
{  \
  /* decoder */ \
  PLUGIN_CODEC_VERSION,               /* codec API version */ \
  &licenseInfo,                       /* license information */ \
  PluginCodec_MediaTypeAudio |        /* audio codec */ \
  PluginCodec_InputTypeRaw |          /* raw input data */ \
  PluginCodec_OutputTypeRaw |         /* raw output data */ \
  PluginCodec_RTPTypeDynamic,         /* dynamic RTP type */ \
  speexW##suffix,                     /* text decription */ \
  speexW##suffix,                     /* source format */ \
  L16Desc,                            /* destination format */ \
  (void *)ordinal,                    /* user data */ \
  8000,                               /* samples per second */ \
  speexInfo[ordinal-2][0],            /* raw bits per second */ \
  20000,                              /* nanoseconds per frame */ \
  SAMPLES_PER_FRAME,                  /* samples per frame */ \
  speexInfo[ordinal-2][1],            /* bytes per frame */ \
  1,                                  /* recommended number of frames per packet */ \
  1,                                  /* maximum number of frames per packet */ \
  0,                                  /* IANA RTP payload code */ \
  NULL,                               /* RTP payload name */ \
  create_decoder,                     /* create codec function */ \
  destroy_decoder,                    /* destroy codec */ \
  codec_decoder,                      /* encode/decode */ \
  NULL,                               /* codec controls */ \
  PluginCodec_H323Codec_nonStandard,  /* h323CapabilityType */ \
  &speexW##suffix##Cap                /* h323CapabilityData */ \
} \

CREATE_SPEEX_CAP_DATA(Narrow-5.95k, Narrow5k95,  2)
CREATE_SPEEX_CAP_DATA(Narrow-8k,    Narrow8k,    3)
CREATE_SPEEX_CAP_DATA(Narrow-11k,   Narrow11k,   4)
CREATE_SPEEX_CAP_DATA(Narrow-15k,   Narrow15k,   5)
CREATE_SPEEX_CAP_DATA(Narrow-18.2k, Narrow18k2,  6)
CREATE_SPEEX_CAP_DATA(Narrow-24.6k, Narrow24k6,  7)

static int speexInfo[][2] = {
  {  5950,  (119+7)/8 },
  {  8000,  (160+7)/8 },
  { 11000,  (220+7)/8 },
  { 15000,  (300+7)/8 },
  { 18200,  (364+7)/8 },
  { 24600,  (492+7)/8 },
};

static struct PluginCodec_Definition speexCodecDefn[] = {
  DECLARE_SPEEX_CODEC(Narrow5k95,  2),
  DECLARE_SPEEX_CODEC(Narrow8k,    3),
  DECLARE_SPEEX_CODEC(Narrow11k,   4),
  DECLARE_SPEEX_CODEC(Narrow15k,   5),
  DECLARE_SPEEX_CODEC(Narrow18k2,  6),
  DECLARE_SPEEX_CODEC(Narrow24k6,  7),

  //DECLARE_SPEEXW_CODEC(Narrow5k95,  2),   // does not work
  DECLARE_SPEEXW_CODEC(Narrow8k,    3),
  //DECLARE_SPEEX_CODEC(Narrow11k,   4),    // does not work
  //DECLARE_SPEEX_CODEC(Narrow15k,   5),    // does not work
  //DECLARE_SPEEX_CODEC(Narrow18k2,  6),    // does not work
  //DECLARE_SPEEX_CODEC(Narrow24k6,  7)     // does not work
};

#define NUM_DEFNS   (sizeof(speexCodecDefn) / sizeof(struct PluginCodec_Definition))

/////////////////////////////////////////////////////////////////////////////

extern "C" {

PLUGIN_CODEC_DLL_API struct PluginCodec_Definition * PLUGIN_CODEC_GET_CODEC_FN(unsigned * count, unsigned version)
{
  *count = NUM_DEFNS;
  return speexCodecDefn;
}

};

