/* Open Digita Services  --  Camera Device Protocol definitions.
  
   Copyright (C) 1998, 1999 James C. Thompson <jim.thompson@pobox.com>
   Copyright (C) 1998, 1999 Viljo Hakala <viljo.hakala@pcuf.fi>
   
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at your
   option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

/*--------------------------------------------------------------------------
  System include files */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#if defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif

/*--------------------------------------------------------------------------
  Local include files */

#include "cameraP.h"
#include "receive.h"

/* Start Editing Here: */

void ensure_buffer_size(ODSCamera camera, unsigned int size)
{
  if (size > 0x10000)
    {
      abort();
    }

  /* Make sure we have a buffer to use. */
  if (camera->buffer == 0) 
    {
      /* The initial buffer size should be at least as large as the
	 host buffer size */
      camera->buffer = malloc(camera->host_size);
      camera->bufsize = camera->host_size;
    }
  else if (camera->bufsize < size) 
    {
      camera->buffer = realloc(camera->buffer, size);
      camera->bufsize = size;
    }
}

ODSResult 
begin_rec(ODSCamera camera,
	  unsigned long *length)
{
  unsigned long   unused;
  short           command;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if ((camera->mode != ods_prereceive) ||
      (camera->receive_message == 0))
    {
      return kODSModeError;
    }

  camera->bufbytes = 0;
  camera->bufused = 0;

  /* Advance the mode */
  camera->mode = ods_receive;

  /* Receive the entire message into one buffer. */
  if ((*camera->receive_message)(camera) != 0)
    {
      return kODSIOError;
    }

  /* Extract the parameters in the message header. */
  rec_uint(camera, length);
  rec_uint(camera, &unused);
  rec_ushort(camera, &command);
  rec_ushort(camera, &(camera->result));

  /* Check the command to ensure that it matches the last sent command.
     If not, something is amiss. */
  if ((~command) != ((unsigned short) camera->command))
    {
      fprintf(stderr, "begin_rec: Command mismatch: sent %02hx, got %02hx\n",
	      (unsigned short) (camera->command), ~command);
    }

  return kODSNoErr;
}

ODSResult
rec_ushort(ODSCamera camera, short *s)
{
  ODSResult rc;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  rc = rec_bytes(camera, s, sizeof(*s), 1);
  swap_short(*s);

  return rc;
}

ODSResult
rec_uint(ODSCamera camera, UInteger * u)
{
  ODSResult rc;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  rc = rec_bytes(camera, u, sizeof(*u), 1);
  swap_long(*u);

  return rc;
}

ODSResult
rec_sint(ODSCamera camera, SInteger * s)
{
  ODSResult rc;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  rc = rec_bytes(camera, s, sizeof(*s), 1);
  swap_long(*s);

  return rc;
}

ODSResult
rec_fixed(ODSCamera camera, Fixed * f)
{
  ODSResult rc;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  rc = rec_bytes(camera, f, sizeof(*f), 1);
  swap_long(*f);

  return rc;
}

ODSResult
rec_boolean(ODSCamera camera, Boolean * b)
{
  ODSResult rc;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  rc = rec_bytes(camera, b, sizeof(*b), 1);
  swap_long(*b);

  return rc;
}

ODSResult
rec_bitflags(ODSCamera camera, BitFlags * f)
{
  ODSResult       rc;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  rc = rec_bytes(camera, f, sizeof(*f), 1);
  swap_long(*f);

  return rc;
}

ODSResult
rec_pname(ODSCamera camera, PName * p)
{
  ODSResult rc;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  rc = rec_bytes(camera, p, sizeof(p), 1);
  swap_long(*p);

  return rc;
}

ODSResult
rec_string(ODSCamera camera, String * s)
{
  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  return rec_bytes(camera, s, sizeof(*s), 1);
}

ODSResult
rec_dosname(ODSCamera camera, DOSName * d)
{
  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  return rec_bytes(camera, d, sizeof(*d), 1);
}

ODSResult
rec_bytes(ODSCamera camera, void *addr, size_t len, int nak)
{
  char *caddr = addr;
  unsigned long	unused;
  (void) nak;

  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  unused = camera->bufbytes - camera->bufused;

  /* If more has been requested than is available, return an error. */
  if (len > unused)
    {
      memset(addr, 0, len);
      return kODSIOError;
    }

  /* Copy the message bytes out into the supplied buffer. */
  if (caddr != 0) 
    {
      memcpy(caddr, ((char *) camera->buffer) + camera->bufused, len);
    }

  /* Increment the number of bytes used. */
  camera->bufused += len;

  return kODSNoErr;
}

ODSResult 
rec_ptvs(ODSCamera camera, PNameTypeValueStruct *ptvs)
{
  rec_pname(camera, &(ptvs->Name));
  rec_uint(camera, (UInteger *) & (ptvs->Type));
  rec_bytes(camera, &(ptvs->Value), sizeof(PNameValue), 1);

  switch (ptvs->Type) 
    {
    case kUInteger:
      swap_long(ptvs->Value.UInt);
      break;

    case kSInteger:
      swap_long(ptvs->Value.Int);
      break;

    case kFixed:
      swap_long(ptvs->Value.fFixed);
      break;

    case kBoolean:
      swap_long(ptvs->Value.fBoolean);
      break;

    case kBitFlags:
      swap_long(ptvs->Value.fBitFlags);
      break;

    case kPName:
      swap_long(ptvs->Value.fPName);
      break;

    case kDOSName:
    case kString:
    case kEnumList:
      break;
    }

  return kODSNoErr;
}


void 
dump_ptvs(PNameTypeValueStruct *ptvs)
{
  char            name[5];

  unsigned long   pname = ptvs->Name;
  swap_long(pname);
  memcpy(name, (const char *) &pname, 4);
  name[4] = '\0';

  printf("%s: ", name);

  switch (ptvs->Type) 
    {
    case kUInteger:
      printf("%08lx\n", ptvs->Value.UInt);
      break;

    case kSInteger:
      printf("%08lx\n", ptvs->Value.Int);
      break;

    case kFixed:
      printf("%08lx\n", ptvs->Value.fFixed);
      break;

    case kBoolean:
      printf("%s\n", ptvs->Value.fFixed ? "true" : "false");
      break;

    case kBitFlags:
      printf("%08lx\n", ptvs->Value.fBitFlags);
      break;

    case kPName:
      pname = ptvs->Value.fPName;
      swap_long(pname);
      memcpy(name, (const char *) &pname, 4);
      name[5] = '\0';
      printf("\"%s\"\n", name);
      break;

    case kDOSName:
      printf("\"%s\"\n", ptvs->Value.fDOSName);
      break;

    case kString:
      printf("\"%s\"\n", ptvs->Value.fString);
      break;

    case kEnumList:
      break;
    }
}

ODSResult 
rec_fileitem(ODSCamera camera, ResFileItem *fileitem)
{
  rec_sint(camera, &(fileitem->DriveNo));
  rec_string(camera, &(fileitem->PathName));
  rec_dosname(camera, &(fileitem->DOSName));
  rec_sint(camera, &(fileitem->FileLength));
  rec_bitflags(camera, &(fileitem->FileStatus));

  return kODSNoErr;
}

ODSResult 
rec_partialtag(ODSCamera camera, PartialTag *partialtag)
{
  rec_uint(camera, &(partialtag->Offset));
  rec_uint(camera, &(partialtag->Length));
  rec_uint(camera, &(partialtag->Filesize));

  return kODSNoErr;
}

ODSResult 
end_rec(ODSCamera camera)
{
  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_receive) 
    {
      return kODSModeError;
    }

  camera->mode = ods_presend;

  return kODSNoErr;
}
