/* Open Digita Services  --  Camera Device Protocol definitions.
   
   Copyright (C) 1998, 1999 James C. Thompson <jim.thompson@pobox.com>
   Copyright (C) 1998, 1999 Viljo Hakala <viljo.hakala@pcuf.fi>
   
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at your
   option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

/*--------------------------------------------------------------------------
  System include files */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#if defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/*--------------------------------------------------------------------------
  Local include files */

#include "ods.h"
#include "cameraP.h"
#include "ods_cmds.h"

/* Start Editing Here: */

typedef struct _camopts 
{
  unsigned int	opt_type : 1;	        /* Device type specified */
  unsigned int	opt_serial : 1;		/* Prefer NSS */
  unsigned int	opt_usb : 1;		/* Prefer Useless Serial Bus */
  unsigned int	opt_irda : 1;		/* Prefer infrared */

  unsigned int	opt_verbosity : 2;	/* How much output to spew */

  const char *	devtype;		/* Type of device to use */
  const char *	device;			/* Device to use. */
  const char *	bitrate;		/* Bitrate for serial devs */

  unsigned int rate;			/* Numeric bitrate */
} camopts;

static const char *
strcdup(const char *orig, const char *src)
{
  if (orig != 0) 
    {
      free((char *) orig);
    }

  if (src != 0) 
    {
#if defined(HAVE_STRDUP)
      return strdup(src);
#else
      char *tmp = malloc(strlen(src) + 1);
      strcpy(tmp, src);
      return tmp;
#endif
    }

  return 0;
}

static void 
_ods_deduce_devtype(ODSCamera camera)
{
  camopts *opts = camera->opts;

  if ((strncmp(opts->device, "/dev/cua", 8) == 0) ||
      (strncmp(opts->device, "/dev/tty", 8) == 0)) 
    {
      opts->opt_serial = !0;
      opts->opt_usb = 0;
      opts->opt_irda = 0;
    }
  else if ((strncmp(opts->device, "/dev/usb", 8) == 0) ||
	   (strncmp(opts->device, "/dev/kodak", 10) == 0))
    {
      opts->opt_serial = 0;
      opts->opt_usb = !0;
      opts->opt_irda = 0;
    }
  else if (strncmp(opts->device, "/dev/irda", 9) == 0) 
    {
      opts->opt_serial = 0;
      opts->opt_usb = 0;
      opts->opt_irda = !0;
    }
}

static void 
_ods_get_devtype(ODSCamera camera)
{
  camopts *opts = camera->opts;

  if (strncmp(opts->devtype, "serial", strlen(opts->devtype)) == 0) 
    {
      opts->opt_serial = !0;
      opts->opt_usb = 0;
      opts->opt_irda = 0;
    } 
  else if (strncmp(opts->devtype, "usb", strlen(opts->devtype)) == 0) 
    {
      opts->opt_serial = 0;
      opts->opt_usb = !0;
      opts->opt_irda = 0;
    }
  else if (strncmp(opts->devtype, "irda", strlen(opts->devtype)) == 0)
    {
      opts->opt_serial = 0;
      opts->opt_usb = 0;
      opts->opt_irda = !0;
    }
}

static void 
_ods_get_bitrate(ODSCamera camera)
{
  camopts        *opts = camera->opts;
  unsigned long   rate = atol(opts->bitrate);

  if (rate != 0) 
    {
      opts->rate = rate;
    }
}

static void 
_ods_init_env(ODSCamera camera)
{
  camopts	*opts = camera->opts;

  const char	*devtype = getenv("ODSTYPE");
  const char	*device = getenv("ODSDEVICE");
  const char	*bitrate = getenv("ODSRATE");

  if (devtype != 0) 
    {
      opts->devtype = strcdup(opts->devtype, devtype);
      opts->opt_type = !0;
      _ods_get_devtype(camera);
    }

  if (device != 0) 
    {
      opts->device = strcdup(opts->device, device);
      _ods_deduce_devtype(camera);
    }

  if (bitrate != 0) 
    {
      opts->bitrate = strcdup(opts->bitrate, bitrate);
      _ods_get_bitrate(camera);
    }
}

static void 
_ods_init_defaults(ODSCamera camera)
{
  camopts        *opts = camera->opts;

  opts->opt_serial = !0;
#if defined(__LINUX__)
  opts->device = strcdup(opts->device, "/dev/ttyS0");
#else
  opts->device = strcdup(opts->device, "/dev/cuaa0");
#endif
  opts->rate = 9600;
}

void 
_ods_help(const char *argv[])
{

  printf("Usage: %s [OPTION]... [COMMAND]...\n", argv[0]);
  printf("Connect to camera and perform COMMANDs.\n");
  printf("Updated: %s\n\n", VERSION);

  printf("Development note: currently some COMMANDs are supported; default\n");
  printf("action is to display this help.  However, all listed\n");
  printf("OPTIONS are now supported.\n\n");

  printf("  -d DEV, --device=DEV       Look for camera at device DEV.  Defaults\n");
  printf("                             to /dev/ttyS0.\n");
  printf("  -r RATE, --rate=RATE       Use bit-rate RATE for serial device.  Defaults\n");
  printf("                             to 9600.  Not applicable to USB or IrDA.\n");
  printf("  -t TYPE, --type=TYPE       Specifies device type, one of: \"serial\"\n");
  printf("                             \"usb\", or \"irda\".  Normally, device type\n");
  printf("                             can be deduced from DEV and need not be\n");
  printf("                             specified\n");
  printf("  -i, --irda                 Same as --type=irda\n");
  printf("  -s, --serial               Same as --type=serial\n");
  printf("  -u, --usb                  Same as --type=usb\n");
  printf("\n");
  printf("  -v  --verbose              More output.\n");
  printf("  -q  --quiet                Less output.\n");
  printf("\n");
  printf("  -l, --list                 List supported commands\n");
  printf("  -h, --help                 Display this help and exit.\n");

  printf("\nReport bugs to jim.thompson@pobox.com\n");
  printf("               viljo.hakala@pcuf.fi\n");
  exit(0);
}

static void 
_ods_init_args(ODSCamera camera, int *argc, const char *argv[])
{
  camopts *opts = camera->opts;

  int i;
  int c;

  opterr = 0;				/* Tell getopt not to report errors. */

  camera->verbosity = 1;

  while (1) 
    {
      int startind = optind ? optind : 1;
      int dont_mangle = 0;

      /* For each camera-specific --option, we also recognize
	 --ods-option.  Does not apply to generic options such as
	 --help. */

      c = getopt(*argc, (char *const *) argv, "-d:r:t:suilhqv");

      if (c == -1)
	{
	  break;
	}

      switch (c) 
	{
	case 1:
	  dont_mangle = !0;
	  break;

	case 'd':

	  opts->device = strcdup(opts->device, optarg);

	  /* If no device type has already been specified, attempt to
	     guess the device type based on the device name. */
	  if (!(opts->opt_type)) 
	    {
	      _ods_deduce_devtype(camera);
	    }
	  break;

	case 't':
	  opts->opt_type = !0;
	  opts->devtype = strcdup(opts->devtype, optarg);
	  _ods_get_devtype(camera);

	  break;

	case 'r':
	  opts->bitrate = strcdup(opts->bitrate, optarg);
	  _ods_get_bitrate(camera);
	  
	  break;

	case 's':
	  opts->opt_type = !0;
	  opts->opt_serial = !0;
	  opts->opt_usb = 0;
	  opts->opt_irda = 0;
	  break;

	case 'u':
	  opts->opt_type = !0;
	  opts->opt_serial = 0;
	  opts->opt_usb = !0;
	  opts->opt_irda = 0;
	  break;

	case 'i':
	  opts->opt_type = !0;
	  opts->opt_serial = 0;
	  opts->opt_usb = 0;
	  opts->opt_irda = !0;
	  break;

	case 'v':
	  camera->verbosity = 2;
	  break;

	case 'q':
	  camera->verbosity = 0;
	  break;

	case 'l':
	  _ods_list_commands();
	  break;

	case 'h':
	  _ods_help(argv);
	  break;

	case '?':
	  _ods_help(argv);
	  dont_mangle = !0;
	  break;

	default:
	  dont_mangle = !0;
	}

      if (!dont_mangle) 
	{
	  int diff = optind - startind;

	  for (i = optind; i < *argc; i++) 
	    {
	      argv[i - diff] = argv[i];
	    }

	  optind -= diff;
	  (*argc) -= diff;
	}
    }
}

#if DEBUG
static void 
_dump_opts(ODSCamera camera)
{
  camopts *opts = camera->opts;

  if (camera->opts == 0) 
    {
      return;
    }

  printf("------------------------\n");

  if (opts->device == 0) 
    {
      printf("no device specified\n");
    } 
  else 
    {
      printf("device = <%s>\n", opts->device);
    }

  if (opts->devtype == 0) 
    {
      printf("no devtype specified\n");
    } 
  else 
    {
      printf("devtype = <%s>\n", opts->devtype);
    }

  if (opts->devtype == 0) 
    {
      printf("no bitrate specified\n");
    } 
  else 
    {
      printf("bitrate = <%s>\n", opts->bitrate);
    }

  printf("opt_devtype = %d\n", opts->opt_type);
  printf("opt_serial = %d\n", opts->opt_serial);
  printf("opt_usb = %d\n", opts->opt_usb);
  printf("opt_irda = %d\n", opts->opt_irda);

  printf("rate = %d\n", opts->rate);

  printf("------------------------\n");
}
#endif

ODSCamera 
ODSInitialize(int *argc, const char *argv[])
{
  ODSCamera	camera;
  camopts *	opts;
  ODSResult	result;

  camera = odsCreateCamera();

  camera->opts = opts = malloc(sizeof(camopts));
  memset(camera->opts, 0, sizeof(camopts));

  _ods_init_defaults(camera);
  _ods_init_env(camera);
  _ods_init_args(camera, argc, argv);

  if (opts->opt_serial) 
    {
      result = odsOpenNSS(camera, camera->opts->device, camera->opts->rate);

      if (result == kODSIOError)
	{
#if defined(DEBUG)
	  printf("Open failed, retrying...\n");
#endif	  
	  /* Try again on error. */
	  odsCloseNSS(camera);
	  result = odsOpenNSS(camera, camera->opts->device, 
			      camera->opts->rate);
	}

      if (result == kODSUnixError)
	{
	  fprintf(stderr,"%s: *** couldn't open device %s.\n",argv[0],
		  camera->opts->device); 
	  exit(1);	
	} 
    }
  else if (opts->opt_usb) 
    {
      result = odsOpenUSB(camera, camera->opts->device);

      if (result != kODSNoErr)
	{
	  fprintf(stderr,"%s: *** couldn't open device %s.\n",argv[0],
		  camera->opts->device); 
	  exit(1);	
	} 
    }
  else if (opts->opt_irda) 
    {
      fprintf(stderr, "%s: *** IrDA not yet supported.\n", argv[0]);
      exit(1);
    }

#if defined(DEBUG)
  _dump_opts(camera);
#endif

  return camera;
}

void 
ODSFinalize(ODSCamera camera)
{
  if (camera == 0) 
    {
      return;
    }

  if (camera->opts != 0) 
    {
      camopts        *opts = camera->opts;

      if (opts->opt_serial)
	{
	  odsCloseNSS(camera);
	} 
      else if (opts->opt_usb) 
	{
	  odsCloseUSB(camera);
	} 
      else if (opts->opt_irda) 
	{
	  /* Perform any shutdown required on the IrDA port */
	}

      strcdup(opts->device, 0);
      strcdup(opts->devtype, 0);
      strcdup(opts->bitrate, 0);

      free(camera->opts);
      camera->opts = 0;
    }

  odsDestroyCamera(camera);
}
