#!/usr/bin/perl -w

use POSIX;
use strict;

# set_time_from_exif: Determine an image's creation time (not its unix
# ctime, but the time at which the photograph was taken -- as stored
# in its OriginalDate EXIF tag) and set the image file's unix times
# (mtime and atime) equal to the EXIF creation time.

sub set_time_from_exif($)
  {
    my $name = shift;		# Get the filename

    # Call exdump to get the creation time.
    my $raw = `exdump.pl --parseable --date-time-original $name`;

    # Date/time-stamps are stored in EXIF files in the form
    # "1999:12:31 23:59:59".  This regular expression will match the
    # date and time.
    my $datere = 
      "\"([0-9][0-9][0-9][0-9]):([0-9][0-9]):([0-9][0-9]) " .
	"([0-9][0-9]):([0-9][0-9]):([0-9][0-9])\"";

    # If the regex for the date and time doesn't match the output of
    # exdump, then issue an error message and continue.
    if (! ($raw =~ /original-date=$datere/))
      {
	print "$name: Unable to read original date/time, skipping.\n";
	return;
      }

    # Retrieve data and time values from the regular expression as
    # matched.
    my $year = $1;
    my $month = $2;
    my $day = $3;
    my $hour = $4;
    my $minute = $5;
    my $second = $6;

#    print "y = $year, m = $month, d = $day, h = $hour, m = $minute, s = $second\n";

    # Modification time from the unix epoch.
    my $mod_time = 0;

    # Compute the unix seconds-since-the-epoch time.
    $mod_time = mktime($second, $minute, $hour, $day, $month - 1, 
		       $year - 1900, 0, 0, 1);

    if ($mod_time)
      {
	# Issue an informational message.
	print "$name: setting time to ", POSIX::ctime($mod_time);

	# Here's the magic: call utime to modify the file's access time
	# and modification time to match the image's creation time.
	utime $mod_time, $mod_time, $name;
      }
    else
      {
	print "$name: unable to compute correct time.\n";
      }
  }

# main

# If no files were specified on the command line, issue a usage statement.
@ARGV || do {
  print "Usage: exdump file [ more files ... ]\n";
  exit;
};

# loop through the specified files, setting the time on each.
for(@ARGV) {
  set_time_from_exif($_);
}
