/*
	main.cc for OBPager, a pager dockapp designed to work with OpenBox or any netwm-compliant window manager.
	
	Copyright (c) 2004 - Roy Wood
	
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	See the COPYING file for a copy of the GNU General Public License.
*/


// Include the STL header(s) we need

#include <iostream>


// Other system necessary headers

#include <sys/types.h>
#include <unistd.h>


// Include the headers for the pager class and also the spiffy custom exception class

#include "OBPager.h"
#include "VerboseException.h"



// As usual in a C++ app, main is pretty small (okay-- it was until I started parsing the args by hand and forking subprocesses)

int main(int argc, char **argv)
{
	// By default, monitor all desktops
	
	int desktopNum = -1;
	
	
	// We have to make sure the child OBPagers are shown in the right order, so, we set up a pipe to talk 
	// to the forked process, and then wait for it to send us a message once it has received its first expose event
	
	// This is the file descriptor/number of the pipe that the child should write to once it has received its first expose event
	
	int childPipeFD = -1;
	
	
	// All the real work is done by the OBPager object
	
	char* activeDesktopFont = NULL;
	char* inactiveDesktopFont = NULL;
	bool forceSync = false;
	
	// Parse user options
	
	for (int i = 1; i < argc; ++i) {
		if (strspn(argv[i], "0123456789") == strlen(argv[i])) {
			// OBPager class expects a zero-based desktop number, so subtract one from what the human passed in
			
			desktopNum = atoi(argv[i]) - 1;
			
			if (desktopNum < 0) {
				desktopNum = 0;
			}
		}
		
		else if (strcmp(argv[i], "--activefont") == 0) {
			if (i + 1 >= argc) {
				std::cerr << "Invalid 'activefont'" << std::endl;
				exit(1);
			}
			
			activeDesktopFont = argv[i + 1];
			
			++i;
		}
		
		else if (strcmp(argv[i], "--inactivefont") == 0) {
			if (i + 1 >= argc) {
				std::cerr << "Invalid 'inactivefont'" << std::endl;
				exit(1);
			}
			
			inactiveDesktopFont = argv[i + 1];
			
			++i;
		}
		
		else if (strcmp(argv[i], "--forcesync") == 0) {
			forceSync = true;
		}
		
		else {
			std::cout << "obpager [--activefont <fontname>] [--inactivefont <fontname>] [--forcesync] [desktop-number]" << std::endl;
			
			exit(0);
		}
	}
	
	
	// If user did not explicitly indicate desktop to monitor, figure out how many there are and fork a process to monitor each one
	
	if (desktopNum < 0) {
		long numDesktops = 0;
		
		// Create an OBPager to use to query the number of desktops (kinda inefficient, I know, but it was the easiest way)
		
		try {
			OBPager obpager(desktopNum, activeDesktopFont, inactiveDesktopFont);
			
			obpager.connectToXServer();
			
			numDesktops = obpager.numDesktops();
		}
		catch (VerboseException &ex) {
			// OBPager tries to throw "VerboseException" objects which include lots of extra info
			
			std::cerr << "Could not create pager: " << ex.whatSrcFile() << ":" << ex.whatSrcLine() << ":" << ex.whatFunction() << ":" << ex.what() << std::endl;
			
			exit(0);
		}
		catch (std::exception &ex) {
			// No VerboseException caught-- settle for a std::exception
			
			std::cerr << "Could not create pager: " << ex.what() << std::endl;
			
			exit(0);
		}
		catch (...) {
			// Oops-- something bad happened, and no further details are available!
			
			std::cerr << "Could not create pager" << std::endl;
			
			exit(0);
		}
		
		
		
		// This is the important bit-- set the desktop this process is to monitor (could be main process or child process, right?)
		
		for (desktopNum = 0; desktopNum < numDesktops; ++desktopNum) {
			// Fork a child for all but the last desktop; for the last one, just fall out of the loop and do things in the current process
			
			if (desktopNum >= numDesktops - 1) {
				break;
			}
			
			
			// As mentioned, we need to communicate with the child process to find out when it is exposed, so set up the pipe pair to do so
			
			int pipefd[2];
			
			if (pipe(pipefd) < 0) {
				std::cerr << "Could not create pipe to communicate with child process, errno = " << errno;
				
				exit(0);
			}
			
			
			// Now fork the child process
			
			pid_t childID = fork();
			
			if (childID == 0) {
				// As the child, we write, not read, so close off the appropriate fd
				
				close(pipefd[0]);
				
				childPipeFD = pipefd[1];
				
				break;
			}
			
			
			// As parent, we read, not write, so close off the appropriate fd
			
			close(pipefd[1]);
			
			
			// Now wait for the message from the child process before we continue
			
			char tempString[256];
			
			int bytesRead = read(pipefd[0], tempString, sizeof(tempString) - 1);
			
			if (bytesRead < 0) {
				std::cerr << "Could not read from pipe, errno = " << errno;
				
				exit(0);
			}
			
			tempString[bytesRead] = '\0';
			
//~ 			std::cout << "Child 0x" << std::hex << childID << std::dec << " said '" << tempString << "'" << std::endl;
			
			close(pipefd[0]);
		}
	}
	
//~ 	pid_t myID = getpid();
//~ 	
//~ 	std::cout << "I am process 0x" << std::hex << myID << std::dec << ", and I am monitoring desktop #" << desktopNum << std::endl;
	
	
	// Show time-- try to create an OBPager, connect to the X server, show the window, and run....
	
	try {
		OBPager obpager(desktopNum, activeDesktopFont, inactiveDesktopFont, childPipeFD);
		
		obpager.connectToXServer();
		
		obpager.createShowWindow();
		
		obpager.run();
	}
	catch (VerboseException &ex) {
		// OBPager tries to throw "VerboseException" objects which include lots of extra info
		
		std::cerr << "Could not create/show/run pager: " << ex.whatSrcFile() << ":" << ex.whatSrcLine() << ":" << ex.whatFunction() << ":" << ex.what() << std::endl;
	}
	catch (std::exception &ex) {
		// No VerboseException caught-- settle for a std::exception
		
		std::cerr << "Could not create/show/run pager: " << ex.what() << std::endl;
	}
	catch (...) {
		// Oops-- something bad happened, and no further details are available!
		
		std::cerr << "Could not create/show/run pager" << std::endl;
	}
	
	
	// Frankie say, "No more...."
	
	return 0;
}
