/*
	OBPager.cc for OBPager, a pager dockapp designed to work with OpenBox or any netwm-compliant window manager.
	
	Copyright (c) 2004 - Roy Wood
	
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	See the COPYING file for a copy of the GNU General Public License.
*/


#include "OBPager.h"
#include "VerboseException.h"

#include <string>
#include <sstream>


// We need to work with these properties

#define UTF8_STRING_PROP "UTF8_STRING"
#define NET_CURRENT_DESKTOP_PROP "_NET_CURRENT_DESKTOP"
#define NUMBER_OF_DESKTOPS_PROP "_NET_NUMBER_OF_DESKTOPS"
#define NET_SUPPORTED_PROP "_NET_SUPPORTED"
#define NET_CLIENT_LIST_STACKING_PROP "_NET_CLIENT_LIST_STACKING"
#define NET_WMDESKTOP_PROP "_NET_WM_DESKTOP"
#define WM_NAME_PROP "WM_NAME"
#define NET_ACTIVE_WINDOW_PROP "_NET_ACTIVE_WINDOW"
#define NET_DESKTOP_GEOMETRY_PROP "_NET_DESKTOP_GEOMETRY"
#define NET_SUPPORTING_WM_CHECK_PROP "_NET_SUPPORTING_WM_CHECK"
#define NET_WM_NAME_PROP "_NET_WM_NAME"


// Some specifics about the dockapp window

#define APPLET_WIDTH 64
#define APPLET_HEIGHT 64
#define APPLET_MARGIN_WIDTH	4
#define APPLET_NAME "obpager"


// Default fonts we use

#define DEFAULT_ACTIVE_FONT "-misc-fixed-medium-*-*-*-20-*-*-*-*-*-*-*"
#define DEFAULT_INACTIVE_FONT "-misc-fixed-medium-*-*-*-10-*-*-*-*-*-*-*"



// Since the Xlib error handler is static, it needs to know which OBPager instance to work with
// We probably could just have a char[ ] to hold the message, but this way we can poke around inside the OBPager, if necessary

OBPager* OBPager::sActiveOBPager = NULL;



/**
 * Our static Xlib error handler
 *
 * @param display is the X Display the error is associated with
 * @param errEvent is the error event
 */

int OBPager::errorHandler(Display* display, XErrorEvent* errEvent)
{
	// Don't bother printing anything-- looks dumb and doesn't help
	
//~ 	static char errText[512];
//~ 	
//~ 	errText[0] = '\0';
//~ 	
//~ 	XGetErrorText(display, errEvent->error_code, errText, sizeof(errText));
//~ 	
//~ 	if (sActiveOBPager != NULL) {
//~ 		std::cerr << "OBPager: Current action: " << sActiveOBPager->mCurrentAction << std::endl;
//~ 	}
//~ 	
//~ 	std::cerr << "OBPager: An Xlib error occurred: " << errText << std::endl;
	
	return errEvent->error_code;
}



/**
 * Set the textual description of the current OBPager action (useful for debugging errors and such)
 *
 * @param actionDescription is the description of the current action being undertaken by this OBPager
 * @param window is the X Window reference of this OBPager
 */

void OBPager::setCurrentAction(const char* actionDescription, Window window)
{
	std::ostringstream msgStream;
	
	msgStream << actionDescription;
	
	if (window != 0) {
		msgStream << ", Window = 0x" << std::hex << window << std::dec;
	}
	
	mCurrentAction = msgStream.str();
}




/**
 * Construct an OBPager object
 *
 * @param desktopNum is the virtual desktop we monitor
 * @param activeDesktopFont is the name of the font to use to display the number of the desktop when it is active
 * @param inactiveDesktopFont is the name of the font to use to display the number of the desktop when it is inactive
 * @param fd is the file descriptor we write when we receive our first expose event
 * @param forceSync indicates whether to call XSynchronize() and thus force synchronization with Xlib
 */
 
OBPager::OBPager(int desktopNum, const char* activeDesktopFont, const char* inactiveDesktopFont, int fd, bool forceSync) : 
	mDesktopNum(desktopNum),
	mActiveDesktopFont((activeDesktopFont == NULL) ? DEFAULT_ACTIVE_FONT : activeDesktopFont),
	mInactiveDesktopFont((inactiveDesktopFont == NULL) ? DEFAULT_INACTIVE_FONT : inactiveDesktopFont),
	mForceSync(forceSync),
	mExposed(false),
	mFD(fd)
{
	// Note that we construct the std::string objects to copy the char*'s our caller passes in
	
	
	// Take note of the fact that we are the active OBPager instance (half-assed version of singleton pattern-- should use a factory....)
	
	sActiveOBPager = this;
}



/**
 * Destruct the OBPager
 */

OBPager::~OBPager()
{
	// Again, this is lame
	
	sActiveOBPager = NULL;
}



/**
 * Connect to the specified X server, or to the default server in $DISPLAY if displayName is null
 *
 * @param displayName is the name of the display to connect to
 */

void OBPager::connectToXServer(char *displayName)
{
	// Connect to the user-specified desktop or whatever is in $DISPLAY?
	
	if (displayName == NULL) {
		displayName = getenv("DISPLAY");
	}
	
	if (displayName == NULL || *displayName == '\0') {
		displayName = ":0.0";
	}
	
	
	// Set Xlib error handler
	
	XSetErrorHandler(errorHandler);
	
	
	setCurrentAction("Opening X display");
	
	
	// Open the X display using the XDisplayKeeper to manage the resource (especially closing it later)
	
	mDisplay.acquire(XOpenDisplay(displayName));
	
	if (mDisplay() == NULL) {
		std::ostringstream errMsgStream;
		
		errMsgStream << "Could not open X display '" << displayName << "'";
		
		THROW_VERBOSE_EXCEPTION(errMsgStream.str());
	}
	
	
	// Force sync Xlib activity?  (debugging only, presumably)
	
	if (mForceSync) {
		setCurrentAction("Setting Xlib synchronized mode");
		
		XSynchronize(mDisplay(), true);
	}
	
	
	// Which screen?
	
	mScreenNum = DefaultScreen(mDisplay());
	
	
	
	// Now intern a whack of atoms we need to work with
	
	setCurrentAction("Interning X atoms");
	
	if ((mAtom_UTF8_STRING = XInternAtom(mDisplay(), UTF8_STRING_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" UTF8_STRING_PROP "'");
	}
	
	if ((mAtom_NET_CURRENT_DESKTOP = XInternAtom(mDisplay(), NET_CURRENT_DESKTOP_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NET_CURRENT_DESKTOP_PROP "'");
	}
	
	if ((mAtom_NET_NUMBER_OF_DESKTOPS = XInternAtom(mDisplay(), NUMBER_OF_DESKTOPS_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NUMBER_OF_DESKTOPS_PROP "'");
	}
	
	if ((mAtom_NET_SUPPORTED = XInternAtom(mDisplay(), NET_SUPPORTED_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NET_SUPPORTED_PROP "'");
	}
	
	if ((mAtom_NET_CLIENT_LIST_STACKING = XInternAtom(mDisplay(), NET_CLIENT_LIST_STACKING_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NET_CLIENT_LIST_STACKING_PROP "'");
	}
	
	if ((mAtom_NET_WMDESKTOP = XInternAtom(mDisplay(), NET_WMDESKTOP_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NET_WMDESKTOP_PROP "'");
	}
	
	if ((mAtom_NET_ACTIVE_WINDOW = XInternAtom(mDisplay(), NET_ACTIVE_WINDOW_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NET_ACTIVE_WINDOW_PROP "'");
	}
	
	if ((mAtom_NET_DESKTOP_GEOMETRY = XInternAtom(mDisplay(), NET_DESKTOP_GEOMETRY_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NET_DESKTOP_GEOMETRY_PROP "'");
	}
	
	if ((mAtom_NET_SUPPORTING_WM_CHECK = XInternAtom(mDisplay(), NET_SUPPORTING_WM_CHECK_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NET_SUPPORTING_WM_CHECK_PROP "'");
	}
	
	if ((mAtom_NET_WM_NAME = XInternAtom(mDisplay(), NET_WM_NAME_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" NET_WM_NAME_PROP "'");
	}
	
	if ((mAtom_WM_NAME = XInternAtom(mDisplay(), WM_NAME_PROP, True)) == None) {
		THROW_VERBOSE_EXCEPTION("Could not intern X Atom for '" WM_NAME_PROP "'");
	}
	
	
	// Not strictly necessary, but for fun let's find out who the window manager is....
	
//~ 	try {
//~ 		Window rootNetWMWindow = getAtomAsLong(mAtom_NET_SUPPORTING_WM_CHECK, XA_WINDOW);
//~ 		
//~ 		Window childNetWMWindow = getAtomAsLong(mAtom_NET_SUPPORTING_WM_CHECK, XA_WINDOW, rootNetWMWindow);
//~ 		
//~ 		if (rootNetWMWindow == childNetWMWindow) {
//~ 			std::vector<std::string> strings;
//~ 			
//~ 			getAtomAsStrings(mAtom_NET_WM_NAME, strings, mAtom_UTF8_STRING, childNetWMWindow);
//~ 			
//~ 			std::cout << "Found a netwm-compliant window manager, '" << strings[0] << "'" << std::endl;
//~ 		}
//~ 	}
//~ 	catch (...) {
//~ 		std::cout << "Could not find a netwm-compliant window manager; carrying on anyway...." << std::endl;
//~ 	}
}



/**
 * Create and show the X window.  As part of this, set up all the other resources we need (e.g. pixmaps, etc.)
 */

void OBPager::createShowWindow()
{
	// We are going to refer to the root window a lot, so set up a local var to avoid retyping the huge macro again and again....
	
	Window rootWindow = RootWindow(mDisplay(), mScreenNum);
	
	
	// A lot of these params could eventually be configurable in a .rc file or as command-line params, so keep 'em all in one spot
	
	int x = 0;
	int y = 0;
	int width = APPLET_WIDTH;
	int height = APPLET_HEIGHT;
	int borderWidth = 0;
	int maxWidth = width;
	int maxHeight = height;
	int minWidth = width;
	int minHeight = height;
	char* appName = APPLET_NAME;
	unsigned long borderColour = BlackPixel(mDisplay(), mScreenNum);
	unsigned long backgroundColour = BlackPixel(mDisplay(), mScreenNum);
	
	
	// Now create the window and store the reference in the XWindowKeeper
	
	setCurrentAction("Creating main X window");
	
	mWindow.acquire(mDisplay(), XCreateSimpleWindow(mDisplay(), rootWindow, x, y, width, height, borderWidth, borderColour, backgroundColour));
	
	if (mWindow() == 0) {
		THROW_VERBOSE_EXCEPTION("Could not create main X Window");
	}
	
	
	// We do all our rendering in a pixmap and then blit it onscreen to avoid flicker, so set up the pixmap
	
	setCurrentAction("Creating X offscreen pixmap");
	
	mPixmap.acquire(mDisplay(), XCreatePixmap(mDisplay(), mWindow(), width, height, DefaultDepth(mDisplay(), mScreenNum)));
	
	if (mPixmap() == 0) {
		THROW_VERBOSE_EXCEPTION("Could not create X Pixmap");
	}
	
	
	// Set up the size/wm/class hints; use XResourceKeepers to automatically release them later
	
	setCurrentAction("Allocating X hint structures");
	
	XResourceKeeper<XSizeHints> mSizeHints;
	
	mSizeHints.acquire(XAllocSizeHints());
	
	if (mSizeHints() == NULL) {
		THROW_VERBOSE_EXCEPTION("Could not allocate XSizeHints structure");
	}
	
	
	XResourceKeeper<XWMHints> mWMHints;
	
	mWMHints.acquire(XAllocWMHints());
	
	if (mSizeHints() == NULL) {
		THROW_VERBOSE_EXCEPTION("Could not allocate XWMHints structure");
	}
	
	
	XResourceKeeper<XClassHint> mClassHints;
	
	mClassHints.acquire(XAllocClassHint());
	
	if (mSizeHints() == NULL) {
		THROW_VERBOSE_EXCEPTION("Could not allocate XClassHint structure");
	}
	
	
	setCurrentAction("Setting X hints");
	
	// Tell the X server all about our size
	
	mSizeHints->min_width = minWidth;
	mSizeHints->min_height = minHeight;
	mSizeHints->max_width = maxWidth;
	mSizeHints->max_height = maxHeight;
	mSizeHints->flags = PSize | PMinSize | PMaxSize;
	
	XSetWMNormalHints(mDisplay(), mWindow(), mSizeHints());
	
	
	// Tell the window manager all about ourselves, too
	
	mWMHints->initial_state = WithdrawnState;
//~ 	mWMHints->initial_state = NormalState;
	mWMHints->input = False;
	mWMHints->icon_pixmap = mPixmap();
	mWMHints->flags = StateHint | InputHint | IconPixmapHint;
	
	XSetWMHints(mDisplay(), mWindow(), mWMHints());
	
	
	// Set the class hints
	
	mClassHints->res_name = appName;
	mClassHints->res_class = appName;
	
	XSetClassHint(mDisplay(), mWindow(), mClassHints());
	
	
	// Tricky bit here-- we define a rectangular mask for our window, thereby creating a transparent border that looks pretty when we are docked
	
	setCurrentAction("Creating offscreen X pixmap");
	
	// First, get the mask as an XBM, and hold it in an auto_ptr so it gets disposed when we exit this block
	
	std::auto_ptr<unsigned char> pixmaskBits(getRectangularXBMMask(width, height, APPLET_MARGIN_WIDTH));
	
	// Next, create the pixmap using the XBM mask
	
	XPixmapKeeper pixmask(mDisplay(), XCreateBitmapFromData(mDisplay(), mWindow(), (char *) pixmaskBits.get(), width, height));
	
	// And finally, use the pixmap mask for our window
	
	XShapeCombineMask(mDisplay(), mWindow(), ShapeBounding, 0, 0, pixmask(), ShapeSet);
	
	
	// Tell the server what events we want to receive
	
	setCurrentAction("Selecting X input events");
	
	XSelectInput(mDisplay(), mWindow(), ExposureMask | StructureNotifyMask | ButtonPressMask);
	
	XSelectInput(mDisplay(), rootWindow, PropertyChangeMask | SubstructureNotifyMask);
	
	
	// Show the window
	
	setCurrentAction("Mapping X window");
	
	XMapWindow(mDisplay(), mWindow());
	
	
	// Now set up the references to the fonts we want
	
	setCurrentAction("Loading X fonts");
	
	const char *fontNameSmall = mInactiveDesktopFont.c_str();
	
	mFontStructSmall.acquire(mDisplay(), XLoadQueryFont(mDisplay(), fontNameSmall));
	
	if (mFontStructSmall() == NULL) {
		std::string errMsg = "Could not load X Font '" + mInactiveDesktopFont + "'";
		
		THROW_VERBOSE_EXCEPTION(errMsg.c_str());
	}
	
	
	const char *fontNameLarge = mActiveDesktopFont.c_str();
	
	mFontStructLarge.acquire(mDisplay(), XLoadQueryFont(mDisplay(), fontNameLarge));
	
	if (mFontStructLarge() == NULL) {
		std::string errMsg = "Could not load X Font '" + mActiveDesktopFont + "'";
		
		THROW_VERBOSE_EXCEPTION(errMsg.c_str());
	}
	
	
	// Set up the graphics context for all our drawing later
	
	setCurrentAction("Setting up X graphics context");
	
	XGCValues gcValues;
	unsigned long gcValueMask = 0;
	
	gcValues.font = mFontStructSmall()->fid;
	gcValues.background = BlackPixel(mDisplay(), mScreenNum);
	gcValues.foreground = WhitePixel(mDisplay(), mScreenNum);
	gcValueMask = GCFont | GCBackground | GCForeground;
	
	mGC.acquire(mDisplay(), XCreateGC(mDisplay(), mWindow(), gcValueMask, &gcValues));
	
	if (mGC() == 0) {
		THROW_VERBOSE_EXCEPTION("Could not create X GC graphics context");
	}
	
	
	// And get the pixel colour references we need for drawing
	
	setCurrentAction("Allocating X colours");
	
	XWindowAttributes rootAttributes;
	XGetWindowAttributes(mDisplay(), rootWindow, &rootAttributes);
	
	mPixelColours[kAppletBorderShadowIndex] = getColourPixel("grey70", rootAttributes.colormap);
	mPixelColours[kActiveDesktopUnfocusedWindowIndex] = getColourPixel("grey60", rootAttributes.colormap);
	mPixelColours[kActiveDesktopFocusedWindowIndex] = getColourPixel("rgb:ff/40/40", rootAttributes.colormap);
	mPixelColours[kActiveDesktopFocusedBorderIndex] = getColourPixel("grey80", rootAttributes.colormap);
	mPixelColours[kActiveDesktopUnfocusedBorderIndex] = getColourPixel("grey50", rootAttributes.colormap);
	mPixelColours[kInactiveDesktopWindowIndex] = getColourPixel("grey30", rootAttributes.colormap);
	mPixelColours[kInactiveDesktopWindowBorderIndex] = getColourPixel("grey50", rootAttributes.colormap);
	mPixelColours[kInactiveDesktopFontColor] = getColourPixel("grey50", rootAttributes.colormap);
	mPixelColours[kActiveDesktopFontColor] = getColourPixel("white", rootAttributes.colormap);
}



/**
 * Get a pixel reference to a named colour
 *
 * @param colourName is the name of the colour (e.g. "grey80" or "rgb:aa/bb/cc")
 * @param colorMap is the Colormap the colour is associated with
 */

unsigned long OBPager::getColourPixel(const char* colourName, const Colormap colorMap) const
{
	// Request the colour from the X server; throw if we run into trouble
	
	XColor colour;
	
	if (!XParseColor(mDisplay(), colorMap, colourName, &colour) || !XAllocColor(mDisplay(), colorMap, &colour)) {
		std::ostringstream errMsgStream;
		
		errMsgStream << "Could not allocate X colour '" << colourName << "'";
		
		THROW_VERBOSE_EXCEPTION(errMsgStream.str());
	}
	
	return colour.pixel;
}



/**
 * Set up a rectangular xbm bitmap mask with a border of zeroes around the center ones.  A pointer to the xbm is returned,
 * and it is the caller's responsibility to delete the xbm later
 *
 * @param width is the pixel width of the output xbm
 * @param height is the pixel height of the output xbm
 * @param indent is the number of pixels to indent around the edges of the rectangular center
 */

unsigned char* OBPager::getRectangularXBMMask(unsigned int width, unsigned int height, unsigned int indent) const
{
	// Figure out how many bytes in each row of the xbm (taking care to round up to the next byte), then allocate the memory for it
	
	unsigned int rowBytes = (width + 7) / 8;
	unsigned char* xbm = new unsigned char[height * rowBytes];
	
	// Figure out total width in bits now
	
	width = rowBytes * 8;
	
	// Iterate over all rows and columns in xbm
	
	for (unsigned int row = 0; row < height; ++row) {
		unsigned char byte = 0;
		
		for (unsigned int col = 0; col < width; ++col) {
			// Shift current byte, set new bit if we are in the center of the rectangular mask area
			
			byte >>= 1;
			
			if (row >= indent && row < height - indent && col >= indent && col < width - indent) {
				byte |= 0x80;
			}
			
			
			// Time to emit the byte?
			
			if ((col + 1) % 8 == 0) {
				xbm[row * rowBytes + col / 8] = byte;
				
				byte = 0;
			}
		}
	}
	
	return xbm;
}



/**
 * Get the value of a window property as a long, using a specified Atom to reference the property
 *
 * @param xAtom is the Atom whose associated property we want to retrieve
 * @param reqType is the type of data to retrieve (e.g. XA_CARDINAL, XA_WINDOW, etc.)
 * @param window is the window whose property we wish to examine
 */

long OBPager::getAtomAsLong(Atom xAtom, Atom reqType, Window window) const
{
	// Use the more-general getAtomAsLongs( ) to get the property
	
	std::vector<long> longs;
	
	getAtomAsLongs(xAtom, longs, reqType, window);
	
	
	// Should we throw if there is more than one?
	
	if (longs.size() <= 0) {
		std::ostringstream errMsgStream;
		
		errMsgStream << "Could not retrieve long value of X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << ")";
		
		THROW_VERBOSE_EXCEPTION(errMsgStream.str());
	}
	
	long oneLong = longs[0];
	
	return oneLong;
}



/**
 * Get the value of a window property as a vector of longs, using a specified Atom to reference the property
 *
 * @param xAtom is the Atom whose associated property we want to retrieve
 * @param longs is the vector to populate with the retrieved longs
 * @param reqType is the type of data to retrieve (e.g. XA_CARDINAL, XA_WINDOW, etc.)
 * @param window is the window whose property we wish to examine
 */

void OBPager::getAtomAsLongs(Atom xAtom, std::vector<long> &longs, Atom reqType, Window window) const
{
	Atom actualType;
	int actualFormat;
	unsigned long numItems, numBytesAfter;
	unsigned char* propertyData = NULL;
	
	
	// Default to the root window
	
	if (window == 0) {
		window = RootWindow(mDisplay(), mScreenNum);
	}
	
	#ifdef __DEBUG__
		std::cout << "Getting X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << "')" << std::endl;
	#endif

	
	// Get the property value (what to use as a max limit?!?!?!?!)
	
	int result = XGetWindowProperty(mDisplay(), window, xAtom, 0, 8192, False, reqType, &actualType, &actualFormat, &numItems, &numBytesAfter, &propertyData);
	
	
	// Stuff the data pointer into an XResourceKeeper so it will be freed-up automatically when it goes out of scope
	// This may seem like overkill, but it makes the code below a little cleaner
	
	XResourceKeeper<unsigned char> keptData(propertyData);
	
	
	// Did we get what we wanted?
	
	if (result != Success || actualType != reqType) {
		std::ostringstream errMsgStream;
		
		errMsgStream << "Cannot get X Property for Atom 0x" << std::hex << xAtom << std::dec << " = '" << XGetAtomName(mDisplay(), xAtom) << "', result = " << result << ", actualType = " << actualType;
		
		THROW_VERBOSE_EXCEPTION(errMsgStream.str());
	}
	
	#ifdef __DEBUG__
		std::cout << "X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << "'): numItems = " << numItems << ", actualType = " << actualType << " ('" << XGetAtomName(mDisplay(), actualType) << "'), format = " << actualFormat << ", numBytesAfter = " << numBytesAfter << std::endl;
	#endif
	
	if (actualFormat != 32 || numBytesAfter != 0) {
		std::ostringstream errMsgStream;
		
		errMsgStream << "Unexpected format/size for X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << "), ";
		errMsgStream << "format = " << actualFormat << ", numBytesAfter = " << numBytesAfter;
		
		THROW_VERBOSE_EXCEPTION(errMsgStream.str());
	}
	
	// Wow-- we got what we asked for, so stuff it into the callers vector
	
	for (unsigned int i = 0; i < numItems; ++i) {
		long resultLong = ((long*) propertyData)[i];
		
		#ifdef __DEBUG__
			std::cout << "X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << "') = " << resultLong << std::endl;
		#endif
		
		longs.push_back(resultLong);
	}
}



/**
 * Get the value of a window property as a vector of strings, using a specified Atom to reference the property
 *
 * @param xAtom is the Atom whose associated property we want to retrieve
 * @param strings is the vector to populate with the retrieved strings
 * @param reqType is the type of data to retrieve (e.g. XA_CARDINAL, XA_WINDOW, etc.)
 * @param window is the window whose property we wish to examine
 */

void OBPager::getAtomAsStrings(Atom xAtom, std::vector<std::string> &strings, Atom reqType, Window window) const
{
	Atom actualType;
	int actualFormat;
	unsigned long numItems, numBytesAfter;
	unsigned char* propertyData = NULL;
	
	// Default to the root window
	
	if (window == 0) {
		window = RootWindow(mDisplay(), mScreenNum);
	}
	
	#ifdef __DEBUG__
		std::cout << "Getting X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << "')" << std::endl;
	#endif

	
	// Get the property value (what to use as a max limit?!?!?!?!)
	
	int result = XGetWindowProperty(mDisplay(), window, xAtom, 0, 8192, False, reqType, &actualType, &actualFormat, &numItems, &numBytesAfter, &propertyData);
	
	
	// Stuff the data pointer into an XResourceKeeper so it will be freed-up automatically when it goes out of scope
	// This may seem like overkill, but it makes the code below a little cleaner
	
	XResourceKeeper<unsigned char> keptData(propertyData);
	
	
	// Did it work?
	
	if (result != Success || actualType != reqType) {
		std::ostringstream errMsgStream;
		
		errMsgStream << "Cannot get X Property for Atom 0x" << std::hex << xAtom << std::dec << " = '" << XGetAtomName(mDisplay(), xAtom) << "'";
		
		THROW_VERBOSE_EXCEPTION(errMsgStream.str());
	}
	
#ifdef __DEBUG__
		std::cout << "X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << "'): numItems = " << numItems << ", actualType = " << actualType << " ('" << XGetAtomName(mDisplay(), actualType) << "'), format = " << actualFormat << ", numBytesAfter = " << numBytesAfter << std::endl;
#endif
	
	if (actualFormat != 8 || numBytesAfter != 0) {
		std::ostringstream errMsgStream;
		
		errMsgStream << "Unexpected format/size for X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << "), ";
		errMsgStream << "format = " << actualFormat << ", numBytesAfter = " << numBytesAfter;
		
		THROW_VERBOSE_EXCEPTION(errMsgStream.str());
	}
	
	
	// All is well-- now extract the null-delimited strings and return them in the caller's vector
	
	std::string currentStr;
	
	for (unsigned int i = 0; i < numItems; ++i) {
		if (propertyData[i] != '\0') {
			currentStr += propertyData[i];
		}
		
		if (propertyData[i] == '\0' || i == numItems - 1) {
#ifdef __DEBUG__
	std::cout << "X Property for Atom 0x" << std::hex << xAtom << std::dec << " ('" << XGetAtomName(mDisplay(), xAtom) << "') = '" << currentStr << "'" << std::endl;
#endif
			
			strings.push_back(currentStr);
			
			currentStr.clear();
		}
	}
}



/**
 * Redraw the pager dockapp window
 */

void OBPager::redrawWindow()
{
	setCurrentAction("Redrawing X window");
	
	
	// We need to reference the root window a lot, so make it easy
	
	Window rootWindow = RootWindow(mDisplay(), mScreenNum);
	
	Window rootReturn;
	int x, y;
	unsigned int border, depth;
	unsigned int pixmapLeft, pixmapTop, pixmapWidth, pixmapHeight;
	
	// How big is the pixmap again?  We could cache this value, but this is easy and makes things dynamic, should we ever allow resizing
	
	if (!XGetGeometry(mDisplay(), mPixmap(), &rootReturn, &x, &y, &pixmapWidth, &pixmapHeight, &border, &depth)) {
		THROW_VERBOSE_EXCEPTION("Can't get geometry of pixmap");
	}
	
	// We actually indent a little bit to support the nifty transparent border in the dock/slit
	
	pixmapLeft = APPLET_MARGIN_WIDTH;
	pixmapTop = APPLET_MARGIN_WIDTH;
	
	pixmapWidth -= 2*APPLET_MARGIN_WIDTH;
	pixmapHeight -= 2*APPLET_MARGIN_WIDTH;
	
	
	// All drawing is done in the pixmap first and then blitted onscreen (minimizes flicker)
	
	
	// Start by clearing the pixmap
	
	XSetForeground(mDisplay(), mGC(), BlackPixel(mDisplay(), mScreenNum));
	
	XFillRectangle(mDisplay(), mPixmap(), mGC(), pixmapLeft, pixmapTop, pixmapWidth, pixmapHeight);
	
	
	// Which desktop is current, and which window is focused?
	
	setCurrentAction("Determining current desktop and active window");
	
	long currentDesktop = getAtomAsLong(mAtom_NET_CURRENT_DESKTOP, XA_CARDINAL);
	Window activeWindow = getAtomAsLong(mAtom_NET_ACTIVE_WINDOW, XA_WINDOW);
	
	
	// How big is the desktop?
	
	setCurrentAction("Determining desktop size");
	
	std::vector<long> tempLongs;
	
	getAtomAsLongs(mAtom_NET_DESKTOP_GEOMETRY, tempLongs, XA_CARDINAL);
	
	int displayWidth = tempLongs[0];
	int displayHeight = tempLongs[1];
	
	
	// Get the list of windows the window manager knows about
	
	setCurrentAction("Getting list of desktop windows");
	
	tempLongs.clear();
	
	getAtomAsLongs(mAtom_NET_CLIENT_LIST_STACKING, tempLongs, XA_WINDOW);
	
	
	// Draw each window that is on our desktop
	
	for (int drawPass = 0; drawPass < 2; ++drawPass) {
		for (std::vector<long>::const_iterator iter = tempLongs.begin(); iter != tempLongs.end(); ++iter) {
			Window window = *iter;
			
			setCurrentAction("Determining desktop number of window", window);
			
			long desktopNum = -1;
			unsigned int width, height;
			
			
			// Catch any exceptions, since sometimes the window manager is a bit slow in updating NET_CLIENT_LIST_STACKING, and includes references to windows that have gone away....
			
			try {
				desktopNum = getAtomAsLong(mAtom_NET_WMDESKTOP, XA_CARDINAL, window);
				
				// Not our desktop?
				
				if (mDesktopNum >= 0 && desktopNum != mDesktopNum) {
					continue;
				}
				
				
				// How big is the window, and where is it located?
				
				if (!XGetGeometry(mDisplay(), window, &rootReturn, &x, &y, &width, &height, &border, &depth)) {
					std::ostringstream errMsgStream;
					
					errMsgStream << "Can't get geometry of window 0x" << std::hex << window << std::dec;
					
					THROW_VERBOSE_EXCEPTION(errMsgStream.str());
				}
				
				
				Window childReturn;
				
				if (!XTranslateCoordinates(mDisplay(), window, rootWindow, x, y, &x, &y, &childReturn)) {
					std::ostringstream errMsgStream;
					
					errMsgStream << "Can't translate coordinates for window 0x" << std::hex << window << std::dec;
					
					THROW_VERBOSE_EXCEPTION(errMsgStream.str());
				}
			}
			catch (...) {
				continue;
			}
			
			
			// Calculate the scaled size and loaction of the window
			
			x = ((pixmapWidth - 1) * x) / displayWidth;
			y = ((pixmapHeight - 1) * y) / displayHeight;
			width = ((pixmapWidth - 1) * width) / displayWidth;
			height = ((pixmapHeight - 1) * height) / displayHeight;
			
			// Draw the window differently, depending on the state of things
			
			if (currentDesktop != mDesktopNum) {
				// Our desktop is not showing, so draw things appropriately
				
				if (drawPass == 0) {
					XSetForeground(mDisplay(), mGC(), mPixelColours[kInactiveDesktopWindowIndex]);
					XFillRectangle(mDisplay(), mPixmap(), mGC(), pixmapLeft + x, pixmapTop + y, width, height);
				}
				else {
					XSetForeground(mDisplay(), mGC(), mPixelColours[kInactiveDesktopWindowBorderIndex]);
					XDrawRectangle(mDisplay(), mPixmap(), mGC(), pixmapLeft + x, pixmapTop + y, width, height);
				}
			}
			else if (window == activeWindow) {
				// Ah-- our desktop is visible, and this window has focus
				
				if (drawPass == 0) {
					XSetForeground(mDisplay(), mGC(), mPixelColours[kActiveDesktopFocusedWindowIndex]);
					XFillRectangle(mDisplay(), mPixmap(), mGC(), pixmapLeft + x, pixmapTop + y, width, height);
				}
				else {
					XSetForeground(mDisplay(), mGC(), mPixelColours[kActiveDesktopFocusedBorderIndex]);
					XDrawRectangle(mDisplay(), mPixmap(), mGC(), pixmapLeft + x, pixmapTop + y, width, height);
				}
			}
			else {
				// Our desktop is visible, but this window does not have focus
				
				if (drawPass == 0) {
					XSetForeground(mDisplay(), mGC(), mPixelColours[kActiveDesktopUnfocusedWindowIndex]);
					XFillRectangle(mDisplay(), mPixmap(), mGC(), pixmapLeft + x, pixmapTop + y, width, height);
				}
				else {
					XSetForeground(mDisplay(), mGC(), mPixelColours[kActiveDesktopUnfocusedBorderIndex]);
					XDrawRectangle(mDisplay(), mPixmap(), mGC(), pixmapLeft + x, pixmapTop + y, width, height);
				}
			}
		}
	}
	
	
	// Draw the desktop number using the appropriate font
	
	char text[2] = { (mDesktopNum % 9) + '1', '\0' };
	int textLength = strlen(text);
	int textWidth, textHeight;
	int direction, ascent, descent;
	XCharStruct overallCharStruct;
	
	if (currentDesktop == mDesktopNum) {
		// Our desktop is visible, so figure out the text width/height, and set the font/colour
		
		XTextExtents(mFontStructLarge(), text, textLength, &direction, &ascent, &descent, &overallCharStruct);
		
		textWidth = overallCharStruct.width;
		textHeight = overallCharStruct.ascent + overallCharStruct.descent;
		
		XGCValues gcValues;
		
		gcValues.font = mFontStructLarge()->fid;
		gcValues.foreground = mPixelColours[kActiveDesktopFontColor];
		unsigned long gcValueMask = GCFont | GCForeground;
		
		XChangeGC(mDisplay(), mGC(), gcValueMask, &gcValues);
	}
	else {
		// Our desktop is not visible, so figure out the text width/height, and set the font/colour
		
		XTextExtents(mFontStructSmall(), text, textLength, &direction, &ascent, &descent, &overallCharStruct);
		
		textWidth = overallCharStruct.width;
		textHeight = overallCharStruct.ascent + overallCharStruct.descent;
		
		XGCValues gcValues;
		
		gcValues.font = mFontStructSmall()->fid;
		gcValues.foreground = mPixelColours[kInactiveDesktopFontColor];
		unsigned long gcValueMask = GCFont | GCForeground;
		
		XChangeGC(mDisplay(), mGC(), gcValueMask, &gcValues);
	}
	
	// Draw the number centered onscreen
	
	int leftX = (pixmapLeft + pixmapWidth / 2) - (textWidth / 2);
	int topY = (pixmapTop + pixmapHeight / 2) + (textHeight / 2);
	XDrawString(mDisplay(), mPixmap(), mGC(), leftX, topY, text, textLength);
	
	
	// Draw the pretty drop-shadow
	
	XSetForeground(mDisplay(), mGC(), mPixelColours[kAppletBorderShadowIndex]);
	
	XDrawLine(mDisplay(), mPixmap(), mGC(), pixmapLeft, pixmapTop + pixmapHeight - 1, pixmapLeft + pixmapWidth - 1, pixmapTop + pixmapHeight - 1);
	XDrawLine(mDisplay(), mPixmap(), mGC(), pixmapLeft + pixmapWidth - 1, pixmapTop + pixmapHeight - 1, pixmapLeft + pixmapWidth - 1, pixmapTop + 1);
	
	
	// And blit the result to the screen
	
	XCopyArea(mDisplay(), mPixmap(), mWindow(), mGC(), pixmapLeft, pixmapTop, pixmapWidth, pixmapHeight, pixmapLeft, pixmapTop);
}



/**
 * Process events forever....
 */

void OBPager::run()
{
	bool alive = true;
	
	while (alive) {
		XEvent event;
		
		// Get the next event and process it
		
		XNextEvent(mDisplay(), &event);
		
		switch (event.type) {
			case Expose: {
				// Eat any other expose events to avoid unnecessary redrawing
				
				while (XCheckTypedEvent(mDisplay(), Expose, &event)) {
					// Do nothing....
				}
				
				if (!mExposed) {
					mExposed = true;
					
					if (mFD > 0) {
						std::stringstream msg;
						
						msg << "I am the keeper of desktop #" << mDesktopNum;
						
						write(mFD, msg.str().c_str(), msg.str().size());
						
						close(mFD);
						
						mFD = 0;
					}
				}
				
				redrawWindow();
				
				break;
			}
			
			case ButtonPress: {
				// Ooh-- user wants to switch desktops!
				
				gotoDesktop(mDesktopNum);
				
				break;
			}
			
			case PropertyNotify: {
				// Aha-- something important changed on the desktop-- better redraw it....
				
				if (event.xproperty.atom == mAtom_NET_CURRENT_DESKTOP) {
					redrawWindow();
				}
				
				else if (event.xproperty.atom == mAtom_NET_CLIENT_LIST_STACKING) {
					redrawWindow();
				}
				
				break;
			}
			
			case CirculateNotify:
			case ConfigureNotify:{
				// Size/order of windows changed-- redraw....
				
				redrawWindow();
				
				break;
			}
		}
	}
}



/**
 * Ask the window manager to change to a different virtual desktop
 *
 * @param newDesktop is the number of the virtual desktop our caller wants to switch to
 */

void OBPager::gotoDesktop(int newDesktop) const
{
	// Send a message to the root window, hoping the window manager is listening....
	
	Window rootWindow = RootWindow(mDisplay(), mScreenNum);
	
	XEvent event;
	
	event.type = ClientMessage;
	event.xclient.type = ClientMessage;
	event.xclient.window = rootWindow;
	event.xclient.message_type= mAtom_NET_CURRENT_DESKTOP;
	event.xclient.format = 32;
	event.xclient.data.l[0] = newDesktop;
	
	XSendEvent(mDisplay(), rootWindow, False, SubstructureNotifyMask, &event);
}



/**
 * Determine the number of virtual desktops
 */

long OBPager::numDesktops()
{
	setCurrentAction("Determining number of desktops");
	
	long numDesktops = getAtomAsLong(mAtom_NET_NUMBER_OF_DESKTOPS, XA_CARDINAL);
	
	return numDesktops;
}



/*
void OBPager::dumpEventInfo(const XEvent& event)
{
	// Which window?
	
	Display *display = event.xany.display;
	Window rootWindow = RootWindow(display, mScreenNum);
	Window window = event.xany.window;
	
	std::cout << "Window = 0x" << std::hex << window << std::dec << std::endl;

	
	// Location and size?
	
	Window rootReturn, childReturn;
	int x, y;
	unsigned int width, height, border, depth;
	
	XGetGeometry(display, window, &rootReturn, &x, &y, &width, &height, &border, &depth);
	
	XTranslateCoordinates(display, window, rootWindow, x, y, &x, &y, &childReturn);
	
	std::cout << "    (x, y) = (" << x << "," << y << ")" << std::endl;
	std::cout << "    width = " << width << ", height = " << height << std::endl;	
	
	
	// Which desktop?
	try {
		long desktopNum = getAtomAsLong(mAtom_NET_WMDESKTOP, XA_CARDINAL, window);
		
		std::cout << "    desktop #" << desktopNum << std::endl;
	}
	catch (...) {
		std::cout << "    desktop #????" << std::endl;
	}
	
	// Name of window?
	
	try {
		std::vector<std::string> names;
		
		getAtomAsStrings(mAtom_WM_NAME, names, XA_STRING, window);
		
		std::cout << "    name = '" << names[0] << "'" << std::endl;
	}
	catch (...) {
		std::cout << "    name = ????" << std::endl;
	}
}
*/
