/*
 * util.c
 *
 * Copyright (c) 2002-2009 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: util.c,v 1.18 2010/11/03 18:28:50 max Exp $
 * $FreeBSD$
 */

#include <bluetooth.h>
#include <ctype.h>
#include <errno.h>
#include <expat.h>
#include <iconv.h>
#include <langinfo.h>
#include <libgen.h>
#include <limits.h>
#include <locale.h>
#include <obex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <readline/readline.h>
#include <readline/history.h>

#include "compat.h"
#include "obexapp.h"
#include "log.h"
#include "util.h"

/*
 * Init locale
 */

static int32_t		 utf16 = 0;
static iconv_t		 utf82locale = (iconv_t) -1;
static iconv_t		 locale2utf8 = (iconv_t) -1;
static iconv_t		 utf16be2locale = (iconv_t) -1;
static iconv_t		 locale2utf16be = (iconv_t) -1;

int
obexapp_util_locale_init(void)
{
	char const	*locale;

	setlocale(LC_CTYPE, "");
	locale = nl_langinfo(CODESET);

	utf16 = (strstr(locale, "UTF-16") != NULL)? 1 : 0;

	/* UTF-8 -> current locale */
	utf82locale = iconv_open(locale, "UTF-8");
	if (utf82locale == (iconv_t) -1)
		return (-1);

	iconv(utf82locale, NULL, NULL, NULL, NULL);

	/* current locale -> UTF-8 */
	locale2utf8 = iconv_open("UTF-8", locale);
	if (locale2utf8 == (iconv_t) -1) {
		obexapp_util_locale_fini();
		return (-1);
	}

	iconv(locale2utf8, NULL, NULL, NULL, NULL);

	/* UTF-16BE -> current locale */
	utf16be2locale = iconv_open(locale, "UTF-16BE");
	if (utf16be2locale == (iconv_t) -1) {
		obexapp_util_locale_fini();
		return (-1);
	}

	iconv(utf16be2locale, NULL, NULL, NULL, NULL);

	/* current locale -> UTF-16BE */
	locale2utf16be = iconv_open("UTF-16BE", locale);
	if (locale2utf16be == (iconv_t) -1) {
		obexapp_util_locale_fini();
		return (-1);
	}

	iconv(locale2utf16be, NULL, NULL, NULL, NULL);

	return (0);
}

/*
 * Fini locale
 */

void
obexapp_util_locale_fini(void)
{
	if (utf82locale != (iconv_t) -1) {
		iconv_close(utf82locale);
		utf82locale = (iconv_t) -1;
	}

	if (locale2utf8 != (iconv_t) -1) {
		iconv_close(locale2utf8);
		locale2utf8 = (iconv_t) -1;
	}

	if (utf16be2locale != (iconv_t) -1) {
		iconv_close(utf16be2locale);
		utf16be2locale = (iconv_t) -1;
	}

	if (locale2utf16be != (iconv_t) -1) {
		iconv_close(locale2utf16be);
		locale2utf16be = (iconv_t) -1;
	}
}

/*
 * Convert string from UTF-8 to current locale
 */

int
obexapp_util_locale_from_utf8(char const *src, size_t srclen,
		char *dst, size_t dstlen)
{
	char const	*s = src;
	char		*d = dst;
	int		 n;

	if (utf82locale == (iconv_t) -1) {
		*d = '\0';
		return (-1);
	}

	iconv(utf82locale, NULL, NULL, NULL, NULL);

	do {
		n = iconv(utf82locale, &s, &srclen, &d, &dstlen);
		if (n < 0) {
			*d = '\0';
			return (-1);
		}
	} while (n > 0);

	*d = '\0';

	return (0);
}

/*
 * Convert string to UTF-8 from current locale
 */

int
obexapp_util_locale_to_utf8(char const *src, size_t srclen,
		char *dst, size_t dstlen)
{
	char const	*s = src;
	char		*d = dst;
	int		 n;

	if (locale2utf8 == (iconv_t) -1) {
		*d = '\0';
		return (-1);
	}

	iconv(locale2utf8, NULL, NULL, NULL, NULL);

	do {
		n = iconv(locale2utf8, &s, &srclen, &d, &dstlen);
		if (n < 0) {
			*d = '\0';
			return (-1);
		}
	} while (n > 0);

	*d = '\0';

	return (0);
}

/*
 * Convert string from UTF-16BE to current locale
 */

int
obexapp_util_locale_from_utf16be(uint8_t const *src, size_t srclen,
		char *dst, size_t dstlen)
{
	char const	*s = (char const *) src;
	char		*d = dst;
	int		 n;

	if (utf16be2locale == (iconv_t) -1) {
		*d = '\0';
		return (-1);
	}

	iconv(utf16be2locale, NULL, NULL, NULL, NULL);

	do {
		n = iconv(utf16be2locale, &s, &srclen, &d, &dstlen);
		if (n < 0) {
			*d = '\0';
			return (-1);
		}
	} while (n > 0);

	*d = '\0';

	if (utf16) {
		if (dstlen == 0)
			return (-1); /* buffer is too short */

		d ++;
		dstlen --;

		*d = '\0';
	}

	return (0);
}

/*
 * Convert string to UTF-16BE from current locale
 */

int
obexapp_util_locale_to_utf16be(char const *src, size_t srclen,
		uint8_t *dst, size_t dstlen)
{
	char const	*s = src;
	char		*d = (char *) dst;
	int		 n;

	if (locale2utf16be == (iconv_t) -1) {
		*d = '\0';
		return (-1);
	}

	iconv(locale2utf16be, NULL, NULL, NULL, NULL);

	do {
		n = iconv(locale2utf16be, &s, &srclen, &d, &dstlen);
		if (n < 0) {
			*d = '\0';
			return (-1);
		}
	} while (n > 0);

	*d = '\0';

	if (dstlen == 0)
		return (-1); /* buffer is too short */

	d ++;
	dstlen --;

	*d = '\0';

	return (0);
}

/*
 * Read upto buffer_length bytes into the buffer from the file descriptor fd
 */

ssize_t
obexapp_util_read(int fd, uint8_t *buffer, size_t buffer_length)
{
	ssize_t length;

again:
	length = read(fd, buffer, buffer_length);
	if (length == -1 && errno == EINTR)
		goto again;

	return (length);
} /* obexapp_util_read */

/*
 * Write buffer_length bytes from the buffer to the descriptor fd
 */

ssize_t
obexapp_util_write(int fd, uint8_t const *buffer, size_t buffer_length)
{
	ssize_t size;
	size_t wrote;

	wrote = 0;
	while (wrote < buffer_length) {
		size = write(fd, buffer + wrote, buffer_length - wrote);
		if (size == -1) {
			if (errno == EINTR)
				continue;

			return (-1);
		}

		wrote += size;
	}

	return (wrote);
} /* obexapp_util_write */

/*
 * Get a string from stdin and chop all non-print chars at the end
 */

char *
obexapp_util_gets(char const *prompt, char *buffer, int buffer_size)
{
	char	*s;

	s = readline(prompt);
	if (s == NULL)
		return (NULL);

	strlcpy(buffer, s, buffer_size);
	add_history(s);
	free(s);

	return (buffer);
} /* obexapp_util_gets */

/*
 * Get next token from string. Just like strsep(3) only with quotation support
 */

char *
obexapp_util_strsep(char **sp, char const *delim)
{
	char	*s = *sp;
	char	*t = s;

	if (s == NULL)
		return (NULL);

	for (*sp += strspn(*sp, delim); **sp != '\0'; (*sp) ++) {
		switch (**sp) {
		case '\\':
			(*sp) ++;
			if ((**sp) == '\0')
				return (NULL);	/* unterminated quote */

			*t = **sp;
			break;

		default:
			if (strchr(delim, **sp) != NULL)
				goto out;	/* end of token */

			*t = **sp;
			break;
		}

		t ++;
	}
out:
	if (t == s)
		return (NULL);	/* no token */

	if (**sp != '\0')
		(*sp) ++;

	*t = '\0';

	return (s);
} /* obexapp_util_strsep */

/*
 * Make temp file name
 */

int
obexapp_util_mkstemp(char const *name, char *temp, int temp_size)
{
	char	n[PATH_MAX];

	/*
	 * Do not rely on dirname(3) accepting const pointer.
	 * Some dirname(3) implementations can modify passed
	 * string, so always pass a copy.
	 */

	strlcpy(n, name, sizeof(n));
	snprintf(temp, temp_size, "%s/XXXXXXXX", dirname(n));

	return (mkstemp(temp));
} /* obexapp_util_mkstemp */

/*
 * Parse XML and print folder listing
 *
 * XXX	libbsdxml(3), which is a verbatim copy of the eXpat XML library version
 *	1.95.5, uses UTF-8 econding internally. This means all XML_Char strings
 *	are UTF-8 encoded. Because OBEX folder listing XML document mostly uses
 *	characters from English alphabet we can get away without coverting most
 * 	of the document back to current locale. The only part of the document
 *	that needs to be converted is the content of the "name" attribute. This
 * 	attribute contains folder and file names that might have national UTF-8
 *	encoded characters.
 */

static int
obexapp_util_xml_unknown_encoding_handler(__unused void *data,
		XML_Char const *name, __unused XML_Encoding *info)
{
	log_err("%s(): Do not know how to handle encoding %s", __func__, name);

	return (0);
}

static int
obexapp_util_xml_string2uint(XML_Char const *string, uint32_t length)
{
	XML_Char	b[16], *e = NULL;
	int		v;

	if (length >= sizeof(b))
		return (-1);

	strncpy(b, string, length);
	b[length] = '\0';

	v = strtol(b, &e, 10);
	if (v == 0 && *e != '\0')
		return (-1);

	return (v);
} /* obexapp_util_xml_string2uint */

static XML_Char * 
obexapp_util_xml_iso8601time2string(XML_Char const *iso8601time,
		XML_Char *string, int string_length)
{
	struct tm	tm;
	time_t		t;

	memset(&tm, 0, sizeof(tm));

	if (strlen(iso8601time) < 15) /* YYYYMMDDThhmmss[z] */
		return (NULL);

	/* Year (YYYY) */
	if ((tm.tm_year = obexapp_util_xml_string2uint(iso8601time, 4)) < 0) 
		return (NULL);
	tm.tm_year -= 1900;
	iso8601time += 4;

	/* Month (MM) */
	if ((tm.tm_mon = obexapp_util_xml_string2uint(iso8601time, 2)) < 0) 
		return (NULL);
	tm.tm_mon -= 1;
	iso8601time += 2;

	/* Day (DD) */
	if ((tm.tm_mday = obexapp_util_xml_string2uint(iso8601time, 2)) < 0) 
		return (NULL);
	iso8601time += 2;

	if (tolower((int)*iso8601time) != 't')
		return (NULL);
	iso8601time ++;

	/* Hour (HH) */
	if ((tm.tm_hour = obexapp_util_xml_string2uint(iso8601time, 2)) < 0) 
		return (NULL);
	iso8601time += 2;

	/* Minute (MM) */
	if ((tm.tm_min = obexapp_util_xml_string2uint(iso8601time, 2)) < 0) 
		return (NULL);
	iso8601time += 2;

	/* Second (SS) */
	if ((tm.tm_sec = obexapp_util_xml_string2uint(iso8601time, 2)) < 0) 
		return (NULL);
	iso8601time += 2;

	switch (tolower((int)*iso8601time)) {
	case 0:
		t = mktime(&tm);
		strftime(string, string_length, "%d-%b-%y %H:%M", localtime(&t));
		break;

	case 'z':
		t = timegm(&tm);
		strftime(string, string_length, "%d-%b-%y %H:%MG", gmtime(&t));
		break;

	default:
		return (NULL);
		/* NOT REACHED */
	}

	return (string);
} /* obexapp_util_xml_iso8601time2string */

static XML_Char const * 
obexapp_util_xml_get_attribute(XML_Char const *attr, XML_Char const **attrs)
{
	int	i;

	for (i = 0; attrs[i] != NULL; i += 2)
		if (strcasecmp(attrs[i], attr) == 0)
			return (attrs[i + 1]);

	return (NULL);
} /* obexapp_util_xml_get_attribute */

static void
obexapp_util_xml_element_start(__unused void *x, XML_Char const *name,
		XML_Char const **attrs)
{
	int		 folder;
	XML_Char const	*uperm = NULL, *gperm = NULL, *operm = NULL,
			*owner = NULL, *group = NULL, *size = NULL,
			*mtime = NULL, *fname = NULL;
	XML_Char	 mtime_buf[32], fname_buf[PATH_MAX];

	if ((folder = strcasecmp(name, "file")) == 0 ||
	    strcasecmp(name, "folder") == 0) {
		uperm = obexapp_util_xml_get_attribute("user-perm", attrs);
		gperm = obexapp_util_xml_get_attribute("group-perm", attrs);
		operm = obexapp_util_xml_get_attribute("other-perm", attrs);
		owner = obexapp_util_xml_get_attribute("owner", attrs);
		group = obexapp_util_xml_get_attribute("group", attrs);
		size  = obexapp_util_xml_get_attribute("size", attrs);

		mtime = obexapp_util_xml_get_attribute("modified", attrs);
		if (mtime != NULL)
			mtime = obexapp_util_xml_iso8601time2string(mtime,
					mtime_buf, sizeof(mtime_buf));

		fname = obexapp_util_xml_get_attribute("name", attrs);
		if (fname != NULL) {
			if (obexapp_util_locale_from_utf8(fname, strlen(fname),
					fname_buf, sizeof(fname_buf)) < 0) {
				log_err("%s(): Could not convert from UTF-8",
					__func__);

				fname = NULL;
			} else
				fname = fname_buf;
		}
	} else if (strcasecmp(name, "parent-folder") == 0) {
		folder = 0;
		fname  = "..";
		owner  = " ";
		group  = " ";
		size   = " ";
		mtime  = " ";
	} else
		return;

	printf("%3s%3s%3s %-8.8s %-8.8s %-10.10s %-16.16s %s%s\n",
		uperm? uperm : " ", gperm? gperm : " ", operm? operm : " ",
		owner? owner : "n/a", group? group : "n/a",
		size ? size  : "n/a",
		mtime? mtime : "n/a",
		fname? fname : "???",
		folder?  "/" : "");
} /* obexapp_util_xml_element_start */

static void
obexapp_util_xml_element_end(__unused void *x, __unused XML_Char const *name)
{
} /* obexapp_util_xml_element_end */

int
obexapp_util_display_folder_listing(char const *ls)
{
	XML_Parser	 p = NULL;
	void		*b = NULL;
	int		 len = strlen(ls), error = -1;

	if ((p = XML_ParserCreate(NULL)) == NULL)
		goto out;

	XML_SetUnknownEncodingHandler(p,
		obexapp_util_xml_unknown_encoding_handler, NULL);

	XML_SetElementHandler(p,
		obexapp_util_xml_element_start,
		obexapp_util_xml_element_end);

	if ((b = XML_GetBuffer(p, len + 1)) == NULL)
		goto out;

	strcpy(b, ls);

	printf("%-9.9s %-8.8s %-8.8s %-10.10s %-16.16s %s\n",
"Access", "Owner", "Group", "Size", "Modified", "Name");

	if ((error = XML_ParseBuffer(p, len, 1)) != XML_STATUS_OK) {
		printf("Could not parse XML: %s\n", XML_ErrorString(error));
		error = -1;
	} else
		error = 0;
out:
	if (p != NULL) {
		XML_ParserFree(p);
		p = NULL;
	}

	return (error);
} /* obexapp_util_display_folder_listing */

/*
 * Translate errno codes into OBEX response codes
 */

int
obexapp_util_errno2response(int error)
{
	switch (error) {
	case ENOENT:
		return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_NOT_FOUND,
					OBEX_RSP_NOT_FOUND));

	case ENOTEMPTY:
	case EACCES:
		return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_FORBIDDEN,
					OBEX_RSP_FORBIDDEN));
	}

	return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_INTERNAL_SERVER_ERROR,
				OBEX_RSP_INTERNAL_SERVER_ERROR));
} /* obexapp_util_errno2response */

