/*
 * transport.c
 *
 * Copyright (c) 2001-2009 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: transport.c,v 1.16 2010/11/03 18:28:50 max Exp $
 * $FreeBSD$
 */

#include <sys/time.h>

#include <bluetooth.h>
#include <errno.h>
#include <obex.h>
#include <fcntl.h>
#include <sdp.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "compat.h"
#include "obexapp.h"
#include "log.h"
#include "transport.h"
#include "util.h"

/*
 * Transport connect callback
 */

int
obexapp_transport_connect(__unused obex_t *handle, void *userdata)
{
	context_p		context = (context_p) userdata;
	struct sockaddr_rfcomm	addr;

	log_debug("%s()", __func__);

	if (context->direct) {
		log_err("%s(): Can not use stdin/stdout in client mode",
			__func__);
		return (-1);
	}

	log_info("%s(): Connecting to %s/%d, pid=%d",
		__func__, bt_ntoa(&context->raddr, NULL),
		context->channel, getpid());

	context->tfd = socket(PF_BLUETOOTH,SOCK_STREAM,BLUETOOTH_PROTO_RFCOMM);
	if (context->tfd < 0) {
		log_err("%s(): Could not create socket. %s (%d)",
			__func__,  strerror(errno), errno);
		return (-1);
	}

	memset(&addr, 0, sizeof(addr));
	addr.rfcomm_len = sizeof(addr);
	addr.rfcomm_family = AF_BLUETOOTH;
	memcpy(&addr.rfcomm_bdaddr, &context->laddr, sizeof(addr.rfcomm_bdaddr));
	addr.rfcomm_channel = 0;

	if (bind(context->tfd, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		log_err("%s(): Could not bind socket. %s (%d)",
			__func__,  strerror(errno), errno);

		close(context->tfd);
		context->tfd = -1;

		return (-1);
	}

	memcpy(&addr.rfcomm_bdaddr, &context->raddr, sizeof(addr.rfcomm_bdaddr));
	addr.rfcomm_channel = context->channel;

	if (connect(context->tfd, (struct sockaddr *) &addr,
			sizeof(addr)) < 0) {
		log_err("%s(): Could not connect socket. %s (%d)",
			__func__,  strerror(errno), errno);

		close(context->tfd);
		context->tfd = -1;

		return (-1);
	}

	return (1);
} /* obexapp_transport_connect */

/*
 * Transport disconnect callback
 */

int
obexapp_transport_disconnect(__unused obex_t *handle, void *userdata)
{
	context_p	context = (context_p) userdata;

	log_debug("%s()", __func__);

	if (context->tfd < 0)
		return (1);

	/*
	 * This is an ugly hack for broken OBEX clients. It seems that few
	 * OBEX clients think that since they have initiated connection they 
	 * have to close it. The scenario is as follows: client sends OBEX
	 * disconnect request. We reply with OBEX disconnect response. Client
	 * sees out reply and reports successful transfer. Now client wants
	 * to disconnect the socket, but since we are done we close the socket
	 * (issue RFCOMM DISC) *before* the client.
	 * 
	 * Here we will try to select() socket to give remote peer a chance to 
	 * close connection, but we will not wait forever.
	 */

	if (context->server) {
		fd_set		rfds;
		struct timeval	tm;

		FD_ZERO(&rfds);
		FD_SET(context->tfd, &rfds);

		tm.tv_sec = 5;
		tm.tv_usec = 0;

		select(context->tfd + 1, &rfds, NULL, NULL, &tm);
	}

	close(context->tfd);
	context->tfd = -1;

	return (1);
} /* obexapp_transport_disconnect */

/*
 * Transport listen callback
 */

int
obexapp_transport_listen(__unused obex_t *handle, void *userdata)
{
	context_p		context = (context_p) userdata;
	struct sockaddr_rfcomm	addr;
	socklen_t		addrlen;
	pid_t			pid;
	int			s;
	sdp_opush_profile_t	opush;
	sdp_ftrn_profile_t	ftrn;

	log_debug("%s()", __func__);

	if (context->direct) { /* Use stdin */
		log_info("%s(): using stdin", __func__);
		context->tfd = STDIN_FILENO;
		return (1);
	}

	s = socket(PF_BLUETOOTH, SOCK_STREAM, BLUETOOTH_PROTO_RFCOMM);
	if (s < 0) {
		log_err("%s(): Could not create socket. %s (%d)",
			__func__,  strerror(errno), errno);
		return (-1);
	}

	memset(&addr, 0, sizeof(addr));
	addr.rfcomm_len = sizeof(addr);
	addr.rfcomm_family = AF_BLUETOOTH;
	addr.rfcomm_channel = context->channel;
	memcpy(&addr.rfcomm_bdaddr, &context->laddr, sizeof(context->laddr));

	if (bind(s, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		log_err("%s(): Could not bind socket. %s (%d)",
			__func__,  strerror(errno), errno);
		return (-1);
	}

	if (listen(s, 10) < 0) {
		log_err("%s(): Could not listen on socket. %s (%d)",
			__func__,  strerror(errno), errno);
		return (-1);
	}

	if (context->channel == 0) {
		addrlen = sizeof(addr);
		if (getsockname(s, (struct sockaddr *) &addr, &addrlen) < 0) {
			log_err("%s(): Could not get socket address. %s (%d)",
				__func__,  strerror(errno), errno);
			return (-1);
		}
		context->channel = addr.rfcomm_channel;
	}

	memset(&opush, 0, sizeof(opush));
	opush.server_channel = context->channel;
	opush.supported_formats_size = 7;
	opush.supported_formats[0] = 0x1;
	opush.supported_formats[1] = 0x2;
	opush.supported_formats[2] = 0x3;
	opush.supported_formats[3] = 0x4;
	opush.supported_formats[4] = 0x5;
	opush.supported_formats[5] = 0x6;
	opush.supported_formats[6] = 0xff; /* Any format */

	if (sdp_register_service(context->ss,
			SDP_SERVICE_CLASS_OBEX_OBJECT_PUSH,
			&context->laddr,
			(void *) &opush, sizeof(opush), NULL) != 0) {
		log_err("%s(): Could not register OPUSH service with " \
			"local SDP daemon. %s (%d)",
			__func__, strerror(sdp_error(context->ss)),
			sdp_error(context->ss));
		sdp_close(context->ss);
		context->ss = NULL;

		return (-1);
	}

	memset(&ftrn, 0, sizeof(ftrn));
	ftrn.server_channel = context->channel;

	if (sdp_register_service(context->ss,
			SDP_SERVICE_CLASS_OBEX_FILE_TRANSFER,
			&context->laddr,
			(void *) &ftrn, sizeof(ftrn), NULL) != 0) {
		log_err("%s(): Could not register FTRN service with " \
			"local SDP daemon. %s (%d)",
			__func__, strerror(sdp_error(context->ss)),
			sdp_error(context->ss));
		sdp_close(context->ss);
		context->ss = NULL;

		return (-1);
	}

	while (1) {
		addrlen = sizeof(addr);
		context->tfd = accept(s, (struct sockaddr *) &addr, &addrlen);
		if (context->tfd < 0) {
			log_err("%s(): Could not accept on socket. %s (%d)",
				__func__,  strerror(errno), errno);
			goto out;
		}

		if ((pid = fork()) == (pid_t) -1) {
			log_err("%s(): Could not fork. %s (%d)",
				__func__,  strerror(errno), errno);
			goto out;
		}

		if (pid == 0) {
			sdp_close(context->ss);
			context->ss = NULL;

			close(s);

			log_info("%s(): Accepted new connection from " \
				"%s/%d, pid=%d", __func__,
				bt_ntoa(&addr.rfcomm_bdaddr, NULL),
				addr.rfcomm_channel, getpid());

			if (daemon(1, 0) < 0) {
				log_err("%s(): Could not daemon. %s (%d)",
					__func__, strerror(errno), errno);

				close(context->tfd);
				context->tfd = -1;

				return (-1);
			}

			memcpy(&context->raddr, &addr.rfcomm_bdaddr,
				sizeof(context->raddr));

			return (1);
		}

		close(context->tfd);
		context->tfd = -1;
	}
out:
	close(s);

	return (-1);
} /* obexapp_transport_listen */

/*
 * Transport write callback
 */

int
obexapp_transport_write(__unused obex_t *handle, void *userdata, 
		uint8_t *buffer, int length) 
{
	context_p	context = (context_p) userdata;
	int		wrote;

	if (context->tfd == -1) {
		log_err("%s(): Invalid transport descriptor", __func__);
		return (-1);
	}

	wrote = obexapp_util_write(context->tfd, buffer, length);
	if (wrote < 0) {
		log_err("%s(): Could write(%d). %s (%d)",
			__func__, context->tfd, strerror(errno), errno);

		return (-1);
	}

	log_debug("%s(): wrote %d bytes", __func__, wrote);

	return (wrote);
} /* obexapp_transport_write */

/*
 * Transport HandleInput callback
 */

int
obexapp_transport_handle_input(obex_t *handle, void *userdata,
		__unused int timeout)
{
	context_p	context = (context_p) userdata;
	int		n;
	fd_set		rfds;

	log_debug("%s()", __func__);

	if (context->tfd == -1) {
		log_err("%s(): Invalid transport descriptor", __func__);
		return (-1);
	}

again:
	FD_ZERO(&rfds);
	FD_SET(context->tfd, &rfds);

	n = select(context->tfd + 1, &rfds, NULL, NULL, NULL);
	if (n < 0 && errno == EINTR)
		goto again;

	n = obexapp_util_read(context->tfd, context->tbuffer, OBEXAPP_BUFFER_SIZE);
	if (n < 0) {
		log_err("%s(): Could not read(%d). %s (%d)", __func__,
			context->tfd, strerror(errno), errno);

		return (n);
	}
	if (n == 0) {
		log_info("%s(): Connection closed", __func__);
		return (-1);
	}

	log_debug("%s(): Got %d bytes", __func__, n);

	return (OBEX_CustomDataFeed(handle, context->tbuffer, n));
} /* obexapp_transport_handle_input */

