/*
 * server.c
 *
 * Copyright (c) 2002-2009 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: server.c,v 1.14 2011/07/13 16:40:20 max Exp $
 * $FreeBSD$
 */

#include <sys/stat.h>
#include <netinet/in.h>

#include <bluetooth.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <grp.h>
#include <inttypes.h>
#include <limits.h>
#include <obex.h>
#include <pwd.h>
#include <sdp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "compat.h"
#include "obexapp.h"
#include "event.h"
#include "log.h"
#include "server.h"
#include "util.h"

/* OBEX capability object */
static char const * const	capability_object = 
"<General>\n" \
" <SN>Unknown</SN>\n" \
" <Manufacturer>Unknown</Manufacturer>\n" \
" <Model>Unknown</Model>\n" \
"</General>\n" \
"\n"
"<Inbox-Objects>\n" \
" <Object Type=\"ANY\" Name-Ext=\"ANY\"/>\n" \
"</Inbox-Objects>\n" \
"\n" \
"<Service-Objects>\n" \
" <Object Type=\"x-obex/folder-listing\"\n" \
"  UUID=\"F9EC7BC4-953C-11d2-984E-525400DC9E09\"/>\n" \
"</Service-Objects>\n" \
"\n" \
"<Services>\n" \
" <Folder-Browsing UUID=\"F9EC7BC4-953C-11d2-984E-525400DC9E09\">\n" \
"  <IrDA Target=\"F9EC7BC4-953C-11d2-984E-525400DC9E09\"/>\n" \
" </Folder-Browsing>\n" \
"</Services>\n";

/* Header and footer for DIR command */
static char const * const	ls_header = 
	"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
	"<!DOCTYPE folder-listing SYSTEM \"obex-folder-listing.dtd\">\n" \
	"<folder-listing version=\"1.0\">\n";

static char const * const	ls_footer =
	"</folder-listing>\n";

static char const * const	ls_parent_folder =
	"<parent-folder/>\n";

static int obexapp_server_set_initial_root (context_p context);
static int obexapp_server_set_device_root  (context_p context);
static int obexapp_server_set_final_root   (context_p context);

/* OBEX request handlers */
static obexapp_request_handler_t	obexapp_server_request_connect;
static obexapp_request_handler_t	obexapp_server_request_disconnect;
static obexapp_request_handler_t	obexapp_server_request_put;
static obexapp_request_handler_t	obexapp_server_request_get;
static obexapp_request_handler_t	obexapp_server_request_setpath;

static int obexapp_server_request_get_capability_object
		(obex_t *handle, obex_object_t *object);
static int obexapp_server_request_get_folder_object
		(obex_t *handle, obex_object_t *object, char const *name);
static int obexapp_server_request_get_file_object
		(obex_t *handle, obex_object_t *object, char const *name);

static int obexapp_server_append_ls
		(context_t *context, char const *s);

/*
 * OBEX server
 */

int
obexapp_server(obex_t *handle)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);
	int		error = -1;

	context->ss = sdp_open_local(NULL);
	if (context->ss == NULL) {
		log_err("%s(): Could not create local SDP session", __func__);
		goto done;
	}

	if (sdp_error(context->ss) != 0) {
		log_err("%s(): Could to open local SDP session. %s (%d)",
			__func__, strerror(sdp_error(context->ss)),
			sdp_error(context->ss));
		goto done;
	}

	log_info("%s: Starting OBEX server", __func__);

	if (OBEX_SetTransportMTU(handle, context->mtu, context->mtu) < 0) {
		log_err("%s(): OBEX_SetTransportMTU failed", __func__);
		goto done;
	}

	if (OBEX_ServerRegister(handle, NULL, 0) < 0) {
		log_err("%s(): OBEX_ServerRegister failed", __func__);
		goto done;
	}

	if (obexapp_server_set_initial_root(context) < 0)
		goto done;
	if (context->vroot && obexapp_server_set_device_root(context) < 0)
		goto done;
	if (obexapp_server_set_final_root(context) < 0)
		goto done;

	log_debug("%s(): Entering event processing loop...", __func__);

	for (error = 0, context->done = 0; !context->done; ) {
		error = OBEX_HandleInput(handle, OBEXAPP_TRANSPORT_TIMEOUT);
		if (error < 0) {
			log_err("%s(): OBEX_HandleInput failed, error=%d",
				__func__, error);
			break;
		}
	}

	log_debug("%s(): Event processing loop completed", __func__);
done:
	if (context->ss != NULL) {
		sdp_close(context->ss);
		context->ss = NULL;
	}

	return (error);
} /* obexapp_server */

/*
 * Set initial server root
 */

static int
obexapp_server_set_initial_root(context_p context)
{
	struct passwd	*pw = NULL;
	char		*ep;

	if (context->user != NULL) {
		context->uid = strtoul(context->user, &ep, 10);
		if (*ep != '\0')
			pw = getpwnam(context->user);
		else
			pw = getpwuid(context->uid);

		if (pw == NULL) {
			log_err("%s(): Unknown user '%s'",
				__func__, context->user);
			return (-1);
		}

		log_debug("%s(): Requested to run as '%s', uid=%d, gid=%d",
			__func__, context->user, pw->pw_uid, pw->pw_gid);

		context->uid = pw->pw_uid;
		context->gid = pw->pw_gid;
	} else {
		context->uid = getuid();
		context->gid = getgid();
	}

	/* Set default root */
	if (context->root[0] == '\0') {
		if (pw == NULL)
			strlcpy(context->root, OBEXAPP_ROOT_DIR, PATH_MAX);
		else
			strlcpy(context->root, pw->pw_dir, PATH_MAX);
	}

	if (chdir(context->root) < 0) {
		log_err("%s(): Could not chdir(%s). %s (%d)",
			__func__, context->root, strerror(errno), errno);
		return (-1);
	}

	log_debug("%s(): Using initial root %s", __func__, context->root);

	return (0);
} /* obexapp_server_set_initial_root */

/*
 * Set device specific server root
 */

static int
obexapp_server_set_device_root(context_p context)
{
	char const	*root[] = { NULL, NULL, NULL };
	struct hostent	*he;
	struct stat	sb;
	int		n;

	n = 0;

	he = bt_gethostbyaddr((char const *) &context->raddr,
			sizeof(bdaddr_t), AF_BLUETOOTH);
	if (he != NULL)
		root[n ++] = (char const *) he->h_name;

	root[n ++] = bt_ntoa(&context->raddr, NULL);

	root[n ++] = "default";

	for (n = 0; n < (int)(sizeof(root)/sizeof(root[0])); n ++) {
		if (root[n] == NULL)
			break;

		log_debug("%s(): Checking for %s/%s subdirectory",
			__func__, context->root, root[n]);

		if (lstat(root[n], &sb) < 0) {
			if (errno == ENOENT)
				continue;

			log_err("%s(): Could not lstat(%s/%s). %s (%d)",
				__func__, context->root, root[n],
				strerror(errno), errno);

			return (-1);
		}

		strlcat(context->root, "/", PATH_MAX);
		strlcat(context->root, root[n], PATH_MAX);

		if (chdir(root[n]) < 0) {
			log_err("%s(): Could not chdir(%s). %s (%d)",
				__func__, context->root, strerror(errno),
				errno);
			return (-1);
		}

		/* If user was not set before, take it from lstat() data */
		if (context->user == NULL) {
			context->uid = sb.st_uid;
			context->gid = sb.st_gid;
		}

		log_debug("%s(): Using device specific root %s, uid=%d, gid=%d",
			__func__, context->root, context->uid, context->gid);

		return (1);
	}

	log_err("%s(): Could not find device specific root for the device " \
		"bdaddr %s (%s)", __func__, root[1],
		root[0]? root[0] : "-no-name-");

	return (-1);
} /* obexapp_server_set_device_root */

/*
 * Finalize server root
 */

static int
obexapp_server_set_final_root(context_p context)
{
	if (context->secure) {
		if (chroot(context->root) < 0) {
			log_err("%s(): Could not chroot(%s). %s (%d)",
				__func__, context->root,
				strerror(errno), errno);
			return (-1);
		}

		strlcpy(context->root, "/", PATH_MAX);
	}

	if (context->gid != getgid() && setgid(context->gid) < 0) {
		log_err("%s(): Could not setgid(%d). %s (%d)",
			__func__, context->gid, strerror(errno), errno);
		return (-1);
	}

	if (context->uid != getuid() && setuid(context->uid) < 0) {
		log_err("%s(): Could not setuid(%d). %s (%d)",
			__func__, context->uid, strerror(errno), errno);
		return (-1);
	}

	log_notice("%s(): Using root %s; Secure mode %s; "
		"Running as uid=%d, gid=%d", __func__, context->root,
		context->secure? "enabled" : "disabled", getuid(), getgid());

	return (0);
} /* obexapp_server_set_final_root */

/*
 * Process OBEX_EV_REQHINT event
 */

void
obexapp_server_request_hint(obex_t *handle, obex_object_t *object, 
			int obex_cmd, __unused int obex_rsp)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);

	log_debug("%s()", __func__);

	switch(obex_cmd) {
	case OBEX_CMD_PUT:
		log_debug("%s(): PUT request. Going to turn streaming on",
			__func__);

		context->pstream = 0;
		context->sfd = -1;

		OBEX_ObjectReadStream(handle, object, NULL);
		OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
		break;

	case OBEX_CMD_CONNECT:
	case OBEX_CMD_DISCONNECT:
	case OBEX_CMD_GET:
	case OBEX_CMD_SETPATH:
		OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
		break;

#if 0 /* XXX FIXME - what to do with these? */
	case OBEX_CMD_COMMAND:
	case OBEX_CMD_ABORT:
	case OBEX_FINAL:
#endif
	default:
		log_notice("%s(): Skipping unsupported command %#x\n",
			__func__, obex_cmd);

		OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED,
			OBEX_RSP_NOT_IMPLEMENTED);
		break;
	}
} /* obexapp_server_request_hint */

/*
 * Process OBEX_EV_REQ event
 */

void
obexapp_server_request(obex_t *handle, obex_object_t *object, 
			int obex_cmd, int obex_rsp)
{
	int	codes;

	log_debug("%s()", __func__);

	switch (obex_cmd) {
	case OBEX_CMD_CONNECT:
		codes = obexapp_server_request_connect(handle, object, obex_rsp);
		break;

	case OBEX_CMD_DISCONNECT:
		codes = obexapp_server_request_disconnect(handle, object, obex_rsp);
		break;

	case OBEX_CMD_PUT:
		codes = obexapp_server_request_put(handle, object, obex_rsp);
		break;

	case OBEX_CMD_GET:
		codes = obexapp_server_request_get(handle, object, obex_rsp);
		break;

	case OBEX_CMD_SETPATH:
		codes = obexapp_server_request_setpath(handle, object, obex_rsp);
		break;

#if 0 /* XXX FIXME - what to do with these? */
	case OBEX_CMD_COMMAND:
	case OBEX_CMD_ABORT:
	case OBEX_FINAL:
#endif

	default:
		log_err("%s(): Unknown OBEX request, obex_cmd=%#x, " \
			"obex_rsp=%#x", __func__, obex_cmd, obex_rsp);

		codes = OBEXAPP_PACK_RSP_CODES(OBEX_RSP_NOT_IMPLEMENTED,
					OBEX_RSP_NOT_IMPLEMENTED);
		break;
	}

	OBEX_ObjectSetRsp(object, OBEXAPP_UNPACK_HI_RSP_CODE(codes), 
			OBEXAPP_UNPACK_LO_RSP_CODE(codes));
} /* obexapp_server_request */

/*
 * Process OBEX_EV_REQDONE event
 */

void
obexapp_server_request_done(obex_t *handle, __unused obex_object_t *object, 
			int obex_cmd, __unused int obex_rsp)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);

	log_debug("%s()", __func__);

	switch (obex_cmd) {
	case OBEX_CMD_DISCONNECT:
		log_debug("%s(): Disconnect completed", __func__);
/* XXX ???	OBEX_TransportDisconnect(handle); */
		context->done = 1;
		break;

	default:
		log_debug("%s(): Command %#x has finished", __func__, obex_cmd);
		break;
	}
} /* obexapp_server_request_done */

/*
 * Process OBEX_CMD_CONNECT request
 */

static int
obexapp_server_request_connect(obex_t *handle, obex_object_t *object,
		__unused int obex_rsp)
{
	obex_headerdata_t	 hv;
	uint8_t			 hi, *data = NULL;
	uint32_t		 hlen;
	uint8_t const		*target = NULL;
	int			 target_len = 0;

	log_debug("%s()", __func__);

	if (OBEX_ObjectGetNonHdrData(object, &data) == sizeof(obex_connect_hdr_t))
		log_debug("%s(): OBEX connect header: version=%#x, " \
			"flags=%#x, mtu=%d", __func__,
			((obex_connect_hdr_t *) data)->version,
			((obex_connect_hdr_t *) data)->flags,
			ntohs(((obex_connect_hdr_t *) data)->mtu));
	else
		log_err("%s(): Invalid OBEX connect header?!", __func__);

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
		case OBEX_HDR_WHO:
			log_debug("%s(): WHO - %*.*s", 
				__func__, hlen, hlen, hv.bs);
			break;

		case OBEX_HDR_TARGET:
			target = hv.bs;
			target_len = hlen;

			if (target_len == 16) {
				log_debug("%s(): Object target - " \
					"UUID %08X-%04X-%04X-%04X-%08X%04X",
					__func__,
					*(uint32_t const *)&target[0],
					*(uint16_t const *)&target[4],
					*(uint16_t const *)&target[6],
					*(uint16_t const *)&target[8],
					*(uint32_t const *)&target[10],
					*(uint16_t const *)&target[14]);
			} else
				log_debug("%s(): Object target - " \
					"unknown format", __func__);
			break;

		default:
			log_debug("%s(): Skipping header, hi=%#x, hlen=%d",
				__func__, hi, hlen);
			break;
		}
	}

	if (target != NULL) {
		hv.bq4 = 1;
		if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_CONNECTION,
				hv, sizeof(hv.bq4), 0) < 0) {
			log_err("%s(): Could not add HDR_CONNECTION", __func__);

			return (OBEXAPP_PACK_RSP_CODES(
					OBEX_RSP_INTERNAL_SERVER_ERROR, 
					OBEX_RSP_INTERNAL_SERVER_ERROR));
		}

		hv.bs = target;
		if (OBEX_ObjectAddHeader(handle, object, 
				OBEX_HDR_WHO, hv, target_len, 0) < 0) {
			log_err("%s(): Could not add HDR_WHO", __func__);

			return (OBEXAPP_PACK_RSP_CODES(
					OBEX_RSP_INTERNAL_SERVER_ERROR, 
					OBEX_RSP_INTERNAL_SERVER_ERROR));
		}
        }

	return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS));
} /* obexapp_server_request_connect */

/*
 * Process OBEX_CMD_DISCONNECT request
 */

static int
obexapp_server_request_disconnect(__unused obex_t *handle,
		__unused obex_object_t *object, __unused int obex_rsp)
{
	log_debug("%s()", __func__);
	return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS));
} /* obexapp_server_request_disconnect */

/*
 * Process OBEX_CMD_PUT request
 */

static int
obexapp_server_request_put(obex_t *handle, obex_object_t *object,
		__unused int obex_rsp)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	 hv;
	uint8_t			 hi;
	uint8_t const		*body = NULL;
	uint32_t		 hlen;
	int			 codes, body_length, body_end;

	log_debug("%s()", __func__);

	context->file[0] = '\0';
	body_length = 0;
	body_end = 0;

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
		case OBEX_HDR_CONNECTION:
			log_debug("%s(): Connection ID - %#x",
				__func__, hv.bq4);
			break;

		case OBEX_HDR_LENGTH:
			log_debug("%s(): Length - %d", __func__, hv.bq4);
			break;

		case OBEX_HDR_NAME:
			if (hlen == 0 || hlen / 2 > PATH_MAX) {
				log_err("%s(): Invalid Name, hlen=%d",
					__func__, hlen);
				codes = OBEXAPP_PACK_RSP_CODES(
						OBEX_RSP_BAD_REQUEST,
						OBEX_RSP_BAD_REQUEST);
				goto done;
			}

			if (obexapp_util_locale_from_utf16be(hv.bs, hlen,
						context->file, PATH_MAX) < 0) {
				log_err("%s(): Could not convert from UTF-16BE",
					__func__);
				codes = OBEXAPP_PACK_RSP_CODES(
						OBEX_RSP_BAD_REQUEST,
						OBEX_RSP_BAD_REQUEST);
				goto done;
			}

			log_debug("%s(): Name - %s, hlen=%d",
				__func__, context->file, hlen);
			break;

		case OBEX_HDR_TYPE:
			log_debug("%s(): Object type - %s", __func__, hv.bs);
			break;

		case OBEX_HDR_BODY:
			log_debug("%s(): Body - length=%d", __func__, hlen);
			body = hv.bs;
			body_length = hlen;
			break;

		case OBEX_HDR_BODY_END:
			log_debug("%s(): Body end", __func__);
			body_end = 1;
			break;

		default:
			log_debug("%s(): Skipping header, hi=%#x, hlen=%d",
				__func__, hi, hlen);
                        break;
		}
	}

	/* XXX must have NAME in PUT */
	if (context->file[0] == '\0') {
		log_warning("%s(): Rejecting PUT without NAME", __func__);
		codes = OBEXAPP_PACK_RSP_CODES(OBEX_RSP_BAD_REQUEST,
				OBEX_RSP_BAD_REQUEST);
		goto done;
	}

	/*
	 * PUT without BODY and BODY_END - delete NAME
	 * PUT with BODY - save BODY to NAME
	 * PUT with BODY_END - create empty NAME
	 * PUT with context->pstream - move context->temp to NAME
	 */

	if (body != NULL || body_end || context->pstream) {
		if (body != NULL || body_end) {
			context->sfd = obexapp_util_mkstemp(context->file, context->temp, PATH_MAX);
			if (context->sfd < 0) {
				log_err("%s(): Could not obexapp_util_mkstemp. %s (%d)",
					__func__, strerror(errno), errno);

				codes = OBEXAPP_PACK_RSP_CODES(
						OBEX_RSP_INTERNAL_SERVER_ERROR,
						OBEX_RSP_INTERNAL_SERVER_ERROR);
				goto done;
			}

			/* Pretend we have a stream */
			context->pstream = OBEXAPP_PSTREAM_END;
		}

		if (body != NULL) {
			if (obexapp_util_write(context->sfd, body, 
					body_length) < 0) {
				log_err("%s(): Could not write(%d). %s (%d)",
					__func__, context->sfd,
					strerror(errno), errno);

				codes = obexapp_util_errno2response(errno);
				goto done;
			}
		}

		if (chmod(context->temp, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP) < 0) {
			log_err("%s(): Could not chmod(%s). %s (%d)",
				__func__, context->temp,
				strerror(errno), errno);

			codes = obexapp_util_errno2response(errno);
			goto done;
		}

		if (rename(context->temp, context->file) < 0) {
			log_err("%s(): Could not rename(%s, %s). %s (%d)",
				__func__, context->temp,
				context->file, strerror(errno), errno);

			codes = obexapp_util_errno2response(errno);
			goto done;
		}
	} else {
		if (remove(context->file) < 0) {
			log_err("%s(): Could not remove(%s). %s (%d)",
				__func__, context->file,
				strerror(errno), errno);

			codes = obexapp_util_errno2response(errno);
			goto done;
		}
	}

	codes = OBEXAPP_PACK_RSP_CODES(OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
done:
	if (context->pstream) {
		context->pstream = 0;
		close(context->sfd);
		context->sfd = -1;
		unlink(context->temp);
	}

	return (codes);
} /* obexapp_server_request_put */

/*
 * Process OBEX_CMD_GET request
 */

static int
obexapp_server_request_get(obex_t *handle, obex_object_t *object,
		__unused int obex_rsp)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	 hv;
	uint8_t			 hi;
	uint32_t		 hlen;
	char const		*object_type = NULL;

	log_debug("%s()", __func__);

	context->file[0] = '\0';

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
		case OBEX_HDR_NAME:
			if (hlen == 0) {
				log_debug("%s(): Got no Name", __func__);
				break;
			}

			if (hlen / 2 > PATH_MAX) {
				log_err("%s(): Name too big, hlen=%d",
					__func__, hlen);

				return (OBEXAPP_PACK_RSP_CODES(
						OBEX_RSP_BAD_REQUEST,
						OBEX_RSP_BAD_REQUEST));
			}

			if (obexapp_util_locale_from_utf16be(hv.bs, hlen,
						context->file, PATH_MAX) < 0) {
				log_err("%s(): Could not convert from UTF-16BE",
					__func__);

				return (OBEXAPP_PACK_RSP_CODES(
						OBEX_RSP_BAD_REQUEST,
						OBEX_RSP_BAD_REQUEST));
			}

			log_debug("%s(): Name - %s, hlen=%d",
				__func__, context->file, hlen);
			break;

		case OBEX_HDR_TYPE:
			object_type = (char const *) hv.bs;
			log_debug("%s(): Object type - %s",
				__func__, object_type);
			break;

		case OBEX_HDR_CONNECTION:
			log_debug("%s(): Connection ID - %#x",
				__func__, hv.bq4);
			break;

		default:
			log_debug("%s(): Skipping header, hi=%#x, hlen=%d",
				__func__, hi, hlen);
			break;
		}
	}

	if (object_type != NULL) {
		if (strcasecmp(object_type, "x-obex/capability") == 0)
			return (obexapp_server_request_get_capability_object(
							handle, object));

		if (strcasecmp(object_type, "x-obex/folder-listing") == 0) {
			if (context->file[0] == '\0') {
				context->file[0] = '.';
				context->file[1] = '\0';
			}

			return (obexapp_server_request_get_folder_object(
							handle, object,
							context->file));
		}
	}

	/* XXX should return object of desired "object_type" (if any) */
	if (context->file[0] == '\0')
		snprintf(context->file, PATH_MAX, "%s/%s", context->root,
				OBEXAPP_DEFAULT_OBJECT);

	return (obexapp_server_request_get_file_object(handle, object, context->file));
} /* obexapp_server_request_get */

/*
 * GET OBEX capability object
 */

static int
obexapp_server_request_get_capability_object(obex_t *handle, obex_object_t *object)
{
	obex_headerdata_t	hv;
	int			len = strlen(capability_object) + 1;

	hv.bq4 = len;
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH,
			hv, sizeof(hv.bq4), 0) < 0) {
		log_err("%s(): Could not add HDR_LENGTH", __func__);
		return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR));
	}

	hv.bs = (uint8_t const *) capability_object;
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY,
			hv, len, 0) < 0) {
		log_err("%s(): Could not add HDR_BODY", __func__);
		return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR));
	}

	return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS));
} /* obexapp_server_request_get_capability_object */

/*
 * GET listing for folder "name"
 */

static int
obexapp_server_request_get_folder_object(obex_t *handle, obex_object_t *object,
		char const *name)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	 hv;
	DIR			*dir = NULL;
	struct dirent		*de = NULL;
	uid_t			 euid, egid, fuid, fgid;
	struct stat		 st;
	struct tm		*c_time = NULL, *m_time = NULL, *a_time = NULL;
	char			 uperm[4], gperm[4], operm[4],
				 name_str[PATH_MAX],
				*ls = NULL;
	int			 codes, fmode;

	if ((dir = opendir(name)) == NULL) {
		log_err("%s(): Could not opendir(%s). %s (%d)",
			__func__, name, strerror(errno), errno);
		return (obexapp_util_errno2response(errno));
	}

	if (fstat(dirfd(dir), &st) < 0) {
		log_err("%s(): Could not fstat(%d). %s (%d)",
			__func__, dirfd(dir), strerror(errno), errno);
		codes = obexapp_util_errno2response(errno);
		goto done;
	}

	fmode = st.st_mode;
	fuid = st.st_uid;
	fgid = st.st_gid;

	context->ls[0] = '\0';

	if (obexapp_server_append_ls(context, ls_header) < 0) {
		log_crit("%s(): Could not assign context->ls", __func__);
		codes =  OBEXAPP_PACK_RSP_CODES(OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR);
		goto done;
	}

	euid = geteuid();
	egid = getegid();

	while ((de = readdir(dir)) != NULL) {
		if (stat(de->d_name, &st) < 0) {
			log_err("%s(): Could not stat(%s). %s (%d)",
				__func__, de->d_name, strerror(errno), errno);
			codes = obexapp_util_errno2response(errno);
			goto done;
		}

		if (S_ISDIR(st.st_mode) && strcmp(de->d_name, ".") == 0)
			continue;

		if (S_ISDIR(st.st_mode) && strcmp(de->d_name, "..") == 0) {
			if (obexapp_server_append_ls(context, ls_parent_folder) < 0) {
				log_crit("%s(): Could not append context->ls",
					__func__);

				codes = OBEXAPP_PACK_RSP_CODES(
						OBEX_RSP_INTERNAL_SERVER_ERROR,
						OBEX_RSP_INTERNAL_SERVER_ERROR);
				goto done;
			}

			continue;
		}

		if (obexapp_util_locale_to_utf8(de->d_name, strlen(de->d_name),
				name_str, sizeof(name_str)) < 0) {
			log_err("%s(): Could not convert to UTF-8", __func__);

			codes = OBEXAPP_PACK_RSP_CODES(
					OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR);
			goto done;
		}

		c_time = gmtime(&st.st_ctime);
		m_time = gmtime(&st.st_mtime);
		a_time = gmtime(&st.st_atime);

		uperm[0] = (st.st_mode & S_IRUSR)? 'R' : '-';
		uperm[1] = (st.st_mode & S_IWUSR)? 'W' : '-';
		uperm[2] = ((euid == fuid) && (fmode & S_IWUSR))?  'D' : '-';

		gperm[0] = (st.st_mode & S_IRGRP)? 'R' : '-';
		gperm[1] = (st.st_mode & S_IWGRP)? 'W' : '-';
		gperm[2] = ((egid == fgid) && (fmode & S_IWGRP))?  'D' : '-';

		operm[0] = (st.st_mode & S_IROTH)? 'R' : '-';
		operm[1] = (st.st_mode & S_IWOTH)? 'W' : '-';
		operm[2] = ((fmode & S_IWOTH) && !(fmode & S_ISTXT))? 'D' : '-';

		uperm[3] = gperm[3] = operm[3] = '\0';

		/*
		 * We could be running chroot()ed, so we can not easily
		 * translate user/group IDs into use/group name. So, don't
		 * even try and just return user/group IDs always.
		 */

		ls = NULL;
		asprintf(&ls,
" <%s name=\"%s\" size=\"%" PRId64 "\" " \
"owner=\"%d\" group=\"%d\" " \
"user-perm=\"%s\" group-perm=\"%s\" other-perm=\"%s\" " \
"created=\"%04d%02d%02dT%02d%02d%02dZ\" " \
"modified=\"%04d%02d%02dT%02d%02d%02dZ\" " \
"accessed=\"%04d%02d%02dT%02d%02d%02dZ\"/>\n",
			(S_ISDIR(st.st_mode) ? "folder" : "file"),
			name_str, st.st_size,
			st.st_uid, st.st_gid,
			uperm, gperm, operm,
			c_time->tm_year + 1900, c_time->tm_mon + 1,
			c_time->tm_mday, c_time->tm_hour, c_time->tm_min,
			c_time->tm_sec,
			m_time->tm_year + 1900, m_time->tm_mon + 1,
			m_time->tm_mday, m_time->tm_hour, m_time->tm_min,
			m_time->tm_sec,
			a_time->tm_year + 1900, a_time->tm_mon + 1,
			a_time->tm_mday, a_time->tm_hour, a_time->tm_min,
			a_time->tm_sec);

		if (ls == NULL) {
			log_crit("%s(): Could not asprintf ls", __func__);
			codes = OBEXAPP_PACK_RSP_CODES(
					OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR);
			goto done;
		}

		if (obexapp_server_append_ls(context, ls) < 0) {
			log_err("%s(): Could not append context->ls", __func__);
			codes = OBEXAPP_PACK_RSP_CODES(
					OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR);
			free(ls);
			goto done;
		}

		free(ls);
	}

	if (obexapp_server_append_ls(context, ls_footer) < 0) {
		log_crit("%s(): Could not append context->ls", __func__);
		codes = OBEXAPP_PACK_RSP_CODES(OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR);
		goto done;
	}

	hv.bq4 = strlen(context->ls) + 1;
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH,
			hv, sizeof(hv.bq4), 0) < 0) {
		log_err("%s(): Could not add HDR_LENGTH", __func__);
		codes = OBEXAPP_PACK_RSP_CODES(
					OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR);
		goto done;
	}

	hv.bs = (uint8_t const *) context->ls;
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY,
			hv, strlen(context->ls) + 1, 0) < 0) {
		log_err("%s(): Could not add HDR_BODY", __func__);
		codes = OBEXAPP_PACK_RSP_CODES(
					OBEX_RSP_INTERNAL_SERVER_ERROR,
					OBEX_RSP_INTERNAL_SERVER_ERROR);
		goto done;
	}

	codes = OBEXAPP_PACK_RSP_CODES(OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
done:
	if (dir != NULL)
		closedir(dir);

	return (codes);
} /* obexapp_server_request_get_folder_object */

/*
 * GET file "name"
 */

static int
obexapp_server_request_get_file_object(obex_t *handle, obex_object_t *object,
		char const *name)
{
	context_p		context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	hv;
	struct stat		st;

	if ((context->sfd = open(name, O_RDONLY)) < 0) {
		log_err("%s(): open(%s) failed. %s (%d)",
			__func__, name, strerror(errno), errno);
				
		return (obexapp_util_errno2response(errno));
	}

	if (fstat(context->sfd, &st) < 0) {
		log_err("%s(): fstat(%d) failed. %s (%d)",
			__func__, context->sfd, strerror(errno), errno);

		close(context->sfd);
		context->sfd = -1;

		return (obexapp_util_errno2response(errno));
	}

	hv.bq4 = st.st_size;
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH, 
			hv, sizeof(hv.bq4), 0) < 0) {
		log_err("%s(): Could not add HDR_LENGTH", __func__);

		return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_INTERNAL_SERVER_ERROR, 
					OBEX_RSP_INTERNAL_SERVER_ERROR));
	}

	hv.bs = NULL;
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY, 
			hv, 0, OBEX_FL_STREAM_START) < 0) {
		log_err("%s(): Could not add HDR_BODY", __func__);

		return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_INTERNAL_SERVER_ERROR, 
					OBEX_RSP_INTERNAL_SERVER_ERROR));
	}

	return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS));
} /* obexapp_server_request_get_file_object */

/*
 * Process OBEX_CMD_SETPATH request
 */

static int
obexapp_server_request_setpath(obex_t *handle, obex_object_t *object,
		__unused int obex_rsp)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	 hv;
	uint8_t			 hi, flags = 0xff, *data = NULL;
	uint32_t		 hlen;
	int			 got_name = 0;

	log_debug("%s()", __func__);

	context->file[0] = '\0';

	if (OBEX_ObjectGetNonHdrData(object, &data) == sizeof(obex_setpath_hdr_t)) {
		flags = ((obex_setpath_hdr_t *) data)->flags;

		log_debug("%s(): OBEX setpath header: flags=%#x, constants=%d",
			__func__, flags, ((obex_setpath_hdr_t *) data)->constants);
	} else
		log_err("%s(): Invalid OBEX setpath header?!", __func__);

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
		case OBEX_HDR_NAME:
			got_name = 1;

			if (hlen == 0)
				break;
			if (hlen / 2 > PATH_MAX) {
				log_err("%s(): Name too big, hlen=%d",
					__func__, hlen);

				return (OBEXAPP_PACK_RSP_CODES(
						OBEX_RSP_BAD_REQUEST,
						OBEX_RSP_BAD_REQUEST));
			}

			if (obexapp_util_locale_from_utf16be(hv.bs, hlen,
						context->file, PATH_MAX) < 0) {
				log_err("%s(): Could not convert from UTF-16BE",
					__func__);

				return (OBEXAPP_PACK_RSP_CODES(
						OBEX_RSP_BAD_REQUEST,
						OBEX_RSP_BAD_REQUEST));
			}

			log_debug("%s(): Name - %s, hlen=%d",
				__func__, context->file, hlen);
			break;

		case OBEX_HDR_CONNECTION:
			log_debug("%s(): Connection ID - %#x",
				__func__, hv.bq4);
			break;

		default:
			log_debug("%s(): Skipping header, hi=%#x, hlen=%d",
				__func__, hi, hlen);
			break;
		}
	}

	if (!got_name) {

		/*
		 * No name and flags == 0x3 (back up one level + don't create
		 * directory) means "cd ..". Everything else is forbidden.
		 */
	
		if (flags != 0x3) {
			log_err("%s(): Invalid flags for 'cd ..', flags=%#x",
				__func__, flags);
				
			return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_FORBIDDEN,
							OBEX_RSP_FORBIDDEN));
		}

		strlcpy(context->file, "..", PATH_MAX);
	}

	if (context->file[0] == '\0') {

		/*
		 * Empty name and flags == 0x2 (don't create directory) means
		 * 'cd /'. Everything else is forbidden
		 */

		if (flags != 0x2) {
			log_err("%s(): Invalid flags for 'cd /', flags=%#x",
				__func__, flags);
				
			return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_FORBIDDEN,
							OBEX_RSP_FORBIDDEN));
		}

		strlcpy(context->file, context->root, PATH_MAX);
	}

	if (flags == 0) {
		if (mkdir(context->file, 0755) < 0 && errno != EEXIST) {
			log_err("%s(): mkdir(%s) failed. %s (%d)",
				__func__, context->file, 
				strerror(errno), errno);

			return (obexapp_util_errno2response(errno));
		}
	} else if (chdir(context->file) < 0) {
		log_err("%s(): chdir(%s) failed. %s (%d)",
			__func__, context->file, strerror(errno), errno);

		return (obexapp_util_errno2response(errno));
	}

	return (OBEXAPP_PACK_RSP_CODES(OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS));
} /* obexapp_server_request_setpath */

/*
 * Append string to the end of context->ls
 */

static int
obexapp_server_append_ls(context_t *context, char const *s)
{
	int	 ls_alen = strlen(s) + 1;
	int	 ls_end = strlen(context->ls);
	int	 ls_avail = context->ls_size - ls_end;
	char	*ls = NULL;

	if (ls_avail < ls_alen) {
		ls = (char *) realloc(context->ls, context->ls_size + ls_alen);

		if (ls == NULL)
			return (-1);

		context->ls = ls;
		context->ls_size += ls_alen;
		ls_avail = context->ls_size - ls_end;
	}

	strlcpy(context->ls + ls_end, s, ls_avail);

	return (0);
} /* obexapp_server_append_ls */

