/*
 * event.c
 *
 * Copyright (c) 2002-2009 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: event.c,v 1.8 2010/10/22 17:04:11 max Exp $
 * $FreeBSD$
 */

#include <bluetooth.h>
#include <obex.h>
#include <stdio.h>
#include <unistd.h>

#include "compat.h"
#include "obexapp.h"
#include "client.h"
#include "event.h"
#include "log.h"
#include "server.h"
#include "stream.h"

/* Common event handlers */
static obexapp_event_handler_t		obexapp_event_stub;
static obexapp_event_handler_t		obexapp_event_progress;
static obexapp_event_handler_t		obexapp_event_link_error;
static obexapp_event_handler_t		obexapp_event_parse_error;
static obexapp_event_handler_t		obexapp_event_abort;

/* Server event table */
static obexapp_event_handler_t * const	server_eh[] = {
	obexapp_event_progress,		/* 0 - OBEX_EV_PROGRESS */
	obexapp_server_request_hint,	/* 1 - OBEX_EV_REQHINT */
	obexapp_server_request,		/* 2 - OBEX_EV_REQ */
	obexapp_server_request_done,	/* 3 - OBEX_EV_REQDONE */
	obexapp_event_link_error,	/* 4 - OBEX_EV_LINKERR */
	obexapp_event_parse_error,	/* 5 - OBEX_EV_PARSEERR */
	obexapp_event_stub,		/* 6 - OBEX_EV_ACCEPTHINT */ 
	obexapp_event_abort,		/* 7 - OBEX_EV_ABORT */
	obexapp_stream_write,		/* 8 - OBEX_EV_STREAMEMPTY */
	obexapp_stream_read,		/* 9 - OBEX_EV_STREAMAVAIL */
	obexapp_event_stub,		/* 10 - OBEX_EV_UNEXPECTED */
	obexapp_event_stub,		/* 11 - OBEX_EV_REQCHECK */
};
#define	server_eh_size	((int)(sizeof(server_eh)/sizeof(server_eh[0])))

/* Client event table */
static obexapp_event_handler_t * const	client_eh[] = {
	obexapp_event_progress,		/* 0 - OBEX_EV_PROGRESS */
	obexapp_event_stub,		/* 1 - OBEX_EV_REQHINT */
	obexapp_event_stub,		/* 2 - OBEX_EV_REQ */
	obexapp_client_request_done,	/* 3 - OBEX_EV_REQDONE */
	obexapp_event_link_error,	/* 4 - OBEX_EV_LINKERR */
	obexapp_event_parse_error,	/* 5 - OBEX_EV_PARSEERR */
	obexapp_event_stub,		/* 6 - OBEX_EV_ACCEPTHINT */ 
	obexapp_event_abort,		/* 7 - OBEX_EV_ABORT */
	obexapp_stream_write,		/* 8 - OBEX_EV_STREAMEMPTY */
	obexapp_stream_read,		/* 9 - OBEX_EV_STREAMAVAIL */
	obexapp_event_stub,		/* 10 - OBEX_EV_UNEXPECTED */
	obexapp_event_stub,		/* 11 - OBEX_EV_REQCHECK */
};
#define	client_eh_size	((int)(sizeof(client_eh)/sizeof(client_eh[0])))

/*
 * Main OBEX event handler
 */

void
obexapp_event_handler(obex_t *handle, obex_object_t *object, int mode,
		int obex_event, int obex_cmd, int obex_rsp)
{
	obexapp_event_handler_t	*h = NULL;
	context_p		context = (context_p) OBEX_GetUserData(handle);

	log_debug("%s(): mode=%d, event=%#x, cmd=%#x, rsp=%#x",
		__func__, mode, obex_event, obex_cmd, obex_rsp);

	if (!context->server) {
		if (obex_event < client_eh_size)
			h = client_eh[obex_event];
	} else {
		if (obex_event < server_eh_size)
			h = server_eh[obex_event];
	}

	if (h != NULL)
		(h)(handle, object, obex_cmd, obex_rsp);
	else
		log_debug("%s(): Unknown event=%#x, cmd=%#x, rsp=%#x, mode=%d",
			__func__, obex_event, obex_cmd, obex_rsp, mode);
} /* obexapp_event_handler */

/*
 * Stub event handler
 */

static void
obexapp_event_stub(__unused obex_t *handle, __unused obex_object_t *object, 
			int obex_cmd, int obex_rsp)
{
	log_debug("%s(): obex_cmd=%#x, obex_rsp=%#x",
		__func__, obex_cmd, obex_rsp);
} /* obexapp_event_stub */

/*
 * Process OBEX_EV_PROGRESS event
 */

static void
obexapp_event_progress(obex_t *handle, __unused obex_object_t *object, 
			__unused int obex_cmd, __unused int obex_rsp)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);

	log_debug("%s(): Made some progress...", __func__);

	if (!context->server && !context->ni && isatty(STDOUT_FILENO)) {
		static char	spinner[] = "\\|/-";
		static uint32_t	spinner_idx = 0;

		printf("%c\b", spinner[spinner_idx ++]);
		fflush(stdout);
		if (spinner_idx == sizeof(spinner)/sizeof(spinner[0]) - 1)
			spinner_idx = 0;
	}
} /* obexapp_event_progress */

/*
 * Process OBEX_EV_LINKERR event
 */

static void
obexapp_event_link_error(obex_t *handle, __unused obex_object_t *object, 
			__unused int obex_cmd, __unused int obex_rsp)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);

	log_warning("%s(): Link has been disconnected", __func__);
	context->done = 1;
} /* obexapp_event_link_error */

/*
 * Process OBEX_EV_PARSEERR event
 * XXX FIXME should we set context->done to 1?
 */

static void
obexapp_event_parse_error(obex_t *handle, __unused obex_object_t *object, 
			__unused int obex_cmd, __unused int obex_rsp)
{
	context_p	context = (context_p) OBEX_GetUserData(handle);

	log_warning("%s(): Malformed data encountered", __func__);
	context->done = 1;
} /* obexapp_event_parse_error */

/*
 * Process OBEX_EV_ABORT event
 * XXX FIXME should we set context->done to 1?
 * XXX FIXME should we clean our mess (i.e. temp files etc.)
 */

static void
obexapp_event_abort(__unused obex_t *handle, __unused obex_object_t *object, 
			int obex_cmd, int obex_rsp)
{
	log_warning("%s(): Request aborted, obex_cmd=%#x, obex_rsp=%#x",
		__func__, obex_cmd, obex_rsp);
} /* obexapp_event_abort */

