/*
 * ntesla - a program for designing Tesla Coils
 * 
 * Copyright (C) 1997  Steven A. Falco
 * 
 * Email contact: sfalco@worldnet.att.net
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define EXTERN extern
#include "protos.h"

/* Solonoid form-factor table taken from "Reference Data for Radio Engineers"
 * Howard W. Sams & Co, Inc. 1975.  Fig 1. page 6-1.
 */
INTERPOLATE SolFF[] = {
	{   0.02,  0.00050 }, {   0.03,  0.00076 }, {   0.04,  0.00100 },
	{   0.06,  0.00150 }, {   0.08,  0.00190 }, {   0.10,  0.00240 },
	{   0.15,  0.00360 }, {   0.20,  0.00460 },

	{   0.30,  0.00660 }, {   0.40,  0.00850 }, {   0.60,  0.01200 },
	{   0.80,  0.01500 }, {   1.00,  0.01800 }, {   1.50,  0.02300 },
	
	{   2.00,  0.02700 }, {   3.00,  0.03200 }, {   4.00,  0.03700 },
	{   5.00,  0.04000 }, {   6.00,  0.04400 }, {   8.00,  0.04800 },
	{  10.00,  0.05100 }, {  15.00,  0.05900 }, {  20.00,  0.06300 },

	{  30.00,  0.07000 }, {  40.00,  0.07500 }, {  50.00,  0.07900 },
	{  60.00,  0.08200 }, {  80.00,  0.08600 }, { 100.00,  0.08800 }
};

/* Medhurst self-capacitance factor taken from "Wireless Engineer", V24.281
 * (Feb 1947) 35, and V24.282 (May 1947) 80 (printed in England).  Supplied by
 * Mark S. Rzeszotarski, Ph.D. via email.
 */
INTERPOLATE MedhurstF[] = {
	{ 0.1,  0.96 }, { 0.15, 0.79 }, { 0.2,  0.70 }, { 0.25, 0.64 },
	{ 0.3,  0.60 }, { 0.35, 0.57 }, { 0.4,  0.54 }, { 0.45, 0.52 },
	{ 0.5,  0.50 }, { 0.6,  0.48 }, { 0.7,  0.47 }, { 0.8,  0.46 },
	{ 0.9,  0.46 }, { 1.0,	0.46 }, { 1.5,	0.47 }, { 2.0,	0.50 },
	{ 2.5,	0.56 }, { 3.0,	0.61 }, { 3.5,	0.67 }, { 4.0,	0.72 },
	{ 4.5,	0.77 }, { 5.0,	0.81 }, { 6.0,  0.92 }, { 7.0,  1.01 }, 
	{ 8.0,  1.12 }, { 9.0,  1.22 }, { 10.0, 1.32 }, { 15.0, 1.86 },
	{ 20.0, 2.36 }, { 25.0, 2.90 }, { 30.0, 3.40 }, { 40.0, 4.60 },
	{ 50.0, 5.80 }
};

/* Q of unshielded solonoid table taken from "Reference Data for Radio
 * Engineers" Howard W. Sams & Co, Inc. 1975.  Fig 3. page 6-4.
 */
INTERPOLATE QF[] = {
	{  0.1,  30.0 }, {  0.2,  47.0 }, {  0.3,  59.0 }, {  0.4,  69.0 },
	{  0.6,  85.0 }, {  0.8,  95.0 }, {  1.0, 100.0 }, {  2.0, 120.0 },
	{  3.0, 130.0 }, {  4.0, 135.0 }, {  6.0, 148.0 }, {  8.0, 149.0 },
	{ 10.0, 150.0 }
};

/* Perform table lookup using binary search, then straight-line interpolate. */
double
binSearch(int maxEntry, INTERPOLATE_P fp, double lookfor)
{
	int ii;
	int minEntry = 0;

	/* Binary search for nearest two elements */
	while((minEntry + 1) < (maxEntry)) {

		/* Pick a trial entry */
		ii = (minEntry + maxEntry) / 2;

		if(lookfor < fp[ii].x_axis) {

			/* Guess something smaller next time */
			maxEntry = ii;
		} else if(lookfor > fp[ii].x_axis) {

			/* Guess something larger next time */
			minEntry = ii;
		} else  {

			/* Exact match */
			return fp[ii].y_axis;
		}
	}

	/* Interpolate */
	return	(lookfor - fp[minEntry].x_axis)
		*
		(fp[maxEntry].y_axis - fp[minEntry].y_axis)
		/
		(fp[maxEntry].x_axis - fp[minEntry].x_axis)
		+
		fp[minEntry].y_axis;
}

/* Find the solonoid form factor for a given ratio of diameter over
 * length.
 */
double
findSolFF(double ratio)
{
	/* Factor is linear with ratio for very long coils */
	if(ratio < 0.02) {
		return 0.0250 * ratio;
	}

	/* Ratio is too extreme */
	if(ratio > 100.0) {
		restrict();
		return -1.0;
	}

	return binSearch(sizeof(SolFF) / sizeof(INTERPOLATE), SolFF, ratio);
}

/* Find the Medhurst factor for distributed capacitance. */
double
findMedhurstF(double ratio)
{
	/* Ratio is too extreme */
	if(ratio < 0.1 || ratio > 50.0) {
		restrict();
		return -1.0;
	}

	return binSearch(sizeof(MedhurstF) / sizeof(INTERPOLATE), MedhurstF,
		ratio);
}

/* Find the Q factor */
double
findQFactor(double ratio)
{
	/* Ratio is too extreme */
	if(ratio < 0.1 || ratio > 10.0) {
		restrict();
		return -1.0;
	}

	return binSearch(sizeof(QF) / sizeof(INTERPOLATE), QF, ratio);
}

/* Display the worst-case restriction to avoid confusion */
void
restrict()
{
	E "\007Length divided by Diameter must be between 0.1 and 10.0 ");
}
