/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include "Program.h"
#include "Fragment.h"
#include "RCX_Image.h"
#include "Bytecode.h"
#include "parser.h"
#include "Symbol.h"
#include "BlockStmt.h"
#include "CallStmt.h"
#include "FunctionDef.h"
#include "Error.h"
#include "RCX_Target.h"
#include "VarTranslator.h"
#include "DeclareStmt.h"
#include "PDebug.h"

Program *gProgram;

Program::Program(const RCX_Target *target) :
	fVarAllocator(target->fMaxGlobalVars, target->fMaxTaskVars)
{
	fTarget = target;
	fCounts[0] = 1;	// leave room for main
	fCounts[1] = 0;
	fMainAdded = false;
	fScopes.InsertHead(new Scope());
	fInitName = Symbol::Get("_init");
	fGlobalDecls = new BlockStmt();
	fVirtualVarCount = 0;
}


Program::~Program()
{
	while(Fragment *f=fTasks.RemoveHead())
		delete f;

	while(Fragment *f=fSubs.RemoveHead())
		delete f;
	
	while(Scope *s=fScopes.RemoveHead())
		delete s;

	while(FunctionDef *func = fFunctions.RemoveHead())
		delete func;
		
	delete fGlobalDecls;
}


int Program::AddMainTask(Fragment *f)
{
	CheckName(f->GetName());
	fMainAdded = true;
	fTasks.InsertHead(f);
	return 0;
}


int Program::AddTask(Fragment *f)
{
	CheckName(f->GetName());
	fTasks.InsertTail(f);
	return fCounts[kRCX_TaskFragment]++;
}


int Program::AddSubroutine(Fragment *f)
{
	CheckName(f->GetName());
	fSubs.InsertTail(f);
	return fCounts[kRCX_SubFragment]++;
}


void Program::AddFunction(FunctionDef *f)
{
	CheckName(f->GetName());
	fFunctions.InsertTail(f);
}


void Program::AddGlobalDecls(BlockStmt *b)
{
	// 'b' should be a block statement containing
	// one or more DeclareStmt instances.  Simply move
	// all of the children of b to the fGlobalDecls
	// block.  Later on (in AllocateGlobals() we'll check
	// to make sure they are really DeclareStmts
	
	while(Stmt *s = b->RemoveHead())
	{
		fGlobalDecls->Add(s);
	}

	delete b;
}


void Program::CheckName(const Symbol *name)
{
	if (Defined(name))
		Error(kErr_SymRedef, name->GetKey()).RaiseLex();
}


int Program::CreateVar(const Symbol *name, bool array)
{
	int var;
	
	var = NextVirtualVar();
	if (fScopes.GetHead()->GetNext()==0)
	{
		// global var
		// probably should remember the name somewhere
	}
	
	DefineVar(name, var, array);
	return var;
}


int Program::NextVirtualVar()
{
	return kVirtualVarBase + fVirtualVarCount++;
}


void Program::DefineVar(const Symbol *name, int var, bool array)
{
	Scope *s = fScopes.GetHead();

	if (s->Contains(name))
	{
		Error(kErr_SymRedef, name->GetKey()).RaiseLex();
	}
	else
		s->Define(name, var, array);
}


int Program::GetVar(const Symbol *name, bool &array)
{
	int var = fScopes.GetHead()->Lookup(name, array);
	if (var == kIllegalVar)
		Error(kErr_Undeclared, name->GetKey());
	
	return var;
}


RCX_Image*	Program::CreateImage()
{
	RCX_Image *image;
	int index;
	
	if (!AllocateGlobals()) return nil;
	if (!PrepareMainTask()) return nil;
	if (!CheckFragments()) return nil;
	
	image = new RCX_Image();
	image->SetTargetType(fTarget->fType);
	
	image->SetFragmentCount(fCounts[0] + fCounts[1]);
	index = 0;
	
	// emit subs
	for(Fragment *sub=fSubs.GetHead(); sub; sub=sub->GetNext())
		EncodeFragment(image, sub, index++);

	// emit tasks
	for(Fragment *task=fTasks.GetHead(); task; task=task->GetNext())
		EncodeFragment(image, task, index++);

	
	// emit global variable names
/*
	for(int i=0; i<varCount; i++)
		image->SetVariable(i, fGlobalVars[i]->GetKey());
*/

	return image;
}


void Program::EncodeFragment(RCX_Image *image, Fragment *f, int index)
{
	Bytecode *b = new Bytecode(fVarAllocator, fTarget);

	// determine variable allocation mode
	int mode;
	if (f->GetType() == kRCX_TaskFragment)
		mode = VarAllocator::kTaskMode;
	else if (f->GetTaskID() == Fragment::kMultiTaskID)
		mode = VarAllocator::kMultiSubMode;
	else
		mode = VarAllocator::kSingleSubMode;

	fVarAllocator.Begin(mode);
		
	f->Emit(*b);
	image->SetFragment(index++, f->GetType(), f->GetNumber(),
		b->GetData(), b->GetLength(), f->GetName()->GetKey());

	f->SetLocalCount(fVarAllocator.End());
	
	delete b;
}


bool Program::PrepareMainTask()
{
	// find main
	if (!fMainAdded)
	{
		Error(kErr_UndefinedMain).Raise(0);
		return false;
	}

	// insert block into main task so we have a place for init code
	Fragment *mainTask = fTasks.GetHead();
	BlockStmt *mainBlock = new BlockStmt();
	
	mainBlock->Add(mainTask->GetBody());
	mainTask->SetBody(mainBlock);
	
	// insert call for init routine into main task
	if (fInitName)
	{
		FunctionDef *f = gProgram->GetFunction(fInitName);
		if (f)
		{
			CallStmt *call = new CallStmt();
			call->SetName(fInitName);
			mainBlock->Prepend(call);
		}
		else
		{
			Error(kErr_UnknownInit, fInitName->GetKey()).Raise(0);
			return false;
		}
	}
	
	// insert init stmts into main task
	mainBlock->Prepend(fGlobalDecls);
	fGlobalDecls = 0;

	return true;
}


bool Program::CheckFragments()
{
	bool ok = true;

	if (fCounts[kRCX_TaskFragment] > fTarget->fMaxTasks)
	{
		Error(kErr_TooManyTasks, fTarget->fMaxTasks).Raise(0);
		ok = false;
	}

	if (fCounts[kRCX_SubFragment] > fTarget->fMaxSubs)
	{
		Error(kErr_TooManySubs, fTarget->fMaxSubs).Raise(0);
		ok = false;
	}
	
	Fragment *f;
	for(f=fTasks.GetHead(); f; f=f->GetNext())
	{
		f->Check();
	}
	
	for(f=fSubs.GetHead(); f; f=f->GetNext())
	{
		f->Check();
	}
		
	return ok;
}


Fragment *Program::GetTask(const Symbol *name)
{
	for(Fragment *f = fTasks.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


Fragment *Program::GetSub(const Symbol *name)
{
	for(Fragment *f = fSubs.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


FunctionDef *Program::GetFunction(const Symbol *name)
{
	for(FunctionDef *f = fFunctions.GetHead(); f; f=f->GetNext())
		if (f->GetName() == name) return f;
	
	return 0;
}


bool Program::Defined(const Symbol *name) const
{
	// this "const" hack is just to avoid writing duplicate
	// versions of GetTask(), GetSub(), and GetFunction() that
	// maintain const-ness
	Program *p = (Program*)(this);

	if (p->GetTask(name)) return true;
	if (p->GetSub(name)) return true;
	if (p->GetFunction(name)) return true;
	
	return fScopes.GetTail()->Contains(name);
}


Scope* Program::PushScope()
{
	Scope *s = new Scope();
	fScopes.InsertHead(s);
	return s;
}


void Program::PopScope()
{
	delete fScopes.RemoveHead();
}


bool Program::AllocateGlobals()
{	
	for(Stmt *s=fGlobalDecls->GetHead(); s; s=s->GetNext())
	{
		DeclareStmt *dec = dynamic_cast<DeclareStmt*>(s);
		PASSERT(dec);
		
		int from = dec->GetVar();
		
		// global scope
		int to = fVarAllocator.Allocate(false, false, dec->GetCount());
		if (to == kIllegalVar)
		{
			Error(kErr_NoMoreVars).Raise(&dec->GetLoc());
			return false;
		}

		TranslateVar(from, to);
	}
	
	return true;
}


void Program::TranslateVar(int from, int to)
{
	VarTranslator vt(from, to);
	
	for(Fragment *f=fTasks.GetHead(); f; f=f->GetNext())
		Apply(f->GetBody(), vt);

	for(Fragment *f=fSubs.GetHead(); f; f=f->GetNext())
		Apply(f->GetBody(), vt);

	for(FunctionDef *f=fFunctions.GetHead(); f; f=f->GetNext())
		Apply(f->GetBody(), vt);

	Apply(fGlobalDecls, vt);
}



bool Program::ReserveVars(int start, int end)
{
	if (end < start) return false;
	
	for(int v=start; v<=end; ++v)
		if (!fVarAllocator.Reserve(v)) return false;
		
	return true;
}
