//
// nono
// Copyright (C) 2022 nono project
// Licensed under nono-license.txt
//

//
// パレットモニターウィンドウ
//

#include "wxpalettemonitor.h"
#include "wxcolor.h"
#include "wxtextscreen.h"
#include "wxuimessage.h"
#include "bt45x.h"
#include "mainapp.h"
#include "videoctlr.h"

//
// パレットビットマップ
//
//
// 情報欄は WXTextScreen 同様に上下と左に DefaultPadding を手動で入れてある。
//
//
//   +------------ DefaultPadding (情報欄の左側)
//   |
// +-|------
// | v          <- DefaultPadding (情報欄の上側)
// |  "Text"
// |            <- DefaultPadding (情報欄の下側、パレット欄の区切り)
// |  "+0 +1 …"
// |--------    <- pal_y0 (ここから以下がパレット領域)
// |  □‥
// :  :
// |  □‥
// |            <- DefaultPadding (パレットの下側)
// +-^-------^
//   |       |
//   +-------+---- DefaultPadding (パレットの左右)
//
// パレット欄は1マスが横 font_height (= font_width * 2)、縦 font_height [px]
// で、マスの右端と左端の1ピクセルを BGPANEL 色にして罫線にする。

// イベントテーブル
wxBEGIN_EVENT_TABLE(WXPalettePanel, inherited)
	EVT_MOUSE_EVENTS(WXPalettePanel::OnMouse)
wxEND_EVENT_TABLE()

// コンストラクタ
WXPalettePanel::WXPalettePanel(wxWindow *parent)
	: inherited(parent)
{
	SetName("PalettePanel");

	// デバイスを取得
	bt45x = gMainApp.FindObject<BT45xDevice>(OBJ_BT45x);
	videoctlr = gMainApp.FindObject<VideoCtlrDevice>(OBJ_VIDEOCTLR);

	cursor = -1;

	// パレットは1行に16個で、行数 (1 or 16) が可変。
	// ここではサイズだけ確定させる。
	const std::vector<Color> *hpal;
	if (videoctlr) {
		hpal = &videoctlr->GetHostPalette();
	} else {
		hpal = &bt45x->GetHostPalette();
	}
	rows = hpal->size() / 16;

	FontChanged();
	PaletteChanged();

	// VM からの通知を受け取る
	WXUIMessage::Connect(UIMessage::PALETTE, this,
		wxCommandEventHandler(WXPalettePanel::OnPaletteChanged));
}

// デストラクタ
WXPalettePanel::~WXPalettePanel()
{
	WXUIMessage::Disconnect(UIMessage::PALETTE, this,
		wxCommandEventHandler(WXPalettePanel::OnPaletteChanged));
}

void
WXPalettePanel::FontChanged()
{
	inherited::FontChanged();

	pal0x  = WXTextScreen::DefaultPadding;	// 左余白
	pal0x += font_width * 4;				// "$00:"

	pal0y  = WXTextScreen::DefaultPadding;	// 上余白
	pal0y += font_height * 2;				// テキスト
	pal0y += WXTextScreen::DefaultPadding;	// テキストとパレットの境界余白
	pal0y += font_height;					// ヘッダ("+0+1"…)

	wxSize size;
	size.x  = pal0x;						// パレットより左全部
	size.x += 16 * font_height;				// パレット部
	size.x += WXTextScreen::DefaultPadding;	// 右余白

	size.y  = pal0y;						// パレットより上全部
	size.y += rows * font_height;			// パレット部
	size.y += WXTextScreen::DefaultPadding;	// 下余白

	// バックバッファのサイズを固定。
	SetMinBitmapSize(size);

	SetSize(size);
	SetMinSize(size);
}

// VM からのパレット変更通知
void
WXPalettePanel::OnPaletteChanged(wxCommandEvent& event)
{
	PaletteChanged();
}

void
WXPalettePanel::PaletteChanged()
{
	// ホストパレットをここでコピー。
	if (videoctlr) {
		pal = videoctlr->GetHostPalette();
		ipal = videoctlr->GetIntPalette();
	} else {
		pal = bt45x->GetHostPalette();
		ipal = bt45x->GetIntPalette();
	}
	Refresh();
}

void
WXPalettePanel::Draw()
{
	bitmap.Fill(BGPANEL);

	// テキスト (2行)
	// XXX snprintf が色々うるさくてアレ…
	char text1[36 + 1 + 40];
	char text2[36 + 1];
	if (cursor < 0) {
		strlcpy(text1, "[   ]", sizeof(text1));
		strlcpy(text2, "Host  R:   G:   B:", sizeof(text2));
	} else {
		// この位置のパレット情報を取得
		__assume(cursor <= 0xff);
		snprintf(text1, sizeof(text1), "[$%02x]", cursor);
		if (videoctlr) {
			uint r = (ipal[cursor] >>  6) & 0x1f;
			uint g = (ipal[cursor] >> 11) & 0x1f;
			uint b = (ipal[cursor] >>  1) & 0x1f;
			uint i =  ipal[cursor]        & 1;
			snprintf(text1 + 5, sizeof(text1) - 5,
				" $%04x R:%02x G:%02x B:%02x I:%d",
				ipal[cursor], r, g, b, i);
			text1[36] = '\0';
		} else {
			if (rows == 1) {
				// Bt454
				uint r =  ipal[cursor] >> 8;
				uint g = (ipal[cursor] >> 4) & 0xf;
				uint b =  ipal[cursor]       & 0xf;
				snprintf(text1 + 5, sizeof(text1) - 5,
					" Guest R:%x  G:%x  B:%x", r, g, b);
			} else {
				// Bt458
				uint r =  ipal[cursor] >> 16;
				uint g = (ipal[cursor] >>  8) & 0xff;
				uint b =  ipal[cursor]        & 0xff;
				snprintf(text1 + 5, sizeof(text1) - 5,
					" Guest R:%02x G:%02x B:%02x", r, g, b);
			}
		}
		snprintf(text2, sizeof(text2), "Host  R:%02x G:%02x B:%02x",
			pal[cursor].r,
			pal[cursor].g,
			pal[cursor].b);
	}
	const int padding = WXTextScreen::DefaultPadding;
	DrawStringSJIS(padding, padding, text1);
	DrawStringSJIS(padding + font_width * 6, padding + font_height, text2);

	// ヘッダ
	DrawStringSJIS(pal0x, pal0y - font_height,
		"+0+1+2+3+4+5+6+7+8+9+a+b+c+d+e+f");
	for (int y = 0; y < rows; y++) {
		char buf[8];
		__assume(y < 16);	// XXX snprintf 対策。うーん
		snprintf(buf, sizeof(buf), "$%02x:", y * 16);
		DrawStringSJIS(padding, pal0y + y * font_height, buf);
	}

	// パレット
	Rect rect(0, 0, font_height - 1, font_height - 1);
	for (int y = 0; y < rows; y++) {
		for (int x = 0; x < 16; x++) {
			int cc = y * 16 + x;
			rect.x = pal0x + x * font_height;
			rect.y = pal0y + y * font_height;
			bitmap.FillRect(pal[cc], rect);
		}
	}
}

void
WXPalettePanel::OnMouse(wxMouseEvent& event)
{
	wxPoint pt = event.GetPosition();
	int new_cursor = -1;

	// pt がパネル内の座標なのに Leaving() になることがある。
	// pt がパネル外の座標なのに Leaving() にならないことがある。
	if (event.Leaving() == false && GetClientRect().Contains(pt)) {
		// マウスカーソルがパネル上にある

		int x = pt.x - pal0x;
		int y = pt.y - pal0y;
		if (x < 0 || y < 0) {
			// パレット外
			goto next;
		}
		x /= font_height;
		y /= font_height;
		if (x >= 16 || y >= rows) {
			// パレット外
			goto next;
		}

		new_cursor = y * 16 + x;
	}

 next:
	if (new_cursor != cursor) {
		cursor = new_cursor;
		Refresh();
	}
}


//
// パレットウィンドウ
//

// コンストラクタ
WXPaletteWindow::WXPaletteWindow(wxWindow *parent, const wxString& name)
	: inherited(parent, wxID_ANY, name)
{
	new WXPalettePanel(this);
	Fit();
}

// デストラクタ
WXPaletteWindow::~WXPaletteWindow()
{
}
