/* #define DEBUG		/* */
/* $Header: /home/vikas/src/nocol/portmon/RCS/portmon.c,v 2.5 2000/04/06 19:31:44 vikas Exp $  */

/* Copyright 1994 Vikas Aggarwal, vikas@navya.com */

/*+ 		portmon
 * FUNCTION:
 * 	Monitor TCP ports and reponses for NOCOL. It can send a string
 * and then parse the responses against a list. Each response can be
 * assigned a severity. Checks simple port connectivity if given a NULL
 * response string.
 *
 * CAVEATS:
 *	1) Uses case insensitive 'strstr' and not a real regular expression.
 *	2) Looks only at the first buffer of the response unless using the
 *	   timeouts to calculate the response time.
 *	3) Does not implement milli-second timers while reading responses
 *	   since that level of accuracy is not really required.
 *
 *  	Vikas Aggarwal, -vikas@navya.com
 */


/*
 * $Log: portmon.c,v $
 * Revision 2.5  2000/04/06 19:31:44  vikas
 * Now replaces all '\0' in the read stream with a '\n'. Needed by
 * a user who has a host which terminates lines with \0
 *
 * Revision 2.4  2000/04/04 04:46:22  vikas
 * Added close() to prevent max file open problem.
 * Better error logging.
 * Now checking for any read buffer even if \n not found.
 *
 * Revision 2.3  2000/01/20 05:27:18  vikas
 * Fixed error in processing sites where we are just testing connectivity.
 * Needed to call process_site() if the responselist was null.
 *
 * Revision 2.2  2000/01/18 04:14:25  vikas
 * Changed arrays to linked lists for large number of hosts.
 * Also, no longer 'blocks' while trying to read from a connected
 * socket, all reading is non-blocking.
 *
 * Revision 2.1  1999/10/19 04:31:17  vikas
 * Added support for reading 'simulconnect' from config file.
 * Also converts \r to '\r' like \n
 * Sends QUITSTRING to remote host before closing socket()
 * Bug fix from frank@ansto.gov.au to prevent buffer overflow.
 *
 * Revision 2.0  1998/10/01 19:36:52  vikas
 * MULTI-THREADED version. Uses 'select' to initate a 'connect' to multiple
 * hosts and processes them all at once. Derived from portmon.c v1.9
 *
 * Revision 1.9  1998/09/27 12:45:27  vikas
 * Changed all arrays to linked lists to have unlimited hosts in the config
 * file.
 *
 * Revision 1.8  1998/08/13 11:31:48  vikas
 * Changed goto into a while() loop. Was core dumping on Solaris with
 * gcc v2.8.1. Patch sent in by durrell@innocence.com.
 *
 * Revision 1.7  1998/07/31 18:32:00  vikas
 * bug in timeout array and increased size of read buffer
 *
 * Revision 1.6  1997/10/19 03:17:17  vikas
 * - cleanup for releasing with nocol v4.2
 * - added code for checking response times
 *
 * Revision 1.5  1994/05/16  02:12:55  vikas
 * bugfix from David Meyer (meyer@phloem.uoregon.edu) to distinguish
 * between the alarm going off and read error (n=0 vs. n<0)
 *
 * Revision 1.4  1994/05/09  01:03:37  vikas
 * Rewrite to use new nocol library 'startup_nocol' function.
 *
 */

/*  */
#ifndef lint
static char rcsid[] = "$Id: portmon.c,v 2.5 2000/04/06 19:31:44 vikas Exp $" ;
#endif

#include "portmon.h"

char	*prognm;
int	debug;

static int 	pollinterval ;
static int	rtimeout = RTIMEOUT;		/* select timeout in seconds */
static int	simulconnects = MAXSIMULCONNECTS;
static char	*configfile, *datafile;
static char 	*sender;
struct _harray	*hostlist;			/* global ptr to list */
extern char	*skip_spaces() ;		/* in libnocol */
static int 	get_inet_address();
static char	*Strcasestr();			/* like strstr() */

main(ac, av)
  int ac;
  char **av;
{
  extern char *optarg;
  extern int  optind;
  int	c ;
  int	fdout;

  if ((prognm = (char *)strrchr (av[0], '/')) == NULL)
    prognm = av[0] ;                        /* no path in program name */
  else
    prognm++ ;                                /* skip leading '/' */

#ifdef SENDER
  sender = SENDER ;
#else                                           /* delete the directory name */
  sender = prognm ;                         /* no path in program name */
#endif


  pollinterval = POLLINTERVAL ;
  hostlist = NULL;

  while ((c = getopt(ac, av, "df:")) != EOF)
    switch(c)
    {
    case 'd':
      debug++;
      break;
    case 'f':
      configfile = optarg ;
      break ;
    case '?':
    default:
      fprintf(stderr, "%s: Unknown flag: %c\n", prognm, c);
      fprintf(stderr, "Usage: %s [-d] [-f <config file>\n] ", prognm);
      exit (1);
    }

  nocol_startup(&configfile, &datafile);

  if ( (fdout = open(datafile, O_RDWR|O_CREAT|O_TRUNC, DATAFILE_MODE)) < 0)
  {
    fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
    perror (datafile);
    nocol_done();
  }

  if (readconfig(fdout) == -1)
    nocol_done();

  lseek(fdout, (off_t)0, SEEK_SET);	/* rewind file */

  signal(SIGPIPE, SIG_IGN);		/* ignore write failures */
  while (1)	/* forever */
  {
    int i;
    struct _harray *h;
    EVENT v;

    h = hostlist;

    while(h != NULL)	/* one entire pass */
    {
      struct _harray *harray[MAXSIMULCONNECTS + 1];

      bzero(harray, sizeof(harray));
      for (i = 0; i < simulconnects && h != NULL; h = h->next)
	harray[i++] = h;

      if (checkports(harray, i) == -1)	/* fills status and testseverity */
      {
	fprintf(stderr, "fatal error, checkports returned -1, exiting\n");
	exit (1);
      }
      for (i = 0; harray[i] != NULL; ++i)
      {
	read(fdout, &v, sizeof v);
	if (harray[i]->status == -1)
	  fprintf (stderr, "%s: Error in checkports, skipping site %s\n",
		   prognm, (harray[i])->hname);
	else
	{
	  update_event(&v, harray[i]->status,
		       /* value */ (u_long)(harray[i]->status),
		       harray[i]->testseverity) ;
	  lseek(fdout, -(off_t)sizeof(v), SEEK_CUR);
	  write(fdout, (char *)&v, sizeof(v));
	}
      }	/* for() */

    }	/* while(h != NULL) */

    lseek(fdout, (off_t)0, SEEK_SET);	/* rewind file */

    if (debug)
      fprintf(stderr, "(debug) %s: sleeping for %d...zzz\n\n", 
	      prognm,pollinterval);
#ifdef SVR4
    bsdsignal(SIGALRM, SIG_DFL);
#else
    signal(SIGALRM, SIG_DFL);		/* wake-up, in case blocked */
#endif
    sleep (pollinterval);

  }	/* while (forever) */

}	/* end:  main()  */

/*
 * This function opens a bunch of non-blocking connects to the remote
 * sites and calls process_host() as each socket becomes ready.
 * Handles multiple sites in parallel.
 */
checkports(hv, nhosts)
  struct _harray  *hv[];	/* list of site structures */
  int nhosts;
{
  int i, nleft;
  int maxfd = 0;
  int sockarray[MAXSIMULCONNECTS];
  time_t elapsedsecs = 0;
  fd_set rset, wset;

  if (nhosts <= 0) return;

  FD_ZERO(&rset); FD_ZERO(&wset);
  nleft = nhosts;

  if (debug > 1) fprintf(stderr, "Testing %d sites simultaneously\n", nhosts);

  /* issue a non-blocking connect to each host */
  for (i = 0; i < nhosts; ++i)
  {
    /* init the array for each host */
    hv[i]->status = 0;	/* assume down */
    hv[i]->testseverity = hv[i]->connseverity;	/* sev if cannot connect */
    hv[i]->wptr = hv[i]->writebuf; hv[i]->rptr = hv[i]->readbuf;
    *(hv[i]->readbuf) = '\0';
    hv[i]->elapsedsecs = 0;

    if ((sockarray[i] = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
      perror("socket");
      return (-1);	/* fatal error */
    }

    /* NBconnect() will return zero/positive if it gets an immediate
     * connection, else it will set errno.
     */
    if (NBconnect(sockarray[i], hv[i]->ipaddr, hv[i]->port) < 0 &&
	errno != EINPROGRESS)
    {		/* some error */
      if (debug)
	fprintf(stderr, "connect() failed for %s:%d- %s\n",
		hv[i]->ipaddr, hv[i]->port, sys_errlist[errno]);
      close(sockarray[i]);
      --nleft;
    }
    else
    {
	FD_SET(sockarray[i], &rset);
	FD_SET(sockarray[i], &wset);
	if (sockarray[i] > maxfd) maxfd = sockarray[i];
    }

  }	/* for (i = 1..nhosts) */

  /* now do a select on all the sockets, and process each one */
  while (nleft > 0) {
    int n;
    fd_set rs, ws;
    time_t curtime;
    struct timeval tval;
    
    rs = rset; ws = wset;
    curtime = time(NULL);
    if (elapsedsecs >= rtimeout)
    {
      for (i = 0; i < nhosts; ++i)
	close(sockarray[i]);
      return 0;
    }
    tval.tv_sec = rtimeout - elapsedsecs;
    tval.tv_usec = 0;
#ifdef DEBUG
    if (debug > 2)
      fprintf(stderr,
	      "\ncheckports() calling select(), timeout %ld, nleft= %d\n",
	      tval.tv_sec, nleft);
#endif
    if ( (n = select(maxfd + 1, &rs, &ws, NULL, &tval)) <= 0)
    {
      if (n < 0 && errno != EINTR)
      {
	perror("select()");
	return (-1);	/* fatal ?? */
      }
      else
      {
	if (debug) fprintf(stderr, "timeout for connect()\n");
	for (i = 0; i < nhosts; ++i)
	  close(sockarray[i]);
	break;	/* out of while */
      }
    }	/* if (n = select) */

    elapsedsecs += (time(NULL) - curtime);	/* stop the clock */

#ifdef DEBUG
    if (debug > 2)
      fprintf(stderr, "select() returned %d, elapsedsecs = %d\n",
	      n, elapsedsecs);
#endif
    /* see which socket is ready from the select() and process it */
    for (i = 0; i < nhosts; ++i) 
    {
      if ( FD_ISSET(sockarray[i], &rs) || FD_ISSET(sockarray[i], &ws) )
      {
	int error;
	int len;
	len = sizeof(error);
	if (getsockopt(sockarray[i], SOL_SOCKET, SO_ERROR, (void *)&error,
		       &len) < 0 || error != 0)
	{	/* some error in connect() */
	  if (debug)
	  {
	    fprintf(stderr, "connect() failed to %s:%d- ", 
		    hv[i]->hname, hv[i]->port);
	    if (error) fprintf(stderr, "%s\n", sys_errlist[error]);
	    else  fprintf(stderr, "%s\n", sys_errlist[errno]);
	  }
	  FD_CLR(sockarray[i], &rset); FD_CLR(sockarray[i], &wset);
	  close(sockarray[i]);	/* ensure this is closed */
	  --nleft;
	  if (debug > 2)
	    fprintf(stderr, "DONE #%d. %s:%d (%s)\n",
		    i, hv[i]->hname, hv[i]->port, hv[i]->ipaddr);
	}
	else	/* ready for reading/writing and connected */
	{
	  hv[i]->elapsedsecs = elapsedsecs;	/* store elapsed time */

	  if (FD_ISSET(sockarray[i], &ws))
	    send_hoststring(sockarray[i], hv[i]);

	  if (FD_ISSET(sockarray[i], &rs) || hv[i]->responselist == NULL)
	    if ( process_host(sockarray[i], hv[i]) == 1 )
	    {
	      FD_CLR(sockarray[i], &rset); FD_CLR(sockarray[i], &wset);
	      close(sockarray[i]);	/* ensure this is closed */
	      --nleft;
	      if (debug > 2)
		fprintf(stderr, "DONE #%d. %s:%d (%s)\n",
			i, hv[i]->hname, hv[i]->port, hv[i]->ipaddr);
	    }

	  if (hv[i]->wptr == NULL || *(hv[i]->wptr) == '\0')
	    FD_CLR(sockarray[i], &wset);		/* nothing to write */
	}	/* if else getsockopt() */

      }	/* if-else FD_ISSET */

    }	/* for() */

  }	/* while(nleft) */

  /* here if timeout or all the connects have been processed */

  if (nleft && debug > 1)
    fprintf(stderr, " %d sites unprocessed (no response)\n", nleft);

  for (i = 0; i < nhosts; ++i)		/* close any open sockets */
    close(sockarray[i]);

  return (0);

}	/* check_ports() */

/*+
 * Send a string to a host
 */
send_hoststring(sock, h)
  int sock;		/* connected socket */
  struct _harray *h;
{
  int n;

  if (debug)
    fprintf(stderr, " (debug) send_hoststring('%s:%d')\n", h->hname, h->port);

  /* if socket is non-blocking, undo this flag */
/*  sflags = fcntl(sock, F_GETFL, 0);		/*  */
/*  fcntl(sock, F_SETFL, sflags ^ O_NONBLOCK);	/* set blocking */
/*  fcntl(sock, F_SETFL, sflags | O_NONBLOCK);	/* set nonblocking */

  /* return if nothing to send */
  if (h->wptr == NULL || *(h->wptr) == '\0')
    return 1;

  /* send our string */
  if ( (n = write(sock, h->wptr, strlen(h->wptr))) < 0)
  {
    if (errno == EWOULDBLOCK)
      return 0;
#ifdef EAGAIN		/* sysV ? */
    if (errno == EAGAIN)
      return 0;
#endif
    if (debug)
      fprintf(stderr, 
	      "Error in write() for '%s:%d' - %s\n", h->hname, h->port,
	      sys_errlist[errno]);
    close (sock);
    h->testseverity = h->connseverity;
    h->status = 0;	/* mark as down */
    return(1);	/* finished testing */
  }	/* end if (couldn't write)  */

  (h->wptr) += n;
  if (*(h->wptr) != '\0')
  {
#ifdef DEBUG
    if (debug > 1)
      fprintf(stderr, "  (debug) Host %s:%d- sent %d bytes\n",
	      h->hname, h->port, n);
#endif
    return 0;
  }

  if (debug)
    fprintf(stderr, "  (debug) Host %s:%d- Sent string '%s'\n",
	    h->hname, h->port, h->writebuf) ;

  return 1;

}	/* send_hoststring() */
  
/*+ 
 * FUNCTION:
 * 	Checks the port for the structure _harray passed to it.
 * Return value is '1' if finished processing, '0' if yet to finish.
 */
process_host(sock, h)
  int sock;		/* connected socket */
  struct _harray *h;
{
  int i, n;
  int sflags;
  int buflen, maxsev;
  register char *r, *s;

  if (debug)
    fprintf(stderr, " (debug) process_host('%s:%d')\n", h->hname, h->port);

  /* if socket is non-blocking, undo this flag */
/*  sflags = fcntl(sock, F_GETFL, 0);		/*  */
/*  fcntl(sock, F_SETFL, sflags ^ O_NONBLOCK);	/* set blocking */
/*  fcntl(sock, F_SETFL, sflags | O_NONBLOCK);	/* set nonblocking */

  if (h->responselist == NULL)		/* no responses to check */
    if (h->wptr == NULL || *(h->wptr) == '\0')	/* done sending */
    {
      if (debug > 1)
	fprintf(stderr,
		" process_host(%s:%d) nothing to send/recieve, done\n",
		h->hname, h->port);
      h->testseverity = E_INFO;
      h->status = 1;		  	/* site up */
      if (h->quitstr && *(h->quitstr))
	write(sock, h->quitstr, strlen(h->quitstr));
      close (sock);
      return(1);	/* done testing */
    }

  /* now fill any remaining read buffer space we have */
  buflen = h->readbuf + sizeof(h->readbuf) - h->rptr;	/* amount we can read*/
  n = read(sock, h->rptr, buflen - 1);
#ifdef DEBUG  	/* */
  if (debug > 2)
    fprintf(stderr, "  read %d bytes from %s:%d\n", n, h->hname, h->port);
#endif	/* */
  if (n < 0)
  {		/* read() error */
    if (errno == EWOULDBLOCK)
      return (0);
#ifdef EAGAIN
    if (errno == EAGAIN)
      return 0;
#endif
    if (debug)
      fprintf(stderr, 
	      "Error in read() for '%s:%d' - %s\n", h->hname, h->port,
	      sys_errlist[errno]);
    close (sock);
    h->testseverity = h->connseverity;
    h->status = 0;	/* mark as down */
    return(1);		/* finished testing */
  }	/* end if (n < 0)  */

  /* if n==0, then we have read end of file, so do a final check_resp() */

  /* replace any \0 in the stream with a \n */
  for (i = 0; i < n ; ++i)
    if ((h->rptr)[i] == '\0')
      (h->rptr)[i] = '\n';

  (h->rptr) += n;		/* increment pointer */
  *(h->rptr) = '\0';

  if (n > 0 && (r = (char *)strrchr(h->readbuf, '\n')) == NULL)  /* no \n */
  {
    if ( n < (buflen - 1) )	/* remaining empty buffer space */
      return (0);		/* need to continue reading */
    else			/* filled buffer, but no \n yet */
      r = h->rptr - 32;		/* set to about 32 chars back from end */
  }

  /* here if end-of-file or found a newline */
  maxsev = check_resp(h->readbuf, h->responselist);

  if (maxsev == -1)
  {	/* no match in response list */
    if (n == 0)		/* end of file, so we are done */
      maxsev = h->connseverity;
    else
    {
      for (++r, s = h->readbuf; *r; )
	*s++ = *r++;	/* shift stuff after \n to start of readbuf */
      *s = '\0';	       	/* lets be safe */
      h->rptr = s;		/* point to next location to be read */
      return 0;			/* still not done */
    }
  }
  else
  {			/* Here if we found a match in check_resp() */
    if (h->timeouts[0] != 0)
    {				/* we are checking port speed */
      if (debug > 1)
	fprintf(stderr,"  (debug) elapsed time= %ld secs\n", h->elapsedsecs);
      if (h->elapsedsecs < h->timeouts[0])	maxsev = E_INFO;
      else if (h->elapsedsecs < h->timeouts[1])	maxsev = E_WARNING;
      else if (h->elapsedsecs < h->timeouts[2])	maxsev = E_ERROR;
      else maxsev = E_CRITICAL;
    }
  }

  if (debug)
    fprintf(stderr," (debug) process_host(%s:%d): returning severity %d\n",
	    h->hname, h->port, maxsev);
  if ( (h->testseverity = maxsev) == E_INFO )
    h->status = 1;
  else
    h->status = 0;

  if (h->quitstr && *(h->quitstr))
    write(sock, h->quitstr, strlen(h->quitstr));
  close(sock);

  return(1);	/* done processing */

}	/* end:  process_host() */

/*+ 
 * FUNCTION:
 * 	Check the list of responses using Strcasestr()
 */
check_resp(readstr, resarr)
  char *readstr;
  struct _response  *resarr ;
{
  struct _response *r;

  if (debug > 1)
    fprintf(stderr, "  (debug) check_resp() '%s'\n", readstr);

  for (r = resarr; r ; r = r->next)
  {
    if (r->response == NULL)
      continue;
    if (Strcasestr(readstr, r->response) != NULL)
    {
#ifdef DEBUG
      if (debug > 1)
	fprintf(stderr,"   (debug) check_resp(): Matched '%s'\n", r->response);
#endif
      return(r->severity);
    }
  }	/* for() */

  if (debug)
    fprintf (stderr, "  check_resp(): No response matched for site\n");

  return(-1);		/* No response matched given list */

}	/* check_resp()  */


/*+ 
 * FUNCTION:
 * 	Duplicate a string. Can handle a NULL ptr.
 */
static char *Strdup(s)
  char *s ;
{
  char *t ;

  if (s == NULL)
  {
    t = (char *)malloc(1);
    *t = '\0';
  }
  else
  {
    t = (char *)malloc(strlen(s) + 1);
    if (t != NULL)
      (char *)strcpy(t, s);
  }

  return (t);
}

/*
 * Our very own strstr() function. Does a case-insensitive match.
 * Finds the first occurrence of the substring 'needle' in the
 * string 'haystack'.
 * Returns
 *	Ptr in 'haystack' to the begining of 'needle'
 *	Ptr to 'haystack' if needle is empty
 *	NULL if needle is not found.
 *
 */
static char *Strcasestr(haystack, needle)
  char *haystack;	/* string */
  char *needle; 	/* string to find */
{
  char p, q;
  size_t needlelen;

  if (haystack == NULL)
    return NULL;

  if (needle == NULL || *needle == 0) 
    return (char *)haystack;
    
  p = *needle++;
  needlelen = strlen(needle);

  for ( ; ; ) 
  {
    register char *s1, *s2;
    int n;

    do
    {		/* match first character */
      if ((q = *haystack++) == 0)
	return (NULL);
    } while ( tolower(q) != tolower(p) );

    s1 = haystack; s2 = needle; n = needlelen;
    do
    {
      if (tolower(*s1) != tolower(*s2++)  || *s1 == 0)	/* */
	break;
      ++s1;
    } while (--n != 0);

    if (n == 0 || *s1 == 0)
      break;	/* found, break out of forever loop */

  }	/* end for */
  haystack--;
  return ((char *)haystack);
}	/* Strcasestr() */

/*+ 
 * FUNCTION:
 * 	Read the config file.
 * POLLINTERVAL ppp
 * HOST  <hostname>  <address>  <var>  <port> <failseverity>  <send string>
 * <severity>	response1 till end of this line
 * <severity>   response2 till end of this line
 *
 */
#define NEXTTOK  (char *)skip_spaces(strtok(NULL, " \t"))

readconfig(fdout)
  int fdout ;				/* output data filename */
{
  int mxsever, i;
  char *j1;				/* temp string pointers */
  FILE *cfd ;
  EVENT v;                            	/* Defined in NOCOL.H */
  char record[MAXLINE] ;
  struct tm *loctime ;
  time_t locclock ;                   	/* Don't use 'long'    */
  struct _harray *h = hostlist;		/* temp ptr */
  struct _response *r;

  if ((cfd = fopen(configfile, "r")) == NULL)
  {
    fprintf(stderr, "%s error (init_sites) ", prognm) ;
    perror (configfile);
    return (-1);
  }

  /*
   * Fill in the static data stuff
   */
  bzero (&v, sizeof(v)) ;

  locclock = time((time_t *)0);
  loctime = localtime((long *)&locclock);

  /* fill in the EVENT structure */
  v.mon = loctime->tm_mon + 1;        v.day = loctime->tm_mday;
  v.hour = loctime->tm_hour;          v.min = loctime->tm_min;
  strncpy (v.sender, sender, sizeof(v.sender) - 1);
  strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
  v.var.threshold = 0 ;
  v.nocop = SETF_UPDOUN (0, n_UNKNOWN); /* Set all to UNKNOWN   */
  v.severity = E_INFO ;

  /*
   * Now parse the config file
   */

  while(fgetLine(cfd, record, MAXLINE - 3) > 0 ) 	/* keeps the \n */
  {
    static int skiphost;
    int port;
    int checkspeed = 0;
    int readquitstr = 0;
    struct sockaddr_in sin;			/* temporary */

    record[strlen(record) - 1] = '\0' ;		/* chop off newline */
    if (*record == '#' || *(skip_spaces(record)) == '\0')
      continue ;

    if (strncasecmp(record, "POLLINTERVAL", strlen("POLLINTERVAL")) == 0)
    {
      strtok(record, " \t");
      j1 = (char *)skip_spaces(strtok(NULL, "")) ;
      if (!j1 || *j1 == '\0' || (pollinterval = atoi(j1)) <= 0)
	fprintf(stderr, "%s: bad or missing pollinterval value\n", prognm);

      pollinterval = (pollinterval > 0 ? pollinterval : POLLINTERVAL);
      if (debug)
	fprintf (stderr, "(debug) %s: Pollinterval = %d\n", 
		 prognm, pollinterval);

      continue;
    }

    if ( strncasecmp(record, "READTIMEOUT", strlen("READTIMEOUT")) == 0 ||
	 strncasecmp(record, "TIMEOUT", strlen("TIMEOUT")) == 0 )
    {
      strtok(record, " \t");
      j1 = (char *)skip_spaces(strtok(NULL, "")) ;
      if (!j1 || *j1 == '\0' || (rtimeout = atoi(j1)) <= 0)
	fprintf(stderr, "%s: bad or missing READTIMEOUT value\n", prognm);

      rtimeout = (rtimeout > 0 ? rtimeout : RTIMEOUT);
      if (debug)
	fprintf (stderr, "(debug) %s: ReadTimeout = %d\n", 
		 prognm, rtimeout);

      continue;
    }

    if (strncasecmp(record, "SIMULCONNECT", strlen("SIMULCONNECT")) == 0)
    {
      strtok(record, " \t");
      j1 = (char *)skip_spaces(strtok(NULL, "")) ;
      if (!j1 || *j1 == '\0' || (simulconnects = atoi(j1)) <= 0)
	fprintf(stderr, "%s: bad or missing SIMULCONNECT value\n", prognm);

      if (simulconnects > MAXSIMULCONNECTS || simulconnects <= 0)
	simulconnects = MAXSIMULCONNECTS;
#ifdef FD_SETSIZE
      if (simulconnects > (FD_SETSIZE - 6))
	simulconnects = FD_SETSIZE - 6;
#endif
      if (debug)
	fprintf (stderr, "(debug) %s: SimultaneousConnects = %d\n", 
		 prognm, simulconnects);

      continue;
    }

    if (strncasecmp(record, "HOST", 4) != 0)	/* Not HOST line */
    {
      if (skiphost || h == NULL)	/* not within a HOST block */
      {
	fprintf(stderr, "readconfig: ignoring config line %s\n", record);
	continue ;
      }

      j1 = strtok(record," \t") ;			/* severity level */
      j1 = (char *)skip_spaces(j1) ;
      switch(tolower(*j1))
      {
      case 'c': case '1': mxsever = E_CRITICAL;	break;
      case 'e': case '2': mxsever = E_ERROR; 	break;
      case 'w': case '3': mxsever = E_WARNING;	break;
      case 'i': case '4': mxsever = E_INFO; 	break;
      case 't': mxsever = E_CRITICAL; ++checkspeed; break;	/* TIME */
      case 'q': ++readquitstr ; break;		/* QUIT string */
      default:  mxsever = E_CRITICAL;		break ;
      }

      j1 = (char *)skip_spaces(strtok(NULL, "")) ;

      if (readquitstr)
      {
	if (j1 == NULL || *j1 == '\0')
	  fprintf(stderr, "%s: missing QUIT string\n");
	else {
	  strcat (j1, "\r\n");
	  h->quitstr = (char *)Strdup(j1);
	}
	continue;
      }		/* if (readquitstr) */

      if (checkspeed && h->timeouts[0] == 0)
      {		 		/* read timeout thresholds */
	if (3 != sscanf(j1, "%u %u %u", &(h->timeouts[0]),
			&(h->timeouts[1]), &(h->timeouts[2])))
	{
	  fprintf(stderr, "%s: need 3 time thresholds, invalid syntax\n",
		  prognm);
	  continue;
	}
	strtok(j1, " \t"); strtok(NULL, " \t"); strtok(NULL, " \t");
	j1 = (char *)skip_spaces(strtok(NULL, ""));
      }		/* end if(checkspeed) */

      if (j1 == NULL || *j1 == '\0')
      {
	fprintf(stderr, "%s: missing response string, skipped\n",prognm);
	fprintf(stderr, "For null responses, dont put any lines\n");
	continue;
      }

      if (h->responselist == NULL)
      {
	r = (struct _response *)malloc(sizeof(struct _response));
	h->responselist = r;
      }
      else
      {
	r->next = (struct _response *)malloc(sizeof(struct _response));
	r = r->next;
      }
      bzero(r, sizeof(struct _response));
      r->response = (char *)Strdup(j1);
      r->severity = mxsever ;

      continue ;	/* next input line */
    }

    /*
     * Here if parsing a HOST line
     */
	
    skiphost = 0;		/* assume valid config */
    strtok(record, " \t");	/* skip HOST keyword */	
	    
    strncpy(v.site.name, NEXTTOK, sizeof(v.site.name) - 1);
    strncpy(v.site.addr, NEXTTOK, sizeof(v.site.addr) - 1);
    if (get_inet_address(&sin, v.site.addr) == 0)        /* bad address */
    {
      fprintf(stderr,
	      "(%s): Error in addr '%s' for site '%s', ignoring\n",
	      prognm, v.site.addr, v.site.name);
      skiphost = 1;
      continue ;
    }
    strncpy(v.var.name, NEXTTOK, sizeof(v.var.name) - 1);
	
    if ((port = atoi(NEXTTOK)) == 0)
    {
      fprintf(stderr,
	      "(%s): Error in port for site '%s', ignoring\n",
	      prognm, v.site.name);
      skiphost = 1;
      continue ;
    }
	
    j1 = NEXTTOK;	/* severity level */
    if (j1 == NULL)
      j1 = "c";
    switch(tolower(*j1))
    {
    case 'c': mxsever = E_CRITICAL; break;
    case 'e': mxsever = E_ERROR; break;
    case 'w': mxsever = E_WARNING; break;
    case 'i': mxsever = E_INFO; break;
    default:  mxsever = E_CRITICAL ; break ;
    }

    /*
     *if (debug)
     *fprintf (stderr, "Name: %s %s, VAR: %s, Port %d SEV: %d\n",
     *       v.site.name,v.site.addr,v.var.name, port, mxsever);
     */
	
    /* the rest of string is SEND */
    if (h) {
      h->next = (struct _harray *)malloc(sizeof(struct _harray));
      h = h->next;
    }
    else {
      h = (struct _harray *)malloc(sizeof(struct _harray));
      if (hostlist == NULL)
	hostlist = h;
    }
    bzero (h, sizeof(struct _harray));

    h->writebuf = NULL ;		/* Initialize.. */
    j1 = (char *)skip_spaces(strtok(NULL, ""));
    if (j1 && *j1)
    {
      /* We NEED a newline at the end of the send strings. http requires 2 */
      raw2newline(j1);
      /* if (*(j1 + strlen(j1) - 1) != '\n')	/* */
      strcat(j1, "\r\n");
      h->writebuf = (char *)Strdup(j1) ;

      /*if(debug)
       *fprintf(stderr, "(debug)Sendstring: >>>%s<<<\n", h->send);
       */
    }

    h->port = port;
    h->connseverity = mxsever ;
    h->hname = (char *)Strdup(v.site.name);
    h->ipaddr = (char *)Strdup(v.site.addr);
	
    if (write (fdout, (char *)&v, sizeof(v)) != sizeof(v))
    {
      fprintf(stderr, "%s (write): %s\n", prognm,sys_errlist[errno]);
      nocol_done() ;
    }

  }	/* end: while */

  fclose (cfd);       		/* Not needed any more */

  if (debug > 1)
    for (h = hostlist, i=0 ; h; h = h->next)
    {
      fprintf(stderr, "#%d. Host=%s %s :%d, MaxSev= %d, Sendstr=%s",
	      i++, h->hname, h->ipaddr, h->port,  h->connseverity,
	      (h->writebuf ? h->writebuf : "\n"));
      for (r = h->responselist; r; r = r->next)
	fprintf(stderr, "\t%s (sev=%d)\n", r->response, r->severity);
    }

  return(1);                          /* All OK  */

}  /* end:  init_sites() */



/*
 * Takes a string containing one or many sequences of "\r\n" and replaces
 * these two chars with a newline. The original string is modified.
 */
char *raw2newline(inbuf)
  char *inbuf;
{
  register char *p, *q;

  if(!inbuf)
    return (char *)NULL;

  p = q = inbuf;

  for (; ;)
  {			/* forever() */
    while (*p && *p != '\\')
      *q++ = *p++;

    if (*p == '\0') {
      *q = '\0';
      return(inbuf);
    }

    switch (*(p+1))
    {
    case 'n':
      *q++ = '\n';
      p = (p+2);	/* skip \ and n */
      break;

    case 'r':
      *q++ = '\r';
      p = (p+2);	/* skip \ and r */
      break;

    default:
      *q++ = *p++;
      break;
    }	/* end switch */

  }	/* end forever () */

}	/* raw2newline() */


/*
 * Start non-blocking connect to remote host
 */
NBconnect(s, host, port)
  int s;	/* socket */
  char *host;
  int port;
{
  int sflags;
  struct sockaddr_in sin;

  if (! get_inet_address(&sin, host))
  {
    fprintf(stderr, "inet_addr() failed for ");
    perror(host);
    return (-1);
  }
  sin.sin_port = htons(port);

  sflags = fcntl(s, F_GETFL, 0);
  fcntl(s, F_SETFL, sflags | O_NONBLOCK);
  return (connect(s, (struct sockaddr *)&sin, sizeof(sin)));

} 

/*
 * Instead of inet_addr() which can only handle IP addresses, this handles
 * hostnames as well...
 * Return 0 on error, 1 if okay.
 */
static int get_inet_address(addr, host)
  struct sockaddr_in *addr;		/* must be a malloced structure */
  char *host;
{
  register struct hostent *hp;

  bzero((char *)addr, sizeof (struct sockaddr_in));

  addr->sin_family = AF_INET;

  addr->sin_addr.s_addr = (u_long) inet_addr(host);
  if (addr->sin_addr.s_addr == -1 || addr->sin_addr.s_addr == 0) {
    if ((hp = gethostbyname(host)) == NULL) {
      fprintf(stderr, "%s is unknown host\n", host);
      return 0;
    }
#ifdef h_addr		/* in netdb.h */
    bcopy((char *)hp->h_addr, (char *)&(addr->sin_addr), hp->h_length);
#else
    bcopy((char *)hp->h_addr_list[0], (char *)&(addr->sin_addr), hp->h_length);
#endif
  }
  return 1;
}
