
/* Handle Automatic Features */

#include "gnutella.h"

#include "interface.h"

#include "auto.h"

#include "search.h"

#include <sys/stat.h>
#include <sys/types.h>
#include <regex.h>


// todo notes: re-scan shared at certain interval

gboolean auto_or(char *, char *);
gboolean auto_and(char *, char *);
void auto_set_param(guint32, gchar *);
gint timeval_subtract(struct _GTimeVal *, struct _GTimeVal *, struct _GTimeVal *);

gboolean auto_enabled = FALSE; // global, flag to show we are working
guint32 auto_save_search_reissue_timeout; // global, re set this value at stop or quit
GSList *sl_auto_record = NULL; // global, pointer to all the list records

guint32 auto_timer = 0;
gint auto_net_stable = 0; // seconds the net has been stable (nodes connected)
gint auto_picked_row = 0; // the list row we picked to use
gint auto_work_row = 0; // the list row we ar working on now
gint auto_search_row = 0; // the search result list row we are on now
gint indicate_cnt = 0; // used for the little gui indicator that we are working
guint32 search_timer = 999999; // count time between searches
guint32 search_timeout = 0; // count time we haven't gotten a new result back
guint32 queue_count = 0; // how many files have been queued from this search
gboolean auto_stopped = FALSE; // flag if we are working or not
gboolean we_searched = FALSE; // TRUE if we have searched once


// Variables set by the auto config file, these are the defaults

gboolean auto_on_at_start = FALSE; // global, Run at program startup?
gboolean auto_log_enabled = FALSE; // Automation log file write enabled
guint32 auto_none_found = 10; // Wait xx minutes if no files are found, set in gui too
guint32 auto_queued_stop = 200; // This many files in the download queue, kill the search and wait
guint32 auto_queued_start = 20; // This many files in the download queue, start searching again
guint32 auto_search_time = 600; // How many seconds must pass before allowing another search
guint32 auto_search_timeout = 180; // Seconds to wait after last results come back before cancel
guint32 auto_flush_timeout = 3600; // Max seconds file will sit in queue, get rid of old data
gint auto_node_kick_mode = 1; // When to kick a node (options 0 to 3) default 1
gint auto_debug_mode = 0; // Debug printout mode, for programmers (level 0 to 5)


// define the keywords to read from the automation config file

enum
{
	kw_none_found = 0, kw_queued_stop, kw_queued_start, kw_search_time, kw_search_timeout,
	kw_flush_timeout, kw_list_filters, kw_list_strings, kw_list_search, kw_on_at_start,
	kw_log_enabled, kw_node_kick_mode, kw_debug_mode, kw_end
};

gchar *kw[] =
{
	"none_found",		// kw_none_found
	"queued_stop",		// kw_queued_stop
	"queued_start",	// kw_queued_start
	"search_time",		// kw_search_time
	"search_timeout",	// kw_search_timeout
	"flush_timeout",	// kw_flush_timeout
	"list_filters",	// kw_list_filters
	"list_strings",	// kw_list_strings
	"list_search",		// kw_list_search
	"on_at_start",		// kw_on_at_start
	"log_enabled",		// kw_log_enabled
	"node_kick_mode",	// kw_node_kick_mode
	"debug_mode",		// kw_debug_mode
	NULL
};

// temp buffers to store the incoming strings

gchar auto_tmp[4096];
gchar buf_list_filters[4096];
gchar buf_list_strings[4096];
gchar buf_list_search[4096];



// Start of filter routines
// Always provide a help printout, including appropriate data that can help in
// debugging problems. Keep the printout short because it may print for every
// file name sent here. Follow examples below.
// Return FALSE to not download this file, TRUE if it passes
// You must also include a entry for your filter in "auto_file_select" and send
// it the proper strings or data


// the "not" filter
gboolean auto_filter_not(gchar *filename, gchar *string, gboolean show_help, gint D)
{
	// auto_or returns TRUE if a word is in there
	// but we return FALSE so the file is not queued
	// Check file against "not" string
	g_strstrip(string);
	if (show_help)
		auto_show_help("Filter: not, file='%s', string='%s'\n", filename, string);

	return !auto_or(filename, string);
}


// the "size" filter
gboolean auto_filter_size(guint32 size, gchar *string, gboolean show_help, gint D)
{
	gchar *maxsize;

	g_strstrip(string);
	if (show_help)
		auto_show_help("Filter: size, Usage='minsize maxsize', size='%d', string='%s'\n", size, string);

	maxsize = string; // copy pointer to local
	while (*maxsize && (*maxsize != ' ')) maxsize++; // move to next "word", if any
	if (!*maxsize) { // only one word in string
		if (*string) if (size < (atol(string) * 1000)) return FALSE; // minsize
		}
	else {
		*maxsize = '\0'; // terminate after minsize
		if (*string) if (size < (atol(string) * 1000)) return FALSE; // minsize
		maxsize++; // move past the 00
		if (*maxsize) if (size > (atol(maxsize) * 1000)) return FALSE; // maxsize
		}
	return TRUE;
}


// the "regex" filter
gboolean auto_filter_regex(gchar *filename, gchar *string, gboolean show_help, gint D)
{
	gint regex_ret;
	regex_t pregex;
	regmatch_t pmatch[1];

	if (show_help)
		auto_show_help("Filter: regex, Usage=(see regex docs), file='%s', string='%s'\n", filename, string);

	regex_ret = 0; // You "compile" a struct before using regexec
	regex_ret = regcomp(&pregex, string, REG_ICASE|REG_EXTENDED );
	if (regex_ret != 0) {
		auto_show_help("Filter: regex, compile failure on '%s'\n", string);
		regfree(&pregex); // free the "compiled" struct
		return FALSE;
		}

	regex_ret = regexec(&pregex, filename, 1, pmatch, 0); // do it
	regfree(&pregex); // free the "compiled" struct

	if (regex_ret != 0) return FALSE;
	return TRUE; // it matches the regex patern
}


// the "or" filter
gboolean auto_filter_or(gchar *filename, gchar *string, gboolean show_help, gint D)
{
	// auto_or returns TRUE if a word is in there
	g_strstrip(string);
	if (show_help)
		auto_show_help("Filter: or, file='%s', string='%s'\n", filename, string);

	return auto_or(filename, string);
}


// the "and" filter
gboolean auto_filter_and(gchar *filename, gchar *string, gboolean show_help, gint D)
{
	// auto_and returns TRUE if all words are in there
	g_strstrip(string);
	if (show_help)
		auto_show_help("Filter: and, file='%s', string='%s'\n", filename, string);

	return auto_and(filename, string);
}


// the "extern" filter
gboolean auto_filter_extern(struct record *rc, struct auto_record *r, gchar *string, gboolean show_help, gint D)
{
	gint x, max;  // notice that we don't g_strstrip the string, so be careful
	gchar *s, tmp[5500], filename[5200], search[512], therest[1030]; // string space is created on stack, very fast
	struct _GTimeVal start_time, end_time, elapsed_time;

	g_get_current_time(&start_time);

	// this may seem like a waste of time, but we must have no / in name, plus it is exactly like it is saved
	strcpy(filename, rc->name);
	s = filename;  // correct the file name first, any slash = underline, for DOS or Linux file systems
	while (*s) { if (*s == '/') *s = '_'; if (*s == '\\') *s = '_'; s++; }

	g_snprintf(search, sizeof(search), "%s", r->search); // only take a limited number of chars
	s = search;  // any slash = underline so we don't mess things up later
	while (*s) { if (*s == '/') *s = '_'; if (*s == '\\') *s = '_'; s++; }

	// We append filename/size/speed/status/search/tag_data to the command line, slash separates
	// Filename will never have a slash in it, we changed them to underscore earlier
	// tag_data is a copy of the actual raw tag data, may be MP3 tag info or "urn:sha1" etc..
	// tag_data may have a forward slash in it! That's why it's last on the line, you deal with it
	if (rc->tag_data != NULL)
		g_snprintf(therest, sizeof(therest), "/%d/%d/%d/%s/%s",
			rc->size, rc->results_set->speed, rc->results_set->status, search, rc->tag_data);
	else
		g_snprintf(therest, sizeof(therest), "/%d/%d/%d/%s", rc->size, rc->results_set->speed, rc->results_set->status, search);

	strcat (filename, therest);
	s = filename;
	strcpy (tmp, string); // start to build the command line
	strcat (tmp, " \""); // we use quotes to send the filename info on one line
	x = strlen(tmp);
	max = sizeof(tmp) - 5; // adjust for end of line stuff we add
	while (*s && x < max) {  // filter out anything that the shell can't handle, use backslashed characters
		tmp[x] = *s;
		if (*s == '$') { tmp[x] = '\\'; x++; tmp[x] = '$';} // dollar sign
		if (*s == '"') { tmp[x] = '\\'; x++; tmp[x] = '"';} // double quote
		if (*s == '\\') { tmp[x] = '\\'; x++; tmp[x] = '\\';} // backslash
		if (*s == '`') { tmp[x] = '\\'; x++; tmp[x] = '`';} // backquote
		x++; s++;
		}
	tmp[x] = '"'; // add the ending quote and terminate the string
	tmp[x + 1] = '\0';

	if (show_help)
		auto_show_help("Filter: extern, Shell command '%s'\n", tmp);

	x = system(tmp); // shell out, return zero to queue the file

	g_get_current_time(&end_time);

	if (timeval_subtract(&elapsed_time, &end_time, &start_time)) {
		elapsed_time.tv_sec = 0;
		elapsed_time.tv_usec = 0;
		}
	if (elapsed_time.tv_sec < 0) elapsed_time.tv_sec = 0;
	if (elapsed_time.tv_usec < 1) elapsed_time.tv_usec = 1;
	elapsed_time.tv_usec = elapsed_time.tv_usec / 1000;

	if (show_help)
		auto_show_help("Filter: extern, Command completed with %d, errno= %d, Time %lds %ldms\n",
				x, errno, elapsed_time.tv_sec, elapsed_time.tv_usec);

	if (x == 0) return TRUE;
	return FALSE;
}


// the "set" filter
gboolean auto_filter_set(gchar *string, gboolean show_help, gint D)
{
	gchar tmp[4300], *s, *k, *v;
	guint32 i, n = 0;
	gboolean end = FALSE;

	if (show_help)
		auto_show_help("Filter: set, string='%s'\n", string);

	strcpy(tmp, string);
	s = tmp;
	while (*s)
	{
		n++;
		while (*s && (*s == ' ' || *s == ',')) s++;
		if (!((*s >= 'A' && *s <= 'Z') || (*s >= 'a' && *s <= 'z'))) continue;
		k = s;
		while (*s =='_' || (*s >= 'A' && *s <= 'Z') || (*s >= 'a' && *s <= 'z') || (*s >= '0' && *s <= '9')) s++;
		if (*s != '=' && *s != ' ') continue;
		v = s;
		while (*s && *s == ' ') s++;
		if (*s != '=') continue;
		*v = 0; s++;
		while (*s && *s == ' ') s++;
		v = s;
		while (*s && *s != ',') s++;
		if (!*s) end = TRUE;
		*s = 0;

		for (i = 0; i < kw_end; i++) if (!g_strcasecmp(k, kw[i])) { auto_set_param(i, v); break; }

		if (i >= kw_end) auto_show_help("Filter: set, error - variable #%u: unknown keyword '%s', ignored\n", n, k);
		if (end) break;
		s++;
	}

	return TRUE; // we always return true on this one
}


// end of filters



// We check the file against all the assigned filters
// Be fast, we check up to 5 files at a time in the other loop

gboolean auto_file_select(struct record *rc, struct auto_record *r, gchar *reason, gint D)
{
	gboolean show_help;
	gchar *filter, *string, *string_start;

	if (!*r->filters && !*r->strings) return TRUE; // no filters and strings entered

	filter = r->filters;
	strncpy(buf_list_strings, r->strings, sizeof(buf_list_strings));
	buf_list_strings[sizeof(buf_list_strings)-1] = '\0'; // strncpy doesn't terminate if at max
	string = buf_list_strings;

		if (!*filter) { // default is "not" if no filter entered
			if (!auto_filter_not(rc->name, string, FALSE, D)) return FALSE;
			else return TRUE;
			}

	while (1) { // move through each filter name
		string_start = string; // mark the start of our string
		while (*string) { // move to the next filter string
			if (strncmp(string, "///", 3) == 0) {
				*string = '\0';
				string += 3;
				break;
				}
			string++;
			}

		if (strncmp(string_start, "???", 3) == 0) { // show help strings?
			show_help = TRUE;
			string_start += 3; // skip past the ???
			}
		else show_help = FALSE;


		// here is where we pick which filter to use, we compare the filter name at each "if"
		// so we put the most commonly used filters first to save CPU time
		// the while loop is a cheapo trick used so we can "break" out and continue
		// with the other while loop
		while(1) {
			if (g_strncasecmp(filter, "not", 3) == 0) { // "not"
				strcpy(reason, "not");
				if (!auto_filter_not(rc->name, string_start, show_help, D)) return FALSE;
				break;
				}
			if (g_strncasecmp(filter, "size", 4) == 0) { // "size"
				strcpy(reason, "size");
				if (!auto_filter_size(rc->size, string_start, show_help, D)) return FALSE;
				break;
				}
			if (g_strncasecmp(filter, "regex", 5) == 0) { // "regex"
				strcpy(reason, "regex");
				if (!auto_filter_regex(rc->name, string_start, show_help, D)) return FALSE;
				break;
				}
			if (g_strncasecmp(filter, "or", 2) == 0) { // "or"
				strcpy(reason, "or");
				if (!auto_filter_or(rc->name, string_start, show_help, D)) return FALSE;
				break;
				}
			if (g_strncasecmp(filter, "and", 3) == 0) { // "and"
				strcpy(reason, "and");
				if (!auto_filter_and(rc->name, string_start, show_help, D)) return FALSE;
				break;
				}
			if (g_strncasecmp(filter, "extern", 6) == 0) { // "extern"
				strcpy(reason, "extern");
				if (!auto_filter_extern(rc, r, string_start, show_help, D)) return FALSE;
				break;
				}
			if (g_strncasecmp(filter, "set", 3) == 0) { // "set"
				strcpy(reason, "set");
				if (!auto_filter_set(string_start, show_help, D)) return FALSE;
				break;
				}
			break;
			}

		while (*filter && (*filter != ' ')) filter++; // move to next filter name
		if (!*filter || !*string) break;
		filter++; // move past the space char
		}

	return TRUE; // file passes, queue it
}



// All the automatic logic, called once every second if auto_enabled
// Here we simulate a very patient, non greedy human brain working all night.
// Try to be as nice as possible to the network at all times.
// If you modify this code you will have to spend the hours it takes to
// test it properly. Comment everything, do it the easy-reading way.
// We depend on the downloads module to filter dups already in the queue or
// download list and files we already have (exact match or larger)

// NOTE: This first version is very simple, more work and testing is needed.
//
// Pick up search term from top of list
// Send search out and wait for results
// Queue all the files that pass the "not" filter up to max
// Get next search term from list, do it all again
//

void auto_brain()
{
	gint D = auto_debug_mode; // debug printout mode

	struct auto_record *r;
	struct search *sch;
	struct results_set *rs;
	struct record *rc;
	struct gnutella_node *n;
	GSList *l = (GSList *) NULL;
	gint i;
	gchar tbuf[260], filter_reason[260];
	time_t tloc;
	gchar indicate_buf[50], indicate_char;
	gchar *indicate_str = "|/-\\"; // 4 characters

	auto_timer++;
	search_timer++;
	search_timeout++;

	// this section is just for display
	indicate_char = indicate_str[indicate_cnt++];
	if (indicate_cnt > 3) indicate_cnt = 0;
	if (auto_timer > 86400)
		g_snprintf(indicate_buf, sizeof(indicate_buf), "       %ud %uh %c",
				auto_timer / 86400, (auto_timer % 86400) / 3600, indicate_char);
	else if (auto_timer > 3600)
				g_snprintf(indicate_buf, sizeof(indicate_buf), "       %uh %um %c",
						auto_timer / 3600, (auto_timer % 3600) / 60, indicate_char);
	else if (auto_timer > 60)
				g_snprintf(indicate_buf, sizeof(indicate_buf), "        %um %us",
						auto_timer / 60, auto_timer % 60);
	else g_snprintf(indicate_buf, sizeof(indicate_buf), "          %us", auto_timer);
	gtk_label_set(GTK_LABEL(label_info_2), indicate_buf); // working indicator
	// end indicators

	strcpy(filter_reason, "none");
	if (on_the_net()) { // check if host connections are stable
		if (++auto_net_stable > 20) { // 20 seconds is a good number
			auto_net_stable = 30; // keep it within limits
			}
		}
	else {
		auto_net_stable = 0; // start count over
		return;
		}

	if (auto_net_stable < 20) {
		g_snprintf(tbuf, 50, "A: Waiting for stable node connections"); // limit to 50 chars
		gtk_label_set(GTK_LABEL(label_left), tbuf); // show user what's going on
		return; // go no further if net is not stable
		}

	if (auto_stopped) {
		if (GTK_CLIST(clist_download_queue)->rows <= auto_queued_start) { // almost out of files, start again
			auto_stopped = FALSE;
			if (D) printf("Auto: queued start\n");
			g_snprintf(tbuf, 50, "A: Need more files, start searching"); // limit to 50 chars
			gtk_label_set(GTK_LABEL(label_left), tbuf); // show user what's going on
			}
		else return;
		}
	else {
		if (GTK_CLIST(clist_download_queue)->rows >= auto_queued_stop) { // enough files queued for now
			while (searches) search_close_current(); // close all searches, next time use new one from list
			auto_stopped = TRUE;
			if (D) printf("Auto: queued stop\n");
			gtk_label_set(GTK_LABEL(label_info_3), "        (idle)");
			g_snprintf(tbuf, 50, "A: Enough files queued, stop searching"); // limit to 50 chars
			gtk_label_set(GTK_LABEL(label_left), tbuf); // show user what's going on
			return;
			}
		}


	if (!searches) { // start a new search if one is not going
		if (auto_picked_row >= GTK_CLIST(clist_auto)->rows) auto_picked_row = 0; // cycle back to top of list
		r = (struct auto_record *) gtk_clist_get_row_data(GTK_CLIST(clist_auto), auto_picked_row);
		if (r) {
			if (search_timer > auto_search_time) {

				if	(we_searched) { // below is for the node kick feature
					if (
						// at last row in list
						((auto_node_kick_mode == 1) && (auto_picked_row == 0)) ||
						// last search queued zero
						((auto_node_kick_mode == 2) && (queue_count == 0)) ||
						// after every search
						(auto_node_kick_mode == 3)
						){
						l = g_slist_last(sl_nodes); // kick first node on list
						if (l) {
							n = (struct gnutella_node *) l->data;
							if (n) {
								node_remove(n, "Automation Kicked", 0);
								if (D) printf("Auto: Kicked first node off\n");
								}
							}
						}
					}

				auto_work_row = auto_picked_row;
				auto_picked_row++; // set up for next time
				new_search(0, r->search); // enter a new search
				we_searched = TRUE; // we searched once already
				auto_search_row = 0; // start at the top of the search list
				search_timer = 0;
				search_timeout = 0;
				queue_count = 0;
				if (D) {
					time(&tloc); // do a time stamp
					strftime(tbuf,sizeof(tbuf),"%d %H:%M:%S",localtime(&tloc));
					printf("\nAuto: ----- %s SEARCH '%s'\n",tbuf,r->search);
					}
				gtk_label_set(GTK_LABEL(label_info_3), "     (searching)");
				g_snprintf(indicate_buf, sizeof(indicate_buf), "Searching");
				gtk_clist_set_text(GTK_CLIST(clist_auto), auto_work_row, 3, indicate_buf);
				g_snprintf(tbuf, 50, "A: Search using item #%d", auto_picked_row); // limit to 50 chars
				gtk_label_set(GTK_LABEL(label_left), tbuf); // show user what's going on
				}
			else {
				// this delay is needed so we don't hammer the network with search queries
				gtk_label_set(GTK_LABEL(label_info_3), "        (idle)");
				g_snprintf(tbuf, 50, "A: New search starts in %d secs", auto_search_time - search_timer); // limit to 50 chars
				gtk_label_set(GTK_LABEL(label_left), tbuf); // show user what's going on
				}
			}
		}

		// Here's where it all happens

	else {
		l = searches;
		sch = (struct search *) l->data;
		if (sch) { // we should always have a search record if "searches" is valid
			i = 0;

			while(1) {
					// get a search result from the list
				rc = (struct record *) gtk_clist_get_row_data(GTK_CLIST(sch->clist), auto_search_row);
				if (rc) { // got new results
					search_timeout = 0; // reset the timeout, we have a result
					rs = rc->results_set; // contains the filename and other data
						// get the "not" string and minsize from our auto list
					r = (struct auto_record *) gtk_clist_get_row_data(GTK_CLIST(clist_auto), auto_work_row);
					if (r) { // check r just in case, but it should be ok
						if (auto_file_select(rc, r, filter_reason, D)) { // should we queue this file?
							download_new(rc->name, rc->name, rc->size, rc->index, rs->ip, rs->port, rs->guid); // queue it
							if (D) printf("Auto: Queue '%s', %d\n",rc->name, rc->size);
							queue_count++;
							g_snprintf(indicate_buf, sizeof(indicate_buf), "   (queued %u)", queue_count);
							gtk_label_set(GTK_LABEL(label_info_3), indicate_buf); // status
							g_snprintf(indicate_buf, sizeof(indicate_buf), "Queued %u", queue_count);
							gtk_clist_set_text(GTK_CLIST(clist_auto), auto_work_row, 3, indicate_buf);
							}
						else if (D) printf("Auto: Not Queued '%s', %d by filter '%s'\n", rc->name, rc->size, filter_reason);
						if (++auto_search_row >= GTK_CLIST(sch->clist)->rows) break; // next row, till end
						if (++i >= 5) break; // only do up to 5 filenames this pass
						}
					}
				else { // no new results yet
					if (search_timeout > auto_search_timeout) { // no new results for quite some time now
						g_snprintf(indicate_buf, sizeof(indicate_buf), "Queued %u", queue_count);
						gtk_clist_set_text(GTK_CLIST(clist_auto), auto_work_row, 3, indicate_buf);
						while (searches) search_close_current(); // close all searches, next time use new one from list
						}
					break;
					}
				} // close the while loop
			}
		} // close else

}




// called after program starts
void auto_init()
{
	if (auto_on_at_start) auto_startup();
	gtk_widget_set_sensitive(button_auto_load_from_file, TRUE);

}


// startup the automatic feature, when they hit the button
void auto_startup()
{
	while (searches) search_close_current(); // close all searches

	auto_enabled = TRUE;
	auto_timer = 0;
	auto_net_stable = 0;
	auto_picked_row = 0;
	auto_work_row = 0;
	auto_search_row = 0;
	indicate_cnt = 0;
	search_timer = 999999;
	search_timeout = 0;
	queue_count = 0;
	auto_stopped = FALSE;
	we_searched = FALSE;

	auto_save_search_reissue_timeout = search_reissue_timeout;
	search_reissue_timeout = 0; // we control search reissue now
	gtk_widget_set_sensitive(button_auto_start, FALSE);
	gtk_widget_set_sensitive(button_auto_stop, TRUE);
	gtk_widget_set_sensitive(button_auto_load_from_file, FALSE);
	gtk_label_set(GTK_LABEL(label_info_1), "AUTOMATION ON");

		// disable entry, passive, filter & stop button on search screen
	gtk_widget_set_sensitive(button_search_passive, FALSE);
	gtk_widget_set_sensitive(button_search_filter, FALSE);
	gtk_widget_set_sensitive(button_search_stop, FALSE);
	gtk_widget_set_sensitive(button_search, FALSE);
	gtk_widget_set_sensitive(entry_search, FALSE);
}


// Stop the automatic mode, when they hit the button
void auto_shutdown()
{
	while (searches) search_close_current(); // close all searches
	auto_enabled = FALSE;
	search_reissue_timeout = auto_save_search_reissue_timeout;
	if (sl_auto_record) gtk_widget_set_sensitive(button_auto_start, TRUE);
	gtk_widget_set_sensitive(button_auto_stop, FALSE);
	gtk_widget_set_sensitive(button_auto_load_from_file, TRUE);
	gtk_label_set(GTK_LABEL(label_info_1), "");
	gtk_label_set(GTK_LABEL(label_info_2), "");
	gtk_label_set(GTK_LABEL(label_info_3), "");
	gtk_label_set(GTK_LABEL(label_left), "");

		// enter, passive & filter button on search screen
	gtk_widget_set_sensitive(button_search_passive, TRUE);
//	gtk_widget_set_sensitive(button_search_filter, TRUE); // disabled for now
	gtk_widget_set_sensitive (entry_search, TRUE);
}



// Load the config file from disk when they press the button
void auto_load_from_file()
{
	gchar cfg_tmp[4096];
	struct auto_record *r;

	gtk_clist_clear(GTK_CLIST(clist_auto)); // clear the list

	while (sl_auto_record) { // clear all data for the list
		r = (struct auto_record *) sl_auto_record->data;
		sl_auto_record = g_slist_remove(sl_auto_record, (gpointer) r);
		if (r->search) g_free(r->search);
		if (r->filters) g_free(r->filters);
		if (r->strings) g_free(r->strings);
		if (r) g_free(r);
		}

	if (cfg_use_local_file) auto_read_from_file("NAPS-auto.txt", TRUE);
	else {
		g_snprintf(cfg_tmp, sizeof(cfg_tmp), "%s/NAPS-auto.txt", config_dir);
		auto_read_from_file(cfg_tmp, TRUE);
		}

}



void auto_remove_entry(struct auto_record *r, gint row)
{
	gtk_clist_remove(GTK_CLIST(clist_auto), row);

	sl_auto_record = g_slist_remove(sl_auto_record, (gpointer) r);

	if (r->search) g_free(r->search);
	if (r->filters) g_free(r->filters);
	if (r->strings) g_free(r->strings);
	if (r) g_free(r);
}



// New entry

void auto_new_entry(gchar *search, gchar *strings, gchar *filters)
{
	struct auto_record *r;
	gchar *titles[4];
	gint row;

	r = (struct auto_record *) g_malloc0(sizeof(struct auto_record));

	r->status = 0;
	r->search = g_strdup(search);
	r->strings = g_strdup(strings);
	r->filters = g_strdup(filters);
	r->time = 0;
	r->info = NULL;
	r->done = FALSE;

	// add it to our list of records
	sl_auto_record = g_slist_prepend(sl_auto_record, (gpointer) r);

	titles[0] = r->search;
	titles[1] = r->strings;
	titles[2] = r->filters;
	titles[3] = NULL;

	row = gtk_clist_append(GTK_CLIST(clist_auto), titles);
	gtk_clist_set_row_data(GTK_CLIST(clist_auto), row, (gpointer) r);

	gtk_widget_set_sensitive(button_auto_start, TRUE);

}



// Do a full OR match of the filename against the "or" string

gboolean auto_or(char *s_filename, char *orstr)
{
  char *a;
  char *q2;
  char *s2;
  char *loc;

  // Copy notstr and filename so strtok doesn't trash them
  q2 = g_strdup(orstr);
  s2 = g_strdup(s_filename);
  g_strdown(q2); // make sure they are both lowercase
  g_strdown(s2);

	// break the notstr string into words ("tokens"). For example,
	// if the notstr is "john smith .jpg" the tokens will be "john",
	// "smith" and "jpg" tabs are handled as spaces

  a = strtok(q2, " \t"); // get first token
  if (a == 0) {
    g_free(q2);
    g_free(s2);
    return FALSE;
  }

	// Check each token to see if it's in the string

  while (a) {
    loc = strstr(s2, a); // is there a match?
    if (loc) { // found it
		g_free(q2);
		g_free(s2);
	 	return TRUE; // it's in there
		}
    a = strtok(0, " \t"); // go to next token
  }

	g_free(q2);
	g_free(s2);
	return FALSE; // if we got this far then it's not in there
}




// Do a full "and" against the filename

gboolean auto_and(char *s_filename, char *andstr)
{
  char *a;
  char *q2;
  char *s2;
  char *loc;
  int i;

  // Copy query and filename so strtok doesn't trash them
  q2 = g_strdup(andstr);
  s2 = g_strdup(s_filename);
  g_strdown(q2); // make sure they are both lowercase
  g_strdown(s2);

// break the query string into words ("tokens"). For example,
// if the query is "john smith .jpg" the tokens will be "john",
// "smith" and "jpg" tabs are handled as spaces

  a = strtok(q2, " \t"); // get first token
  if (a == 0) {
    g_free(q2);
    g_free(s2);
    return FALSE;
  }

// Check each token to see if it's in the string

  while (a) {
    loc = strstr(s2, a); // is there a match?
    if (loc == 0) {
		g_free(q2);
		g_free(s2);
	 	return FALSE; // all must match
		}

    // "Erase" the matched text, so that two occurrances of the
    // same keyword in query will require two occurrances of that
    // word in the string.
    for(i=0; a[i]; i++) {
		if (loc[i]) {
          loc[i] = ' ';
        }
      }
    a = strtok(0, " \t"); // go to next token
  }

	g_free(q2);
	g_free(s2);
	return TRUE; // if we got this far we have a match
}



// used to set the variables and load the list

void auto_set_param(guint32 keyword, gchar *value)
{
	gint32 i = atol(value);

	switch (keyword)
	{
		case kw_none_found: { if (i > 1 && i < 1048576) auto_none_found = i; return; }
		case kw_queued_stop: { if (i > 1 && i < 1048576) auto_queued_stop = i; return; }
		case kw_queued_start: { if (i > 1 && i < 1048576) auto_queued_start = i; return; }
		case kw_search_time: { if (i > 300 && i < 1048576) auto_search_time = i; return; }
		case kw_search_timeout: { if (i > 60 && i < 1048576) auto_search_timeout = i; return; }
		case kw_flush_timeout: { if (i > 1800 && i < 1048576) auto_flush_timeout = i; return; }
		case kw_list_filters: { strncpy(buf_list_filters, value, sizeof(buf_list_filters)); return; }
		case kw_list_strings: { strncpy(buf_list_strings, value, sizeof(buf_list_strings)); return; }
		case kw_on_at_start: { auto_on_at_start = i ? TRUE : FALSE; return; }
		case kw_log_enabled: { auto_log_enabled = i ? TRUE : FALSE; return; }
		case kw_node_kick_mode: { auto_node_kick_mode = i; return; }
		case kw_debug_mode: { auto_debug_mode = i; return; }

		// special case, enter all into the clist, including previous values
		case kw_list_search: {
			strncpy(buf_list_search, value, sizeof(buf_list_search)); // limit the size
			buf_list_filters[sizeof(buf_list_filters)-1] = '\0';
			buf_list_strings[sizeof(buf_list_strings)-1] = '\0'; // strncpy doesn't terminate if at max
			buf_list_search[sizeof(buf_list_search)-1] = '\0';
			auto_new_entry(buf_list_search, buf_list_strings, buf_list_filters);
			buf_list_filters[0] = '\0'; // clear these
			buf_list_strings[0] = '\0';
			return;
			}
	}
}


// write out the automation settings (done if there is no existing file)

void auto_write_to_file(gchar *path)
{

	FILE *f;

	f = fopen(path, "w");

	if (!f) {
		g_warning("Unable to open output file %s (%s)\n", path, g_strerror(errno));
		return;
		}

	fprintf(f, "\n# This is the NapShare automation default configuration file.\n");
	fprintf(f, "# You may edit it if you're careful. See the README file for more info.\n");
	fprintf(f, "# Put this file in the same place as your other NapShare configuration\n");
	fprintf(f, "# files.\n\n");

	fprintf(f, "# Wait xx minutes if no files are found, set via gui\n"
			"%s = %u\n\n", kw[kw_none_found], auto_none_found);
	fprintf(f, "# This many files in the download queue, kill the search and wait\n"
			"%s = %u\n\n", kw[kw_queued_stop], auto_queued_stop);
	fprintf(f, "# This many files in the download queue, start searching again\n"
			"%s = %u\n\n", kw[kw_queued_start], auto_queued_start);
	fprintf(f, "# How many seconds must pass before allowing another search (>300)\n"
			"%s = %u\n\n", kw[kw_search_time], auto_search_time);
	fprintf(f, "# Seconds to wait after last results come back before cancel search (>60)\n"
			"%s = %u\n\n", kw[kw_search_timeout], auto_search_timeout);
	fprintf(f, "# Max seconds file will sit in queue, get rid of old data (>1800)\n"
			"%s = %u\n\n", kw[kw_flush_timeout], auto_flush_timeout);
	fprintf(f, "# Start automation when program starts?\n"
			"%s = %u\n\n", kw[kw_on_at_start], auto_on_at_start);
	fprintf(f, "# Automation log file write enabled\n"
			"%s = %u\n\n", kw[kw_log_enabled], auto_log_enabled);
	fprintf(f, "# When to kick a node, 0=normal, 1=end of list, 2=previous search\n");
	fprintf(f, "# queued zero, 3=after every search (see README for more info)\n"
			"%s = %u\n\n", kw[kw_node_kick_mode], auto_node_kick_mode);
	fprintf(f, "# Automation debug printout mode, for programmers (level 0 to 5)\n"
			"%s = %u\n\n", kw[kw_debug_mode], auto_debug_mode);

	fprintf(f, "\n# The following values are entered into the automation list at each\n");
	fprintf(f, "# occurrence of the value for \"list_search\". Set \"list_filters\" and\n");
	fprintf(f, "# \"list_strings\" first. 4K limit on each string.\n\n");

	fprintf(f, "list_filters = \"or not size\"\n");
	fprintf(f, "list_strings = \"mpg mpeg avi divx asf wmf wmv///url htm txt zip mp3///3000\"\n");
	fprintf(f, "list_search = \"digital video\"\n\n");

	fprintf(f, "list_filters = \"or not size\"\n");
	fprintf(f, "list_strings = \"jpg jpeg///url htm txt zip mp3///5\"\n");
	fprintf(f, "list_search = \"pictures\"\n\n");

	fprintf(f, "list_filters = \"or not size\"\n");
	fprintf(f, "list_strings = \"mp3 wav///url htm txt zip mpg mpeg avi divx asf wmf wmv///300\"\n");
	fprintf(f, "list_search = \"sounds\"\n\n");

	fprintf(f, "list_filters = \"or not size\"\n");
	fprintf(f, "list_strings = \"jpg jpeg///url htm txt zip mp3///5\"\n");
	fprintf(f, "list_search = \"people in the park\"\n\n");

	fprintf(f, "list_filters = \"or not size\"\n");
	fprintf(f, "list_strings = \"jpg jpeg///url htm txt zip mp3///5\"\n");
	fprintf(f, "list_search = \"startup screen\"\n\n");

	fprintf(f, "list_strings = \"url htm txt zip mp3 mpg mpeg avi divx asf wmf wmv\"\n");
	fprintf(f, "list_search = \"THESE ARE EXAMPLES\"\n\n");

	fprintf(f, "list_strings = \"url htm txt zip mp3 mpg mpeg avi divx asf wmf wmv\"\n");
	fprintf(f, "list_search = \"SEE THE README FILE!\"\n\n");


	fprintf(f, "\n\n");

	fclose(f);
}


// read in automation settings from file

void auto_read_from_file(gchar *path, gboolean quiet)
{
	FILE *f;
	gchar *s, *k, *v;
	guint32 i, n = 0;

	static gchar *err = "Bad line %u in NAPS-auto.txt file, ignored\n";
	buf_list_filters[0] = '\0'; // in case someone forgets these
	buf_list_strings[0] = '\0';

	for (i = 0; i < 2; i++) { // if no file, write one then try reading it, only once
		f = fopen(path, "r");
		if (!f) { // no file? write the defaults
			auto_write_to_file(path);
			}
		}
	if (!f) { // still no file, throw a error
		if (!quiet) g_warning("Unable to open file %s (%s)\n", path, g_strerror(errno));
		return;
	}

	while (fgets(auto_tmp, sizeof(auto_tmp), f)) // fgets terminates string if at max size
	{
		n++;
		s = auto_tmp;
		while (*s && (*s == ' ' || *s == '\t')) s++;
		if (!((*s >= 'A' && *s <= 'Z') || (*s >= 'a' && *s <= 'z'))) continue;
		k = s;
		while (*s =='_' || (*s >= 'A' && *s <= 'Z') || (*s >= 'a' && *s <= 'z') || (*s >= '0' && *s <= '9')) s++;
		if (*s != '=' && *s != ' ' && *s != '\t') { fprintf(stderr, err, n); continue; }
		v = s;
		while (*s == ' ' || *s == '\t') s++;
		if (*s != '=') { fprintf(stderr, err, n); continue; }
		*v = 0; s++;
		while (*s == ' ' || *s == '\t') s++;
		if (*s == '"')
		{
			v = ++s;
			while (*s && *s != '\n' && *s != '"') s++;
			if (!*s || *s == '\n') { fprintf(stderr, err, n); continue; }
		}
		else { v = s; while (*s && *s != '\n' && *s != ' ' && *s != '\t') s++; }
		*s = 0;

		for (i = 0; i < kw_end; i++) if (!g_strcasecmp(k, kw[i])) { auto_set_param(i, v); break; }

		if (i >= kw_end) fprintf(stderr, "'NAPS-auto.txt' file, line %u: unknown keyword '%s', ignored\n", n, k);
	}

	fclose(f);

}




// Print the help output to STDOUT or log file, just like printf
void auto_show_help(const char *strings, ...)
{
	va_list ap;

	va_start (ap, strings);
//	if (auto_log_enabled) vfprintf (auto_log_fs, strings, ap);
/*	else */ vprintf (strings, ap);
	va_end (ap);
}


// subtract two GTimeVal's
gint timeval_subtract(struct _GTimeVal *result, struct _GTimeVal *x, struct _GTimeVal *y)
{
       /* Perform the carry for the later subtraction by updating y. */
       if (x->tv_usec < y->tv_usec) {
         int nsec = (y->tv_usec - x->tv_usec) / 1000000 + 1;
         y->tv_usec -= 1000000 * nsec;
         y->tv_sec += nsec;
       }
       if (x->tv_usec - y->tv_usec > 1000000) {
         int nsec = (x->tv_usec - y->tv_usec) / 1000000;
         y->tv_usec += 1000000 * nsec;
         y->tv_sec -= nsec;
       }

       /* Compute the time remaining to wait.
          tv_usec is certainly positive. */
       result->tv_sec = x->tv_sec - y->tv_sec;
       result->tv_usec = x->tv_usec - y->tv_usec;

       /* Return 1 if result is negative. */
       return x->tv_sec < y->tv_sec;
}

