/* $NetBSD$ */

/*-
 * Copyright (C) 2006 Jared D. McNeill <jmcneill@invisible.ca>.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Jared D. McNeill.
 * 4. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#include "porting.h"

__RCSID("$NetBSD$");

#include <sys/types.h>
#include <sys/stat.h>

#include <arpa/inet.h>

#include <prop/proplib.h>

#include <pthread.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>

#include "conf.h"

/* server configuration */
static mtftpd_conf_t mc;

static int
conf_add_file(prop_object_iterator_t iter)
{
	struct sockaddr_in sin;
	prop_dictionary_t pd;
	prop_object_t po;
	mtftpd_file_t *mf;
	int rv;

	po = prop_object_iterator_next(iter);
	if (po == NULL || prop_object_type(po) != PROP_TYPE_DICTIONARY)
		return 0;

	pd = (prop_dictionary_t)po;

	mf = malloc(sizeof(mtftpd_file_t));
	if (mf == NULL) {
		perror("malloc");
		return 0;
	}

	po = prop_dictionary_get(pd, "FileName");
	if (po != NULL && prop_object_type(po) == PROP_TYPE_STRING) {
		const char *str;

		str = prop_string_cstring_nocopy((prop_string_t)po);
		strlcpy(mf->mf_filename, str, MAXPATHLEN);
		strlcpy(mf->mf_filepath, mc.mc_tftproot, MAXPATHLEN);
		strlcat(mf->mf_filepath, "/", MAXPATHLEN);
		strlcat(mf->mf_filepath, str, MAXPATHLEN);
	} else {
		fprintf(stderr, "FileName not specified\n");
		exit(1);
	}

	po = prop_dictionary_get(pd, "InetAddress");
	if (po != NULL && prop_object_type(po) == PROP_TYPE_STRING) {
		const char *str;

		str = prop_string_cstring_nocopy((prop_string_t)po);
		inet_aton(str, &mf->mf_inetaddr);
	} else {
		fprintf(stderr, "InetAddress not specified for %s\n",
		    mf->mf_filename);
		exit(1);
	}

	po = prop_dictionary_get(pd, "Port");
	if (po != NULL && prop_object_type(po) == PROP_TYPE_NUMBER)
		mf->mf_port = (uint16_t)prop_number_integer_value(
		    (prop_number_t)po);
	else {
		fprintf(stderr, "Port not specified for %s\n",
		    mf->mf_filename);
		exit(1);
	}

	po = prop_dictionary_get(pd, "PacketDelay");
	if (po != NULL && prop_object_type(po) == PROP_TYPE_NUMBER)
		mf->mf_packetdelay = (uint32_t)prop_number_integer_value(
		    (prop_number_t)po);
	else
		mf->mf_packetdelay = 100000;

	po = prop_dictionary_get(pd, "Burst");
	if (po != NULL && prop_object_type(po) == PROP_TYPE_NUMBER)
		mf->mf_burst = (uint32_t)prop_number_integer_value(
		    (prop_number_t)po);
	else
		mf->mf_burst = 1;

	po = prop_dictionary_get(pd, "Retry");
	if (po != NULL && prop_object_type(po) == PROP_TYPE_NUMBER)
		mf->mf_retry = (uint32_t)prop_number_integer_value(
		    (prop_number_t)po);
	else
		mf->mf_retry = 5;

	/* initialize private members */
	pthread_mutex_init(&mf->mf_lock, NULL);
	mf->mf_sendcnt = mf->mf_running = 0;
	mf->mf_buf = malloc(mc.mc_blksize + 4);
	if (mf->mf_buf == NULL) {
		fprintf(stderr, "malloc(%d) failed: %s\n", mc.mc_blksize,
		    strerror(errno));
		exit(1);
	}
	mf->mf_sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (mf->mf_sock < 0) {
		fprintf(stderr, "socket(...) failed: %s\n", strerror(errno));
		exit(1);
	}
	memset(&sin, 0, sizeof(struct sockaddr_in));
	sin.sin_family = AF_INET;
	sin.sin_port = htons(mf->mf_port);
	sin.sin_addr.s_addr = INADDR_ANY;
	rv = bind(mf->mf_sock, (struct sockaddr *)&sin,
	    sizeof(struct sockaddr_in));
	if (rv == -1) {
		fprintf(stderr, "unable to bind to UDP port %d: %s\n",
		    mf->mf_port, strerror(errno));
		exit(1);
	}
	rv = setsockopt(mf->mf_sock, IPPROTO_IP, IP_MULTICAST_TTL,
	    &mc.mc_ttl, sizeof(mc.mc_ttl));
	if (rv == -1) {
		fprintf(stderr, "unable to set multicast TTL to %d: %s\n",
		    mc.mc_ttl, strerror(errno));
		exit(1);
	}

	syslog(LOG_INFO, "adding file %s at %s:%d (%d/%d retry %d)\n",
	    mf->mf_filename, inet_ntoa(mf->mf_inetaddr),
	    mf->mf_port, mf->mf_burst, mf->mf_packetdelay, mf->mf_retry);

	TAILQ_INSERT_TAIL(&mc.mc_mf, mf, mf_files);

	return 1;
}

void
conf_init(void)
{
	TAILQ_INIT(&mc.mc_mf);
}

mtftpd_conf_t *
conf_reload(const char *filename)
{
	mtftpd_conf_t *pmc;
	prop_dictionary_t p;
	struct stat st;
	int fd;
	char *xml;

	pmc = NULL;
	xml = NULL;

	fd = open(filename, O_RDONLY);
	if (fd < 0) {
		perror("open");
		goto cleanup;
	}

	if (fstat(fd, &st) != 0) {
		perror("fstat");
		goto cleanup;
	}

	xml = malloc(st.st_size);
	if (xml == NULL) {
		perror("malloc");
		goto cleanup;
	}

	read(fd, xml, st.st_size);

	p = prop_dictionary_internalize(xml);
	if (p != NULL) {
		prop_object_t po;

		pmc = &mc;

		po = prop_dictionary_get(p, "TFTPRoot");
		if (po != NULL && prop_object_type(po) == PROP_TYPE_STRING) {
			const char *str;

			str = prop_string_cstring_nocopy((prop_string_t)po);
			strlcpy(mc.mc_tftproot, str, MAXPATHLEN);
		} else {
			fprintf(stderr, "TFTPRoot not specified\n");
			exit(1);
		}

		po = prop_dictionary_get(p, "Port");
		if (po != NULL && prop_object_type(po) == PROP_TYPE_NUMBER)
			mc.mc_port = (uint16_t)prop_number_integer_value(
			    (prop_number_t)po);
		else
			mc.mc_port = 69;

		po = prop_dictionary_get(p, "TTL");
		if (po != NULL && prop_object_type(po) == PROP_TYPE_NUMBER)
			mc.mc_ttl = (uint8_t)prop_number_integer_value(
			    (prop_number_t)po);
		else
			mc.mc_ttl = 2;

		po = prop_dictionary_get(p, "BlockSize");
		if (po != NULL && prop_object_type(po) == PROP_TYPE_NUMBER)
			mc.mc_blksize = (uint16_t)prop_number_integer_value(
			    (prop_number_t)po);
		else
			mc.mc_blksize = 512;

		po = prop_dictionary_get(p, "FileList");
		if (po != NULL && prop_object_type(po) == PROP_TYPE_ARRAY) {
			prop_object_iterator_t iter;

			iter = prop_array_iterator((prop_array_t)po);
			while (conf_add_file(iter))
				;
		}
	}

cleanup:
	if (xml != NULL)
		free(xml);
	if (fd >= 0)
		close(fd);
	return pmc;
}
