/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/mman.h>

#ifdef __CYGWIN__
#include <windows.h>
HANDLE hFileMapping;
#endif

#include "mgl2.h"
#include "mglcol.h"
#include "event_man.h"
#include "em_comm.h"
#include "mglkey.h"
#include "keymap.h"
#include "config.h"

#define USE_KEYMAP

#undef NULL
#define NULL (void *)0

#ifndef __CYGWIN__
#define sigset(sig, fun) {\
	struct sigaction sa; \
	sa.sa_handler = fun; \
	sa.sa_flags = SA_RESTART; \
	memset(&(sa.sa_mask),0,sizeof(sa.sa_mask)); \
	sigaction(sig, &sa, NULL); \
}
#else
/* sa.sa_flags = SA_RESTART; \  XXX */
#define sigset(sig, fun) {\
	struct sigaction sa; \
	sa.sa_handler = fun; \
	memset(&(sa.sa_mask),0,sizeof(sa.sa_mask)); \
	sigaction(sig, &sa, NULL); \
}
#endif

#define COLOR_FOCUS	   packMC(0,15,15)

#define TASKBAR_IS_BOTTOM
#define TASKBAR_WIDTH	16

#define MK_MDOWN	MK_V1
#define MK_MMOVE	MK_V2
#define MK_MUP		MK_V3

#define MK_ICON_MAX	30
#define MK_ICON(fd)	((fd < MK_ICON_MAX)?(MK_V10 + fd):MK_V10)

#ifdef USE_KEYMAP
#define MK_FOCUS	MKE_SWITCH_FOCUS
#define MK_WINDOW	MKE_SWITCH_WINDOW
#else
#define MK_FOCUS	MK_F10
#define MK_WINDOW	MK_F9
#endif
//#define MK_DUMP_SCREEN	MK_F8

#define MK_POWER	(-1)		/* key of tb_attach */
#define MK_CLOCK	MK_V4		/* key of tb_attach */
#define MK_IM		('O'&0x1f)	/* im popup */

extern struct screen *current_screen;

#define mgl_color_model (physical_screen->type & ST_KINDMASK)
char cs_curdir[256];
char cs_workdir[256];
char cs_sock_name[32];
char cs_name[32];
int cs_sock_fd;
int cs_screen_is_fb = 0;
char *cs_screen_addr;
char cs_screen_name[32];
extern char mgl_screen_name[32];
extern void *mgl_screen_addr;
extern int mgl_screen_offset;
extern int mgl_screen_realwidth;
extern int mgl_screen_bpp;
extern int mgl_screen_realtype;
extern int mgl_screen_type;
extern struct screen *physical_screen;
extern int mgl_noclient;
extern char mgl_display[];

#ifdef USE_SENDIMAGE
const static int recv_image = 1;
#else
const static int recv_image = 0;
#endif

int disp_request = 1;

int first_connect = 1;

struct screen_v1 bg_main1_v1 = { 0, 1, 4, 2, "\xcc\xff"};
struct screen_v1 bg_main2_v1 = { 0, 1, 4, 2, "\xcc\x33\x33\x33\x33"};
struct screen_v1 bg_mini1_v1 = { 0, 1, 3, 3, "\xc0\x30\x0c"};
struct screen_v1 bg_mini2_v1 = { 0, 1, 4, 5, "\x00\xc0\xf0\xfc\xff"};

struct screen *bg_main1;
struct screen *bg_main2;
struct screen *bg_mini1;
struct screen *bg_mini2;

struct screen *power_image;
struct screen *clock_image;
struct screen *im_image;
long long last_update_clk;
long long last_update_pwr;
int mode_clk;

int get_apm(int *level);
int draw_power(struct screen *s);
int draw_time(struct screen *s);

#ifdef GP_DYNAMIC
#define SVR_SUPPORT_NINE
#define nine	virtual_key
#define create_nine(x,y,xs,ys,kind) create_gesture_pad(x,y,xs,ys)
#define nine_attach vk_attach
#define nine_detach vk_detach
int nine_attached;
struct screen *nine_image;
static struct nine *svr_nine;
static int old_modifier_mode;
int MK_NINE = MK_F8;
static void draw_gp_status(int mode) {
	static int old_mode;
	struct screen *s = NULL;
	extern char *icon_nine,*icon_nine_num;

	if ((mgl_modifier_status ^ old_mode) & MGL_SKS_NUM_LOCKED) {
	    old_mode = mgl_modifier_status;
	    if (mgl_modifier_status & MGL_SKS_NUM_LOCKED) {
		s = conv_screen_from_mgr(icon_nine_num,STK_NATIVE);
	    } else {
		s = conv_screen_from_mgr(icon_nine,STK_NATIVE);
	    }
	    if (s) {
	        bitblt(nine_image,0,0,s,0,0,16,16,0);
		free_screen(s);
	    }
	}
	push_screen(nine_image);
	if (mode) {
		set_color(COLOR_BLACK);
	} else {
		set_color(COLOR_LIGHTGRAY);
	}
	draw_line(0,4,4,0);
	draw_line(4,0,10,0);
	draw_line(10,0,14,4);
	draw_line(14,4,14,12);
	draw_line(14,12,10,15);
	draw_line(10,15,4,15);
	draw_line(4,15,0,12);
	draw_line(0,12,0,4);
	pop_screen();
	tb_show();
}
#else

#undef SVR_SUPPORT_NINE_TEST
/*
 mglsvr  㡼ˤϤ򥵥ݡȤ뤿 ե졼
ƥѡ( ƤӽФ¤ϷޤĤĤ뤬API ȤƤ ؿ̾ѹ
ʤͽ)

ʤߤ nine Ȥϡͭ󤬺äƤ 㡼ϥ⥸塼(β)
 
*/

#ifdef SVR_SUPPORT_NINE_TEST
#define SVR_SUPPORT_NINE

struct nine {
	struct virtual_key *vk;	/* global */
	int status;		/* global or local */
	int flags;		/* local */
	void *link;		/* local */
	void *opt;		/* local */
};

int nine_initialized;

int nine_init() {
	return 0;
}

int dmy_func(int type,int x,int y) {
printf("callback %d %d %d\n",type,x,y);
	return 0;
}

struct nine *create_nine(int x,int y,int xs,int ys,int kind) {
	struct nine *n;

	if (!nine_initialized) {
		if (nine_init() < 0) {
			return NULL;
		}
	}
	n = malloc(sizeof(struct nine));
	n->vk = create_virtual_key3(x,y,xs,ys,MK_V1,MK_V2,MK_V3);
	if (!n->vk) {
		free(n);
		return NULL;
	}
	n->vk->callback = dmy_func;
printf("create_nine %d %d %d %d\n",x,y,xs,ys);
	return n;
}

void nine_attach(struct virtual_key *parent,struct nine *n) {
	vk_attach(parent,n->vk);
printf("NINE attached\n");
}

void nine_detach(struct nine *n,int clean) {
	vk_detach(n->vk,clean);
printf("NINE detached\n");
	if (clean) {
		free(n);
	}
}

#endif

#ifdef SVR_SUPPORT_NINE
int nine_attached;
struct screen *nine_image;
static struct nine *svr_nine;
int MK_NINE = MK_F8;
#endif
#endif
char *icon_mini="\
#MGR000200160016
++++++++++++++++
+++WWWWWWWWW++++
++WXXXXXXXXXW+++
+WXXXXXXXXXXXW++
WXXXXXXXXXXXXXW+
WXXXXXXXXXXXXXW+
WXXXXXXXXXXXXXW+
WXXXXXXXXXXXXXW+
WXXXXXXXXXXXXXW+
WXXXXXXXXXXXXXW+
WXXXXXXXXXXXXXW+
WXXXXXXXXXXXXXW+
+WXXXXXXXXXXXW++
++WXXXXXXXXXW+++
+++WWWWWWWWW++++
++++++++++++++++
";

char *icon_main="\
#MGR000200160016
++++++++++++++++
WWWWWWWWWWWWWWW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WjjjjjjjjjjjjjW+
WWWWWWWWWWWWWWW+
++++++++++++++++
";

char *icon_ac="\
#MGR000200160016
++++++++++++++++
++++++++++++++++
++++++++++++++++
++++++++++++++++
+++++++@@@@@++++
++++++@@@@@@++++
+++++@@@@@@@@@@+
+++++@@@@@@@++++
@@@@@@@@@@@@++++
+++++@@@@@@@++++
+++++@@@@@@@@@@+
++++++@@@@@@++++
+++++++@@@@@++++
++++++++++++++++
++++++++++++++++
++++++++++++++++
";

char *icon_batt="\
#MGR000200160016
++++++++++++++++
++++++++++++++++
++++++++++++++++
++++++++++++++++
@@@@@@@@@@@@@@++
@GGGGGGGGGGGG@++
@GGGGGGGGGGGG@++
@GGGGGGGGGGGG@@@
@GGGGGGGGGGGG@@@
@GGGGGGGGGGGG@@@
@GGGGGGGGGGGG@++
@GGGGGGGGGGGG@++
@@@@@@@@@@@@@@++
++++++++++++++++
++++++++++++++++
++++++++++++++++
";

char *icon_im="\
#MGR000200160016
++++++++++++++++
+++++++@++++++++
++++++@r@+++++++
+++++@rrr@++++++
++++@rrrrr@+++++
+++@rrrrrrr@++++
++@rrrrrrrrr@+++
+@rrrrrrrrrrr@++
+@@@@@rrr@@@@@++
+++++@rrr@++++++
+++++@rrr@++++++
+++++@rrr@++++++
+++++@rrr@++++++
+++++@rrr@++++++
+++++@rrr@++++++
+++++@@@@@++++++
";

char *icon_nine="\
#MGR000200160016
++++++++++++++++
++++ggggggg+++++
+++gkkkkkkkg++++
++gkkkkkgkkkg+++
+gkkkkkggkkkkg++
+gkkkkggkkkkkg++
+gkkkggkkkkkkg++
+gkkggkkkkkkkg++
+gkgggggggggkg++
+gkkkkkkggkkkg++
+gkkkkgggkkkkg++
+gkkkggkkkkkkg++
++gkkgkkkkkkg+++
+++gkkkkkkkg++++
++++ggggggg+++++
++++++++++++++++
";

char *icon_nine_num="\
#MGR000200160016
++++++++++++++++
++++ggggggg+++++
+++gkkkkkkkg++++
++gkkkkkkkkkg+++
+gkkggkkkggkkg++
+gkkgggkkggkkg++
+gkkgggkkggkkg++
+gkkggkgkggkkg++
+gkkggkgkggkkg++
+gkkggkkgggkkg++
+gkkggkkgggkkg++
+gkkggkkkggkkg++
++gkkkkkkkkgg+++
+++gkkkkkkkg++++
++++ggggggg+++++
++++++++++++++++
";

#define SWS_INITIALIZING	1
#define SWS_FORGROUND		3
#define SWS_BACKGROUND		4
#define SWS_FORGROUNDING	5
#define SWS_BACKGROUNDING	6

struct screen_window {
	int type;
	int login;
	int stat;
	int where;
	int xs,ys;
	int fd;
	int key_mode;
	struct screen *icon;
	struct screen_window *link;
} *sw_list;

struct screen_window *sw_disp(int i);

#define MAX_SCREEN_MAP	10

#define SMS_NONE		0
#define SMS_FORGROUND		2
#define SMS_BACKGROUND		3
#define SMS_FORGROUNDING	4
#define SMS_BACKGROUNDING	5

struct screen_map {
	int type;
	int stat;
	int x,y,xs,ys;
	struct screen_window *sw;
} screen_map[MAX_SCREEN_MAP];

int screen_map_cnt;
struct screen_map *sm_cur;
struct screen_map *sm_taskbar;
struct screen_map *sm_mini1;
struct screen_map *sm_mini2;
struct screen_map *sm_main1;
struct screen_map *sm_main2;

int where_mini;
int where_main;
char *cmdline;

#define HAS_TASKBAR	1
#define HAS_MINIAPLI	2
#define HAS_DUALMAIN	4

#define MINI_XS		160
int mini_ys = 240 - 16;
#define MINI_YS		(mini_ys)

#undef link

extern int (*mgl_event_manager_initiator)();

#ifndef SVR_EVENT_MANAGER_INITIATOR
#define SVR_EVENT_MANAGER_INITIATOR emx11_init
#endif
extern int SVR_EVENT_MANAGER_INITIATOR();
#undef EVENT_MANAGER_INITIATOR

screen_map_print(int i) {
	struct screen_map *m = &screen_map[i];

	printf("%d %s %s %x\n",i
		,(m->type == AT_MAIN)?"AT_MAIN"
		:(m->type == AT_MINIAPLI)?"AT_MINIAPLI"
		:(m->type == AT_TASKBAR)?"AT_TASKBAR"
		:(m->type == AT_PALM)?"AT_PALM" :"unkown"

		,(m->stat == SMS_NONE)?"SMS_NONE"
		:(m->stat == SMS_FORGROUND)?"SMS_FORGROUND"
		:(m->stat == SMS_FORGROUNDING)?"SMS_FORGROUNDING"
		:(m->stat == SMS_BACKGROUND)?"SMS_BACKGROUND"
		:(m->stat == SMS_BACKGROUNDING)?"SMS_BACKGROUNDING":"unkown"
		,m->sw);
}

screen_map_init(int type) {
	int x,y,xs,ys;
	int mini_num = 0;
	int mini_pad = 0;
	int tys;
	int taskbar_width = TASKBAR_WIDTH;
	struct screen_map *sm;
	x = 0;
	y = 0;
	xs = SCREEN_WIDTH;
	ys = SCREEN_HEIGHT;

	if (type & HAS_MINIAPLI) {
		xs = MINI_XS;
		x = SCREEN_WIDTH - xs;
		tys = ys - TASKBAR_WIDTH;
		if (tys >= MINI_YS) {
			tys -= MINI_YS;
			mini_num ++;
		}
		if (tys >= MINI_YS) {
			tys -= MINI_YS;
			mini_num ++;
		}
		if (tys >= TASKBAR_WIDTH) {
			mini_pad = (tys - TASKBAR_WIDTH)/2;
			taskbar_width *= 2;
		} else if (tys >= 0) {
			mini_pad = tys/2;
		}
		if (type & HAS_TASKBAR) {
			sm = &screen_map[screen_map_cnt++];
			sm->type = AT_TASKBAR;
			sm->stat = SMS_NONE;
			sm->xs = xs;
			sm->ys = taskbar_width;
			sm->x = x;
#ifdef TASKBAR_IS_BOTTOM
			sm->y = y + ys - sm->ys;
#else
			sm->y = y;
			y += sm->ys;
#endif
			sm->sw = NULL;
			ys -= sm->ys;
			sm_taskbar = sm;
		}
		if (mini_num >= 2) {
			where_mini |= (1<<screen_map_cnt);
			sm = &screen_map[screen_map_cnt++];
			sm->type = AT_MINIAPLI;
			sm->stat = SMS_NONE;
			sm->xs = xs;
			sm->ys = MINI_YS;
			sm->x = x;
			sm->y = y + sm->ys + mini_pad;
			sm->sw = NULL;
			ys -= sm->ys;
			sm_mini2 = sm;
		}
		where_mini |= (1<<screen_map_cnt);
		sm = &screen_map[screen_map_cnt++];
		sm->type = AT_MINIAPLI;
		sm->stat = SMS_NONE;
		sm->xs = xs;
		sm->ys = MINI_YS;
		sm->x = x;
		sm->y = y;
		sm->sw = NULL;

		x = 0;
		y = 0;
		xs = SCREEN_WIDTH - 160;
		ys = SCREEN_HEIGHT;
		type &=  ~HAS_TASKBAR;
		sm_mini1 = sm;
	}
	if (type & HAS_TASKBAR) {
		sm = &screen_map[screen_map_cnt++];
		sm->type = AT_TASKBAR;
		sm->stat = SMS_NONE;
		sm->xs = xs;
		sm->ys = TASKBAR_WIDTH;
		sm->x = x;
#ifdef TASKBAR_IS_BOTTOM
		sm->y = y + ys - sm->ys;
#else
		sm->y = y;
		y += sm->ys;
#endif
		ys -= sm->ys;
		sm_taskbar = sm;
		sm->sw = NULL;
	}
	if ((type & HAS_DUALMAIN)) {
		where_main |= (1<<screen_map_cnt);
		sm = &screen_map[screen_map_cnt++];
		sm->type = AT_MAIN;
		sm->stat = SMS_NONE;
		sm->xs = xs;
		sm->ys = ys/2;
		sm->x = x;
		sm->y = y;
		sm->sw = NULL;
		ys -= sm->ys;
		y += sm->ys;
		sm_main2 = sm;
	}
	where_main |= (1<<screen_map_cnt);
	sm = &screen_map[screen_map_cnt++];
	sm->type = AT_MAIN;
	sm->stat = SMS_NONE;
	sm->xs = xs;
	sm->ys = ys;
	sm->x = x;
	sm->y = y;
	sm->sw = NULL;
	sm_main1 = sm;
	if (where_mini == 0) {
		/* miniapli can run in main (if main is PsPC size) */
		if ((xs >= 160) && (xs <= 240) && (ys >= 240 - TASKBAR_WIDTH)) {
			where_mini = where_main;
		}
	}
//printf("where_mini %x where_main %x\n",where_mini,where_main);
}

int screen_map_find(struct screen_window *sw) {
	int i;
	int pass;
	struct screen_map *m;

	for (i=0; i< screen_map_cnt; i++) {
		m = &screen_map[i];
		if (m->sw == sw) {
			return i;
		}
	}
	return -1;
}

int screen_map_size(int type) {
	int i;
	struct screen_map *m;

	for (i=0; i< screen_map_cnt; i++) {
		m = &screen_map[i];
		if (type & m->type) {
			return i;
		}
	}
	return -1;
}

screen_map_focus(int id) {
	int i,j;
	struct screen_map *m;
	if ((id >= 0) && (id < screen_map_cnt)) {
		sm_cur = &screen_map[id];
		if (sm_cur->sw) return;
	}
	for (i=0; i< screen_map_cnt; i++) {
		m = &screen_map[i];
		if (m == sm_cur) break;
	}
	for (j=0; j< screen_map_cnt; j++) {
		i = (i+1) % screen_map_cnt;
		m = &screen_map[i];
		if ( ((m->type == AT_MAIN) || (m->type == AT_MINIAPLI))
		    && m->sw) {
			sm_cur = m;
			return;
		}
	}
	sm_cur = NULL;
#if 0
	if (sm_cur == NULL) {
		sm_cur = sm_main1;
		return;
	}
	if (sm_cur == sm_main1) {
		if (sm_main2) {
			sm_cur = sm_main2;
		} else if (sm_mini1) {
			sm_cur = sm_mini1;
		}
	} else if (sm_cur == sm_main2) {
		if (sm_mini1) {
			sm_cur = sm_mini1;
		} else {
			sm_cur = sm_main1;
		}
	} else if (sm_cur == sm_mini1) {
		if (sm_mini2) {
			sm_cur = sm_mini2;
		} else {
			sm_cur = sm_main1;
		}
	} else if (sm_cur == sm_mini2) {
		sm_cur = sm_main1;
	}
#endif
}

int screen_map_disp() {
	int i;
	struct screen_map *m;
	struct screen_window *sw;

	for (i=0; i< screen_map_cnt; i++) {
		m = &screen_map[i];
		if (m->stat != SMS_NONE) 
			continue;
		if (m->type == AT_TASKBAR)
			continue;
		sw = sw_disp(i);
		if (sw) {
//printf("find to disp \n");
		    /* forgrounding */
//printf("---");
//screen_map_print(i);
		    m->sw = sw;
		    screen_map_forground(m);
//printf("===");
//screen_map_print(i);
		} else if (m == sm_main1) {
		     bitblt(NULL,m->x,m->y,bg_main1,0,0,m->xs,m->ys,BLT_TILING);
		} else if (m == sm_main2) {
		     bitblt(NULL,m->x,m->y,bg_main2,0,0,m->xs,m->ys,BLT_TILING);
		} else if (m == sm_mini1) {
		     bitblt(NULL,m->x,m->y,bg_mini1,0,0,m->xs,m->ys,BLT_TILING);
		} else if (m == sm_mini2) {
		     bitblt(NULL,m->x,m->y,bg_mini2,0,0,m->xs,m->ys,BLT_TILING);
		}
	}
}

sw_add_top(struct screen_window *sw) {
	sw->link = sw_list;
	sw_list = sw;
}

sw_add(struct screen_window *sw) {
	struct screen_window *p;

	sw->link = NULL;
	if (sw_list == NULL) {
		sw_list = sw;
	} else {
		p = sw_list;
		while (p->link) {
			p = p->link;
		}
		p->link = sw;
	}
}

sw_remove(struct screen_window *sw) {
	struct screen_window *p;

	if (!sw_list) return;

	if (sw_list == sw) {
		sw_list = sw->link;
	} else {
		p = sw_list;
		while (p->link) {
			if (p->link == sw) {
				p->link = sw->link;
				break;
			}
			p = p->link;
		}
	}
	sw->link = NULL;
}

sw_free(struct screen_window *sw) {
	int i;

	if (sw->login) exit(0);
	i = screen_map_find(sw);
	if (i >= 0) {
		screen_map[i].sw = NULL;
		screen_map[i].stat = SMS_NONE;
		disp_request++;
	}
	sw_remove(sw);
	close(sw->fd);
	if (sw->icon) {
		tb_detach(sw->icon);
		free_screen(sw->icon);
	}
	free(sw);
	tb_disp();
}

sw_alloc(int fd) {
	struct screen_window *sw;
	sw = malloc(sizeof(*sw));
	sw->stat = SWS_INITIALIZING;
	sw->where = 0;
	sw->fd = fd;
	sw->link = NULL;
	sw->icon = NULL;
	sw->type = 0;
	sw->login = first_connect;
	sw->key_mode = 0;
	first_connect = 0;
	sw_add(sw);
}

sw_set_icon(struct screen_window *sw,char *icon) {
	char num[32];
	struct screen *old = NULL;
	if (sw->icon) {
		old = sw->icon;
		sw->icon = NULL;
	}
	if (icon == NULL) {
		if (sw->type == AT_MAIN) {
			sw->icon = conv_screen_from_mgr(icon_main,STK_NATIVE);
		} else {
			sw->icon = conv_screen_from_mgr(icon_mini,STK_NATIVE);
		}
		set_color(COLOR_WHITE);
		set_font(12,FA_BOLD);
		push_screen(sw->icon);
		sprintf(num,"%2d",sw->fd);
		if ((0 <= sw->fd) && (sw->fd < 10)) {
			draw_font(4,2,num[1],DIR_NORTH);
		} else if (sw->fd < 100) {
			draw_font(0,2,num[0],DIR_NORTH);
			draw_font(6,2,num[1],DIR_NORTH);
		}
		pop_screen();
	} else {
		sw->icon = conv_screen_from_mgr(icon,STK_NATIVE);
		if (sw->icon == NULL) {
			sw->icon = old;
			return;
		}
	}
	if (old) {
		tb_change(sw->icon,old);
		free_screen(old);
	} else {
		tb_attach(sw->icon,MK_ICON(sw->fd),0);
	}
	tb_show();
}

tb_forground(int fd) {
	int i;
	struct screen_window *sw;

//printf("tb_forground %d\n",fd);
	sw = sw_list;
	while (sw) {
		if (sw->fd == fd) {
//printf("---- find \n");
			if (sw->stat == SWS_FORGROUND) {
				i = screen_map_find(sw);
				set_focus(i);
			} else {
				sw_forground(sw,1);
			}
			break;
		}
		sw = sw->link;
	}
}

tb_disp() {
	if (sm_cur && sm_cur->sw) {
		tb_set_focus(sm_cur->sw->icon, sm_cur->sw->type == AT_MAIN);
	}
	tb_show();
}

int sw_send(struct screen_window *sw,struct emc_packet *es) {
	int r,i;

#ifdef DEBUG_COMM
	emc_packet_print("Send",es);
#endif

	r = send(sw->fd,es,sizeof(*es),0);
	if (r != sizeof(*es)) {
		perror(kind2name(es->kind));
		sw_free(sw);
	}
	return r;
}

int sw_recv(struct screen_window *sw,struct emc_packet *es,int n) {
#if 0 /* old */
	int r,i;
	r = recv(sw->fd,es,sizeof(*es)*n,0);
printf("sizeof(es) = %d\n",sizeof(*es));
	if (r == 0) {
if (r == 0) printf("read 0 length\n");
		perror("read packet");
		sw_free(sw);
		return r;
	}
#else
	int r,ii,sz,l;
	char *buf;

	buf = (char *)es;
	sz = sizeof(*es)*n;
	ii = 0;
	while (ii < sz) {
		l = sz - ii;
		r = recv(sw->fd,buf + ii,l,0);
		if ((r < 0) && (errno = EAGAIN))
				continue;
		if (r <= 0) {
			perror("read packet");
			sw_free(sw);
			return;
		}
		ii += r;
	}
	r = sz;
#endif
#ifdef DEBUG_COMM
	emc_packet_print("Recv",es);
#endif
	return r;
}

struct screen_window *sw_disp(int i) {
	struct screen_window *sw;

	sw = sw_list;
	while (sw) {
		if ((sw->stat == SWS_BACKGROUND) && (sw->where & (1<<i))) {
//printf("disp to %x \n",(1<<i));
			sw_remove(sw);
			sw_add(sw);
			break;
		}
		sw = sw->link;
	}
	return sw;
}

screen_map_forground(struct screen_map *m) {
	int i,r;
	struct emc_packet e;

	if (!m->sw || (m->sw->stat != SWS_BACKGROUND) )
		return;
	m->stat = SMS_FORGROUNDING;
	m->sw->stat = SWS_FORGROUNDING;
	e.kind = REQ_DISPMODE;
	e.args[0] = REQ_SHOW;
	e.args[1] = m->x;
	e.args[2] = m->y;
#ifdef USE_SENDIMAGE
	if (recv_image) {
	   e.args[1] = 0;
	   e.args[2] = 0;
	}
#endif
	sw_send(m->sw,&e);
}

screen_map_background(struct screen_map *m) {
	int i,r;
	struct emc_packet e;

	if ((sm_cur == m) && sm_cur->sw) {
		e.kind = EVENT_FOCUS;
		e.args[0] = FOCUS_OFF;
		sw_send(sm_cur->sw,&e);
		tb_disp();
	}
	if (!m->sw || (m->stat != SMS_FORGROUND))
		return;
	m->stat = SMS_BACKGROUNDING;
	m->sw->stat = SWS_BACKGROUNDING;
	e.kind = REQ_DISPMODE;
	e.args[0] = REQ_HIDE;
	e.args[1] = m->x;
	e.args[2] = m->y;
	sw_send(m->sw,&e);
}

sw_forground(struct screen_window *sw,int force) {
	int i;
	struct screen_map *m;
	int can_disp=0;

	if (sw->stat != SWS_BACKGROUND) return;

	sw_remove(sw);
	sw_add_top(sw);

	/* pass 1 : finding NULL */
	for (i=0; i < screen_map_cnt; i++) {
//screen_map_print(i);
		m = &screen_map[i];
		if (sw->where & (1<<i)) {
			can_disp++;
			if (m->sw == NULL) {
				disp_request++;
				return;
			}
		}
	}
	/* pass 2 : finding ! cureent */
	if (can_disp >= 1) for (i=0; i < screen_map_cnt; i++) {
		m = &screen_map[i];
		if ( (m != sm_cur) && (sw->where & (1<<i)) ) {
			screen_map_background(m);
			return;
		}
	}
	if (!force) return;

	/* pass 3 : anyway */
	for (i=0; i < screen_map_cnt; i++) {
		m = &screen_map[i];
		if (sw->where & (1<<i)) {
			screen_map_background(m);
			return;
		}
	}
}

execute_client_req(struct screen_window *sw) {
	struct emc_packet er[10],es;
	int x,y,xs,ys;
	int i,r,e,n,read_len;

	read_len = sw_recv(sw,er,1);
	if (read_len <= 0) {
		return;
	}
	n = read_len / sizeof(struct emc_packet);
	for (e=0; e<n; e++) {
	    switch(er[e].kind) {
	    case REQ_CONNECT:
		if (sw->stat != SWS_INITIALIZING) {
			sw_free(sw);
			return;
		}
		for (i=0; i<128; i++) {
			es.kind = PUT_KEYMAP;
			es.args[0] = i;
			es.args[1] = mgl_keymap[i][0];
			es.args[2] = mgl_keymap[i][1];
			es.args[3] = mgl_keymap[i][2];
			if ((es.args[1] >= 0) || (es.args[2] >= 0)) {
				r = sw_send(sw,&es);
				if (r < 0) {
					return;
				}
			}
		}
		i = screen_map_size(er[e].args[0]);
		sw->type = er[e].args[0];
		if (i>=0) {
			if (screen_map[i].type == AT_MINIAPLI) {
				sw->where = where_mini;
//printf("attach to mini %x\n",sw->where);
			} else {
//printf("attach to main %x\n",sw->where);
				sw->where = where_main;
			}
			sw->xs = screen_map[i].xs;
			sw->ys = screen_map[i].ys;
			es.kind = RPLY_CONNECT;
			es.args[0] = screen_map[i].xs;
			es.args[1] = screen_map[i].ys;
			es.args[2] = mgl_screen_realwidth;
			es.args[3] = SCREEN_HEIGHT;
#ifdef USE_SENDIMAGE
			if (recv_image) {
			    es.args[2] = es.args[0];
			    es.args[3] = es.args[1];
			}
#endif
			es.args[4] =  (mgl_screen_bpp << 16)
				    | (mgl_screen_type << 8)
				    | (mgl_screen_realtype) 
				    | (recv_image << 24);
			es.args[5] = mgl_screen_offset;
			if (!recv_image)
			   strncpy((char *)(&es.args[6]),mgl_screen_name,16);
		} else {
			sw->where = 0;
			sw->xs = -1;
			sw->ys = -1;
			es.kind = RPLY_CONNECT;
			es.args[0] = -1;
			es.args[1] = -1;
			es.args[2] = mgl_screen_realwidth;
			es.args[3] = SCREEN_HEIGHT;
		}
		r = sw_send(sw,&es);
		if (r < 0) {
			return;
		}
		break;
	    case ACCEPT_SIZE:
		if ((er[e].args[0] == 0) || (sw->stat != SWS_INITIALIZING)) {
//printf("not accept\n");
			if (sw->login) {
				first_connect = 1;
			}
			sw_free(sw);
			return;
		}
		sw->stat = SWS_BACKGROUND;

		sw_set_icon(sw,NULL);
		sw_forground(sw,0);
		break;
	    case REQ_REFRESH:
		i = screen_map_find(sw);
		if ((i>=0) && (screen_map[i].stat == SMS_FORGROUND)) {
			em_rec_point(current_screen
				,er[e].args[0] + screen_map[i].x
				,er[e].args[1] + screen_map[i].y);
			em_rec_point(current_screen
				,er[e].args[2] + screen_map[i].x
				,er[e].args[3] + screen_map[i].y);
//printf("refresh %d %d - %d %d\n",r_minx,r_miny,r_maxx,r_maxy);
			//refresh();
#ifdef USE_SENDIMAGE
			if (recv_image) {
				struct screen *s;
				int ii,r,l;
				int x = er[e].args[0] + screen_map[i].x;
				int y = er[e].args[1] + screen_map[i].y;
				int xs = er[e].args[2] - er[e].args[0] +1;
				int ys = er[e].args[3] - er[e].args[1] +1;
				int sz = xs * ys * mgl_screen_bpp / 8;
				int szp; /* pre-read size */
				char buf[sz];

				szp = read_len-(e+1)* sizeof(struct emc_packet);
				memcpy(buf,&er[e+1],szp);
				e = n;
				ii = szp;
				while (ii < sz) {
					l = 8192;
					if ((sz - ii) < l) l = sz - ii;
			    		r = recv(sw->fd,buf + ii,l,0);
					if ((r < 0) && (errno = EAGAIN))
						continue;
					if (r <= 0) {
						sw_free(sw);
						return;
					}
					ii += r;
				}
//printf("recv1 %d %d %d n = %d\n",sz,xs,ys,ii);
				s = create_memscreen(xs,ys,buf,STK_NATIVE,0);
				if (s) {
				    bitblt(current_screen
					,x,y ,s,0,0 ,xs,ys,0);
				    free_screen(s);
				}
			}
#endif
		}
#ifdef USE_SENDIMAGE
		else {
			if (recv_image) {
				int xs = er[e].args[2] - er[e].args[0] +1;
				int ys = er[e].args[3] - er[e].args[1] +1;
				int sz = xs * ys * mgl_screen_bpp / 8;
				int szp; /* pre-read size */
				char buf[sz];
				int ii,r,l;

				szp = read_len-(e+1)* sizeof(struct emc_packet);
				memcpy(buf,&er[e+1],szp);
				e = n;

//printf("recv2 %d %d %d\n",sz,xs,ys);
				ii = szp;
				while (ii < sz) {
					l = 8192;
					if ((sz - ii) < l) l = sz - ii;
			    		r = recv(sw->fd,buf+ii,l,0);
					if ((r < 0) && (errno = EAGAIN))
						continue;
					if (r <= 0) {
						sw_free(sw);
						return;
					}
					ii += r;
				}
			}
		}
#endif
		break;
	    case RPLY_DISPMODE:
		if (sw->stat == SWS_FORGROUNDING) {
			sw->stat = SWS_FORGROUND;
			i = screen_map_find(sw);
			if (i >= 0) {
				screen_map[i].stat = SMS_FORGROUND;
			}
			if (!sm_cur) sm_cur = &screen_map[i];
			if (sm_cur == &screen_map[i]) {
				es.kind = EVENT_FOCUS;
				es.args[0] = FOCUS_ON;
				sw_send(sm_cur->sw,&es);
				mgl_set_keymode(sm_cur->sw->key_mode);
				tb_disp();
			}
		} else if (sw->stat == SWS_BACKGROUNDING) {
			sw->stat = SWS_BACKGROUND;
			i = screen_map_find(sw);
			if (i >= 0) {
				screen_map[i].stat = SMS_NONE;
				screen_map[i].sw = NULL;
				disp_request++;
			}
		} else {
			fprintf(stderr,"err %x %d\n",er[e].kind,er[e].args[0]);
		}
		tb_disp();
		break;
	    case SET_ICON:
		{
			int r,len,rr,mod;
			struct screen *s;
			char buf[1024];

			len = er[e].args[0];
			if (len > 1024) {
				sw_free(sw);
				break;
			}
			rr = read_len - (e+1)* sizeof(struct emc_packet);
//printf("set_icon len = %d rr = %d\n",len,rr);
			/* adjust event buffer */
			if (rr <= len) {
			    memmove(buf,&er[e+1],rr);
			    r = recv(sw->fd,buf+rr,len-rr,0);
			    if (r != len-rr) {
				sw_free(sw);
				break;
			    }
			    n = e+1;
			} else {
			    memmove(buf,&er[e+1],len);
			    memmove(&er[e+1],(void *)(&er[e+1])+len,rr-len);
			    mod = (rr-len) % sizeof(struct emc_packet);
			    if (mod) {
				mod = sizeof(struct emc_packet) - mod;
			    	r = recv(sw->fd,(void *)(&er[e+1])+(rr-len),mod,0);
			        if (r != mod) {
				    sw_free(sw);
				    break;
			        }
			    }
			    read_len = read_len - len + mod;
			    n = read_len / sizeof(struct emc_packet);
			}
			sw_set_icon(sw,buf);
		}
		break;
	    case EXIT_NOTIFY:
		sw_free(sw);
		break;
	    case SET_KEY_MODE:
		sw->key_mode = er[e].args[0];

		/* λ sm_cur  NULL ΤȤ롣Х? */
		if(sm_cur && (sm_cur->sw == sw)) {
			mgl_set_keymode(sw->key_mode);
		}
		break;
	    default:
		fprintf(stderr,"unkown event %x %d\n",er[e].kind,er[e].args[0]);
	    }
	}
}

execute_send_event() {
	extern int key_buf_cnt;
	int c,cc,x,y;
	struct emc_packet e;
	struct screen_window *sw,*z;

	while (key_buf_cnt) {
		c = get_key(-1);
		cc = c & ~MGL_SKM_MASK;
#ifdef USE_KEYMAP
		if ((mgl_key_mode & MGL_SK_KM_MASK) != MGL_SK_RAW) {
			if (cc == MK_F9) cc = MK_WINDOW;
			if (cc == MK_F10) cc = MK_FOCUS;
		}
#else
		if ((mgl_key_mode & MGL_SK_KM_MASK) == MGL_SK_RAW) {
//printf("get_key %3d %s\n",c & 0x7f,(c & 0x80)?"Rel":"Prs");
			cc = -1;
			switch (c) {
			case 68:	cc = MK_F10; break;
			case 67:	cc = MK_F9; break;
			case 66:	cc = MK_F8; break;
			}
		} else {
			cc = c & ~MGL_SKM_MASK;
		}
#endif
		x = vk_x;
		y = vk_y;

		if (c == MK_CLOCK) {
			last_update_clk = 0LL;
			mode_clk = !mode_clk;
			draw_clock(clock_image);
			tb_show();
			continue;
		}
		if (sm_cur == NULL)
			continue;
#ifdef MK_DUMP_SCREEN
		if (cc == MK_DUMP_SCREEN) {
			struct screen *s;
#if 0
			s = create_subscreen(NULL,sm_cur->x,sm_cur->y
				,sm_cur->xs,sm_cur->ys);
#else
			s = current_screen;
#endif
			if (!s) continue;
			write_screen_xpm("screen",s);
			write_screen_mgr("screen",s,mgl_color_model);
#if 0
			free_screen(s);
#endif
			continue;
		} else
#endif
#ifdef SVR_SUPPORT_NINE
		if (cc == MK_NINE) {
			if (nine_attached) {
				draw_gp_status(0);
				nine_detach(svr_nine,0);
			} else {
				draw_gp_status(1);
				nine_attach(NULL,svr_nine);
			}
			nine_attached = !nine_attached;
		} else
#endif
		if (cc == MK_FOCUS) {
printf("recv MK_FOCUS %x\n",c);
			set_focus(-1);
			return;
		} else
		if (cc == MK_WINDOW) {
printf("recv MK_WINDOW %x\n",c);
			screen_map_background(sm_cur);
			return;
		}
		if ((c == MK_MDOWN) && sm_taskbar
			&& (x >= sm_taskbar->x)&&(y >= sm_taskbar->y)
			&& (x < sm_taskbar->x + sm_taskbar->xs)
			&& (y < sm_taskbar->y + sm_taskbar->ys) ) {
//printf(" MDOWN %d %d\n",x,y);

			tb_mouse(x,y);
			return;
		}
#ifdef SVR_SUPPORT_NINE
		if ((mgl_modifier_status ^ old_modifier_mode) & MGL_SKS_NUM_LOCKED) {
				old_modifier_mode = mgl_modifier_status;
				draw_gp_status(nine_attached);
		}
#endif
		if (sm_cur->sw == NULL)
			continue;

		sw = sm_cur->sw;

		switch(c) {
		case MK_MDOWN:
			e.kind = EVENT_MPRESS;
			e.args[0] = x - sm_cur->x;
			e.args[1] = y - sm_cur->y;
			e.args[2] = mgl_button_shift;
			break;
		case MK_MUP:
			e.kind = EVENT_MRELEASE;
			e.args[0] = x - sm_cur->x;
			e.args[1] = y - sm_cur->y;
			break;
		case MK_MMOVE:
			e.kind = EVENT_MMOVE;
			e.args[0] = x - sm_cur->x;
			e.args[1] = y - sm_cur->y;
			break;
		default:
//printf("send key %08x\n",c);
			e.kind = EVENT_KEY;
			e.args[0] = 1;
			e.args[1] = c;
			break;
		}
		if (e.kind != EVENT_KEY) {
		    if ((x >= sm_cur->x) && ( e.args[0] < sm_cur->xs)
			  && (y >= sm_cur->y) && ( e.args[1] < sm_cur->ys) ) {
			sw_send(sw,&e);
		    }
		} else {
		    if ((MK_V10 <= c) && (c < MK_V10 + MK_ICON_MAX)) {
		    	tb_forground(c - MK_V10);
		    } else {
			sw_send(sw,&e);
		    }
		}
	}
}

set_focus(int i) {
	struct emc_packet e;
	int change = 0;
	if (sm_cur->sw) {
		e.kind = EVENT_FOCUS;
		e.args[0] = FOCUS_OFF;
		sw_send(sm_cur->sw,&e);
		change = 1;
	}
	screen_map_focus(i);
	if (sm_cur->sw) {
		e.kind = EVENT_FOCUS;
		e.args[0] = FOCUS_ON;
		sw_send(sm_cur->sw,&e);
		change = 1;
		mgl_set_keymode(sm_cur->sw->key_mode);
	}
	if (change)
		tb_disp();
}

#define  AC_LINE 0
#define  BATTERY 1

int draw_clock(struct screen *s) {
	int w,h;
	long long t;
	time_t tt;
	struct tm *tm;
	char buf1[10],buf2[10];

	t = millitime();
	if (!((last_update_clk == 0) || (t - last_update_clk > 10000LL))) {
		return 0;
	}
	last_update_clk = t;
	tt = time(0);
	tm = localtime(&tt);
	sprintf(buf1,"%2d:%02d",tm->tm_hour,tm->tm_min);
	sprintf(buf2,"%2d/%-2d",tm->tm_mon+1,tm->tm_mday);
	push_screen(s);
	set_font(12,0);
	set_color(COLOR_LIGHTGRAY);
	clear_screen();
	set_color(COLOR_WHITE);
	fill_rect(1,1,s->width-2,s->height-3);
	set_color(COLOR_BLACK);

	if (s->width >= 64) {
		draw_string(0,2,buf2,DIR_NORTH);
		draw_string(0+32,2,buf1,DIR_NORTH);
	} else if (s->width >= 32) {
		if (mode_clk) draw_string(0,2,buf2,DIR_NORTH);
		else          draw_string(0,2,buf1,DIR_NORTH);
	}
	pop_screen();
	return 1;
}

int draw_power(struct screen *s) {
	int lev;
	struct screen *new;
	int w,h;
	long long t;

	t = millitime();
	if (!((last_update_pwr == 0) || (t - last_update_pwr > 10000LL))) {
		return 0;
	}
	last_update_pwr = t;
	if (get_apm(&lev)==AC_LINE) {
		new = conv_screen_from_mgr(icon_ac,STK_NATIVE);
		bitblt(s,0,0,new,0,0,16,16,0);
		free_screen(new);
	} else {
		new = conv_screen_from_mgr(icon_batt,STK_NATIVE);
		bitblt(s,0,0,new,0,0,16,16,0);
		free_screen(new);
		if (lev >= 100) w = 12;
		else w = lev * 12 / 100;
		h = 7;
		push_screen(s);
		set_color(COLOR_BLACK);
		fill_rect(1+w,5,12-w,h);
		pop_screen();
	}
	return 1;
}

#if defined (__FreeBSD__)
#include <sys/ioctl.h>
#include <machine/apm_bios.h>

#define APMDEV  "/dev/apm"

int get_apm(int *level) {
    int fd;
    struct apm_info info;

    /* Init */
    *level = 0;

    /* device open */
    fd = open(APMDEV, O_RDONLY); 
    if(fd == -1){
        return(AC_LINE);
    }

    /* get info */
    if(ioctl(fd, APMIO_GETINFO, &info) == -1){
        close(fd);
        return(AC_LINE);
    }
    close(fd);

    /* power line */
    if(info.ai_acline == 1){
        return(AC_LINE);
    }

    /* battery level */
    if(info.ai_batt_stat != 255){
        if(info.ai_batt_stat <= 3){
            switch(info.ai_batt_stat){
            case 0: /* high */
                *level = 100;
                break;
            case 1: /* low */
                *level = 30;
                break;
            case 2: /* critical */
                *level = 10;
                break;
            }
        }
    }

    if(info.ai_batt_life != 255){
        if(info.ai_batt_life <= 100){
            *level = info.ai_batt_life;
            if(*level < 0){
                *level = 0;
            }
        }
    }

    if(*level == 0){
        return(AC_LINE);
    }else{
        return(BATTERY);
    }
}
#elif defined (__linux__)
/* from xbatt-1.2beta3 */
#define APMDEV  "/proc/apm"

int get_apm(int *level) {
    FILE *fp;
    char buffer[64];
    int  battLife;
    char driver_version[64];
    int  apm_bios_info_major;
    int  apm_bios_info_minor;
    int  apm_bios_info_flags;
    int  ac_line_status;
    int  battery_status;
    int  battery_flag;
    int  time_units;
    char units[64];


    /* Init */
    *level = 0;

    /* device open */
    fp = fopen(APMDEV, "r"); 
    if(!fp){
        return(AC_LINE);
    }

    while(fgets(buffer, 64, fp) != NULL){
        /*
         * for linux-1.3.58 or later
         */
        if(sscanf(buffer, "%s %d.%d 0x%x 0x%x 0x%x 0x%x %d%% %d %s\n",
                  driver_version,
                  &apm_bios_info_major,
                  &apm_bios_info_minor,
                  &apm_bios_info_flags,
                  &ac_line_status,
                  &battery_status,
                  &battery_flag,
                  &battLife,
                  &time_units,
                  units) == 10) {
            *level = battLife;
            if(ac_line_status == 0x01){
                *level = 0;
            }
        }
        /*
         * for apm_bios-0.5
         */
        else
        if(sscanf(buffer, "Battery life: %d%%\n", &battLife) == 1){
            *level = battLife;
        }else
        if(strcmp("AC: on line\n", buffer)==0){
            *level = 0;
        }
    }
    fclose(fp);

    if(*level == 0){
        return(AC_LINE);
    }else{
        return(BATTERY);
    }
}
#else
int get_apm(int *level) {
    /* Init */
    *level = 0;

    /*  ac_line */
    return(AC_LINE);
}
#endif


ems_term() {
	int r;
	char path[256];

#if 0
	FILE* co;
	co = fopen("/dev/console", "w");

	fprintf(co, "exec ems_term()\n");
#endif

	close(cs_sock_fd);

	sprintf(path,"%s/%s",cs_workdir,cs_screen_name);
	unlink(path);
	sprintf(path,"%s/%s",cs_workdir,cs_sock_name);
	unlink(path);

	/* fprintf(co, "unlink OK\n"); */
#if 0
        /* 
         * mglsvrcons ϰʲ loop ƤޤｪλƤʤ
         * Τ text screen ˼Ԥ롣
         * θפ˻פΤǳ
         */

	while(sw_list) {
		sw_free(sw_list);
	}
#endif
	/* fprintf(co, "end of ems_term()\n"); fclose(co); */

}


ems_init() {
	char *p,*name;
	char buf[32];
	int sd,fd;
	int flag,i,r;
	struct sockaddr_un addr;
	char *v;

	name = "";

	if (name = ttyname(0)) {
		if (p = strrchr(name,'/')) {
			name = p+1;
		}
	}
	if ((geteuid() != 0 ) && (p = getenv("HOME"))) {
		sprintf(cs_workdir,"%s/.mgl",p);
	} else {
		strcpy(cs_workdir,"/var/run");
	}

	strcpy(cs_name,name);
	sprintf(cs_sock_name,"mgl.sock.%s",name);
	sprintf(cs_screen_name,"mgl.screen.%s",name);
	getcwd(cs_curdir,sizeof(cs_curdir));

	r = chdir(cs_workdir);
	if ((geteuid()!=0) && (r < 0)) {
		printf("creating ${HOME}/.mgl\n");
		mkdir(cs_workdir,0700);
		r = chdir(cs_workdir);
	}
	if (r < 0) {
		printf("cannot chdir to ${HOME}/.mgl or /var/run\n");
		exit(1);
	}
	sd = socket( AF_LOCAL, SOCK_STREAM, 0);
	if (sd < 0) {
		perror("socket");
		exit(1);
	}
	flag = 1;
        //r = setsockopt(sd, SOL_SOCKET,SO_REUSEADDR,(char *)&flag,sizeof(int));
	if (r < 0) {
		perror("sockopt");
	}
	//memset(&addr,sizeof(addr),0);
	addr.sun_family = AF_UNIX;
	strcpy(addr.sun_path,cs_sock_name);

	r = bind(sd,(struct sockaddr *)&addr, sizeof(struct sockaddr_un));
	if (r != 0) {
		perror("bind");
		exit(1);
	}
	chmod(cs_sock_name,0777);
	fd = open(cs_screen_name,O_CREAT|O_RDWR,0666);
	if (fd < 0) {
		perror("creat");
		exit(1);
	}
	lseek(fd,MAX_SCREEN_SIZE-1,0);
	write(fd,buf,1);

#ifdef VIRTUAL_FRAME_ADDR
	v = mmap((void *)VIRTUAL_FRAME_ADDR,MAX_SCREEN_SIZE+MAX_SCREENATTR_SIZE
		,PROT_READ|PROT_WRITE,MAP_SHARED|MAP_FIXED,fd,0);
printf("try to mmap %08x -> %08x\n",VIRTUAL_FRAME_ADDR,v);
#else
  #ifndef __CYGWIN__
	v = mmap(NULL,MAX_SCREEN_SIZE+MAX_SCREENATTR_SIZE
		,PROT_READ|PROT_WRITE,MAP_SHARED,fd,0);
  #else
	hFileMapping = CreateFileMapping((HANDLE) 0xffffffff,
					 NULL,
					 PAGE_READWRITE,
					 0,
					 MAX_SCREEN_SIZE+MAX_SCREENATTR_SIZE,
					 "mgl2_screen");
	if(hFileMapping == NULL) {
		perror("mmap");
		exit(1);
	}
	v = (char *) MapViewOfFile(hFileMapping,
				   FILE_MAP_ALL_ACCESS,
				   0,
				   0,
				   0);
  #endif /* !__CYGWIN__ */
#endif
	if (v == MAP_FAILED) {
		perror("mmap");
		exit(1);
	}
	chown(cs_sock_name,getuid(),getgid());
	chmod(cs_sock_name,0700);
	chown(cs_screen_name,getuid(),getgid());
	chmod(cs_screen_name,0600);
	chdir(cs_curdir);
	r = listen(sd,5);
	if (r != 0) {
		perror("listen");
	}
	cs_sock_fd = sd;
	cs_screen_addr = mgl_screen_addr = v;
	mgl_add_term_func((void (*)())ems_term,0);
}

ems_postinit() {
	char path[256];
	int fbsize;

	sprintf(path,"%s/%s",cs_workdir,cs_screen_name);
	if (mgl_screen_name[0]) {
		chown(mgl_screen_name,getuid(),getgid());
		chmod(mgl_screen_name,0600);
		//strncpy(cs_screen_name,mgl_screen_name,16);
		//cs_screen_name[16] = 0;
		fbsize = MAX_SCREENATTR_SIZE;
		truncate(path,fbsize);
	} else {
		fbsize = mgl_screen_realwidth * SCREEN_HEIGHT 
				* mgl_screen_bpp / 8;
		fbsize += MAX_SCREENATTR_SIZE;
		truncate(path,fbsize);
	}
}

#ifdef USE_SPAWN
#ifndef HAS_SPAWN
static int spawnv(char *path, char *argv[]) {
	int pid;

	pid = fork();
	if (pid == 0) {
		mgl_ignore_auto_close();
		execv(path,argv);
		exit(0);
	}
	return pid;
}
#endif
#endif

char *env_mglim;

main(int argc,char *argv[]) {
	fd_set fds;
	int fdnum = 0;
	int c,i,r,rfd;
	struct screen_window *sw;
	struct virtual_key *v;
	struct screen_map *m;
	char *p;
	extern char *optarg;
	int type = 0;
	int ticks = 15 * 1000; /* 15 sec */

	p = mgl_getenv("MGLIM");
	if (p) {
		env_mglim = malloc(strlen(p) + 1);
		if (env_mglim) strcpy(env_mglim,p);
	}
	unsetenv("MGLIM");
#if 0
	SCREEN_WIDTH = 240;
	SCREEN_HEIGHT = 320;
#else
	SCREEN_WIDTH = 640;
	SCREEN_HEIGHT = 240;
#endif
	type |= HAS_TASKBAR;
	type |= HAS_MINIAPLI;

	while ((c = getopt(argc,argv,"s:tmde:"))!=EOF) {
		switch (c) {
		case 's': 
			if (!strcmp("ppc",optarg)) {
				SCREEN_WIDTH = 240;
				SCREEN_HEIGHT = 320;
				type &= ~HAS_MINIAPLI;
			}
			else if (!strcmp("hpc",optarg)) {
				SCREEN_WIDTH = 640;
				SCREEN_HEIGHT = 240;
			}
			else if (!strcmp("cga",optarg)) {
				SCREEN_WIDTH = 640;
				SCREEN_HEIGHT = 200;
				MINI_YS = 200 - 16;
			}
			else if (!strcmp("320x240",optarg)) {
				SCREEN_WIDTH = 320;
				SCREEN_HEIGHT = 240;
				type &= ~HAS_TASKBAR;
				type &= ~HAS_MINIAPLI;
			}
			else if (!strcmp("512x348",optarg)) {
				SCREEN_WIDTH = 512;
				SCREEN_HEIGHT = 348;
			}
			else if (!strcmp("640x400",optarg)) {
				SCREEN_WIDTH = 640;
				SCREEN_HEIGHT = 400;
			}
			else if (!strcmp("800x600",optarg)) {
				SCREEN_WIDTH = 800;
				SCREEN_HEIGHT = 600;
			}
			else if (!strcmp("640x870",optarg)) {
				SCREEN_WIDTH = 640;
				SCREEN_HEIGHT = 870;
			}
			else if (!strcmp("vga",optarg)) {
				SCREEN_WIDTH = 640;
				SCREEN_HEIGHT = 480;
			}
			break;
		case 't':
			type ^= HAS_TASKBAR;
			break;
		case 'm':
			type ^= HAS_MINIAPLI;
			break;
		case 'd':
			type ^= HAS_DUALMAIN;
			break;
		case 'e':
			cmdline = optarg;
		}
	}
	ems_init();
	mgl_noclient = 1;
	mgl_display[0] = 0;
	mgl_event_manager_initiator = SVR_EVENT_MANAGER_INITIATOR;

	if (!open_graph() ) {
	        fprintf(stderr,"open_graph fail.\n");
		exit(1);
	}

	ems_postinit();

	sigset(SIGPIPE,SIG_IGN);
	sigset(SIGINT,SIG_IGN);
	sigset(SIGQUIT,SIG_IGN);
	sigset(SIGCHLD,SIG_IGN);
	v = create_virtual_key3(0,0,SCREEN_WIDTH,SCREEN_HEIGHT
					,MK_MDOWN,MK_MMOVE,MK_MUP);
#ifdef USE_KEYMAP
	for (i = 0; i< MKE_SIZE; i++) {
		sym_trans_tab[i] = MKE_BASE+i;
	}
#endif
	vk_attach(NULL,v);
	screen_map_init(type);

	bg_main1 = conv_screen_from_v1(&bg_main1_v1,STK_NATIVE);
	bg_main2 = conv_screen_from_v1(&bg_main2_v1,STK_NATIVE);
	bg_mini1 = conv_screen_from_v1(&bg_mini1_v1,STK_NATIVE);
	bg_mini2 = conv_screen_from_v1(&bg_mini2_v1,STK_NATIVE);

	if (sm_main1) {
		m = sm_main1;
		printf("main1     %d %d %d %d\n",m->x,m->y,m->xs,m->ys);
	}
	if (sm_main2) {
		m = sm_main2;
		printf("main2     %d %d %d %d\n",m->x,m->y,m->xs,m->ys);
	}
	if (sm_mini1) {
		m = sm_mini1;
		printf("mini1     %d %d %d %d\n",m->x,m->y,m->xs,m->ys);
	}
	if (sm_mini2) {
		m = sm_mini2;
		printf("mini2     %d %d %d %d\n",m->x,m->y,m->xs,m->ys);
	}
	if (sm_taskbar) {
		m = sm_taskbar;
		printf("taskbar   %d %d %d %d\n",m->x,m->y,m->xs,m->ys);
		set_color(COLOR_LIGHTGRAY);
		fill_rect(m->x,m->y,m->xs,m->ys);
		refresh();
	}
	if (sm_taskbar) {
		m = sm_taskbar;
		tb_init(m->x,m->y,m->xs,m->ys);
	}
	i = screen_map_size(AT_MAIN);
	if (i >= 0) {
	    printf("MAIN size %d %d \n",screen_map[i].xs,screen_map[i].ys);
	}
	i = screen_map_size(AT_MINIAPLI);
	if (i >= 0) {
	    printf("MINIAPLI size %d %d \n",screen_map[i].xs,screen_map[i].ys);
	}

	power_image = conv_screen_from_mgr(icon_ac,STK_NATIVE);
	tb_attach(power_image,MK_POWER,1);
	clock_image = create_memscreen(32,16,NULL,STK_NATIVE,0);
	tb_attach(clock_image,MK_CLOCK,1);

	im_image = conv_screen_from_mgr(icon_im,STK_NATIVE);
	tb_attach(im_image,MK_IM,0);

#ifdef SVR_SUPPORT_NINE
	if (sm_taskbar) {
		int x,y,xs,ys;
		m = sm_taskbar;
		x = m->x;
		xs = m->xs;
		y = 0;
		ys = SCREEN_HEIGHT - m->ys;
		svr_nine = create_nine(x,y,xs,ys,0);
		if (svr_nine) {
			nine_image = conv_screen_from_mgr(icon_nine,STK_NATIVE);
			tb_attach(nine_image,MK_NINE,0);
		}
	}
#endif
	if (cmdline) {
		char *args[100];
		char **ap = args;
		char *shell_name;
		char *shell;

#ifdef USE_SPAWN
		setenv("MGL_DISPLAY",cs_name,1);
		if (env_mglim)
			setenv("MGLIM",env_mglim,1);

		if (getenv("SHELL")) {
			shell = getenv("SHELL");
		} else {
			shell = "/bin/sh";
		}
		shell_name = shell;
		if (shell[0] == '/') {
			shell_name = strrchr(shell,'/') + 1;
		}
		*ap++ = shell_name;
		*ap++ = "-c";
		*ap++ = cmdline;
		*ap++ = NULL;
		spawnv(shell,args);
#else
		if (fork() == 0) {
			mgl_ignore_auto_close();
			setegid(getgid());
			seteuid(getuid());
			setenv("MGL_DISPLAY",cs_name,1);
			if (env_mglim)
				setenv("MGLIM",env_mglim,1);

			if (getenv("SHELL")) {
				shell = getenv("SHELL");
			} else {
				shell = "/bin/sh";
			}
			shell_name = shell;
			if (shell[0] == '/') {
				shell_name = strrchr(shell,'/') + 1;
			}
			*ap++ = shell_name;
			*ap++ = "-c";
			*ap++ = cmdline;
			*ap++ = NULL;
			execv(shell,args);
			exit(1);
		}
#endif
	}
	while (1) {
		int f;
		if (disp_request) {
			screen_map_disp();
			disp_request = 0;
		}
		f = draw_power(power_image);
		f |=  draw_clock(clock_image);
		if (f) {
			tb_show();
		}
		FD_ZERO(&fds);
		FD_SET(0,&fds);
		FD_SET(cs_sock_fd,&fds);
		fdnum = cs_sock_fd + 1;
		sw = sw_list;
		while (sw) {
			FD_SET(sw->fd,&fds);
			if (sw->fd >= fdnum) 
				fdnum = sw->fd+1;
			sw = sw->link;
		}
//printf("start key_select\n");
		rfd = key_select(fdnum,&fds,ticks);
//printf("end key_select %d\n",rfd);
		if (rfd < 0) {
			perror("key_select");
			continue;
		}
		if ((rfd > 0) && FD_ISSET(0,&fds)) {
			r -= 1;
			execute_send_event();
		}
		if ((rfd > 0) && FD_ISSET(cs_sock_fd,&fds)) {
			r -= 1;
//printf("accept ing\n");
			r = accept(cs_sock_fd,(struct sockaddr *)0,(int *)0);
			if (r < 0) {
				perror("accept");
				continue;
			}
//printf("done accept\n");
			sw_alloc(r);
		}
		if (rfd > 0) {
			sw = sw_list;
			while (sw) {
				if (FD_ISSET(sw->fd,&fds)) {
					execute_client_req(sw);
					break;
				}
				sw = sw->link;
			}
		}
	}
}
