/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#define MGL_PREFIX
#include "mgl2.h"
#include "draw_engine.h"
#include <stdlib.h>


#ifndef NULL
#define NULL (void *)0
#endif

static struct screen *s_freelist;
static int s_freecnt;

struct screen *de_create_memscreen(int xs,int ys,char *bitmap,int op) {
	struct screen *ret;
	int type = ST_ALLOCED;

	if (s_freelist) {
		ret = s_freelist;
		s_freelist = *(struct screen **)s_freelist;
		s_freecnt--;
	} else {
		ret = (struct screen *)malloc(sizeof(*ret));
//printf("malloc memscreen\n");
	}
        if(!ret){
		perror("malloc");
		return (struct screen *)NULL;
        }

	ret->type = type;
	ret->de = NULL;
	ret->op = 0; 
	if (op) {
		ret->op = op;
		ret->type |= ST_RESERVEOP;
	}
	ret->wbytes = 0;
	ret->width =  xs; 
	ret->height = ys;
	ret->bitmap = bitmap;
	ret->off_x = 0;
	ret->off_y = 0;
	ret->need_clipping=1;	/* for test ? */
	ret->plbytes = 0;
	if (current_screen) {
		ret->_pen_font = current_screen->_pen_font;
		ret->_pen_color = current_screen->_pen_color;
	}
	return ret;
}

struct screen *de_create_subscreen(struct screen *org, int x, int y,
                                int xs, int ys,int opt) {
	struct screen *ret;

	if ((x < 0) || (y < 0)) return NULL;
	
	if (x + xs > org->width) {
		xs = org->width - x;
		if (xs < 0) xs = 0;
	}
	if (y + ys > org->height) {
		ys = org->height -y;
		if (ys < 0) ys = 0;
	}

	if (s_freelist) {
		ret = s_freelist;
		s_freelist = *(struct screen **)s_freelist;
		s_freecnt --;
	} else {
		ret = (struct screen *)malloc(sizeof(*ret));
//printf("malloc memscreen(sub)\n");
	}
       	if(!ret){
		perror("malloc");
		return (struct screen *)NULL;
       	}
	ret->type = ST_SUBSCREEN | ST_ALLOCED | (org->type & ST_KINDMASK );
	ret->op = 0;
	if (org->type & ST_RESERVEOP) {
		ret->type |= ST_RESERVEOP;
		ret->op = org->op;
	}
	ret->de = org->de;
	ret->wbytes = org->wbytes;
	ret->width = xs;
	ret->height = ys;
	ret->bitmap = (char *)org;
	ret->off_x = org->off_x + x;
	ret->off_y = org->off_y + y;
	ret->need_clipping=1;	/* for test ? */
	ret->plbytes = org->plbytes;
	ret->_pen_color = org->_pen_color;
	ret->_pen_font = org->_pen_font;
	return ret;
}

void de_free_screen(struct screen *s) {
	if (s->type & ST_ALLOCED_BITMAP) free(s->bitmap);
	if (s->type & ST_ALLOCED) {
		if (s_freecnt > 30) {
			free((void *)s);
//printf("free screen s_freecnt %d\n",s_freecnt);
		} else {
			*(struct screen **)s = s_freelist;
			s_freelist = s;
			s_freecnt ++;
		}
	}
}


static void gen_get_pixstream(struct screen *s,int x, int y, int *buf, int length, int dir,int op
	, struct draw_engine *self) {

	switch (dir & 3) {
	case DIR_NORTH:
		while (length-- > 0) {
			*buf++ = get_pixel(s,x++,y,op);
		}
		break;
	case DIR_WEST:
		while (length-- > 0) {
			*buf++ = get_pixel(s,x,y--,op);
		}
		break;
	case DIR_SOUTH:
		while (length-- > 0) {
			*buf++ = get_pixel(s,x--,y,op);
		}
		break;
	case DIR_EAST:
		while (length-- > 0) {
			*buf++ = get_pixel(s,x,y++,op);
		}
		break;
	}
}

static void gen_put_pixstream(struct screen * s,int x, int y, int *buf, int length, int dir
	, struct draw_engine *self) {
	int x1,y1;
	int x2,y2;

	CLIP_PIXSTREAM(s,x,y,buf,length,dir,x1,y1,x2,y2);
	s->need_clipping--;
	switch (dir & 3) {
	case DIR_NORTH:
		while (length-- > 0) {
			put_pixel(s,x++,y,*buf++);
		}
		break;
	case DIR_WEST:
		while (length-- > 0) {
			put_pixel(s,x,y--,*buf++);
		}
		break;
	case DIR_SOUTH:
		while (length-- > 0) {
			put_pixel(s,x--,y,*buf++);
		}
		break;
	case DIR_EAST:
		while (length-- > 0) {
			put_pixel(s,x,y++,*buf++);
		}
		break;
	}
	s->need_clipping++;
}

static void gen_put_pixstream_rect(struct screen *s,int x, int y, int *buf, int length, int dir
	,int width, struct draw_engine *self) {

	switch (dir & 3) {
	case DIR_NORTH:
		while (length > width) {
			put_pixstream(s,x,y,buf,width,dir,self);
			length -= width;
			buf += width;
			y++;
		}
		break;
	case DIR_WEST:
		while (length > width) {
			put_pixstream(s,x,y,buf,width,dir,self);
			length -= width;
			buf += width;
			x++;
		}
		break;
	case DIR_SOUTH:
		while (length > width) {
			put_pixstream(s,x,y,buf,width,dir,self);
			length -= width;
			buf += width;
			y--;
		}
		break;
	case DIR_EAST:
		while (length > width) {
			put_pixstream(s,x,y,buf,width,dir,self);
			length -= width;
			buf += width;
			x--;
		}
		break;
	}
	if (length) {
		put_pixstream(s,x,y,buf,length,dir,self);
	}
}

static void gen_draw_line_vertical(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
//printf("gen_draw_line_vert y(%d-%d),x (%d)\n",y1,y2,x1);
	CLIP_VLINE(s,x1,y1,x2,y2);
	s->need_clipping--;
	while(y1 <= y2) {
		draw_pixel(s,x1,y1++);
	}
	s->need_clipping++;
}


static void gen_draw_line_horizontal(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
//printf("gen_draw_line_horizon x(%d-%d),y (%d)\n",x1,x2,y1);
	CLIP_HLINE(s,x1,y1,x2,y2);

	s->need_clipping--;
	while(x1 <= x2) {
		draw_pixel(s,x1++,y1);
	}
	s->need_clipping++;
}

static void gen_clear_screen_vertical(struct screen *s,struct draw_engine *self) {
	int x1,y1,x2,y2;

//printf("gen_clear_screen_vert \n");
	x1 = 0;
	x2 = s->width-1;
	y1 = 0;
	y2 = s->height-1;

	s->need_clipping--;
	while (x1 <= x2) {
	    draw_line_vertical(s,x1,y1,x1,y2,self);
	    x1++;
	}
	s->need_clipping++;
}

static void gen_clear_screen_horizontal(struct screen *s,struct draw_engine *self) {
	int x1,y1,x2,y2;

//printf("gen_clear_screen_horizon \n");
	x1 = 0;
	x2 = s->width-1;
	y1 = 0;
	y2 = s->height-1;

	s->need_clipping--;
	while (y1 <= y2) {
	    draw_line_horizontal(s,x1,y1,x2,y1,self);
	    y1++;
	}
	s->need_clipping++;
}

#define ABS(a) (((a)<0) ? -(a) : (a))
static void gen_draw_line(struct screen *s,int x1, int y1, int x2, int y2, struct draw_engine *self) {
    int dx = x2 - x1;
    int dy = y2 - y1;
    int ax = ABS(dx) << 1;
    int ay = ABS(dy) << 1;
    int sx = (dx >= 0) ? 1 : -1;
    int sy = (dy >= 0) ? 1 : -1;

    int x = x1;
    int y = y1;

    if (y1 == y2) {
	if (x2 < x1) {
		x1 = x2;
		x2 = x;
	}
	draw_line_horizontal(s,x1,y1,x2,y2,self);
	return;
    }
    if (x1 == x2) {
	if (y2 < y1) {
		y1 = y2;
		y2 = y;
	}
	draw_line_vertical(s,x1,y1,x2,y2,self);
	return;
    }

    if (ax > ay) {
	int d = ay - (ax >> 1);
	while (x != x2) {
//printf("draw_pixel (%d,%d)\n",x,y);
	    draw_pixel(s,x, y);

	    if (d > 0 || (d == 0 && sx == 1)) {
		y += sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    } else {
	int d = ax - (ay >> 1);
	while (y != y2) {
//printf("draw_pixel (%d,%d)\n",x,y);
	    draw_pixel(s,x, y);

	    if (d > 0 || (d == 0 && sy == 1)) {
		x += sx;
		d -= ay;
	    }
	    y += sy;
	    d += ax;
	}
    }
//printf("draw_pixel (%d,%d)\n",x,y);
    draw_pixel(s,x, y);
}
#define MAX_SCREEN_SIZE	800

void bitblt_generic(struct screen *dst, int dx, int dy, struct screen *src
	, int sx, int sy, int xsize, int ysize, int op) {
	int i,j,s;
	int rsize;
	int tmp[MAX_SCREEN_SIZE];

//printf("bitblt_generic (%d,%d) from (%d,%d), size(%d,%d) op %08x\n"
//,dx,dy,sx,sy,xsize,ysize,op);
	CLIP_BITBLT(dst,dx,dy,src,sx,sy,xsize,ysize);

	if (src->type & ST_RESERVEOP) {
		op = src->op;
	}
	if (op & BLT_TILING) {
	    int ssx;
	    ssx = sx;
//printf("TILING sw %d sh %d\n",src->width,src->height);
	    for (i=0; i<ysize; i++,sy++,dy++) {
		sx = ssx;
		if (sy >= src->height) sy = 0;

		rsize = 0;
		while (rsize < xsize) {
			s = xsize - rsize;
			if (s > src->width - sx) {
				s = src->width - sx;
			}
//printf("TILING get (%d,%d) len %d!\n",sx,sy,s);
			src->de->_get_pixstream(src,sx,sy,tmp+rsize,s
					,DIR_NORTH,op,src->de);
			sx = 0;
			rsize += s;
		}
//printf("TILING put (%d,%d) %d len %d!\n",dx,dy,sy,xsize);
		dst->de->_put_pixstream(dst,dx,dy,tmp,xsize,DIR_NORTH,dst->de);
	    }
	    goto ret;
	}

	if ((dst != src) || (dy <= sy)) {
	    for (i=0; i<ysize; i++,sy++,dy++) {
		src->de->_get_pixstream(src,sx,sy,tmp,xsize,DIR_NORTH,op,src->de);
//printf("put_pixstream %d,%d = %x %x %x \n",sx,sy,tmp[0],tmp[1],tmp[2]);
		dst->de->_put_pixstream(dst,dx,dy,tmp,xsize,DIR_NORTH,dst->de);
	    }
	} else {
	    dy += ysize -1;
	    sy += ysize -1;
	    for (i=0; i<ysize; i++,sy--,dy--) {
		src->de->_get_pixstream(src,sx,sy,tmp,xsize,DIR_NORTH,op,src->de);
//printf("put_pixstream %d,%d = %x %x %x \n",sx,sy,tmp[0],tmp[1],tmp[2]);
		dst->de->_put_pixstream(dst,dx,dy,tmp,xsize,DIR_NORTH,dst->de);
	    }
	}
ret:
}

static void gen_bitblt(struct screen *dst, int dx, int dy, struct screen *src,
	 int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {

	if (src == NULL) src = current_screen;
	if (dst == NULL) dst = current_screen;
	if ((xsize < 0) || (ysize < 0)) {
		xsize = src->width;
		ysize = src->height;
	}
	CLIP_BITBLT(dst,dx,dy,src,sx,sy,xsize,ysize);

	dst->need_clipping--;
	if (src->type & ST_RESERVEOP) {
		op = src->op;
	}

#if 0
	if ((dst->de != self) || (src->de != self)) {
		bitblt_generic(dst,dx,dy,src,sx,sy,xsize,ysize,op);
		goto ret;
	}
#endif

	if ((src->type & ST_RESERVEOP) && (dst != src) && (op & BLT_MASKING)) {
		op = src->op;
		if (!(op & BLT_TILING) && bitblt_reserved_mask ) {
			bitblt_reserved_mask(dst,dx,dy,src,sx,sy
				,xsize,ysize,op,self);
			goto ret;
		}
		if ((op & BLT_TILING) && bitblt_reserved_masktile) {
			bitblt_reserved_masktile(dst,dx,dy,src,sx,sy
				,xsize,ysize,op,self);
			goto ret;
		}
	} else if ((src == dst) && (op == 0) 
		&& bitblt_scroll_forward && bitblt_scroll_backward) {
		if ((dy < sy) || ( (dy == sy) && (dx < sx ))) {
			bitblt_scroll_forward(dst,dx,dy,src,sx,sy
				,xsize,ysize,op,self);
			goto ret;
		} else {
			bitblt_scroll_backward(dst,dx,dy,src,sx,sy
				,xsize,ysize,op,self);
			goto ret;
		}
	} else if ((src != dst) && (op == 0) && bitblt_copy 
		&& (src->de->_bitblt_copy == dst->de->_bitblt_copy) ) {
		bitblt_copy(dst,dx,dy,src,sx,sy
			,xsize,ysize,op,self);
		goto ret;
	}
	bitblt_generic(dst,dx,dy,src,sx,sy,xsize,ysize,op);
ret:
	dst->need_clipping++;
}

setup_draw_engine(struct draw_engine *self, int hint) {
	/* level 0 */
	if (!create_subscreen) {
		printf("_create_subscreen not defined\n");
		exit(1);
	}
	if (!free_screen) {
		printf("_free_screen not defined\n");
		exit(1);
	}
	if (!get_pixel) {
		printf("_get_pixel not defined\n");
		exit(1);
	}
	if (!put_pixel) {
		printf("_put_pixel not defined\n");
		exit(1);
	}
	if (!draw_pixel) {
		printf("_draw_pixel not defined\n");
		exit(1);
	}
	if (!set_color) {
		printf("_set_color not defined\n");
		exit(1);
	}
	/* level 1 */
	if (!draw_line_vertical) {
		draw_line_vertical = gen_draw_line_vertical;
	}
	if (!draw_line_horizontal) {
		draw_line_horizontal = gen_draw_line_horizontal;
	}
	if (!draw_line) {
		draw_line = gen_draw_line;
	}
	if (hint & DE_FAST_VERTICAL) {
	    if (!clear_screen) {
		clear_screen = gen_clear_screen_vertical;
	    }
	} else {
	    if (!clear_screen) {
		clear_screen = gen_clear_screen_horizontal;
	    }
	}
	if (!get_pixstream) {
		get_pixstream = gen_get_pixstream;
	}
	if (!put_pixstream) {
		put_pixstream = gen_put_pixstream;
	}
	if (!put_pixstream_rect) {
		put_pixstream_rect = gen_put_pixstream_rect;
	}
	/* bitblt */
	if (!bitblt) {
		bitblt = gen_bitblt;
	}
}
