/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#define MGL_PREFIX
#include "config.h"
#include "mgl2.h"
#include "draw_engine.h"
#include <stdlib.h>

#ifdef USE_LOCAL_MEMSET
#define memset	mgl_memset
#endif
#ifdef USE_LOCAL_MEMMOVE
#define memmove	mgl_memmove
#endif

#define lp_color	(s->_pen_color.local_color)
#define lp_dodither	(s->_pen_color.flags)

#ifndef NULL
#define NULL ((void *)0)
#endif

static struct draw_engine dec192;

/*
  0 %       20%       40%       60%    80%     100%
  .....    #....    #..#.    #.##.    ####.    #####
  .....    ..#..    ..#.#    #.#.#    #.###    #####

*/
static const int dither_map[2][5][16] = {
{  {0,1,1,1,1   , 1,2,2,2,2,  2,3,3,3,3,  3},
   {0,0,0,0,1   , 1,1,1,1,2,  2,2,2,2,3,  3},
   {0,0,0,1,1   , 1,1,1,2,2,  2,2,2,3,3,  3},
   {0,0,1,1,1   , 1,1,2,2,2,  2,2,3,3,3,  3},
   {0,0,0,0,0   , 1,1,1,1,1,  2,2,2,2,2,  3},
}, {
   {0,0,0,1,1   , 1,1,1,2,2,  2,2,2,3,3,  3},
   {0,0,0,0,0   , 1,1,1,1,1,  2,2,2,2,2,  3},
   {0,1,1,1,1   , 1,2,2,2,2,  2,3,3,3,3,  3},
   {0,0,0,0,1   , 1,1,1,1,2,  2,2,2,2,3,  3},
   {0,0,1,1,1   , 1,1,2,2,2,  2,2,3,3,3,  3},
}
};

static struct screen *dec192_create_memscreen(int xs,int ys,char *bitmap,int op) {
	struct screen *ret;
	int wb;
	int type = STK_GENERIC_192COLOR | ST_ALLOCED;

	wb = xs;
	ret = de_create_memscreen(xs,ys,bitmap,op);
	if (!ret) return NULL;

	if (!bitmap) {
		bitmap = (char *)malloc(ys * wb);
        	if(!bitmap){
			perror("malloc");
			return (struct screen *)NULL;
        	}
		type |= ST_ALLOCED_BITMAP;
		memset(bitmap,0,ys * wb);
	}

	ret->type |= type;
	ret->de = &dec192;
	ret->wbytes = wb;
	ret->bitmap = bitmap;
	return ret;
}

static struct screen *dec192_create_subscreen(struct screen *org, int x, int y,
                                int xs, int ys,int opt) {
	struct screen *ret;

	ret = de_create_subscreen(org,x,y,xs,ys,opt);
	if (!ret) return NULL;

	if (opt & CSS_AS_MEMSCREEN) {
		ret->type &= ~ST_SUBSCREEN;
		ret->bitmap = org->bitmap + (ret->off_y * ret->wbytes)
			+ (ret->off_x);
		ret->off_y = 0;
		ret->off_x = 0;
	}
	return ret;
}

static void dec192_free_screen(struct screen *s) {
	de_free_screen(s);
}


static int dec192_get_pixel(struct screen *s,int x, int y,int op) {
	unsigned char *p;
	int col,ret;

	if (x < 0 || x >= s->width
	   || y < 0 || y >= s->height) return 0;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x;
	col = *p;

	ret = CONV_FROM_COL192(col);
	if ((op & BLT_MASKING) && (ret == (op & BLT_MASKCOL))) {
		ret = COLOR_TRANSPARENT;
	}
//printf("dec192_get_pixel (%d %d) = %08x\n",x,y,ret);
	return ret;
}

static void dec192_put_pixel(struct screen *s,int x, int y, int col) {
	char *p;

	if (col & COLOR_TRANSPARENT) return;
	if (col < 0) return;

	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x;
	if (col & COLOR_DITHER) {
		int dx = (x + (y>>2) + (y>>4))%5;
		int dy = y % 2;
		int dc = dither_map[dy][dx][col & 0xf]
			| (dither_map[dy][dx][(col>>4) & 0xf] << 2);
		*p = dc | CONV_TO_COL192(col & ~0xff);
	} else {
		*p = CONV_TO_COL192(col);
	}
	return;
}

static void dec192_get_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir,int op
		,struct draw_engine *self) {
	unsigned char *p;
	int i;
	int d;
	int r;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x;

	switch(dir) {
	case DIR_NORTH:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = CONV_FROM_COL192(*p);
			if (r == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p++;
		} else {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
 		    for (i=0; i<length; i++) {
			*buf++ = CONV_FROM_COL192(*p);
			p++;
		    }
#else
		    int len4 = length/4;
		    length -= len4 * 4;
 		    for (i=0; i<len4; i++) {
			buf[0] = CONV_FROM_COL192(p[0]);
			buf[1] = CONV_FROM_COL192(p[1]);
			buf[2] = CONV_FROM_COL192(p[2]);
			buf[3] = CONV_FROM_COL192(p[3]);
			buf += 4;
			p += 4;
		    }
 		    for (i=0; i<length; i++) {
			*buf++ = CONV_FROM_COL192(*p);
			p++;
		    }
#endif
		}
		break;
	case DIR_WEST:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = CONV_FROM_COL192(*p);
			if (r == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p -= s->wbytes;
		} else for (i=0; i<length; i++) {
			*buf++ = CONV_FROM_COL192(*p);
			p -= s->wbytes;
		}
		break;
	case DIR_SOUTH:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = CONV_FROM_COL192(*p);
			if (r == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p--;
		} else for (i=0; i<length; i++) {
			*buf++ = CONV_FROM_COL192(*p);
			p--;
		}
		break;
	case DIR_EAST:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = CONV_FROM_COL192(*p);
			if (r == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p += s->wbytes;
		} else for (i=0; i<length; i++) {
			*buf++ = CONV_FROM_COL192(*p);
			p += s->wbytes;
		}
		break;
	}
}

static void dec192_put_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir
		,struct draw_engine *self) {
	char *p;
	int col,mask;
	int d,i;
	int dx=0,dy=0;
	int dodither = 0;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x;

	switch(dir) {
	case DIR_NORTH:
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
	    if (*buf & COLOR_DITHER) {
		dx = (x + (y>>2) + (y>>4))%5;
		dy = y % 2;
		dodither = 1;
	    }
	    for (i=0; i<length;i++,x++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		    if (dodither) {
			int dc = dither_map[dy][dx][col & 0xf]
				| (dither_map[dy][dx][(col>>4) & 0xf] << 2);
			*p = dc | CONV_TO_COL192(col & ~0xff);
		    } else {
		        *p = CONV_TO_COL192(col);
		    }
		}
		p++;
		if (dodither) if (++dx >= 5) dx = 0;
	    }
#else
	    if (*buf & COLOR_DITHER) {
		int len5,pre_len;
		dx = (x + (y>>2) + (y>>4))%5;
		dy = y % 2;

		pre_len = (5-dx)%5;
		length -= pre_len;
		len5 = length/5;
		length -= len5 * 5;
		for (i=0; i<pre_len;i++) {
		    col = *buf++;
		    if (!(col & COLOR_TRANSPARENT)) {
			int dc = dither_map[dy][dx][col & 0xf]
				| (dither_map[dy][dx][(col>>4) & 0xf] << 2);
			*p = dc | CONV_TO_COL192(col & ~0xff);
		    }
		    p++;
		    dx++;
		}
		for (i=0; i<len5; i++) {
		    col = buf[0];
		    if (!(col & COLOR_TRANSPARENT)) {
			int dc = dither_map[dy][0][col & 0xf]
				| (dither_map[dy][0][(col>>4) & 0xf] << 2);
			p[0] = dc | CONV_TO_COL192(col & ~0xff);
		    }
		    col = buf[1];
		    if (!(col & COLOR_TRANSPARENT)) {
			int dc = dither_map[dy][1][col & 0xf]
				| (dither_map[dy][1][(col>>4) & 0xf] << 2);
			p[1] = dc | CONV_TO_COL192(col & ~0xff);
		    }
		    col = buf[2];
		    if (!(col & COLOR_TRANSPARENT)) {
			int dc = dither_map[dy][2][col & 0xf]
				| (dither_map[dy][2][(col>>4) & 0xf] << 2);
			p[2] = dc | CONV_TO_COL192(col & ~0xff);
		    }
		    col = buf[3];
		    if (!(col & COLOR_TRANSPARENT)) {
			int dc = dither_map[dy][3][col & 0xf]
				| (dither_map[dy][3][(col>>4) & 0xf] << 2);
			p[3] = dc | CONV_TO_COL192(col & ~0xff);
		    }
		    col = buf[4];
		    if (!(col & COLOR_TRANSPARENT)) {
			int dc = dither_map[dy][4][col & 0xf]
				| (dither_map[dy][4][(col>>4) & 0xf] << 2);
			p[4] = dc | CONV_TO_COL192(col & ~0xff);
		    }
		    buf += 5;
		    p += 5;
		}
		for (i=0; i<length;i++) {
		    col = *buf++;
		    if (!(col & COLOR_TRANSPARENT)) {
			int dc = dither_map[dy][i][col & 0xf]
				| (dither_map[dy][i][(col>>4) & 0xf] << 2);
			*p = dc | CONV_TO_COL192(col & ~0xff);
		    }
		    p++;
		}
	    } else {
		int len4;
		len4 = length/4;
		length -= len4 * 4;
	        for (i=0; i<len4; i++) {
		    col = buf[0];
		    if (!(col & COLOR_TRANSPARENT))
		        p[0] = CONV_TO_COL192(col);
		    col = buf[1];
		    if (!(col & COLOR_TRANSPARENT))
		        p[1] = CONV_TO_COL192(col);
		    col = buf[2];
		    if (!(col & COLOR_TRANSPARENT))
		        p[2] = CONV_TO_COL192(col);
		    col = buf[3];
		    if (!(col & COLOR_TRANSPARENT))
		        p[3] = CONV_TO_COL192(col);
		    buf += 4;
		    p += 4;
		}
	        for (i=0; i<length;i++,x++) {
		    col = *buf++;
		    if (!(col & COLOR_TRANSPARENT)) {
		        *p = CONV_TO_COL192(col);
		    }
		}
	    }
#endif
	    break;
	case DIR_WEST:
	    for (i=0; i<length;i++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		    if (col & COLOR_DITHER) {
			int dx = (x + (y>>2) + (y>>4))%5;
			int dy = y % 2;
			int dc = dither_map[dy][dx][col & 0xf]
				| (dither_map[dy][dx][(col>>4) & 0xf] << 2);
			*p = dc | CONV_TO_COL192(col & ~0xff);
		    } else {
		        *p = CONV_TO_COL192(col);
		    }
		}
		p -= s->wbytes;
	    }
	    break;
	case DIR_SOUTH:
	    if (*buf & COLOR_DITHER) {
		dx = (x + (y>>2) + (y>>4))%5;
		dy = y % 2;
		dodither = 1;
	    }
	    for (i=0; i<length;i++,x--) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		    if (dodither) {
			int dc = dither_map[dy][dx][col & 0xf]
				| (dither_map[dy][dx][(col>>4) & 0xf] << 2);
			*p = dc | CONV_TO_COL192(col & ~0xff);
		    } else {
		        *p = CONV_TO_COL192(col);
		    }
		}
	        p--;
		if (dodither) if (--dx < 0) dx = 4;
	    }
	    break;
	case DIR_EAST:
	    for (i=0; i<length;i++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		    if (col & COLOR_DITHER) {
			int dx = (x + (y>>2) + (y>>4))%5;
			int dy = y % 2;
			int dc = dither_map[dy][dx][col & 0xf]
				| (dither_map[dy][dx][(col>>4) & 0xf] << 2);
			*p = dc | CONV_TO_COL192(col & ~0xff);
		    } else {
		        *p = CONV_TO_COL192(col);
		    }
		}
		p += s->wbytes;
	    }
	    break;
	}
	return;
}

static void dec192_set_color(struct screen *s,int col) {
	lp_color = CONV_TO_COL192(col);
	lp_dodither = 0;
	if (col & COLOR_DITHER) 
		lp_dodither = 1;
	if ((CONV_FROM_COL192(lp_color) | COLOR_DITHER) == col) {
		lp_dodither = 0;
	}
	return;
}

static void dec192_draw_pixel(struct screen *s,int x, int y) {
	char *p;
	int mask;

//printf("dec192_draw_pixel (%d,%d) %02x %04x\n",x,y,lp_color,CONV_FROM_COL192(lp_color));
	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x;

	if (pen_color == COLOR_REVERSE) {
		*p ^= 0x3;
	} else if (lp_dodither) {
		int dx = (x + (y>>2) + (y>>4))%5;
		int dy = y % 2;
		int dc = dither_map[dy][dx][pen_color & 0xf]
			| (dither_map[dy][dx][(pen_color>>4) & 0xf] << 2);
		*p = dc | CONV_TO_COL192(pen_color & ~0xff);
	} else {
		*p = lp_color;
	}
      return;
}


static void dec192_draw_line_horizontal(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
	char *p;
	int wb;
	int i;
	int off_x,off_y;

	CLIP_HLINE(s,x1,y1,x2,y2);
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;

	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	wb = (x2 + 1 - x1);
	if (wb) {
	     if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	     } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + x1;

	    if (pen_color == COLOR_REVERSE) {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		for (i=0; i< wb; i++) {
		    *p++ ^= 0x3;
		}
#else
		mgl_memxor(p, 0x3, wb);
#endif
	    } else if (lp_dodither) {
		for (i=0; i< wb; i++,x1++) {
		   int dx = (x1 + (y1>>2) + (y1>>4))%5;
		   int dy = y1 % 2;
		   int dc = dither_map[dy][dx][pen_color & 0xf]
		       | (dither_map[dy][dx][(pen_color>>4) & 0xf] <<2);
	           *p++ = dc | CONV_TO_COL192(pen_color & ~0xff);
		}
	    } else {
		memset(p, lp_color, wb);
	    }
	}
	s->need_clipping++;
    return;
}

static void dec192_clear_screen(struct screen *s,struct draw_engine *self) {
	int x1,y1,x2,y2;
	int off_x,off_y;
	char *p,*pp;
	int wb;
	int i;
	int y;

	x1 = 0;
	x2 = s->width-1;
	y1 = 0;
	y2 = s->height-1;
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;
	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	wb = (x2 + 1 - x1);
	if (wb) {
	    if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	    } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + x1;
	    for (y=y1; y<=y2; y++) {
		if (pen_color == COLOR_REVERSE) {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		    pp = p;
		    for (i=0; i< wb; i++) {
			*pp++ ^= 0x3;
		    }
#else
		    mgl_memxor(p, 0x3, wb);
#endif
		} else if (lp_dodither) {
		    pp = p;
		    for (i=0; i< wb; i++,pp++) {
		        int dx = (x1 + i + (y>>2) + (y>>4))%5;
		        int dy = y % 2;
		        int dc= dither_map[dy][dx][pen_color & 0xf]
			     | (dither_map[dy][dx][(pen_color>>4) & 0xf] << 2);
			*pp = dc | CONV_TO_COL192(pen_color & ~0xff);
		    }
		} else {
		    memset(p, lp_color, wb);
	        }
		p += s->wbytes;
	    }
	}
	s->need_clipping++;
}

static void dec192_bitblt_copy(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		dp = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		dp = (char *)dst->bitmap;
	}
	if (src->type & ST_SUBSCREEN) {
		sp = ((struct screen *)(src->bitmap))->bitmap;
		src->wbytes = 
			((struct screen *)(src->bitmap))->wbytes;
	} else {
		sp = (char *)src->bitmap;
	}
	dp += dy * dst->wbytes + dx;
	sp += sy * src->wbytes + sx;
	len = xsize;

	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp += dst->wbytes;
		sp += src->wbytes;
	}
	return;
}

static void dec192_bitblt_scroll_forward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		p = (char *)dst->bitmap;
	}
	dp = p + dy * dst->wbytes + dx;
	sp = p + sy * dst->wbytes + sx;
	len = xsize;
	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp += dst->wbytes;
		sp += dst->wbytes;
	}
	return;
}

static void dec192_bitblt_scroll_backward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		p = (char *)dst->bitmap;
	}
	dy += ysize - 1;
	sy += ysize - 1;
	dp = p + dy * dst->wbytes + dx;
	sp = p + sy * dst->wbytes + sx;
	len = xsize;
	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp -= dst->wbytes;
		sp -= dst->wbytes;
	}
	return;
}
void dec192_init(int type) {
	dec192._create_subscreen = dec192_create_subscreen;
	dec192._free_screen = dec192_free_screen;
	dec192._put_pixel = dec192_put_pixel;
	dec192._get_pixel = dec192_get_pixel;

	dec192._set_color = dec192_set_color;
	dec192._draw_pixel = dec192_draw_pixel;

	//dec192._draw_line = dec192_draw_line;

	//dec192._draw_line_vertical = dec192_draw_line_vertical;
	dec192._draw_line_horizontal = dec192_draw_line_horizontal;

	dec192._clear_screen = dec192_clear_screen;

	dec192._get_pixstream = dec192_get_pixstream;
	dec192._put_pixstream = dec192_put_pixstream;
	//dec192._put_pixstream_rect = dec192_put_pixstream__rect;

	//dec192._bitblt = dec192_bitblt;
	dec192._bitblt_scroll_forward = dec192_bitblt_scroll_forward;
	dec192._bitblt_scroll_backward = dec192_bitblt_scroll_backward;
	dec192._bitblt_copy = dec192_bitblt_copy;
	//dec192._bitblt_reserved_mask = dec192_bitblt_reserved_mask;
	//dec192._bitblt_reserved_masktile = dec192_bitblt_reserver_masktile;
	//dec192._bitblt_gen = dec192_bitblt_gen;
	setup_draw_engine(&dec192,0);
	_create_memscreen[type] = dec192_create_memscreen;
}
