/* sqMgl.c -- support for the MGL2.
 *    written by  Koji Suzuki
 *
 *    modified by Toshi Fujita
 *	MONO SUPPORT
 *	EVENTS RECEPTION BUG FIX
 *	PEN SWAY FIX
 *	KEYBORAD BUFFER FIX
 *	CANNA IM SUPPORT (need Squeak Image Change)
 *
 * Original: sqlXWindow.c
 * Original-Author: Ian Piumarta <ian.piumarta@inria.fr>
 *
 * Last edited: Mon Jan 17 18:00:35 2000 by piumarta (Ian Piumarta) on pingu
 *
 * Code to support displays deeper than 8 bits contributed by: Kazuki YASUMATSU
 *	<kyasu@crl.fujixerox.co.jp> <Kazuki.Yasumatsu@fujixerox.co.jp>
 *
 * Support for cursor and keypad editing keys based on code contributed by:
 *	Stefan Matthias Aust <sma@kiel.netsurf.de>
 *
 * Support for intelligent visual class selection contributed by:
 *	Bill Cattey <wdc@MIT.EDU>
 *
 * Support European accented characters in selections contributed by:
 *	Bert Freudenberg <bert@isgnw.CS.Uni-Magdeburg.De>
 *
 * BUGS: this file is too long; it should be split into two: one for
 *		Unix and the other for X.
 *	 icon stuff should be removed (window manager's responsibility).
 *	 RCS stuff has been removed.
 */

#include "sq.h"

#include <stdio.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/utsname.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>

#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif

#include <mgl2.h>
#include <mglcol.h>
#define mgl_color_model (physical_screen->type & ST_KINDMASK)

/*** Variables -- Imported from Virtual Machine ***/
extern unsigned char *memory;
extern int interruptKeycode;
extern int interruptPending;
extern int interruptCheckCounter;
extern int savedWindowSize;

/*** Variables -- image and path names ***/
#define IMAGE_NAME_SIZE MAXPATHLEN

char imageName[MAXPATHLEN+1];		/* full path to image */
char shortImageName[MAXPATHLEN+1];	/* just the base name */
char vmPath[MAXPATHLEN+1];		/* full path to interpreter's directory */

#define DefaultHeapSize		20	/* megabytes */

int initialHeapSize;

int    initialArgc;
char **initialArgv;

#undef USE_ITIMER			/* severely confuses GNU's profil() */

#ifdef USE_ITIMER
unsigned int	lowResMSecs= 0;
#define	LOW_RES_TICK_MSECS	20	/* 1/50 second resolution */
#endif


#define SqueakWhite	0
#define SqueakBlack	

int		 asmAlign= 1;
int		 sleepWhenUnmapped= 0;
int		 noJitter= 0;
int		 withSpy= 0;
int		 noTitle= 0;
int		 fullScreen= 0;
struct timeval	 startUpTime;

/* maximum input polling frequency */
#define	MAXPOLLSPERSEC	33


#define MOD_SHIFT	1
#define MOD_CAPS	2
#define MOD_CTRL	4
#define MOD_META	8

/*** Functions ***/

#ifdef ioMSecs
# undef ioMSecs
#endif

int  HandleEvents(void);
void RecordFullPathForImageName(char *localImageName);
void SetUpTimers(void);
void usage(void);
void imageNotFound(char *imageName);
void ParseArguments(int argc, char **argv);
void segv(int ignored);

int strtobkm(char *str);

extern void aioPollForIO(int, int);	/* see sqUnixNetwork.c */
extern void auPollForIO(void);		/* see sqUnixSound.c */

time_t convertToSqueakTime(time_t);	/* unix epoch -> Squeak epoch */


//#define	EVENT_LOG
//#define DISP_DEBUG_MSG	//dispaly debug message
//#define AVOID_PEN_SWAY 3	//avoid pen swaying
//#define	PEN_SWAYING	//pen move chagne the root tapped position (AVOID_PEN_SWAY should be defined)
#define	ONE_BIT_DEPTH		//Mono surpport
#define	KEY_BUFFER_SIZE	256	//Accumulate Keys
//#define	IM_SURPPORT		//Need Multilingualized Squeak Image

/*** VM Home Directory Path ***/

int vmPathSize(void)
{
  return strlen(vmPath);
}

int vmPathGetLength(int sqVMPathIndex, int length)
{
  char *stVMPath= (char *)sqVMPathIndex;
  int count, i;

  count= strlen(vmPath);
  count= (length < count) ? length : count;

  /* copy the file name into the Squeak string */
  for (i= 0; i < count; i++)
    stVMPath[i]= vmPath[i];

  return count;
}

#ifdef ONE_BIT_DEPTH
int sq_color_map2[2];
#endif

int sq_color_map[256];
int sq_color_map16[16];
int sq_color_map4[4];
struct virtual_key *sq_vk;

#ifdef EVENT_LOG
FILE	*eventLogFp;
#endif

int SetUpWindow() {
	int i,j;
	int r,g,b;
	if (!open_graph()) {
		exit(1);
	}
	sq_vk = create_virtual_key3(0,0,SCREEN_WIDTH,SCREEN_HEIGHT
		,MK_V1,MK_V3,MK_V2);
		/* down,move,up*/
	vk_attach(NULL,sq_vk);

	mgl_set_keymode(MGL_SK_EXTRANSLATED | MGL_SK_EXMOUSE_EVENT);

#ifdef ONE_BIT_DEPTH
	sq_color_map2[0] = COLOR_WHITE;
	sq_color_map2[1] = COLOR_BLACK;
#endif
	sq_color_map4[ 0] = COLOR_WHITE;
	sq_color_map4[ 1] = COLOR_BLACK;
	sq_color_map4[ 2] = COLOR_LIGHTGRAY;
	sq_color_map4[ 3] = COLOR_DARKGRAY;

	sq_color_map16[ 0] = COLOR_WHITE;
	sq_color_map16[ 1] = COLOR_BLACK;
	sq_color_map16[ 2] = packMC( 0, 0,10); 
	sq_color_map16[ 3] = packMC( 0, 0, 7);  /*  1/2 */
	for (i=j=0; i<12; i++,j++) {
	    if (j==0 || j==15 || j==7 || j==10) {
		j++;
	    }
	    sq_color_map16[4+i] = packMC(0,0,j);
	}

	sq_color_map[ 0] = COLOR_WHITE;
	sq_color_map[ 1] = COLOR_BLACK;
	sq_color_map[ 2] = COLOR_LIGHTGRAY;
	sq_color_map[ 3] = packMC( 0, 7, 7);  /*  1/2 */

	sq_color_map[ 4] = packMC(MCH_RED,10,10);
	sq_color_map[ 5] = packMC(MCH_GREEN,10,10);
	sq_color_map[ 6] = packMC(MCH_BLUE,10,10);
	sq_color_map[ 7] = packMC(MCH_CYAN,10,10);
	sq_color_map[ 8] = packMC(MCH_YELLOW,10,10);
	sq_color_map[ 9] = packMC(MCH_MAGENTA,10,10);

	sq_color_map[10] = packMC(0, 1, 1);  /* 1/8 */
	sq_color_map[11] = packMC(0, 3, 3);  /* 2/8 */
	sq_color_map[12] = packMC(0, 5, 5);  /* 3/8 */
	sq_color_map[13] = packMC(0, 9, 9);  /* 5/8 */
	sq_color_map[14] = packMC(0,11,11);  /* 6/8 */
	sq_color_map[15] = packMC(0,13,13);  /* 7/8 */


	sq_color_map[16] = packMC(0, 0, 0);  /*  1/32 */
	sq_color_map[17] = packMC(0, 0, 0);  /*  2/32 */
	sq_color_map[18] = packMC(0, 1, 1);  /*  3/32 */

	sq_color_map[19] = packMC(0, 2, 2);  /*  5/32 */
	sq_color_map[20] = packMC(0, 2, 2);  /*  6/32 */
	sq_color_map[21] = packMC(0, 3, 3);  /*  7/32 */

	sq_color_map[22] = packMC(0, 4, 4);  /*  9/32 */
	sq_color_map[23] = packMC(0, 4, 4);  /* 10/32 */
	sq_color_map[24] = packMC(0, 5, 5);  /* 11/32 */

	sq_color_map[25] = packMC(0, 6, 6);  /* 13/32 */
	sq_color_map[26] = packMC(0, 6, 6);  /* 14/32 */
	sq_color_map[27] = packMC(0, 7, 7);  /* 15/32 */

	sq_color_map[28] = packMC(0, 8, 8);  /* 17/32 */
	sq_color_map[29] = packMC(0, 8, 8);  /* 18/32 */
	sq_color_map[30] = packMC(0, 9, 9);  /* 19/32 */

	sq_color_map[31] = packMC(0,10,10);  /* 21/32 */
	sq_color_map[32] = packMC(0,10,10);  /* 22/32 */
	sq_color_map[33] = packMC(0,11,11);  /* 23/32 */

	sq_color_map[34] = packMC(0,12,12);  /* 25/32 */
	sq_color_map[35] = packMC(0,12,12);  /* 26/32 */
	sq_color_map[36] = packMC(0,13,13);  /* 27/32 */

	sq_color_map[37] = packMC(0,14,14);  /* 29/32 */
	sq_color_map[38] = packMC(0,14,14);  /* 30/32 */
	sq_color_map[39] = packMC(0,15,15);  /* 31/32 */

	for (r=0; r < 6; r++)
	  for (g=0; g < 6; g++)
	    for (b=0; b < 6; b++)
	    {
		sq_color_map[40 + 36 * r + 6 * b + g] 
			= mc_from_rgb(packRGB(r*3,g*3,b*3));
	    }
}

int sq_button = 0;
int sq_modifier = 0;
#ifdef KEY_BUFFER_SIZE
int sq_keys[KEY_BUFFER_SIZE+1];
#else
int sq_keys[1];
#endif
int sq_keys_len;

#ifdef AVOID_PEN_SWAY
// for adjusting for pen device (not using vk_x and vk_y directly)
// Just after mouse down, if current mouse position is inside of
// +-AVOID_PEN_SWAY, mouse_move event is ignored.
static int sqMouseX = 0;
static int sqMouseY = 0;
static int sqMouseDownX = 0;
static int sqMouseDownY = 0;
#define cSqMouseStatusUp	0
#define cSqMouseStatusDown	1
#define cSqMouseStatusMoved	2
static int sqMouseStatus = cSqMouseStatusUp;
#endif

int HandleEvents(void)
{
  int c,b;
  
  /* quick check for asynchronous socket i/o */
  aioPollForIO(0, 0);
  auPollForIO();
  
#ifdef IM_SURPPORT
  c = get_key_im(0);
#else
  c = get_key(0);
#endif

#ifdef DISP_DEBUG_MSG
  if (sq_button) printf("(%x)", c);
#endif
  
  if (c < 0) return 0;

#ifdef DISP_DEBUG_MSG
printf("get_key():%X\n", c);
#endif

  if (c == MK_V1) { /* down */
	b = 4;
	if ((mgl_button_shift == 1) || (sq_modifier & MOD_CTRL)) {
		b = 2;
	} else if ((mgl_button_shift == 2) || (sq_modifier & MOD_META)) {
		b = 1;
	}
#if 1
	sq_button = b | (sq_modifier << 3);
#else
	sq_button |= b;
	sq_button = ( sq_button & 0x7 ) | (sq_modifier << 3);
#endif

#ifdef AVOID_PEN_SWAY
	sqMouseX = sqMouseDownX = vk_x;
	sqMouseY = sqMouseDownY = vk_y;
	sqMouseStatus = cSqMouseStatusDown;
#endif
#ifdef DISP_DEBUG_MSG
	printf("mouse pressed\n");
#endif

#ifdef EVENT_LOG
	fprintf(eventLogFp, "mouse pressed:%x\n", sq_button);
#endif
	return 1;
  } else if (c == MK_V2) { /* up */
	b = 4;
	if ((mgl_button_shift == 1) || (sq_modifier & MOD_CTRL)) {
		b = 2;
	} else if ((mgl_button_shift == 2) || (sq_modifier & MOD_META)) {
		b = 1;
	}
#if 1
	sq_button = sq_modifier << 3;
#else
	sq_button &= ~b;
	sq_button = ( sq_button & 0x7 ) | (sq_modifier << 3);
#endif

#ifdef AVOID_PEN_SWAY
	sqMouseStatus = cSqMouseStatusUp;
#endif
#ifdef DISP_DEBUG_MSG
	printf("mouse released\n");
#endif

#ifdef EVENT_LOG
	fprintf(eventLogFp, "mouse release:%x\n", sq_button);
#endif
	return 1;
  } else if (c == MK_V3) { /* move */
#ifdef AVOID_PEN_SWAY
	if (sqMouseStatus == cSqMouseStatusDown) {
		if ( ((sqMouseDownX + AVOID_PEN_SWAY) > vk_x) && ((sqMouseDownX - AVOID_PEN_SWAY) < vk_x)
		&&  ((sqMouseDownY + AVOID_PEN_SWAY) > vk_y) && ((sqMouseDownY - AVOID_PEN_SWAY) < vk_y) ) {
			// move is ignored
#ifdef PEN_SWAYING
			// change the root tap positon, it could be happier not used.
			sqMouseDownX = vk_x;
			sqMouseDownY = vk_y;
#endif
			return 0;
		}
	}
	sqMouseStatus = cSqMouseStatusMoved;
	sqMouseX = vk_x;
	sqMouseY = vk_y;
#endif
#ifdef DISP_DEBUG_MSG
	printf("mouse moved\n");
#endif

#ifdef EVENT_LOG
	fprintf(eventLogFp, "mosue moved:%d(%d,%d)\n", sq_button, vk_x, vk_y);
#endif
	return 1;
  }
  sq_modifier = 0;
  if (c & MGL_SKM_MASK) {
	//sq_modifier = 0;
	if (c & MGL_SKM_SHIFT) {
		sq_modifier |= MOD_SHIFT;
	}
	if (c & MGL_SKM_CAPS) {
		sq_modifier |= MOD_CAPS;
	}
	if (c & MGL_SKM_CTRL) {
		sq_modifier |= MOD_CTRL;
	}
	if (c & MGL_SKM_ALT) {
		sq_modifier |= MOD_META;
	}
	if (c & MGL_SKM_NOTICE)
		return 1;
  }
  c &= ~MGL_SKM_MASK;
  switch(c) {
  case MK_LEFT:		c = 28; break;
  case MK_UP:		c = 30; break;
  case MK_RIGHT:	c = 29; break;
  case MK_DOWN:		c = 31; break;
  case MK_INS:		c =  5; break;
  case MK_PAGE_UP:	c = 11; break;
  case MK_PAGE_DOWN:	c = 12; break;
  case MK_HOME:		c =  1; break;
  case MK_END:		c =  4; break;
  }
  if (c == interruptKeycode) {
	interruptPending = true;
	interruptCheckCounter = 0;
  } else if (sq_keys_len < KEY_BUFFER_SIZE) {
      sq_keys[sq_keys_len++] = (sq_modifier << 8) | (c & 0xff);
  }
#ifdef EVENT_LOG
	fprintf(eventLogFp, "keyEvent:");
	{ int i;
	  for(i=0; i<sq_keys_len; i++) {
		fprintf(eventLogFp, "%x\n", sq_keys[i]);
	  }
	}
#endif
  return 1;
}



/*** I/O Primitives ***/


int ioFormPrint(int bitsAddr, int width, int height, int depth,
		double hScale, double vScale, int landscapeFlag)
{

  fprintf(stderr,
	  "Sorry, a headless VM cannot print Forms.  If you\n"
	  "*really* need this then let me know, since there\n"
	  "is a (rather painful to implement) solution.\n");
#ifdef DISP_DEBUG_MSG
  printf("ioFormPrint()\n");
#endif
  return false;

}


int ioBeep(void)
{
#ifdef DISP_DEBUG_MSG
  printf("ioBeep()\n");
#endif
  return 0;
}


int ioGetButtonState(void)
{
  //ioProcessEvents();  /* process all pending events */
  HandleEvents();
  
#ifdef DISP_DEBUG_MSG
  if (sq_button)	printf("sq_button=%x\n", sq_button);
  else			printf(".");
#endif

  return sq_button;
}

int ioGetKeystroke(void)
{
  int keystate;

  //ioProcessEvents();  /* process all pending events */
  HandleEvents();
  
  if (sq_keys_len == 0) 
  	return -1;  /* keystroke buffer is empty */
#ifdef KEY_BUFFER_SIZE
  sq_keys_len--;
  keystate = sq_keys[0];
  {
     int i;
     for(i=0; i<sq_keys_len; i++)
     	sq_keys[i] = sq_keys[i+1];
  }
#else
  sq_keys_len = 0;
  keystate = sq_keys[0];
#endif

#ifdef IM_SURPPORT
  if (keystate & 0x80)	keystate &= 0x00ff;
#endif
  sq_button= ((keystate >> 5) & 0xF8) | (sq_button & 0x7);
#ifdef EVENT_LOG
	fprintf(eventLogFp, "KEY:%x(%x), Button:%x\n", keystate, sq_modifier,sq_button);
#endif

  return keystate;
}


int ioLowResMSecs(void)
{
#ifdef USE_ITIMER
  return lowResMSecs;
#else
  return ioMSecs();
#endif
}


int ioMSecs(void)
{
  struct timeval now;
  gettimeofday(&now, 0);
  if ((now.tv_usec -= startUpTime.tv_usec) < 0) {
    now.tv_usec += 1000000;
    now.tv_sec -= 1;
  }
  now.tv_sec -= startUpTime.tv_sec;
  return (now.tv_usec / 1000 + now.tv_sec * 1000);
}

int ioMicroMSecs(void)
{
  /* return the highest available resolution of the millisecond clock */
  return ioMSecs();	/* this already to the nearest millisecond */
}

int ioRelinquishProcessorForMicroseconds(int microSeconds)
{
  /* sleep in select() for immediate response to socket i/o */
  aioPollForIO(microSeconds, 0);
  return microSeconds;
}

int ioMousePoint(void)
{
  ioProcessEvents();  /* process all pending events */
  /* x is high 16 bits; y is low 16 bits */
#ifdef AVOID_PEN_SWAY
#ifdef DISP_DEBUG_MSG
  if (sq_button) printf("X:%d,Y:%d\n", sqMouseX, sqMouseY);
#endif
  return (sqMouseX << 16) | sqMouseY;

#else
#ifdef DISP_DEBUG_MSG
  if (sq_button) printf("X:%d,Y:%d\n", vk_x, vk_y);
#endif
  return (vk_x << 16) | vk_y;
#endif
}

int ioPeekKeystroke(void)
{
  int keystate;

  //ioProcessEvents();  /* process all pending events */
  HandleEvents();
  
  if (sq_keys_len == 0) 
  	return -1;  /* keystroke buffer is empty */
#ifdef EVENT_LOG
  fprintf(eventLogFp, "ioPeekKeystroke:%x\n", sq_keys[0]);
#endif
  keystate = sq_keys[0];
  return keystate;
}

/* this should be rewritten to use SIGIO and/or the interval timers */
int ioProcessEvents(void)
{
#if 0
  while(HandleEvents()) ;
#else
  static unsigned long nextPollTick= 0;

  if ((unsigned long)ioLowResMSecs() > nextPollTick)
    {
      /* time to process events! */
      while (HandleEvents())
	{
	  /* process all pending events */
	}
    /* wait a while before trying again */
      nextPollTick= ioLowResMSecs() + (1000 / MAXPOLLSPERSEC);
    }
#endif
  return 0;
}

/* returns the size of the Squeak window */
int ioScreenSize(void)
{
#ifdef DISP_DEBUG_MSG
  printf("ioScreenSize() h:%d,w:%d\n", SCREEN_HEIGHT, SCREEN_WIDTH);
#endif
  /* w is high 16 bits; h is low 16 bits */
  return (SCREEN_WIDTH << 16) | (SCREEN_HEIGHT & 0xFFFF);
}

time_t convertToSqueakTime(time_t unixTime)
{
#ifdef HAVE_TM_GMTOFF
  unixTime+= localtime(&unixTime)->tm_gmtoff;
#else
# ifdef HAVE_TIMEZONE
  unixTime+= ((daylight) * 60*60) - timezone;
# else
#  error: cannot determine timezone correction
# endif
#endif
  /* Squeak epoch is Jan 1, 1901.  Unix epoch is Jan 1, 1970: 17 leap years
     and 52 non-leap years later than Squeak. */
  return unixTime + ((52*365UL + 17*366UL) * 24*60*60UL);
}

/* returns the local wall clock time */
int ioSeconds(void)
{
  return convertToSqueakTime(time(0));
}



int ioSetCursorWithMask(int cursorBitsIndex, int cursorMaskIndex,
			int offsetX, int offsetY)
{
#ifdef DISP_DEBUG_MSG
  printf("ioSetCursorWithMask()\n");
#endif
  return 0;
}


int ioSetCursor(int cursorBitsIndex, int offsetX, int offsetY)
{
  /* Deprecated: forward to new version with explicit mask. */
  ioSetCursorWithMask(cursorBitsIndex, null, offsetX, offsetY);
  return 0;
}


int ioSetFullScreen(int fullScreen)
{
#ifdef DISP_DEBUG_MSG
  printf("ioSetFullScreen %d\n",fullScreen);
#endif
  return 0;
}



int ioForceDisplayUpdate(void)
{
#ifdef DISP_DEBUG_MSG
  printf("ioForceDisplayUpdate\n");
#endif

#if 1
  //ioProcessEvents();
  HandleEvents();
#endif
  return 0;
}


int ioShowDisplay(int dispBitsIndex, int width, int height, int depth,
		  int affectedL, int affectedR, int affectedT, int affectedB)
{
 int x,y;
 unsigned char *p,*pp;
 int len;
 int buf[1024];

#if 0
fprintf(stderr,"ioShowDisplay %d %d %d %d %d %d %d\n",width,height,depth,
		affectedL,affectedR,affectedT,affectedB);
#endif

#ifdef DISP_DEBUG_MSG
  printf("ioShowDisplay()\n");
#endif

  if((affectedR <= affectedL) || (affectedT >= affectedB))
     return 1;

#ifdef DISP_DEBUG_MSG
  printf("ioShowDisplay():2\n");
#endif

  if (depth == 8) {
    if (affectedR > width) affectedR= width;
    if (affectedB > height) affectedB= height;
    affectedL = affectedL & ~3;
    affectedR = (affectedR +3)& ~3;

    len = affectedR - affectedL;
    pp = (unsigned char *)dispBitsIndex + affectedT*width + affectedL;

//printf(" ----- %d %d\n",affectedL,affectedR);
  for (y=affectedT; y < affectedB; y++) {
	p = pp;
#ifdef WORDS_BIGENDIAN
	for (x=0; x<len; x+=4,p+=4) {
		buf[x  ] = sq_color_map[p[0]];
		buf[x+1] = sq_color_map[p[1]];
		buf[x+2] = sq_color_map[p[2]];
		buf[x+3] = sq_color_map[p[3]];
	}
#else
	for (x=0; x<len; x+=4,p+=4) {
		buf[x  ] = sq_color_map[p[3]];
		buf[x+1] = sq_color_map[p[2]];
		buf[x+2] = sq_color_map[p[1]];
		buf[x+3] = sq_color_map[p[0]];
	}
#endif
	put_pixstream(affectedL,y,buf,len,DIR_NORTH);
	pp += width;
     }
  } else if (depth == 4) {
    if (affectedR > width) affectedR= width;
    if (affectedB > height) affectedB= height;
    affectedL = affectedL & ~7;
    affectedR = (affectedR +7)& ~7;

    len = (affectedR - affectedL);
    pp = (unsigned char *)dispBitsIndex + (affectedT*width + affectedL)/2;

//printf(" ----- %d %d\n",affectedL,affectedR);
  for (y=affectedT; y < affectedB; y++) {
	p = pp;
#ifdef WORDS_BIGENDIAN
	for (x=0; x<len; x+=8,p+=4) {
		buf[x+0] = sq_color_map16[p[0]&0xf];
		buf[x+1] = sq_color_map16[p[0]>>4];
		buf[x+2] = sq_color_map16[p[1]&0xf];
		buf[x+3] = sq_color_map16[p[1]>>4];
		buf[x+4] = sq_color_map16[p[2]&0xf];
		buf[x+5] = sq_color_map16[p[2]>>4];
		buf[x+6] = sq_color_map16[p[3]&0xf];
		buf[x+7] = sq_color_map16[p[3]>>4];
	}
#else
	for (x=0; x<len; x+=8,p+=4) {
		buf[x+0] = sq_color_map16[p[3]>>4];
		buf[x+1] = sq_color_map16[p[3]&0xf];
		buf[x+2] = sq_color_map16[p[2]>>4];
		buf[x+3] = sq_color_map16[p[2]&0xf];
		buf[x+4] = sq_color_map16[p[1]>>4];
		buf[x+5] = sq_color_map16[p[1]&0xf];
		buf[x+6] = sq_color_map16[p[0]>>4];
		buf[x+7] = sq_color_map16[p[0]&0xf];
	}
#endif
	put_pixstream(affectedL,y,buf,len,DIR_NORTH);
	pp += width/2;
     }
  } else if (depth == 2) {
    if (affectedR > width) affectedR= width;
    if (affectedB > height) affectedB= height;
    affectedL = affectedL & ~15;
    affectedR = (affectedR +15)& ~15;

    len = (affectedR - affectedL);
    pp = (unsigned char *)dispBitsIndex + (affectedT*width + affectedL)/4;

//printf(" ----- %d %d\n",affectedL,affectedR);
  for (y=affectedT; y < affectedB; y++) {
	p = pp;
#ifdef WORDS_BIGENDIAN
	for (x=0; x<len; x+=16,p+=4) {
		buf[x+0] =  sq_color_map4[p[3]&0x3];
		buf[x+1] =  sq_color_map4[(p[3]>>2)&0x3];
		buf[x+2] =  sq_color_map4[(p[3]>>4)&0x3];
		buf[x+3] =  sq_color_map4[p[3]>>6];
		buf[x+4] =  sq_color_map4[p[2]&0x3];
		buf[x+5] =  sq_color_map4[(p[2]>>2)&0x3];
		buf[x+6] =  sq_color_map4[(p[2]>>4)&0x3];
		buf[x+7] =  sq_color_map4[p[2]>>6];
		buf[x+8] = sq_color_map4[p[1]&0x3];
		buf[x+9] = sq_color_map4[(p[1]>>2)&0x3];
		buf[x+10] =  sq_color_map4[(p[1]>>4)&0x3];
		buf[x+11] =  sq_color_map4[p[1]>>6];
		buf[x+12] = sq_color_map4[p[0]&0x3];
		buf[x+13] = sq_color_map4[(p[0]>>2)&0x3];
		buf[x+14] = sq_color_map4[(p[0]>>4)&0x3];
		buf[x+15] = sq_color_map4[p[0]>>6];
	}
#else
	for (x=0; x<len; x+=16,p+=4) {
		buf[x+0] =  sq_color_map4[p[3]>>6];
		buf[x+1] =  sq_color_map4[(p[3]>>4)&0x3];
		buf[x+2] =  sq_color_map4[(p[3]>>2)&0x3];
		buf[x+3] =  sq_color_map4[p[3]&0x3];
		buf[x+4] =  sq_color_map4[p[2]>>6];
		buf[x+5] =  sq_color_map4[(p[2]>>4)&0x3];
		buf[x+6] =  sq_color_map4[(p[2]>>2)&0x3];
		buf[x+7] =  sq_color_map4[p[2]&0x3];
		buf[x+8] =  sq_color_map4[p[1]>>6];
		buf[x+9] =  sq_color_map4[(p[1]>>4)&0x3];
		buf[x+10] = sq_color_map4[(p[1]>>2)&0x3];
		buf[x+11] = sq_color_map4[p[1]&0x3];
		buf[x+12] = sq_color_map4[p[0]>>6];
		buf[x+13] = sq_color_map4[(p[0]>>4)&0x3];
		buf[x+14] = sq_color_map4[(p[0]>>2)&0x3];
		buf[x+15] = sq_color_map4[p[0]&0x3];
	}
#endif
	put_pixstream(affectedL,y,buf,len,DIR_NORTH);
	pp += width/4;
     }
  }
#ifdef ONE_BIT_DEPTH
   else if (depth == 1) {
    if (affectedR > width) affectedR= width;
    if (affectedB > height) affectedB= height;
    affectedL = affectedL & ~31;
    affectedR = (affectedR +31)& ~31;

    len = (affectedR - affectedL);
    pp = (unsigned char *)dispBitsIndex + (affectedT*width + affectedL)/8;

//printf(" ----- %d %d\n",affectedL,affectedR);
  for (y=affectedT; y < affectedB; y++) {
	p = pp;
#ifdef WORDS_BIGENDIAN
	for (x=0; x<len; x+=32,p+=4) {
		buf[x+0] =  sq_color_map2[p[3]&0x1];
		buf[x+1] =  sq_color_map2[(p[3]>>1)&0x1];
		buf[x+2] =  sq_color_map2[(p[3]>>2)&0x1];
		buf[x+3] =  sq_color_map2[(p[3]>>3)&0x1];
		buf[x+4] =  sq_color_map2[(p[3]>>4)&0x1];
		buf[x+5] =  sq_color_map2[(p[3]>>5)&0x1];
		buf[x+6] =  sq_color_map2[(p[3]>>6)&0x1];
		buf[x+7] =  sq_color_map2[(p[3]>>7)&0x1];
		buf[x+8] =  sq_color_map2[p[2]&0x1];
		buf[x+9] =  sq_color_map2[(p[2]>>1)&0x1];
		buf[x+10] =  sq_color_map2[(p[2]>>2)&0x1];
		buf[x+11] =  sq_color_map2[(p[2]>>3)&0x1];
		buf[x+12] =  sq_color_map2[(p[2]>>4)&0x1];
		buf[x+13] =  sq_color_map2[(p[2]>>5)&0x1];
		buf[x+14] =  sq_color_map2[(p[2]>>6)&0x1];
		buf[x+15] =  sq_color_map2[(p[2]>>7)&0x1];
		buf[x+16] =  sq_color_map2[p[1]&0x1];
		buf[x+17] =  sq_color_map2[(p[1]>>1)&0x1];
		buf[x+18] =  sq_color_map2[(p[1]>>2)&0x1];
		buf[x+19] =  sq_color_map2[(p[1]>>3)&0x1];
		buf[x+20] =  sq_color_map2[(p[1]>>4)&0x1];
		buf[x+21] =  sq_color_map2[(p[1]>>5)&0x1];
		buf[x+22] =  sq_color_map2[(p[1]>>6)&0x1];
		buf[x+23] =  sq_color_map2[(p[1]>>7)&0x1];
		buf[x+24] =  sq_color_map2[p[0]&0x1];
		buf[x+25] =  sq_color_map2[(p[0]>>1)&0x1];
		buf[x+26] =  sq_color_map2[(p[0]>>2)&0x1];
		buf[x+27] =  sq_color_map2[(p[0]>>3)&0x1];
		buf[x+28] =  sq_color_map2[(p[0]>>4)&0x1];
		buf[x+29] =  sq_color_map2[(p[0]>>5)&0x1];
		buf[x+30] =  sq_color_map2[(p[0]>>6)&0x1];
		buf[x+31] =  sq_color_map2[(p[0]>>7)&0x1];
	}
#else
	for (x=0; x<len; x+=32,p+=4) {
		buf[x+0] =  sq_color_map2[(p[3]>>7)&0x1];
		buf[x+1] =  sq_color_map2[(p[3]>>6)&0x1];
		buf[x+2] =  sq_color_map2[(p[3]>>5)&0x1];
		buf[x+3] =  sq_color_map2[(p[3]>>4)&0x1];
		buf[x+4] =  sq_color_map2[(p[3]>>3)&0x1];
		buf[x+5] =  sq_color_map2[(p[3]>>2)&0x1];
		buf[x+6] =  sq_color_map2[(p[3]>>1)&0x1];
		buf[x+7] =  sq_color_map2[p[3]&0x1];
		buf[x+8] =  sq_color_map2[(p[2]>>7)&0x1];
		buf[x+9] =  sq_color_map2[(p[2]>>6)&0x1];
		buf[x+10] =  sq_color_map2[(p[2]>>5)&0x1];
		buf[x+11] =  sq_color_map2[(p[2]>>4)&0x1];
		buf[x+12] =  sq_color_map2[(p[2]>>3)&0x1];
		buf[x+13] =  sq_color_map2[(p[2]>>2)&0x1];
		buf[x+14] =  sq_color_map2[(p[2]>>1)&0x1];
		buf[x+15] =  sq_color_map2[p[2]&0x1];
		buf[x+16] =  sq_color_map2[(p[1]>>7)&0x1];
		buf[x+17] =  sq_color_map2[(p[1]>>6)&0x1];
		buf[x+18] =  sq_color_map2[(p[1]>>5)&0x1];
		buf[x+19] =  sq_color_map2[(p[1]>>4)&0x1];
		buf[x+20] =  sq_color_map2[(p[1]>>3)&0x1];
		buf[x+21] =  sq_color_map2[(p[1]>>2)&0x1];
		buf[x+22] =  sq_color_map2[(p[1]>>1)&0x1];
		buf[x+23] =  sq_color_map2[p[1]&0x1];
		buf[x+24] =  sq_color_map2[(p[0]>>7)&0x1];
		buf[x+25] =  sq_color_map2[(p[0]>>6)&0x1];
		buf[x+26] =  sq_color_map2[(p[0]>>5)&0x1];
		buf[x+27] =  sq_color_map2[(p[0]>>4)&0x1];
		buf[x+28] =  sq_color_map2[(p[0]>>3)&0x1];
		buf[x+29] =  sq_color_map2[(p[0]>>2)&0x1];
		buf[x+30] =  sq_color_map2[(p[0]>>1)&0x1];
		buf[x+31] =  sq_color_map2[p[0]&0x1];
	}
#endif
	put_pixstream(affectedL,y,buf,len,DIR_NORTH);
	pp += width/8;
     }
  }
#endif
  //refresh();
  return 0;
}


int ioHasDisplayDepth(int i)
{
#ifdef ONE_BIT_DEPTH
  if (i == 1)	return true;
#endif
  if (mgl_color_model == STK_GENERIC_4COLOR) {
	return (i == 2)?true:false;
  } else if (mgl_color_model == STK_GENERIC_16COLOR) {
#if 1
	if (i == 2) return true;
#endif
	return (i==4)?true:false;
  }
#if 1
  if (i == 2) return true;
  if (i == 4) return true;
#endif
  return (i == 8)?true:false;
}

int ioSetDisplayMode(int width, int height, int depth, int fullscreenFlag)
{
  fprintf(stderr, "ioSetDisplayMode(%d, %d, %d, %d)\n",
	  width, height, depth, fullscreenFlag);
  
  return 0;
}




/*** Image File Naming ***/

void RecordFullPathForImageName(char *localImageName)
{
  int i;
  /* get canonical path to image */
  if (!realpath(localImageName, vmPath))
    imageNotFound(localImageName);
  strcpy(imageName, vmPath);
  /* truncate vmPath to dirname */
  for (i= strlen(vmPath); i >= 0; i--)
    if ('/' == vmPath[i])
      {
	vmPath[i+1]= '\0';
	return;
      }
  /* this might just work in an emergency... */
  strcpy(imageName, vmPath);
}

int imageNameSize(void)
{
  return strlen(imageName);
}

int imageNameGetLength(int sqImageNameIndex, int length)
{
  char *sqImageName= (char *)sqImageNameIndex;
  int count, i;

  count= strlen(imageName);
  count= (length < count) ? length : count;

  /* copy the file name into the Squeak string */
  for (i= 0; i < count; i++)
    sqImageName[i]= imageName[i];

  return count;
}

int imageNamePutLength(int sqImageNameIndex, int length)
{
  char *sqImageName= (char *)sqImageNameIndex;
  int count, i;

  count= (IMAGE_NAME_SIZE < length) ? IMAGE_NAME_SIZE : length;

  /* copy the file name into a null-terminated C string */
  for (i= 0; i < count; i++) imageName[i]= sqImageName[i];
  imageName[count]= 0;


  return count;
}

/*** Timing support ***/

#ifdef USE_ITIMER
void sigalrm(int signum)
{
  lowResMSecs+= LOW_RES_TICK_MSECS;
}
#endif

void SetUpTimers(void)
{
  /* set up the micro/millisecond clock */
  gettimeofday(&startUpTime, 0);
#ifdef USE_ITIMER
  /* set up the low-res (50th second) millisecond clock */
  /* WARNING: all system calls must check for EINTR!!! */
  {
    struct sigaction sa;
    sigset_t ss1, ss2;
    sigemptyset(&ss1);
    sigprocmask(SIG_BLOCK, &ss1, &ss2);
    sa.sa_handler= sigalrm;
    sa.sa_mask= ss2;
#ifdef SA_RESTART
    sa.sa_flags= SA_RESTART;
#else
    sa.sa_flags= 0;	/* assume that we have default BSD behaviour */
#endif
#ifdef __linux__
    sa.sa_restorer= 0;
#endif
    sigaction(SIGALRM, &sa, 0);
  }
  {
    struct itimerval iv;
    iv.it_interval.tv_sec= 0;
    iv.it_interval.tv_usec= LOW_RES_TICK_MSECS * 1000;
    iv.it_value.tv_sec= 0;
    iv.it_value.tv_usec= LOW_RES_TICK_MSECS;
    setitimer(ITIMER_REAL, &iv, 0);
  }
#endif
}

/*** X selection handling ***/


int clipboardSize(void)
{
  return 0;
}


/* claim ownership of the X selection, providing the given string to requestors */
int clipboardWriteFromAt(int count, int byteArrayIndex, int startIndex)
{
  return 0;
}

/* transfer the X selection into the given byte array; optimise local requests */
int clipboardReadIntoAt(int count, int byteArrayIndex, int startIndex)
{
  return 0;
}

/*** Profiling ***/

int clearProfile(void) { return 0; }
int dumpProfile(void) { return 0; }
int startProfiling(void) { return 0; }
int stopProfiling(void) { return 0; }

/*** Access to system attributes and command-line arguments ***/

static char *getSpecialAttribute(int id)
{
  static char buf[256];
  struct utsname info;
  uname(&info);
  switch (id)
    {
    case -1:	/* Squeak PID */
      sprintf(buf, "%d", (int)getpid());
     break;
    case 1001:	/* OS name */
      strcpy(buf, info.sysname);
      break;
    case 1002:	/* OS version */
      strcpy(buf, info.release);
      break;
    case 1003:	/* processor type */
      strcpy(buf, info.machine);
      break;
    default:
      buf[0]= '\0';
    }
  return buf;
}

int attributeSize(int id)
{
  if ((id < 0) || (id > 1000))
    return strlen(getSpecialAttribute(id));
  if (id < initialArgc)
    return strlen(initialArgv[id]);
  return 0;
}

int getAttributeIntoLength(int id, int byteArrayIndex, int length)
{
  char *attrIndex= (char *)byteArrayIndex;
  char *arg;
  if ((id < 0) || (id > 1000))
    arg= getSpecialAttribute(id);
  else if (id < initialArgc)
    arg= initialArgv[id];
  else
    return 0;
  while (length--)
    *attrIndex++= *arg++;
  return 0;
}

/*** Command line ***/

static char *progName;

void usage()
{
  printf("Usage: %s [<options>] [<imageName>]\n\n", progName);
  printf("<options> are:\n");
  printf("  -align <n>           align functions at n-bytes (jit)\n");
#ifdef CCACHE
  printf("  -ccache <size>       set context cache size (default: 16)\n");
#endif
  printf("  -display <dpy>       display on <dpy> (default: $DISPLAY)\n");
  printf("  -fullscreen          occupy the entire screen\n");
  printf("  -help                print this help message, then exit\n");
  printf("  -lazy                go to sleep when main window unmapped\n");
  printf("  -memory <size>[mk]   set initial memory size (default: %dm)\n",
	 DefaultHeapSize);
  printf("  -nojit               disable the dynamic (runtime) translator\n");
  printf("  -notitle             disable the Squeak window title bar\n");
  printf("  -spy                 enable the system spy (if using the jit)\n");
  printf("  -version             print version information, then exit\n");
  printf("  -xasync              don't serialize display updates\n");
  printf("\nNotes:\n");
  printf("  <imageName> defaults to 'Squeak"SQVERSION".image'.\n");
  printf("  Using 'unix:0' for <dpy> may improve local display performance.\n");
  printf("  -xshm only works when Squeak is running on the X server host.\n");
  exit(1);
}

void imageNotFound(char *imageName)
{
  /* image file is not found */
  fprintf(stderr, "Could not open the Squeak image file '%s'.\n\n", imageName);
  fprintf(stderr, "There are two ways to open a Squeak image file.  You can:\n");
  fprintf(stderr, "  1. Put the name of the image file on the command line\n");
  fprintf(stderr, "     when you run Squeak (use '-help' for more information).\n");
  fprintf(stderr, "  2. Save an image (from inside Squeak) with the name 'clone'\n");
  fprintf(stderr, "     and run Squeak from the directory containing it.\n");
  exit(1);
}

int strtobkm(char *str) {
  char *suffix;
  int value= strtol(str, &suffix, 10);
  switch (*suffix) {
  case 'k': case 'K':
    value*= 1024;
    break;
  case 'm': case 'M':
    value*= 1024*1024;
    break;
  }
  return value;
}

void ParseArguments(int argc, char **argv)
{
  char *ev;

  initialArgc= 3;                  /* VM name + image name + document name */
  initialArgv= malloc((argc+3) * sizeof(char*));       /* +3 is worst case */

  if (initialArgv == 0)
    {
      fprintf(stderr, "memory allocation failed\n");
      exit(1);
    }
  initialArgv[0]= argv[0];
  initialArgv[1]= shortImageName;
  initialArgv[2]= "";              /* document */

  /* defaults */

  if ((ev= getenv("SQUEAK_ALIGN")))
    asmAlign= atoi(ev);
  else
    asmAlign= 1;
  if (asmAlign < 1) asmAlign= 1;
  if (asmAlign > 16) asmAlign= 16;

  if ((ev= getenv("SQUEAK_IMAGE")))
    strcpy(shortImageName, ev);
  else
    strcpy(shortImageName, "Squeak"SQVERSION".image");

  if ((ev= getenv("SQUEAK_MEMORY")))
    initialHeapSize= strtobkm(ev);
  else
    initialHeapSize= DefaultHeapSize * 1024 * 1024;

  if (getenv("SQUEAK_LAZY"))
    sleepWhenUnmapped= 1;
  else
    sleepWhenUnmapped= 0;

  if (getenv("SQUEAK_NOJIT"))
    noJitter= 1;
  else
    noJitter= 0;

  if (getenv("SQUEAK_SPY"))
    withSpy= 1;
  else
    withSpy= 0;

  if (getenv("SQUEAK_NOTITLE"))
    noTitle= 1;
  else
    noTitle= 0;

  if (getenv("SQUEAK_FULLSCREEN"))
    fullScreen= 1;
  else
    fullScreen= 0;

#ifdef CCACHE
  if (ev= getenv("SQUEAK_CCACHE"))
    stackCacheEntries= atoi(ev);
  else
    stackCacheEntries= 0;
  contextCacheEntries= stackCacheEntries;
#endif

  argc--;
  argv++;	/* skip VM name */

  while (argc && **argv == '-')	/* more options to parse */
    {
      if (!strcmp(*argv, "-help"))
	usage();

      if (!strcmp(*argv, "-align"))
	{
	  argc--;	/* skip -align */
	  argv++;
	  if (!argc) usage();
	  asmAlign= atoi(*argv++);
	  argc--;
	  if (asmAlign < 1) asmAlign= 1;
	  if (asmAlign > 16) asmAlign= 16;
	  continue;
	}


      if (!strcmp(*argv, "-memory"))
	{
	  argc--;	/* skip -memory */
	  argv++;
	  if (!argc) usage();
	  initialHeapSize= strtobkm(*argv++);
	  argc--;
	  continue;
	}

#ifdef CCACHE
      if (!strcmp(*argv, "-ccache"))
	{
	  argc--;	/* skip -ccache */
	  argv++;
	  if (!argc) usage();
	  stackCacheEntries= contextCacheEntries= atoi(*argv++);
	  argc--;
	  if (contextCacheEntries < 2)
	    {
	      fprintf(stderr, "context cache size must be at least 2\n");
	      exit(1);
	    }
	  continue;
	}
#endif

      if (!strcmp(*argv, "-lazy"))
	{
	  argc--;	/* skip -lazy */
	  argv++;
	  sleepWhenUnmapped= 1;
	  continue;
	}

      if (!strcmp(*argv, "-notitle"))
	{
	  argc--;	/* skip -notitle */
	  argv++;
	  noTitle= 1;
	  continue;
	}

      if (!strcmp(*argv, "-nojit"))
	{
	  argc--;	/* skip -nojit */
	  argv++;
	  noJitter= 1;
	  continue;
	}

      if (!strcmp(*argv, "-spy"))
	{
	  argc--;	/* skip -nojit */
	  argv++;
	  withSpy= 1;
	  continue;
	}

      if (!strcmp(*argv, "-fullscreen"))
	{
	  argc--;	/* skip -fullscreen */
	  argv++;
	  fullScreen= 1;
	  continue;
	}

      if (!strcmp(*argv, "-version"))
	{
	  extern int vm_serial;
	  extern char *vm_date, *vm_version, *cc_version, *ux_version;
	  fprintf(stderr, "%s %s #%d %s%s %s\n%s\n",
		  VM_HOST, vm_version, vm_serial,
		  " ",
		  vm_date, cc_version, ux_version);
	  exit(0);
	}

      /* first unrecognized option - keep for Squeak */
      /* but print a warning because we got no image file yet */
      fprintf(stderr, "Warning: no imagefile specified "
 	      "but additional parameters found\n");
      break;
    }

  if ((argc != 0) && (**argv != '-'))
    {
      strcpy(shortImageName, *argv++);
      --argc;
    }

  if ((argc != 0) && (**argv != '-'))
    {
      initialArgv[2]= *argv++;
      --argc;
    }

  if ((argc != 0) && (**argv != '-'))
    {
      initialArgv[2]= *argv++;
      --argc;
    }

  if (0 == strstr(shortImageName, ".image"))
    strcat(shortImageName, ".image");
}


/*** Segmentation fault handler ***/

#include <signal.h>

void segv(int ignore)
{
  error("Segmentation fault");
}

#ifdef __alpha__
/* headers for setsysinfo (see below) */
# include <sys/sysinfo.h>
# include <sys/proc.h>
#endif

int main(int argc, char **argv)
{
  signal(SIGSEGV, segv);

  /* initialisation */

#ifdef __alpha__
  /* disable printing of unaligned access exceptions */
  {
    int buf[2]= { SSIN_UACPROC, UAC_NOPRINT };
    if (setsysinfo(SSI_NVPAIRS, buf, 1, 0, 0, 0) < 0) {
      perror("setsysinfo(UAC_NOPRINT)");
    }
  }
#endif

  progName= argv[0];
  ParseArguments(argc, argv);
  SetUpTimers();


  sqFileInit();
  joystickInit();

#ifdef HAVE_TZSET
  tzset();	/* should _not_ be necessary! */
#endif

  RecordFullPathForImageName(shortImageName);

  /* check the interpreter's size assumptions for basic data types */
  if (sizeof(int) != 4)    error("This C compiler's integers are not 32 bits.");
  if (sizeof(double) != 8) error("This C compiler's floats are not 64 bits.");
  if (sizeof(time_t) != 4) error("This C compiler's time_t's are not 32 bits.");

  /* read the image file and allocate memory for Squeak heap */
  {
    FILE *f= fopen(imageName, "r");

    if (f == NULL)
      imageNotFound(imageName);
    readImageFromFileHeapSize(f, initialHeapSize);
    fclose(f);
  }

  SetUpWindow();

#ifdef EVENT_LOG
	eventLogFp = fopen("sqMgl.log", "w");
#endif

#if defined(HAVE_LIBDL) && !defined(J_PROFILE)
  if (!noJitter)
    {
      extern void *loadExternalFunctionFromModule(char *, char *);
      /* first try to find an internal dynamic compiler... */
      void *comp= loadExternalFunctionFromModule("j_interpret", 0);
      /* ...and if that fails... */
      if (!comp)
	/* ...try to find an external one */
	comp= loadExternalFunctionFromModule("j_interpret", "SqueakCompiler");
      if (comp)
	{
	  ((void(*)(void))comp)();
	  return 0;
	}
    }
#endif /*HAVE_LIBDL*/

  /* run Squeak */
#ifndef J_PROFILE
  interpret();
#else
  j_interpret();
#endif

  return 0;
}


int ioExit(void)
{
#ifdef EVENT_LOG
	fclose(eventLogFp);
#endif
  exit(0);
}
