/*
 * Copyright (c) 2004, 2018, 2023 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <inttypes.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <time.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <arpa/inet.h>
#include <netinet/in.h>

#include "mdd.h"
#include "config.h"
#include "packets.h"

#define ST_RUNNING	0
#define ST_FINISHING	1
#define ST_FINISHED	2

void
sender(char *file_name, off_t start, off_t sendmax, size_t block_size,
       in_addr_t group_addr, unsigned char ttl, unsigned int ifindex)
{
	int fd, sd, rd;
	struct sockaddr_in send_addr;
	struct sockaddr_in recv_addr;
	ssize_t readen;
	off_t offset;
	off_t new_offset;
	off_t file_size = 0;
	struct stat st;
	struct mdd_packet *packet;
	size_t packet_size;
	struct ip_mreq ms;
	struct ip_mreq mr;
	in_addr_t src_addr;
	int water_mark;
	int on;
	int status;
	struct timeval now;
	struct timeval wait = { SHUTDOWN_WAIT_SEC, SHUTDOWN_WAIT_USEC };
	struct timeval time_to_stop;

	if ((fd = open(file_name, O_RDONLY, 0)) == -1) {
		printf("Open \"%s\" failed: %s\n", file_name, strerror(errno));
		exit(-1);
	}

	if (sendmax != -1) {
		file_size = sendmax;
	} else {
		if (fstat(fd, &st) == 0)
			file_size = st.st_size;
	}

	if ((sd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1) {
		printf("Cannot create UDP/IP socket: %s\n", strerror(errno));
		exit(-1);
	}

	on = 1;
	if ((setsockopt(sd, SOL_SOCKET, SO_REUSEPORT, &on, sizeof(on))) == -1) {
		perror("cannot set SO_REUSEPORT");
		exit(-1);
	}
#ifdef notanymore
	if ((setsockopt(sd, SOL_SOCKET, SO_BROADCAST, &on, sizeof(on))) == -1) {
		perror("cannot use broadcast");
		exit(-1);
	}
#endif

	if (debug)
		printf("Setting TTL to %d\n", (int)ttl);
        if ((setsockopt(sd, IPPROTO_IP, IP_MULTICAST_TTL,
            &ttl, sizeof(ttl))) == -1) {
                perror("cannot set IP_MULTICAST_TTL");
                exit(-1);
        }

	src_addr = local_addr(group_addr, ifindex);

	if (set_multicast_if(sd, &src_addr) != 0) {
		perror("set_multicast_if for sender socket failed");
		exit(-1);
	}

	(void)memset(&send_addr, 0, sizeof(send_addr));
	send_addr.sin_family = AF_INET;
#ifndef __linux__
	send_addr.sin_len = sizeof(send_addr);
#endif
	send_addr.sin_addr.s_addr = src_addr;
	send_addr.sin_port = htons(DATA_SEND_PORT);
	if ((bind(sd, (struct sockaddr *)&send_addr,
	    sizeof(send_addr))) == -1) {
		 printf("Cannot bind sender socket to UDP port %d: %s\n",
		    DATA_SEND_PORT, strerror(errno));
		exit(-1);
	}

	water_mark = sizeof(struct mdd_packet);
	if ((setsockopt(sd, SOL_SOCKET, SO_SNDLOWAT,
	    &water_mark, sizeof(water_mark))) == -1) {
		perror("cannot set send low water mark");
		exit(-1);
	}

	 bzero(&ms, sizeof(ms));
	 ms.imr_multiaddr.s_addr = group_addr;
	 ms.imr_interface.s_addr = src_addr;
	 if ((setsockopt(sd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
	     &ms, sizeof(ms))) == -1) {
		 perror("cannot set IP_ADD_MEMBERSHIP");
		 exit(-1);
	 }

	if ((rd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1) {
		printf("Cannot create UDP/IP socket: %s\n", strerror(errno));
		exit(-1);
	}

	on = 1;
	if ((setsockopt(rd, SOL_SOCKET, SO_REUSEPORT, &on, sizeof(on))) == -1) {
		perror("cannot set SO_REUSEPORT");
		exit(-1);
	}

	(void)memset(&recv_addr, 0, sizeof(recv_addr));
	recv_addr.sin_family = AF_INET;
#ifndef __linux__
	recv_addr.sin_len = sizeof(recv_addr);
#endif
	recv_addr.sin_addr.s_addr = INADDR_ANY;
	recv_addr.sin_port = htons(RESEND_REQ_PORT);
	if ((bind(rd, (struct sockaddr *)&recv_addr,
	    sizeof(recv_addr))) == -1) {
		printf("Cannot bind receiver socket to UDP port %d: %s\n",
		    RESEND_REQ_PORT, strerror(errno));
		exit(-1);
	}

	if (fcntl(rd, F_SETFL, fcntl(rd, F_GETFL)|O_NONBLOCK) == -1) {
		perror("cannot set non blocking receiving");
		exit(-1);
	}

	water_mark = sizeof(struct mdd_packet);
	if ((setsockopt(rd, SOL_SOCKET, SO_RCVLOWAT,
	    &water_mark, sizeof(water_mark))) == -1) {
		perror("cannot set recveive low water mark");
		exit(-1);
	}

	bzero(&mr, sizeof(mr));
	mr.imr_multiaddr.s_addr = group_addr;
	mr.imr_interface.s_addr = src_addr;
	if ((setsockopt(rd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
	    &mr, sizeof(mr))) == -1) {
		perror("cannot set IP_ADD_MEMBERSHIP");
		exit(-1);
	}

	packet_size = block_size + sizeof(struct mdd_packet);
	if ((packet = malloc(packet_size)) == NULL) {
		perror("Cannot allocate buffer");
		exit(-1);
	}
	if (debug)
		printf("packet size: %zd\n", packet_size);

	send_addr.sin_addr.s_addr = group_addr;
	status = ST_RUNNING;
	offset = 0;
	do {
		struct sockaddr_in from_sin;
		char from_str[20];
		socklen_t from_salen;
		int mdd_operation;
		off_t mdd_offset;
		size_t mdd_data_size;

		new_offset = lseek(fd, start + offset, SEEK_SET);
		mdd_operation = OP_DATA_DUMP;
		mdd_offset = offset;
		readen = read(fd, &packet->mdd_data, block_size);

		/* When reading a device, we get an error at the end */
		if (readen == -1)
			readen = 0;

		if (sendmax != -1) {
			if (offset + readen > sendmax) {
				if (offset < sendmax)
					readen = sendmax - offset;
				else
					readen = 0;
			}
		}

		mdd_data_size = readen;
		if (readen == 0 && status == ST_RUNNING) {
			(void)gettimeofday(&now, NULL);
			timeradd(&now, &wait, &time_to_stop);
			status = ST_FINISHING;
			if (verbose)
				printf("Shutting down in %g.%06ld\n",
				    difftime(wait.tv_sec, 0),
				    (long)wait.tv_usec);
		}
		if (status == ST_FINISHING) {
			mdd_operation |= OP_SHUTDOWN;
			(void)gettimeofday(&now, NULL);
			if (timercmp(&now, &time_to_stop, >))
				status = ST_FINISHED;
		}

		packet->mdd_operation = htole32(mdd_operation);
		packet->mdd_offset = htole64(mdd_offset);
		packet->mdd_data_size = htole64(mdd_data_size);

		if (sendto(sd, packet, packet_size, 0,
		    (struct sockaddr *)&send_addr, sizeof(send_addr)) == -1) {
			if (debug) {
				printf("Send failed (offset %" PRId64 "): "
				    "%s, retry\n", offset, strerror(errno));
			}
		} else {
			offset = new_offset + readen;
		}

		from_salen = sizeof(from_sin);
		if (recvfrom(rd, packet, sizeof(struct mdd_packet),
		    0, (struct sockaddr *)&from_sin, &from_salen) > 0) {
			inet_ntop(AF_INET, &from_sin.sin_addr,
			    from_str, sizeof(from_str));

			mdd_operation = le32toh(packet->mdd_operation);
			mdd_offset = le64toh(packet->mdd_offset);
			mdd_data_size = (size_t)le64toh(packet->mdd_data_size);
			if (debug) {
				printf("Packet received from %s\n", from_str);
				printf("operation = %d\n", mdd_operation);
				printf("offset = %" PRId64 "\n", mdd_offset);
				printf("current offset = %" PRId64 "\n",offset);
			}
			if (mdd_offset <= offset) {
				if (verbose) {
					printf("resend request from %s "
					    "%" PRId64 " -> %" PRId64 "\n",
					    from_str, offset, mdd_offset);
				}
				offset = mdd_offset;
				if (status == ST_FINISHING) {
					status = ST_RUNNING;
					if (verbose)
						printf("Abort shutting down\n");
				}
			}
			if (debug)
				printf("new offset = %" PRId64 "\n", offset);

			if (!debug && !verbose && !quiet)
				print_report(offset, file_size);

		}

		if (debug)
			printf("new offset = %" PRId64 ", readen = %zd\n",
			    offset, readen);

	} while (status != ST_FINISHED);

	if (verbose)
		printf("sender finished\n");

        if ((setsockopt(sd, IPPROTO_IP, IP_DROP_MEMBERSHIP,
            &ms, sizeof(ms))) == -1) {
                perror("cannot set IP_DROP_MEMBERSHIP");
        }

        if ((setsockopt(rd, IPPROTO_IP, IP_DROP_MEMBERSHIP,
            &mr, sizeof(mr))) == -1) {
                perror("cannot set IP_DROP_MEMBERSHIP");
        }

	return;
}
