// license:GPL-2.0+
// copyright-holders:Juergen Buchmueller, Angelo Salese
/******************************************************************************

    Atari 400/800

    Juergen Buchmueller, June 1998

    TODO (generic):
    - add cassette support;
    - add floppy .atx support;
    - Investigate supported RAM sizes and OS versions in different models;
    - Fix various keyboard differences;
    - Freddy emulation for 800XLF?
    - Add support for proto boards and expansions (a1400xl, C/PM board, etc.)
    - a130xe: support extended bank readback for Antic;
    - a1200xl: requires reading TRIG3 high for detecting a cart inserted,
      depends on above;
    - a600xl, a1200xl: crashes on MMU test in Acid800;
    - slot support for PBI/ECI bus;
    - slot support for overlay DIY HW mods:
      \- PokeyMAX
         (with stereo support via second Pokey alias accessed to $d280-$d2ff,
         cfr. yoomp);
      \- Ultimate1MB;
      \- Covox;
      \- VBXE "VideoBoard XE";
      \- Incognito;
      \- Rapidus;
      \- AKI PS/2 keyboard inteface;
      \- RAMBO XL and COMPY RAM expansion;
      \- APE Warp+ OS 32-in-1;
      \- MyBIOS for MyIDE-II;
      \- Bit-3 Full-View 80
         (technically maps in cart CCTL space, but installs in RAM card slot 3
         and overrides ANTIC+GTIA layer when enabled);

    2009-05 FP changes:
     Factored out MESS specific code from MAME
     Added skeleton support for other XL/XE machines (VERY preliminary):
     - a600xl based on maxaflex emulation in MAME
     - a1200xl sharing a800xl code without BASIC
     - a65xe, a65xea, a130xe, a800xe, xegs sharing a800xl code (and this is wrong
      at least for xegs)
     Added proper dumps and labels, thanks to Freddy Offenga researches (a few
     are still marked BAD_DUMP while waiting for crc confirmation, since they
     have been obtained by splitting whole dumps)

    2013-11-06 Robert Tuccitto:
    Updated Palette per 'CGIA D020577' and 'GTIA C014805', including
    normalized grayscale with proper color gradient.  Added Phase Shift
    values 24.7 thru 27.7 degrees in 0.5 degree increments.  Enabled
    Phase Shift 26.2 degrees as default.

    2013-11-23 Robert Tuccitto:
    Added palette notes

******************************************************************************/

#include "emu.h"
#include "atari400.h"

#include "cpu/m6502/m6502.h"
#include "machine/6821pia.h"
#include "machine/bankdev.h"
#include "machine/input_merger.h"
#include "machine/ram.h"
#include "machine/timer.h"
#include "sound/dac.h"
#include "sound/pokey.h"

#include "bus/a800/a800_slot.h"
#include "bus/a800/a800_carts.h"
#include "bus/a800/a8sio.h"
#include "bus/vcs_ctrl/ctrl.h"

#include "screen.h"
#include "softlist_dev.h"
#include "speaker.h"

/******************************************************************************
    Atari 800 memory map (preliminary)

    ***************** read access *******************
    range     short   description
    0000-9FFF RAM     main memory
    A000-BFFF RAM/ROM RAM or (banked) ROM cartridges
    C000-CFFF ROM     unused or monitor ROM

    ********* GTIA    ********************************

    D000      m0pf    missile 0 playfield collisions
    D001      m1pf    missile 1 playfield collisions
    D002      m2pf    missile 2 playfield collisions
    D003      m3pf    missile 3 playfield collisions
    D004      p0pf    player 0 playfield collisions
    D005      p1pf    player 1 playfield collisions
    D006      p2pf    player 2 playfield collisions
    D007      p3pf    player 3 playfield collisions
    D008      m0pl    missile 0 player collisions
    D009      m1pl    missile 1 player collisions
    D00A      m2pl    missile 2 player collisions
    D00B      m3pl    missile 3 player collisions
    D00C      p0pl    player 0 player collisions
    D00D      p1pl    player 1 player collisions
    D00E      p2pl    player 2 player collisions
    D00F      p3pl    player 3 player collisions
    D010      but0    button stick 0
    D011      but1    button stick 1
    D012      but2    button stick 2
    D013      but3    button stick 3
    D014      xff     unused
    D015      xff     unused
    D016      xff     unused
    D017      xff     unused
    D018      xff     unused
    D019      xff     unused
    D01A      xff     unused
    D01B      xff     unused
    D01C      xff     unused
    D01D      xff     unused
    D01E      xff     unused
    D01F      cons    console keys
    D020-D0FF repeated 7 times

    D100-D1FF xff

    ********* POKEY   ********************************
    D200      pot0    paddle 0
    D201      pot1    paddle 1
    D202      pot2    paddle 2
    D203      pot3    paddle 3
    D204      pot4    paddle 4
    D205      pot5    paddle 5
    D206      pot6    paddle 6
    D207      pot7    paddle 7
    D208      potb    all paddles
    D209      kbcode  keyboard scan code
    D20A      random  random number generator
    D20B      xff     unused
    D20C      xff     unused
    D20D      serin   serial input
    D20E      irqst   IRQ status
    D20F      skstat  sk status
    D210-D2FF repeated 15 times

    ********* PIO     ********************************
    D300      porta   read pio port A
    D301      portb   read pio port B
    D302      pactl   read pio port A control
    D303      pbctl   read pio port B control
    D304-D3FF repeated 63 times

    ********* ANTIC   ********************************
    D400      xff     unused
    D401      xff     unused
    D402      xff     unused
    D403      xff     unused
    D404      xff     unused
    D405      xff     unused
    D406      xff     unused
    D407      xff     unused
    D408      xff     unused
    D409      xff     unused
    D40A      xff     unused
    D40B      vcount  vertical (scanline) counter
    D40C      penh    light pen horizontal pos
    D40D      penv    light pen vertical pos
    D40E      xff     unused
    D40F      nmist   NMI status

    D500-D7FF xff     unused memory

    D800-DFFF ROM     floating point ROM
    E000-FFFF ROM     bios ROM

    ***************** write access *******************
    range     short   description
    0000-9FFF RAM     main memory
    A000-BFFF RAM/ROM RAM or (banked) ROM
    C000-CFFF ROM     unused or monitor ROM

    ********* GTIA    ********************************
    D000      hposp0  player 0 horz position
    D001      hposp1  player 1 horz position
    D002      hposp2  player 2 horz position
    D003      hposp3  player 3 horz position
    D004      hposm0  missile 0 horz position
    D005      hposm1  missile 0 horz position
    D006      hposm2  missile 0 horz position
    D007      hposm3  missile 0 horz position
    D008      sizep0  size player 0
    D009      sizep1  size player 0
    D00A      sizep2  size player 0
    D00B      sizep3  size player 0
    D00C      sizem   size missiles
    D00D      grafp0  graphics data for player 0
    D00E      grafp1  graphics data for player 1
    D00F      grafp2  graphics data for player 2
    D010      grafp3  graphics data for player 3
    D011      grafm   graphics data for missiles
    D012      colpm0  color for player/missile 0
    D013      colpm1  color for player/missile 1
    D014      colpm2  color for player/missile 2
    D015      colpm3  color for player/missile 3
    D016      colpf0  color 0 playfield
    D017      colpf1  color 1 playfield
    D018      colpf2  color 2 playfield
    D019      colpf3  color 3 playfield
    D01A      colbk   background playfield
    D01B      prior   priority select
    D01C      vdelay  delay until vertical retrace
    D01D      gractl  graphics control
    D01E      hitclr  clear collisions
    D01F      wcons   write console (speaker)
    D020-D0FF repeated 7 times

    D100-D1FF xff     unused

    ********* POKEY   ********************************
    D200      audf1   frequency audio chan #1
    D201      audc1   control audio chan #1
    D202      audf2   frequency audio chan #2
    D203      audc2   control audio chan #2
    D204      audf3   frequency audio chan #3
    D205      audc3   control audio chan #3
    D206      audf4   frequency audio chan #4
    D207      audc4   control audio chan #4
    D208      audctl  audio control
    D209      stimer  start timer
    D20A      skres   sk reset
    D20B      potgo   start pot AD conversion
    D20C      xff     unused
    D20D      serout  serial output
    D20E      irqen   IRQ enable
    D20F      skctl   sk control
    D210-D2FF repeated 15 times

    ********* PIO     ********************************
    D300      porta   write pio port A (output or mask)
    D301      portb   write pio port B (output or mask)
    D302      pactl   write pio port A control
    D303      pbctl   write pio port B control
    D304-D3FF         repeated

    ********* ANTIC   ********************************
    D400      dmactl  write DMA control
    D401      chactl  write character control
    D402      dlistl  write display list lo
    D403      dlisth  write display list hi
    D404      hscrol  write horz scroll
    D405      vscrol  write vert scroll
    D406      xff     unused
    D407      pmbash  player/missile base addr hi
    D408      xff     unused
    D409      chbash  character generator base addr hi
    D40A      wsync   wait for hsync
    D40B      xff     unused
    D40C      xff     unused
    D40D      xff     unused
    D40E      nmien   NMI enable
    D40F      nmires  NMI reset

    D500-D7FF xff     unused memory

    D800-DFFF ROM     floating point ROM
    E000-FFFF ROM     BIOS ROM
******************************************************************************/

class a400_state : public atari_common_state
{
public:
	a400_state(const machine_config &mconfig, device_type type, const char *tag)
		: atari_common_state(mconfig, type, tag)
		, m_ram(*this, RAM_TAG)
		, m_pia(*this, "pia")
		, m_dac(*this, "dac")
		, m_sio(*this, "sio")
		, m_region_maincpu(*this, "maincpu")
		, m_cartleft(*this, "cartleft")
		, m_cart_rd4_view(*this, "cart_rd4_view")
		, m_cart_rd5_view(*this, "cart_rd5_view")
		, m_ctrl(*this, "ctrl%u", 1U)
	{ }

	void a400(machine_config &config);
	void a400pal(machine_config &config);

protected:
	void atari_common_nodac(machine_config &config);
	void atari_common(machine_config &config);

	void a400_mem(address_map &map) ATTR_COLD;
	TIMER_DEVICE_CALLBACK_MEMBER(a400_interrupt);

	virtual void machine_start() override ATTR_COLD;
	virtual void machine_reset() override ATTR_COLD;

	// TODO: these two should really be inside ram_device instead
	template <unsigned StartBase> uint8_t ram_r(address_space &space, offs_t offset)
	{
		const offs_t memory_offset = StartBase + offset;

		if (memory_offset < m_ram->size())
			return m_ram->pointer()[memory_offset];

		// TODO: floating bus
		return space.unmap();
	}

	template <unsigned StartBase> void ram_w(offs_t offset, uint8_t data)
	{
		const offs_t memory_offset = StartBase + offset;

		if (memory_offset < m_ram->size())
			m_ram->pointer()[memory_offset] = data;
	}

	virtual uint8_t djoy_b_r();

private:
	void a400_palette(palette_device &palette) const;

	void gtia_cb(uint8_t data);

	uint8_t djoy_0_1_r();
	void djoy_0_1_w(uint8_t data);
	uint8_t djoy_2_3_r();
	void djoy_2_3_w(uint8_t data);

protected:
	optional_device<ram_device> m_ram;
	optional_device<pia6821_device> m_pia;
	optional_device<dac_bit_interface> m_dac;
	optional_device<a8sio_device> m_sio;
	required_memory_region m_region_maincpu;
	optional_device<a800_cart_slot_device> m_cartleft;
	memory_view m_cart_rd4_view, m_cart_rd5_view;
	optional_device_array<vcs_control_port_device, 4> m_ctrl;

	void hw_iomap(address_map &map) ATTR_COLD;

	int m_cart_rd4_enabled = 0, m_cart_rd5_enabled = 0;
	void cart_rd4_w( int state );
	void cart_rd5_w( int state );

	virtual void area_8000_map(address_map &map) ATTR_COLD;
	virtual void area_a000_map(address_map &map) ATTR_COLD;
};

class a800_state : public a400_state
{
public:
	a800_state(const machine_config &mconfig, device_type type, const char *tag)
		: a400_state(mconfig, type, tag)
		, m_cartright(*this, "cartright")
	{ }

	void a800(machine_config &config);
	void a800pal(machine_config &config);

protected:
//  virtual void machine_start() override ATTR_COLD;
	virtual void machine_reset() override ATTR_COLD;

private:
	required_device<a800_cart_slot_device> m_cartright;

	void a800_mem(address_map &map) ATTR_COLD;
};

class a1200xl_state : public a400_state
{
public:
	a1200xl_state(const machine_config &mconfig, device_type type, const char *tag)
		: a400_state(mconfig, type, tag)
		, m_kernel_view(*this, "kernel_view")
		, m_selftest_view(*this, "selftest_view")
	{ }

	void a1200xl(machine_config &config);

protected:
	void atari_xl_common(machine_config &config);

	void a1200xl_mem(address_map &map) ATTR_COLD;

	virtual void machine_start() override ATTR_COLD;
	virtual void machine_reset() override ATTR_COLD;

	TIMER_DEVICE_CALLBACK_MEMBER(xl_interrupt);

	memory_view m_kernel_view;
	memory_view m_selftest_view;

	virtual void portb_cb(uint8_t data);
	virtual uint8_t djoy_b_r() override;

	void selftest_map(memory_view::memory_view_entry &block, bool is_rom_mapping);
private:
	void pia_portb_w(uint8_t data);

	uint8_t m_mmu;
};

class a800xl_state : public a1200xl_state
{
public:
	a800xl_state(const machine_config &mconfig, device_type type, const char *tag)
		: a1200xl_state(mconfig, type, tag)
		, m_basic_view(*this, "basic_view")
	{ }

	void a800xl(machine_config &config);
	void a800xlpal(machine_config &config);

protected:
	virtual void machine_reset() override ATTR_COLD;

	virtual void portb_cb(uint8_t data) override;

	void a800xl_mem(address_map &map) ATTR_COLD;

	virtual void area_a000_map(address_map &map) override ATTR_COLD;

	memory_view m_basic_view;
};

class a600xl_state : public a800xl_state
{
public:
	a600xl_state(const machine_config &mconfig, device_type type, const char *tag)
		: a800xl_state(mconfig, type, tag)
	{ }

	void a600xl(machine_config &config);
};

class a130xe_state : public a800xl_state
{
public:
	a130xe_state(const machine_config &mconfig, device_type type, const char *tag)
		: a800xl_state(mconfig, type, tag)
		, m_ext_view(*this, "ext_view")
		, m_ext_bank(*this, "ext_bank")
	{ }

	void a130xe(machine_config &config);

private:
	virtual void machine_reset() override ATTR_COLD;

	void a130xe_mem(address_map &map) ATTR_COLD;

	virtual void portb_cb(uint8_t data) override;

	memory_view m_ext_view;
	required_device<address_map_bank_device> m_ext_bank;

	void extram_map(address_map &map) ATTR_COLD;

};

class xegs_state : public a800xl_state
{
public:
	xegs_state(const machine_config &mconfig, device_type type, const char *tag)
		: a800xl_state(mconfig, type, tag)
		, m_bank(*this, "bank")
	{ }

	void xegs(machine_config &config);

private:
	virtual void machine_start() override ATTR_COLD;
	virtual void machine_reset() override ATTR_COLD;

	virtual void portb_cb(uint8_t data) override;

	required_memory_bank m_bank;

	void xegs_mem(address_map &map) ATTR_COLD;
};

class a5200_state : public a400_state
{
public:
	a5200_state(const machine_config &mconfig, device_type type, const char *tag)
		: a400_state(mconfig, type, tag)
		, m_cart(*this, "cartslot")
	{ }

	void a5200(machine_config &config);
	void a5200a(machine_config &config);

private:
	virtual void machine_start() override ATTR_COLD;
	virtual void machine_reset() override ATTR_COLD;

	required_device<a5200_cart_slot_device> m_cart;

	TIMER_DEVICE_CALLBACK_MEMBER(a5200_interrupt);

	void a5200_mem(address_map &map) ATTR_COLD;
};

/**************************************************************
 *
 * Memory maps
 *
 **************************************************************/

// TODO: transparent support for PBI (XL) / ECI (XE series), at 0xd1xx, 0xd6xx, 0xd7xx + ROM at 0xd800-0xdfff

void a400_state::hw_iomap(address_map &map)
{
	map(0x0000, 0x00ff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
//  map(0x0100, 0x01ff).noprw();
	map(0x0200, 0x02ff).rw(m_pokey, FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0x0300, 0x03ff).rw(m_pia, FUNC(pia6821_device::read_alt), FUNC(pia6821_device::write_alt));
	map(0x0400, 0x04ff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	map(0x0500, 0x05ff).rw(m_cartleft, FUNC(a800_cart_slot_device::read_cctl), FUNC(a800_cart_slot_device::write_cctl));
//  map(0x0600, 0x07ff).noprw();
}

void a400_state::area_8000_map(address_map &map)
{
	map(0x8000, 0x9fff).view(m_cart_rd4_view);
	m_cart_rd4_view[0](0x8000, 0x9fff).rw(FUNC(a400_state::ram_r<0x8000>), FUNC(a400_state::ram_w<0x8000>));
	m_cart_rd4_view[1](0x8000, 0x9fff).rw(m_cartleft, FUNC(a800_cart_slot_device::read_cart<0>), FUNC(a800_cart_slot_device::write_cart<0>));
}

void a400_state::area_a000_map(address_map &map)
{
	map(0xa000, 0xbfff).view(m_cart_rd5_view);
	m_cart_rd5_view[0](0xa000, 0xbfff).rw(FUNC(a400_state::ram_r<0xa000>), FUNC(a400_state::ram_w<0xa000>));
	m_cart_rd5_view[1](0xa000, 0xbfff).rw(m_cartleft, FUNC(a800_cart_slot_device::read_cart<1>), FUNC(a800_cart_slot_device::write_cart<1>));
}


// a400/a800 explicitly expects floating bus for unmapped ranges, will punt with value_low()
void a400_state::a400_mem(address_map &map)
{
	map.unmap_value_high();
	map(0x0000, 0x7fff).rw(FUNC(a400_state::ram_r<0x0000>), FUNC(a400_state::ram_w<0x0000>));
	area_8000_map(map);
	area_a000_map(map);
	map(0xc000, 0xcfff).rom();
	map(0xd000, 0xd7ff).m(*this, FUNC(a400_state::hw_iomap));
	map(0xd800, 0xffff).rom();
}

void a800_state::a800_mem(address_map &map)
{
	a400_mem(map);
	// TODO: stub for right cart handling, to be eventually honored internally
	// ...
}

// XL specifics

void a1200xl_state::selftest_map(memory_view::memory_view_entry &block, bool is_rom_mapping)
{
	block(0x4000, 0x4fff).rw(FUNC(a1200xl_state::ram_r<0x4000>), FUNC(a1200xl_state::ram_w<0x4000>));
	if (is_rom_mapping)
		block(0x5000, 0x57ff).rom().region("maincpu", 0xd000);
	else
		block(0x5000, 0x57ff).rw(FUNC(a1200xl_state::ram_r<0x5000>), FUNC(a1200xl_state::ram_w<0x5000>));
	block(0x5800, 0x7fff).rw(FUNC(a1200xl_state::ram_r<0x5800>), FUNC(a1200xl_state::ram_w<0x5800>));
}

// from a800xl onward HW I/O map punches thru kernel view,
// for simplicity we just delay mapping former.

// same as a800xl except no built-in basic (just maps RAM there)
void a1200xl_state::a1200xl_mem(address_map &map)
{
	map.unmap_value_high();
	map(0x0000, 0x3fff).rw(FUNC(a1200xl_state::ram_r<0x0000>), FUNC(a1200xl_state::ram_w<0x0000>));
	map(0x4000, 0x7fff).view(m_selftest_view);
	selftest_map(m_selftest_view[0], false);
	selftest_map(m_selftest_view[1], true);
	area_8000_map(map);
	area_a000_map(map);
	map(0xc000, 0xffff).view(m_kernel_view);
	m_kernel_view[0](0xc000, 0xffff).rw(FUNC(a1200xl_state::ram_r<0xc000>), FUNC(a1200xl_state::ram_w<0xc000>));
	m_kernel_view[1](0xc000, 0xffff).rom().region("maincpu", 0xc000);
	map(0xd000, 0xd7ff).m(*this, FUNC(a1200xl_state::hw_iomap));
}

void a800xl_state::area_a000_map(address_map &map)
{
	map(0xa000, 0xbfff).view(m_cart_rd5_view);
	m_cart_rd5_view[0](0xa000, 0xbfff).view(m_basic_view);
	m_basic_view[0](0xa000, 0xbfff).rw(FUNC(a800xl_state::ram_r<0xa000>), FUNC(a800xl_state::ram_w<0xa000>));
	m_basic_view[1](0xa000, 0xbfff).rom().region("maincpu", 0xa000);
	m_cart_rd5_view[1](0xa000, 0xbfff).rw(m_cartleft, FUNC(a800_cart_slot_device::read_cart<1>), FUNC(a800_cart_slot_device::write_cart<1>));
}

void a800xl_state::a800xl_mem(address_map &map)
{
	map.unmap_value_high();
	map(0x0000, 0x3fff).rw(FUNC(a800xl_state::ram_r<0x0000>), FUNC(a800xl_state::ram_w<0x0000>));
	map(0x4000, 0x7fff).view(m_selftest_view);
	selftest_map(m_selftest_view[0], false);
	selftest_map(m_selftest_view[1], true);
	area_8000_map(map);
	area_a000_map(map);
	map(0xc000, 0xffff).view(m_kernel_view);
	m_kernel_view[0](0xc000, 0xffff).rw(FUNC(a800xl_state::ram_r<0xc000>), FUNC(a800xl_state::ram_w<0xc000>));
	m_kernel_view[1](0xc000, 0xffff).rom().region("maincpu", 0xc000);
	map(0xd000, 0xd7ff).m(*this, FUNC(a800xl_state::hw_iomap));
}

// selftest ROM has still priority over regular a130xe extended RAM
// cfr. a130xe MMU test in Acid800
void a130xe_state::extram_map(address_map &map)
{
	map(0x00000, 0x0ffff).mirror(0x10000).rw(FUNC(a130xe_state::ram_r<0x10000>), FUNC(a130xe_state::ram_w<0x10000>));
	map(0x11000, 0x117ff).mirror(0x0c000).rom().region("maincpu", 0xd000).unmapw();
}

void a130xe_state::a130xe_mem(address_map &map)
{
	map.unmap_value_high();
	map(0x0000, 0x3fff).rw(FUNC(a130xe_state::ram_r<0x0000>), FUNC(a130xe_state::ram_w<0x0000>));
	map(0x4000, 0x7fff).view(m_ext_view);
	m_ext_view[0](0x4000, 0x7fff).view(m_selftest_view);
	selftest_map(m_selftest_view[0], false);
	selftest_map(m_selftest_view[1], true);
	m_ext_view[1](0x4000, 0x7fff).m(m_ext_bank, FUNC(address_map_bank_device::amap8));
	area_8000_map(map);
	area_a000_map(map);
	map(0xc000, 0xffff).view(m_kernel_view);
	m_kernel_view[0](0xc000, 0xffff).rw(FUNC(a130xe_state::ram_r<0xc000>), FUNC(a130xe_state::ram_w<0xc000>));
	m_kernel_view[1](0xc000, 0xffff).rom().region("maincpu", 0xc000);
	map(0xd000, 0xd7ff).m(*this, FUNC(a130xe_state::hw_iomap));
}


void xegs_state::xegs_mem(address_map &map)
{
	map.unmap_value_high();
	map(0x0000, 0x4fff).ram();
	map(0x5000, 0x57ff).view(m_selftest_view);
	m_selftest_view[0](0x5000, 0x57ff).ram();
	m_selftest_view[1](0x5000, 0x57ff).rom().region("maincpu", 0xd000);
	map(0x5800, 0x7fff).ram();

	map(0x8000, 0x9fff).view(m_cart_rd4_view);
	m_cart_rd4_view[0](0x8000, 0x9fff).ram();
	m_cart_rd4_view[1](0x8000, 0x9fff).rw(m_cartleft, FUNC(a800_cart_slot_device::read_cart<0>), FUNC(a800_cart_slot_device::write_cart<0>));

	map(0xa000, 0xbfff).view(m_cart_rd5_view);
	m_cart_rd5_view[0](0xa000, 0xbfff).view(m_basic_view);
	m_basic_view[0](0xa000, 0xbfff).ram();
	m_basic_view[1](0xa000, 0xbfff).bankr(m_bank);
	m_cart_rd5_view[1](0xa000, 0xbfff).rw(m_cartleft, FUNC(a800_cart_slot_device::read_cart<1>), FUNC(a800_cart_slot_device::write_cart<1>));

	map(0xc000, 0xffff).view(m_kernel_view);
	m_kernel_view[0](0xc000, 0xffff).ram();
	m_kernel_view[1](0xc000, 0xffff).rom().region("maincpu", 0xc000);
	map(0xd000, 0xd7ff).m(*this, FUNC(xegs_state::hw_iomap));
}


void a5200_state::a5200_mem(address_map &map)
{
	map(0x0000, 0x3fff).ram();
	map(0x4000, 0xbfff).rw(m_cart, FUNC(a5200_cart_slot_device::read_cart), FUNC(a5200_cart_slot_device::write_cart));
	map(0xc000, 0xcfff).rw(m_gtia, FUNC(gtia_device::read), FUNC(gtia_device::write));
	map(0xd400, 0xd4ff).rw(m_antic, FUNC(antic_device::read), FUNC(antic_device::write));
	// 0xe000-0xe7ff - Expansion?
	map(0xe800, 0xefff).rw(m_pokey, FUNC(pokey_device::read), FUNC(pokey_device::write));
	map(0xf000, 0xffff).rom();
}



/**************************************************************
 *
 * Input ports
 *
 **************************************************************/


#define JOYSTICK_DELTA          10
#define JOYSTICK_SENSITIVITY    200

static INPUT_PORTS_START( atari_artifacting )
	PORT_START("artifacts")
	PORT_CONFNAME(0x40, 0x00, "Television Artifacts" )
	PORT_CONFSETTING(0x00, DEF_STR( Off ) )
	PORT_CONFSETTING(0x40, DEF_STR( On ) )
INPUT_PORTS_END



static INPUT_PORTS_START( atari_console )
	PORT_START("console")
	PORT_BIT(0x04, 0x04, IPT_KEYBOARD) PORT_NAME("CONS.2: Option") PORT_CODE(KEYCODE_F3)
	PORT_BIT(0x02, 0x02, IPT_KEYBOARD) PORT_NAME("CONS.1: Select") PORT_CODE(KEYCODE_F2)
	PORT_BIT(0x01, 0x01, IPT_KEYBOARD) PORT_NAME("CONS.0: Start") PORT_CODE(KEYCODE_F1)
INPUT_PORTS_END



/* 2009-04 FP:
Small note about natural keyboard support: currently,
- "Break" is mapped to 'F1'
- "Clear" is mapped to 'F2'
- "Atari" is mapped to 'F3'                         */

static INPUT_PORTS_START( atari_keyboard )
	PORT_START("keyboard.0")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_L) PORT_CHAR('l') PORT_CHAR('L')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_J) PORT_CHAR('j') PORT_CHAR('J')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_COLON) PORT_CHAR(';') PORT_CHAR(':')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Break") PORT_CODE(KEYCODE_BACKSPACE) PORT_CHAR(UCHAR_MAMEKEY(F1))
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_K) PORT_CHAR('k') PORT_CHAR('K')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_QUOTE) PORT_CHAR('+') PORT_CHAR('\\')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_BACKSLASH) PORT_CHAR('*') PORT_CHAR('^')

	PORT_START("keyboard.1")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_O) PORT_CHAR('o') PORT_CHAR('O')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_P) PORT_CHAR('p') PORT_CHAR('P')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_U) PORT_CHAR('u') PORT_CHAR('U')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Return") PORT_CODE(KEYCODE_ENTER) PORT_CHAR(13)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_I) PORT_CHAR('i') PORT_CHAR('I')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_OPENBRACE) PORT_CHAR('-') PORT_CHAR('_')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_CLOSEBRACE) PORT_CHAR('=') PORT_CHAR('|')

	PORT_START("keyboard.2")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_V) PORT_CHAR('v') PORT_CHAR('V')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_C) PORT_CHAR('c') PORT_CHAR('C')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_B) PORT_CHAR('b') PORT_CHAR('B')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_X) PORT_CHAR('x') PORT_CHAR('X')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Z) PORT_CHAR('z') PORT_CHAR('Z')

	PORT_START("keyboard.3")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_4) PORT_CHAR('4') PORT_CHAR('$')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_3) PORT_CHAR('3') PORT_CHAR('#')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_6) PORT_CHAR('6') PORT_CHAR('&')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Escape") PORT_CODE(KEYCODE_ESC) PORT_CHAR(27)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_5) PORT_CHAR('5') PORT_CHAR('%')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_2) PORT_CHAR('2') PORT_CHAR('\"')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_1) PORT_CHAR('1') PORT_CHAR('!')

	PORT_START("keyboard.4")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_COMMA) PORT_CHAR(',') PORT_CHAR('[')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_SPACE) PORT_CHAR(' ')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_STOP) PORT_CHAR('.') PORT_CHAR(']')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_N) PORT_CHAR('n') PORT_CHAR('N')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_M) PORT_CHAR('m') PORT_CHAR('M')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_SLASH) PORT_CHAR('/') PORT_CHAR('?')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Atari") PORT_CODE(KEYCODE_RCONTROL) PORT_CHAR(UCHAR_MAMEKEY(F3))

	PORT_START("keyboard.5")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_R) PORT_CHAR('r') PORT_CHAR('R')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_E) PORT_CHAR('e') PORT_CHAR('E')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Y) PORT_CHAR('y') PORT_CHAR('Y')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Tab") PORT_CODE(KEYCODE_TAB) PORT_CHAR(UCHAR_MAMEKEY(TAB))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_T) PORT_CHAR('t') PORT_CHAR('T')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_W) PORT_CHAR('w') PORT_CHAR('W')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_Q) PORT_CHAR('q') PORT_CHAR('Q')

	PORT_START("keyboard.6")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_9) PORT_CHAR('9') PORT_CHAR('(')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_0) PORT_CHAR('0') PORT_CHAR(')')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_7) PORT_CHAR('7') PORT_CHAR('\'')
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("BackS  Delete") PORT_CODE(KEYCODE_BACKSLASH2) PORT_CHAR(8) PORT_CHAR(UCHAR_MAMEKEY(DEL))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_8) PORT_CHAR('8') PORT_CHAR('@')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("<  Clear") PORT_CODE(KEYCODE_MINUS) PORT_CHAR('<') PORT_CHAR(UCHAR_MAMEKEY(F2))
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME(">  Insert") PORT_CODE(KEYCODE_EQUALS) PORT_CHAR('>') PORT_CHAR(UCHAR_MAMEKEY(INSERT))

	PORT_START("keyboard.7")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_F) PORT_CHAR('f') PORT_CHAR('F')
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_H) PORT_CHAR('h') PORT_CHAR('H')
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_D) PORT_CHAR('d') PORT_CHAR('D')
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Lowr  Caps") PORT_CODE(KEYCODE_CAPSLOCK) PORT_CHAR(UCHAR_MAMEKEY(CAPSLOCK))
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_G) PORT_CHAR('g') PORT_CHAR('G')
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_S) PORT_CHAR('s') PORT_CHAR('S')
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_CODE(KEYCODE_A) PORT_CHAR('a') PORT_CHAR('A')

	PORT_START("fake")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Shift") PORT_CODE(KEYCODE_LSHIFT) PORT_CODE(KEYCODE_RSHIFT) PORT_CHAR(UCHAR_SHIFT_1)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Ctrl") PORT_CODE(KEYCODE_LCONTROL) PORT_CHAR(UCHAR_SHIFT_2)
INPUT_PORTS_END



static INPUT_PORTS_START( a800 )
	PORT_INCLUDE( atari_artifacting )
	PORT_INCLUDE( atari_console )
	PORT_INCLUDE( atari_keyboard )
INPUT_PORTS_END

static INPUT_PORTS_START( a1200xl )
	PORT_INCLUDE( a800 )

	// option jumpers, available on a1200xl only
	// J1 causes a self-test if installed
	// J2 to J4 are unused by BIOS but may eventually be read by SW ...
	PORT_START("J1")
	PORT_BIT( 0x7f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_CONFNAME( 0x80, 0x80, "Enable self-test check (J1)" )
	PORT_CONFSETTING(    0x00, DEF_STR( Yes ) )
	PORT_CONFSETTING(    0x80, DEF_STR( No ) )

	PORT_START("J2")
	PORT_BIT( 0x7f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_CONFNAME( 0x80, 0x80, "Unused (J2)" )
	PORT_CONFSETTING(    0x80, DEF_STR( Off ) )
	PORT_CONFSETTING(    0x00, DEF_STR( On ) )

	PORT_START("J3")
	PORT_BIT( 0x7f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_CONFNAME( 0x80, 0x80, "Unused (J3)" )
	PORT_CONFSETTING(    0x80, DEF_STR( Off ) )
	PORT_CONFSETTING(    0x00, DEF_STR( On ) )

	PORT_START("J4")
	PORT_BIT( 0x7f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_CONFNAME( 0x80, 0x80, "Unused (J4)" )
	PORT_CONFSETTING(    0x80, DEF_STR( Off ) )
	PORT_CONFSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END

static INPUT_PORTS_START( a5200 )
	PORT_INCLUDE( atari_artifacting )

	PORT_START("djoy_b")    /* lower/upper buttons */
	PORT_BIT(0x01, 0x01, IPT_BUTTON1) PORT_PLAYER(1)
	PORT_BIT(0x02, 0x02, IPT_BUTTON1) PORT_PLAYER(2)
	PORT_BIT(0x04, 0x04, IPT_BUTTON1) PORT_PLAYER(3)
	PORT_BIT(0x08, 0x08, IPT_BUTTON1) PORT_PLAYER(4)
	PORT_BIT(0x10, 0x10, IPT_BUTTON2) PORT_PLAYER(1)
	PORT_BIT(0x20, 0x20, IPT_BUTTON2) PORT_PLAYER(2)
	PORT_BIT(0x40, 0x40, IPT_BUTTON2) PORT_PLAYER(3)
	PORT_BIT(0x80, 0x80, IPT_BUTTON2) PORT_PLAYER(4)

	PORT_START("keypad.0")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("(Break)") PORT_CODE(KEYCODE_PAUSE)    // is this correct?
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[#]") PORT_CODE(KEYCODE_ENTER_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[0]") PORT_CODE(KEYCODE_0_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[*]") PORT_CODE(KEYCODE_PLUS_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad.1")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("Reset") PORT_CODE(KEYCODE_F3)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[9]") PORT_CODE(KEYCODE_9_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[8]") PORT_CODE(KEYCODE_8_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[7]") PORT_CODE(KEYCODE_7_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad.2")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME(DEF_STR(Pause)) PORT_CODE(KEYCODE_F2)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[6]") PORT_CODE(KEYCODE_6_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[5]") PORT_CODE(KEYCODE_5_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[4]") PORT_CODE(KEYCODE_4_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("keypad.3")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_START)    PORT_NAME("Start")
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[3]") PORT_CODE(KEYCODE_3_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[2]") PORT_CODE(KEYCODE_2_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("[1]") PORT_CODE(KEYCODE_1_PAD)
	PORT_BIT(0xf0, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("analog_0")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(1)

	PORT_START("analog_1")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(1)

	PORT_START("analog_2")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(2)

	PORT_START("analog_3")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(2)

	PORT_START("analog_4")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(3)

	PORT_START("analog_5")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(3)

	PORT_START("analog_6")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_X) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(4)

	PORT_START("analog_7")
	PORT_BIT(0xff, 0x72, IPT_AD_STICK_Y) PORT_SENSITIVITY(JOYSTICK_SENSITIVITY) PORT_KEYDELTA(JOYSTICK_DELTA) PORT_MINMAX(0x00,0xe4) PORT_PLAYER(4)

INPUT_PORTS_END



static INPUT_PORTS_START( a5200a )
	PORT_INCLUDE( a5200 )

	PORT_MODIFY("djoy_b")    /* lower/upper buttons */
	PORT_BIT(0x04, 0x04, IPT_UNUSED)
	PORT_BIT(0x08, 0x08, IPT_UNUSED)
	PORT_BIT(0x40, 0x40, IPT_UNUSED)
	PORT_BIT(0x80, 0x80, IPT_UNUSED)

	PORT_MODIFY("analog_4")
	PORT_BIT(0xff, 0x72, IPT_UNUSED)

	PORT_MODIFY("analog_5")
	PORT_BIT(0xff, 0x72, IPT_UNUSED)

	PORT_MODIFY("analog_6")
	PORT_BIT(0xff, 0x72, IPT_UNUSED)

	PORT_MODIFY("analog_7")
	PORT_BIT(0xff, 0x72, IPT_UNUSED)
INPUT_PORTS_END

/***************************************************************
Atari 5200 Palette Notes:

Palette on a modern flat panel display (LCD, LED, Plasma, etc.)
appears different from a traditional CRT. The most outstanding
difference is Hue 1x, the hue begin point. Hue 1x looks very
'green' (~-60 to -45 degrees - depending on how poor or well it
handles the signal conversion and its calibration) on a modern
flat panel display, as opposed to 'gold' (~-33 degrees) on a
CRT.  The official technical document, "GTIA C014805 NTSC"
stipulates Hue 1x as gold.

The "Atari 5200 Field Service Manual" provides two different
sets of instructions in harmony with utilizing the "PAM
Diagnostic SALT Cartridge v1.1".  In one account it states the
color just below and above the reference bar to be within one
shade of each other.

Under the same reference document, directions are given for it
to be the same color.  Phase Shift 25.7 degrees matches Hue 1x,
15x and the color below the reference bar.

However, if the system is adjusted within the first several
minutes of running, the warm up, consistent system run time,
causes Hue 15x (F$) to become stronger/darker gold (More brown
then ultimately red-brown); as well as leans Hue 14x (E$) more
brown than green.  Once achieving a phase shift of 27.7,
Hue 14x (E$) and Hue 15x (F$) near-exact match Hue 1x and 2x
respectively.

Accounting for system 'warm-up', phase shifting, as well as the
instructions for it to be within one shade of each other, would
make Phase Shift 26.2 degrees or 26.7 degrees a realistic
logical choice.

It also collaborates with the official "GTIA C014805 NTSC"
document for color order: Hue 1x = Gold, Hue 2x = Orange,
Hue 15x (F$) = Light-Orange; Phase Shift 26.2 places
Hue 15x (F$) between Hue 1x, Gold and Hue 2x, Orange;
a Light Orange in color.  Color descriptions are best measured
in the middle of the brightness scale.

It should be mentioned that Green-Yellow is referenced at
Hue 13x (D$), nowhere near Hue 1x.  A Green-Yellow Hue 1x is
how the palette is manipulated and modified (in part) under
a modern flat panel display.

Note though, even a properly calibrated console, at power on,
the phase shift appears as low as ~23 degrees and after a
considerable consistent runtime, can be as high as ~28 degrees.
In general, the low end of ~23 degrees lasts for maybe several
seconds, whereas higher values such as ~25-27 degrees is the
most dominant during system run time.

Additionally, the blue to red (And consequently blue to green)
ratio proportions may appear different on a modern flat panel
display than a CRT in some instances for the Atari 5200 system.
Furthermore, you may have some variation of proportions even
within the same display type.

One side effect of this on the console's palette is that some
values of red may appear too pinkish - Too much blue to red.
This is not the same as a traditional tint-hue control
adjustment; rather, can be demonstrated by changing the blue
ratio values via MAME HLSL settings.

Lastly, the Atari 2600 & 7800 NTSC color palettes hold the same
hue structure order and have similar appearance differences
dependent upon display type.
***************************************************************/
/**************************************************************
 *
 * Palette - Phase Shift 26.2
 *
 **************************************************************/

static const uint8_t atari_colors[256*3] =
{
	/* Grey */
	0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
	0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
	0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
	0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,
	/* Gold */
	0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
	0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
	0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
	0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,
	/* Orange */
	0x31,0x00,0x00, 0x42,0x06,0x00, 0x53,0x17,0x00, 0x64,0x28,0x00,
	0x75,0x39,0x00, 0x86,0x4A,0x00, 0x97,0x5B,0x0B, 0xA8,0x6C,0x1C,
	0xB9,0x7D,0x2D, 0xCA,0x8E,0x3E, 0xDB,0x9F,0x4F, 0xEC,0xB0,0x60,
	0xFD,0xC1,0x71, 0xFF,0xD2,0x86, 0xFF,0xE3,0x9D, 0xFF,0xF4,0xB3,
	/* Red-Orange */
	0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x08,0x00, 0x71,0x19,0x00,
	0x82,0x2A,0x0F, 0x93,0x3B,0x20, 0xA4,0x4C,0x31, 0xB5,0x5D,0x42,
	0xC6,0x6E,0x53, 0xD7,0x7F,0x64, 0xE8,0x90,0x75, 0xF9,0xA1,0x86,
	0xFF,0xB2,0x9A, 0xFF,0xC3,0xB0, 0xFF,0xD4,0xC6, 0xFF,0xE5,0xDC,
	/* Pink */
	0x3E,0x00,0x06, 0x4F,0x00,0x12, 0x60,0x00,0x1E, 0x71,0x0E,0x2E,
	0x82,0x1F,0x3F, 0x93,0x30,0x50, 0xA4,0x41,0x61, 0xB5,0x52,0x72,
	0xC6,0x63,0x83, 0xD7,0x74,0x94, 0xE8,0x85,0xA5, 0xF9,0x96,0xB6,
	0xFF,0xA7,0xCB, 0xFF,0xB8,0xE1, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,
	/* Purple */
	0x32,0x00,0x38, 0x43,0x00,0x44, 0x54,0x00,0x50, 0x65,0x0C,0x5F,
	0x76,0x1D,0x70, 0x87,0x2E,0x81, 0x98,0x3F,0x92, 0xA9,0x50,0xA3,
	0xBA,0x61,0xB4, 0xCB,0x72,0xC5, 0xDC,0x83,0xD6, 0xED,0x94,0xE4,
	0xFE,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,
	/* Purple-Blue */
	0x1B,0x00,0x5F, 0x2C,0x00,0x6B, 0x3D,0x00,0x77, 0x4E,0x11,0x88,
	0x5F,0x22,0x99, 0x70,0x33,0xAA, 0x81,0x44,0xBB, 0x92,0x55,0xCC,
	0xA3,0x66,0xDD, 0xB4,0x77,0xED, 0xC5,0x88,0xED, 0xD6,0x99,0xED,
	0xE7,0xAA,0xED, 0xF8,0xBB,0xED, 0xFF,0xCC,0xF0, 0xFF,0xDD,0xF5,
	/* Blue 1 */
	0x00,0x00,0x72, 0x10,0x00,0x7E, 0x21,0x0D,0x8E, 0x32,0x1E,0x9F,
	0x43,0x2F,0xB0, 0x54,0x40,0xC1, 0x65,0x51,0xD2, 0x76,0x62,0xE3,
	0x87,0x73,0xF4, 0x98,0x84,0xF9, 0xA9,0x95,0xF9, 0xBA,0xA6,0xF9,
	0xCB,0xB7,0xF9, 0xDC,0xC8,0xF9, 0xED,0xD9,0xF9, 0xFE,0xEA,0xF9,
	/* Blue 2 */
	0x00,0x00,0x65, 0x00,0x0C,0x7A, 0x05,0x1D,0x8E, 0x16,0x2E,0x9F,
	0x27,0x3F,0xB0, 0x38,0x50,0xC1, 0x49,0x61,0xD2, 0x5A,0x72,0xE3,
	0x6B,0x83,0xF4, 0x7C,0x94,0xFF, 0x8D,0xA5,0xFF, 0x9E,0xB6,0xFF,
	0xAF,0xC7,0xFF, 0xC0,0xD8,0xFF, 0xD1,0xE9,0xFF, 0xE2,0xFA,0xFF,
	/* Light-Blue */
	0x00,0x0D,0x48, 0x00,0x1E,0x5E, 0x00,0x2F,0x74, 0x00,0x40,0x8A,
	0x11,0x51,0x9B, 0x22,0x62,0xAC, 0x33,0x73,0xBD, 0x44,0x84,0xCE,
	0x55,0x95,0xDF, 0x66,0xA6,0xF0, 0x77,0xB7,0xFF, 0x88,0xC8,0xFF,
	0x99,0xD9,0xFF, 0xAA,0xEA,0xFF, 0xBB,0xFB,0xFF, 0xCC,0xFF,0xFF,
	/* Turquoise */
	0x00,0x1C,0x1C, 0x00,0x2D,0x32, 0x00,0x3E,0x49, 0x00,0x4F,0x5F,
	0x05,0x60,0x73, 0x16,0x71,0x84, 0x27,0x82,0x95, 0x38,0x93,0xA6,
	0x49,0xA4,0xB7, 0x5A,0xB5,0xC8, 0x6B,0xC6,0xD9, 0x7C,0xD7,0xEA,
	0x8D,0xE8,0xFB, 0x9E,0xF9,0xFF, 0xAF,0xFF,0xFF, 0xC0,0xFF,0xFF,
	/* Green-Blue */
	0x00,0x25,0x0B, 0x00,0x36,0x10, 0x00,0x47,0x18, 0x00,0x58,0x2E,
	0x07,0x69,0x42, 0x18,0x7A,0x53, 0x29,0x8B,0x64, 0x3A,0x9C,0x75,
	0x4B,0xAD,0x86, 0x5C,0xBE,0x97, 0x6D,0xCF,0xA8, 0x7E,0xE0,0xB9,
	0x8F,0xF1,0xCA, 0xA0,0xFF,0xDA, 0xB1,0xFF,0xE6, 0xC2,0xFF,0xF2,
	/* Green */
	0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x04,0x5A,0x1A,
	0x15,0x6B,0x1A, 0x26,0x7C,0x22, 0x37,0x8D,0x33, 0x48,0x9E,0x44,
	0x59,0xAF,0x55, 0x6A,0xC0,0x66, 0x7B,0xD1,0x77, 0x8C,0xE2,0x88,
	0x9D,0xF3,0x99, 0xAE,0xFF,0xA8, 0xBF,0xFF,0xB4, 0xD0,0xFF,0xC0,
	/* Yellow-Green */
	0x00,0x21,0x0A, 0x00,0x32,0x0F, 0x0A,0x43,0x11, 0x1B,0x54,0x11,
	0x2C,0x65,0x11, 0x3D,0x76,0x11, 0x4E,0x87,0x11, 0x5F,0x98,0x1E,
	0x70,0xA9,0x2F, 0x81,0xBA,0x40, 0x92,0xCB,0x51, 0xA3,0xDC,0x62,
	0xB4,0xED,0x73, 0xC5,0xFE,0x84, 0xD6,0xFF,0x90, 0xE7,0xFF,0x9C,
	/* Orange-Green */
	0x05,0x13,0x04, 0x16,0x24,0x04, 0x27,0x35,0x04, 0x38,0x46,0x04,
	0x49,0x57,0x04, 0x5A,0x68,0x04, 0x6B,0x79,0x04, 0x7C,0x8A,0x09,
	0x8D,0x9B,0x1A, 0x9E,0xAC,0x2B, 0xAF,0xBD,0x3C, 0xC0,0xCE,0x4D,
	0xD1,0xDF,0x5E, 0xE2,0xF0,0x6F, 0xF3,0xFF,0x80, 0xFF,0xFF,0x8D,
	/* Light-Orange */
	0x21,0x02,0x00, 0x32,0x13,0x00, 0x43,0x24,0x00, 0x54,0x35,0x00,
	0x65,0x46,0x00, 0x76,0x57,0x00, 0x87,0x68,0x00, 0x98,0x79,0x0C,
	0xA9,0x8A,0x1D, 0xBA,0x9B,0x2E, 0xCB,0xAC,0x3F, 0xDC,0xBD,0x50,
	0xED,0xCE,0x61, 0xFE,0xDF,0x72, 0xFF,0xF0,0x87, 0xFF,0xFF,0x9D
};


/* Initialise the palette */
void a400_state::a400_palette(palette_device &palette) const
{
	for (unsigned i = 0; i < std::size(atari_colors) / 3; i++)
		palette.set_pen_color(i, atari_colors[i * 3], atari_colors[i * 3 + 1], atari_colors[i * 3 + 2]);
}
/******************************************************************
    PALETTE - PHASE 24.7 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x30,0x00,0x00, 0x41,0x07,0x00, 0x52,0x18,0x00, 0x63,0x29,0x00,
    0x74,0x3A,0x00, 0x85,0x4B,0x00, 0x96,0x5C,0x0A, 0xA7,0x6D,0x1B,
    0xB8,0x7E,0x2C, 0xC9,0x8F,0x3D, 0xDA,0xA0,0x4E, 0xEB,0xB1,0x5F,
    0xFC,0xC2,0x70, 0xFF,0xD3,0x85, 0xFF,0xE4,0x9B, 0xFF,0xF5,0xB1,

RED-ORANGE
    0x3D,0x00,0x00, 0x4E,0x00,0x00, 0x5F,0x09,0x00, 0x70,0x1A,0x00,
    0x81,0x2B,0x09, 0x92,0x3C,0x1A, 0xA3,0x4D,0x2B, 0xB4,0x5E,0x3C,
    0xC5,0x6F,0x4D, 0xD6,0x80,0x5E, 0xE7,0x91,0x6F, 0xF8,0xA2,0x80,
    0xFF,0xB3,0x94, 0xFF,0xC4,0xAA, 0xFF,0xD5,0xC0, 0xFF,0xE6,0xD6,

PINK
    0x3F,0x00,0x00, 0x50,0x00,0x09, 0x61,0x00,0x15, 0x72,0x10,0x26,
    0x83,0x21,0x37, 0x94,0x32,0x48, 0xA5,0x43,0x59, 0xB6,0x54,0x6A,
    0xC7,0x65,0x7B, 0xD8,0x76,0x8C, 0xE9,0x87,0x9D, 0xFA,0x98,0xAE,
    0xFF,0xA9,0xC2, 0xFF,0xBA,0xD8, 0xFF,0xCB,0xEE, 0xFF,0xDC,0xF4,

PURPLE
    0x36,0x00,0x2E, 0x47,0x00,0x3A, 0x58,0x00,0x46, 0x69,0x0C,0x55,
    0x7A,0x1D,0x66, 0x8B,0x2E,0x77, 0x9C,0x3F,0x88, 0xAD,0x50,0x99,
    0xBE,0x61,0xAA, 0xCF,0x72,0xBB, 0xE0,0x83,0xCC, 0xF1,0x94,0xDD,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x23,0x00,0x55, 0x34,0x00,0x61, 0x45,0x00,0x6D, 0x56,0x0F,0x7E,
    0x67,0x20,0x8F, 0x78,0x31,0xA0, 0x89,0x42,0xB1, 0x9A,0x53,0xC2,
    0xAB,0x64,0xD3, 0xBC,0x75,0xE4, 0xCD,0x86,0xEA, 0xDE,0x97,0xEA,
    0xEF,0xA8,0xEA, 0xFF,0xB9,0xEA, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

BLUE1
    0x09,0x00,0x6E, 0x1A,0x00,0x7A, 0x2B,0x08,0x88, 0x3C,0x19,0x99,
    0x4D,0x2A,0xAA, 0x5E,0x3B,0xBB, 0x6F,0x4C,0xCC, 0x80,0x5D,0xDD,
    0x91,0x6E,0xEE, 0xA2,0x7F,0xF4, 0xB3,0x90,0xF4, 0xC4,0xA1,0xF4,
    0xD5,0xB2,0xF4, 0xE6,0xC3,0xF4, 0xF7,0xD4,0xF4, 0xFF,0xE5,0xF7,

BLUE2
    0x00,0x00,0x6D, 0x00,0x05,0x80, 0x10,0x16,0x91, 0x21,0x27,0xA2,
    0x32,0x38,0xB3, 0x43,0x49,0xC4, 0x54,0x5A,0xD5, 0x65,0x6B,0xE6,
    0x76,0x7C,0xF7, 0x87,0x8D,0xFF, 0x98,0x9E,0xFF, 0xA9,0xAF,0xFF,
    0xBA,0xC0,0xFF, 0xCB,0xD1,0xFF, 0xDC,0xE2,0xFF, 0xED,0xF3,0xFF

LIGHT-BLUE
    0x00,0x05,0x57, 0x00,0x16,0x6E, 0x00,0x27,0x84, 0x09,0x38,0x97,
    0x1A,0x49,0xA8, 0x2B,0x5A,0xB9, 0x3C,0x6B,0xCA, 0x4D,0x7C,0xDB,
    0x5E,0x8D,0xEC, 0x6F,0x9E,0xFD, 0x80,0xAF,0xFF, 0x91,0xC0,0xFF,
    0xA2,0xD1,0xFF, 0xB3,0xE2,0xFF, 0xC4,0xF3,0xFF, 0xD5,0xFF,0xFF,

TURQUOISE
    0x00,0x15,0x34, 0x00,0x26,0x4A, 0x00,0x37,0x60, 0x00,0x48,0x77,
    0x0A,0x59,0x8A, 0x1B,0x6A,0x9B, 0x2C,0x7B,0xAC, 0x3D,0x8C,0xBD,
    0x4E,0x9D,0xCE, 0x5F,0xAE,0xDF, 0x70,0xBF,0xF0, 0x81,0xD0,0xFF,
    0x92,0xE1,0xFF, 0xA3,0xF2,0xFF, 0xB4,0xFF,0xFF, 0xC5,0xFF,0xFF

GREEN-BLUE
    0x00,0x21,0x0A, 0x00,0x32,0x1F, 0x00,0x43,0x35, 0x00,0x54,0x4B,
    0x04,0x65,0x60, 0x15,0x76,0x71, 0x26,0x87,0x82, 0x37,0x98,0x93,
    0x48,0xA9,0xA4, 0x59,0xBA,0xB5, 0x6A,0xCB,0xC6, 0x7B,0xDC,0xD7,
    0x8C,0xED,0xE8, 0x9D,0xFE,0xF9, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1D,
    0x0A,0x6B,0x30, 0x1B,0x7C,0x41, 0x2C,0x8D,0x52, 0x3D,0x9E,0x63,
    0x4E,0xAF,0x74, 0x5F,0xC0,0x85, 0x70,0xD1,0x96, 0x81,0xE2,0xA7,
    0x92,0xF3,0xB8, 0xA3,0xFF,0xC8, 0xB4,0xFF,0xD3, 0xC5,0xFF,0xDF,

YELLOW-GREEN
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x0A,0x59,0x18,
    0x1B,0x6A,0x18, 0x2C,0x7B,0x18, 0x3D,0x8C,0x27, 0x4E,0x9D,0x38,
    0x5F,0xAE,0x49, 0x70,0xBF,0x5A, 0x81,0xD0,0x6B, 0x92,0xE1,0x7C,
    0xA3,0xF2,0x8D, 0xB4,0xFF,0x9C, 0xC5,0xFF,0xA8, 0xD6,0xFF,0xB4,

ORANGE-GREEN
    0x00,0x1E,0x09, 0x00,0x2F,0x0E, 0x11,0x40,0x0E, 0x22,0x51,0x0E,
    0x33,0x62,0x0E, 0x44,0x73,0x0E, 0x55,0x84,0x0E, 0x66,0x95,0x17,
    0x77,0xA6,0x28, 0x88,0xB7,0x39, 0x99,0xC8,0x4A, 0xAA,0xD9,0x5B,
    0xBB,0xEA,0x6C, 0xCC,0xFB,0x7D, 0xDD,0xFF,0x8A, 0xEE,0xFF,0x96,

LIGHT-ORANGE
    0x0A,0x11,0x02, 0x1B,0x22,0x02, 0x2C,0x33,0x02, 0x3D,0x44,0x02,
    0x4E,0x55,0x02, 0x5F,0x66,0x02, 0x70,0x77,0x02, 0x81,0x88,0x09,
    0x92,0x99,0x1A, 0xA3,0xAA,0x2B, 0xB4,0xBB,0x3C, 0xC5,0xCC,0x4D,
    0xD6,0xDD,0x5E, 0xE7,0xEE,0x6F, 0xF8,0xFF,0x80, 0xFF,0xFF,0x8F,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 25.2 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x30,0x00,0x00, 0x41,0x07,0x00, 0x52,0x18,0x00, 0x63,0x29,0x00,
    0x74,0x3A,0x00, 0x85,0x4B,0x00, 0x96,0x5C,0x0A, 0xA7,0x6D,0x1B,
    0xB8,0x7E,0x2C, 0xC9,0x8F,0x3D, 0xDA,0xA0,0x4E, 0xEB,0xB1,0x5F,
    0xFC,0xC2,0x70, 0xFF,0xD3,0x85, 0xFF,0xE4,0x9B, 0xFF,0xF5,0xB1,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x09,0x00, 0x71,0x1A,0x00,
    0x82,0x2B,0x0B, 0x93,0x3C,0x1C, 0xA4,0x4D,0x2D, 0xB5,0x5E,0x3E,
    0xC6,0x6F,0x4F, 0xD7,0x80,0x60, 0xE8,0x91,0x71, 0xF9,0xA2,0x82,
    0xFF,0xB3,0x96, 0xFF,0xC4,0xAC, 0xFF,0xD5,0xC2, 0xFF,0xE6,0xD8,

PINK
    0x3F,0x00,0x00, 0x50,0x00,0x0C, 0x61,0x00,0x18, 0x72,0x0F,0x28,
    0x83,0x20,0x39, 0x94,0x31,0x4A, 0xA5,0x42,0x5B, 0xB6,0x53,0x6C,
    0xC7,0x64,0x7D, 0xD8,0x75,0x8E, 0xE9,0x86,0x9F, 0xFA,0x97,0xB0,
    0xFF,0xA8,0xC5, 0xFF,0xB9,0xDB, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

PURPLE
    0x35,0x00,0x31, 0x46,0x00,0x3D, 0x57,0x00,0x49, 0x68,0x0C,0x58,
    0x79,0x1D,0x69, 0x8A,0x2E,0x7A, 0x9B,0x3F,0x8B, 0xAC,0x50,0x9C,
    0xBD,0x61,0xAD, 0xCE,0x72,0xBE, 0xDF,0x83,0xCF, 0xF0,0x94,0xE0,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x20,0x00,0x59, 0x31,0x00,0x65, 0x42,0x00,0x71, 0x53,0x10,0x82,
    0x64,0x21,0x93, 0x75,0x32,0xA4, 0x86,0x43,0xB5, 0x97,0x54,0xC6,
    0xA8,0x65,0xD7, 0xB9,0x76,0xE8, 0xCA,0x87,0xEB, 0xDB,0x98,0xEB,
    0xEC,0xA9,0xEB, 0xFD,0xBA,0xEB, 0xFF,0xCB,0xEF, 0xFF,0xDC,0xF4,

BLUE1
    0x05,0x00,0x70, 0x16,0x00,0x7C, 0x27,0x09,0x8B, 0x38,0x1A,0x9C,
    0x49,0x2B,0xAD, 0x5A,0x3C,0xBE, 0x6B,0x4D,0xCF, 0x7C,0x5E,0xE0,
    0X8D,0x6F,0xF1, 0x9E,0x80,0xF6, 0xAF,0x91,0xF6, 0xC0,0xA2,0xF6,
    0xD1,0xB3,0xF6, 0xE2,0xC4,0xF6, 0xF3,0xD5,0xF6, 0xFF,0xE6,0xF7,

BLUE2
    0x00,0x00,0x6B, 0x00,0x08,0x7E, 0x0C,0x19,0x91, 0x1D,0x2A,0xA2,
    0x2E,0x3B,0xB3, 0x3F,0x4C,0xC4, 0x50,0x5D,0xD5, 0x61,0x6E,0xE6,
    0x72,0x7F,0xF7, 0x83,0x90,0xFF, 0x94,0xA1,0xFF, 0xA5,0xB2,0xFF,
    0xB6,0xC3,0xFF, 0xC7,0xD4,0xFF, 0xD8,0xE5,0xFF, 0xE9,0xF6,0xFF,

LIGHT-BLUE
    0x00,0x08,0x52, 0x00,0x19,0x68, 0x00,0x2A,0x7F, 0x05,0x3B,0x93,
    0x16,0x4C,0xA4, 0x27,0x5D,0xB5, 0x38,0x6E,0xC6, 0x49,0x7F,0xD7,
    0x5A,0x90,0xE8, 0x6B,0xA1,0xF9, 0x7C,0xB2,0xFF, 0x8D,0xC3,0xFF,
    0x9E,0xD4,0xFF, 0xAF,0xE5,0xFF, 0xC0,0xF6,0xFF, 0xD1,0xFF,0xFF,

TURQUOISE
    0x00,0x17,0x2D, 0x00,0x28,0x43, 0x00,0x39,0x59, 0x00,0x4A,0x6F,
    0x08,0x5B,0x83, 0x19,0x6C,0x94, 0x2A,0x7D,0xA5, 0x3B,0x8E,0xB6,
    0x4C,0x9F,0xC7, 0x5D,0xB0,0xD8, 0x6E,0xC1,0xE9, 0x7F,0xD2,0xFA,
    0x90,0xE3,0xFF, 0xA1,0xF4,0xFF, 0xB2,0xFF,0xFF, 0xC3,0xFF,0xFF,

GREEN-BLUE
    0x00,0x23,0x0A, 0x00,0x34,0x15, 0x00,0x45,0x2B, 0x00,0x56,0x41,
    0x04,0x67,0x56, 0x15,0x78,0x67, 0x26,0x89,0x78, 0x37,0x9A,0x89,
    0x48,0xAB,0x9A, 0x59,0xBC,0xAB, 0x6A,0xCD,0xBC, 0x7B,0xDE,0xCD,
    0x8C,0xEF,0xDE, 0x9D,0xFF,0xEE, 0xAE,0xFF,0xFA, 0xBF,0xFF,0xFF,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x0D,0x6B,0x25, 0x1E,0x7C,0x36, 0x2F,0x8D,0x47, 0x40,0x9E,0x58,
    0x51,0xAF,0x69, 0x62,0xC0,0x7A, 0x73,0xD1,0x8B, 0x84,0xE2,0x9C,
    0x95,0xF3,0xAD, 0xA6,0xFF,0xBD, 0xB7,0xFF,0xC9, 0xC8,0xFF,0xD4,

YELLOW-GREEN
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x00,0x46,0x15, 0x10,0x57,0x15,
    0x21,0x68,0x15, 0x32,0x79,0x15, 0x43,0x8A,0x1C, 0x54,0x9B,0x2D,
    0x65,0xAC,0x3E, 0x76,0xBD,0x4F, 0x87,0xCE,0x60, 0x98,0xDF,0x71,
    0xA9,0xF0,0x82, 0xBA,0xFF,0x93, 0xCB,0xFF,0x9F, 0xDC,0xFF,0xAA,

ORANGE-GREEN
    0x00,0x1B,0x08, 0x08,0x2C,0x0B, 0x19,0x3D,0x0B, 0x2A,0x4E,0x0B,
    0x3B,0x5F,0x0B, 0x4C,0x70,0x0B, 0x5D,0x81,0x0B, 0x6E,0x92,0x11,
    0x7F,0xA3,0x22, 0x90,0xB4,0x33, 0xA1,0xC5,0x44, 0xB2,0xD6,0x55,
    0xC3,0xE7,0x66, 0xD4,0xF8,0x77, 0xE5,0xFF,0x85, 0xF6,0xFF,0x91,

LIGHT-ORANGE
    0x12,0x0C,0x00, 0x23,0x1D,0x00, 0x34,0x2E,0x00, 0x45,0x3F,0x00,
    0x56,0x50,0x00, 0x67,0x61,0x00, 0x78,0x72,0x00, 0x89,0x83,0x08,
    0x9A,0x94,0x19, 0xAB,0xA5,0x2A, 0xBC,0xB6,0x3B, 0xCD,0xC7,0x4C,
    0xDE,0xD8,0x5D, 0xEF,0xE9,0x6E, 0xFF,0xFA,0x80, 0xFF,0xFF,0x92,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 25.7 SHIFT
GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x31,0x00,0x00, 0x42,0x06,0x00, 0x53,0x17,0x00, 0x64,0x28,0x00,
    0x75,0x39,0x00, 0x86,0X4A,0x00, 0x97,0x5B,0x0A, 0xA8,0x6C,0x1B,
    0xB9,0x7D,0x2C, 0xCA,0x8E,0x3D, 0xDB,0x9F,0x4E, 0xEC,0xB0,0x5F,
    0xFD,0xC1,0x70, 0xFF,0xD2,0x85, 0xFF,0xE3,0x9C, 0xFF,0xF4,0xB2,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x08,0x00, 0x71,0x19,0x00,
    0x82,0x2A,0x0D, 0x93,0x3B,0x1E, 0xA4,0x4C,0x2F, 0xB5,0x5D,0x40,
    0xC6,0x6E,0x51, 0xD7,0x7F,0x62, 0xE8,0x90,0x73, 0xF9,0xA1,0x83,
    0xFF,0xB2,0x98, 0xFF,0xC3,0xAE, 0xFF,0xD4,0xC4, 0xFF,0xE5,0xDA,

PINK
    0x3F,0x00,0x03, 0x50,0x00,0x0F, 0x61,0x00,0x1B, 0x72,0x0F,0x2B,
    0x83,0x20,0x3C, 0x94,0x31,0x4D, 0xA5,0x42,0x5E, 0xB6,0x53,0x6F,
    0xC7,0x64,0x80, 0xD8,0x75,0x91, 0xE9,0x86,0xA2, 0xFA,0x97,0xB3,
    0xFF,0xA8,0xC8, 0xFF,0xB9,0xDE, 0xFF,0xCA,0xEF, 0xFF,0xDB,0xF4,

PURPLE
    0x33,0x00,0x35, 0x44,0x00,0x41, 0x55,0x00,0x4C, 0x66,0x0C,0x5C,
    0x77,0x1D,0x6D, 0x88,0x2E,0x7E, 0x99,0x3F,0x8F, 0xAA,0x50,0xA0,
    0xBB,0x61,0xB1, 0xCC,0x72,0xC2, 0xDD,0x83,0xD3, 0xEE,0x94,0xE4,
    0xFF,0xA5,0xE4, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x1D,0x00,0x5C, 0x2E,0x00,0x68, 0x40,0x00,0x74, 0x51,0x10,0x84,
    0x62,0x21,0x95, 0x73,0x32,0xA6, 0x84,0x43,0xB7, 0x95,0x54,0xC8,
    0xA6,0x65,0xD9, 0xB7,0x76,0xEA, 0xC8,0x87,0xEB, 0xD9,0x98,0xEB,
    0xE9,0xA9,0xEC, 0xFB,0xBA,0xEB, 0xFF,0xCB,0xEF, 0xFF,0xDC,0xF4,

BLUE1
    0x02,0x00,0x71, 0x13,0x00,0x7D, 0x24,0x0B,0x8C, 0x35,0x1C,0x9D,
    0x46,0x2D,0xAE, 0x57,0x3E,0xBF, 0x68,0x4F,0xD0, 0x79,0x60,0xE1,
    0x8A,0x71,0xF2, 0x9B,0x82,0xF7, 0xAC,0x93,0xF7, 0xBD,0xA4,0xF7,
    0xCE,0xB5,0xF7, 0xDF,0xC6,0xF7, 0xF0,0xD7,0xF7, 0xFF,0xE8,0xF8,

BLUE2
    0x00,0x00,0x68, 0x00,0x0A,0x7C, 0x08,0x1B,0x90, 0x19,0x2C,0xA1,
    0x2A,0x3D,0xB2, 0x3B,0x4E,0xC3, 0x4C,0x5F,0xD4, 0x5D,0x70,0xE5,
    0x6E,0x81,0xF6, 0x7F,0x92,0xFF, 0x90,0xA3,0xFF, 0xA1,0xB4,0xFF,
    0xB2,0xC5,0xFF, 0xC3,0xD6,0xFF, 0xD4,0xE7,0xFF, 0xE5,0xF8,0xFF,

LIGHT-BLUE
    0x00,0x0A,0x4D, 0x00,0x1B,0x63, 0x00,0x2C,0x79, 0x02,0x3D,0x8F,
    0x13,0x4E,0xA0, 0x24,0x5F,0xB1, 0x35,0x70,0xC2, 0x46,0x81,0xD3,
    0x57,0x92,0xE4, 0x68,0xA3,0xF5, 0x79,0xB4,0xFF, 0x8A,0xC5,0xFF,
    0x9B,0xD6,0xFF, 0xAC,0xE7,0xFF, 0xBD,0xF8,0xFF, 0xCE,0xFF,0xFF,

TURQUOISE
    0x00,0x1A,0x26, 0x00,0x2B,0x3C, 0x00,0x3C,0x52, 0x00,0x4D,0x68,
    0x06,0x5E,0x7C, 0x17,0x6F,0x8D, 0x28,0x80,0x9E, 0x39,0x91,0xAF,
    0x4A,0xA2,0xC0, 0x5B,0xB3,0xD1, 0x6C,0xC4,0xE2, 0x7D,0xD5,0xF3,
    0x8E,0xE6,0xFF, 0x9F,0xF7,0xFF, 0xB0,0xFF,0xFF, 0xC1,0xFF,0xFF,

GREEN-BLUE
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x00,0x46,0x22, 0x00,0x57,0x38,
    0x05,0x68,0x4D, 0x16,0x79,0x5E, 0x27,0x8A,0x6F, 0x38,0x9B,0x80,
    0x49,0xAC,0x91, 0x5A,0xBD,0xA2, 0x6B,0xCE,0xB3, 0x7C,0xDF,0xC4,
    0x8D,0xF0,0xD5, 0x9E,0xFF,0xE5, 0xAF,0xFF,0xF1, 0xC0,0xFF,0xFD,

GREEN
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x10,0x6B,0x1B, 0x21,0x7C,0x2C, 0x32,0x8D,0x3D, 0x43,0x9E,0x4E,
    0x54,0xAF,0x5F, 0x65,0xC0,0x70, 0x76,0xD1,0x81, 0x87,0xE2,0x92,
    0x98,0xF3,0xA3, 0xA9,0xFF,0xB3, 0xBA,0xFF,0xBF, 0xCB,0xFF,0xCB,

YELLOW-GREEN
    0x00,0x23,0x0A, 0x00,0x34,0x10, 0x04,0x45,0x13, 0x15,0x56,0x13,
    0x26,0x67,0x13, 0x37,0x78,0x13, 0x48,0x89,0x14, 0x59,0x9A,0x25,
    0x6A,0xAB,0x36, 0x7B,0xBC,0x47, 0x8C,0xCD,0x58, 0x9D,0xDE,0x69,
    0xAE,0xEF,0x7A, 0xBF,0xFF,0x8B, 0xD0,0xFF,0x97, 0xE1,0xFF,0xA3,

ORANGE-GREEN
    0x00,0x17,0x07, 0x0E,0x28,0x08, 0x1F,0x39,0x08, 0x30,0x4A,0x08,
    0x41,0x5B,0x08, 0x52,0x6C,0x08, 0x63,0x7D,0x08, 0x74,0x8E,0x0D,
    0x85,0x9F,0x1E, 0x96,0xB0,0x2F, 0xA7,0xC1,0x40, 0xB8,0xD2,0x51,
    0xC9,0xE3,0x62, 0xDA,0xF4,0x73, 0xEB,0xFF,0x82, 0xFC,0xFF,0x8E,

LIGHT-ORANGE
    0x19,0x07,0x00, 0x2A,0x18,0x00, 0x3B,0x29,0x00, 0x4C,0x3A,0x00,
    0x5D,0x4B,0x00, 0x6E,0x5C,0x00, 0x7F,0x6D,0x00, 0x90,0x7E,0x09,
    0xA1,0x8F,0x1A, 0xB2,0xA0,0x2B, 0xC3,0xB1,0x3C, 0xD4,0xC2,0x4D,
    0xE5,0xD3,0x5E, 0xF6,0xE4,0x6F, 0xFF,0xF5,0x82, 0xFF,0xFF,0x96,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 26.7 SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x06,0x00, 0x54,0x17,0x00, 0x65,0x28,0x00,
    0x79,0x39,0x00, 0x87,0x4A,0x00, 0x98,0x5B,0x0C, 0xA9,0x6C,0x1D,
    0xBA,0x7D,0x2E, 0xCB,0x8E,0x3F, 0xDC,0x9F,0x50, 0xED,0xB0,0x61,
    0xFE,0xC1,0x72, 0xFF,0xD2,0x87, 0xFF,0xE3,0x9E, 0xFF,0xF4,0xB4,

RED-ORANGE
    0x3E,0x00,0x00, 0x4F,0x00,0x00, 0x60,0x07,0x00, 0x71,0x18,0x00,
    0x82,0x29,0x10, 0x93,0x3A,0x21, 0xA4,0x4B,0x32, 0xB5,0x5C,0x43,
    0xC6,0x6D,0x54, 0xD7,0x7E,0x65, 0xE8,0x8F,0x76, 0xF9,0xA0,0x87,
    0xFF,0xB1,0x9C, 0xFF,0xC2,0xB2, 0xFF,0xD3,0xC8, 0xFF,0xE4,0xDE,

PINK
    0x3E,0x00,0x09, 0x4F,0x00,0x15, 0x60,0x00,0x21, 0x71,0x0E,0x31,
    0x82,0x1F,0x42, 0x93,0x30,0x53, 0xA4,0x41,0x64, 0xB5,0x52,0x75,
    0xC6,0x63,0x86, 0xD7,0x74,0x97, 0xE8,0x85,0xA8, 0xF9,0x96,0xB9,
    0xFF,0xA7,0xCE, 0xFF,0xB8,0xE4, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,

PURPLE
    0x30,0x00,0x3D, 0x41,0x00,0x48, 0x52,0x00,0x54, 0x63,0x0C,0x64,
    0x74,0x1D,0x75, 0x85,0x2E,0x86, 0x96,0x3F,0x97, 0xA7,0x50,0xA8,
    0xB8,0x61,0xB9, 0xC9,0x72,0xCA, 0xDA,0x83,0xDB, 0xEB,0x94,0xE5,
    0xFC,0xA5,0xE5, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x18,0x00,0x62, 0x29,0x00,0x6E, 0x3A,0x01,0x7A, 0x4B,0x12,0x8B,
    0x5C,0x23,0x9C, 0x6D,0x34,0xAD, 0x7E,0x45,0xBE, 0x8F,0x56,0xCF,
    0xA0,0x67,0xE0, 0xB1,0x78,0xEE, 0xC2,0x89,0xEE, 0xD3,0x9A,0xEE,
    0xE4,0xAB,0xEE, 0xF5,0xBC,0xEE, 0xFF,0xCD,0xE0, 0xFF,0xDE,0xF5,

BLUE1
    0x00,0x00,0x72, 0x0C,0x00,0x7F, 0x1D,0x0E,0x8F, 0x2E,0x1F,0xA0,
    0x3F,0x30,0xB1, 0x50,0x41,0xC2, 0x61,0x52,0xD3, 0x72,0x63,0xE4,
    0x83,0x74,0xF5, 0x94,0x85,0xFA, 0xA5,0x96,0xFA, 0xB6,0xA7,0xFA,
    0xC7,0xB8,0xFA, 0xD8,0xC9,0xFA, 0xE9,0xDA,0xFA, 0xFA,0xE8,0xFA,

BLUE2
    0x00,0x00,0x62, 0x00,0x0F,0x77, 0x01,0x20,0x8D, 0x12,0x31,0x9E,
    0x23,0x42,0xAF, 0x34,0x53,0xC0, 0x45,0x64,0xD1, 0x56,0x75,0xE2,
    0x67,0x86,0xF3, 0x78,0x97,0xFF, 0x89,0xA8,0xFF, 0x9A,0xB9,0xFF,
    0xAB,0xCA,0xFF, 0xBC,0xDB,0xFF, 0xCD,0xEC,0xFF, 0xDE,0xFD,0xFF,

LIGHT-BLUE
    0x00,0x10,0x42, 0x00,0x21,0x58, 0x00,0x32,0x6E, 0x00,0x43,0x84,
    0x0E,0x54,0x96, 0x1F,0x65,0xA7, 0x30,0x76,0xB8, 0x41,0x87,0xC9,
    0x52,0x98,0xDA, 0x63,0xA9,0xEB, 0x74,0xBA,0xFC, 0x85,0xCB,0xFF,
    0x96,0xDC,0xFF, 0xA7,0xED,0xFF, 0xB8,0xFE,0xFF, 0xC9,0xFF,0xFF,

TURQUOISE
    0x00,0x1E,0x14, 0x00,0x2F,0x2A, 0x00,0x40,0x40, 0x00,0x51,0x56,
    0x04,0x62,0x6B, 0x15,0x73,0x7C, 0x26,0x84,0x8D, 0x37,0x95,0x9E,
    0x48,0xA6,0xAF, 0x59,0xB7,0xC0, 0x6A,0xC8,0xD1, 0x7B,0xD9,0xE2,
    0x8C,0xEA,0xF3, 0x9D,0xFB,0xFF, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x00,0x59,0x25,
    0x08,0x6A,0x38, 0x19,0x7B,0x49, 0x2A,0x8C,0x5A, 0x3B,0x9D,0x6B,
    0x4C,0xAE,0x7C, 0x5D,0xBF,0x8D, 0x6E,0xD0,0x9E, 0x7F,0xE1,0xAF,
    0x90,0xF2,0xC0, 0xA1,0xFF,0xD0, 0xB2,0xFF,0xDC, 0xC3,0xFF,0xE8,

GREEN
    0x00,0x26,0x0B, 0x00,0x37,0x10, 0x00,0x48,0x16, 0x08,0x59,0x18,
    0x19,0x6A,0x18, 0x2A,0x7B,0x18, 0x3B,0x8C,0x29, 0x4C,0x9D,0x3A,
    0x5D,0xAE,0x4B, 0x6E,0xBF,0x5C, 0x7F,0xD0,0x6D, 0x90,0xE1,0x7E,
    0xA1,0xF2,0x8F, 0xB2,0xFF,0x9F, 0xC3,0xFF,0xAB, 0xD4,0xFF,0xB7,

YELLOW-GREEN
    0x00,0x1E,0x09, 0x00,0x2F,0x0E, 0x11,0x40,0x0E, 0x22,0x51,0x0E,
    0x33,0x62,0x0E, 0x44,0x73,0x0E, 0x55,0x84,0x0E, 0x66,0x95,0x17,
    0x77,0xA6,0x28, 0x88,0xB7,0x39, 0x99,0xC8,0x4A, 0xAA,0xD9,0x5B,
    0xBB,0xEA,0x6C, 0xCC,0xFB,0x7D, 0xDD,0xFF,0x8A, 0xEE,0xFF,0x96,

ORANGE-GREEN
    0x0D,0x0F,0x01, 0x1E,0x20,0x01, 0x2F,0x31,0x01, 0x40,0x42,0x01,
    0x51,0x53,0x01, 0x62,0x64,0x01, 0x73,0x75,0x01, 0x84,0x86,0x08,
    0x95,0x97,0x19, 0xA6,0xA8,0x2A, 0xB7,0xB9,0x3B, 0xC8,0xCA,0x4C,
    0xD9,0xDB,0x5D, 0xEA,0xEC,0x6E, 0xFB,0xFD,0x7F, 0xFF,0xFF,0x8F,

LIGHT-ORANGE
    0x28,0x00,0x00, 0x39,0x0E,0x00, 0x4A,0x1F,0x00, 0x5B,0x30,0x00,
    0x6C,0x41,0x00, 0x7D,0x52,0x00, 0x8E,0x63,0x00, 0x9F,0x74,0x10,
    0xB0,0x85,0x21, 0xC1,0x96,0x32, 0xD2,0xA7,0x43, 0xE3,0xB8,0x54,
    0xF4,0xC9,0x65, 0xFF,0xDA,0x78, 0xFF,0xEB,0x8E, 0xFF,0xFC,0xA4,
*******************************************************************

*******************************************************************
    PALETTE - PHASE 27.2 SHIFT


GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xF7,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x05,0x00, 0x54,0x16,0x00, 0x65,0x27,0x00,
    0x76,0x38,0x00, 0x87,0X49,0x00, 0x98,0x5A,0x0C, 0xA9,0x6B,0x1D,
    0xBA,0x7C,0x2E, 0xCB,0x8D,0x3F, 0xDC,0x9E,0x50, 0xED,0xAF,0x61,
    0xFE,0xC0,0x72, 0xFF,0xD1,0x88, 0xFF,0xE2,0x9E, 0xFF,0xF3,0xB4,

RED-ORANGE
    0x3F,0x00,0x00, 0x50,0x00,0x00, 0x61,0x07,0x00, 0x72,0x18,0x01,
    0x83,0x29,0x12, 0x94,0x3A,0x23, 0xA5,0x4B,0x34, 0xB6,0x5C,0x45,
    0xC7,0x6D,0x56, 0xD8,0x7E,0x67, 0xE9,0x8F,0x78, 0xFA,0xA0,0x89,
    0xFF,0xB1,0x9E, 0xFF,0xC2,0xB4, 0xFF,0xD3,0xCA, 0xFF,0xE4,0xE0,

PINK
    0x3E,0x00,0x0C, 0x4F,0x00,0x18, 0x60,0x00,0x24, 0x71,0x0E,0x34,
    0x82,0x1F,0x45, 0x93,0x30,0x56, 0xA4,0x41,0x67, 0xB5,0x52,0x78,
    0xC6,0x63,0x89, 0xD7,0x74,0x9A, 0xE8,0x85,0xAB, 0xF9,0x96,0xB6,
    0xFF,0xA7,0xD1, 0xFF,0xB8,0xE7, 0xFF,0xC9,0xEF, 0xFF,0xDA,0xF4,

PURPLE
    0x2F,0x00,0x3F, 0x40,0x00,0x4B, 0x51,0x00,0x57, 0x62,0x0C,0x66,
    0x73,0x1D,0x77, 0x84,0x2E,0x88, 0x95,0x3F,0x99, 0xA6,0x50,0xAA,
    0xB7,0x61,0xBB, 0xC8,0x72,0xCC, 0xD9,0x83,0xDD, 0xEA,0x94,0xE5,
    0xFB,0xA5,0xE5, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x16,0x00,0x64, 0x27,0x00,0x70, 0x38,0x02,0x7D, 0x49,0x13,0x8E,
    0x5A,0x24,0x9F, 0x6B,0x35,0xB0, 0x7C,0x46,0xC1, 0x8D,0x57,0xD2,
    0x9E,0x68,0xE3, 0xAF,0x79,0xEF, 0xC0,0x8A,0xEF, 0xD1,0x9D,0xEF,
    0xE2,0xAC,0xEF, 0xF3,0xBD,0xEF, 0xFF,0xCE,0xF0, 0xFF,0xDF,0xF5,

BLUE1
    0x00,0x00,0x71, 0x09,0x00,0x7F, 0x1A,0x10,0x90, 0x2B,0x21,0xA1,
    0x3C,0x32,0xB2, 0x4D,0x43,0xC3, 0x5E,0x54,0xD4, 0x6F,0x65,0xE5,
    0x80,0x76,0xF6, 0x91,0x87,0xFC, 0xA2,0x98,0xFC, 0xB3,0xA9,0xFC,
    0xC4,0xBA,0xFC, 0xD5,0xCB,0xFC, 0xE6,0xDC,0xFC, 0xF7,0xED,0xFC,

BLUE2
    0x00,0x00,0x5E, 0x00,0x11,0x74, 0x00,0x22,0x8A, 0x0F,0x33,0x9C,
    0x20,0x44,0xAD, 0x31,0x55,0xBE, 0x42,0x66,0xCF, 0x53,0x77,0xE0,
    0x64,0x88,0xF1, 0x75,0x99,0xFF, 0x86,0xAA,0xFF, 0x97,0xBB,0xFF,
    0xA8,0xCC,0xFF, 0xB9,0xDD,0xFF, 0xCA,0xEE,0xFF, 0xDB,0xFF,0xFF,

LIGHT-BLUE
    0x00,0x12,0x3B, 0x00,0x23,0x51, 0x00,0x34,0x68, 0x00,0x45,0x7E,
    0x0C,0x56,0x90, 0x1D,0x67,0xA1, 0x2E,0x78,0xB2, 0x3F,0x89,0xC3,
    0x50,0x9A,0xD4, 0x61,0xAB,0xE5, 0x72,0xBC,0xF6, 0x83,0xCD,0xFF,
    0x94,0xDE,0xFF, 0xA5,0xEF,0xFF, 0xB6,0xFF,0xFF, 0xC7,0xFF,0xFF,

TURQUOISE
    0x00,0x20,0x0C, 0x00,0x31,0x22, 0x00,0x42,0x38, 0x00,0x53,0x4E,
    0x04,0x64,0x63, 0x15,0x75,0x74, 0x26,0x86,0x85, 0x37,0x97,0x96,
    0x48,0xA8,0xA7, 0x59,0xB9,0xB8, 0x6A,0xCA,0xC9, 0x7B,0xDB,0xDA,
    0x8C,0xEC,0xEB, 0x9D,0xFD,0xFC, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1C,
    0x0B,0x6B,0x2F, 0x1C,0x7C,0x40, 0x2D,0x8D,0x51, 0x3E,0x9E,0x62,
    0x4F,0xAF,0x73, 0x60,0xC0,0x84, 0x71,0xD1,0x95, 0x82,0xE2,0xA6,
    0x93,0xF3,0xB7, 0xA4,0xFF,0xC6, 0xB5,0xFF,0xD2, 0xC6,0xFF,0xDE,

GREEN
    0x00,0x25,0x0B, 0x00,0x36,0x10, 0x00,0x47,0x15, 0x0D,0x58,0x16,
    0x1E,0x69,0x16, 0x2F,0x7A,0x16, 0x40,0x8B,0x21, 0x51,0x9C,0x32,
    0x62,0xAD,0x43, 0x73,0xBE,0x54, 0x84,0xCF,0x65, 0x95,0xE0,0x76,
    0xA6,0xF1,0x87, 0xB7,0xFF,0x98, 0xC8,0xFF,0xA3, 0xD9,0xFF,0xAF,

YELLOW-GREEN
    0x00,0x1B,0x08, 0x06,0x2C,0x0B, 0x17,0x3D,0x0B, 0x28,0x4E,0x0B,
    0x39,0x5F,0x0B, 0x4A,0x70,0x0B, 0x5B,0x81,0x0B, 0x6C,0x92,0x12,
    0x7D,0xA3,0x23, 0x8E,0xB4,0x34, 0x9F,0xC5,0x45, 0xB0,0xD6,0x56,
    0xC1,0xE7,0x67, 0xD2,0xF8,0x78, 0xE3,0xFF,0x86, 0xF4,0xFF,0x92,

ORANGE-GREEN
    0x13,0x0B,0x00, 0x24,0x1C,0x00, 0x35,0x2D,0x00, 0x46,0x3E,0x00,
    0x57,0x4F,0x00, 0x68,0x60,0x00, 0x79,0x71,0x00, 0x8A,0x82,0x08,
    0x9B,0x93,0x19, 0xAC,0xA4,0x2A, 0xBD,0xB5,0x3B, 0xCE,0xC6,0x4C,
    0xDF,0xD7,0x5D, 0xF0,0xE8,0x6E, 0xFF,0xF9,0x7F, 0xFF,0xFF,0x92,

LIGHT-ORANGE
    0x2D,0x00,0x00, 0x3E,0x0A,0x00, 0x4F,0x1B,0x00, 0x60,0x2C,0x00,
    0x71,0x3D,0x00, 0x82,0x4E,0x00, 0x93,0x5F,0x05, 0xA4,0x70,0x16,
    0xB5,0x81,0x27, 0xC4,0x90,0x37, 0xD7,0xA3,0x49, 0xE8,0xB4,0x5A,
    0xF9,0xC5,0x6B, 0xFF,0xD6,0x80, 0xFF,0xE7,0x96, 0xFF,0xF8,0xAC,
*******************************************************************

*******************************************************************
    PALETTE - 27.7 PHASE SHIFT

GREY
    0x00,0x00,0x00, 0x11,0x11,0x11, 0x22,0x22,0x22, 0x33,0x33,0x33,
    0x44,0x44,0x44, 0x55,0x55,0x55, 0x66,0x66,0x66, 0x77,0x77,0x77,
    0x88,0x88,0x88, 0x99,0x99,0x99, 0xAA,0xAA,0xAA, 0xBB,0xBB,0xBB,
    0xCC,0xCC,0xCC, 0xDD,0xDD,0xDD, 0xEE,0xEE,0xEE, 0xFF,0xFF,0xFF,

GOLD
    0x1A,0x07,0x00, 0x2B,0x18,0x00, 0x3C,0x29,0x00, 0x4D,0x3A,0x00,
    0x5E,0x4B,0x00, 0x6F,0x5C,0x00, 0x80,0x6D,0x00, 0x91,0x7E,0x09,
    0xA2,0x8F,0x1A, 0xB3,0xA0,0x2B, 0xC4,0xB1,0x3C, 0xD5,0xC2,0x4D,
    0xE6,0xD3,0x5E, 0xF7,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

ORANGE
    0x32,0x00,0x00, 0x43,0x05,0x00, 0x54,0x16,0x00, 0x65,0x27,0x00,
    0x76,0x38,0x00, 0x87,0x49,0x00, 0x98,0x5A,0x0C, 0xA9,0x6B,0x1D,
    0xBA,0x7C,0x2E, 0xCB,0x8D,0x3F, 0xDC,0x9E,0x50, 0xED,0xAF,0x61,
    0xFE,0xC0,0x72, 0xFF,0xD1,0x88, 0xFF,0xE2,0x9E, 0xFF,0xF3,0xB4,

RED-ORANGE
    0x3F,0x00,0x00, 0x50,0x00,0x00, 0x61,0x06,0x00, 0x72,0x17,0x03,
    0x83,0x28,0x14, 0x94,0x39,0x25, 0xA5,0x4A,0x36, 0xB6,0x5B,0x47,
    0xC7,0x6C,0x58, 0xD8,0x7D,0x69, 0xE9,0x8E,0x7A, 0xFA,0x9F,0x8B,
    0xFF,0xB0,0x9F, 0xFF,0xC1,0xB5, 0xFF,0xD2,0xCB, 0xFF,0xE3,0xE1,

PINK
    0x3D,0x00,0x10, 0x4E,0x00,0x1C, 0x5F,0x00,0x27, 0x70,0x0D,0x37,
    0x81,0x1E,0x48, 0x92,0x2F,0x59, 0xA3,0x40,0x6A, 0xB4,0x51,0x7B,
    0xC5,0x62,0x8C, 0xD6,0x73,0x9D, 0xE7,0x84,0xAE, 0xF8,0x95,0xBF,
    0xFF,0xA6,0xD3, 0xFF,0xB7,0xE9, 0xFF,0xC8,0xEE, 0xFF,0xD9,0xF4,

PURPLE
    0x2D,0x00,0x42, 0x3E,0x00,0x4E, 0x4F,0x00,0x5A, 0x60,0x0C,0x6A,
    0x71,0x1D,0x7B, 0x82,0x2E,0x8C, 0x93,0x3F,0x9D, 0xA4,0x50,0xAE,
    0xB5,0x61,0xBF, 0xC6,0x72,0xD0, 0xD7,0x83,0xE1, 0xE8,0x94,0xE6,
    0xF9,0xA5,0xE6, 0xFF,0xB6,0xE9, 0xFF,0xC7,0xEE, 0xFF,0xD8,0xF3,

PURPLE-BLUE
    0x13,0x00,0x67, 0x24,0x00,0x73, 0x35,0x03,0x80, 0x46,0x14,0x91,
    0x57,0x25,0xA2, 0x68,0x36,0xB3, 0x79,0x47,0xC4, 0x8A,0x58,0xD5,
    0x9B,0x69,0xE6, 0xAC,0x7A,0xF0, 0xBD,0x8B,0xF0, 0xCE,0x9C,0xF0,
    0xDF,0xAD,0xF0, 0xF0,0xBE,0xF0, 0xFF,0xCF,0xF1, 0xFF,0xE0,0xF6,

BLUE1
    0x00,0x00,0x70, 0x05,0x01,0x80, 0x16,0x12,0x91, 0x27,0x23,0xA2,
    0x38,0x34,0xB3, 0x49,0x45,0xC4, 0x5A,0x56,0xD5, 0x6B,0x67,0xE6,
    0x7C,0x78,0xF7, 0x8D,0x89,0xFE, 0x9E,0x9A,0xFE, 0xAF,0xAB,0xFE,
    0xC0,0xBC,0xFE, 0xD1,0xCD,0xFE, 0xE2,0xDE,0xFE, 0xF3,0xEF,0xFE,

BLUE2
    0x00,0x03,0x5B, 0x00,0x14,0x71, 0x00,0x25,0x87, 0x0C,0x36,0x9A,
    0x1D,0x47,0xAB, 0x2E,0x58,0xBC, 0x3F,0x69,0xCD, 0x50,0x7A,0xDE,
    0x61,0x8B,0xEF, 0x72,0x9C,0xFF, 0x83,0xAD,0xFF, 0x94,0xBE,0xFF,
    0xA5,0xCF,0xFF, 0xB6,0xE0,0xFF, 0xC7,0xF1,0xFF, 0xD8,0xFF,0xFF,

LIGHT-BLUE
    0x00,0x15,0x35, 0x00,0x26,0x4B, 0x00,0x37,0x61, 0x00,0x48,0x78,
    0x0A,0x59,0x8B, 0x1B,0x6A,0x9C, 0x2C,0x7B,0xAD, 0x3D,0x8C,0xBE,
    0x4E,0x9D,0xCF, 0x5F,0xAE,0xE0, 0x70,0xBF,0xF1, 0x81,0xD0,0xFF,
    0x92,0xE1,0xFF, 0xA3,0xF2,0xFF, 0xB4,0xFF,0xFF, 0xC5,0xFF,0xFF,

TURQUOISE
    0x00,0x22,0x0A, 0x00,0x33,0x19, 0x00,0x44,0x2F, 0x00,0x55,0x45,
    0x04,0x66,0x5A, 0x15,0x77,0x6B, 0x26,0x88,0x7C, 0x37,0x99,0x8D,
    0x48,0xAA,0x9E, 0x59,0xBB,0xAF, 0x6A,0xCC,0xC0, 0x7B,0xDD,0xD1,
    0x8C,0xEE,0xE2, 0x9D,0xFF,0xF3, 0xAE,0xFF,0xFF, 0xBF,0xFF,0xFF,

GREEN-BLUE
    0x00,0x27,0x0C, 0x00,0x38,0x11, 0x00,0x49,0x16, 0x00,0x5A,0x1B,
    0x0D,0x6B,0x25, 0x1E,0x7C,0x36, 0x2F,0x8D,0x47, 0x40,0x9E,0x58,
    0x51,0xAF,0x69, 0x62,0xC0,0x7A, 0x73,0xD1,0x8B, 0x84,0xE2,0x9C,
    0x95,0xF3,0xAD, 0xA6,0xFF,0xBD, 0xB7,0xFF,0xC9, 0xC8,0xFF,0xD4,

GREEN
    0x00,0x24,0x0B, 0x00,0x35,0x10, 0x01,0x46,0x15, 0x12,0x57,0x15,
    0x23,0x68,0x15, 0x34,0x79,0x15, 0x45,0x8A,0x19, 0x56,0x9B,0x2A,
    0x67,0xAC,0x3B, 0x78,0xBD,0x4C, 0x89,0xCE,0x5D, 0x9A,0xDF,0x6E,
    0xAB,0xF0,0x7F, 0xBC,0xFF,0x8F, 0xCD,0xFF,0x9B, 0xDE,0xFF,0xA7,

YELLOW-GREEN
    0x00,0x18,0x07, 0x00,0x29,0x0C, 0x1E,0x3A,0x08, 0x2F,0x4B,0x08,
    0x40,0x5C,0x08, 0x51,0x6D,0x08, 0x62,0x7E,0x08, 0x73,0x8F,0x0D,
    0x84,0xA0,0x1E, 0x95,0xB1,0x2F, 0xA6,0xC2,0x40, 0xB7,0xD3,0x51,
    0xC8,0xE4,0x62, 0xD9,0xF5,0x73, 0xEA,0xFF,0x82, 0xFB,0xFF,0x8E,

ORANGE-GREEN
    0x1B,0x07,0x00, 0x2C,0x18,0x00, 0x3D,0x29,0x00, 0x4E,0x3A,0x00,
    0x5F,0x4B,0x00, 0x70,0x5C,0x00, 0x81,0x6D,0x00, 0x92,0x7E,0x09,
    0xA3,0x8F,0x1A, 0xB4,0xA0,0x2B, 0xC5,0xB1,0x3C, 0xD6,0xC2,0x4D,
    0xE7,0xD3,0x5E, 0xF8,0xE4,0x6F, 0xFF,0xF5,0x83, 0xFF,0xFF,0x97,

LIGHT-ORANGE
    0x33,0x00,0x00, 0x44,0x05,0x00, 0x55,0x16,0x00, 0x66,0x27,0x00,
    0x77,0x38,0x00, 0x88,0x49,0x00, 0x99,0x5A,0x0D, 0xAA,0x6B,0x1E,
    0xBB,0x7C,0x2F, 0xCC,0x8D,0x40, 0xDD,0x9E,0x51, 0xEE,0xAF,0x62,
    0xFF,0xC0,0x73, 0xFF,0xD1,0x89, 0xFF,0xE2,0x9F, 0xFF,0xF3,0xB5
*******************************************************************/

/**************************************************************
 *
 *    Memory setup
 *
 **************************************************************/

void a400_state::cart_rd4_w(int state)
{
	m_cart_rd4_enabled = state;
	m_cart_rd4_view.select(m_cart_rd4_enabled);
}

void a400_state::cart_rd5_w(int state)
{
	m_cart_rd5_enabled = state;
	m_cart_rd5_view.select(m_cart_rd5_enabled);
}

TIMER_DEVICE_CALLBACK_MEMBER( a400_state::a400_interrupt )
{
	m_antic->generic_interrupt(4);
}

TIMER_DEVICE_CALLBACK_MEMBER( a1200xl_state::xl_interrupt )
{
	m_antic->generic_interrupt(2);
}

TIMER_DEVICE_CALLBACK_MEMBER( a5200_state::a5200_interrupt )
{
	m_antic->generic_interrupt(4);
}

void a400_state::machine_start()
{
	save_item(NAME(m_cart_rd4_enabled));
	save_item(NAME(m_cart_rd5_enabled));

	if (m_sio.found())
		m_sio->ready_w(1);
}

void a1200xl_state::machine_start()
{
	a400_state::machine_start();
	save_item(NAME(m_mmu));
}

void xegs_state::machine_start()
{
	a1200xl_state::machine_start();
	m_bank->configure_entries(0, 2, memregion("maincpu")->base() + 0x8000, 0x2000);
}

void a400_state::machine_reset()
{
	m_pokey->write(15, 0);

	// TODO: stub reset state
	if (!m_cartleft->exists())
	{
		m_cart_rd4_enabled = 0;
		m_cart_rd5_enabled = 0;
	}
	else
		std::tie(m_cart_rd4_enabled, m_cart_rd5_enabled) = m_cartleft->get_initial_rd_state();

	m_cart_rd4_view.select(m_cart_rd4_enabled);
	m_cart_rd5_view.select(m_cart_rd5_enabled);
}

void a800_state::machine_reset()
{
	a400_state::machine_reset();

	// TODO: stub reset state, verify if any can be run stand-alone
/*
    if (!m_cartright->exists())
    {
        m_cart_rd4_enabled = false;
        m_cart_rd4_view.select(0);
    }*/
}

void a1200xl_state::machine_reset()
{
	a400_state::machine_reset();
	m_mmu = 0x83;
	m_kernel_view.select(1);
	m_selftest_view.select(0);
}

void a800xl_state::machine_reset()
{
	a1200xl_state::machine_reset();
	m_basic_view.select(0);
}

void xegs_state::machine_reset()
{
	a800xl_state::machine_reset();
	m_bank->set_entry(0);
}

void a130xe_state::machine_reset()
{
	a800xl_state::machine_reset();
	m_ext_view.select(0);
	m_ext_bank->set_bank(3);
}

void a5200_state::machine_start()
{
}

void a5200_state::machine_reset()
{
	m_pokey->write(15, 0);
}

/**************************************************************
 *
 * GTIA interface
 *
 **************************************************************/

void a400_state::gtia_cb(uint8_t data)
{
	m_dac->write(BIT(data, 3));
}

/**************************************************************
 *
 * PIA interface
 *
 **************************************************************/

uint8_t a400_state::djoy_0_1_r()
{
	return (m_ctrl[0]->read_joy() & 0x0f) | (m_ctrl[1]->read_joy() & 0x0f) << 4;
}

void a400_state::djoy_0_1_w(uint8_t data)
{
	m_ctrl[0]->joy_w(data & 0x0f);
	m_ctrl[1]->joy_w(data >> 4);
}

uint8_t a400_state::djoy_2_3_r()
{
	return (m_ctrl[2]->read_joy() & 0x0f) | (m_ctrl[3]->read_joy() & 0x0f) << 4;
}

void a400_state::djoy_2_3_w(uint8_t data)
{
	m_ctrl[2]->joy_w(data & 0x0f);
	m_ctrl[3]->joy_w(data >> 4);
}

uint8_t a400_state::djoy_b_r()
{
	uint8_t b = 0;
	for (int i = 0; i < 4; i++)
		if (!m_ctrl[i].found() || BIT(m_ctrl[i]->read_joy(), 5))
			b |= 1 << i;
	return b;
}

uint8_t a1200xl_state::djoy_b_r()
{
	uint8_t b = 0;
	for (int i = 0; i < 2; i++)
		if (!m_ctrl[i].found() || BIT(m_ctrl[i]->read_joy(), 5))
			b |= 1 << i;

	// TODO: TRIG2 (0) for XEGS keyboard disconnected, always 1 otherwise
	b |= 1 << 2;
	// TODO: TRIG3 should really read from RD5 instead
	// - non-64KB SDX boot (will hang in kernel during POST by boot conflict with BASIC if not handled properly)
	// - a1200xl cart boot (expects this to be reversed)
	// - returning m_cartleft->exists() makes BASIC disable thru OPTION to not work and makes maxflash to fail loading most .xex
//  b |= m_cart_rd5_enabled << 3;
	b |= !m_cartleft->exists() << 3;

	return b;
}

void a1200xl_state::pia_portb_w(uint8_t data)
{
	const u8 dir = m_pia->port_b_z_mask();

	// On a1200xl BIOS jumps from PC=0xc550, expecting ROM to be still
	// enabled at PC=0xe40c, needing direction to work properly.
	const u8 new_mmu = (data & ~dir) | (m_mmu & dir);

	if (m_mmu != new_mmu)
	{
		m_mmu = new_mmu;
		portb_cb(m_mmu);
	}
}

/*
 * x--- ---- /DRE (1) 0x5000-0x57ff self-test ROM enabled if also kernel ROM enabled
 *                (0) RAM or NOP
 * ---- --x- /BAE (0) 0xa000-0xbfff BASIC ROM enabled (1) disabled
 * ---- ---x /OSE (1) 0xd800-0xffff kernel ROM enable (0) RAM
 */
void a800xl_state::portb_cb(uint8_t data)
{
	m_kernel_view.select(BIT(data, 0));
	m_basic_view.select(!BIT(data, 1));
	m_selftest_view.select((data & 0x81) == 0x01);
}

/*
 * same as a800xl plus:
 * ---- xx-- LED states (specific for this variant only)
 * Note: basic ROM not present so bit 1 doesn't do anything
 */
void a1200xl_state::portb_cb(uint8_t data)
{
	m_kernel_view.select(BIT(data, 0));
	m_selftest_view.select((data & 0x81) == 0x01);
}

/*
 * same as a800xl plus:
 * -x-- ---- (0) enables Missile Command ROM at 0xa000-0xbfff, ignored if BASIC ROM bit 1 also 0
 *           (BASIC has higher priority)
 *
 * To access Missile Command with a keyboard connected
 * hold console SELECT switch while booting the machine
 */
void xegs_state::portb_cb(uint8_t data)
{
	m_kernel_view.select(BIT(data, 0));
	const bool game_rom = !BIT(data, 6);
	const bool basic_rom = !BIT(data, 1);
	if (game_rom || basic_rom)
	{
		m_basic_view.select(1);
		m_bank->set_entry(basic_rom);
	}
	else
		m_basic_view.select(0);
	m_selftest_view.select((data & 0x81) == 0x01);
}

/*
 * same as a800xl plus:
 * --x- ---- ANTIC extended memory access enable
 * ---x ---- CPU extended memory access enable
 * ---- xx-- 0x4000-0x7fff extended RAM bank select
 */
void a130xe_state::portb_cb(uint8_t data)
{
	a800xl_state::portb_cb(data);
	if (!BIT(data, 4))
	{
		m_ext_view.select(1);
		const bool selftest_enabled = (data & 0x81) == 0x01;
		m_ext_bank->set_bank((data & 0xc) >> 2 | (selftest_enabled << 2));
	}
	else
		m_ext_view.select(0);

	// TODO: ANTIC extended memory access enable
	// Prince of Persia homebrew has it always on.
	// May be used for the intro screen (where text don't draw at the time of this writing)
}

// TODO: propagate portb to RAMBO XL and COMPY sub-devices for a800xl and onward
/*
 * For RAMBO, same as a130xe plus:
 * -xx- ---- A2-A3 select bits for RAM bank
 * ---x ---- /RAME (1) 0x4000-0x7fff selects main system RAM (0) extended RAM access
 * NB: if A0-A3 selects banks $0-$3 then main system RAM is used,
 *     only $4-$f and onward really accesses the sub-board RAM installed.
 */

/**************************************************************
 *
 * Machine drivers
 *
 **************************************************************/

// note: both screen setups are actually non-interlaced, and always 240 lines
void atari_common_state::config_ntsc_screen(machine_config &config)
{
	// 15.69975KHz x 59.9271 Hz
	m_screen->set_raw(XTAL(14'318'181), 912, antic_device::MIN_X, antic_device::MAX_X, 262, antic_device::MIN_Y, antic_device::MAX_Y);
	m_gtia->set_region(GTIA_NTSC);
}

void atari_common_state::config_pal_screen(machine_config &config)
{
	// 15.55655KHz x 49.86074 Hz, master clock rated at 14.18757 MHz
	// TODO: confirm hsync
	m_screen->set_raw(XTAL(3'546'800) * 4, 912, antic_device::MIN_X, antic_device::MAX_X, 312, antic_device::MIN_Y, antic_device::MAX_Y);
	m_gtia->set_region(GTIA_PAL);
}

void a400_state::atari_common_nodac(machine_config &config)
{
	/* basic machine hardware */
	M6502(config, m_maincpu, pokey_device::FREQ_17_EXACT);

	/* video hardware */
	SCREEN(config, m_screen, SCREEN_TYPE_RASTER);
	m_screen->set_screen_update("antic", FUNC(antic_device::screen_update));
	m_screen->set_palette("palette");
//  m_screen->set_video_attributes(VIDEO_UPDATE_SCANLINE);

	PALETTE(config, "palette", FUNC(a400_state::a400_palette), std::size(atari_colors) / 3);

	/* sound hardware */
	SPEAKER(config, "speaker").front_center();
	POKEY(config, m_pokey, pokey_device::FREQ_17_EXACT);
	m_pokey->set_keyboard_callback(FUNC(a400_state::a800_keyboard));
	m_pokey->irq_w().set_inputline(m_maincpu, m6502_device::IRQ_LINE);
	m_pokey->add_route(ALL_OUTPUTS, "speaker", 1.0);
}

void a400_state::atari_common(machine_config &config)
{
	atari_common_nodac(config);

	INPUT_MERGER_ANY_HIGH(config, "mainirq").output_handler().set_inputline(m_maincpu, m6502_device::IRQ_LINE);
	m_pokey->irq_w().set("mainirq", FUNC(input_merger_device::in_w<0>));
	m_pokey->pot_r<0>().set(m_ctrl[0], FUNC(vcs_control_port_device::read_pot_y));
	m_pokey->pot_r<1>().set(m_ctrl[0], FUNC(vcs_control_port_device::read_pot_x));
	m_pokey->pot_r<2>().set(m_ctrl[1], FUNC(vcs_control_port_device::read_pot_y));
	m_pokey->pot_r<3>().set(m_ctrl[1], FUNC(vcs_control_port_device::read_pot_x));
	m_pokey->pot_r<4>().set(m_ctrl[2], FUNC(vcs_control_port_device::read_pot_y));
	m_pokey->pot_r<5>().set(m_ctrl[2], FUNC(vcs_control_port_device::read_pot_x));
	m_pokey->pot_r<6>().set(m_ctrl[3], FUNC(vcs_control_port_device::read_pot_y));
	m_pokey->pot_r<7>().set(m_ctrl[3], FUNC(vcs_control_port_device::read_pot_x));
	m_pokey->oclk_w().set(m_sio, FUNC(a8sio_device::clock_out_w));
	m_pokey->sod_w().set(m_sio, FUNC(a8sio_device::data_out_w));

	DAC_1BIT(config, "dac", 0).add_route(ALL_OUTPUTS, "speaker", 0.03);

	/* internal ram */
	RAM(config, m_ram).set_default_size("48K");

	ATARI_GTIA(config, m_gtia, 0);
	m_gtia->read_callback().set_ioport("console");
	m_gtia->write_callback().set(FUNC(a400_state::gtia_cb));
	m_gtia->trigger_callback().set(FUNC(a400_state::djoy_b_r));

	ATARI_ANTIC(config, m_antic, 0);
	m_antic->set_gtia_tag(m_gtia);

	/* devices */
	PIA6821(config, m_pia);
	m_pia->readpa_handler().set(FUNC(a400_state::djoy_0_1_r));
	m_pia->writepa_handler().set(FUNC(a400_state::djoy_0_1_w));
	m_pia->readpb_handler().set(FUNC(a400_state::djoy_2_3_r));
	m_pia->writepb_handler().set(FUNC(a400_state::djoy_2_3_w));
	m_pia->ca2_handler().set(m_sio, FUNC(a8sio_device::motor_w));
	m_pia->cb2_handler().set(m_sio, FUNC(a8sio_device::command_w));
	m_pia->irqa_handler().set("mainirq", FUNC(input_merger_device::in_w<1>));
	m_pia->irqb_handler().set("mainirq", FUNC(input_merger_device::in_w<2>));

	a8sio_device &sio(A8SIO(config, m_sio, "fdc"));
	//sio.clock_in().set(m_pokey, FUNC(pokey_device::bclk_w));
	sio.data_in().set(m_pokey, FUNC(pokey_device::sid_w));
	sio.proceed().set(m_pia, FUNC(pia6821_device::ca1_w));
	sio.interrupt().set(m_pia, FUNC(pia6821_device::cb1_w));

	A800_CART_SLOT(config, m_cartleft, a800_left, nullptr);
	m_cartleft->rd4_callback().set(FUNC(a400_state::cart_rd4_w));
	m_cartleft->rd5_callback().set(FUNC(a400_state::cart_rd5_w));

	/* software lists */
	SOFTWARE_LIST(config, "flop_list").set_original("a800_flop");
	SOFTWARE_LIST(config, "cass_list").set_original("a800_cass");
	SOFTWARE_LIST(config, "cart_list").set_original("a800");
	SOFTWARE_LIST(config, "xegs_list").set_compatible("xegs");

	VCS_CONTROL_PORT(config, m_ctrl[0], a800_control_port_devices, "joy");
	VCS_CONTROL_PORT(config, m_ctrl[1], a800_control_port_devices, "joy");
	VCS_CONTROL_PORT(config, m_ctrl[2], a800_control_port_devices, "joy");
	VCS_CONTROL_PORT(config, m_ctrl[3], a800_control_port_devices, "joy");
}

// memory map A400 + NTSC screen
void a400_state::a400(machine_config &config)
{
	atari_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a400_state::a400_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a400_state::a400_interrupt), "screen", 0, 1);

	config_ntsc_screen(config);

	// TODO: confirm all of the official config
	// 4KB was the release model,
	// shouldn't be possible to reach 48K but added as convenience for now.
	m_ram->set_default_size("16K");
	m_ram->set_extra_options("4K,48K");
}

// memory map A400 + PAL screen
void a400_state::a400pal(machine_config &config)
{
	a400(config);

	config_pal_screen(config);
}

// memory map A800 + NTSC screen + Right cartslot
void a800_state::a800(machine_config &config)
{
	atari_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a800_state::a800_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a800_state::a400_interrupt), "screen", 0, 1);

	config_ntsc_screen(config);

	A800_CART_SLOT(config, m_cartright, a800_right, nullptr);

	// TODO: confirm all of the official config
	// 8K release model, 48K possible here?
	m_ram->set_default_size("16K");
	m_ram->set_extra_options("8K,48K");
}


// memory map A800 + PAL screen + Right cartslot
void a800_state::a800pal(machine_config &config)
{
	a800(config);

	config_pal_screen(config);
}

// memory map A1200XL+ MMU via PIA portB
void a1200xl_state::atari_xl_common(machine_config &config)
{
	atari_common(config);

	TIMER(config, "scantimer").configure_scanline(FUNC(a1200xl_state::xl_interrupt), "screen", 0, 1);

	m_pokey->pot_r<4>().set_constant(0xff);
	m_pokey->pot_r<5>().set_constant(0xff);
	m_pokey->pot_r<6>().set_constant(0xff);
	m_pokey->pot_r<7>().set_constant(0xff);

	m_pia->readpb_handler().set_constant(0x83);
	m_pia->writepb_handler().set(FUNC(a1200xl_state::pia_portb_w));

	config_ntsc_screen(config);

	m_ram->set_default_size("16K");

	config.device_remove("ctrl3");
	config.device_remove("ctrl4");
}

void a1200xl_state::a1200xl(machine_config &config)
{
	atari_xl_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a1200xl_state::a1200xl_mem);

	m_pokey->pot_r<4>().set_ioport("J1");
	m_pokey->pot_r<5>().set_ioport("J2");
	m_pokey->pot_r<6>().set_ioport("J3");
	m_pokey->pot_r<7>().set_ioport("J4");
	// TODO: console LEDs for this model only

	// retail has 64KB minimum
	m_ram->set_default_size("64K");
//  m_ram->set_extra_options("16K,32K,48K");
}

// memory map A800XL + NTSC screen + MMU via PIA portB
void a800xl_state::a800xl(machine_config &config)
{
	atari_xl_common(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a800xl_state::a800xl_mem);

	m_ram->set_default_size("64K");
}

// memory map A600XL (same as 800XL but less RAM) + NTSC screen + MMU via PIA portB
void a600xl_state::a600xl(machine_config &config)
{
	a800xl(config);

	m_ram->set_default_size("16K");
	m_ram->set_extra_options("32K,48K,64K");
}

// memory map A800XL + PAL screen + MMU via PIA portB
void a800xl_state::a800xlpal(machine_config &config)
{
	a800xl(config);

	m_maincpu->set_clock(1773000);

	config_pal_screen(config);

	m_pokey->set_clock(1773000);
}

// memory map A130XE
void a130xe_state::a130xe(machine_config &config)
{
	a800xl(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a130xe_state::a130xe_mem);

	// minimum RAM 128KB, further sizes with optional modules
	m_ram->set_default_size("128K");

	ADDRESS_MAP_BANK(config, m_ext_bank).set_map(&a130xe_state::extram_map).set_options(ENDIANNESS_LITTLE, 8, 16 + 1, 0x4000);
}

// memory map XEGS
void xegs_state::xegs(machine_config &config)
{
	a800xl(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &xegs_state::xegs_mem);

	// not installable unless with DIY mods
	config.device_remove("ram");

//  m_cartleft->set_default_option("xegs");
	m_cartleft->set_is_xegs(true);

	// uses exact same slot connector
	SOFTWARE_LIST(config.replace(), "cart_list").set_compatible("a800");
	SOFTWARE_LIST(config.replace(), "xegs_list").set_original("xegs");
}

// memory map A5200, different ports, less RAM
void a5200_state::a5200(machine_config &config)
{
	atari_common_nodac(config);

	m_maincpu->set_addrmap(AS_PROGRAM, &a5200_state::a5200_mem);
	TIMER(config, "scantimer").configure_scanline(FUNC(a5200_state::a5200_interrupt), "screen", 0, 1);

	// Not used but exposed via expansion port
	//m_pokey->serin_r().set_constant(0);
	//m_pokey->serout_w().set_nop();
	m_pokey->pot_r<0>().set_ioport("analog_0");
	m_pokey->pot_r<1>().set_ioport("analog_1");
	m_pokey->pot_r<2>().set_ioport("analog_2");
	m_pokey->pot_r<3>().set_ioport("analog_3");
	m_pokey->pot_r<4>().set_ioport("analog_4");
	m_pokey->pot_r<5>().set_ioport("analog_5");
	m_pokey->pot_r<6>().set_ioport("analog_6");
	m_pokey->pot_r<7>().set_ioport("analog_7");
	m_pokey->set_keyboard_callback(FUNC(a5200_state::a5200_keypads));
	m_pokey->add_route(ALL_OUTPUTS, "speaker", 1.0);

	ATARI_GTIA(config, m_gtia, 0);
	m_gtia->trigger_callback().set_ioport("djoy_b");

	ATARI_ANTIC(config, m_antic, 0);
	m_antic->set_gtia_tag(m_gtia);

	config_ntsc_screen(config);

	A5200_CART_SLOT(config, m_cart, a5200_carts, nullptr);

	/* Software lists */
	SOFTWARE_LIST(config, "cart_list").set_original("a5200");

	// not installable like at all?
	//config.device_remove("ram");
}

/*
From Analog Computing Magazine, issue 16 (1984-02):
  Newer releases of the 5200 incorporate some minor hardware changes.
  Controller ports 3 and 4 have been eliminated, making POT4 through
  POT7, TRIG2, TRIG3, and bit 1 of CONSOL useless. A few of the
  connector pins have been redefined. Pin 2 of the I/O expansion
  connector now carries POKEY's Audio Out signal. Three pins on the
  cartridge connector have changed to accomodate the new 2600 adapter.
  The system clock, 02, is output on pin 14, isolated through a diode.
  An alternate video input is taken from pin 24 and is also isolated
  through a diode. Pin 30 provides an alternate audio input.

  There is space on the newer boards for circuitry for a PAL (European
  TV standard) version of the 5200. Also, on power-up, the monitor
  program checks for the PAL version by examining the GTIA register PAL
  after step 2 of the initialization routine. It also checks the
  cartridge program for PAL compatibility. The byte at $BFE7 should
  read $02 if compatible, or $00 if not. This is the only important
  change to the monitor program. There are some additional hardware
  changes, but none affects the machine's operation from the
  programmer's view.
*/
void a5200_state::a5200a(machine_config &config)
{
	a5200(config);

	m_pokey->pot_r<4>().set_constant(0);
	m_pokey->pot_r<5>().set_constant(0);
	m_pokey->pot_r<6>().set_constant(0);
	m_pokey->pot_r<7>().set_constant(0);
}


/**************************************************************
 *
 * ROM loading
 *
 **************************************************************/

ROM_START( a400 )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. B")
	ROMX_LOAD( "co12499b.rom",  0xe000, 0x1000, BAD_DUMP CRC(d818f3e8) SHA1(bcdec2188f6a6a5bfc1df4e383bd828d34b5c4ac), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599b.rom",  0xf000, 0x1000, BAD_DUMP CRC(c1690a9b) SHA1(c5248e8565574fd39ae1c3f4f356aa4cac07df95), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "reva", "OS Rev. A")
	ROMX_LOAD( "co12499a.rom",  0xe000, 0x1000, BAD_DUMP CRC(29f64e17) SHA1(abf7ec488c6b600f1b7f30bdc7f8a2bf6a727675), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599a.rom",  0xf000, 0x1000, BAD_DUMP CRC(bc533f0c) SHA1(e217148495fa747fe5488132d8d22533e68c7e58), ROM_BIOS(1) )    // CRC and label waiting for confirmation
ROM_END

ROM_START( a400pal )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_LOAD( "co15199.rom", 0xe000, 0x1000, BAD_DUMP CRC(8e547f56) SHA1(1bd746ea798b723bfb18495a7facca113183d713) )    // Rev. A - CRC and label waiting for confirmation
	ROM_LOAD( "co15299.rom", 0xf000, 0x1000, BAD_DUMP CRC(be55b413) SHA1(d88afae49b08e75943d0258cb580e5d34756414a) )    // Rev. A - CRC and label waiting for confirmation
ROM_END

ROM_START( a800 )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. B")
	ROMX_LOAD( "co12499b.rom",  0xe000, 0x1000, BAD_DUMP CRC(d818f3e8) SHA1(bcdec2188f6a6a5bfc1df4e383bd828d34b5c4ac), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599b.rom",  0xf000, 0x1000, BAD_DUMP CRC(c1690a9b) SHA1(c5248e8565574fd39ae1c3f4f356aa4cac07df95), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "reva", "OS Rev. A")
	ROMX_LOAD( "co12499a.rom",  0xe000, 0x1000, BAD_DUMP CRC(29f64e17) SHA1(abf7ec488c6b600f1b7f30bdc7f8a2bf6a727675), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co14599a.rom",  0xf000, 0x1000, BAD_DUMP CRC(bc533f0c) SHA1(e217148495fa747fe5488132d8d22533e68c7e58), ROM_BIOS(1) )    // CRC and label waiting for confirmation
ROM_END

ROM_START( a800pal )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co12399b.rom", 0xd800, 0x0800, CRC(6a5d766e) SHA1(01a6044f7a81d409c938e7dfde0a1af5832229d2) )
	ROM_LOAD( "co15199.rom", 0xe000, 0x1000, BAD_DUMP CRC(8e547f56) SHA1(1bd746ea798b723bfb18495a7facca113183d713) )    // Rev. A - CRC and label waiting for confirmation
	ROM_LOAD( "co15299.rom", 0xf000, 0x1000, BAD_DUMP CRC(be55b413) SHA1(d88afae49b08e75943d0258cb580e5d34756414a) )    // Rev. A - CRC and label waiting for confirmation
ROM_END

ROM_START( a1200xl )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_SYSTEM_BIOS(0, "default", "OS Rev. 11")
	ROMX_LOAD( "co60616b.rom", 0xc000, 0x2000, BAD_DUMP CRC(6e29ec8d) SHA1(3f9c06d6b4d261f3d5bf4354e3cff0c17b9347b9), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co60617b.rom", 0xe000, 0x2000, BAD_DUMP CRC(d73ce29a) SHA1(64790242d902643fe0c40dd842749f1fe461831b), ROM_BIOS(0) )    // CRC and label waiting for confirmation
	ROM_SYSTEM_BIOS(1, "rev10", "OS Rev. 10")
	ROMX_LOAD( "co60616a.rom", 0xc000, 0x2000, BAD_DUMP CRC(0391386b) SHA1(7c176657c88b89b8a69bf021fa8e0939efc0dff2), ROM_BIOS(1) )    // CRC and label waiting for confirmation
	ROMX_LOAD( "co60617a.rom", 0xe000, 0x2000, BAD_DUMP CRC(b502f1e7) SHA1(6688db57d97fa570aef5c15cef3e5fb2688879c2), ROM_BIOS(1) )    // CRC and label waiting for confirmation
ROM_END

ROM_START( a600xl )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co60302a.rom", 0xa000, 0x2000, CRC(f0202fb3) SHA1(7ad88dd99ff4a6ee66f6d162074db6f8bef7a9b6) )    // Rev. B
	ROM_LOAD( "co62024.rom",  0xc000, 0x4000, CRC(643bcc98) SHA1(881d030656b40bbe48f15a696b28f22c0b752ab0) )    // Rev. 1
ROM_END

ROM_START( a800xl )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co60302a.rom", 0xa000, 0x2000, CRC(f0202fb3) SHA1(7ad88dd99ff4a6ee66f6d162074db6f8bef7a9b6) )   // Rev. B
	ROM_LOAD( "co61598b.rom", 0xc000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2
ROM_END

#define rom_a800xlp rom_a800xl

ROM_START( a65xe )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0xa000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "co61598b.rom", 0xc000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2
ROM_END

ROM_START( a65xea )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "basic_ar.rom", 0xa000, 0x2000, CRC(c899f4d6) SHA1(043df191d1fe402e792266a108e147ffcda35130) )   // is this correct? or shall we use Rev. C?
//  ROM_LOAD( "c101700.rom",  0xc000, 0x4000, CRC(7f9a76c8) SHA1(57eb6d87850a763f11767f53d4eaede186f831a2) )   // this was from Savetz and has wrong bits!
	ROM_LOAD( "c101700.rom",  0xc000, 0x4000, CRC(45f47988) SHA1(a36b8b20f657580f172749bb0625c08706ed824c) )   // Rev. 3B ?
ROM_END

ROM_START( a130xe )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0xa000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "co61598b.rom", 0xc000, 0x4000, CRC(1f9cd270) SHA1(ae4f523ba08b6fd59f3cae515a2b2410bbd98f55) )   // Rev. 2
ROM_END

ROM_START( a800xe )
	ROM_REGION(0x10000, "maincpu", 0)
	ROM_LOAD( "co24947a.rom", 0xa000, 0x2000, CRC(7d684184) SHA1(3693c9cb9bf3b41bae1150f7a8264992468fc8c0) )   // Rev. C
	ROM_LOAD( "c300717.rom",  0xc000, 0x4000, CRC(29f133f7) SHA1(f03b9b93000ee84abb9cf8d6367241006f172182) )   // Rev. 3
ROM_END

ROM_START( xegs )
	ROM_REGION(0x10000, "maincpu", ROMREGION_ERASE00)
	ROM_LOAD( "c101687.rom", 0x8000, 0x8000, CRC(d50260d1) SHA1(0e0625ab2473f8431640df3ac8af61925760b9b9) )    // Rev. C + Rev. 4 + Missile Command
ROM_END


ROM_START( a5200 )
	ROM_REGION(0x10000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD( "co19156.u8", 0xf800, 0x0800, CRC(4248d3e3) SHA1(6ad7a1e8c9fad486fbec9498cb48bf5bc3adc530) )
ROM_END

ROM_START( a5200a )
	ROM_REGION(0x10000, "maincpu", ROMREGION_ERASEFF)
	ROM_SYSTEM_BIOS(0, "default", "2-Port")
	ROMX_LOAD( "co19156a.a8", 0xf800, 0x0800, CRC(c2ba2613) SHA1(1d2a3f00109d75d2d79fecb565775eb95b7d04d5), ROM_BIOS(0) ) // This breaks "K-Razy Shoot-Out", "Mountain King", and "Pitfall!"
	ROM_SYSTEM_BIOS(1, "4port", "4-Port")
	ROMX_LOAD( "co19156.u8", 0xf800, 0x0800, CRC(4248d3e3) SHA1(6ad7a1e8c9fad486fbec9498cb48bf5bc3adc530), ROM_BIOS(1) )
ROM_END


/**************************************************************
 *
 * Game driver(s)
 *
 **************************************************************/

/*     YEAR  NAME    PARENT  COMPAT  MACHINE    INPUT   CLASS       INIT        COMPANY  FULLNAME */
COMP( 1979, a400,    0,      0,      a400,      a800,   a400_state,   empty_init, "Atari", "Atari 400 (NTSC)",     0)
COMP( 1979, a400pal, a400,   0,      a400pal,   a800,   a400_state,   empty_init, "Atari", "Atari 400 (PAL)",      0)
COMP( 1979, a800,    0,      0,      a800,      a800,   a800_state,   empty_init, "Atari", "Atari 800 (NTSC)",     0)
COMP( 1979, a800pal, a800,   0,      a800pal,   a800,   a800_state,   empty_init, "Atari", "Atari 800 (PAL)",      0)
COMP( 1982, a1200xl, a800,   0,      a1200xl,   a1200xl,a1200xl_state, empty_init, "Atari", "Atari 1200XL",         MACHINE_NOT_WORKING )      // 64k RAM
COMP( 1983, a600xl,  a800xl, 0,      a600xl,    a800,   a600xl_state, empty_init, "Atari", "Atari 600XL",          MACHINE_IMPERFECT_GRAPHICS )      // 16k RAM
COMP( 1983, a800xl,  0,      0,      a800xl,    a800,   a800xl_state, empty_init, "Atari", "Atari 800XL (NTSC)",   MACHINE_IMPERFECT_GRAPHICS )      // 64k RAM
COMP( 1983, a800xlp, a800xl, 0,      a800xlpal, a800,   a800xl_state, empty_init, "Atari", "Atari 800XL (PAL)",    MACHINE_IMPERFECT_GRAPHICS )      // 64k RAM
COMP( 1986, a65xe,   a800xl, 0,      a800xl,    a800,   a800xl_state, empty_init, "Atari", "Atari 65XE",           MACHINE_IMPERFECT_GRAPHICS )      // 64k RAM
COMP( 1986, a65xea,  a800xl, 0,      a800xl,    a800,   a800xl_state, empty_init, "Atari", "Atari 65XE (Arabic)",  MACHINE_NOT_WORKING )
COMP( 1986, a130xe,  a800xl, 0,      a130xe,    a800,   a130xe_state, empty_init, "Atari", "Atari 130XE",          MACHINE_NOT_WORKING )      // 128k RAM
COMP( 1986, a800xe,  a800xl, 0,      a800xl,    a800,   a800xl_state, empty_init, "Atari", "Atari 800XE",          MACHINE_IMPERFECT_GRAPHICS )      // 64k RAM
COMP( 1987, xegs,    0,      0,      xegs,      a800,   xegs_state,   empty_init, "Atari", "Atari XE Game System", MACHINE_IMPERFECT_GRAPHICS )  // 64k RAM

CONS( 1982, a5200,   0,      0,      a5200,     a5200,  a5200_state,  empty_init, "Atari", "Atari 5200",           0)
CONS( 1983, a5200a,  a5200,  0,      a5200a,    a5200a, a5200_state,  empty_init, "Atari", "Atari 5200 (2-port)",  0)
