// license:BSD-3-Clause
// copyright-holders: Robbbert
/*******************************************************************************************************

PINBALL
Inder S.A. of Spain

All manuals are in Spanish (including the 'English' ones).
The schematics for Brave Team, Canasta are too blurry to read.
Each game has different hardware. Note with switches: "Veleta" is normally closed.
Order of dips on "new cpu" games: 3,2,1,0,4,5,6,7 (so SL1(1) uses bit 3 of input definition).

Setting up:
- First time run, the displays will all show zero. Hit F3. The game will now be working.
- If you want to change the "Handicap" (High Score) setting, you may have to wipe the NVRAM before it
   will take.

Note
- La Rana is not a pinball machine - it looks like an upright arcade game with a frog's mouth and some
   holes on a lower panel, while some indicators are on the upper panel. The main scores are on the
   marquee. Each of 4 players has 3 digits. There's other buttons on the marquee too. The game has
   no screen.
   Gameplay and description can be found on the patent fill (it's a traditional Spanish game, it's
   easy to find more info about how to play it):
     https://worldwide.espacenet.com/patent/search/family/008268025/publication/ES1014799U?q=pn%3DES1014799U
   Photos and a (partial) manual can be found at: https://www.recreativas.org/la-rana-1051-inder
   It can be played by pressing 5 (insert coin), press 1 (start) for each player, press A (start timer),
   press various keys to score. Don't press X or the game instantly ends. When the timer runs out it
   switches to the next player, or ends.

Status:
- All games are playable with sound.

ToDo:
- The "new CPU" machines are lacking mechanical sounds. The output bits vary per game.
- La Rana: playable. Needs its own layout. Inputs to be figured out.

********************************************************************************************************/

#include "emu.h"
#include "genpin.h"

#include "cpu/z80/z80.h"
#include "machine/74157.h"
#include "machine/7474.h"
#include "machine/i8255.h"
#include "sound/ay8910.h"
#include "sound/msm5205.h"
#include "sound/sn76496.h"
#include "speaker.h"

#include "inder.lh"


namespace {

class inder_state : public genpin_class
{
public:
	inder_state(const machine_config &mconfig, device_type type, const char *tag)
		: genpin_class(mconfig, type, tag)
		, m_maincpu(*this, "maincpu")
		, m_audiocpu(*this, "audiocpu")
		, m_sn(*this, "sn")
		, m_msm(*this, "msm")
		, m_7a(*this, "7a")
		, m_9a(*this, "9a")
		, m_9b(*this, "9b")
		, m_13(*this, "13")
		, m_p_audio(*this, "audiorom")
		, m_io_keyboard(*this, "X%d", 0U)
		, m_digits(*this, "digit%d", 0U)
		, m_io_outputs(*this, "out%d", 0U)
	{ }

	void inder(machine_config &config);
	void brvteam(machine_config &config);
	void canasta(machine_config &config);
	void lapbylap(machine_config &config);

	void init_0();
	void init_1();

private:
	virtual void machine_reset() override ATTR_COLD;
	virtual void machine_start() override ATTR_COLD;
	u8 ppic_r();
	void ppia_w(u8 data);
	void ppib_w(u8 data);
	void ppic_w(u8 data);
	void ppi60a_w(u8 data);
	void ppi60b_w(u8 data);
	void ppi64a_w(u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[i] = BIT(data, i); }
	void ppi64b_w(u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[8U+i] = BIT(data, i); }
	void ppi64c_w(u8 data);
	void ppi68a_w(u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[16U+i] = BIT(data, i); }
	void ppi68b_w(u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[24U+i] = BIT(data, i) ^ 1; }
	void ppi68c_w(u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[32U+i] = BIT(data, i); }
	void ppi6ca_w(u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[40U+i] = BIT(data, i); }
	void ppi6cb_w(u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[48U+i] = BIT(data, i); }
	void ppi6cc_w(u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[56U+i] = BIT(data, i); }
	u8 sw_r();
	void sw_w(offs_t offset, u8 data);
	void sol_brvteam_w(u8 data);
	void sol_canasta_w(u8 data);
	void sn_w(u8 data);
	u8 sndcmd_r();
	void sndbank_w(u8 data);
	void sndcmd_w(u8 data);
	void sndcmd_lapbylap_w(u8 data);
	void lamp_w(offs_t offset, u8 data) { for (u8 i = 0; i < 8; i++) m_io_outputs[8U+offset*8+i] = BIT(data, i); }
	void disp_w(offs_t offset, u8 data);
	void qc7a_w(int state);
	void q9a_w(int state);
	void qc9b_w(int state);
	void update_mus();
	void brvteam_map(address_map &map) ATTR_COLD;
	void canasta_map(address_map &map) ATTR_COLD;
	void main_map(address_map &map) ATTR_COLD;
	void audio_map(address_map &map) ATTR_COLD;
	void lapbylap_main_map(address_map &map) ATTR_COLD;
	void lapbylap_audio_map(address_map &map) ATTR_COLD;

	bool m_pc0 = false;
	u8 m_game = 0U;
	u8 m_portc = 0U;
	u8 m_row = 0U;
	u8 m_segment[8]{};
	u8 m_sndcmd = 0U;
	u8 m_sndbank = 0U;
	u32 m_sound_addr = 0U;
	u32 m_audio_size = 0U;

	required_device<cpu_device> m_maincpu;
	optional_device<cpu_device> m_audiocpu;
	optional_device<sn76489_device> m_sn;
	optional_device<msm5205_device> m_msm;
	optional_device<ttl7474_device> m_7a;
	optional_device<ttl7474_device> m_9a;
	optional_device<ttl7474_device> m_9b;
	optional_device<hct157_device> m_13;
	optional_region_ptr<u8> m_p_audio;
	required_ioport_array<11> m_io_keyboard;
	output_finder<50> m_digits;
	output_finder<64> m_io_outputs;
};

void inder_state::brvteam_map(address_map &map)
{
	map(0x0000, 0x1fff).rom();
	map(0x2000, 0x20ff).w(FUNC(inder_state::disp_w));
	map(0x4000, 0x43ff).ram(); // pair of 2114
	map(0x4400, 0x44ff).ram().share("nvram"); // pair of 5101, battery-backed
	map(0x4800, 0x480a).rw(FUNC(inder_state::sw_r), FUNC(inder_state::sw_w));
	map(0x4900, 0x4900).w(FUNC(inder_state::sol_brvteam_w));
	map(0x4901, 0x4907).w(FUNC(inder_state::lamp_w));
	map(0x4b00, 0x4b00).w(FUNC(inder_state::sn_w));
}

void inder_state::canasta_map(address_map &map)
{
	map(0x0000, 0x1fff).rom();
	map(0x2000, 0x20ff).w(FUNC(inder_state::disp_w));
	map(0x4000, 0x43ff).ram(); // pair of 2114
	map(0x4400, 0x44ff).ram().share("nvram"); // pair of 5101, battery-backed
	map(0x4800, 0x480a).rw(FUNC(inder_state::sw_r), FUNC(inder_state::sw_w));
	map(0x4900, 0x4900).w(FUNC(inder_state::sol_canasta_w));
	map(0x4901, 0x4907).w(FUNC(inder_state::lamp_w));
	map(0x4b00, 0x4b00).w("ay", FUNC(ay8910_device::address_w));
	map(0x4b01, 0x4b01).r("ay", FUNC(ay8910_device::data_r));
	map(0x4b02, 0x4b02).w("ay", FUNC(ay8910_device::data_w));
}

void inder_state::lapbylap_main_map(address_map &map)
{
	map(0x0000, 0x1fff).rom();
	map(0x2000, 0x20ff).w(FUNC(inder_state::disp_w));
	map(0x4000, 0x43ff).ram(); // pair of 2114
	map(0x4400, 0x44ff).ram().share("nvram"); // pair of 5101, battery-backed
	map(0x4800, 0x480a).rw(FUNC(inder_state::sw_r), FUNC(inder_state::sw_w));
	map(0x4900, 0x4900).w(FUNC(inder_state::sol_canasta_w));
	map(0x4901, 0x4907).w(FUNC(inder_state::lamp_w));
	map(0x4b00, 0x4b00).w(FUNC(inder_state::sndcmd_lapbylap_w));
}

void inder_state::lapbylap_audio_map(address_map &map)
{
	map(0x0000, 0x1fff).rom();
	map(0x8000, 0x87ff).ram(); // 6116
	map(0x9000, 0x9000).w("ay1", FUNC(ay8910_device::address_w));
	map(0x9001, 0x9001).r("ay1", FUNC(ay8910_device::data_r));
	map(0x9002, 0x9002).w("ay1", FUNC(ay8910_device::data_w));
	map(0xa000, 0xa000).w("ay2", FUNC(ay8910_device::address_w));
	map(0xa001, 0xa001).r("ay2", FUNC(ay8910_device::data_r));
	map(0xa002, 0xa002).w("ay2", FUNC(ay8910_device::data_w));
}

void inder_state::main_map(address_map &map)
{
	map(0x0000, 0x3fff).rom();
	map(0x4000, 0x47ff).mirror(0x1800).ram().share("nvram"); // 6116, battery-backed
	map(0x6000, 0x6003).mirror(0x13fc).rw("ppi60", FUNC(i8255_device::read), FUNC(i8255_device::write));
	map(0x6400, 0x6403).mirror(0x13fc).rw("ppi64", FUNC(i8255_device::read), FUNC(i8255_device::write));
	map(0x6800, 0x6803).mirror(0x13fc).rw("ppi68", FUNC(i8255_device::read), FUNC(i8255_device::write));
	map(0x6c00, 0x6c03).mirror(0x131c).rw("ppi6c", FUNC(i8255_device::read), FUNC(i8255_device::write));
	map(0x6c20, 0x6c3f).mirror(0x1300).w(FUNC(inder_state::sndcmd_w));
	map(0x6c60, 0x6c7f).mirror(0x1300).w(FUNC(inder_state::disp_w));
	map(0x6ce0, 0x6ce0).nopw();
}

void inder_state::audio_map(address_map &map)
{
	map(0x0000, 0x1fff).rom();
	map(0x2000, 0x27ff).mirror(0x1800).ram(); // 6116
	map(0x4000, 0x4003).mirror(0x1ffc).rw("ppi", FUNC(i8255_device::read), FUNC(i8255_device::write));
	map(0x6000, 0x6000).w(FUNC(inder_state::sndbank_w));
	map(0x8000, 0x8000).r(FUNC(inder_state::sndcmd_r));
}

static INPUT_PORTS_START( brvteam )
	PORT_START("X0")
	PORT_DIPNAME( 0x03, 0x01, "Coin Slot 1") // sw G,H
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C )) // slot 2: 1 moneda 4 partidas  // selection 00 is same as 01
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_5C )) // slot 2: 1 moneda 5 partidas
	PORT_DIPSETTING(    0x03, "4 moneda 6 partidas") // slot 2: 1 moneda 6 partidas
	PORT_DIPNAME( 0x08, 0x08, "Balls") // sw E
	PORT_DIPSETTING(    0x08, "3")
	PORT_DIPSETTING(    0x00, "5")
	PORT_DIPNAME( 0x30, 0x20, "Points for free game") // sw C,D
	PORT_DIPSETTING(    0x00, "850000")
	PORT_DIPSETTING(    0x10, "800000")
	PORT_DIPSETTING(    0x20, "750000")
	PORT_DIPSETTING(    0x30, "700000")
	PORT_BIT( 0xc4, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X1")
	PORT_DIPNAME( 0x03, 0x03, "High Score") //"Handicap"  // sw O,P
	PORT_DIPSETTING(    0x00, "990000")
	PORT_DIPSETTING(    0x01, "950000")
	PORT_DIPSETTING(    0x02, "900000")
	PORT_DIPSETTING(    0x03, "850000")
	PORT_DIPNAME( 0x04, 0x00, "Maximum Credits") // sw N
	PORT_DIPSETTING(    0x04, "10")
	PORT_DIPSETTING(    0x00, "20")
	PORT_BIT( 0xf8, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X2") // bank of unused dipswitches
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X3")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X4")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X5") // Contactos 50-57
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 ) // "Monedero B"
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X6") // 60-67
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP67")

	PORT_START("X7") // 70-77
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP71")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP77")

	PORT_START("X8") // 80-87
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP80")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP81")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP82")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP83")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP84")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP85")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP86")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP87")

	PORT_START("X9") // 90-97
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP90")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP92")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP93")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP94")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_COLON) PORT_NAME("INP95")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_QUOTE) PORT_NAME("INP96")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_ENTER) PORT_NAME("INP97")

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( canasta )
	PORT_START("X0")
	PORT_DIPNAME( 0x03, 0x00, "Coin Slot 1") // sw G,H
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C )) // slot 2: 1 moneda 4 partidas
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_5C )) // slot 2: 1 moneda 5 partidas
	PORT_DIPSETTING(    0x02, "4 moneda 6 partidas") // slot 2: 1 moneda 6 partidas
	PORT_DIPSETTING(    0x03, "4 moneda 9 partidas") // slot 2: 1 moneda 9 partidas
	PORT_DIPNAME( 0x08, 0x08, "Balls") // sw E
	PORT_DIPSETTING(    0x08, "3")
	PORT_DIPSETTING(    0x00, "5")
	PORT_DIPNAME( 0x30, 0x20, "Points for 1st free game") // sw C,D
	PORT_DIPSETTING(    0x00, "1900000")
	PORT_DIPSETTING(    0x10, "1800000")
	PORT_DIPSETTING(    0x20, "1700000")
	PORT_DIPSETTING(    0x30, "1500000")
	PORT_DIPNAME( 0xc0, 0xc0, "Points for 2nd free game") // sw A,B
	PORT_DIPSETTING(    0x00, "NA")
	PORT_DIPSETTING(    0x40, "2900000")
	PORT_DIPSETTING(    0x80, "2700000")
	PORT_DIPSETTING(    0xc0, "2500000")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X1")
	PORT_DIPNAME( 0x03, 0x03, "High Score") //"Handicap"  // sw O,P
	PORT_DIPSETTING(    0x00, "3500000")
	PORT_DIPSETTING(    0x01, "3000000")
	PORT_DIPSETTING(    0x02, "2400000")
	PORT_DIPSETTING(    0x03, "2000000")
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X2") // bank of unused dipswitches
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X3")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X4")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X5") // Contactos 50-57
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 ) // "Monedero B"
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X6") // 60-67
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP67")

	PORT_START("X7") // 70-77
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP71")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP77")

	PORT_START("X8") // 80-87
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP80")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP81")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP82")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP83")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP84")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP85")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP86")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP87")

	PORT_START("X9") // 90-97
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP90")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP92")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP93")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP94")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP96")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP97")

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( lapbylap )
	PORT_START("X0")
	PORT_DIPNAME( 0x03, 0x03, "Coin Slot 1") // sw G,H
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C )) // slot 2: 1 moneda 4 partidas
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C )) // and 4c_3c; slot 2: 1 moneda 3 partidas
	PORT_DIPNAME( 0x08, 0x08, "Balls") // sw E
	PORT_DIPSETTING(    0x08, "3")
	PORT_DIPSETTING(    0x00, "5")
	PORT_DIPNAME( 0x30, 0x20, "Points for free game") // sw C,D
	PORT_DIPSETTING(    0x00, "2900000")
	PORT_DIPSETTING(    0x10, "2700000")
	PORT_DIPSETTING(    0x20, "2500000")
	PORT_DIPSETTING(    0x30, "2300000")
	PORT_BIT( 0xc4, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X1")
	PORT_DIPNAME( 0x03, 0x03, "High Score") //"Handicap"  // sw O,P
	PORT_DIPSETTING(    0x00, "4600000")
	PORT_DIPSETTING(    0x01, "4400000")
	PORT_DIPSETTING(    0x02, "4200000")
	PORT_DIPSETTING(    0x03, "4000000")
	PORT_DIPNAME( 0x0c, 0x08, "Extra Ball Award??") //"Comienzo Secuenzia Diana Bola Extra"  // sw M,N
	PORT_DIPSETTING(    0x04, "50000")
	PORT_DIPSETTING(    0x08, "25000")
	PORT_DIPSETTING(    0x0c, "10000")
	PORT_DIPNAME( 0x10, 0x10, "Extra Ball Derribo??") //"Bola Extra en 1st Derribo Completo"  // sw L
	PORT_DIPSETTING(    0x00, DEF_STR(No))
	PORT_DIPSETTING(    0x10, DEF_STR(Yes)) // "Si"
	PORT_DIPNAME( 0x20, 0x20, "Especiales Laterales??") //need help here guys...  // sw K
	PORT_DIPSETTING(    0x00, "Derribo Lateral Dianas")
	PORT_DIPSETTING(    0x20, "Derribo Completo Dianas Laterales")
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X2")
	PORT_DIPNAME( 0x03, 0x03, "High Score Returns??") //"Handicap de Vueltas"  // sw W,X
	PORT_DIPSETTING(    0x00, "40")
	PORT_DIPSETTING(    0x01, "35")
	PORT_DIPSETTING(    0x02, "30")
	PORT_DIPSETTING(    0x03, "25")
	PORT_DIPNAME( 0x40, 0x40, "Bola Extra En Rampa??") //nfi  // sw R
	PORT_DIPSETTING(    0x00, "Derribo Completo")
	PORT_DIPSETTING(    0x40, "Derribo Lateral")
	PORT_DIPNAME( 0x80, 0x80, "Apagado de dianas")  // sw Q
	PORT_DIPSETTING(    0x80, DEF_STR(Easy)) // "Facil"
	PORT_DIPSETTING(    0x00, DEF_STR(Hard)) // "Dificil"
	PORT_BIT( 0x3c, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X3")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X4")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("X5") // Contactos 50-57
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 ) // "Monedero B"
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X6") // 60-67
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP67")

	PORT_START("X7") // 70-77
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP71")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP77")

	PORT_START("X8") // 80-87
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP80")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP81")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP82")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP83")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP84")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP85")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP86")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP87")

	PORT_START("X9") // 90-97
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP90")
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP92")
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP93")
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP94")
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP95")
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_COLON) PORT_NAME("INP96")
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_KEYPAD ) PORT_CODE(KEYCODE_QUOTE) PORT_NAME("INP97")

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( pinmoonl )
	PORT_START("X0")
	PORT_DIPNAME( 0x08, 0x00, "Balls")
	PORT_DIPSETTING(    0x00, "3")
	PORT_DIPSETTING(    0x08, "5")
	PORT_DIPNAME( 0x02, 0x00, "Coin Slot 1")
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C )) // slot 2: 1 moneda 4 partidas
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C )) // and 4c_3c; slot 2: 1 moneda 3 partidas
	PORT_DIPNAME( 0x30, 0x00, "Points for free game")
	PORT_DIPSETTING(    0x00, "2600000")
	PORT_DIPSETTING(    0x10, "3000000")
	PORT_DIPSETTING(    0x20, "3400000")
	PORT_DIPSETTING(    0x30, "3800000")

	PORT_START("X1")
	PORT_DIPNAME( 0x03, 0x00, "High Score") //"Handicap"
	PORT_DIPSETTING(    0x00, "4800000")
	PORT_DIPSETTING(    0x01, "5000000")
	PORT_DIPSETTING(    0x02, "5200000")
	PORT_DIPSETTING(    0x03, "5400000")
	PORT_DIPNAME( 0x10, 0x00, "Especial en Picabolas")
	PORT_DIPSETTING(    0x00, "1st Derribo")
	PORT_DIPSETTING(    0x10, "2nd Derribo")
	PORT_DIPNAME( 0x20, 0x00, "Bola Extra En Rampas")
	PORT_DIPSETTING(    0x00, "4 dianas")
	PORT_DIPSETTING(    0x20, "2 dianas")

	PORT_START("X2")
	PORT_DIPNAME( 0x03, 0x00, "High Score Returns??") //"Handicap de Vueltas"
	PORT_DIPSETTING(    0x00, "20")
	PORT_DIPSETTING(    0x01, "25")
	PORT_DIPSETTING(    0x02, "30")
	PORT_DIPSETTING(    0x03, "35")
	PORT_DIPNAME( 0x80, 0x00, "Apagado de dianas")
	PORT_DIPSETTING(    0x00, DEF_STR(Easy)) // "Facil"
	PORT_DIPSETTING(    0x80, DEF_STR(Hard)) // "Dificil"

	PORT_START("X3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_COIN2 ) // "Monedero B"
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP32")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP40")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP41")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP42")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP43")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP44")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP45")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP46")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP47")

	PORT_START("X5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP50")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP51")
	PORT_BIT( 0x04, IP_ACTIVE_LOW,  IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP52")
	PORT_BIT( 0x08, IP_ACTIVE_LOW,  IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP53")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP54")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP55")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP56")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP57")

	PORT_START("X6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP67")

	PORT_START("X7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COLON) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_QUOTE) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_ENTER) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_OPENBRACE) PORT_NAME("INP77")

	PORT_START("X8")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X9")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( pinclown )
	PORT_START("X0")
	PORT_DIPNAME( 0x08, 0x00, "Balls")
	PORT_DIPSETTING(    0x00, "3")
	PORT_DIPSETTING(    0x08, "5")
	PORT_DIPNAME( 0x02, 0x00, "Coin Slot 1")
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C )) // slot 2: 1 moneda 4 partidas
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C )) // and 4c_3c; slot 2: 1 moneda 3 partidas
	PORT_DIPNAME( 0x30, 0x00, "Points for free game")
	PORT_DIPSETTING(    0x00, "2800000")
	PORT_DIPSETTING(    0x10, "3200000")
	PORT_DIPSETTING(    0x20, "3600000")
	PORT_DIPSETTING(    0x30, "4000000")

	PORT_START("X1")
	PORT_DIPNAME( 0x03, 0x00, "High Score") //"Handicap"
	PORT_DIPSETTING(    0x00, "4800000")
	PORT_DIPSETTING(    0x01, "5000000")
	PORT_DIPSETTING(    0x02, "5200000")
	PORT_DIPSETTING(    0x03, "5400000")
	PORT_DIPNAME( 0x20, 0x00, "Bola Extra En Pasillos Inferiores")
	PORT_DIPSETTING(    0x00, "2")
	PORT_DIPSETTING(    0x20, "3")

	PORT_START("X2")
	PORT_DIPNAME( 0x20, 0x00, "Quita bola extra al 2nd derribo")
	PORT_DIPSETTING(    0x00, DEF_STR(Yes))
	PORT_DIPSETTING(    0x20, DEF_STR(No))
	PORT_DIPNAME( 0x80, 0x00, "Apagado de dianas")
	PORT_DIPSETTING(    0x00, DEF_STR(Easy)) // "Facil"
	PORT_DIPSETTING(    0x80, DEF_STR(Hard)) // "Dificil"

	PORT_START("X3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_COIN2 ) // "Monedero B"
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP40")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP42")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP43")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP44")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP46")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP47")

	PORT_START("X5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP50")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP53")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP54")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP55")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP56")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP57")

	PORT_START("X6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP67")

	PORT_START("X7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP77")

	PORT_START("X8")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X9")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( corsario )
	PORT_START("X0")
	PORT_DIPNAME( 0x08, 0x00, "Balls")
	PORT_DIPSETTING(    0x00, "3")
	PORT_DIPSETTING(    0x08, "5")
	PORT_DIPNAME( 0x02, 0x00, "Coin Slot 1")
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ))
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C )) // and 4c_3c
	PORT_DIPNAME( 0x30, 0x00, "Points for free game")
	PORT_DIPSETTING(    0x00, "3000000")
	PORT_DIPSETTING(    0x10, "3500000")
	PORT_DIPSETTING(    0x20, "4000000")
	PORT_DIPSETTING(    0x30, "4500000")

	PORT_START("X1")
	PORT_DIPNAME( 0x03, 0x00, "High Score") //"Handicap"
	PORT_DIPSETTING(    0x00, "4500000")
	PORT_DIPSETTING(    0x01, "5000000")
	PORT_DIPSETTING(    0x02, "5500000")
	PORT_DIPSETTING(    0x03, "6000000")

	PORT_START("X2")

	PORT_START("X3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP40")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP41")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP42")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP43")
	PORT_BIT( 0x10, IP_ACTIVE_LOW,  IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP44")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP46")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP47")

	PORT_START("X5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP50")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP51")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP52")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP53")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP54")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP57")

	PORT_START("X6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP67")

	PORT_START("X7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP77")

	PORT_START("X8")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X9")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( mundial )
	PORT_START("X0")
	PORT_DIPNAME( 0x08, 0x00, "Balls")
	PORT_DIPSETTING(    0x00, "3")
	PORT_DIPSETTING(    0x08, "5")
	PORT_DIPNAME( 0x02, 0x00, "Coin Slot 1")
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C )) // slot 2: 1 moneda 4 partidas
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C )) // and 4c_3c; slot 2: 1 moneda 3 partidas
	PORT_DIPNAME( 0x30, 0x00, "Points for free game")
	PORT_DIPSETTING(    0x00, "3600000")
	PORT_DIPSETTING(    0x10, "4000000")
	PORT_DIPSETTING(    0x20, "4400000")
	PORT_DIPSETTING(    0x30, "4800000")

	PORT_START("X1")
	PORT_DIPNAME( 0x03, 0x00, "High Score") //"Handicap"
	PORT_DIPSETTING(    0x00, "5500000")
	PORT_DIPSETTING(    0x01, "5700000")
	PORT_DIPSETTING(    0x02, "5900000")
	PORT_DIPSETTING(    0x03, "6100000")

	PORT_START("X2")
	PORT_DIPNAME( 0x20, 0x00, "Dianas")
	PORT_DIPSETTING(    0x00, "Alternativos")
	PORT_DIPSETTING(    0x20, "Fijos")

	PORT_START("X3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_COIN2 ) // "Monedero B"
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP32")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP40")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP41")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP42")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP43")
	PORT_BIT( 0x10, IP_ACTIVE_LOW,  IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP44")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP45")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP46")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP47")

	PORT_START("X5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP50")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP51")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP52")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP53")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP54")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP55")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP57")

	PORT_START("X6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP67")

	PORT_START("X7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_LOW,  IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COLON) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_QUOTE) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_ENTER) PORT_NAME("INP77")

	PORT_START("X8")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X9")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( atleta )
	PORT_START("X0")
	PORT_DIPNAME( 0x08, 0x00, "Balls")
	PORT_DIPSETTING(    0x00, "3")
	PORT_DIPSETTING(    0x08, "5")
	PORT_DIPNAME( 0x02, 0x00, "Coin Slot 1")
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ))
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C )) // and 4c_3c
	PORT_DIPNAME( 0x30, 0x00, "Points for free game")
	PORT_DIPSETTING(    0x00, "3500000")
	PORT_DIPSETTING(    0x10, "4000000")
	PORT_DIPSETTING(    0x20, "4500000")
	PORT_DIPSETTING(    0x30, "5000000")

	PORT_START("X1")
	PORT_DIPNAME( 0x08, 0x00, "Dianas") // "Dificultad bola extra dianas posterior bancada"
	PORT_DIPSETTING(    0x00, DEF_STR(Easy)) // "Facil"
	PORT_DIPSETTING(    0x08, DEF_STR(Hard)) // "Dificil"
	PORT_DIPNAME( 0x03, 0x00, "High Score") //"Handicap"
	PORT_DIPSETTING(    0x00, "5500000")
	PORT_DIPSETTING(    0x01, "6000000")
	PORT_DIPSETTING(    0x02, "6500000")
	PORT_DIPSETTING(    0x03, "7000000")

	PORT_START("X2")

	PORT_START("X3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP32")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP40")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP41")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP42")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP43")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP44")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP45")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP46")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP47")

	PORT_START("X5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP50")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP51")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP52")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP53")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP54")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP55")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP56")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP57")

	PORT_START("X6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP67")

	PORT_START("X7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_LOW,  IPT_KEYPAD ) PORT_CODE(KEYCODE_COLON) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_QUOTE) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_ENTER) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_OPENBRACE) PORT_NAME("INP77")

	PORT_START("X8")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X9")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( ind250cc )
	PORT_START("X0")
	PORT_DIPNAME( 0x08, 0x00, "Balls")
	PORT_DIPSETTING(    0x00, "3")
	PORT_DIPSETTING(    0x08, "5")
	PORT_DIPNAME( 0x02, 0x00, "Coins")
	PORT_DIPSETTING(    0x00, "Slot1 1coin/1game; Slot2 1coin/4games")
	PORT_DIPSETTING(    0x02, "Slot1 2coins/1game; Slot2 1coin/3games")
	PORT_DIPNAME( 0x30, 0x00, "Points for free game")
	PORT_DIPSETTING(    0x00, "3000000")
	PORT_DIPSETTING(    0x10, "3300000")
	PORT_DIPSETTING(    0x20, "3500000")
	PORT_DIPSETTING(    0x30, "3800000")

	PORT_START("X1")
	PORT_DIPNAME( 0x03, 0x00, "High Score") //"Handicap"
	PORT_DIPSETTING(    0x00, "5000000")
	PORT_DIPSETTING(    0x01, "5200000")
	PORT_DIPSETTING(    0x02, "5400000")
	PORT_DIPSETTING(    0x03, "5600000")

	PORT_START("X2")
	PORT_DIPNAME( 0x20, 0x20, "Quita bola extra en passillos 1,2,y 3")
	PORT_DIPSETTING(    0x00, DEF_STR(No))
	PORT_DIPSETTING(    0x20, DEF_STR(Yes))

	PORT_START("X3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_COIN2 ) // "Monedero B"
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP32")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP40")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP41")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP42")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP43")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP44")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP45")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP46")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP47")

	PORT_START("X5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP50")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP51")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP52")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP53")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP54")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP55")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP56")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP57")

	PORT_START("X6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP67")

	PORT_START("X7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP73")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COLON) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_QUOTE) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_ENTER) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_OPENBRACE) PORT_NAME("INP77")

	PORT_START("X8")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X9")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( larana ) // to be determined, need the manual
	PORT_START("X0")
	PORT_DIPNAME( 0x01, 0x00, "Coin control 1")
	PORT_DIPSETTING(    0x00, "Off")
	PORT_DIPSETTING(    0x01, "On")
	PORT_DIPNAME( 0x02, 0x00, "Coin control 2")
	PORT_DIPSETTING(    0x00, "Off")
	PORT_DIPSETTING(    0x02, "On")
	PORT_DIPNAME( 0x04, 0x00, "Coin control 3")
	PORT_DIPSETTING(    0x00, "Off")
	PORT_DIPSETTING(    0x04, "On")
	PORT_DIPNAME( 0x08, 0x00, "Coin control 4")
	PORT_DIPSETTING(    0x00, "Off")
	PORT_DIPSETTING(    0x08, "On")
	PORT_DIPNAME( 0x30, 0x00, "Points for free game")
	PORT_DIPSETTING(    0x00, "3600000")
	PORT_DIPSETTING(    0x10, "4000000")
	PORT_DIPSETTING(    0x20, "4400000")
	PORT_DIPSETTING(    0x30, "4800000")

	PORT_START("X1")
	PORT_DIPNAME( 0x08, 0x00, "Balls")
	PORT_DIPSETTING(    0x00, "3")
	PORT_DIPSETTING(    0x08, "5")
	PORT_DIPNAME( 0x03, 0x00, "High Score") //"Handicap"
	PORT_DIPSETTING(    0x00, "5600000")
	PORT_DIPSETTING(    0x01, "6000000")
	PORT_DIPSETTING(    0x02, "6400000")
	PORT_DIPSETTING(    0x03, "6800000")
	PORT_DIPNAME( 0x10, 0x00, "Dificultad de Especiales")
	PORT_DIPSETTING(    0x00, "Varios por partida")
	PORT_DIPSETTING(    0x10, "Solo uno por partida")

	PORT_START("X2") // not on the pcb

	PORT_START("X3")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 ) // "Monedero A"
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_A) PORT_NAME("INP32")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_9) PORT_NAME("Tilt") // "Falta"
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START1 ) // "Pulsador Partidas"
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_2_PAD) PORT_NAME("Reset") // "Puesta a cero"
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_1_PAD) PORT_NAME("Accounting info") // "Test economico"
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_0_PAD) PORT_NAME("Test") // "Test tecnico"

	PORT_START("X4")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_B) PORT_NAME("INP40")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_C) PORT_NAME("INP41")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_D) PORT_NAME("INP42")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_E) PORT_NAME("INP43")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_F) PORT_NAME("INP44")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_G) PORT_NAME("INP45")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_H) PORT_NAME("INP46")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_I) PORT_NAME("INP47")

	PORT_START("X5")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_J) PORT_NAME("INP50")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_K) PORT_NAME("INP51")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_L) PORT_NAME("INP52")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_M) PORT_NAME("INP53")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_N) PORT_NAME("INP54")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_O) PORT_NAME("INP55")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_P) PORT_NAME("INP56")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Q) PORT_NAME("INP57")

	PORT_START("X6")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_R) PORT_NAME("INP60")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_S) PORT_NAME("INP61")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_T) PORT_NAME("INP62")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_U) PORT_NAME("INP63")
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_V) PORT_NAME("INP64")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_W) PORT_NAME("INP65")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Y) PORT_NAME("INP66")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_Z) PORT_NAME("INP67")

	PORT_START("X7")
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COMMA) PORT_NAME("INP70")
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_X) PORT_NAME("Outhole")
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_STOP) PORT_NAME("INP72")
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_SLASH) PORT_NAME("INP74")
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_COLON) PORT_NAME("INP75")
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_QUOTE) PORT_NAME("INP76")
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_KEYPAD ) PORT_CODE(KEYCODE_ENTER) PORT_NAME("INP77")

	PORT_START("X8")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X9")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("X10")
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END

u8 inder_state::sw_r()
{
	if (m_row < 11)
		return m_io_keyboard[m_row]->read();
	else
		return 0;
}

void inder_state::sw_w(offs_t offset, u8 data)
{
	m_row = offset;
}

void inder_state::sn_w(u8 data)
{
	m_sn->write(bitswap<8>(data, 0, 1, 2, 3, 4, 5, 6, 7));
}

void inder_state::sndcmd_lapbylap_w(u8 data)
{
	m_sndcmd = data;
	m_audiocpu->pulse_input_line(INPUT_LINE_NMI, attotime::zero);
}

void inder_state::sndcmd_w(u8 data)
{
	m_sndcmd = data;
}

u8 inder_state::sndcmd_r()
{
	return m_sndcmd;
}

// "bobinas"
void inder_state::sol_brvteam_w(u8 data)
{
	if ((data & 0xee) && BIT(data, 4)) // solenoid selected & activated
	{
		if (BIT(data, 1))
			m_samples->start(0, 7); // left sling near bumpers "canon izq"

		if (BIT(data, 2))
			m_samples->start(1, 7); // right sling near bumpers "canon der"

		if (BIT(data, 3))
			m_samples->start(0, 5); // outhole

		if (BIT(data, 5))
			m_samples->start(2, 0); // left bumper

		if (BIT(data, 6))
			m_samples->start(3, 0); // right bumper

		if (BIT(data, 7))
			m_samples->start(4, 0); // middle bumper
	}
	for (u8 i = 0; i < 8; i++)
		m_io_outputs[i] = (BIT(data, i) && BIT(data, 4));
}

// no slings in this game
void inder_state::sol_canasta_w(u8 data)
{
	if ((data & 0xee) && BIT(data, 4)) // solenoid selected & activated
	{
		if (BIT(data, 3))
			m_samples->start(0, 5); // outhole

		if (BIT(data, 5))
			m_samples->start(2, 0); // left bumper

		if (BIT(data, 6))
			m_samples->start(3, 0); // right bumper

		if (BIT(data, 7))
			m_samples->start(4, 0); // middle bumper
	}
	for (u8 i = 0; i < 8; i++)
		m_io_outputs[i] = (BIT(data, i) && BIT(data, 4));
}

void inder_state::disp_w(offs_t offset, u8 data)
{
	u8 i;
	if (offset < 8)
		m_segment[offset] = data;
	else
	// From here, only used on old cpu board
	if (offset > 0x40)
	{
		offset = (offset >> 3) & 7;
		for (i = 0; i < 5; i++)
			m_digits[i*10+offset] = m_segment[i];
	}
}

void inder_state::ppi60a_w(u8 data)
{
	if (data)
		for (u8 i = 0; i < 8; i++)
			if (BIT(data, i))
				m_row = i;
}

void inder_state::ppi60b_w(u8 data)
{
	if (data & 7)
		for (u8 i = 0; i < 3; i++)
			if (BIT(data, i))
				m_row = i+8;
}

void inder_state::ppi64c_w(u8 data)
{
	u8 i;
	data &= 15;
	if (BIT(data, 3)) // 8 to 15)
	{
		data ^= 15; // now 7 to 0

		for (i = 0; i < 5; i++)
		{
			if (m_game && (i == 4))  // mundial,clown,250cc,atleta have credit and ball displays swapped
				data ^= 4;
			m_digits[i*10+data] = m_segment[i];
		}
	}
}

void inder_state::sndbank_w(u8 data)
{
	m_sndbank = data;
	u8 i;
	// look for last rom enabled
	for (i = 0; i < 4; i++)
		if (!(BIT(data, i)))
			m_sound_addr = (m_sound_addr & 0x0ffff) | (i << 16);
	update_mus();
}

void inder_state::update_mus()
{
	if (!m_audio_size)
		return;
	if ((m_sound_addr < m_audio_size) && (m_sndbank != 0xff))
		m_13->ba_w(m_p_audio[m_sound_addr]);
	else
		m_13->ba_w(0);
}

void inder_state::qc7a_w(int state)
{
	m_msm->reset_w(state);
	m_9a->clear_w(!state);
	m_9b->clear_w(!state);
}

void inder_state::q9a_w(int state)
{
	m_pc0 = state;
}

void inder_state::qc9b_w(int state)
{
	m_9a->d_w(state);
	m_9b->d_w(state);
	m_13->select_w(state);
}

u8 inder_state::ppic_r()
{
	return (m_pc0 ? 1 : 0) | m_portc;
}

void inder_state::ppia_w(u8 data)
{
	m_sound_addr = (m_sound_addr & 0xffff00) | data;
	update_mus();
}

void inder_state::ppib_w(u8 data)
{
	m_sound_addr = (m_sound_addr & 0xff00ff) | (data << 8);
	update_mus();
}

void inder_state::ppic_w(u8 data)
{
	// pc4 - READY line back to cpu board, but not used
	m_msm->s1_w(BIT(data, 5));
	m_7a->clock_w(BIT(data, 6));
	m_7a->preset_w(!BIT(data, 7));
	m_9a->preset_w(!BIT(data, 7));
	m_portc = data & 0xfe;
}


void inder_state::machine_start()
{
	genpin_class::machine_start();

	m_digits.resolve();
	m_io_outputs.resolve();

	save_item(NAME(m_pc0));
	save_item(NAME(m_game));
	save_item(NAME(m_portc));
	save_item(NAME(m_row));
	save_item(NAME(m_segment));
	save_item(NAME(m_sndcmd));
	save_item(NAME(m_sndbank));
	save_item(NAME(m_sound_addr));

	m_audio_size = 0;
	if (memregion("audiorom"))
		m_audio_size = memregion("audiorom")->bytes();
}

void inder_state::machine_reset()
{
	genpin_class::machine_reset();
	for (u8 i = 0; i < m_io_outputs.size(); i++)
		m_io_outputs[i] = 0;

	std::fill(std::begin(m_segment), std::end(m_segment), 0);
	m_sound_addr = 0;
	m_sndbank = 0xff;
	m_row = 0;
	if (m_7a.found())
	{
		m_7a->clear_w(1);
		update_mus();
	}
}

void inder_state::init_0()
{
	if (m_7a.found())
	{
		m_7a->d_w(0);
		m_7a->clear_w(0);
		m_9b->preset_w(1);
	}
	m_game = 0;
}

void inder_state::init_1()
{
	init_0();
	m_game = 1;
}

void inder_state::brvteam(machine_config &config)
{
	// basic machine hardware
	Z80(config, m_maincpu, XTAL(5'000'000) / 2);
	m_maincpu->set_addrmap(AS_PROGRAM, &inder_state::brvteam_map);
	m_maincpu->set_periodic_int(FUNC(inder_state::irq0_line_hold), attotime::from_hz(250)); // NE556

	NVRAM(config, "nvram", nvram_device::DEFAULT_ALL_1);

	// Video
	config.set_default_layout(layout_inder);

	// Sound
	genpin_audio(config);
	SPEAKER(config, "snvol").front_center();
	SN76489(config, m_sn, XTAL(8'000'000) / 2).add_route(ALL_OUTPUTS, "snvol", 2.0); // jumper choice of 2 or 4 MHz
}

void inder_state::canasta(machine_config &config)
{
	// basic machine hardware
	Z80(config, m_maincpu, XTAL(5'000'000) / 2);
	m_maincpu->set_addrmap(AS_PROGRAM, &inder_state::canasta_map);
	m_maincpu->set_periodic_int(FUNC(inder_state::irq0_line_hold), attotime::from_hz(250)); // NE556

	NVRAM(config, "nvram", nvram_device::DEFAULT_ALL_1);

	// Video
	config.set_default_layout(layout_inder);

	// Sound
	genpin_audio(config);
	SPEAKER(config, "ayvol").front_center();
	AY8910(config, "ay", XTAL(4'000'000) / 2).add_route(ALL_OUTPUTS, "ayvol", 1.0);
}

void inder_state::lapbylap(machine_config &config)
{
	// basic machine hardware
	Z80(config, m_maincpu, XTAL(5'000'000) / 2);
	m_maincpu->set_addrmap(AS_PROGRAM, &inder_state::lapbylap_main_map);
	m_maincpu->set_periodic_int(FUNC(inder_state::irq0_line_hold), attotime::from_hz(250)); // NE556

	Z80(config, m_audiocpu, XTAL(2'000'000));
	m_audiocpu->set_addrmap(AS_PROGRAM, &inder_state::lapbylap_audio_map);
	m_audiocpu->set_periodic_int(FUNC(inder_state::irq0_line_hold), attotime::from_hz(250)); // NE555

	NVRAM(config, "nvram", nvram_device::DEFAULT_ALL_1);

	// Video
	config.set_default_layout(layout_inder);

	// Sound
	genpin_audio(config);
	SPEAKER(config, "ayvol").front_center();
	AY8910(config, "ay1", XTAL(2'000'000)).add_route(ALL_OUTPUTS, "ayvol", 1.0); // same xtal that drives subcpu
	ay8910_device &ay2(AY8910(config, "ay2", XTAL(2'000'000))); // same xtal that drives subcpu
	ay2.port_a_read_callback().set(FUNC(inder_state::sndcmd_r));
	ay2.add_route(ALL_OUTPUTS, "ayvol", 1.0);
}

void inder_state::inder(machine_config &config)
{
	// basic machine hardware
	Z80(config, m_maincpu, XTAL(5'000'000) / 2);
	m_maincpu->set_addrmap(AS_PROGRAM, &inder_state::main_map);
	m_maincpu->set_periodic_int(FUNC(inder_state::irq0_line_hold), attotime::from_hz(250)); // NE556

	Z80(config, m_audiocpu, XTAL(5'000'000) / 2);
	m_audiocpu->set_addrmap(AS_PROGRAM, &inder_state::audio_map);
	m_audiocpu->set_periodic_int(FUNC(inder_state::irq0_line_hold), attotime::from_hz(250)); // NE555

	NVRAM(config, "nvram", nvram_device::DEFAULT_ALL_1);

	// Video
	config.set_default_layout(layout_inder);

	// Sound
	genpin_audio(config);
	SPEAKER(config, "msmvol").front_center();
	MSM5205(config, m_msm, 384_kHz_XTAL);
	m_msm->vck_callback().set(m_9a, FUNC(ttl7474_device::clock_w));
	m_msm->vck_callback().append(m_9b, FUNC(ttl7474_device::clock_w)); // order of writes is sensitive
	m_msm->set_prescaler_selector(msm5205_device::S96_4B); // 4KHz 4-bit
	m_msm->add_route(ALL_OUTPUTS, "msmvol", 1.0);

	// Devices
	i8255_device &ppi60(I8255A(config, "ppi60"));
	ppi60.out_pa_callback().set(FUNC(inder_state::ppi60a_w));
	ppi60.out_pb_callback().set(FUNC(inder_state::ppi60b_w));
	ppi60.in_pc_callback().set(FUNC(inder_state::sw_r));

	i8255_device &ppi64(I8255A(config, "ppi64"));
	ppi64.out_pa_callback().set(FUNC(inder_state::ppi64a_w));
	ppi64.out_pb_callback().set(FUNC(inder_state::ppi64b_w));
	ppi64.out_pc_callback().set(FUNC(inder_state::ppi64c_w));

	i8255_device &ppi68(I8255A(config, "ppi68"));
	ppi68.out_pa_callback().set(FUNC(inder_state::ppi68a_w));
	ppi68.out_pb_callback().set(FUNC(inder_state::ppi68b_w));
	ppi68.out_pc_callback().set(FUNC(inder_state::ppi68c_w));

	i8255_device &ppi6c(I8255A(config, "ppi6c"));
	ppi6c.out_pa_callback().set(FUNC(inder_state::ppi6ca_w));
	ppi6c.out_pb_callback().set(FUNC(inder_state::ppi6cb_w));
	ppi6c.out_pc_callback().set(FUNC(inder_state::ppi6cc_w));

	i8255_device &ppi(I8255A(config, "ppi"));
	ppi.out_pa_callback().set(FUNC(inder_state::ppia_w));
	ppi.out_pb_callback().set(FUNC(inder_state::ppib_w));
	ppi.in_pc_callback().set(FUNC(inder_state::ppic_r));
	ppi.out_pc_callback().set(FUNC(inder_state::ppic_w));

	TTL7474(config, m_7a, 0);
	m_7a->comp_output_cb().set(FUNC(inder_state::qc7a_w));

	TTL7474(config, m_9a, 0); // HCT74
	m_9a->output_cb().set(FUNC(inder_state::q9a_w));

	TTL7474(config, m_9b, 0); // HCT74
	m_9b->comp_output_cb().set(FUNC(inder_state::qc9b_w));

	HCT157(config, m_13, 0);
	m_13->out_callback().set("msm", FUNC(msm5205_device::data_w));
}


/*-------------------------------------------------------------------
/ Brave Team (1985)
/-------------------------------------------------------------------*/
ROM_START(brvteam)
	ROM_REGION(0x2000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("brv-tea.m0", 0x0000, 0x1000, CRC(1fa72160) SHA1(0fa779ce2604599adff1e124d0b161b69094a614))
	ROM_LOAD("brv-tea.m1", 0x1000, 0x1000, CRC(4f02ca47) SHA1(68ec7d48c335a1ddd808feaeccac04a4f63d1a33))
ROM_END

/*-------------------------------------------------------------------
/ Canasta '86' (1986)
/-------------------------------------------------------------------*/
ROM_START(canasta)
	ROM_REGION(0x2000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("c860.bin", 0x0000, 0x1000, CRC(b1f79e52) SHA1(8e9c616f9be19d056da2f86778539d62c0885bac))
	ROM_LOAD("c861.bin", 0x1000, 0x1000, CRC(25ae3994) SHA1(86dcda3278fbe0e57b8ff4858b955d067af414ce))
ROM_END

/*-------------------------------------------------------------------
/ Lap By Lap (1986)
/-------------------------------------------------------------------*/
ROM_START(lapbylap)
	ROM_REGION(0x2000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("lblr0.bin", 0x0000, 0x1000, CRC(2970f31a) SHA1(01fb774de19944bb3a19577921f84ab5b6746afb))
	ROM_LOAD("lblr1.bin", 0x1000, 0x1000, CRC(94787c10) SHA1(f2a5b07e57222ee811982eb220c239e34a358d6f))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("lblsr0.bin", 0x0000, 0x2000, CRC(cbaddf02) SHA1(8207eebc414d90328bfd521190d508b88bb870a2))
ROM_END

/*-------------------------------------------------------------------
/ Moon Light (1987)
/-------------------------------------------------------------------*/
ROM_START(pinmoonl)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("ci-3.bin", 0x0000, 0x2000, CRC(56b901ae) SHA1(7269d1a100c378b21454f9f80f5bd9fbb736c222))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("ci-11.bin", 0x0000, 0x2000, CRC(a0732fe4) SHA1(54f62cd81bdb7e1924acb67ddbe43eb3d0a4eab0))

	ROM_REGION(0x40000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("ci-24.bin", 0x00000, 0x10000, CRC(6406bd18) SHA1(ae45ed9e8b1fd278a36a68b780352dbbb6ee781e))
	ROM_LOAD("ci-23.bin", 0x10000, 0x10000, CRC(eac346da) SHA1(7c4c26ae089dda0dcd7300fd1ecabf5a91099c41))
	ROM_LOAD("ci-22.bin", 0x20000, 0x10000, CRC(379740da) SHA1(83ad13ab7f1f37c78397d8e830bd74c5a7aea758))
	ROM_LOAD("ci-21.bin", 0x30000, 0x10000, CRC(0febb4a7) SHA1(e6cc1b26ddfe9cd58da29de2a50a83ce50afe323))

	ROM_REGION(0x4000, "bonus", ROMREGION_ERASEFF) // On "Bonus expansion board"
	ROM_LOAD("inder_sa_m17_moon_ligh_bonus_rom_1.ci1", 0x0000, 0x2000, CRC(2c45fd1e) SHA1(4cd1336271d77d9acb7c81a5461546842e946dc3))
	ROM_LOAD("inder_sa_m17_moon_ligh_bonus_rom_2.ci2", 0x2000, 0x2000, CRC(c0dc1523) SHA1(511657cc01fbbb5146e32ef26a9b259250df3fb6))
ROM_END

/*-------------------------------------------------------------------
/ Clown (1988)
/-------------------------------------------------------------------*/
ROM_START(pinclown)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("clown_a.bin", 0x0000, 0x2000, CRC(b7c3f9ab) SHA1(89ede10d9e108089da501b28f53cd7849f791a00))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("clown_b.bin", 0x0000, 0x2000, CRC(c223c961) SHA1(ed5180505b6ebbfb9451f67a44d07df3555c8f8d))

	ROM_REGION(0x40000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("clown_c.bin", 0x00000, 0x10000, CRC(dff89319) SHA1(3745a02c3755d11ea7fb552f7a5df2e8bbee2c29))
	ROM_LOAD("clown_d.bin", 0x10000, 0x10000, CRC(cce4e1dc) SHA1(561c9331d2d110d34cf250cd7b25be16a72a1d79))
	ROM_LOAD("clown_e.bin", 0x20000, 0x10000, CRC(98263526) SHA1(509764e65847637824ba93f7e6ce926501c431ce))
	ROM_LOAD("clown_f.bin", 0x30000, 0x10000, CRC(5f01b531) SHA1(116b1670ef4d5c054bb09dc55aa7d5d3ca047079))
ROM_END

/*-------------------------------------------------------------------
/ Corsario (1989)
/-------------------------------------------------------------------*/
ROM_START(corsario)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("0-corsar.bin", 0x0000, 0x2000, CRC(800f6895) SHA1(a222e7ea959629202686815646fc917ffc5a646c))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("a-corsar.bin", 0x0000, 0x2000, CRC(e14b7918) SHA1(5a5fc308b0b70fe041b81071ba4820782b6ff988))

	ROM_REGION(0x40000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("b-corsar.bin", 0x00000, 0x10000, CRC(7f155828) SHA1(e459c81b2c2e47d4276344d8d6a08c2c6242f941))
	ROM_LOAD("c-corsar.bin", 0x10000, 0x10000, CRC(047fd722) SHA1(2385507459f85c68141adc7084cb51dfa02462f6))
	ROM_LOAD("d-corsar.bin", 0x20000, 0x10000, CRC(10d8b448) SHA1(ed1918e6c55eba07dde31b9755c9403e073cad98))
	ROM_LOAD("e-corsar.bin", 0x30000, 0x10000, CRC(918ee349) SHA1(17cded8b5626c91e400d26332a160704f2fd2b55))
ROM_END

/*-------------------------------------------------------------------
/ Mundial 90 (1990)
/ Soon after the launch of Mundial 90, Inder detected a bug on the
/ program ROM that could cause the catapult frame to break, so they
/ sent a letter (dated June 1990) with an updated program ROM and
/ recalling the buggy one. It is unclear if the supported set is the
/ fixed or the older one, as it matches one dumped from an EEPROM
/ with an original Inder label dated 18-April-1990.
/-------------------------------------------------------------------*/
ROM_START(mundial)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("mundial.cpu", 0x0000, 0x2000, CRC(b615e69b) SHA1(d129eb6f2943af40ddffd0da1e7a711b58f65b3c))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("snd11.bin", 0x0000, 0x2000, CRC(2cebc1a5) SHA1(e0dae2b1ce31ff436b55ceb1ec71d39fc56694da))

	ROM_REGION(0x40000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("snd24.bin", 0x00000, 0x10000, CRC(603bfc3c) SHA1(8badd9731243270ce5b8003373ed09ec7eac6ca6))
	ROM_LOAD("snd23.bin", 0x10000, 0x10000, CRC(2868ce6f) SHA1(317457763f764be08cbe6a5dd4008ba2257c9d78))
	ROM_LOAD("snd22.bin", 0x20000, 0x10000, CRC(2559f874) SHA1(cbf57f29e394d5dc320e7dcbd2625f6c96412a06))
	ROM_LOAD("snd21.bin", 0x30000, 0x10000, CRC(7a8f7402) SHA1(39666ba2634fe9c720c2c9bcc9ccc73874ed85e7))
ROM_END

/*-------------------------------------------------------------------
/ Atleta (1991)
/-------------------------------------------------------------------*/
ROM_START(atleta)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("atleta0.cpu", 0x0000, 0x2000, CRC(5f27240f) SHA1(8b77862fa311d703b3af8a1db17e13b17dca7ec6))
	ROM_LOAD("atleta1.cpu", 0x2000, 0x2000, CRC(12bef582) SHA1(45e1da318141d9228bc91a4e09fff6bf6f194235))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("atletaa.snd", 0x0000, 0x2000, CRC(051c5329) SHA1(339115af4a2e3f1f2c31073cbed1842518d5916e))

	ROM_REGION(0x40000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("atletab.snd", 0x00000, 0x10000, CRC(7f155828) SHA1(e459c81b2c2e47d4276344d8d6a08c2c6242f941))
	ROM_LOAD("atletac.snd", 0x10000, 0x10000, CRC(20456363) SHA1(b226400dac35dedc039a7e03cb525c6033b24ebc))
	ROM_LOAD("atletad.snd", 0x20000, 0x10000, CRC(6518e3a4) SHA1(6b1d852005dabb76c7c65b87ecc9ee1422f16737))
	ROM_LOAD("atletae.snd", 0x30000, 0x10000, CRC(1ef7b099) SHA1(08400db3e238baf1673a2da604c999db6be30ffe))
ROM_END

/*-------------------------------------------------------------------
/ La Rana (1991)
/ Main PCB: Inder SA UCPU87 (only 2 of 4 I8255 sockets populated)
/ Sound PCB: USVYS (only 1 of 8 speech ROM sockets populated)
/-------------------------------------------------------------------*/
ROM_START(larana)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("inder_sa_mod_la_rana_0_050790.bin", 0x00000, 0x02000, CRC(ba94618f) SHA1(0fd6ffe9a6ef514c1dbf8856b881a54bf184e863))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("inder_sa_mod_la_rana_a_050690.bin", 0x00000, 0x02000, CRC(1513fd92) SHA1(6ca0723f5d7c86b844476a4830c8fc3744cbf918))

	ROM_REGION(0x10000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("inder_sa_mod_la_rana_b_200690.bin", 0x00000, 0x10000, CRC(3aaa7c7d) SHA1(4a8531b6859fc1f2a4bb63a51da35e9081b7e88b))
ROM_END

// This sets gives, by default, 2 launches per coin. Other changes are unknown.
ROM_START(larana2)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("larana_alt.bin",                    0x00000, 0x02000, CRC(81a6c8c3) SHA1(e3678e18704d0be0a1e9ece178babec7d728e510))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("inder_sa_mod_la_rana_a_050690.bin", 0x00000, 0x02000, CRC(1513fd92) SHA1(6ca0723f5d7c86b844476a4830c8fc3744cbf918))

	ROM_REGION(0x10000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("inder_sa_mod_la_rana_b_200690.bin", 0x00000, 0x10000, CRC(3aaa7c7d) SHA1(4a8531b6859fc1f2a4bb63a51da35e9081b7e88b))
ROM_END

/*-------------------------------------------------------------------
/ 250 CC (1992)
/-------------------------------------------------------------------*/
ROM_START(ind250cc)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("0-250cc.bin", 0x0000, 0x2000, CRC(753d82ec) SHA1(61950336ba571f9f75f2fc31ccb7beaf4e05dddc))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("a-250cc.bin", 0x0000, 0x2000, CRC(b64bdafb) SHA1(eab6d54d34b44187d454c1999e4bcf455183d5a0))

	ROM_REGION(0x40000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("b-250cc.bin", 0x00000, 0x10000, CRC(884c31c8) SHA1(23a838f1f0cb4905fa8552579b5452134f0fc9cc))
	ROM_LOAD("c-250cc.bin", 0x10000, 0x10000, CRC(5a1dfa1d) SHA1(4957431d87be0bb6d27910b718f7b7edcd405fff))
	ROM_LOAD("d-250cc.bin", 0x20000, 0x10000, CRC(a0940387) SHA1(0e06483e3e823bf4673d8e0bd120b0a6b802035d))
	ROM_LOAD("e-250cc.bin", 0x30000, 0x10000, CRC(538b3274) SHA1(eb76c41a60199bb94aec4666222e405bbcc33494))
ROM_END

/*---------------------------------------------------------------------------------------------------------------
/ Unknown gambling game on Inder pinball hardware.
/ According to PinMAME dev:
/ By inspecting the program ROMs, it seems that the game will "spin" several lamps so a pattern emerges.
/ If a certain pattern appears (a win), two to five tries can be made to increase your win, using three buttons.
/ You may also wait, not pushing anything, in which case the win stays the same.
/ If a win is made, some sort of payout starts (probably coins being dispensed).
/--------------------------------------------------------------------------------------------------------------*/
ROM_START(indunkgam)
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("0.ci3", 0x0000, 0x2000, CRC(f819d4a4) SHA1(f8f73dadbf9d45af8fa004e92e660a6014c31f68))
	ROM_LOAD("1.ci4", 0x2000, 0x2000, CRC(31837bed) SHA1(27edb9dec0e0368b5144922bc9e23fc9f69da34b))

	ROM_REGION(0x2000, "audiocpu", ROMREGION_ERASEFF)
	ROM_LOAD("audiocpu.snd", 0x0000, 0x2000, NO_DUMP)

	ROM_REGION(0x40000, "audiorom", ROMREGION_ERASEFF)
	ROM_LOAD("audiorom.snd", 0x00000, 0x10000, NO_DUMP)
ROM_END

} // Anonymous namespace


// old CPU board, 6 digits, sn76489
GAME(1985,  brvteam,  0, brvteam,  brvteam,  inder_state, empty_init,  ROT0, "Inder", "Brave Team",    MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )

// old CPU board, 7 digits, ay8910
GAME(1986,  canasta,  0, canasta,  canasta,  inder_state, empty_init,  ROT0, "Inder", "Canasta '86'",  MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )

// old CPU board, 7 digits, sound cpu with 2x ay8910
GAME(1986,  lapbylap, 0, lapbylap, lapbylap, inder_state, empty_init,  ROT0, "Inder", "Lap By Lap",    MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )

// new CPU board, sound board with msm5205
GAME(1987,  pinmoonl, 0,      inder, pinmoonl, inder_state, init_0, ROT0, "Inder", "Moon Light (Inder)", MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )
GAME(1988,  pinclown, 0,      inder, pinclown, inder_state, init_1, ROT0, "Inder", "Clown (Inder)",      MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )
GAME(1989,  corsario, 0,      inder, corsario, inder_state, init_1, ROT0, "Inder", "Corsario",           MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )
GAME(1990,  mundial,  0,      inder, mundial,  inder_state, init_1, ROT0, "Inder", "Mundial 90",         MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )
GAME(1991,  atleta,   0,      inder, atleta,   inder_state, init_1, ROT0, "Inder", "Atleta",             MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )
GAME(1991,  larana,   0,      inder, larana,   inder_state, init_0, ROT0, "Inder", "La Rana (set 1)",    MACHINE_IS_SKELETON_MECHANICAL | MACHINE_SUPPORTS_SAVE )
GAME(1991,  larana2,  larana, inder, larana,   inder_state, init_0, ROT0, "Inder", "La Rana (set 2)",    MACHINE_IS_SKELETON_MECHANICAL | MACHINE_SUPPORTS_SAVE )
GAME(1992,  ind250cc, 0,      inder, ind250cc, inder_state, init_1, ROT0, "Inder", "250 CC",             MACHINE_MECHANICAL | MACHINE_SUPPORTS_SAVE )

// Unknown sound hardware, unknown machine (using 'larana' inputs until proper ones are figured out).
GAME(1991, indunkgam, 0, inder, larana, inder_state, init_0, ROT0, "Inder", "unknown gambling game on Inder pinball hardware", MACHINE_IS_SKELETON_MECHANICAL | MACHINE_SUPPORTS_SAVE )
