/*
	File:		MacFork.cc

	Function:	Extract resources from a mac-style resource fork

	Author:		Andrew Willmott, (c) 1999

	Notes:		
*/

#include <stdio.h>
#include <stdlib.h>
#include "arg.h"

#include "Resource.h"
#include <netinet/in.h>

const Int NO_ID = 65536;

static const Char *gDataTypeID = 0;

Void SaveAsData(ResourceFile &resFile, ResShort id)
{
	ResLong     dataLen;
	Char        *data;
	FILE        *dataOut;
	Char        filename[1024];
	const Char  *name;	

	data = (Char*) resFile.GetResource(gDataTypeID, id, &dataLen);

	if (data)
	{
		printf("saving %s %d as data\n", gDataTypeID, id);

		name = resFile.GetResourceName(gDataTypeID, id);
		if (name)
			sprintf(filename, "%s.data", name);
		else
			sprintf(filename, "id_%d.data", id);

		dataOut = fopen(filename, "wb");

		fwrite(data, dataLen, 1, dataOut);

		fclose(dataOut);
	}
	else
		printf("%s %d doesn't exist\n", gDataTypeID, id);
}

Void SavePICT(ResourceFile &resFile, ResShort id)
{
	Char        *pict;
	ResLong     pictLen;
	FILE        *pictOut;
	ResLong     zero = 0;
	Char        filename[1024];
	const Char  *name;
	
	pict = (Char*) resFile.GetResource("PICT" , id, &pictLen);

	if (pict)
	{
		printf("saving PICT %d\n", id);

		name = resFile.GetResourceName("PICT" , id);
		if (name)
			sprintf(filename, "%s.pict", name);
		else
			sprintf(filename, "pict_%d.pict", id);

		pictOut = fopen(filename, "wb");

		fwrite(&zero, 4, 512 / 4, pictOut);	// write header
		fwrite(pict, pictLen, 1, pictOut);

		fclose(pictOut);
	}
	else
		printf("PICT %d doesn't exist\n", id);
}

struct ClrInfo
{
	ResShort	red;
	ResShort	green;
	ResShort	blue;
	ResShort	dummy1[5];
};

struct Palette
{
	ResShort	entries;
	ResShort	dummy1[7];
	ClrInfo		info[1];
};

Void SavePalette(ResourceFile &resFile, ResShort id)
{
	ResLong     len;
	Palette     *pltt;
	Int         i;
	FILE        *plttOut;
	Char        filename[1024];
	const Char  *name;
	
	pltt = (Palette*) resFile.GetResource("pltt", id, &len);
	if (pltt)
	{
		printf("saving pltt %d\n", id);

		name = resFile.GetResourceName("pltt" , id);
		if (name)
			sprintf(filename, "%s.pltt", name);
		else
			sprintf(filename, "pltt_%d.pltt", id);

		plttOut = fopen(filename, "wb");

		for (i = 0; i < ntohs(pltt->entries); i++)
			fprintf(plttOut, "%d %d %d\n", 
				Int((ntohs(pltt->info[i].red) + 0.5) / 256), 
				Int((ntohs(pltt->info[i].green) + 0.5) / 256), 
				Int((ntohs(pltt->info[i].blue) + 0.5) / 256)
			);

		fclose(plttOut);
	}
	else
		printf("pltt %d doesn't exist\n", id);
}

static Byte kStdPalette[256][3] =
{
#include "mac-std-palette.txt"
};

Void SaveIconFamily(ResourceFile &resFile, ResShort id)
{
	Byte        *icon;
	ResLong     iconLen;
	ResLong     i;
	FILE        *iconOut;
	Char        stub[1024], filename[1024];
	const Char  *name;
	
	name = resFile.GetResourceName("ICN#" , id);
	if (name)
		sprintf(stub, "%s", name);
	else
		sprintf(stub, "icon_%d", id);

	icon = (Byte*) resFile.GetResource("icl8" , id, &iconLen);
	if (icon)
	{
		printf("saving icl8 %d\n", id);

		FILE *iconOut;

		sprintf(filename, "%s.ppm", stub);
		iconOut = fopen(filename, "wb");
		fprintf(iconOut, "P6\n%d %d\n255\n", 32, 32);

		/* XXX should also do gamma correction here */
		for (i = 0; i < iconLen; i++)
			fprintf(iconOut, "%c%c%c", 
				kStdPalette[icon[i]][0],
				kStdPalette[icon[i]][1],
				kStdPalette[icon[i]][2]
			);
		fclose(iconOut);
	}
	else
		printf("icl8 %d doesn't exist\n", id);
	
	icon = (Byte*) resFile.GetResource("ICN#" , id, &iconLen);
	if (icon)
	{
		printf("saving ICN# %d\n", id);

		sprintf(filename, "%s-bw.pbm", stub);
		iconOut = fopen(filename, "wb");
		fprintf(iconOut, "P4\n%d %d\n", 32, 32);
		fwrite(icon, iconLen / 2, 1, iconOut);
		fclose(iconOut);
		sprintf(filename, "%s-mask.pbm", stub);
		iconOut = fopen(filename, "wb");
		fprintf(iconOut, "P4\n%d %d\n", 32, 32);
		fwrite(icon + iconLen / 2, iconLen / 2, 1, iconOut);
		fclose(iconOut);
	}
	else
		printf("ICN# %d doesn't exist\n", id);
}

typedef Void (*ResFunc)(ResourceFile &resFile, ResShort id);

Void IterateResFunc(ResourceFile &resFile, ResFunc resFunc, ResIDParam typeID, Int resID)
{
	if (resID == NO_ID)
	{
		Int		i, j;

		i = resFile.GetTypeNumber(typeID);
		if (i >= 0)
			for (j = 0; j < resFile.GetNumResources(i); j++)
				resFunc(resFile, resFile.resLists[i][j].id);
	}
	else
		resFunc(resFile, resID);
}


int main(int argc, char **argv)
{
	ResourceFile    resFile;
	Int				i;
	Int				pict, iconFam, pltt, resID = NO_ID;
	const Char		*filename = 0, *listTypeID = 0;
	Arg_form		*arg_format;

	// set options
	
	arg_format = arg_to_form(0,
		"", 									"usage: macfork [options]",
		"", 									"",
		
		"%S",		&filename,					"name of resource fork",
		
		"-list %S", &listTypeID,				"list resources of specified type only",
		"-pict",	ARG_FLAG(&pict),			"save PICT (as .pict file)",
		"-pltt",	ARG_FLAG(&pltt),			"save pltt (as text)",
		"-icon",	ARG_FLAG(&iconFam), 		"save icon (as .ppm files)",
		"-dump %S", &gDataTypeID,				"save resource of specified type as data file",
		"[%d]",		&resID,						"resource ID (if not given, all resources are selected)",

		"", 	"",
		"", 	"examples:",
		"", 	"  macfork Finder.rsrc                     list all finder resources",
		"", 	"  macfork Finder.rsrc -list 'snd '        list finder's 'snd ' resources",
		"", 	"  macfork Finder.rsrc -icon               extract all finder icons",
		"", 	"  macfork Finder.rsrc -pict 2764          extract PICT with id 2764",
		"", 	"  macfork Finder.rsrc -dump STR# 5000     save STR# resource 5000 as data file",

		0);

	// No arguments: print usage
	if (argc == 1)
	{
		arg_form_print(arg_format);
		exit(0);
	}
	// parse arguments
	if (arg_parse_argv(argc, argv, arg_format) < 0)
		exit(1);

	resFile.Open(filename);

	if (argc == 2)
	{
		printf("        ID       Size      Name\n");
		resFile.PrintAllResources();
	}
	else if (listTypeID)
	{
		if ((i = resFile.GetTypeNumber(listTypeID)) >= 0)
		{
			printf("        ID       Size      Name\n");
			resFile.PrintResources(i);
		}
		else
			fprintf(stderr, "No resources of that type!");
	}
	else if (pict)
		IterateResFunc(resFile, SavePICT, "PICT", resID);
	else if (pltt)
		IterateResFunc(resFile, SavePalette, "pltt", resID);
	else if (iconFam)
		IterateResFunc(resFile, SaveIconFamily, "ICN#", resID);
	else if (gDataTypeID)
		IterateResFunc(resFile, SaveAsData, gDataTypeID, resID);
	else
		printf("no conversion specified\n");
			
	resFile.Close();
	
	return(0);
}
