/* Copyright (C) 2000-2003 Markus Lausser (sgop@users.sf.net)
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>

#include "files.h"
#include "utils.h"
#include "dialog.h"
#include "exec.h"

file_t* file_new() {
  file_t* result;

  result = g_malloc(sizeof(file_t));
  result->longname = NULL;
  result->winname = NULL;
  result->shortname = NULL;
  result->filename = NULL;
  result->size = 0;
  result->user = NULL;
  result->net = NULL;
  result->ip = 0;
  result->md5 = NULL;
  result->bitrate = 0;
  result->frequency = 0;
  result->duration = 0;
  result->ping = 0;
  result->linespeed = 0;
  result->vbr = 0;
  result->visible = 0;
  result->media_type = 0;

  return result;
}

void file_destroy(gpointer data) {
  file_t *file = data;

  if (!file) return;
  if (file->longname) g_free(file->longname);
  if (file->winname) g_free(file->winname);
  if (file->user) g_free(file->user);
  if (file->md5) g_free(file->md5);
  g_free(file);
}

void file_set_longname(file_t* file, char* name) {
  if (!file || !name) return;

  if (file->longname) g_free(file->longname);
  file->longname = g_strdup(name);
  if (file->winname) g_free(file->winname);
  file->winname = g_strdup(name);
  local_to_napster_hide(file->winname);
  file->shortname = get_short_name(file->longname);
  file->filename = get_file_name(file->shortname);
  file->media_type = mtype_get(file->filename);
}

void file_set_winname(file_t* file, char* name) {
  if (!file || !name) return;

  if (file->longname) g_free(file->longname);
  file->longname = g_strdup(name);
  if (file->winname) g_free(file->winname);
  file->winname = g_strdup(name);
  napster_to_local(file->longname);
  file->shortname = get_short_name(file->longname);
  file->filename = get_file_name(file->shortname);
  file->media_type = mtype_get(file->filename);
}

file_t*
file_create_from_browse_response(net_t* net, char *data) {
  file_t *file;
  char* user;
  char* name;
  char* md5;
  char* size;
  char* rate;
  char* freq;
  char* dur;

  user = arg(data, 0);
  name = arg(NULL, 0);
  md5  = arg(NULL, 0);
  size = arg(NULL, 0);
  rate = arg(NULL, 0);
  freq = arg(NULL, 0);
  dur  = arg(NULL, 0);

  if (!dur) return NULL;

  file = file_new();
  if (!file) return NULL;
  file_set_winname(file, name);
  file->user = g_strdup(user);
  file->md5 = g_strdup(md5);
  file->net = net;
  file->size = strtoul(size, NULL, 0);
  file->bitrate = atoi(rate);
  file->frequency = atoi(freq);
  file->duration = atoi(dur);

  return file;
}

file_t*
file_create_from_search_response(net_t* net, char *data) {
  file_t *file;
  char* name;
  char* md5;
  char* size;
  char* rate;
  char* freq;
  char* dur;
  char* user;
  char* ip;
  char* speed;

  name  = arg(data, 0);
  md5   = arg(NULL, 0);
  size  = arg(NULL, 0);
  rate  = arg(NULL, 0);
  freq  = arg(NULL, 0);
  dur   = arg(NULL, 0);
  user  = arg(NULL, 0);
  ip    = arg(NULL, 0);
  speed = arg(NULL, 0);

  if (!speed) return NULL;

  file = file_new();
  if (!file) return NULL;
  file_set_winname(file, name);
  file->md5 = g_strdup(md5);
  file->size = strtoul(size, NULL, 10);
  file->bitrate = atoi(rate);
  file->frequency = atoi(freq);
  file->duration = atoi(dur);
  file->user = g_strdup(user);
  file->ip = BSWAP32(strtoul(ip, NULL, 10));
  file->linespeed = atoi(speed);
  file->net = net;

  return file;
}

file_t*
file_create_from_local(char *longname) {
  file_t *file;

  file = file_new();
  if (!file) return NULL;
  file_set_longname(file, longname);
  file->user = g_strdup("");
  file->md5 = g_strdup("00000000000000000000000000000000");
  file->bitrate = 32;
  file->frequency = 32000;
  file->duration = 60;

  return file;
}

int
file_compare(file_t* file1, file_t* file2) {
  if (file1->size != file2->size) return 1;
  //  if (file1->bitrate != file2->bitrate) return 1;
  //  if (file1->frequency != file2->frequency) return 1;
  //  if (file1->duration != file2->duration) return 1;
  if (strcmp(file1->longname, file2->longname)) return 1;
  return 0;
}

file_t* 
file_duplicate(file_t * file) {
  file_t *dest;

  dest = file_new();
  dest->longname = g_strdup(file->longname);
  dest->winname = g_strdup(file->winname);
  dest->media_type = file->media_type;
  dest->shortname = get_short_name(dest->longname);
  dest->filename = get_file_name(dest->shortname);
  if (file->user) dest->user = g_strdup(file->user);
  if (file->md5) dest->md5 = g_strdup(file->md5);
  dest->net = file->net;
  dest->size = file->size;
  dest->ip = file->ip;
  dest->bitrate = file->bitrate;
  dest->frequency = file->frequency;
  dest->duration = file->duration;
  dest->linespeed = file->linespeed;
  dest->media_type = file->media_type;
  dest->ping = file->ping;
  dest->vbr = file->vbr;

  return dest;
}

static char*
create_open_command(char* filename, char* app) {
  int cnt = 0;
  int fcnt = 0;
  char* pos;
  char* pos2;
  char* result;
  char* fname;
  int quo;
  
  pos = app;
  while (*pos) {
    if (*pos == '%' && (pos[1] == 'f' || pos[1] == 'F'))
      fcnt++;
    pos++;
  }

  if (fcnt == 0) return NULL;

  pos = filename;
  quo = 0;
  while (*pos) {
    if (*pos == '\'') {
      if (quo) { cnt++; quo = 0; }
      cnt++;
    } else {
      if (!quo) { cnt++; quo = 1; }
    }
    pos++;
  }
  if (quo) cnt++;
  
  fname = g_malloc0(strlen(filename)+cnt+1);
  pos = filename;
  pos2 = fname;
  quo = 0;
  while (*pos) {
    if (*pos == '\'') {
      if (quo) { *pos2++ = '\''; quo = 0; }
      *pos2++ = '\\';
    } else {
      if (!quo) { *pos2++ = '\''; quo = 1; }
    }
    *pos2++ = *pos++;
  }
  if (quo) *pos2++ = '\'';

  quo = strlen(fname);
  result = g_malloc0(strlen(app) + (quo-2)*fcnt + 1);

  pos = app;
  pos2 = result;
  while (*pos) {
    if (*pos == '%' && (pos[1] == 'f' || pos[1] == 'F')) {
      strcpy(pos2, fname);
      pos += 2;
      pos2 += quo;
    } else {
      *pos2++ = *pos++;
    }
  }
  g_free(fname);
  return result;
}

void play_file(char *filename, char* app) {
  struct stat st;
  char* command;

  //  printf("open [%s] with [%s]\n", filename, app?app:"(null)"); 
  if (!filename || !app) return;

  //  printf("checking file [%s]\n", filename);
  if (stat(filename, &st) < 0) {
    command = g_strdup_printf("File '%s' not found", filename);
    info_dialog(command);
    g_free(command);
    return;
  }

  command = create_open_command(filename, app);

  if (!command)
    exec_command_safe(0, 0, app, filename, 0);
  else
    exec_command_safe(0, 0, command, NULL, 0);

  g_free(command);
}

