#include "config.h"

#include "dht_controller.h"

#include "dht/dht_router.h"
#include "src/manager.h"
#include "torrent/exceptions.h"
#include "torrent/throttle.h"
#include "torrent/net/socket_address.h"
#include "torrent/net/network_config.h"
#include "torrent/utils/log.h"

#define LT_LOG(log_fmt, ...)                                            \
  lt_log_print_subsystem(torrent::LOG_DHT_CONTROLLER, "dht_controller", log_fmt, __VA_ARGS__);

namespace torrent::tracker {

DhtController::DhtController() = default;

DhtController::~DhtController() {
  stop();
}

bool
DhtController::is_valid() {
  auto lock = std::lock_guard(m_lock);
  return m_router != nullptr;
}

bool
DhtController::is_active() {
  auto lock = std::lock_guard(m_lock);
  return m_router && m_router->is_active();
}

bool
DhtController::is_receiving_requests() {
  auto lock = std::lock_guard(m_lock);
  return m_receive_requests;
}

uint16_t
DhtController::port() {
  auto lock = std::lock_guard(m_lock);
  return m_port;
}

void
DhtController::initialize(const Object& dht_cache) {
  auto lock = std::lock_guard(m_lock);

  if (m_router != nullptr)
    throw internal_error("DhtController::initialize() called with DHT already active.");

  // TODO: Create a seperate router listen open function, which will be called when bind address /
  // block ipv4/6 changes.

  // TODO: Bind address should be set at start, not initialize.

  auto bind_address = config::network_config()->bind_address_or_any_and_null();

  LT_LOG("initializing : %s", sa_pretty_str(bind_address.get()).c_str());

  try {
    // TODO: This should not be an internal error, just do it here for now.
    if (bind_address == nullptr)
      throw internal_error("DhtController::initialize() called but no valid bind address.");

    m_router = std::make_unique<DhtRouter>(dht_cache, bind_address.get());

  } catch (const torrent::local_error& e) {
    LT_LOG("initialization failed : %s", e.what());
  }
}

bool
DhtController::start() {
  auto lock = std::lock_guard(m_lock);

  if (m_router == nullptr)
    throw internal_error("DhtController::start() called without initializing first.");

  auto port = config::network_config()->override_dht_port();

  if (port == 0)
    port = config::network_config()->listen_port_or_throw();

  LT_LOG("starting : port:%d", port);

  try {
    m_router->start(port);
    m_port = port;

  } catch (const torrent::local_error& e) {
    LT_LOG("start failed : %s", e.what());
    return false;
  }

  return true;
}

void
DhtController::stop() {
  auto lock = std::lock_guard(m_lock);

  if (!m_router)
    return;

  LT_LOG("stopping", 0);

  m_router->stop();
  m_port = 0;
}

void
DhtController::set_receive_requests(bool state) {
  auto lock = std::lock_guard(m_lock);
  m_receive_requests = state;
}

void
DhtController::add_node(const sockaddr* sa, int port) {
  auto lock = std::lock_guard(m_lock);

  if (m_router)
    m_router->contact(sa, port);
}

void
DhtController::add_node(const std::string& host, int port) {
  auto lock = std::lock_guard(m_lock);

  if (m_router)
    m_router->add_contact(host, port);
}

Object*
DhtController::store_cache(Object* container) {
  auto lock = std::lock_guard(m_lock);

  if (!m_router)
    throw internal_error("DhtController::store_cache() called but DHT not initialized.");

  return m_router->store_cache(container);
}

DhtController::statistics_type
DhtController::get_statistics() {
  auto lock = std::lock_guard(m_lock);

  if (!m_router)
    throw internal_error("DhtController::get_statistics() called but DHT not initialized.");

  return m_router->get_statistics();
}

void
DhtController::reset_statistics() {
  auto lock = std::lock_guard(m_lock);

  if (!m_router)
    throw internal_error("DhtController::reset_statistics() called but DHT not initialized.");

  m_router->reset_statistics();
}

// TOOD: Throttle needs to be made thread-safe.

void
DhtController::set_upload_throttle(Throttle* t) {
  auto lock = std::lock_guard(m_lock);

  if (!m_router)
    throw internal_error("DhtController::set_upload_throttle() called but DHT not initialized.");

  if (m_router->is_active())
    throw internal_error("DhtController::set_upload_throttle() called while DHT server active.");

  m_router->set_upload_throttle(t->throttle_list());
}

void
DhtController::set_download_throttle(Throttle* t) {
  auto lock = std::lock_guard(m_lock);

  if (!m_router)
    throw internal_error("DhtController::set_download_throttle() called but DHT not initialized.");

  if (m_router->is_active())
    throw internal_error("DhtController::set_download_throttle() called while DHT server active.");

  m_router->set_download_throttle(t->throttle_list());
}

void
DhtController::announce(const HashString& info_hash, TrackerDht* tracker) {
  auto lock = std::lock_guard(m_lock);

  if (!m_router)
    throw internal_error("DhtController::announce() called but DHT not initialized.");

  m_router->announce(info_hash, tracker);
}

void
DhtController::cancel_announce(const HashString* info_hash, const torrent::TrackerDht* tracker) {
  auto lock = std::lock_guard(m_lock);

  if (!m_router)
    throw internal_error("DhtController::cancel_announce() called but DHT not initialized.");

  m_router->cancel_announce(info_hash, tracker);
}

} // namespace torrent::tracker
