/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include <libsyncml/syncml.h>

#include <libsyncml/syncml_internals.h>
#include <libsyncml/sml_elements_internals.h>
#include <libsyncml/sml_command_internals.h>
#include <libsyncml/sml_devinf_internals.h>
#include <libsyncml/sml_session_internals.h>

#include "sml_xml_parse.h"
#include "sml_xml_parse_internals.h"

#define BUFFER_SIZE 500

/**
 * @defgroup Parser SyncML XML Parser
 * @ingroup PrivateAPI
 * @brief Interfaces to parse syncml xml messages
 * 
 */
/*@{*/

static SmlBool _smlXmlParserStep(SmlXmlParser *parser)
{
	SmlBool ret = FALSE;
	do {
		ret = (xmlTextReaderRead(parser->reader) == 1) ? TRUE : FALSE;
	} while (ret && (xmlTextReaderNodeType(parser->reader) == XML_READER_TYPE_DOCUMENT_TYPE || \
		xmlTextReaderNodeType(parser->reader) == XML_READER_TYPE_WHITESPACE || \
		xmlTextReaderNodeType(parser->reader) == XML_READER_TYPE_SIGNIFICANT_WHITESPACE));
		
	return ret;
}

static SmlBool _smlXmlParserStepData(SmlXmlParser *parser)
{
	SmlBool ret = FALSE;
	do {
		ret = (xmlTextReaderRead(parser->reader) == 1) ? TRUE : FALSE;
	} while (ret && (xmlTextReaderNodeType(parser->reader) == XML_READER_TYPE_DOCUMENT_TYPE || \
		xmlTextReaderNodeType(parser->reader) == XML_READER_TYPE_WHITESPACE));
		
	return ret;
}

static SmlBool _smlXmlParserExpectNode(SmlXmlParser *parser, int type, SmlBool empty, const char *name, SmlError **error)
{
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No node at all");
		return FALSE;
	}
	
	if (xmlTextReaderNodeType(parser->reader) != type) {
		smlErrorSet(error, SML_ERROR_GENERIC, "wrong node type");
		return FALSE;
	}
	
		
	switch (type) {
		case XML_NODE_START:
		case XML_NODE_CLOSE:
			if (name) {
				if (!xmlTextReaderConstName(parser->reader)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "no name");
					return FALSE;
				}
			
				if (strcmp((char *)xmlTextReaderConstName(parser->reader), name)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Wrong name");
					return FALSE;
				}
			}
			break;
		case XML_NODE_TEXT:
			if (!empty && !xmlTextReaderConstName(parser->reader)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Empty.");
				return FALSE;
			}
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown node type");
			return FALSE;
	}
	
	return TRUE;
}

static SmlBool _smlXmlParserGetID(SmlXmlParser *parser, unsigned int *id, const char *name, SmlError **error)
{
	smlAssert(parser);
	smlAssert(id);
	
	if (*id) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Id already set");
		goto error;
	}
	
	if (!_smlXmlParserExpectNode(parser, XML_NODE_TEXT, FALSE, NULL, error))
		goto error;
	
	*id = atoi(g_strstrip((char *)xmlTextReaderConstValue(parser->reader)));
	
	if (!_smlXmlParserExpectNode(parser, XML_NODE_CLOSE, FALSE, name, error))
		goto error;
	
	return TRUE;

error:
	return FALSE;
}

static SmlBool _smlXmlParserGetString(SmlXmlParser *parser, char **string, const char *name, SmlError **error)
{
	smlAssert(parser);
	smlAssert(string);
	
	if (*string) {
		smlErrorSet(error, SML_ERROR_GENERIC, "string already set");
		goto error;
	}
	
	/* required for empty elements like <FwV/> */
	if (xmlTextReaderIsEmptyElement(parser->reader)) {
		*string = g_strdup("");
		return TRUE;
	}
	
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No node at all");
		goto error_clear;
	}
	
	if (xmlTextReaderNodeType(parser->reader) == XML_NODE_TEXT) {
		*string = g_strstrip(g_strdup((char *)xmlTextReaderConstValue(parser->reader)));
		
		if (!_smlXmlParserExpectNode(parser, XML_NODE_CLOSE, FALSE, name, error))
			goto error_clear;
	} else if (xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
		*string = g_strdup("");
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "wrong node type");
		goto error_clear;
	}
	
	return TRUE;

error_clear:
	*string = NULL;
error:
	return FALSE;
}

static SmlBool _smlXmlParserGetData(SmlXmlParser *parser, char **string, unsigned int *size, const char *name, SmlError **error)
{
	smlAssert(parser);
	smlAssert(string);
	int rc = 0;
	
	if (*string) {
		smlErrorSet(error, SML_ERROR_GENERIC, "string already set");
		return FALSE;
	}
	
	
	xmlBuffer *buffer = xmlBufferCreateSize(BUFFER_SIZE);
	if (!buffer) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to create new buffer");
		goto error;
	}
	
	xmlTextWriter *writer = xmlNewTextWriterMemory(buffer, 0);
	if (!writer) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to create new writer");
		goto error_free_buffer;
	}
		
	while (1) {
		if (!_smlXmlParserStepData(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_writer;
		}
		
		switch (xmlTextReaderNodeType(parser->reader)) {
			case XML_NODE_CLOSE:
				if (!strcmp((char *)xmlTextReaderConstName(parser->reader), name))
					goto out;
					
				if (!xmlTextReaderIsEmptyElement(parser->reader))
					rc = xmlTextWriterEndElement(writer);

				if (rc < 0) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Unable to end node");
					goto error_free_writer;
				}
				break;
			case XML_NODE_START:
				if (!xmlTextReaderIsEmptyElement(parser->reader))
					rc = xmlTextWriterStartElementNS(writer, NULL, xmlTextReaderConstName(parser->reader), NULL);
				else
					rc = xmlTextWriterWriteElementNS(writer, NULL, xmlTextReaderConstName(parser->reader), NULL, (const xmlChar*)"");

				if (rc < 0) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Unable to start node");
					goto error_free_writer;
				}
				break;
			case XML_NODE_CDATA:
			case XML_NODE_TEXT:
				rc = xmlTextWriterWriteRaw(writer, xmlTextReaderConstValue(parser->reader));
				if (rc < 0) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Unable to add string");
					goto error_free_writer;
				}
				break;
			case XML_READER_TYPE_SIGNIFICANT_WHITESPACE:
				rc = xmlTextWriterWriteRaw(writer, xmlTextReaderConstValue(parser->reader));
				if (rc < 0) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Unable to add string");
					goto error_free_writer;
				}
				break;
			default:
				smlErrorSet(error, SML_ERROR_GENERIC, "Unknown node type");
				goto error_free_writer;
		}
	}

out:
	if (xmlTextWriterEndDocument(writer) < 0) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to end writer");
		goto error_free_writer;
	}
	
	
	xmlFreeTextWriter(writer);
	*string = (char *)xmlBufferContent(buffer);
	if (size)
		*size = xmlBufferLength(buffer) - 1;
	
	//Hack to avoid memory copy
	g_free(buffer);
	
	return TRUE;

error_free_writer:
	xmlFreeTextWriter(writer);
error_free_buffer:
	xmlBufferFree(buffer);
error:
	*string = NULL;
	*size = 0;
	return FALSE;
}

static SmlBool _smlSyncHeaderParseDTD(SmlProtocolVersion *version, SmlXmlParser *parser, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, version, parser, error);
	smlAssert(parser);
	smlAssert(version);
	
	if (*version) {
		smlErrorSet(error, SML_ERROR_GENERIC, "dtd already set");
		smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
		return FALSE;
	}
	
	char *dtd = NULL;
	if (!_smlXmlParserGetString(parser, &dtd, SML_ELEMENT_VERDTD, error))
		goto error;
	
	if (!strcmp(dtd, "1.0"))
		*version = SML_VERSION_10;
	else if (!strcmp(dtd, "1.1"))
		*version = SML_VERSION_11;
	else if (!strcmp(dtd, "1.2"))
		*version = SML_VERSION_12;
	else {
		g_free(dtd);
		smlErrorSet(error, SML_ERROR_GENERIC, "Unknown VerDTD");
		goto error;
	}
	g_free(dtd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	*version = SML_VERSION_UNKNOWN;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlSyncHeaderParseProto(SmlProtocolType *type, SmlXmlParser *parser, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, type, parser, error);
	smlAssert(parser);
	smlAssert(type);
	
	if (*type) {
		smlErrorSet(error, SML_ERROR_GENERIC, "protocol already set");
		smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
		return FALSE;
	}
	
	char *proto = NULL;
	if (!_smlXmlParserGetString(parser, &proto, SML_ELEMENT_VERPROTO, error))
		goto error;
	
	if (!strcmp(proto, SML_VERSION_STRING_10) || !strcmp(proto, SML_VERSION_STRING_11) || !strcmp(proto, SML_VERSION_STRING_12))
		*type = SML_PROTOCOL_SYNCML;
	else {
		g_free(proto);
		smlErrorSet(error, SML_ERROR_GENERIC, "Unknown VerProto");
		goto error;
	}
	g_free(proto);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	*type = SML_PROTOCOL_UNKNOWN;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}


static SmlBool _smlLocationParse(SmlLocation **location, SmlXmlParser *parser, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, location, parser, error);
	smlAssert(parser);
	smlAssert(location);
	
	if (*location) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Location already set");
		smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
		return FALSE;
	}
	
	*location = smlTryMalloc0(sizeof(SmlLocation), error);
	if (!location)
		goto error;
	(*location)->refCount = 1;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_location;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TARGET) && \
			xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCE) && \
			xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_location;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_LOCURI)) {
			if (!_smlXmlParserGetString(parser, &((*location)->locURI), SML_ELEMENT_LOCURI, error))
				goto error_free_location;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_LOCNAME)) {
			if (!_smlXmlParserGetString(parser, &((*location)->locName), SML_ELEMENT_LOCNAME, error))
				goto error_free_location;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node. expected SyncHdr");
			goto error_free_location;
		}
	}
	
	if (!(*location)->locURI) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No locURI set");
		goto error_free_location;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_location:
	smlLocationUnref(*location);
error:
	*location = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlAnchorParse(SmlAnchor **anchor, SmlXmlParser *parser, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, anchor, parser, error);
	smlAssert(parser);
	smlAssert(anchor);
	
	if (*anchor) {
		smlErrorSet(error, SML_ERROR_GENERIC, "anchor already set");
		smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
		return FALSE;
	}
	
	*anchor = smlTryMalloc0(sizeof(SmlAnchor), error);
	if (!anchor)
		goto error;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_anchor;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ANCHOR) && \
			xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_anchor;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_NEXT)) {
			if (!_smlXmlParserGetString(parser, &((*anchor)->next), SML_ELEMENT_NEXT, error))
				goto error_free_anchor;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_LAST)) {
			if (!_smlXmlParserGetString(parser, &((*anchor)->last), SML_ELEMENT_LAST, error))
				goto error_free_anchor;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error_free_anchor;
		}
	}
	
	if (!(*anchor)->next) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No next set");
		goto error_free_anchor;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_anchor:
	smlAnchorFree(*anchor);
error:
	*anchor = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlChalMetaParse(SmlXmlParser *parser, char **format, char **type, char **nonce, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p, %p, %p, %p)", __func__, parser, format, type, nonce, error);
	smlAssert(parser);
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error;
		}

		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error;
		}
		
		if (type && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TYPE)) {
			if (!_smlXmlParserGetString(parser, type, SML_ELEMENT_TYPE, error))
				goto error;
		} else if (format && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_FORMAT)) {
			if (!_smlXmlParserGetString(parser, format, SML_ELEMENT_FORMAT, error))
				goto error;
		} else if (format && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_NEXTNONCE)) {
			if (!_smlXmlParserGetString(parser, nonce, SML_ELEMENT_NEXTNONCE, error))
				goto error;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node: %s", xmlTextReaderConstName(parser->reader));
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	if (format)
		*format = NULL;
	if (nonce)
		*nonce = NULL;
	if (type)
		*type = NULL;

	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlCommandMetaParse(SmlXmlParser *parser, char **format, char **type, SmlAnchor **anchor, unsigned int *size, int *maxobjsize, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p, %p, %p, %p)", __func__, parser, format, type, anchor, size, maxobjsize, error);
	smlAssert(parser);
	
	if (maxobjsize)
		*maxobjsize = -1;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error;
		}

		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MEM)) {
			/* Ignored for now */
			if (xmlTextReaderNext(parser->reader) != 1) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Unable to skip mem node");
				goto error;
			}
		}

		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error;
		}
		
		if (type && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TYPE)) {
			if (!_smlXmlParserGetString(parser, type, SML_ELEMENT_TYPE, error))
				goto error;
		} else if (anchor && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ANCHOR)) {
			if (!_smlAnchorParse(anchor, parser, error))
				goto error;
		} else if (format && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_FORMAT)) {
			if (!_smlXmlParserGetString(parser, format, SML_ELEMENT_FORMAT, error))
				goto error;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MARK)) {
			/* Ignore the mark for now */
			char *mark = NULL;
			if (!_smlXmlParserGetString(parser, &mark, SML_ELEMENT_MARK, error))
				goto error;
			g_free(mark);
		} else if (size && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SIZE)) {
			if (!_smlXmlParserGetID(parser, size, SML_ELEMENT_SIZE, error))
				goto error;
		} else if (maxobjsize && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAXOBJSIZE)) {
			unsigned int loc_maxobjsize = 0;
			if (!_smlXmlParserGetID(parser, &loc_maxobjsize, SML_ELEMENT_MAXOBJSIZE, error))
				goto error;
			*maxobjsize = loc_maxobjsize;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_VERSION)) {
			/* Ignore the version for now */
			char *version = NULL;
			if (!_smlXmlParserGetString(parser, &version, SML_ELEMENT_VERSION, error))
				goto error;
			g_free(version);
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node: %s", xmlTextReaderConstName(parser->reader));
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	if (format)
		*format = NULL;
	if (anchor)
		*anchor = NULL;
	if (type)
		*type = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlItem *_smlItemParse(SmlXmlParser *parser, SmlCommand *cmd, SmlCommandType type, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %p)", __func__, parser, cmd, type, error);
	smlAssert(parser);

	if (parser->gotMoreData) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Last item already had more data set");
		goto error;
	}
	
	SmlItem *item = smlItemNew(0, error);
	if (!item)
		goto error;
	
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_item;
	}
	
	while (1) {
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ITEM) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_item;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCE)) {
			SmlLocation *source = NULL;
			if (!_smlLocationParse(&source, parser, error))
				goto error_free_item;
			
			smlItemSetSource(item, source);
			smlLocationUnref(source);
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TARGET)) {
			SmlLocation *target = NULL;
			if (!_smlLocationParse(&target, parser, error))
				goto error_free_item;
			
			smlItemSetTarget(item, target);
			smlLocationUnref(target);
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META)) {
			switch (type) {
				case SML_COMMAND_TYPE_ALERT:;
					/* Some phones send the maxobjsize in the alert */
					if (!_smlCommandMetaParse(parser, NULL, &(cmd->private.alert.contentType), &cmd->private.alert.anchor, NULL, &(cmd->private.alert.maxObjSize), error))
						goto error_free_item;
					break;
				case SML_COMMAND_TYPE_ADD:
				case SML_COMMAND_TYPE_DELETE:	
				case SML_COMMAND_TYPE_REPLACE:
					if (!_smlCommandMetaParse(parser, NULL, &(item->contenttype), NULL, &cmd->size, NULL, error))
						goto error_free_item;
					break;
				case SML_COMMAND_TYPE_RESULTS:
				case SML_COMMAND_TYPE_PUT:;
					/* Some phones send the type information for the devinf put
					 * not in the put itself but in the item */
					if (!_smlCommandMetaParse(parser, NULL, &(item->contenttype), NULL, NULL, NULL, error))
						goto error_free_item;
						
					if (!item->contenttype)
						goto error_free_item;
					break;
				default:
					smlErrorSet(error, SML_ERROR_GENERIC, "Unknown command type");
					goto error_free_item;
			}
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DATA)) {
			switch (type) {
				case SML_COMMAND_TYPE_ALERT:
					if (!_smlXmlParserStep(parser)) {
						smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
						goto error_free_item;
					}
					
					SmlAnchor *anchor = NULL;
					if (!_smlAnchorParse(&anchor, parser, error))
						goto error_free_item;
					
					item->anchor = anchor;
					
					if (!_smlXmlParserStep(parser)) {
						smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
						goto error_free_item;
					}
					break;
				default:;
					char *data = NULL;
					unsigned int size = 0;
					if (!_smlXmlParserGetData(parser, &data, &size, SML_ELEMENT_DATA, error))
						goto error_free_item;
					
					if (!smlItemAddData(item, data, size, error)) {
						g_free(data);
						goto error_free_item;
					}
					
					g_free(data);
			}
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MOREDATA)) {
			if (parser->version == SML_VERSION_10) {
				smlErrorSet(error, SML_ERROR_GENERIC, "SyncML 1.0 does not allow MoreData");
				goto error_free_item;
			}
			
			item->moreData = TRUE;
			parser->gotMoreData = TRUE;
			if (!_smlXmlParserStep(parser)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes2");
				goto error_free_item;
			}
			
			if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MOREDATA) && \
				xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
				if (!_smlXmlParserStep(parser)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes3");
					goto error_free_item;
				}
			}
			continue;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node %s", xmlTextReaderConstName(parser->reader));
			goto error_free_item;
		}
		
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_item;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return item;

error_free_item:
	smlItemUnref(item);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlCred *_smlCredParse(SmlXmlParser *parser, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, parser, error);
	smlAssert(parser);
	char *format = NULL;
	char *type = NULL;
	
	SmlCred *cred = smlTryMalloc0(sizeof(SmlCred), error);
	if (!cred)
		goto error;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_cred;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CRED) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_cred;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META)) {
			if (!_smlCommandMetaParse(parser, &format, &type, NULL, NULL, NULL, error))
				goto error_free_cred;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DATA)) {
			if (!_smlXmlParserGetString(parser, &cred->data, SML_ELEMENT_DATA, error))
				goto error_free_cred;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node %s", xmlTextReaderConstName(parser->reader));
			goto error_free_cred;
		}
	}
		
	if (!cred->data)  {
		smlErrorSet(error, SML_ERROR_GENERIC, "Creds misformed");
		goto error_free_cred;
	}
	
	if (!format || !strcmp(format, SML_BASE64)) {
		cred->format = SML_FORMAT_TYPE_BASE64;
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unknown format");
		goto error_free_cred;
	}
	
	if (!type || !strcmp(type, SML_AUTH_BASIC)) {
		cred->type = SML_AUTH_TYPE_BASIC;
	} else if (!strcmp(type, SML_AUTH_MD5)) {
		cred->type = SML_AUTH_TYPE_MD5;
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unknown type");
		goto error_free_cred;
	}
	
	if (format)
		g_free(format);
	if (type)
		g_free(type);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cred);
	return cred;

error_free_cred:
	smlCredFree(cred);
error:
	if (format)
		g_free(format);
	if (type)
		g_free(type);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlChal *_smlChalParse(SmlXmlParser *parser, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, parser, error);
	smlAssert(parser);
	char *format = NULL;
	char *type = NULL;
	
	SmlChal *chal = smlTryMalloc0(sizeof(SmlChal), error);
	if (!chal)
		goto error;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_chal;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CHAL) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_chal;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META)) {
			if (!_smlChalMetaParse(parser, &format, &type, &chal->nonce, error))
				goto error_free_chal;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node %s", xmlTextReaderConstName(parser->reader));
			goto error_free_chal;
		}
	}
		
	if (!format || !strcmp(format, SML_BASE64)) {
		chal->format = SML_FORMAT_TYPE_BASE64;
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unknown format");
		goto error_free_chal;
	}
	
	if (!type || !strcmp(type, SML_AUTH_BASIC)) {
		chal->type = SML_AUTH_TYPE_BASIC;
	} else if (!strcmp(type, SML_AUTH_MD5)) {
		chal->type = SML_AUTH_TYPE_MD5;
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unknown type");
		goto error_free_chal;
	}
	
	if (format)
		g_free(format);
	if (type)
		g_free(type);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, chal);
	return chal;

error_free_chal:
	smlChalFree(chal);
error:
	if (format)
		g_free(format);
	if (type)
		g_free(type);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool _smlChangeParse(SmlXmlParser *parser, SmlCommand **cmd, SmlCommandType type, const char *name, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %s, %p)", __func__, parser, type, name, error);
	smlAssert(parser);
	smlAssert(name);
	char *contenttype = NULL;
	
	*cmd = smlCommandNew(type, error);
	if (!*cmd)
		goto error;	
	(*cmd)->refCount = 1;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), name) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDID)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->cmdID), SML_ELEMENT_CMDID, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ITEM)) {
			(*cmd)->private.change.item = _smlItemParse(parser, *cmd, type, error);
			if (!(*cmd)->private.change.item)
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META)) {
			if (!_smlCommandMetaParse(parser, NULL, &contenttype, NULL, &(*cmd)->size, NULL, error))
				goto error_free_cmd;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error_free_cmd;
		}
	}
	
	if (!(*cmd)->cmdID) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No cmdid set");
		goto error_free_cmd;
	}
	
	if (!(*cmd)->private.change.item) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No item set");
		goto error_free_cmd;
	}
	
	/* some clients set the contentype in the item, don't overwrite it with NULL */
	if ((*cmd)->private.change.item->contenttype == NULL) {
		if (contenttype) {
			(*cmd)->private.change.item->contenttype = contenttype;
			contenttype = NULL;
		}
	}
	
	if ((*cmd)->size)
		(*cmd)->private.change.item->size = (*cmd)->size;
	
	switch (type) {
		case SML_COMMAND_TYPE_ADD:
			(*cmd)->private.change.type = SML_CHANGE_ADD;
			break;
		case SML_COMMAND_TYPE_REPLACE:
			(*cmd)->private.change.type = SML_CHANGE_REPLACE;
			break;
		case SML_COMMAND_TYPE_DELETE:
			(*cmd)->private.change.type = SML_CHANGE_DELETE;
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown change type set");
			goto error_free_cmd;
	}
	
	if ((*cmd)->private.change.item->source) {
		(*cmd)->source = smlLocationClone((*cmd)->private.change.item->source, error);
		if (!(*cmd)->source)
			goto error_free_cmd;
	}
	
	if ((*cmd)->private.change.item->target) {
		(*cmd)->target = smlLocationClone((*cmd)->private.change.item->target, error);
		if (!(*cmd)->target)
			goto error_free_cmd;
	}
	
	/* Step once more */
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_cmd;
	}
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, *cmd);
	return TRUE;

error_free_cmd:
	smlCommandUnref(*cmd);
error:
	if (contenttype)
		g_free(contenttype);
	*cmd = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlMessageParseSynchdrMeta(SmlXmlParser *parser, unsigned int *maxmsgsize, unsigned int *maxobjsize, char **emi, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, parser, maxmsgsize, maxobjsize, error);
	smlAssert(parser);
	smlAssert(maxmsgsize);
	smlAssert(maxobjsize);
	smlAssert(emi);

	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META) && \
			xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAXMSGSIZE)) {
			if (!_smlXmlParserGetID(parser, maxmsgsize, SML_ELEMENT_MAXMSGSIZE, error))
				goto error;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAXOBJSIZE)) {
			if (!_smlXmlParserGetID(parser, maxobjsize, SML_ELEMENT_MAXOBJSIZE, error))
				goto error;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_EMI)) {
			if (!_smlXmlParserGetString(parser, emi, SML_ELEMENT_EMI, error))
				goto error;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node. expected MaxMsgSize, MaxObjSize or EMI. Instead of that: %s", xmlTextReaderConstName(parser->reader));
			goto error;
		}
	}
	
	if (!(*maxmsgsize) && !(*maxobjsize)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No maxmsgsize set");
		goto error;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	*maxmsgsize = 0;
	*maxobjsize = 0;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlAlertParse(SmlXmlParser *parser, SmlCommand **cmd, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, cmd, error);
	smlAssert(parser);
	smlAssert(cmd);
		
	*cmd = smlCommandNew(SML_COMMAND_TYPE_ALERT, error);
	if (!*cmd)
		goto error;
	(*cmd)->refCount = 1;
	(*cmd)->private.alert.maxObjSize = -1;
		
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ALERT) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDID)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->cmdID), SML_ELEMENT_CMDID, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ITEM)) {
			SmlItem *item = _smlItemParse(parser, (*cmd), SML_COMMAND_TYPE_ALERT, error);
			if (!item)
				goto error_free_cmd;
			
			(*cmd)->target = item->target;
			item->target = NULL;
			(*cmd)->source = item->source;
			item->source = NULL;
			
			smlItemUnref(item);
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DATA)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->private.alert.type), SML_ELEMENT_DATA, error))
				goto error_free_cmd;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node %s", xmlTextReaderConstName(parser->reader));
			goto error_free_cmd;
		}
	}
	
	if (!(*cmd)->private.alert.type) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No alert type set");
		goto error_free_cmd;
	}
	
	/* Step once more */
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_cmd;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_cmd:
	smlCommandUnref(*cmd);
error:
	*cmd = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlParserResult _smlCommandSyncParse(SmlXmlParser *parser, SmlCommand **cmd, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, cmd, error);
	smlAssert(parser);
	smlAssert(cmd);
		
	*cmd = smlCommandNew(SML_COMMAND_TYPE_SYNC, error);
	if (!*cmd)
		goto error;
	(*cmd)->private.sync.maxObjSize = -1;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_cmd;
		}
		
		if (xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE && !strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNC))
			break;
		else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ADD) || \
		!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_REPLACE) || \
		!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DELETE))
			break;
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDID)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->cmdID), SML_ELEMENT_CMDID, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ITEM)) {
			SmlItem *item = _smlItemParse(parser, (*cmd), SML_COMMAND_TYPE_SYNC, error);
			if (!item)
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META)) {
			if (!_smlCommandMetaParse(parser, NULL, NULL, NULL, NULL, &((*cmd)->private.sync.maxObjSize), error))
				goto error;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TARGET)) {
			if (!_smlLocationParse(&(*cmd)->target, parser, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCE)) {
			if (!_smlLocationParse(&(*cmd)->source, parser, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_NUMBEROFCHANGES)) {
			(*cmd)->private.sync.hasNumChanged = TRUE;
			if (!_smlXmlParserGetID(parser, &((*cmd)->private.sync.numChanged), SML_ELEMENT_NUMBEROFCHANGES, error))
				goto error_free_cmd;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node: %s", xmlTextReaderConstName(parser->reader));
			goto error_free_cmd;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_cmd:
	smlCommandUnref(*cmd);
error:
	*cmd = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlMapItem *_smlMapItemParse(SmlXmlParser *parser, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, parser, error);
	smlAssert(parser);
	
	SmlMapItem *item = smlTryMalloc0(sizeof(SmlMapItem), error);
	if (!item)
		goto error;
	item->refCount = 1;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_item;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAPITEM) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_item;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCE)) {
			if (!_smlLocationParse(&item->source, parser, error))
				goto error_free_item;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TARGET)) {
			if (!_smlLocationParse(&item->target, parser, error))
				goto error_free_item;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node %s", xmlTextReaderConstName(parser->reader));
			goto error_free_item;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return item;

error_free_item:
	smlMapItemUnref(item);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool _smlCommandMapParse(SmlXmlParser *parser, SmlCommand **cmd, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, cmd, error);
	smlAssert(parser);
	smlAssert(cmd);
	
	*cmd = smlCommandNew(SML_COMMAND_TYPE_MAP, error);
	if (!*cmd)
		goto error;
	(*cmd)->refCount = 1;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAP) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDID)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->cmdID), SML_ELEMENT_CMDID, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAPITEM)) {
			SmlMapItem *item = _smlMapItemParse(parser, error);
			if (!item)
				goto error_free_cmd;
			(*cmd)->private.map.items = g_list_append((*cmd)->private.map.items, item);
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TARGET)) {
			if (!_smlLocationParse(&(*cmd)->target, parser, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCE)) {
			if (!_smlLocationParse(&(*cmd)->source, parser, error))
				goto error_free_cmd;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error_free_cmd;
		}
	}
	
	/* Step once more */
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_cmd;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_cmd:
	smlCommandUnref(*cmd);
error:
	*cmd = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlCommandAccessParse(SmlXmlParser *parser, SmlCommand **cmd, SmlCommandType type, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %p)", __func__, parser, cmd, type, error);
	smlAssert(parser);
	smlAssert(cmd);
	char *contenttype = NULL;
		
	*cmd = smlCommandNew(type, error);
	if (!*cmd)
		goto error;
	(*cmd)->refCount = 1;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_cmd;
		}
		
		if ((*cmd)->type == SML_COMMAND_TYPE_PUT &&!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_PUT) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if ((*cmd)->type == SML_COMMAND_TYPE_GET &&!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_GET) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDID)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->cmdID), SML_ELEMENT_CMDID, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ITEM)) {
			(*cmd)->private.access.item = _smlItemParse(parser, (*cmd), (*cmd)->type, error);
			if (!(*cmd)->private.access.item)
				goto error_free_cmd;
			
			(*cmd)->target = (*cmd)->private.access.item->target;
			(*cmd)->private.access.item->target = NULL;
			(*cmd)->source = (*cmd)->private.access.item->source;
			(*cmd)->private.access.item->source = NULL;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META)) {
			char *format = NULL;
			if (!_smlCommandMetaParse(parser, &format, &contenttype, NULL, NULL, NULL, error))
				goto error_free_cmd;
			g_free(format);
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error_free_cmd;
		}
	}
	
	if (!(*cmd)->private.access.item) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Put/Get is missing item");
		goto error_free_cmd;
	}
	
	/* We only use the content type of the put command if the item itself did not have
	 * a content type set */
	if (!(*cmd)->private.access.item->contenttype && contenttype)
		(*cmd)->private.access.item->contenttype = g_strdup(contenttype);
	
	if (contenttype)
		g_free(contenttype);
	
	/* Step once more */
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_cmd;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_cmd:
	smlCommandUnref(*cmd);
error:
	*cmd = NULL;
	if (contenttype)
		g_free(contenttype);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlResultsParse(SmlXmlParser *parser, SmlCommand **cmd, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, cmd, error);
	smlAssert(parser);
	smlAssert(cmd);
	char *contenttype = NULL;
	char *locURI = NULL;
		
	*cmd = smlCommandNew(SML_COMMAND_TYPE_RESULTS, error);
	if (!*cmd)
		goto error;
	(*cmd)->refCount = 1;
	
	(*cmd)->private.results.status = smlTryMalloc0(sizeof(SmlStatus), error);
	if (!(*cmd)->private.results.status)
		goto error;
	(*cmd)->private.results.status->refCount = 1;
	(*cmd)->private.results.status->result = (*cmd);
	(*cmd)->private.results.status->type = SML_COMMAND_TYPE_RESULTS;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_cmd;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_RESULTS) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_cmd;
		}
		
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDID)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->cmdID), SML_ELEMENT_CMDID, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MSGREF)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->private.results.status->msgRef), SML_ELEMENT_MSGREF, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDREF)) {
			if (!_smlXmlParserGetID(parser, &((*cmd)->private.results.status->cmdRef), SML_ELEMENT_CMDREF, error))
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCEREF)) {
			if (!_smlXmlParserGetString(parser, &locURI, SML_ELEMENT_SOURCEREF, error))
				goto error_free_cmd;
			
			(*cmd)->private.results.status->sourceRef = smlLocationNew(locURI, NULL, error);
			g_free(locURI);
			locURI = NULL;
			if (!(*cmd)->private.results.status->sourceRef)
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TARGETREF)) {
			if (!_smlXmlParserGetString(parser, &locURI, SML_ELEMENT_TARGETREF, error))
				goto error_free_cmd;
				
			(*cmd)->private.results.status->targetRef = smlLocationNew(locURI, NULL, error);
			g_free(locURI);
			locURI = NULL;
			if (!(*cmd)->private.results.status->targetRef)
				goto error_free_cmd;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ITEM)) {
			(*cmd)->private.results.status->item = _smlItemParse(parser, (*cmd), (*cmd)->type, error);
			if (!(*cmd)->private.results.status->item)
				goto error_free_cmd;
			
			(*cmd)->target = (*cmd)->private.results.status->item->target;
			(*cmd)->private.results.status->item->target = NULL;
			(*cmd)->source = (*cmd)->private.results.status->item->source;
			(*cmd)->private.results.status->item->source = NULL;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META)) {
			char *format = NULL;
			if (!_smlCommandMetaParse(parser, &format, &contenttype, NULL, NULL, NULL, error))
				goto error_free_cmd;
			g_free(format);
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error_free_cmd;
		}
	}
	
	if (!(*cmd)->private.results.status->item) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Result is missing item");
		goto error_free_cmd;
	}
	
	/* We only use the content type of the put command if the item itself did not have
	 * a content type set */
	if (!(*cmd)->private.results.status->item->contenttype) {
		if (!contenttype) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Result is missing content type");
			goto error_free_cmd;
		}
		
		(*cmd)->private.results.status->item->contenttype = g_strdup(contenttype);
	}
	
	if (contenttype)
		g_free(contenttype);
	
	/* Step once more */
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_cmd;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_cmd:
	smlCommandUnref(*cmd);
error:
	*cmd = NULL;
	if (contenttype)
		g_free(contenttype);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** @brief Start the parsing
 * 
 * This will set everything up and parse until the SyncHdr
 * 
 */
SmlBool smlXmlParserStart(SmlXmlParser *parser, const char *data, unsigned int size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %p)", __func__, parser, data, size, error);
	smlAssert(parser);
	smlAssert(data);
	smlAssert(size);
	
	//Fixme debug if
	char *debugstr = smlPrintBinary(data, size);
	smlTrace(TRACE_INTERNAL, "Xml input: %s", debugstr);
	smlLog("received-%i.xml", data, size);
	g_free(debugstr);
	
	parser->got_command = FALSE;
	
	/* Create the new parser */
	parser->reader = xmlReaderForMemory(data, size, "/", NULL, XML_PARSE_NONET | XML_PARSE_NOERROR | XML_PARSE_NOWARNING | XML_PARSE_NOCDATA);
	if (!parser->reader) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to create new reader");
		goto error;
	}
	xmlSubstituteEntitiesDefault(1);
	
	/* Check the First Node (SyncML)*/
	if (!_smlXmlParserExpectNode(parser, XML_NODE_START, FALSE, SML_ELEMENT_SYNCML, error))
		goto error_free_reader;
	
	/* Check the Second Node (SyncHdr)*/
	if (!_smlXmlParserExpectNode(parser, XML_NODE_START, FALSE, SML_ELEMENT_SYNCHDR, error))
		goto error_free_reader;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_reader:
	xmlFreeTextReader(parser->reader);
error:
	parser->reader = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlXmlParserEnd(SmlXmlParser *parser, SmlBool *final, SmlBool *end, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, parser, final, end, error);
	smlAssert(parser);
	SmlBool got_final = FALSE;
	
	/* Check for final */
	if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_FINAL)) {
		got_final = TRUE;
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_FINAL) && \
			xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			if (!_smlXmlParserStep(parser)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
				goto error;
			}
		}
	}
	
	if (final)
		*final = got_final;
	
	/* Check for the end. We can only decide if this was the last message if this
	 * was the final message */
	if (end) {
		if (got_final)
			*end = parser->got_command ? FALSE : TRUE;
		else
			*end = FALSE;
	}
	
	/* Check the closing SyncBody*/
	if (strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNCBODY) || \
		xmlTextReaderNodeType(parser->reader) != XML_NODE_CLOSE) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Wrong closing node");
		goto error;
	}

	/* Check the next node (SyncML)*/
	if (!_smlXmlParserExpectNode(parser, XML_NODE_CLOSE, FALSE, SML_ELEMENT_SYNCML, error))
		goto error;
		
	xmlFreeTextReader(parser->reader);
	parser->reader = NULL;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlXmlParserFree(SmlXmlParser *parser)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, parser);
	smlAssert(parser);
	
	if (parser->reader)
		xmlFreeTextReader(parser->reader);
	
	//xmlCleanupParser();
	
	g_free(parser);

	smlTrace(TRACE_EXIT, "%s", __func__);
}	

SmlBool smlXmlParserGetHeader(SmlXmlParser *parser, SmlHeader **header, SmlCred **cred, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, header, cred, error);
	smlAssert(parser);
	smlAssert(header);
	smlAssert(cred);
	
	if (!xmlTextReaderConstName(parser->reader) || strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNCHDR) || \
		xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Wrong starting node");
		goto error;
	}
	
	parser->version = SML_VERSION_UNKNOWN;
	
	*header = smlTryMalloc0(sizeof(SmlHeader), error);
	if (!*header)
		goto error;
	
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_header;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNCHDR) && \
			xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_header;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_VERDTD)) {
			if (!_smlSyncHeaderParseDTD(&((*header)->version), parser, error))
				goto error_free_header;
			parser->version = (*header)->version;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_VERPROTO)) {
			if (!_smlSyncHeaderParseProto(&((*header)->protocol), parser, error))
				goto error_free_header;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MSGID)) {
			if (!_smlXmlParserGetID(parser, &((*header)->messageID), SML_ELEMENT_MSGID, error))
				goto error_free_header;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SESSIONID)) {
			if (!_smlXmlParserGetString(parser, &((*header)->sessionID), SML_ELEMENT_SESSIONID, error))
				goto error_free_header;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TARGET)) {
			if (!_smlLocationParse(&((*header)->target), parser, error))
				goto error_free_header;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCE)) {
			if (!_smlLocationParse(&((*header)->source), parser, error))
				goto error_free_header;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_META)) {
			if (!_smlMessageParseSynchdrMeta(parser, &((*header)->maxmsgsize), &((*header)->maxobjsize), &((*header)->emi), error))
				goto error_free_header;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CRED)) {
			*cred = _smlCredParse(parser, error);
			if (!(*cred))
				goto error_free_header;
		} 
		
		else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node. expected SyncHdr");
			goto error_free_header;
		}
	}
	
	if (!(*header)->protocol) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No protocol set");
		goto error_free_header;
	}
	
	if (!(*header)->version) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No dtd version set");
		goto error_free_header;
	}
	
	if (!(*header)->sessionID) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No sessionID set");
		goto error_free_header;
	}
	
	if (!(*header)->target) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No target set");
		goto error_free_header;
	}
	
	if (!(*header)->source) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No source set");
		goto error_free_header;
	}
	
	if (!(*header)->messageID) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No msgid set");
		goto error_free_header;
	}
	
	/* Check the Next Node (SyncBody)*/
	if (!_smlXmlParserExpectNode(parser, XML_NODE_START, FALSE, SML_ELEMENT_SYNCBODY, error))
		goto error_free_header;
	
	/* Step once more */
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_header;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_header:
	smlHeaderFree(*header);
	*header = NULL;
	
	if (*cred)
		smlCredFree(*cred);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlXmlParserGetStatus(SmlXmlParser *parser, SmlStatus **status, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, status, error);
	smlAssert(parser);
	smlAssert(status);
	
	char *locURI = NULL;
	SmlItem *item = NULL;
	SmlChal *chal = NULL;
	
	/* Lets check if the next node is a command node */
	if (smlCommandTypeFromString((char *)xmlTextReaderConstName(parser->reader), NULL) != SML_COMMAND_TYPE_UNKNOWN && xmlTextReaderNodeType(parser->reader) == XML_NODE_START) {
		*status = NULL;
		smlTrace(TRACE_EXIT, "%s: Next is command", __func__);
		return TRUE;
	}
	
	/* Lets check if the next node is a final node */
	if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_FINAL) && (xmlTextReaderNodeType(parser->reader) == XML_NODE_START || xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE)) {
		*status = NULL;
		smlTrace(TRACE_EXIT, "%s: Next is final", __func__);
		return TRUE;
	}
	
	/* Lets check if the next node is a syncbody node */
	if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNCBODY) && xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
		*status = NULL;
		smlTrace(TRACE_EXIT, "%s: Next is closing syncbody", __func__);
		return TRUE;
	}
	
	if (strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_STATUS) || xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Wrong starting node");
		goto error;
	}
	
	*status = smlStatusNew(SML_ERROR_UNKNOWN, 0, 0, NULL, NULL, SML_COMMAND_TYPE_UNKNOWN, error);
	if (!*status)
		goto error;
		
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_status;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_STATUS) && \
			xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_status;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDID)) {
			//Errr... who cares?
			if (!_smlXmlParserExpectNode(parser, XML_NODE_TEXT, FALSE, NULL, error))
				goto error_free_status;
			if (!_smlXmlParserExpectNode(parser, XML_NODE_CLOSE, FALSE, SML_ELEMENT_CMDID, error))
				goto error_free_status;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MSGREF)) {
			if (!_smlXmlParserGetID(parser, &((*status)->msgRef), SML_ELEMENT_MSGREF, error))
				goto error_free_status;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMDREF)) {
			if (!_smlXmlParserGetID(parser, &((*status)->cmdRef), SML_ELEMENT_CMDREF, error))
				goto error_free_status;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CMD)) {
			char *cmdname = NULL;
			if (!_smlXmlParserGetString(parser, &cmdname, SML_ELEMENT_CMD, error))
				goto error_free_status;
			(*status)->type = smlCommandTypeFromString(cmdname, error);
			g_free(cmdname);
			if ((*status)->type == SML_COMMAND_TYPE_UNKNOWN)
				goto error_free_status;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DATA)) {
			if (!_smlXmlParserGetString(parser, &((*status)->data), SML_ELEMENT_DATA, error))
				goto error_free_status;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCEREF)) {
			if (!_smlXmlParserGetString(parser, &locURI, SML_ELEMENT_SOURCEREF, error))
				goto error_free_status;
			
			(*status)->sourceRef = smlLocationNew(locURI, NULL, error);
			g_free(locURI);
			locURI = NULL;
			if (!(*status)->sourceRef)
				goto error_free_status;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TARGETREF)) {
			if (!_smlXmlParserGetString(parser, &locURI, SML_ELEMENT_TARGETREF, error))
				goto error_free_status;
				
			(*status)->targetRef = smlLocationNew(locURI, NULL, error);
			g_free(locURI);
			locURI = NULL;
			if (!(*status)->targetRef)
				goto error_free_status;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_ITEM)) {
			item = _smlItemParse(parser, NULL, SML_COMMAND_TYPE_ALERT, error);
			if (!item)
				goto error_free_status;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CHAL)) {
			chal = _smlChalParse(parser, error);
			if (!chal)
				goto error_free_status;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error;
		}
	}
	
	/* Step once more */
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_status;
	}
	
	if (item && (*status)->type != SML_COMMAND_TYPE_ALERT) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Got wrong item");
		smlItemUnref(item);
		goto error_free_status;
	}
	
	if (item && (*status)->type == SML_COMMAND_TYPE_ALERT) {
		(*status)->anchor = item->anchor;
		item->anchor = NULL;
		smlItemUnref(item);
	}
	
	if (!(*status)->msgRef) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No msgref set");
		goto error_free_status;
	}
	
	if ((*status)->type == SML_COMMAND_TYPE_UNKNOWN) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No cmd set");
		goto error_free_status;
	}
	
	smlTrace(TRACE_INTERNAL, "Got status %p with: cmdRef %i, msgRef %i, type %i, data %s", *status, (*status)->cmdRef, (*status)->msgRef, (*status)->type, (*status)->data);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_status:
	smlStatusUnref(*status);
error:
	*status = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlParserResult smlXmlParserGetCommand(SmlXmlParser *parser, SmlCommand **cmd, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, cmd, error);
	smlAssert(parser);
	smlAssert(cmd);
	*cmd = NULL;
	
	/* Lets check if the next node is a final node */
	if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_FINAL) && (xmlTextReaderNodeType(parser->reader) == XML_NODE_START || xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE)) {
		*cmd = NULL;
		smlTrace(TRACE_EXIT, "%s: Next is final", __func__);
		return SML_PARSER_RESULT_OTHER;
	}
	
	/* Lets check if the next node is a syncbody node */
	if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNCBODY) && xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
		*cmd = NULL;
		smlTrace(TRACE_EXIT, "%s: Next is closing syncbody", __func__);
		return SML_PARSER_RESULT_OTHER;
	}
	
	/* Lets check if the next node status node */
	if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_STATUS) && xmlTextReaderNodeType(parser->reader) == XML_NODE_START) {
		*cmd = NULL;
		smlTrace(TRACE_EXIT, "%s: Next is status", __func__);
		return SML_PARSER_RESULT_STATUS;
	}
	
	SmlCommandType type = smlCommandTypeFromString((char *)xmlTextReaderConstName(parser->reader), error);
	if (!type)
		goto error;
	
	SmlParserResult result = SML_PARSER_RESULT_NORMAL;
	
	switch (type) {
		case SML_COMMAND_TYPE_ALERT:
			if (!_smlAlertParse(parser, cmd, error))
				goto error;
			break;
		case SML_COMMAND_TYPE_PUT:
		case SML_COMMAND_TYPE_GET:
			if (!_smlCommandAccessParse(parser, cmd, type, error))
				goto error;
			break;
		case SML_COMMAND_TYPE_SYNC:
			if (xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
				if (!_smlXmlParserStep(parser)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
					goto error_free_cmd;
				}
				
				smlTrace(TRACE_EXIT, "%s: SML_PARSER_RESULT_CLOSE", __func__);
				return SML_PARSER_RESULT_CLOSE;
			} else {
				if (!_smlCommandSyncParse(parser, cmd, error))
					goto error;
				
				result = SML_PARSER_RESULT_OPEN;
			}
			break;
		case SML_COMMAND_TYPE_MAP:
			if (!_smlCommandMapParse(parser, cmd, error))
				goto error;
			break;
		case SML_COMMAND_TYPE_ADD:
			if (!_smlChangeParse(parser, cmd, SML_COMMAND_TYPE_ADD, SML_ELEMENT_ADD, error))
				goto error;
			break;
		case SML_COMMAND_TYPE_REPLACE:
			if (!_smlChangeParse(parser, cmd, SML_COMMAND_TYPE_REPLACE, SML_ELEMENT_REPLACE, error))
				goto error;
			break;
		case SML_COMMAND_TYPE_DELETE:
			if (!_smlChangeParse(parser, cmd, SML_COMMAND_TYPE_DELETE, SML_ELEMENT_DELETE, error))
				goto error;
			break;
		case SML_COMMAND_TYPE_RESULTS:
			if (!_smlResultsParse(parser, cmd, error))
				goto error;
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unsupported command type");
			goto error;
	}
	
	if (!(*cmd)->cmdID) {
		smlErrorSet(error, SML_ERROR_GENERIC, "No cmdid set");
		goto error_free_cmd;
	}
	
	parser->got_command = TRUE;
	
	smlTrace(TRACE_EXIT, "%s: %i", __func__, result);
	return result;

error_free_cmd:
	smlCommandUnref(*cmd);
error:
	*cmd = NULL;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return SML_PARSER_RESULT_ERROR;
}

SmlXmlParser *smlXmlParserNew(SmlParserFunctions *functions, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, functions,error);
	smlAssert(functions);
	
	SmlXmlParser *parser = smlTryMalloc0(sizeof(SmlXmlParser), error);
	if (!parser)
		goto error;
	
	functions->start = (SmlParserStartFunction)smlXmlParserStart;
	functions->free = (SmlParserFreeFunction)smlXmlParserFree;
	functions->end = (SmlParserEndFunction)smlXmlParserEnd;
	functions->get_header = (SmlParserHeaderFunction)smlXmlParserGetHeader;
	functions->get_cmd = (SmlParserCommandFunction)smlXmlParserGetCommand;
	functions->get_status = (SmlParserStatusFunction)smlXmlParserGetStatus;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, parser);
	return parser;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool _smlXmlDevInfDataStoreParseDSMem(SmlXmlParser *parser, SmlDevInfDataStore *datastore, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, datastore, error);
	smlAssert(parser);
	smlAssert(datastore);
		
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DSMEM) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAXID)) {
			if (!_smlXmlParserGetID(parser, &(datastore->maxid), SML_ELEMENT_MAXID, error))
				goto error;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SHAREDMEM)) {	
			datastore->sharedMem = TRUE;
			if (!_smlXmlParserStep(parser)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
				goto error;
			}
			
			if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SHAREDMEM) && \
				xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
				if (!_smlXmlParserStep(parser)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes2");
					goto error;
				}
			}
			continue;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAXMEM)) {
			if (!_smlXmlParserGetID(parser, &(datastore->maxmem), SML_ELEMENT_MAXMEM, error))
				goto error;	
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlXmlDevInfDataStoreParseSyncCap(SmlXmlParser *parser, SmlDevInfDataStore *datastore, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, datastore, error);
	smlAssert(parser);
	smlAssert(datastore);
		
	while (1) {
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNCCAP) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNCTYPE)) {
			SmlDevInfSyncCap synccap = SML_DEVINF_SYNCTYPE_UNKNOWN;
			if (!_smlXmlParserGetID(parser, &synccap, SML_ELEMENT_SYNCTYPE, error))
				goto error;
			
			if (synccap != SML_DEVINF_SYNCTYPE_UNKNOWN)
				smlDevInfDataStoreSetSyncCap(datastore, synccap, TRUE);
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlXmlDevInfDataStoreParseRxTx(SmlXmlParser *parser, const char *element, char **cttype, char **version, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %s, %p, %p, %p)", __func__, parser, element, cttype, version, error);
	smlAssert(parser);
	smlAssert(element);
	smlAssert(cttype);
	
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error;
	}
	
	while (1) {
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), element) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node: %s %i", (char *)xmlTextReaderConstName(parser->reader), xmlTextReaderNodeType(parser->reader));
			goto error;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CTTYPE)) {
			if (!_smlXmlParserGetString(parser, cttype, SML_ELEMENT_CTTYPE, error))
				goto error;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_VERCT)) {
			/* Some phones send an empty VerCT for formats that dont have a version
			 * (like notes). */
			if (!_smlXmlParserStep(parser)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "No node at all");
				goto error;
			}
			
			if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_VERCT) && \
				xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
				*version = g_strdup("");
			} else if (xmlTextReaderNodeType(parser->reader) == XML_NODE_TEXT) {
				*version = g_strstrip(g_strdup((char *)xmlTextReaderConstValue(parser->reader)));
				
				if (!_smlXmlParserExpectNode(parser, XML_NODE_CLOSE, FALSE, SML_ELEMENT_VERCT, error))
					goto error;
			} else {
				*version = g_strdup("");
				continue;
			}
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node");
			goto error;
		}
		
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static SmlBool _smlXmlDevInfDataStoreParse(SmlXmlParser *parser, SmlDevInf *devinf, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, parser, devinf, error);
	smlAssert(devinf);
	smlAssert(parser);
	
	SmlDevInfDataStore *datastore = smlTryMalloc0(sizeof(SmlDevInfDataStore), error);
	if (!datastore)
		goto error;
	datastore->refCount = 1;
	
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_datastore;
	}
		
	while (1) {
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DATASTORE) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_datastore;
		}
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SOURCEREF)) {
			if (!_smlXmlParserGetString(parser, &(datastore->sourceref), SML_ELEMENT_SOURCEREF, error))
				goto error_free_datastore;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DISPLAYNAME)) {
			if (!_smlXmlParserGetString(parser, &(datastore->displayname), SML_ELEMENT_DISPLAYNAME, error))
				goto error_free_datastore;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAXGUIDSIZE)) {
			if (!_smlXmlParserGetID(parser, &(datastore->maxGUIDSize), SML_ELEMENT_MAXGUIDSIZE, error))
				goto error_free_datastore;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_RXPREF)) {
			if (!_smlXmlDevInfDataStoreParseRxTx(parser, SML_ELEMENT_RXPREF, &(datastore->rxPrefContentType), &(datastore->rxPrefVersion), error))
				goto error_free_datastore;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_RX)) {
			if (!_smlXmlDevInfDataStoreParseRxTx(parser, SML_ELEMENT_RX, &(datastore->rxContentType), &(datastore->rxVersion), error))
				goto error_free_datastore;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TXPREF)) {
			if (!_smlXmlDevInfDataStoreParseRxTx(parser, SML_ELEMENT_TXPREF, &(datastore->txPrefContentType), &(datastore->txPrefVersion), error))
				goto error_free_datastore;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_TX)) {
			if (!_smlXmlDevInfDataStoreParseRxTx(parser, SML_ELEMENT_TX, &(datastore->txContentType), &(datastore->txVersion), error))
				goto error_free_datastore;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SYNCCAP)) {
			if (!_smlXmlDevInfDataStoreParseSyncCap(parser, datastore, error))
				goto error_free_datastore;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DSMEM)) { 
			if (!_smlXmlDevInfDataStoreParseDSMem(parser, datastore, error))
				goto error_free_datastore;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node (%s)", (char *)xmlTextReaderConstName(parser->reader));
			goto error_free_datastore;
		}
		
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_datastore;
		}
	}
	
	smlDevInfAddDataStore(devinf, datastore);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_datastore:
	smlDevInfDataStoreUnref(datastore);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlDevInf *smlXmlDevInfParse(const char *data, unsigned int size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, data, size, error);
	smlAssert(data);
	smlAssert(size);
	
	//Fixme debug if
	char *debugstr = smlPrintBinary(data, size);
	smlTrace(TRACE_INTERNAL, "Xml devinf input: %s", debugstr);
	g_free(debugstr);
	
	SmlXmlParser *parser = smlTryMalloc0(sizeof(SmlXmlParser), error);
	if (!parser)
		goto error;
	
	/* Create the new parser */
	parser->reader = xmlReaderForMemory(data, size, "/", NULL, XML_PARSE_NONET | XML_PARSE_NOERROR | XML_PARSE_NOWARNING | XML_PARSE_NOCDATA | XML_PARSER_SUBST_ENTITIES);
	if (!parser->reader) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to create new reader");
		goto error_free_parser;
	}
	xmlSubstituteEntitiesDefault(1);
	
	/* Check the First Node (DevInf)*/
	if (!_smlXmlParserExpectNode(parser, XML_NODE_START, FALSE, SML_ELEMENT_DEVINF, error))
		goto error_free_reader;
	
	SmlDevInf *devinf = smlTryMalloc0(sizeof(SmlDevInf), error);
	if (!devinf)
		goto error_free_reader;

	smlDevInfRef(devinf);
	
	if (!_smlXmlParserStep(parser)) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
		goto error_free_devinf;
	}
		
	while (1) {
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DEVINF) && \
		xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
			break;
		} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
			goto error_free_devinf;
		}
		
		
		if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_VERDTD)) {
			char *verdtd = NULL;
			if (!_smlXmlParserGetString(parser, &verdtd, SML_ELEMENT_VERDTD, error))
				goto error_free_devinf;
			
			if (verdtd && !strcmp(verdtd, "1.1"))
				devinf->version = SML_DEVINF_VERSION_11;
			else if (verdtd && !strcmp(verdtd, "1.0"))
				devinf->version = SML_DEVINF_VERSION_10;
			else {
				g_free(verdtd);
				smlErrorSet(error, SML_ERROR_GENERIC, "Unknown devinf version");
				goto error_free_devinf;
			}
			g_free(verdtd);
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MAN)) {
			if (!_smlXmlParserGetString(parser, &(devinf->manufacturer), SML_ELEMENT_MAN, error))
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_MOD)) {
			if (!_smlXmlParserGetString(parser, &(devinf->model), SML_ELEMENT_MOD, error))
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_OEM)) {
			if (!_smlXmlParserGetString(parser, &(devinf->oem), SML_ELEMENT_OEM, error))
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_FWV)) {
			if (!_smlXmlParserGetString(parser, &(devinf->firmwareVersion), SML_ELEMENT_FWV, error))
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SWV)) {
			if (!_smlXmlParserGetString(parser, &(devinf->softwareVersion), SML_ELEMENT_SWV, error))
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_HWV)) {
			if (!_smlXmlParserGetString(parser, &(devinf->hardwareVersion), SML_ELEMENT_HWV, error))
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DEVID)) {
			if (!_smlXmlParserGetString(parser, &(devinf->devid), SML_ELEMENT_DEVID, error))
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DEVTYPE)) {
			char *devtype = NULL;
			if (!_smlXmlParserGetString(parser, &devtype, SML_ELEMENT_DEVTYPE, error))
				goto error_free_devinf;
			
			devinf->devtyp = smlDevInfDevTypeFromString(devtype, error);
			g_free(devtype);
			
			if (devinf->devtyp == SML_DEVINF_DEVTYPE_UNKNOWN)
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_UTC)) {
			if (devinf->version == SML_DEVINF_VERSION_10) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Devinf 1.0 does not allow UTC");
				goto error_free_devinf;
			}
			
			devinf->supportsUTC = TRUE;
			if (!_smlXmlParserStep(parser)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
				goto error_free_devinf;
			}
			
			if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_UTC) && \
				xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
				if (!_smlXmlParserStep(parser)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes2");
					goto error_free_devinf;
				}
			}
			continue;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SUPPORTLARGEOBJS)) {
			if (devinf->version == SML_DEVINF_VERSION_10) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Devinf 1.0 does not allow large objects");
				goto error_free_devinf;
			}
			
			devinf->supportsLargeObjs = TRUE;
			if (!_smlXmlParserStep(parser)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
				goto error_free_devinf;
			}
			
			if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SUPPORTLARGEOBJS) && \
				xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
				if (!_smlXmlParserStep(parser)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes2");
					goto error_free_devinf;
				}
			}
			continue;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SUPPORTNUMBEROFCHANGES)) {
			if (devinf->version == SML_DEVINF_VERSION_10) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Devinf 1.0 does not allow number of changes");
				goto error_free_devinf;
			}
			
			devinf->supportsNumberOfChanges = TRUE;
			if (!_smlXmlParserStep(parser)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
				goto error_free_devinf;
			}
			
			if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_SUPPORTNUMBEROFCHANGES) && \
				xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
				if (!_smlXmlParserStep(parser)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes2");
					goto error_free_devinf;
				}
			}
			continue;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_DATASTORE)) {
			if (!_smlXmlDevInfDataStoreParse(parser, devinf, error))
				goto error_free_devinf;
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CTCAP)) {
			if (!_smlXmlParserStep(parser)) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
				goto error_free_devinf;
			}
			
			while (1) {
				if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_CTCAP) &&
				    xmlTextReaderNodeType(parser->reader) == XML_NODE_CLOSE) {
					break;
				} else if (xmlTextReaderNodeType(parser->reader) != XML_NODE_START) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Not a start node");
					goto error_free_devinf;
				}
				
				const char *elem_name;
				char *value = NULL;
				SmlDevInfCTCapType type;
				
				elem_name = (const char*)xmlTextReaderConstName(parser->reader);
				type = smlDevInfCTCapTypeFromString(elem_name, NULL);
				if (type != SML_DEVINF_CTCAP_UNKNOWN) {
					_smlXmlParserGetString(parser, &value, elem_name, error);
					smlDevInfAddCTCap(devinf, type, value);
					g_free(value);
				}

				if (!_smlXmlParserStep(parser)) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
					goto error_free_devinf;
				}
			}
		} else if (!strcmp((char *)xmlTextReaderConstName(parser->reader), SML_ELEMENT_EXT)) {
			/* Ignored for now */
			smlTrace(TRACE_INTERNAL, "Skipping ext node");
			if (xmlTextReaderNext(parser->reader) != 1) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Unable to skip ext node");
				goto error_free_parser;
			}
			continue;
		} else {
			smlErrorSet(error, SML_ERROR_GENERIC, "wrong initial node: %s", (char *)xmlTextReaderConstName(parser->reader));
			goto error_free_devinf;
		}
		
		if (!_smlXmlParserStep(parser)) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Missing nodes");
			goto error_free_devinf;
		}
	}
		
	xmlFreeTextReader(parser->reader);
	parser->reader = NULL;
	g_free(parser);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return devinf;

error_free_devinf:
	smlDevInfUnref(devinf);
error_free_reader:
	xmlFreeTextReader(parser->reader);
error_free_parser:
	parser->reader = NULL;
	g_free(parser);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/*@}*/
