/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include <libsyncml/syncml.h>
#include "sml_devinf_obj.h"

#include <libsyncml/syncml_internals.h>
#include "sml_devinf_obj_internals.h"
#include <libsyncml/sml_session_internals.h>
#include <libsyncml/sml_elements_internals.h>
#include <libsyncml/sml_command_internals.h>

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

static void _get_devinf_reply(SmlSession *session, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, userdata);
	SmlError *error = NULL;
	SmlDevInfAgent *agent = userdata;
	
	if (smlStatusIsResult(status)) {
		SmlCommand *result = smlStatusGetResult(status);
		
		agent->recvDevInf = smlDevInfFromResult(result, &error);
		if (!agent->recvDevInf)
			goto error;
		
		SmlStatus *reply = smlCommandNewReply(result, SML_NO_ERROR, &error);
		if (!reply)
			goto error;
	
		if (!smlSessionSendReply(session, reply, &error)) {
			smlStatusUnref(reply);
			goto error;
		}
		
		smlStatusUnref(reply);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	smlSessionDispatchEvent(session, SML_SESSION_EVENT_ERROR, NULL, NULL, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

static void _devinf_reply(SmlSession *session, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, userdata);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/


static SmlBool _send_devinf(SmlDevInfAgent *agent, SmlSession *session, SmlCommand *get, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, agent, session, get, error);
	SmlCommand *result = NULL;
	SmlCommand *cmd = NULL;
	
	if (!agent->devinfSent) {
		if (get) {
			if (smlSessionGetVersion(session) == SML_VERSION_10)
				result = smlDevInfNewResult(get, agent->devinf, SML_DEVINF_VERSION_10, error);
			else
				result = smlDevInfNewResult(get, agent->devinf, SML_DEVINF_VERSION_11, error);
			
			if (!result)
				goto error;
			
			if (!smlSessionSendCommand(session, result, NULL, _devinf_reply, agent, error)) {
				smlCommandUnref(result);
				goto error;
			}
			
			smlCommandUnref(result);
			
			SmlStatus *reply = smlCommandNewReply(get, SML_NO_ERROR, error);
			if (!reply)
				goto error;
			
			if (!smlSessionSendReply(session, reply, error)) {
				smlStatusUnref(reply);
				goto error;
			}
			
			smlStatusUnref(reply);
		} else {
			if (smlSessionGetVersion(session) == SML_VERSION_10)
				cmd = smlDevInfNewPut(agent->devinf, SML_DEVINF_VERSION_10, error);
			else
				cmd = smlDevInfNewPut(agent->devinf, SML_DEVINF_VERSION_11, error);
			
			if (!cmd)
				goto error;
			
			if (!smlSessionSendCommand(session, cmd, NULL, _devinf_reply, agent, error)) {
				smlCommandUnref(cmd);
				goto error;
			}
			
			smlCommandUnref(cmd);
		}
		agent->devinfSent = TRUE;
	} else {
		smlTrace(TRACE_INTERNAL, "Already sent the devinf!");
	
		/* We return an generic error if we dont want to send the devinf to a get
		 * request. Hope that all devices like this */
		SmlStatus *reply = smlCommandNewReply(get, SML_ERROR_GENERIC, error);
		if (!reply)
			goto error;
		
		if (!smlSessionSendReply(session, reply, error)) {
			smlStatusUnref(reply);
			goto error;
		}
		
		smlStatusUnref(reply);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}
	
static void _recv_devinf(SmlSession *session, SmlCommand *cmd, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, cmd, userdata);
	SmlDevInfAgent *agent = userdata;
	SmlError *error = NULL;
	char *data = NULL;
	unsigned int size = 0;
	
	if (!smlItemGetData(cmd->private.access.item, &data, &size, &error))
		goto error;
	
	agent->recvDevInf = smlDevInfParse(data, size, &error);
	if (!agent->recvDevInf)
		goto error;
	
	SmlStatus *reply = smlCommandNewReply(cmd, SML_NO_ERROR, &error);
	if (!reply)
		goto error;

	if (!smlSessionSendReply(session, reply, &error)) {
		smlStatusUnref(reply);
		goto error;
	}
	
	smlStatusUnref(reply);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

static void _request_devinf(SmlSession *session, SmlCommand *cmd, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, cmd, userdata);
	SmlDevInfAgent *agent = userdata;
	SmlError *error = NULL;
	
	if (!_send_devinf(agent, session, cmd, &error))
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
}

SmlDevInfAgent *smlDevInfAgentNew(SmlDevInf *devinf, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, devinf, error);
	smlAssert(devinf);
	
	SmlDevInfAgent *agent = smlTryMalloc0(sizeof(SmlDevInfAgent), error);
	if (!agent)
		goto error;

	agent->devinfSent = FALSE;
	agent->devinf = devinf;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, agent);
	return agent;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}


void smlDevInfAgentFree(SmlDevInfAgent *agent)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, agent);
	smlAssert(agent);
	
	g_free(agent);
	
	smlTrace(TRACE_EXIT, "%s", __func__);	
}

/* Get the devinf that was sent in the session. Returns FALSE if no devinf was received yet. */
SmlDevInf *smlDevInfAgentGetDevInf(SmlDevInfAgent *agent)
{
	smlAssert(agent);
	return agent->recvDevInf;
}

/** Issues a put request on the session if needed */
SmlBool smlDevInfAgentSendDevInf(SmlDevInfAgent *agent, SmlSession *session, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, agent, session, error);
	smlAssert(agent);
	
	if (!_send_devinf(agent, session, NULL, error))
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** Issues a Get request for the devinf on the session if needed */
SmlBool smlDevInfAgentRequestDevInf(SmlDevInfAgent *agent, SmlSession *session, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, agent, session, error);
	smlAssert(agent);
	SmlCommand *get = NULL;
	
	if (agent->recvDevInf) {
		smlTrace(TRACE_EXIT, "%s: Already have the devinf", __func__);
		return TRUE;
	}
	
	if (smlSessionGetVersion(session) == SML_VERSION_10)
		get = smlDevInfNewGet(SML_DEVINF_VERSION_10, error);
	else
		get = smlDevInfNewGet(SML_DEVINF_VERSION_11, error);
	
	if (!get)
		goto error;
		
	if (!smlSessionSendCommand(session, get, NULL, _get_devinf_reply, agent, error)) {
		smlCommandUnref(get);
		goto error;
	}
	
	smlCommandUnref(get);
			
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlDevInfAgentRegisterSession(SmlDevInfAgent *agent, SmlManager *manager, SmlSession *session, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, agent, manager, error);
	smlAssert(agent);
	smlAssert(manager);

	SmlLocation *devinf11 = smlLocationNew("./devinf11", NULL, error);
	SmlLocation *devinf10 = smlLocationNew("./devinf10", NULL, error);

	if (!devinf11 || !devinf10)
		goto error;

	if (!smlManagerObjectRegister(manager, SML_COMMAND_TYPE_PUT, session, NULL, devinf10, NULL, _recv_devinf, NULL, agent, error))
		goto error_free_loc;
	if (!smlManagerObjectRegister(manager, SML_COMMAND_TYPE_PUT, session, NULL, devinf11, NULL, _recv_devinf, NULL, agent, error))
		goto error_free_loc;

	if (!smlManagerObjectRegister(manager, SML_COMMAND_TYPE_GET, session, devinf10, NULL, NULL, _request_devinf, NULL, agent, error))
		goto error_free_loc;
	if (!smlManagerObjectRegister(manager, SML_COMMAND_TYPE_GET, session, devinf11, NULL, NULL, _request_devinf, NULL, agent, error))
		goto error_free_loc;

	smlLocationUnref(devinf10);
	smlLocationUnref(devinf11);

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_loc:
	if (devinf10)
		smlLocationUnref(devinf10);
	if (devinf11)
		smlLocationUnref(devinf11);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlDevInfAgentRegister(SmlDevInfAgent *agent, SmlManager *manager, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, agent, manager, error);

	SmlBool retval = smlDevInfAgentRegisterSession(agent, manager, NULL, error);

	
	if (!retval)
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/*@}*/
