/* $Id: init.c,v 1.11.2.1 2004/10/27 12:40:47 cegger Exp $
******************************************************************************

   LibGIC - Init/Exit functions

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include "config.h"
#include <stdlib.h>
#include <string.h>

#include <ggi/gg.h>
#include <ggi/internal/gic.h>
#include <ggi/internal/gic_debug.h>


/* Exported variables */
uint32 _gicDebugState = 0;
int _gicDebugSync = 0;


/* Global variables */
void *_gic_global_lock = NULL;
void *_gicconfhandle = NULL;

/* Static variables */
static int _gicLibIsUp = 0;
static char gicconfstub[512] = GICCONFDIR;
static char *gicconfdir = gicconfstub + GICTAGLEN;


/* 
 * Returns the directory wher global config files are kept
 */

static const char *gicGetConfDir(void)
{
#if defined(__WIN32__) && !defined(__CYGWIN__)
	/* On Win32 we allow overriding of the compiled in path. */
	const char *envdir = getenv("GGI_CONFDIR");
	if (envdir)
		return envdir;
#endif
	return gicconfdir;
}


/*
 * Initialize the structures for the library
 */

int gicInit(void)
{
	int err = 0;
	const char *str, *confdir;
	char *conffile;


	_gicLibIsUp++;
	if (_gicLibIsUp > 1)
		return 0;	/* Initialize only at first call. */

	err = ggInit();
	if (err != GGI_OK) {
		fprintf(stderr, "LibGIC: unable to initialize LibGG\n");
		return err;
	}

	str = getenv("GIC_DEBUGSYNC");
	if (str != NULL) {
		_gicDebugSync = 1;
	}

	str = getenv("GIC_DEBUG");
	if (str != NULL) {
		_gicDebugState = atoi(str);
		GICDPRINT_CORE("%s Debugging=%d\n",
				_gicDebugSync ? "sync" : "async",
				_gicDebugState);
	}


	confdir = gicGetConfDir();
	conffile = malloc(strlen(confdir) + 1 + strlen(GICCONFFILE) + 1);
	if (!conffile) {
		fprintf(stderr,
			"LibGIC: unable to allocate memory for config filename.\n");
	} else {
#ifdef HAVE_SNPRINTF
		snprintf(conffile,
			 strlen(confdir) + strlen(GICCONFFILE) + 2,
			 "%s/%s", confdir, GICCONFFILE);
#else
		sprintf(conffile, "%s/%s", confdir, GICCONFFILE);
#endif
		err = ggLoadConfig(conffile, &_gicconfhandle);
		if (err == GGI_OK) {
			free(conffile);
			return GGI_OK;
		}

		fprintf(stderr,
			"LibGIC: fatal error - could not load %s\n",
			conffile);
		free(conffile);
	}

	_gicLibIsUp--;
	ggExit();
	return err;
}

int gicExit(void)
{
	GICDPRINT_CORE("gicExit() called\n");
	if (!_gicLibIsUp)
		return GGI_ENOTALLOC;

	if (_gicLibIsUp > 1) {
		/* Exit only last call */
		_gicLibIsUp--;
		return _gicLibIsUp;
	}


	GICDPRINT_CORE("gicExit: really destroying.\n");

	ggFreeConfig(_gicconfhandle);
	ggExit();
	_gicLibIsUp = 0;

	GICDPRINT_CORE("gicExit: done!\n");
	return 0;
}

#define GIC_DLINIT_SYM         "GICdlinit"

gic_handle *gicOpen(const char *drivers, ...)
{
	gic_handle *rc;
	char expbuf[1024], resbuf[1024], *current;

	GICDPRINT_LIBS("_gicOpen(\"%s\",...) called \n",
		       drivers ? drivers : "(null)");

	if (!drivers)
		drivers = "default";
	ggConfigExpandAlias(_gicconfhandle, drivers, expbuf,
			    sizeof(expbuf));
	GICDPRINT_LIBS("expn is: %s.\n", expbuf);
	current = expbuf;

	rc = malloc(sizeof(gic_handle));
	if (rc) {
		rc->input = NULL;
		rc->reclist = NULL;

		GICDPRINT_LIBS("_gicOpen(\"%s\",...) called \n",
			       drivers ? drivers : "(null)");

		while (*current) {
			void *handle;
			gic_recognizerdriver *(*init) (void);
			gic_recognizerdriver *driver;

			if (*current == ':') {
				current++;
				continue;
			}
			current =
			    ggParseTarget(current, resbuf, (int)sizeof(resbuf));
			if (!current)
				break;

			GICDPRINT_LIBS("match says: %s\n",
				       ggMatchConfig(_gicconfhandle,
						     resbuf, NULL));

			handle =
			    ggMLoadModule(_gicconfhandle, resbuf, NULL, 0);

			GICDPRINT_LIBS("handle(%s)=%p\n", resbuf, handle);
			if (handle == NULL) {
				continue;
			}

			init = ggGetSymbolAddress(handle, GIC_DLINIT_SYM);

			GICDPRINT_LIBS("init=%p\n", init);

			if (init == NULL) {
				ggFreeModule(handle);
				continue;
			}
			driver = init();
			GICDPRINT_LIBS("driver=%p\n", driver);
			if (driver == NULL) {
				ggFreeModule(handle);
				continue;
			}

			_gicRecognizerDriverRegister(rc, driver, handle);
		}
	}
	return rc;
}


int gicClose(gic_handle_t hand)
{
	free(hand);
	return 0;
}


int gicInputRegister(gic_handle_t hand, gii_input_t inp)
{
	hand->input = inp;
	return 0;
}
