/**
 * @file zip.c zip module
 * 
 * $Id: zip.c,v 1.14 2003/01/02 05:07:13 defrhymes Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/comprex.h>
#include <libcomprex/internal.h>
#include <zip.h>

static size_t 
__readFuncZip(void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	ZipFile *fileData;
	CxFile *file;
	CxFP *parentFp;
	size_t totalSize, remainingSize, result; 
	
	/* Get the data for the file within the archive */
	file = fp->file;
	fileData = (ZipFile *)fp->moduleData;

	parentFp = (CxFP *)cxGetFileArchive(file)->moduleData;

	/* you seek where I tell you to! */
	if(cxTell(parentFp) != fileData->curOffset)
		cxSeek(parentFp, fileData->curOffset, SEEK_SET);

	totalSize     = (size * nmemb);
	remainingSize = cxGetFileCompressedSize(file) - 
		        (fileData->curOffset - fileData->startOffset);
	
	/* Don't read past the end of the file */
	if(totalSize > remainingSize)
		totalSize = remainingSize;

	/* Read from the parent's fp, duh! */
	result = cxRead(ptr, size, nmemb, parentFp);

	return result;
}

static void
__seekFuncZip (CxFP *fp, long offset, int whence)
{
	CxFile *file;
	ZipFile *fileData;

	file = fp->file;
	fileData = (ZipFile *)fp->moduleData;

	switch(whence)
	{
		case SEEK_SET:
				fileData->curOffset = fileData->startOffset + offset;
				break;
		case SEEK_CUR:
				fileData->curOffset += offset;
				break;
		case SEEK_END:
				fileData->curOffset = fileData->startOffset
						      + cxGetFileCompressedSize(file)
						      - offset;
				break;
	}
}

static size_t
__altReadFunc (void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	return 0;
}

static size_t
__writeFuncZip (const void *ptr, size_t size, size_t nmemb, CxFP *fp)
{
	return 0;
}	

static void
__closeFuncZip (CxFP *fp)
{
	if (fp->moduleData != NULL)
	{	
		free(fp->moduleData);
		fp->moduleData = NULL;
	}
}

static CxStatus
readArchive(CxArchive *archive, CxFP *fp)
{
	ZipLocalHeader  header;
 	ZipStatus       status;
	ZipExtra        zip_extra;
	int 		first_header_check = 1;
	CxDirectory *root;
	root = cxGetArchiveRoot(archive);	

	/* fprintf(stdout, "READING ZIP ARHIVE!\n"); */

	while ((status = cxZipReadLocalHeader(&header, fp)) == ZIP_SUCCESS)
	{
		
		char *temp, *basePath = NULL, *baseName = NULL;
		const char *headerName; /* Cause I copy Chippy! */
		int     nameLength;
		ZipUnixExtra *unix_extra;
		CxDirectory *parentDir;
			

		if (*header.filename == '.' && header.filename[1] == '/')
			headerName = header.filename + 1;
		else
			headerName = header.filename;

		if (!strcmp(header.filename, "/"))
				continue;
				
		nameLength = strlen(headerName);
	 	/* fprintf(stdout, "        -- Reading header #%d\n", first_header_check);*/	
		/* Now check to see whether this is a file or a directory 
		 * We rely on what appears to be a "/" at the end of any directory
		 * name, and a lack of "/" for a file.
		 */
		if (headerName[nameLength - 1] == '/')
		{
			/* A directory! yay! */
			CxDirectory *dir;
			char *fullPath = strdup(headerName);

			fullPath[nameLength - 1] = '\0'; /* Get rid of that trailing /, aww */
			cxSplitPath(fullPath, &basePath, &baseName);

			if (baseName != NULL && baseName[0] == '.' && 
			    baseName[1] == '\0')
			{
				free(baseName);
				free(fullPath);
				if (basePath != NULL)
					free(basePath);

				continue;
			}
			
			dir = cxNewDirectory();
			cxSetDirName(dir, baseName);
			/*
			fprintf(stdout, "   DIRECTORY\n");
			fprintf(stdout, "      ---Name: %s\n", baseName);
			fprintf(stdout, "      ---Path: %s\n", fullPath);
			free(baseName);
			free(fullPath);
			*/
			
			
			/* Now add it to the correct parent dir */
			if (basePath != NULL)
			{
				parentDir = cxGetDirectory(root, basePath);
				free(basePath);
			}
			else 
				parentDir = root;

			cxDirAddSubDir(parentDir, dir);

		}
		else
		{
			/* it's a file who's spoon is too big :-/ */
			CxFile *file;
			ZipFile       *zipFile;
			
			file = cxNewFile();

			temp = cxGetBaseName(header.filename);
			cxSetFileName(file, temp);
			/*
			fprintf(stdout, "   FILE\n");
			fprintf(stdout, "     ---FileName: %s\n", temp);
			fprintf(stdout, "     ---Path:     %s\n", header.filename);
			free(temp);
			*/
			
			cxSetFileSize(file, header.uncompressedSize);
			cxSetFileDate(file, cxDosDateToUnix(header.mtime));
			cxSetFileCompressedSize(file, header.compressedSize);

			MEM_CHECK(zipFile = (ZipFile *)malloc(sizeof(ZipFile)));
			zipFile->startOffset = header.fileOffset;
			zipFile->curOffset   = header.fileOffset;
			zipFile->compression = header.compression;
			zipFile->flag 	     = header.flag;
			switch(header.compression)
			{
				case ZIP_DEFLATED:
					zipFile->callback = __inflateReadFunc;
					break;
				default:
					/* All other unsupported compression methods */
					zipFile->callback = __altReadFunc;
			}
			file->moduleData = zipFile;
			
			zip_extra.field  = header.extraField;
			zip_extra.length = header.extraLength;
			zip_extra.data   = (ZipUnixExtra *)NULL;
			/* Unix oriented -- maybe put in some other #ifdef WIN32 blah
			 * in the future to have ntfs or other extra type support
			 */
			if (!cxZipFindExtra(&zip_extra, ZIP_PKWARE_UNIX_EXTRA))
			{
				cxZipFindExtra(&zip_extra, ZIP_UNIX2_IZ_EXTRA);
				cxZipFindExtra(&zip_extra, ZIP_TSTAMP_EXTRA);
				if(&zip_extra.data == NULL)
					cxZipFindExtra(&zip_extra, ZIP_UNIX1_IZ_EXTRA);
				
			}	
					
			if ((unix_extra = (ZipUnixExtra *)zip_extra.data) != NULL)
			{
				if (unix_extra->mtime > 0)
					cxSetFileDate(file, (time_t)unix_extra->mtime);
				if (unix_extra->uid > 0)
					cxSetFileUid(file, (uid_t)unix_extra->uid);
				if (unix_extra->gid > 0)
					cxSetFileGid(file, (gid_t)unix_extra->gid);
				/* TODO: Handle these links in the future */
				if (unix_extra->link != NULL)
					free(unix_extra->link);

				free(unix_extra);
			}

			basePath = cxGetBasePath(header.filename);
			if (basePath != NULL)
			{
				parentDir = cxGetDirectory(root, basePath);
				free(basePath);
			}
			else
				parentDir = root;

			cxDirAddFile(parentDir, file);	
			
		}

		/* Clean up the header data */
		free(header.filename);
		if (header.extraField != NULL)
			free(header.extraField);

		first_header_check++;
	}  /* end of while */
	
	if (status == ZIP_ERROR && first_header_check == 1)
		return CX_INVALID_FORMAT;
	else
	{
		/* If we haven't stopped because we've reached the last file
		 * then something else is wrong
		 */
		/*if (status != ZIP_LAST_FILE)
			return CX_CORRUPT; */
	}

	cxSetArchiveType(archive, CX_ARCHIVE_MULTI);

	/* Store the file pointer in moduleData */
	archive->moduleData = fp;

	return CX_SUCCESS;
}

static CxStatus
saveArchive(CxArchive *archive, CxFP *fp)
{
	return CX_NOT_SUPPORTED;
}

static void
closeArchive(CxArchive *archive)
{
	archive->moduleData = NULL;
}


static CxFP *
openFile(CxFile *file, CxAccessMode mode)
{
	ZipFile *fileData;
	CxArchive *archive;
	CxFP *fp;

	if(!CX_IS_MODE_READ_ONLY(mode))
		return NULL;

	archive = cxGetFileArchive(file);
	fp = cxNewFp();
	fileData = file->moduleData;
	
	cxSetReadFunc(fp, fileData->callback);
	cxSetWriteFunc(fp, __writeFuncZip);
	cxSetSeekFunc(fp, __seekFuncZip);
	cxSetCloseFunc(fp, __closeFuncZip);
	
	fp->moduleData = fileData;

	cxSeek((CxFP *)archive->moduleData, fileData->startOffset, SEEK_SET);
	__cxZipInflateInit2(file);

	return fp;
}


static void
destroyFile(CxFile *file)
{
	if (file->moduleData != NULL)
	{
		free(file->moduleData);
		
		file->moduleData = NULL;
	}	
}		

static char
supportsExtension(const char *ext)
{
	if(!strcasecmp(ext, "zip"))
			return 1;
	
	return 0;
}

static CxArchiveOps ops = 
{
	readArchive,
	saveArchive,
	closeArchive,
	openFile,
	destroyFile,
	supportsExtension
};

static void
__moduleInit(CxModuleType type)
{
}

CX_INIT_ARCHIVE_MODULE(zip, __moduleInit, ops);
