/**
 * @file libcomprex/module.c Module API
 * 
 * $Id: module.c,v 1.28 2003/01/01 06:22:36 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/module.h>
#include <libcomprex/internal.h>

#ifdef DYNAMIC_MODS
# include <ltdl.h>

# define USER_MODULES_COUNT (sizeof(user_modules) / sizeof(*user_modules))
# define SYS_MODULES_COUNT  (sizeof(system_modules) / sizeof(*system_modules))

static const char *user_modules[] =
{
	".gnupdate/comprex"
};

static const char *system_modules[] =
{
	COMPREX_LIBDIR,
	"/usr/lib/comprex", "/usr/local/lib/comprex"
};


static int ltdl_refCount = 0;

# define LOADERS_UNINITIALIZED -4444

static int errors = LOADERS_UNINITIALIZED;
#endif /* DYNAMIC_MODS */


static CxModule *firstArchiveModule = NULL;
static CxModule *lastArchiveModule  = NULL;

static CxModule *firstSchemeModule = NULL;
static CxModule *lastSchemeModule  = NULL;

static int __initialized = 0;


#ifdef HAVE_ATEXIT
# define EXIT_FUNC(proc) static void (proc)(void)
#else
# ifdef HAVE_ON_EXIT
#  define EXIT_FUNC(proc) static void (proc)(int i, void *v)
# else
#  define EXIT_FUNC(proc) static void (proc)(void)
# endif
#endif

STATIC_ARCHIVE_INIT
STATIC_SCHEME_INIT


#ifdef DYNAMIC_MODS
static void __ltdlInit(void);
#endif

#if 0
EXIT_FUNC(__uninitialize)
{
	cxCleanup();
}
#endif

static void
__initialize()
{
	if (__initialized == 0)
	{
#if 0
		/*
		 * NOTE: This can cause segfaults when __uninitialize is called on
		 *       exit.
		 */

		/*
		 * This is a good place to initialize the atexit for the entire
		 * library.
		 */
#ifdef HAVE_ATEXIT
		atexit(__uninitialize);
#else
# ifdef HAVE_ON_EXIT
		on_exit(__uninitialize, NULL);
# endif /* HAVE_ON_EXIT */
#endif /* HAVE_ATEXIT */
#endif /* 0 */

#ifdef DYNAMIC_MODS
		__ltdlInit();
#endif

		static_archive_init();
		static_scheme_init();

		__initialized = 1;
	}
}

#ifdef DYNAMIC_MODS
static void
__ltdlInit(void)
{
	/* Do this only once! */

	if (errors == LOADERS_UNINITIALIZED)
	{
		errors = lt_dlinit();

		/* Initialize libltdl's memory management. */
		lt_dlmalloc = malloc;
		lt_dlfree   = free;
	}

	if (errors != 0)
	{
		const char *dlerror = lt_dlerror();

		fprintf(stderr,
				_("libcomprex: error: failed to initialize ltdl: %s\n"),
				dlerror);
		exit(1);
	}
}

static void
__ltdlExit(void)
{
	if (errors != LOADERS_UNINITIALIZED)
	{
		ltdl_refCount = 0;
		errors = LOADERS_UNINITIALIZED;
		lt_dlexit();
	}
}

static CxModule *
__cxLoadModule(const char *file, CxModuleType type)
{
	CxModule *(*initModule)(CxModuleType type);
	CxModule *module;
	lt_dlhandle handle;

	handle = lt_dlopenext(file);

	if (handle == NULL)
	{
		const char *dlerror = lt_dlerror();

		fprintf(stderr, _("libcomprex: error: failed to open %s: %s\n"),
				file, dlerror);

		return NULL;
	}

	initModule = lt_dlsym(handle, "initComprexModule");

	if (initModule == NULL)
	{
		lt_dlclose(handle);

		return NULL;
	}

	module = initModule(type);

	if (module == NULL)
	{
		lt_dlclose(handle);
		free(module);

		return NULL;
	}

	module->handle = handle;

	ltdl_refCount++;

	return module;
}

static void
__cxUnloadModule(CxModule *module)
{
	if (module->handle != NULL)
	{
		int result;
		
		ltdl_refCount--;

		result = lt_dlclose((lt_dlhandle)module->handle);

		module->handle = NULL;
	}

	if (ltdl_refCount == 0)
		__ltdlExit();
}

static char **
__trimModuleList(char **list, int *num)
{
	int n, size = 0, found = 0;
	char **ret = NULL;
	char **c   = NULL;
	char **d   = NULL;

	if (list == NULL)
		return NULL;

	if (*num == 0)
		return list;

	n = *num;

	for (c = list; c - list < n; c++)
	{
		char *ext;

		if (*c == NULL)
			continue;

		ext = strrchr(*c, '.');

		if (ext != NULL)
		{
			*ext = '\0';

			/* Don't add the same loader multiple times... */
			found = 0;

			for (d = ret; d - ret < size; d++)
			{
				if (!strcmp(*d, *c))
				{
					found = 1;
					break;
				}
			}

			if (!found)
			{
				ret = realloc(ret, (size + 1) * sizeof(char *));

				ret[size] = strdup(*c);

				size++;
			}
		}

		if (*c != NULL)
			free(*c);
	}

	if (list != NULL)
		free(list);

	*num = size;

	return ret;
}

static void
__scanModulesInDir(char ***list, int *num_ret, const char *dir)
{
	char **files;
	char buffer[4096];
	int offset = *num_ret;
	int num;
	int i;
	
	/* Get the files in the directory */
	files = cxListDir(dir, &num, "lib");

	if (num > 0)
	{
		/* The directory is not empty. */

		*num_ret += num;
		
		if (*list == NULL)
		{
			MEM_CHECK(*list = (char **)malloc(*num_ret * sizeof(char *)));
		
		}
		else
		{
			MEM_CHECK(*list = (char **)realloc(*list, *num_ret *
											   sizeof(char *)));
		}

		for (i = 0; i < num; i++)
		{
			sprintf(buffer, "%s/%s", dir, files[i]);
			(*list)[offset + i] = strdup(buffer);
		}

		cxFreeDirList(files, num);
	}
}

static char **
__scanModules(int *num_ret, CxModuleType type)
{
	const char *homeDir;
	char **list = NULL;
	char   buffer[4096];
	int    i;

	*num_ret = 0;

	/* Get the user's home directory */
	homeDir = cxGetHomeDir();

	/* Loop through the user module directories. */
	for (i = 0; i < USER_MODULES_COUNT; i++)
	{
		sprintf(buffer, "%s/%s/%s", homeDir, user_modules[i],
				(type == CX_MODULE_ARCHIVE ? "archive" : "scheme"));
		
		__scanModulesInDir(&list, num_ret, buffer);
	}

	/* Loop through the system module directories. */
	for (i = 0; i < SYS_MODULES_COUNT; i++)
	{
		sprintf(buffer, "%s/%s", system_modules[i],
				(type == CX_MODULE_ARCHIVE ? "archive" : "scheme"));

		__scanModulesInDir(&list, num_ret, buffer);
	}

	/* Get rid of duplicates and invalid entries. */
	list = __trimModuleList(list, num_ret);

	return list;
}
#endif /* DYNAMIC_MODS */

static void
__loadAllModules(CxModuleType type)
{
#ifdef DYNAMIC_MODS
	int i, num;
	char **list;
#endif

	__initialize();

#ifdef DYNAMIC_MODS
	/* Scan for the file modules. */
	list = __scanModules(&num, type);

	if (list == NULL)
		return;

	for (i = 0; i < num; i++)
	{
		cxLoadModule(list[i], type);

		free(list[i]);
	}

	free(list);
#endif
}

CxModule *
cxRegisterModule(const char *name, void *_ops, CxModuleType type)
{
	CxModule *module;

	if (name == NULL || _ops == NULL)
		return NULL;

	/* Build the module structure */
	MEM_CHECK(module = (CxModule *)malloc(sizeof(CxModule)));
	memset(module, 0, sizeof(CxModule));

	if (type == CX_MODULE_ARCHIVE)
	{
		CxArchiveOps *ops = (CxArchiveOps *)_ops;

		if (ops->supportsExtension == NULL ||
			ops->readArchive       == NULL ||
			ops->openFile          == NULL ||
			ops->destroyFile       == NULL)
		{
			free(module);

			return NULL;
		}
		
		module->ops.archive = ops;
	}
	else if (type == CX_MODULE_SCHEME)
	{
		CxSchemeOps *ops = (CxSchemeOps *)_ops;

		if (ops->get == NULL || ops->supports == NULL)
		{
			free(module);

			return NULL;
		}
	}
	else
	{
		free(module);

		return NULL;
	}

	module->name = strdup(name);
	module->type = type;

	if (type == CX_MODULE_ARCHIVE)
	{
		if (firstArchiveModule == NULL)
			firstArchiveModule = module;

		module->prev = lastArchiveModule;

		if (lastArchiveModule != NULL)
			lastArchiveModule->next = module;

		lastArchiveModule = module;
	}
	else
	{
		if (firstSchemeModule == NULL)
			firstSchemeModule = module;

		module->prev = lastSchemeModule;

		if (lastSchemeModule != NULL)
			lastSchemeModule->next = module;

		lastSchemeModule = module;
	}

	module->next = NULL;

	return module;
}

CxModule *
cxLoadModule(const char *name, CxModuleType type)
{
#ifndef DYNAMIC_MODS
	return NULL;
#else
	CxModule *module;
	
	if (name == NULL || *name == '\0')
		return NULL;

	module = __cxLoadModule(name, type);

	if (module == NULL)
		return NULL;

	module->filename = strdup(name);

	return module;
#endif /* DYNAMIC_MODS */
}

void
cxUnloadModule(CxModule *module)
{
	if (module == NULL)
		return;

#ifdef DYNAMIC_MODS
	__cxUnloadModule(module);
#endif
	
	if (module->prev != NULL)
		module->prev->next = module->next;
	else
	{
		if (module->type == CX_MODULE_ARCHIVE)
			firstArchiveModule = module->next;
		else
			firstSchemeModule = module->next;
	}

	if (module->next != NULL)
		module->next->prev = module->prev;
	else
	{
		if (module->type == CX_MODULE_ARCHIVE)
			lastArchiveModule = module->prev;
		else
			lastSchemeModule = module->prev;
	}

	if (module->filename != NULL) free(module->filename);
	if (module->name     != NULL) free(module->name);

	free(module);
}

CxModule *
cxGetModule(const char *name, CxModuleType type)
{
	CxModule *module;

	for (module = cxGetFirstModule(type);
		 module != NULL;
		 module = module->next)
	{
		if (!strcmp(module->name, name))
			return module;
	}

	/* Not found. Load it. */
	module = cxLoadModule(name, type);
	
	return module;
}

void
cxLinkModule(CxModule **ptr)
{
	if (ptr == NULL || *ptr == NULL)
		return;

	CX_LINK(*ptr);
}

void
cxUnlinkModule(CxModule **ptr)
{
	CxModule *module;
	
	if (ptr == NULL || *ptr == NULL)
		return;

	module = *ptr;

	CX_UNLINK(module);

	if (module->refCount == 0)
	{
		cxUnloadModule(module);
	}

	*ptr = NULL;
}

CxModule *
cxGetFirstModule(CxModuleType type)
{
	if (type == CX_MODULE_ARCHIVE)
	{
		if (firstArchiveModule == NULL)
			__loadAllModules(CX_MODULE_ARCHIVE);

		return firstArchiveModule;
	}
	else
	{
		if (firstSchemeModule == NULL)
			__loadAllModules(CX_MODULE_SCHEME);

		return firstSchemeModule;
	}
}

void
cxCleanupModules()
{
	CxModule *module, *nextModule;

	for (module = firstArchiveModule; module != NULL; module = nextModule)
	{
		nextModule = module->next;

		cxUnloadModule(module);
	}

	for (module = firstSchemeModule; module != NULL; module = nextModule)
	{
		nextModule = module->next;

		cxUnloadModule(module);
	}

	firstArchiveModule = NULL;
	lastArchiveModule  = NULL;
	firstSchemeModule  = NULL;
	lastSchemeModule   = NULL;
}

void
cxCleanup()
{
	cxCleanupModules();
	cxCleanupEnvInfo();

#ifdef DYNAMIC_MODS
	__ltdlExit();
#endif

	__initialized = 0;
}
