#![allow(clippy::wildcard_imports)]
use std::os::raw::c_uint;
use x11_dl::keysym::*;
use x11_dl::xlib;

pub type XKeysym = c_uint;
pub type ModMask = c_uint;
pub type Button = c_uint;

pub fn clean_mask(mut mask: ModMask) -> ModMask {
    mask &= !(xlib::Mod2Mask | xlib::LockMask);
    mask & (xlib::ShiftMask
        | xlib::ControlMask
        | xlib::Mod1Mask
        | xlib::Mod3Mask
        | xlib::Mod4Mask
        | xlib::Mod5Mask)
}

#[must_use]
pub fn into_modmask(keys: &[String]) -> ModMask {
    let mut mask = 0;
    for s in keys {
        mask |= into_mod(s);
    }
    clean_mask(mask)
}

#[must_use]
pub fn into_mod(key: &str) -> ModMask {
    match key {
        "None" => xlib::AnyModifier,
        "Shift" => xlib::ShiftMask,
        "Control" => xlib::ControlMask,
        "Mod1" | "Alt" => xlib::Mod1Mask,
        //"Mod2" => xlib::Mod2Mask,     // NOTE: we are ignoring the state of Numlock
        //"NumLock" => xlib::Mod2Mask,  // this is left here as a reminder
        "Mod3" => xlib::Mod3Mask,
        "Mod4" | "Super" => xlib::Mod4Mask,
        "Mod5" => xlib::Mod5Mask,
        _ => 0,
    }
}

// We allow this because this function is simply a mapping wrapper.
#[allow(clippy::too_many_lines)]
#[must_use]
pub fn into_keysym(key: &str) -> Option<XKeysym> {
    match key {
        "BackSpace" => Some(XK_BackSpace),
        "Tab" => Some(XK_Tab),
        "Linefeed" => Some(XK_Linefeed),
        "Clear" => Some(XK_Clear),
        "Return" => Some(XK_Return),
        "Pause" => Some(XK_Pause),
        "Scroll_Lock" => Some(XK_Scroll_Lock),
        "Sys_Req" => Some(XK_Sys_Req),
        "Escape" => Some(XK_Escape),
        "Delete" => Some(XK_Delete),
        "Multi_key" => Some(XK_Multi_key),
        "Kanji" => Some(XK_Kanji),
        "Muhenkan" => Some(XK_Muhenkan),
        "Henkan_Mode" => Some(XK_Henkan_Mode),
        "Henkan" => Some(XK_Henkan),
        "Romaji" => Some(XK_Romaji),
        "Hiragana" => Some(XK_Hiragana),
        "Katakana" => Some(XK_Katakana),
        "Hiragana_Katakana" => Some(XK_Hiragana_Katakana),
        "Zenkaku" => Some(XK_Zenkaku),
        "Hankaku" => Some(XK_Hankaku),
        "Zenkaku_Hankaku" => Some(XK_Zenkaku_Hankaku),
        "Touroku" => Some(XK_Touroku),
        "Massyo" => Some(XK_Massyo),
        "Kana_Lock" => Some(XK_Kana_Lock),
        "Kana_Shift" => Some(XK_Kana_Shift),
        "Eisu_Shift" => Some(XK_Eisu_Shift),
        "Eisu_toggle" => Some(XK_Eisu_toggle),
        "Home" => Some(XK_Home),
        "Left" => Some(XK_Left),
        "Up" => Some(XK_Up),
        "Right" => Some(XK_Right),
        "Down" => Some(XK_Down),
        "Prior" => Some(XK_Prior),
        "Page_Up" => Some(XK_Page_Up),
        "Next" => Some(XK_Next),
        "Page_Down" => Some(XK_Page_Down),
        "End" => Some(XK_End),
        "Begin" => Some(XK_Begin),
        "Win_L" => Some(XK_Win_L),
        "Win_R" => Some(XK_Win_R),
        "App" => Some(XK_App),
        "Select" => Some(XK_Select),
        "Print" => Some(XK_Print),
        "Execute" => Some(XK_Execute),
        "Insert" => Some(XK_Insert),
        "Undo" => Some(XK_Undo),
        "Redo" => Some(XK_Redo),
        "Menu" => Some(XK_Menu),
        "Find" => Some(XK_Find),
        "Cancel" => Some(XK_Cancel),
        "Help" => Some(XK_Help),
        "Break" => Some(XK_Break),
        "Mode_switch" => Some(XK_Mode_switch),
        "script_switch" => Some(XK_script_switch),
        "Num_Lock" => Some(XK_Num_Lock),
        "KP_Space" => Some(XK_KP_Space),
        "KP_Tab" => Some(XK_KP_Tab),
        "KP_Enter" => Some(XK_KP_Enter),
        "KP_F1" => Some(XK_KP_F1),
        "KP_F2" => Some(XK_KP_F2),
        "KP_F3" => Some(XK_KP_F3),
        "KP_F4" => Some(XK_KP_F4),
        "KP_Home" => Some(XK_KP_Home),
        "KP_Left" => Some(XK_KP_Left),
        "KP_Up" => Some(XK_KP_Up),
        "KP_Right" => Some(XK_KP_Right),
        "KP_Down" => Some(XK_KP_Down),
        "KP_Prior" => Some(XK_KP_Prior),
        "KP_Page_Up" => Some(XK_KP_Page_Up),
        "KP_Next" => Some(XK_KP_Next),
        "KP_Page_Down" => Some(XK_KP_Page_Down),
        "KP_End" => Some(XK_KP_End),
        "KP_Begin" => Some(XK_KP_Begin),
        "KP_Insert" => Some(XK_KP_Insert),
        "KP_Delete" => Some(XK_KP_Delete),
        "KP_Equal" => Some(XK_KP_Equal),
        "KP_Multiply" => Some(XK_KP_Multiply),
        "KP_Add" => Some(XK_KP_Add),
        "KP_Separator" => Some(XK_KP_Separator),
        "KP_Subtract" => Some(XK_KP_Subtract),
        "KP_Decimal" => Some(XK_KP_Decimal),
        "KP_Divide" => Some(XK_KP_Divide),
        "KP_0" => Some(XK_KP_0),
        "KP_1" => Some(XK_KP_1),
        "KP_2" => Some(XK_KP_2),
        "KP_3" => Some(XK_KP_3),
        "KP_4" => Some(XK_KP_4),
        "KP_5" => Some(XK_KP_5),
        "KP_6" => Some(XK_KP_6),
        "KP_7" => Some(XK_KP_7),
        "KP_8" => Some(XK_KP_8),
        "KP_9" => Some(XK_KP_9),
        "F1" => Some(XK_F1),
        "F2" => Some(XK_F2),
        "F3" => Some(XK_F3),
        "F4" => Some(XK_F4),
        "F5" => Some(XK_F5),
        "F6" => Some(XK_F6),
        "F7" => Some(XK_F7),
        "F8" => Some(XK_F8),
        "F9" => Some(XK_F9),
        "F10" => Some(XK_F10),
        "F11" => Some(XK_F11),
        "L1" => Some(XK_L1),
        "F12" => Some(XK_F12),
        "L2" => Some(XK_L2),
        "F13" => Some(XK_F13),
        "L3" => Some(XK_L3),
        "F14" => Some(XK_F14),
        "L4" => Some(XK_L4),
        "F15" => Some(XK_F15),
        "L5" => Some(XK_L5),
        "F16" => Some(XK_F16),
        "L6" => Some(XK_L6),
        "F17" => Some(XK_F17),
        "L7" => Some(XK_L7),
        "F18" => Some(XK_F18),
        "L8" => Some(XK_L8),
        "F19" => Some(XK_F19),
        "L9" => Some(XK_L9),
        "F20" => Some(XK_F20),
        "L10" => Some(XK_L10),
        "F21" => Some(XK_F21),
        "R1" => Some(XK_R1),
        "F22" => Some(XK_F22),
        "R2" => Some(XK_R2),
        "F23" => Some(XK_F23),
        "R3" => Some(XK_R3),
        "F24" => Some(XK_F24),
        "R4" => Some(XK_R4),
        "F25" => Some(XK_F25),
        "R5" => Some(XK_R5),
        "F26" => Some(XK_F26),
        "R6" => Some(XK_R6),
        "F27" => Some(XK_F27),
        "R7" => Some(XK_R7),
        "F28" => Some(XK_F28),
        "R8" => Some(XK_R8),
        "F29" => Some(XK_F29),
        "R9" => Some(XK_R9),
        "F30" => Some(XK_F30),
        "R10" => Some(XK_R10),
        "F31" => Some(XK_F31),
        "R11" => Some(XK_R11),
        "F32" => Some(XK_F32),
        "R12" => Some(XK_R12),
        "F33" => Some(XK_F33),
        "R13" => Some(XK_R13),
        "F34" => Some(XK_F34),
        "R14" => Some(XK_R14),
        "F35" => Some(XK_F35),
        "R15" => Some(XK_R15),
        "Shift_L" => Some(XK_Shift_L),
        "Shift_R" => Some(XK_Shift_R),
        "Control_L" => Some(XK_Control_L),
        "Control_R" => Some(XK_Control_R),
        "Caps_Lock" => Some(XK_Caps_Lock),
        "Shift_Lock" => Some(XK_Shift_Lock),
        "Meta_L" => Some(XK_Meta_L),
        "Meta_R" => Some(XK_Meta_R),
        "Alt_L" => Some(XK_Alt_L),
        "Alt_R" => Some(XK_Alt_R),
        "Super_L" => Some(XK_Super_L),
        "Super_R" => Some(XK_Super_R),
        "Hyper_L" => Some(XK_Hyper_L),
        "Hyper_R" => Some(XK_Hyper_R),
        "space" => Some(XK_space),
        "exclam" => Some(XK_exclam),
        "quotedbl" => Some(XK_quotedbl),
        "numbersign" => Some(XK_numbersign),
        "dollar" => Some(XK_dollar),
        "percent" => Some(XK_percent),
        "ampersand" => Some(XK_ampersand),
        "apostrophe" => Some(XK_apostrophe),
        "quoteright" => Some(XK_quoteright),
        "parenleft" => Some(XK_parenleft),
        "parenright" => Some(XK_parenright),
        "asterisk" => Some(XK_asterisk),
        "plus" => Some(XK_plus),
        "comma" => Some(XK_comma),
        "minus" => Some(XK_minus),
        "period" => Some(XK_period),
        "slash" => Some(XK_slash),
        "0" => Some(XK_0),
        "1" => Some(XK_1),
        "2" => Some(XK_2),
        "3" => Some(XK_3),
        "4" => Some(XK_4),
        "5" => Some(XK_5),
        "6" => Some(XK_6),
        "7" => Some(XK_7),
        "8" => Some(XK_8),
        "9" => Some(XK_9),
        "colon" => Some(XK_colon),
        "semicolon" => Some(XK_semicolon),
        "less" => Some(XK_less),
        "equal" => Some(XK_equal),
        "greater" => Some(XK_greater),
        "question" => Some(XK_question),
        "at" => Some(XK_at),
        "A" => Some(XK_A),
        "B" => Some(XK_B),
        "C" => Some(XK_C),
        "D" => Some(XK_D),
        "E" => Some(XK_E),
        "F" => Some(XK_F),
        "G" => Some(XK_G),
        "H" => Some(XK_H),
        "I" => Some(XK_I),
        "J" => Some(XK_J),
        "K" => Some(XK_K),
        "L" => Some(XK_L),
        "M" => Some(XK_M),
        "N" => Some(XK_N),
        "O" => Some(XK_O),
        "P" => Some(XK_P),
        "Q" => Some(XK_Q),
        "R" => Some(XK_R),
        "S" => Some(XK_S),
        "T" => Some(XK_T),
        "U" => Some(XK_U),
        "V" => Some(XK_V),
        "W" => Some(XK_W),
        "X" => Some(XK_X),
        "Y" => Some(XK_Y),
        "Z" => Some(XK_Z),
        "bracketleft" => Some(XK_bracketleft),
        "backslash" => Some(XK_backslash),
        "bracketright" => Some(XK_bracketright),
        "asciicircum" => Some(XK_asciicircum),
        "underscore" => Some(XK_underscore),
        "grave" => Some(XK_grave),
        "quoteleft" => Some(XK_quoteleft),
        "a" => Some(XK_a),
        "b" => Some(XK_b),
        "c" => Some(XK_c),
        "d" => Some(XK_d),
        "e" => Some(XK_e),
        "f" => Some(XK_f),
        "g" => Some(XK_g),
        "h" => Some(XK_h),
        "i" => Some(XK_i),
        "j" => Some(XK_j),
        "k" => Some(XK_k),
        "l" => Some(XK_l),
        "m" => Some(XK_m),
        "n" => Some(XK_n),
        "o" => Some(XK_o),
        "p" => Some(XK_p),
        "q" => Some(XK_q),
        "r" => Some(XK_r),
        "s" => Some(XK_s),
        "t" => Some(XK_t),
        "u" => Some(XK_u),
        "v" => Some(XK_v),
        "w" => Some(XK_w),
        "x" => Some(XK_x),
        "y" => Some(XK_y),
        "z" => Some(XK_z),
        "braceleft" => Some(XK_braceleft),
        "bar" => Some(XK_bar),
        "braceright" => Some(XK_braceright),
        "asciitilde" => Some(XK_asciitilde),
        "nobreakspace" => Some(XK_nobreakspace),
        "exclamdown" => Some(XK_exclamdown),
        "cent" => Some(XK_cent),
        "sterling" => Some(XK_sterling),
        "currency" => Some(XK_currency),
        "yen" => Some(XK_yen),
        "brokenbar" => Some(XK_brokenbar),
        "section" => Some(XK_section),
        "diaeresis" => Some(XK_diaeresis),
        "copyright" => Some(XK_copyright),
        "ordfeminine" => Some(XK_ordfeminine),
        "guillemotleft" => Some(XK_guillemotleft),
        "notsign" => Some(XK_notsign),
        "hyphen" => Some(XK_hyphen),
        "registered" => Some(XK_registered),
        "macron" => Some(XK_macron),
        "degree" => Some(XK_degree),
        "plusminus" => Some(XK_plusminus),
        "twosuperior" => Some(XK_twosuperior),
        "threesuperior" => Some(XK_threesuperior),
        "acute" => Some(XK_acute),
        "mu" => Some(XK_mu),
        "paragraph" => Some(XK_paragraph),
        "periodcentered" => Some(XK_periodcentered),
        "cedilla" => Some(XK_cedilla),
        "onesuperior" => Some(XK_onesuperior),
        "masculine" => Some(XK_masculine),
        "guillemotright" => Some(XK_guillemotright),
        "onequarter" => Some(XK_onequarter),
        "onehalf" => Some(XK_onehalf),
        "threequarters" => Some(XK_threequarters),
        "questiondown" => Some(XK_questiondown),
        "Agrave" => Some(XK_Agrave),
        "Aacute" => Some(XK_Aacute),
        "Acircumflex" => Some(XK_Acircumflex),
        "Atilde" => Some(XK_Atilde),
        "Adiaeresis" => Some(XK_Adiaeresis),
        "Aring" => Some(XK_Aring),
        "AE" => Some(XK_AE),
        "Ccedilla" => Some(XK_Ccedilla),
        "Egrave" => Some(XK_Egrave),
        "Eacute" => Some(XK_Eacute),
        "Ecircumflex" => Some(XK_Ecircumflex),
        "Ediaeresis" => Some(XK_Ediaeresis),
        "Igrave" => Some(XK_Igrave),
        "Iacute" => Some(XK_Iacute),
        "Icircumflex" => Some(XK_Icircumflex),
        "Idiaeresis" => Some(XK_Idiaeresis),
        "ETH" => Some(XK_ETH),
        "Eth" => Some(XK_Eth),
        "Ntilde" => Some(XK_Ntilde),
        "Ograve" => Some(XK_Ograve),
        "Oacute" => Some(XK_Oacute),
        "Ocircumflex" => Some(XK_Ocircumflex),
        "Otilde" => Some(XK_Otilde),
        "Odiaeresis" => Some(XK_Odiaeresis),
        "multiply" => Some(XK_multiply),
        "Ooblique" => Some(XK_Ooblique),
        "Ugrave" => Some(XK_Ugrave),
        "Uacute" => Some(XK_Uacute),
        "Ucircumflex" => Some(XK_Ucircumflex),
        "Udiaeresis" => Some(XK_Udiaeresis),
        "Yacute" => Some(XK_Yacute),
        "THORN" => Some(XK_THORN),
        "Thorn" => Some(XK_Thorn),
        "ssharp" => Some(XK_ssharp),
        "agrave" => Some(XK_agrave),
        "aacute" => Some(XK_aacute),
        "acircumflex" => Some(XK_acircumflex),
        "atilde" => Some(XK_atilde),
        "adiaeresis" => Some(XK_adiaeresis),
        "aring" => Some(XK_aring),
        "ae" => Some(XK_ae),
        "ccedilla" => Some(XK_ccedilla),
        "egrave" => Some(XK_egrave),
        "eacute" => Some(XK_eacute),
        "ecircumflex" => Some(XK_ecircumflex),
        "ediaeresis" => Some(XK_ediaeresis),
        "igrave" => Some(XK_igrave),
        "iacute" => Some(XK_iacute),
        "icircumflex" => Some(XK_icircumflex),
        "idiaeresis" => Some(XK_idiaeresis),
        "eth" => Some(XK_eth),
        "ntilde" => Some(XK_ntilde),
        "ograve" => Some(XK_ograve),
        "oacute" => Some(XK_oacute),
        "ocircumflex" => Some(XK_ocircumflex),
        "otilde" => Some(XK_otilde),
        "odiaeresis" => Some(XK_odiaeresis),
        "division" => Some(XK_division),
        "oslash" => Some(XK_oslash),
        "ugrave" => Some(XK_ugrave),
        "uacute" => Some(XK_uacute),
        "ucircumflex" => Some(XK_ucircumflex),
        "udiaeresis" => Some(XK_udiaeresis),
        "yacute" => Some(XK_yacute),
        "thorn" => Some(XK_thorn),
        "ydiaeresis" => Some(XK_ydiaeresis),
        "Aogonek" => Some(XK_Aogonek),
        "breve" => Some(XK_breve),
        "Lstroke" => Some(XK_Lstroke),
        "Lcaron" => Some(XK_Lcaron),
        "Sacute" => Some(XK_Sacute),
        "Scaron" => Some(XK_Scaron),
        "Scedilla" => Some(XK_Scedilla),
        "Tcaron" => Some(XK_Tcaron),
        "Zacute" => Some(XK_Zacute),
        "Zcaron" => Some(XK_Zcaron),
        "Zabovedot" => Some(XK_Zabovedot),
        "aogonek" => Some(XK_aogonek),
        "ogonek" => Some(XK_ogonek),
        "lstroke" => Some(XK_lstroke),
        "lcaron" => Some(XK_lcaron),
        "sacute" => Some(XK_sacute),
        "caron" => Some(XK_caron),
        "scaron" => Some(XK_scaron),
        "scedilla" => Some(XK_scedilla),
        "tcaron" => Some(XK_tcaron),
        "zacute" => Some(XK_zacute),
        "doubleacute" => Some(XK_doubleacute),
        "zcaron" => Some(XK_zcaron),
        "zabovedot" => Some(XK_zabovedot),
        "Racute" => Some(XK_Racute),
        "Abreve" => Some(XK_Abreve),
        "Lacute" => Some(XK_Lacute),
        "Cacute" => Some(XK_Cacute),
        "Ccaron" => Some(XK_Ccaron),
        "Eogonek" => Some(XK_Eogonek),
        "Ecaron" => Some(XK_Ecaron),
        "Dcaron" => Some(XK_Dcaron),
        "Dstroke" => Some(XK_Dstroke),
        "Nacute" => Some(XK_Nacute),
        "Ncaron" => Some(XK_Ncaron),
        "Odoubleacute" => Some(XK_Odoubleacute),
        "Rcaron" => Some(XK_Rcaron),
        "Uring" => Some(XK_Uring),
        "Udoubleacute" => Some(XK_Udoubleacute),
        "Tcedilla" => Some(XK_Tcedilla),
        "racute" => Some(XK_racute),
        "abreve" => Some(XK_abreve),
        "lacute" => Some(XK_lacute),
        "cacute" => Some(XK_cacute),
        "ccaron" => Some(XK_ccaron),
        "eogonek" => Some(XK_eogonek),
        "ecaron" => Some(XK_ecaron),
        "dcaron" => Some(XK_dcaron),
        "dstroke" => Some(XK_dstroke),
        "nacute" => Some(XK_nacute),
        "ncaron" => Some(XK_ncaron),
        "odoubleacute" => Some(XK_odoubleacute),
        "udoubleacute" => Some(XK_udoubleacute),
        "rcaron" => Some(XK_rcaron),
        "uring" => Some(XK_uring),
        "tcedilla" => Some(XK_tcedilla),
        "abovedot" => Some(XK_abovedot),
        "Hstroke" => Some(XK_Hstroke),
        "Hcircumflex" => Some(XK_Hcircumflex),
        "Iabovedot" => Some(XK_Iabovedot),
        "Gbreve" => Some(XK_Gbreve),
        "Jcircumflex" => Some(XK_Jcircumflex),
        "hstroke" => Some(XK_hstroke),
        "hcircumflex" => Some(XK_hcircumflex),
        "idotless" => Some(XK_idotless),
        "gbreve" => Some(XK_gbreve),
        "jcircumflex" => Some(XK_jcircumflex),
        "Cabovedot" => Some(XK_Cabovedot),
        "Ccircumflex" => Some(XK_Ccircumflex),
        "Gabovedot" => Some(XK_Gabovedot),
        "Gcircumflex" => Some(XK_Gcircumflex),
        "Ubreve" => Some(XK_Ubreve),
        "Scircumflex" => Some(XK_Scircumflex),
        "cabovedot" => Some(XK_cabovedot),
        "ccircumflex" => Some(XK_ccircumflex),
        "gabovedot" => Some(XK_gabovedot),
        "gcircumflex" => Some(XK_gcircumflex),
        "ubreve" => Some(XK_ubreve),
        "scircumflex" => Some(XK_scircumflex),
        "kra" => Some(XK_kra),
        "kappa" => Some(XK_kappa),
        "Rcedilla" => Some(XK_Rcedilla),
        "Itilde" => Some(XK_Itilde),
        "Lcedilla" => Some(XK_Lcedilla),
        "Emacron" => Some(XK_Emacron),
        "Gcedilla" => Some(XK_Gcedilla),
        "Tslash" => Some(XK_Tslash),
        "rcedilla" => Some(XK_rcedilla),
        "itilde" => Some(XK_itilde),
        "lcedilla" => Some(XK_lcedilla),
        "emacron" => Some(XK_emacron),
        "gcedilla" => Some(XK_gcedilla),
        "tslash" => Some(XK_tslash),
        "ENG" => Some(XK_ENG),
        "eng" => Some(XK_eng),
        "Amacron" => Some(XK_Amacron),
        "Iogonek" => Some(XK_Iogonek),
        "Eabovedot" => Some(XK_Eabovedot),
        "Imacron" => Some(XK_Imacron),
        "Ncedilla" => Some(XK_Ncedilla),
        "Omacron" => Some(XK_Omacron),
        "Kcedilla" => Some(XK_Kcedilla),
        "Uogonek" => Some(XK_Uogonek),
        "Utilde" => Some(XK_Utilde),
        "Umacron" => Some(XK_Umacron),
        "amacron" => Some(XK_amacron),
        "iogonek" => Some(XK_iogonek),
        "eabovedot" => Some(XK_eabovedot),
        "imacron" => Some(XK_imacron),
        "ncedilla" => Some(XK_ncedilla),
        "omacron" => Some(XK_omacron),
        "kcedilla" => Some(XK_kcedilla),
        "uogonek" => Some(XK_uogonek),
        "utilde" => Some(XK_utilde),
        "umacron" => Some(XK_umacron),
        "overline" => Some(XK_overline),
        "kana_fullstop" => Some(XK_kana_fullstop),
        "kana_openingbracket" => Some(XK_kana_openingbracket),
        "kana_closingbracket" => Some(XK_kana_closingbracket),
        "kana_comma" => Some(XK_kana_comma),
        "kana_conjunctive" => Some(XK_kana_conjunctive),
        "kana_middledot" => Some(XK_kana_middledot),
        "kana_WO" => Some(XK_kana_WO),
        "kana_a" => Some(XK_kana_a),
        "kana_i" => Some(XK_kana_i),
        "kana_u" => Some(XK_kana_u),
        "kana_e" => Some(XK_kana_e),
        "kana_o" => Some(XK_kana_o),
        "kana_ya" => Some(XK_kana_ya),
        "kana_yu" => Some(XK_kana_yu),
        "kana_yo" => Some(XK_kana_yo),
        "kana_tsu" => Some(XK_kana_tsu),
        "kana_tu" => Some(XK_kana_tu),
        "prolongedsound" => Some(XK_prolongedsound),
        "kana_A" => Some(XK_kana_A),
        "kana_I" => Some(XK_kana_I),
        "kana_U" => Some(XK_kana_U),
        "kana_E" => Some(XK_kana_E),
        "kana_O" => Some(XK_kana_O),
        "kana_KA" => Some(XK_kana_KA),
        "kana_KI" => Some(XK_kana_KI),
        "kana_KU" => Some(XK_kana_KU),
        "kana_KE" => Some(XK_kana_KE),
        "kana_KO" => Some(XK_kana_KO),
        "kana_SA" => Some(XK_kana_SA),
        "kana_SHI" => Some(XK_kana_SHI),
        "kana_SU" => Some(XK_kana_SU),
        "kana_SE" => Some(XK_kana_SE),
        "kana_SO" => Some(XK_kana_SO),
        "kana_TA" => Some(XK_kana_TA),
        "kana_CHI" => Some(XK_kana_CHI),
        "kana_TI" => Some(XK_kana_TI),
        "kana_TSU" => Some(XK_kana_TSU),
        "kana_TU" => Some(XK_kana_TU),
        "kana_TE" => Some(XK_kana_TE),
        "kana_TO" => Some(XK_kana_TO),
        "kana_NA" => Some(XK_kana_NA),
        "kana_NI" => Some(XK_kana_NI),
        "kana_NU" => Some(XK_kana_NU),
        "kana_NE" => Some(XK_kana_NE),
        "kana_NO" => Some(XK_kana_NO),
        "kana_HA" => Some(XK_kana_HA),
        "kana_HI" => Some(XK_kana_HI),
        "kana_FU" => Some(XK_kana_FU),
        "kana_HU" => Some(XK_kana_HU),
        "kana_HE" => Some(XK_kana_HE),
        "kana_HO" => Some(XK_kana_HO),
        "kana_MA" => Some(XK_kana_MA),
        "kana_MI" => Some(XK_kana_MI),
        "kana_MU" => Some(XK_kana_MU),
        "kana_ME" => Some(XK_kana_ME),
        "kana_MO" => Some(XK_kana_MO),
        "kana_YA" => Some(XK_kana_YA),
        "kana_YU" => Some(XK_kana_YU),
        "kana_YO" => Some(XK_kana_YO),
        "kana_RA" => Some(XK_kana_RA),
        "kana_RI" => Some(XK_kana_RI),
        "kana_RU" => Some(XK_kana_RU),
        "kana_RE" => Some(XK_kana_RE),
        "kana_RO" => Some(XK_kana_RO),
        "kana_WA" => Some(XK_kana_WA),
        "kana_N" => Some(XK_kana_N),
        "voicedsound" => Some(XK_voicedsound),
        "semivoicedsound" => Some(XK_semivoicedsound),
        "kana_switch" => Some(XK_kana_switch),
        "Arabic_comma" => Some(XK_Arabic_comma),
        "Arabic_semicolon" => Some(XK_Arabic_semicolon),
        "Arabic_question_mark" => Some(XK_Arabic_question_mark),
        "Arabic_hamza" => Some(XK_Arabic_hamza),
        "Arabic_maddaonalef" => Some(XK_Arabic_maddaonalef),
        "Arabic_hamzaonalef" => Some(XK_Arabic_hamzaonalef),
        "Arabic_hamzaonwaw" => Some(XK_Arabic_hamzaonwaw),
        "Arabic_hamzaunderalef" => Some(XK_Arabic_hamzaunderalef),
        "Arabic_hamzaonyeh" => Some(XK_Arabic_hamzaonyeh),
        "Arabic_alef" => Some(XK_Arabic_alef),
        "Arabic_beh" => Some(XK_Arabic_beh),
        "Arabic_tehmarbuta" => Some(XK_Arabic_tehmarbuta),
        "Arabic_teh" => Some(XK_Arabic_teh),
        "Arabic_theh" => Some(XK_Arabic_theh),
        "Arabic_jeem" => Some(XK_Arabic_jeem),
        "Arabic_hah" => Some(XK_Arabic_hah),
        "Arabic_khah" => Some(XK_Arabic_khah),
        "Arabic_dal" => Some(XK_Arabic_dal),
        "Arabic_thal" => Some(XK_Arabic_thal),
        "Arabic_ra" => Some(XK_Arabic_ra),
        "Arabic_zain" => Some(XK_Arabic_zain),
        "Arabic_seen" => Some(XK_Arabic_seen),
        "Arabic_sheen" => Some(XK_Arabic_sheen),
        "Arabic_sad" => Some(XK_Arabic_sad),
        "Arabic_dad" => Some(XK_Arabic_dad),
        "Arabic_tah" => Some(XK_Arabic_tah),
        "Arabic_zah" => Some(XK_Arabic_zah),
        "Arabic_ain" => Some(XK_Arabic_ain),
        "Arabic_ghain" => Some(XK_Arabic_ghain),
        "Arabic_tatweel" => Some(XK_Arabic_tatweel),
        "Arabic_feh" => Some(XK_Arabic_feh),
        "Arabic_qaf" => Some(XK_Arabic_qaf),
        "Arabic_kaf" => Some(XK_Arabic_kaf),
        "Arabic_lam" => Some(XK_Arabic_lam),
        "Arabic_meem" => Some(XK_Arabic_meem),
        "Arabic_noon" => Some(XK_Arabic_noon),
        "Arabic_ha" => Some(XK_Arabic_ha),
        "Arabic_heh" => Some(XK_Arabic_heh),
        "Arabic_waw" => Some(XK_Arabic_waw),
        "Arabic_alefmaksura" => Some(XK_Arabic_alefmaksura),
        "Arabic_yeh" => Some(XK_Arabic_yeh),
        "Arabic_fathatan" => Some(XK_Arabic_fathatan),
        "Arabic_dammatan" => Some(XK_Arabic_dammatan),
        "Arabic_kasratan" => Some(XK_Arabic_kasratan),
        "Arabic_fatha" => Some(XK_Arabic_fatha),
        "Arabic_damma" => Some(XK_Arabic_damma),
        "Arabic_kasra" => Some(XK_Arabic_kasra),
        "Arabic_shadda" => Some(XK_Arabic_shadda),
        "Arabic_sukun" => Some(XK_Arabic_sukun),
        "Arabic_switch" => Some(XK_Arabic_switch),
        "Serbian_dje" => Some(XK_Serbian_dje),
        "Macedonia_gje" => Some(XK_Macedonia_gje),
        "Cyrillic_io" => Some(XK_Cyrillic_io),
        "Ukrainian_ie" => Some(XK_Ukrainian_ie),
        "Ukranian_je" => Some(XK_Ukranian_je),
        "Macedonia_dse" => Some(XK_Macedonia_dse),
        "Ukrainian_i" => Some(XK_Ukrainian_i),
        "Ukranian_i" => Some(XK_Ukranian_i),
        "Ukrainian_yi" => Some(XK_Ukrainian_yi),
        "Ukranian_yi" => Some(XK_Ukranian_yi),
        "Cyrillic_je" => Some(XK_Cyrillic_je),
        "Serbian_je" => Some(XK_Serbian_je),
        "Cyrillic_lje" => Some(XK_Cyrillic_lje),
        "Serbian_lje" => Some(XK_Serbian_lje),
        "Cyrillic_nje" => Some(XK_Cyrillic_nje),
        "Serbian_nje" => Some(XK_Serbian_nje),
        "Serbian_tshe" => Some(XK_Serbian_tshe),
        "Macedonia_kje" => Some(XK_Macedonia_kje),
        "Byelorussian_shortu" => Some(XK_Byelorussian_shortu),
        "Cyrillic_dzhe" => Some(XK_Cyrillic_dzhe),
        "Serbian_dze" => Some(XK_Serbian_dze),
        "numerosign" => Some(XK_numerosign),
        "Serbian_DJE" => Some(XK_Serbian_DJE),
        "Macedonia_GJE" => Some(XK_Macedonia_GJE),
        "Cyrillic_IO" => Some(XK_Cyrillic_IO),
        "Ukrainian_IE" => Some(XK_Ukrainian_IE),
        "Ukranian_JE" => Some(XK_Ukranian_JE),
        "Macedonia_DSE" => Some(XK_Macedonia_DSE),
        "Ukrainian_I" => Some(XK_Ukrainian_I),
        "Ukranian_I" => Some(XK_Ukranian_I),
        "Ukrainian_YI" => Some(XK_Ukrainian_YI),
        "Ukranian_YI" => Some(XK_Ukranian_YI),
        "Cyrillic_JE" => Some(XK_Cyrillic_JE),
        "Serbian_JE" => Some(XK_Serbian_JE),
        "Cyrillic_LJE" => Some(XK_Cyrillic_LJE),
        "Serbian_LJE" => Some(XK_Serbian_LJE),
        "Cyrillic_NJE" => Some(XK_Cyrillic_NJE),
        "Serbian_NJE" => Some(XK_Serbian_NJE),
        "Serbian_TSHE" => Some(XK_Serbian_TSHE),
        "Macedonia_KJE" => Some(XK_Macedonia_KJE),
        "Byelorussian_SHORTU" => Some(XK_Byelorussian_SHORTU),
        "Cyrillic_DZHE" => Some(XK_Cyrillic_DZHE),
        "Serbian_DZE" => Some(XK_Serbian_DZE),
        "Cyrillic_yu" => Some(XK_Cyrillic_yu),
        "Cyrillic_a" => Some(XK_Cyrillic_a),
        "Cyrillic_be" => Some(XK_Cyrillic_be),
        "Cyrillic_tse" => Some(XK_Cyrillic_tse),
        "Cyrillic_de" => Some(XK_Cyrillic_de),
        "Cyrillic_ie" => Some(XK_Cyrillic_ie),
        "Cyrillic_ef" => Some(XK_Cyrillic_ef),
        "Cyrillic_ghe" => Some(XK_Cyrillic_ghe),
        "Cyrillic_ha" => Some(XK_Cyrillic_ha),
        "Cyrillic_i" => Some(XK_Cyrillic_i),
        "Cyrillic_shorti" => Some(XK_Cyrillic_shorti),
        "Cyrillic_ka" => Some(XK_Cyrillic_ka),
        "Cyrillic_el" => Some(XK_Cyrillic_el),
        "Cyrillic_em" => Some(XK_Cyrillic_em),
        "Cyrillic_en" => Some(XK_Cyrillic_en),
        "Cyrillic_o" => Some(XK_Cyrillic_o),
        "Cyrillic_pe" => Some(XK_Cyrillic_pe),
        "Cyrillic_ya" => Some(XK_Cyrillic_ya),
        "Cyrillic_er" => Some(XK_Cyrillic_er),
        "Cyrillic_es" => Some(XK_Cyrillic_es),
        "Cyrillic_te" => Some(XK_Cyrillic_te),
        "Cyrillic_u" => Some(XK_Cyrillic_u),
        "Cyrillic_zhe" => Some(XK_Cyrillic_zhe),
        "Cyrillic_ve" => Some(XK_Cyrillic_ve),
        "Cyrillic_softsign" => Some(XK_Cyrillic_softsign),
        "Cyrillic_yeru" => Some(XK_Cyrillic_yeru),
        "Cyrillic_ze" => Some(XK_Cyrillic_ze),
        "Cyrillic_sha" => Some(XK_Cyrillic_sha),
        "Cyrillic_e" => Some(XK_Cyrillic_e),
        "Cyrillic_shcha" => Some(XK_Cyrillic_shcha),
        "Cyrillic_che" => Some(XK_Cyrillic_che),
        "Cyrillic_hardsign" => Some(XK_Cyrillic_hardsign),
        "Cyrillic_YU" => Some(XK_Cyrillic_YU),
        "Cyrillic_A" => Some(XK_Cyrillic_A),
        "Cyrillic_BE" => Some(XK_Cyrillic_BE),
        "Cyrillic_TSE" => Some(XK_Cyrillic_TSE),
        "Cyrillic_DE" => Some(XK_Cyrillic_DE),
        "Cyrillic_IE" => Some(XK_Cyrillic_IE),
        "Cyrillic_EF" => Some(XK_Cyrillic_EF),
        "Cyrillic_GHE" => Some(XK_Cyrillic_GHE),
        "Cyrillic_HA" => Some(XK_Cyrillic_HA),
        "Cyrillic_I" => Some(XK_Cyrillic_I),
        "Cyrillic_SHORTI" => Some(XK_Cyrillic_SHORTI),
        "Cyrillic_KA" => Some(XK_Cyrillic_KA),
        "Cyrillic_EL" => Some(XK_Cyrillic_EL),
        "Cyrillic_EM" => Some(XK_Cyrillic_EM),
        "Cyrillic_EN" => Some(XK_Cyrillic_EN),
        "Cyrillic_O" => Some(XK_Cyrillic_O),
        "Cyrillic_PE" => Some(XK_Cyrillic_PE),
        "Cyrillic_YA" => Some(XK_Cyrillic_YA),
        "Cyrillic_ER" => Some(XK_Cyrillic_ER),
        "Cyrillic_ES" => Some(XK_Cyrillic_ES),
        "Cyrillic_TE" => Some(XK_Cyrillic_TE),
        "Cyrillic_U" => Some(XK_Cyrillic_U),
        "Cyrillic_ZHE" => Some(XK_Cyrillic_ZHE),
        "Cyrillic_VE" => Some(XK_Cyrillic_VE),
        "Cyrillic_SOFTSIGN" => Some(XK_Cyrillic_SOFTSIGN),
        "Cyrillic_YERU" => Some(XK_Cyrillic_YERU),
        "Cyrillic_ZE" => Some(XK_Cyrillic_ZE),
        "Cyrillic_SHA" => Some(XK_Cyrillic_SHA),
        "Cyrillic_E" => Some(XK_Cyrillic_E),
        "Cyrillic_SHCHA" => Some(XK_Cyrillic_SHCHA),
        "Cyrillic_CHE" => Some(XK_Cyrillic_CHE),
        "Cyrillic_HARDSIGN" => Some(XK_Cyrillic_HARDSIGN),
        "Greek_ALPHAaccent" => Some(XK_Greek_ALPHAaccent),
        "Greek_EPSILONaccent" => Some(XK_Greek_EPSILONaccent),
        "Greek_ETAaccent" => Some(XK_Greek_ETAaccent),
        "Greek_IOTAaccent" => Some(XK_Greek_IOTAaccent),
        "Greek_IOTAdiaeresis" => Some(XK_Greek_IOTAdiaeresis),
        "Greek_OMICRONaccent" => Some(XK_Greek_OMICRONaccent),
        "Greek_UPSILONaccent" => Some(XK_Greek_UPSILONaccent),
        "Greek_UPSILONdieresis" => Some(XK_Greek_UPSILONdieresis),
        "Greek_OMEGAaccent" => Some(XK_Greek_OMEGAaccent),
        "Greek_accentdieresis" => Some(XK_Greek_accentdieresis),
        "Greek_horizbar" => Some(XK_Greek_horizbar),
        "Greek_alphaaccent" => Some(XK_Greek_alphaaccent),
        "Greek_epsilonaccent" => Some(XK_Greek_epsilonaccent),
        "Greek_etaaccent" => Some(XK_Greek_etaaccent),
        "Greek_iotaaccent" => Some(XK_Greek_iotaaccent),
        "Greek_iotadieresis" => Some(XK_Greek_iotadieresis),
        "Greek_iotaaccentdieresis" => Some(XK_Greek_iotaaccentdieresis),
        "Greek_omicronaccent" => Some(XK_Greek_omicronaccent),
        "Greek_upsilonaccent" => Some(XK_Greek_upsilonaccent),
        "Greek_upsilondieresis" => Some(XK_Greek_upsilondieresis),
        "Greek_upsilonaccentdieresis" => Some(XK_Greek_upsilonaccentdieresis),
        "Greek_omegaaccent" => Some(XK_Greek_omegaaccent),
        "Greek_ALPHA" => Some(XK_Greek_ALPHA),
        "Greek_BETA" => Some(XK_Greek_BETA),
        "Greek_GAMMA" => Some(XK_Greek_GAMMA),
        "Greek_DELTA" => Some(XK_Greek_DELTA),
        "Greek_EPSILON" => Some(XK_Greek_EPSILON),
        "Greek_ZETA" => Some(XK_Greek_ZETA),
        "Greek_ETA" => Some(XK_Greek_ETA),
        "Greek_THETA" => Some(XK_Greek_THETA),
        "Greek_IOTA" => Some(XK_Greek_IOTA),
        "Greek_KAPPA" => Some(XK_Greek_KAPPA),
        "Greek_LAMDA" => Some(XK_Greek_LAMDA),
        "Greek_LAMBDA" => Some(XK_Greek_LAMBDA),
        "Greek_MU" => Some(XK_Greek_MU),
        "Greek_NU" => Some(XK_Greek_NU),
        "Greek_XI" => Some(XK_Greek_XI),
        "Greek_OMICRON" => Some(XK_Greek_OMICRON),
        "Greek_PI" => Some(XK_Greek_PI),
        "Greek_RHO" => Some(XK_Greek_RHO),
        "Greek_SIGMA" => Some(XK_Greek_SIGMA),
        "Greek_TAU" => Some(XK_Greek_TAU),
        "Greek_UPSILON" => Some(XK_Greek_UPSILON),
        "Greek_PHI" => Some(XK_Greek_PHI),
        "Greek_CHI" => Some(XK_Greek_CHI),
        "Greek_PSI" => Some(XK_Greek_PSI),
        "Greek_OMEGA" => Some(XK_Greek_OMEGA),
        "Greek_alpha" => Some(XK_Greek_alpha),
        "Greek_beta" => Some(XK_Greek_beta),
        "Greek_gamma" => Some(XK_Greek_gamma),
        "Greek_delta" => Some(XK_Greek_delta),
        "Greek_epsilon" => Some(XK_Greek_epsilon),
        "Greek_zeta" => Some(XK_Greek_zeta),
        "Greek_eta" => Some(XK_Greek_eta),
        "Greek_theta" => Some(XK_Greek_theta),
        "Greek_iota" => Some(XK_Greek_iota),
        "Greek_kappa" => Some(XK_Greek_kappa),
        "Greek_lamda" => Some(XK_Greek_lamda),
        "Greek_lambda" => Some(XK_Greek_lambda),
        "Greek_mu" => Some(XK_Greek_mu),
        "Greek_nu" => Some(XK_Greek_nu),
        "Greek_xi" => Some(XK_Greek_xi),
        "Greek_omicron" => Some(XK_Greek_omicron),
        "Greek_pi" => Some(XK_Greek_pi),
        "Greek_rho" => Some(XK_Greek_rho),
        "Greek_sigma" => Some(XK_Greek_sigma),
        "Greek_finalsmallsigma" => Some(XK_Greek_finalsmallsigma),
        "Greek_tau" => Some(XK_Greek_tau),
        "Greek_upsilon" => Some(XK_Greek_upsilon),
        "Greek_phi" => Some(XK_Greek_phi),
        "Greek_chi" => Some(XK_Greek_chi),
        "Greek_psi" => Some(XK_Greek_psi),
        "Greek_omega" => Some(XK_Greek_omega),
        "Greek_switch" => Some(XK_Greek_switch),
        "leftradical" => Some(XK_leftradical),
        "topleftradical" => Some(XK_topleftradical),
        "horizconnector" => Some(XK_horizconnector),
        "topintegral" => Some(XK_topintegral),
        "botintegral" => Some(XK_botintegral),
        "vertconnector" => Some(XK_vertconnector),
        "topleftsqbracket" => Some(XK_topleftsqbracket),
        "botleftsqbracket" => Some(XK_botleftsqbracket),
        "toprightsqbracket" => Some(XK_toprightsqbracket),
        "botrightsqbracket" => Some(XK_botrightsqbracket),
        "topleftparens" => Some(XK_topleftparens),
        "botleftparens" => Some(XK_botleftparens),
        "toprightparens" => Some(XK_toprightparens),
        "botrightparens" => Some(XK_botrightparens),
        "leftmiddlecurlybrace" => Some(XK_leftmiddlecurlybrace),
        "rightmiddlecurlybrace" => Some(XK_rightmiddlecurlybrace),
        "topleftsummation" => Some(XK_topleftsummation),
        "botleftsummation" => Some(XK_botleftsummation),
        "topvertsummationconnector" => Some(XK_topvertsummationconnector),
        "botvertsummationconnector" => Some(XK_botvertsummationconnector),
        "toprightsummation" => Some(XK_toprightsummation),
        "botrightsummation" => Some(XK_botrightsummation),
        "rightmiddlesummation" => Some(XK_rightmiddlesummation),
        "lessthanequal" => Some(XK_lessthanequal),
        "notequal" => Some(XK_notequal),
        "greaterthanequal" => Some(XK_greaterthanequal),
        "integral" => Some(XK_integral),
        "therefore" => Some(XK_therefore),
        "variation" => Some(XK_variation),
        "infinity" => Some(XK_infinity),
        "nabla" => Some(XK_nabla),
        "approximate" => Some(XK_approximate),
        "similarequal" => Some(XK_similarequal),
        "ifonlyif" => Some(XK_ifonlyif),
        "implies" => Some(XK_implies),
        "identical" => Some(XK_identical),
        "radical" => Some(XK_radical),
        "includedin" => Some(XK_includedin),
        "includes" => Some(XK_includes),
        "intersection" => Some(XK_intersection),
        "union" => Some(XK_union),
        "logicaland" => Some(XK_logicaland),
        "logicalor" => Some(XK_logicalor),
        "partialderivative" => Some(XK_partialderivative),
        "function" => Some(XK_function),
        "leftarrow" => Some(XK_leftarrow),
        "uparrow" => Some(XK_uparrow),
        "rightarrow" => Some(XK_rightarrow),
        "downarrow" => Some(XK_downarrow),
        "blank" => Some(XK_blank),
        "soliddiamond" => Some(XK_soliddiamond),
        "checkerboard" => Some(XK_checkerboard),
        "ht" => Some(XK_ht),
        "ff" => Some(XK_ff),
        "cr" => Some(XK_cr),
        "lf" => Some(XK_lf),
        "nl" => Some(XK_nl),
        "vt" => Some(XK_vt),
        "lowrightcorner" => Some(XK_lowrightcorner),
        "uprightcorner" => Some(XK_uprightcorner),
        "upleftcorner" => Some(XK_upleftcorner),
        "lowleftcorner" => Some(XK_lowleftcorner),
        "crossinglines" => Some(XK_crossinglines),
        "horizlinescan1" => Some(XK_horizlinescan1),
        "horizlinescan3" => Some(XK_horizlinescan3),
        "horizlinescan5" => Some(XK_horizlinescan5),
        "horizlinescan7" => Some(XK_horizlinescan7),
        "horizlinescan9" => Some(XK_horizlinescan9),
        "leftt" => Some(XK_leftt),
        "rightt" => Some(XK_rightt),
        "bott" => Some(XK_bott),
        "topt" => Some(XK_topt),
        "vertbar" => Some(XK_vertbar),
        "emspace" => Some(XK_emspace),
        "enspace" => Some(XK_enspace),
        "em3space" => Some(XK_em3space),
        "em4space" => Some(XK_em4space),
        "digitspace" => Some(XK_digitspace),
        "punctspace" => Some(XK_punctspace),
        "thinspace" => Some(XK_thinspace),
        "hairspace" => Some(XK_hairspace),
        "emdash" => Some(XK_emdash),
        "endash" => Some(XK_endash),
        "signifblank" => Some(XK_signifblank),
        "ellipsis" => Some(XK_ellipsis),
        "doubbaselinedot" => Some(XK_doubbaselinedot),
        "onethird" => Some(XK_onethird),
        "twothirds" => Some(XK_twothirds),
        "onefifth" => Some(XK_onefifth),
        "twofifths" => Some(XK_twofifths),
        "threefifths" => Some(XK_threefifths),
        "fourfifths" => Some(XK_fourfifths),
        "onesixth" => Some(XK_onesixth),
        "fivesixths" => Some(XK_fivesixths),
        "careof" => Some(XK_careof),
        "figdash" => Some(XK_figdash),
        "leftanglebracket" => Some(XK_leftanglebracket),
        "decimalpoint" => Some(XK_decimalpoint),
        "rightanglebracket" => Some(XK_rightanglebracket),
        "marker" => Some(XK_marker),
        "oneeighth" => Some(XK_oneeighth),
        "threeeighths" => Some(XK_threeeighths),
        "fiveeighths" => Some(XK_fiveeighths),
        "seveneighths" => Some(XK_seveneighths),
        "trademark" => Some(XK_trademark),
        "signaturemark" => Some(XK_signaturemark),
        "trademarkincircle" => Some(XK_trademarkincircle),
        "leftopentriangle" => Some(XK_leftopentriangle),
        "rightopentriangle" => Some(XK_rightopentriangle),
        "emopencircle" => Some(XK_emopencircle),
        "emopenrectangle" => Some(XK_emopenrectangle),
        "leftsinglequotemark" => Some(XK_leftsinglequotemark),
        "rightsinglequotemark" => Some(XK_rightsinglequotemark),
        "leftdoublequotemark" => Some(XK_leftdoublequotemark),
        "rightdoublequotemark" => Some(XK_rightdoublequotemark),
        "prescription" => Some(XK_prescription),
        "minutes" => Some(XK_minutes),
        "seconds" => Some(XK_seconds),
        "latincross" => Some(XK_latincross),
        "hexagram" => Some(XK_hexagram),
        "filledrectbullet" => Some(XK_filledrectbullet),
        "filledlefttribullet" => Some(XK_filledlefttribullet),
        "filledrighttribullet" => Some(XK_filledrighttribullet),
        "emfilledcircle" => Some(XK_emfilledcircle),
        "emfilledrect" => Some(XK_emfilledrect),
        "enopencircbullet" => Some(XK_enopencircbullet),
        "enopensquarebullet" => Some(XK_enopensquarebullet),
        "openrectbullet" => Some(XK_openrectbullet),
        "opentribulletup" => Some(XK_opentribulletup),
        "opentribulletdown" => Some(XK_opentribulletdown),
        "openstar" => Some(XK_openstar),
        "enfilledcircbullet" => Some(XK_enfilledcircbullet),
        "enfilledsqbullet" => Some(XK_enfilledsqbullet),
        "filledtribulletup" => Some(XK_filledtribulletup),
        "filledtribulletdown" => Some(XK_filledtribulletdown),
        "leftpointer" => Some(XK_leftpointer),
        "rightpointer" => Some(XK_rightpointer),
        "club" => Some(XK_club),
        "diamond" => Some(XK_diamond),
        "heart" => Some(XK_heart),
        "maltesecross" => Some(XK_maltesecross),
        "dagger" => Some(XK_dagger),
        "doubledagger" => Some(XK_doubledagger),
        "checkmark" => Some(XK_checkmark),
        "ballotcross" => Some(XK_ballotcross),
        "musicalsharp" => Some(XK_musicalsharp),
        "musicalflat" => Some(XK_musicalflat),
        "malesymbol" => Some(XK_malesymbol),
        "femalesymbol" => Some(XK_femalesymbol),
        "telephone" => Some(XK_telephone),
        "telephonerecorder" => Some(XK_telephonerecorder),
        "phonographcopyright" => Some(XK_phonographcopyright),
        "caret" => Some(XK_caret),
        "singlelowquotemark" => Some(XK_singlelowquotemark),
        "doublelowquotemark" => Some(XK_doublelowquotemark),
        "cursor" => Some(XK_cursor),
        "leftcaret" => Some(XK_leftcaret),
        "rightcaret" => Some(XK_rightcaret),
        "downcaret" => Some(XK_downcaret),
        "upcaret" => Some(XK_upcaret),
        "overbar" => Some(XK_overbar),
        "downtack" => Some(XK_downtack),
        "upshoe" => Some(XK_upshoe),
        "downstile" => Some(XK_downstile),
        "underbar" => Some(XK_underbar),
        "jot" => Some(XK_jot),
        "quad" => Some(XK_quad),
        "uptack" => Some(XK_uptack),
        "circle" => Some(XK_circle),
        "upstile" => Some(XK_upstile),
        "downshoe" => Some(XK_downshoe),
        "rightshoe" => Some(XK_rightshoe),
        "leftshoe" => Some(XK_leftshoe),
        "lefttack" => Some(XK_lefttack),
        "righttack" => Some(XK_righttack),
        "hebrew_doublelowline" => Some(XK_hebrew_doublelowline),
        "hebrew_aleph" => Some(XK_hebrew_aleph),
        "hebrew_bet" => Some(XK_hebrew_bet),
        "hebrew_beth" => Some(XK_hebrew_beth),
        "hebrew_gimel" => Some(XK_hebrew_gimel),
        "hebrew_gimmel" => Some(XK_hebrew_gimmel),
        "hebrew_dalet" => Some(XK_hebrew_dalet),
        "hebrew_daleth" => Some(XK_hebrew_daleth),
        "hebrew_he" => Some(XK_hebrew_he),
        "hebrew_waw" => Some(XK_hebrew_waw),
        "hebrew_zain" => Some(XK_hebrew_zain),
        "hebrew_zayin" => Some(XK_hebrew_zayin),
        "hebrew_chet" => Some(XK_hebrew_chet),
        "hebrew_het" => Some(XK_hebrew_het),
        "hebrew_tet" => Some(XK_hebrew_tet),
        "hebrew_teth" => Some(XK_hebrew_teth),
        "hebrew_yod" => Some(XK_hebrew_yod),
        "hebrew_finalkaph" => Some(XK_hebrew_finalkaph),
        "hebrew_kaph" => Some(XK_hebrew_kaph),
        "hebrew_lamed" => Some(XK_hebrew_lamed),
        "hebrew_finalmem" => Some(XK_hebrew_finalmem),
        "hebrew_mem" => Some(XK_hebrew_mem),
        "hebrew_finalnun" => Some(XK_hebrew_finalnun),
        "hebrew_nun" => Some(XK_hebrew_nun),
        "hebrew_samech" => Some(XK_hebrew_samech),
        "hebrew_samekh" => Some(XK_hebrew_samekh),
        "hebrew_ayin" => Some(XK_hebrew_ayin),
        "hebrew_finalpe" => Some(XK_hebrew_finalpe),
        "hebrew_pe" => Some(XK_hebrew_pe),
        "hebrew_finalzade" => Some(XK_hebrew_finalzade),
        "hebrew_finalzadi" => Some(XK_hebrew_finalzadi),
        "hebrew_zade" => Some(XK_hebrew_zade),
        "hebrew_zadi" => Some(XK_hebrew_zadi),
        "hebrew_qoph" => Some(XK_hebrew_qoph),
        "hebrew_kuf" => Some(XK_hebrew_kuf),
        "hebrew_resh" => Some(XK_hebrew_resh),
        "hebrew_shin" => Some(XK_hebrew_shin),
        "hebrew_taw" => Some(XK_hebrew_taw),
        "hebrew_taf" => Some(XK_hebrew_taf),
        "Hebrew_switch" => Some(XK_Hebrew_switch),
        "XF86XK_ModeLock" | "XF86ModeLock" => Some(XF86XK_ModeLock),
        "XF86XK_MonBrightnessUp" | "XF86MonBrightnessUp" => Some(XF86XK_MonBrightnessUp),
        "XF86XK_MonBrightnessDown" | "XF86MonBrightnessDown" => Some(XF86XK_MonBrightnessDown),
        "XF86XK_KbdLightOnOff" | "XF86KbdLightOnOff" => Some(XF86XK_KbdLightOnOff),
        "XF86XK_KbdBrightnessUp" | "XF86KbdBrightnessUp" => Some(XF86XK_KbdBrightnessUp),
        "XF86XK_KbdBrightnessDown" | "XF86KbdBrightnessDown" => Some(XF86XK_KbdBrightnessDown),
        "XF86XK_Standby" | "XF86Standby" => Some(XF86XK_Standby),
        "XF86XK_AudioLowerVolume" | "XF86AudioLowerVolume" => Some(XF86XK_AudioLowerVolume),
        "XF86XK_AudioMute" | "XF86AudioMute" => Some(XF86XK_AudioMute),
        "XF86XK_AudioRaiseVolume" | "XF86AudioRaiseVolume" => Some(XF86XK_AudioRaiseVolume),
        "XF86XK_AudioPlay" | "XF86AudioPlay" => Some(XF86XK_AudioPlay),
        "XF86XK_AudioStop" | "XF86AudioStop" => Some(XF86XK_AudioStop),
        "XF86XK_AudioPrev" | "XF86AudioPrev" => Some(XF86XK_AudioPrev),
        "XF86XK_AudioNext" | "XF86AudioNext" => Some(XF86XK_AudioNext),
        "XF86XK_HomePage" | "XF86HomePage" => Some(XF86XK_HomePage),
        "XF86XK_Mail" | "XF86Mail" => Some(XF86XK_Mail),
        "XF86XK_Start" | "XF86Start" => Some(XF86XK_Start),
        "XF86XK_Search" | "XF86Search" => Some(XF86XK_Search),
        "XF86XK_AudioRecord" | "XF86AudioRecord" => Some(XF86XK_AudioRecord),
        "XF86XK_Calculator" | "XF86Calculator" => Some(XF86XK_Calculator),
        "XF86XK_Memo" | "XF86Memo" => Some(XF86XK_Memo),
        "XF86XK_ToDoList" | "XF86ToDoList" => Some(XF86XK_ToDoList),
        "XF86XK_Calendar" | "XF86Calendar" => Some(XF86XK_Calendar),
        "XF86XK_PowerDown" | "XF86PowerDown" => Some(XF86XK_PowerDown),
        "XF86XK_ContrastAdjust" | "XF86ContrastAdjust" => Some(XF86XK_ContrastAdjust),
        "XF86XK_RockerUp" | "XF86RockerUp" => Some(XF86XK_RockerUp),
        "XF86XK_RockerDown" | "XF86RockerDown" => Some(XF86XK_RockerDown),
        "XF86XK_RockerEnter" | "XF86RockerEnter" => Some(XF86XK_RockerEnter),
        "XF86XK_Back" | "XF86Back" => Some(XF86XK_Back),
        "XF86XK_Forward" | "XF86Forward" => Some(XF86XK_Forward),
        "XF86XK_Stop" | "XF86Stop" => Some(XF86XK_Stop),
        "XF86XK_Refresh" | "XF86Refresh" => Some(XF86XK_Refresh),
        "XF86XK_PowerOff" | "XF86PowerOff" => Some(XF86XK_PowerOff),
        "XF86XK_WakeUp" | "XF86WakeUp" => Some(XF86XK_WakeUp),
        "XF86XK_Eject" | "XF86Eject" => Some(XF86XK_Eject),
        "XF86XK_ScreenSaver" | "XF86ScreenSaver" => Some(XF86XK_ScreenSaver),
        "XF86XK_WWW" | "XF86WWW" => Some(XF86XK_WWW),
        "XF86XK_Sleep" | "XF86Sleep" => Some(XF86XK_Sleep),
        "XF86XK_Favorites" | "XF86Favorites" => Some(XF86XK_Favorites),
        "XF86XK_AudioPause" | "XF86AudioPause" => Some(XF86XK_AudioPause),
        "XF86XK_AudioMedia" | "XF86AudioMedia" => Some(XF86XK_AudioMedia),
        "XF86XK_MyComputer" | "XF86MyComputer" => Some(XF86XK_MyComputer),
        "XF86XK_VendorHome" | "XF86VendorHome" => Some(XF86XK_VendorHome),
        "XF86XK_LightBulb" | "XF86LightBulb" => Some(XF86XK_LightBulb),
        "XF86XK_Shop" | "XF86Shop" => Some(XF86XK_Shop),
        "XF86XK_History" | "XF86History" => Some(XF86XK_History),
        "XF86XK_OpenURL" | "XF86OpenURL" => Some(XF86XK_OpenURL),
        "XF86XK_AddFavorite" | "XF86AddFavorite" => Some(XF86XK_AddFavorite),
        "XF86XK_HotLinks" | "XF86HotLinks" => Some(XF86XK_HotLinks),
        "XF86XK_BrightnessAdjust" | "XF86BrightnessAdjust" => Some(XF86XK_BrightnessAdjust),
        "XF86XK_Finance" | "XF86Finance" => Some(XF86XK_Finance),
        "XF86XK_Community" | "XF86Community" => Some(XF86XK_Community),
        "XF86XK_AudioRewind" | "XF86AudioRewind" => Some(XF86XK_AudioRewind),
        "XF86XK_BackForward" | "XF86BackForward" => Some(XF86XK_BackForward),
        "XF86XK_Launch0" | "XF86Launch0" => Some(XF86XK_Launch0),
        "XF86XK_Launch1" | "XF86Launch1" => Some(XF86XK_Launch1),
        "XF86XK_Launch2" | "XF86Launch2" => Some(XF86XK_Launch2),
        "XF86XK_Launch3" | "XF86Launch3" => Some(XF86XK_Launch3),
        "XF86XK_Launch4" | "XF86Launch4" => Some(XF86XK_Launch4),
        "XF86XK_Launch5" | "XF86Launch5" => Some(XF86XK_Launch5),
        "XF86XK_Launch6" | "XF86Launch6" => Some(XF86XK_Launch6),
        "XF86XK_Launch7" | "XF86Launch7" => Some(XF86XK_Launch7),
        "XF86XK_Launch8" | "XF86Launch8" => Some(XF86XK_Launch8),
        "XF86XK_Launch9" | "XF86Launch9" => Some(XF86XK_Launch9),
        "XF86XK_LaunchA" | "XF86LaunchA" => Some(XF86XK_LaunchA),
        "XF86XK_LaunchB" | "XF86LaunchB" => Some(XF86XK_LaunchB),
        "XF86XK_LaunchC" | "XF86LaunchC" => Some(XF86XK_LaunchC),
        "XF86XK_LaunchD" | "XF86LaunchD" => Some(XF86XK_LaunchD),
        "XF86XK_LaunchE" | "XF86LaunchE" => Some(XF86XK_LaunchE),
        "XF86XK_LaunchF" | "XF86LaunchF" => Some(XF86XK_LaunchF),
        "XF86XK_ApplicationLeft" | "XF86ApplicationLeft" => Some(XF86XK_ApplicationLeft),
        "XF86XK_ApplicationRight" | "XF86ApplicationRight" => Some(XF86XK_ApplicationRight),
        "XF86XK_Book" | "XF86Book" => Some(XF86XK_Book),
        "XF86XK_CD" | "XF86CD" => Some(XF86XK_CD),
        "XF86XK_Calculater" | "XF86Calculater" => Some(XF86XK_Calculater),
        "XF86XK_Clear" | "XF86Clear" => Some(XF86XK_Clear),
        "XF86XK_Close" | "XF86Close" => Some(XF86XK_Close),
        "XF86XK_Copy" | "XF86Copy" => Some(XF86XK_Copy),
        "XF86XK_Cut" | "XF86Cut" => Some(XF86XK_Cut),
        "XF86XK_Display" | "XF86Display" => Some(XF86XK_Display),
        "XF86XK_DOS" | "XF86DOS" => Some(XF86XK_DOS),
        "XF86XK_Documents" | "XF86Documents" => Some(XF86XK_Documents),
        "XF86XK_Excel" | "XF86Excel" => Some(XF86XK_Excel),
        "XF86XK_Explorer" | "XF86Explorer" => Some(XF86XK_Explorer),
        "XF86XK_Game" | "XF86Game" => Some(XF86XK_Game),
        "XF86XK_Go" | "XF86Go" => Some(XF86XK_Go),
        "XF86XK_iTouch" | "XF86iTouch" => Some(XF86XK_iTouch),
        "XF86XK_LogOff" | "XF86LogOff" => Some(XF86XK_LogOff),
        "XF86XK_Market" | "XF86Market" => Some(XF86XK_Market),
        "XF86XK_Meeting" | "XF86Meeting" => Some(XF86XK_Meeting),
        "XF86XK_MenuKB" | "XF86MenuKB" => Some(XF86XK_MenuKB),
        "XF86XK_MenuPB" | "XF86MenuPB" => Some(XF86XK_MenuPB),
        "XF86XK_MySites" | "XF86MySites" => Some(XF86XK_MySites),
        "XF86XK_New" | "XF86New" => Some(XF86XK_New),
        "XF86XK_News" | "XF86News" => Some(XF86XK_News),
        "XF86XK_OfficeHome" | "XF86OfficeHome" => Some(XF86XK_OfficeHome),
        "XF86XK_Open" | "XF86Open" => Some(XF86XK_Open),
        "XF86XK_Option" | "XF86Option" => Some(XF86XK_Option),
        "XF86XK_Paste" | "XF86Paste" => Some(XF86XK_Paste),
        "XF86XK_Phone" | "XF86Phone" => Some(XF86XK_Phone),
        "XF86XK_Q" | "XF86Q" => Some(XF86XK_Q),
        "XF86XK_Reply" | "XF86Reply" => Some(XF86XK_Reply),
        "XF86XK_Reload" | "XF86Reload" => Some(XF86XK_Reload),
        "XF86XK_RotateWindows" | "XF86RotateWindows" => Some(XF86XK_RotateWindows),
        "XF86XK_RotationPB" | "XF86RotationPB" => Some(XF86XK_RotationPB),
        "XF86XK_RotationKB" | "XF86RotationKB" => Some(XF86XK_RotationKB),
        "XF86XK_Save" | "XF86Save" => Some(XF86XK_Save),
        "XF86XK_ScrollUp" | "XF86ScrollUp" => Some(XF86XK_ScrollUp),
        "XF86XK_ScrollDown" | "XF86ScrollDown" => Some(XF86XK_ScrollDown),
        "XF86XK_ScrollClick" | "XF86ScrollClick" => Some(XF86XK_ScrollClick),
        "XF86XK_Send" | "XF86Send" => Some(XF86XK_Send),
        "XF86XK_Spell" | "XF86Spell" => Some(XF86XK_Spell),
        "XF86XK_SplitScreen" | "XF86SplitScreen" => Some(XF86XK_SplitScreen),
        "XF86XK_Support" | "XF86Support" => Some(XF86XK_Support),
        "XF86XK_TaskPane" | "XF86TaskPane" => Some(XF86XK_TaskPane),
        "XF86XK_Terminal" | "XF86Terminal" => Some(XF86XK_Terminal),
        "XF86XK_Tools" | "XF86Tools" => Some(XF86XK_Tools),
        "XF86XK_Travel" | "XF86Travel" => Some(XF86XK_Travel),
        "XF86XK_UserPB" | "XF86UserPB" => Some(XF86XK_UserPB),
        "XF86XK_User1KB" | "XF86User1KB" => Some(XF86XK_User1KB),
        "XF86XK_User2KB" | "XF86User2KB" => Some(XF86XK_User2KB),
        "XF86XK_Video" | "XF86Video" => Some(XF86XK_Video),
        "XF86XK_WheelButton" | "XF86WheelButton" => Some(XF86XK_WheelButton),
        "XF86XK_Word" | "XF86Word" => Some(XF86XK_Word),
        "XF86XK_Xfer" | "XF86Xfer" => Some(XF86XK_Xfer),
        "XF86XK_ZoomIn" | "XF86ZoomIn" => Some(XF86XK_ZoomIn),
        "XF86XK_ZoomOut" | "XF86ZoomOut" => Some(XF86XK_ZoomOut),
        "XF86XK_Away" | "XF86Away" => Some(XF86XK_Away),
        "XF86XK_Messenger" | "XF86Messenger" => Some(XF86XK_Messenger),
        "XF86XK_WebCam" | "XF86WebCam" => Some(XF86XK_WebCam),
        "XF86XK_MailForward" | "XF86MailForward" => Some(XF86XK_MailForward),
        "XF86XK_Pictures" | "XF86Pictures" => Some(XF86XK_Pictures),
        "XF86XK_Music" | "XF86Music" => Some(XF86XK_Music),
        "XF86XK_Battery" | "XF86Battery" => Some(XF86XK_Battery),
        "XF86XK_Bluetooth" | "XF86Bluetooth" => Some(XF86XK_Bluetooth),
        "XF86XK_WLAN" | "XF86WLAN" => Some(XF86XK_WLAN),
        "XF86XK_UWB" | "XF86UWB" => Some(XF86XK_UWB),
        "XF86XK_AudioForward" | "XF86AudioForward" => Some(XF86XK_AudioForward),
        "XF86XK_AudioRepeat" | "XF86AudioRepeat" => Some(XF86XK_AudioRepeat),
        "XF86XK_AudioRandomPlay" | "XF86AudioRandomPlay" => Some(XF86XK_AudioRandomPlay),
        "XF86XK_Subtitle" | "XF86Subtitle" => Some(XF86XK_Subtitle),
        "XF86XK_AudioCycleTrack" | "XF86AudioCycleTrack" => Some(XF86XK_AudioCycleTrack),
        "XF86XK_CycleAngle" | "XF86CycleAngle" => Some(XF86XK_CycleAngle),
        "XF86XK_FrameBack" | "XF86FrameBack" => Some(XF86XK_FrameBack),
        "XF86XK_FrameForward" | "XF86FrameForward" => Some(XF86XK_FrameForward),
        "XF86XK_Time" | "XF86Time" => Some(XF86XK_Time),
        "XF86XK_Select" | "XF86Select" => Some(XF86XK_Select),
        "XF86XK_View" | "XF86View" => Some(XF86XK_View),
        "XF86XK_TopMenu" | "XF86TopMenu" => Some(XF86XK_TopMenu),
        "XF86XK_Red" | "XF86Red" => Some(XF86XK_Red),
        "XF86XK_Green" | "XF86Green" => Some(XF86XK_Green),
        "XF86XK_Yellow" | "XF86Yellow" => Some(XF86XK_Yellow),
        "XF86XK_Blue" | "XF86Blue" => Some(XF86XK_Blue),
        "XF86XK_Suspend" | "XF86Suspend" => Some(XF86XK_Suspend),
        "XF86XK_Hibernate" | "XF86Hibernate" => Some(XF86XK_Hibernate),
        "XF86XK_TouchpadToggle" | "XF86TouchpadToggle" => Some(XF86XK_TouchpadToggle),
        "XF86XK_TouchpadOn" | "XF86TouchpadOn" => Some(XF86XK_TouchpadOn),
        "XF86XK_TouchpadOff" | "XF86TouchpadOff" => Some(XF86XK_TouchpadOff),
        "XF86XK_AudioMicMute" | "XF86AudioMicMute" => Some(XF86XK_AudioMicMute),
        "XF86XK_Switch_VT_1" | "XF86Switch_VT_1" => Some(XF86XK_Switch_VT_1),
        "XF86XK_Switch_VT_2" | "XF86Switch_VT_2" => Some(XF86XK_Switch_VT_2),
        "XF86XK_Switch_VT_3" | "XF86Switch_VT_3" => Some(XF86XK_Switch_VT_3),
        "XF86XK_Switch_VT_4" | "XF86Switch_VT_4" => Some(XF86XK_Switch_VT_4),
        "XF86XK_Switch_VT_5" | "XF86Switch_VT_5" => Some(XF86XK_Switch_VT_5),
        "XF86XK_Switch_VT_6" | "XF86Switch_VT_6" => Some(XF86XK_Switch_VT_6),
        "XF86XK_Switch_VT_7" | "XF86Switch_VT_7" => Some(XF86XK_Switch_VT_7),
        "XF86XK_Switch_VT_8" | "XF86Switch_VT_8" => Some(XF86XK_Switch_VT_8),
        "XF86XK_Switch_VT_9" | "XF86Switch_VT_9" => Some(XF86XK_Switch_VT_9),
        "XF86XK_Switch_VT_10" | "XF86Switch_VT_10" => Some(XF86XK_Switch_VT_10),
        "XF86XK_Switch_VT_11" | "XF86Switch_VT_11" => Some(XF86XK_Switch_VT_11),
        "XF86XK_Switch_VT_12" | "XF86Switch_VT_12" => Some(XF86XK_Switch_VT_12),
        "XF86XK_Ungrab" | "XF86Ungrab" => Some(XF86XK_Ungrab),
        "XF86XK_ClearGrab" | "XF86ClearGrab" => Some(XF86XK_ClearGrab),
        "XF86XK_Next_VMode" | "XF86Next_VMode" => Some(XF86XK_Next_VMode),
        "XF86XK_Prev_VMode" | "XF86Prev_VMode" => Some(XF86XK_Prev_VMode),
        "XF86XK_LogWindowTree" | "XF86LogWindowTree" => Some(XF86XK_LogWindowTree),
        "XF86XK_LogGrabInfo" | "XF86LogGrabInfo" => Some(XF86XK_LogGrabInfo),
        "ISO_Lock" => Some(XK_ISO_Lock),
        "ISO_Level2_Latch" => Some(XK_ISO_Level2_Latch),
        "ISO_Level3_Shift" => Some(XK_ISO_Level3_Shift),
        "ISO_Level3_Latch" => Some(XK_ISO_Level3_Latch),
        "ISO_Level3_Lock" => Some(XK_ISO_Level3_Lock),
        "ISO_Level5_Shift" => Some(XK_ISO_Level5_Shift),
        "ISO_Level5_Latch" => Some(XK_ISO_Level5_Latch),
        "ISO_Level5_Lock" => Some(XK_ISO_Level5_Lock),
        "ISO_Group_Shift" => Some(XK_ISO_Group_Shift),
        "ISO_Group_Latch" => Some(XK_ISO_Group_Latch),
        "ISO_Group_Lock" => Some(XK_ISO_Group_Lock),
        "ISO_Next_Group" => Some(XK_ISO_Next_Group),
        "ISO_Next_Group_Lock" => Some(XK_ISO_Next_Group_Lock),
        "ISO_Prev_Group" => Some(XK_ISO_Prev_Group),
        "ISO_Prev_Group_Lock" => Some(XK_ISO_Prev_Group_Lock),
        "ISO_First_Group" => Some(XK_ISO_First_Group),
        "ISO_First_Group_Lock" => Some(XK_ISO_First_Group_Lock),
        "ISO_Last_Group" => Some(XK_ISO_Last_Group),
        "ISO_Last_Group_Lock" => Some(XK_ISO_Last_Group_Lock),
        "ISO_Left_Tab" => Some(XK_ISO_Left_Tab),
        "ISO_Move_Line_Up" => Some(XK_ISO_Move_Line_Up),
        "ISO_Move_Line_Down" => Some(XK_ISO_Move_Line_Down),
        "ISO_Partial_Line_Up" => Some(XK_ISO_Partial_Line_Up),
        "ISO_Partial_Line_Down" => Some(XK_ISO_Partial_Line_Down),
        "ISO_Partial_Space_Left" => Some(XK_ISO_Partial_Space_Left),
        "ISO_Partial_Space_Right" => Some(XK_ISO_Partial_Space_Right),
        "ISO_Set_Margin_Left" => Some(XK_ISO_Set_Margin_Left),
        "ISO_Set_Margin_Right" => Some(XK_ISO_Set_Margin_Right),
        "ISO_Release_Margin_Left" => Some(XK_ISO_Release_Margin_Left),
        "ISO_Release_Margin_Right" => Some(XK_ISO_Release_Margin_Right),
        "ISO_Release_Both_Margins" => Some(XK_ISO_Release_Both_Margins),
        "ISO_Fast_Cursor_Left" => Some(XK_ISO_Fast_Cursor_Left),
        "ISO_Fast_Cursor_Right" => Some(XK_ISO_Fast_Cursor_Right),
        "ISO_Fast_Cursor_Up" => Some(XK_ISO_Fast_Cursor_Up),
        "ISO_Fast_Cursor_Down" => Some(XK_ISO_Fast_Cursor_Down),
        "ISO_Continuous_Underline" => Some(XK_ISO_Continuous_Underline),
        "ISO_Discontinuous_Underline" => Some(XK_ISO_Discontinuous_Underline),
        "ISO_Emphasize" => Some(XK_ISO_Emphasize),
        "ISO_Center_Object" => Some(XK_ISO_Center_Object),
        "ISO_Enter" => Some(XK_ISO_Enter),
        "dead_grave" => Some(XK_dead_grave),
        "dead_acute" => Some(XK_dead_acute),
        "dead_circumflex" => Some(XK_dead_circumflex),
        "dead_tilde" => Some(XK_dead_tilde),
        "dead_perispomeni" => Some(XK_dead_perispomeni),
        "dead_macron" => Some(XK_dead_macron),
        "dead_breve" => Some(XK_dead_breve),
        "dead_abovedot" => Some(XK_dead_abovedot),
        "dead_diaeresis" => Some(XK_dead_diaeresis),
        "dead_abovering" => Some(XK_dead_abovering),
        "dead_doubleacute" => Some(XK_dead_doubleacute),
        "dead_caron" => Some(XK_dead_caron),
        "dead_cedilla" => Some(XK_dead_cedilla),
        "dead_ogonek" => Some(XK_dead_ogonek),
        "dead_iota" => Some(XK_dead_iota),
        "dead_voiced_sound" => Some(XK_dead_voiced_sound),
        "dead_semivoiced_sound" => Some(XK_dead_semivoiced_sound),
        "dead_belowdot" => Some(XK_dead_belowdot),
        "dead_hook" => Some(XK_dead_hook),
        "dead_horn" => Some(XK_dead_horn),
        "dead_stroke" => Some(XK_dead_stroke),
        "dead_abovecomma" => Some(XK_dead_abovecomma),
        "dead_psili" => Some(XK_dead_psili),
        "dead_abovereversedcomma" => Some(XK_dead_abovereversedcomma),
        "dead_dasia" => Some(XK_dead_dasia),
        "dead_doublegrave" => Some(XK_dead_doublegrave),
        "dead_belowring" => Some(XK_dead_belowring),
        "dead_belowmacron" => Some(XK_dead_belowmacron),
        "dead_belowcircumflex" => Some(XK_dead_belowcircumflex),
        "dead_belowtilde" => Some(XK_dead_belowtilde),
        "dead_belowbreve" => Some(XK_dead_belowbreve),
        "dead_belowdiaeresis" => Some(XK_dead_belowdiaeresis),
        "dead_invertedbreve" => Some(XK_dead_invertedbreve),
        "dead_belowcomma" => Some(XK_dead_belowcomma),
        "dead_currency" => Some(XK_dead_currency),
        "dead_lowline" => Some(XK_dead_lowline),
        "dead_aboveverticalline" => Some(XK_dead_aboveverticalline),
        "dead_belowverticalline" => Some(XK_dead_belowverticalline),
        "dead_longsolidusoverlay" => Some(XK_dead_longsolidusoverlay),
        "dead_a" => Some(XK_dead_a),
        "dead_A" => Some(XK_dead_A),
        "dead_e" => Some(XK_dead_e),
        "dead_E" => Some(XK_dead_E),
        "dead_i" => Some(XK_dead_i),
        "dead_I" => Some(XK_dead_I),
        "dead_o" => Some(XK_dead_o),
        "dead_O" => Some(XK_dead_O),
        "dead_u" => Some(XK_dead_u),
        "dead_U" => Some(XK_dead_U),
        "dead_small_schwa" => Some(XK_dead_small_schwa),
        "dead_capital_schwa" => Some(XK_dead_capital_schwa),
        "dead_greek" => Some(XK_dead_greek),
        "First_Virtual_Screen" => Some(XK_First_Virtual_Screen),
        "Prev_Virtual_Screen" => Some(XK_Prev_Virtual_Screen),
        "Next_Virtual_Screen" => Some(XK_Next_Virtual_Screen),
        "Last_Virtual_Screen" => Some(XK_Last_Virtual_Screen),
        "Terminate_Server" => Some(XK_Terminate_Server),
        "AccessX_Enable" => Some(XK_AccessX_Enable),
        "AccessX_Feedback_Enable" => Some(XK_AccessX_Feedback_Enable),
        "RepeatKeys_Enable" => Some(XK_RepeatKeys_Enable),
        "SlowKeys_Enable" => Some(XK_SlowKeys_Enable),
        "BounceKeys_Enable" => Some(XK_BounceKeys_Enable),
        "StickyKeys_Enable" => Some(XK_StickyKeys_Enable),
        "MouseKeys_Enable" => Some(XK_MouseKeys_Enable),
        "MouseKeys_Accel_Enable" => Some(XK_MouseKeys_Accel_Enable),
        "Overlay1_Enable" => Some(XK_Overlay1_Enable),
        "Overlay2_Enable" => Some(XK_Overlay2_Enable),
        "AudibleBell_Enable" => Some(XK_AudibleBell_Enable),
        "Pointer_Left" => Some(XK_Pointer_Left),
        "Pointer_Right" => Some(XK_Pointer_Right),
        "Pointer_Up" => Some(XK_Pointer_Up),
        "Pointer_Down" => Some(XK_Pointer_Down),
        "Pointer_UpLeft" => Some(XK_Pointer_UpLeft),
        "Pointer_UpRight" => Some(XK_Pointer_UpRight),
        "Pointer_DownLeft" => Some(XK_Pointer_DownLeft),
        "Pointer_DownRight" => Some(XK_Pointer_DownRight),
        "Pointer_Button_Dflt" => Some(XK_Pointer_Button_Dflt),
        "Pointer_Button1" => Some(XK_Pointer_Button1),
        "Pointer_Button2" => Some(XK_Pointer_Button2),
        "Pointer_Button3" => Some(XK_Pointer_Button3),
        "Pointer_Button4" => Some(XK_Pointer_Button4),
        "Pointer_Button5" => Some(XK_Pointer_Button5),
        "Pointer_DblClick_Dflt" => Some(XK_Pointer_DblClick_Dflt),
        "Pointer_DblClick1" => Some(XK_Pointer_DblClick1),
        "Pointer_DblClick2" => Some(XK_Pointer_DblClick2),
        "Pointer_DblClick3" => Some(XK_Pointer_DblClick3),
        "Pointer_DblClick4" => Some(XK_Pointer_DblClick4),
        "Pointer_DblClick5" => Some(XK_Pointer_DblClick5),
        "Pointer_Drag_Dflt" => Some(XK_Pointer_Drag_Dflt),
        "Pointer_Drag1" => Some(XK_Pointer_Drag1),
        "Pointer_Drag2" => Some(XK_Pointer_Drag2),
        "Pointer_Drag3" => Some(XK_Pointer_Drag3),
        "Pointer_Drag4" => Some(XK_Pointer_Drag4),
        "Pointer_Drag5" => Some(XK_Pointer_Drag5),
        "Pointer_EnableKeys" => Some(XK_Pointer_EnableKeys),
        "Pointer_Accelerate" => Some(XK_Pointer_Accelerate),
        "Pointer_DfltBtnNext" => Some(XK_Pointer_DfltBtnNext),
        "Pointer_DfltBtnPrev" => Some(XK_Pointer_DfltBtnPrev),
        "ch" => Some(XK_ch),
        "Ch" => Some(XK_Ch),
        "CH" => Some(XK_CH),
        "c_h" => Some(XK_c_h),
        "C_h" => Some(XK_C_h),
        "C_H" => Some(XK_C_H),
        _ => None,
    }
}
