/*
 * Copyright (c) 2015-2016 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <util.h>
#include <libgen.h>
#include <strings.h>
#include <syslog.h>
#include <getopt.h>
#include <fcntl.h>
#include <err.h>
#include <sysexits.h>

#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/mount.h>
#include <sys/statvfs.h>

#include <dev/fssvar.h>

#define MAX_FSSDEV 256
#define FSS_TTL 22

char atexit_msg[MAXPATHLEN] = { 0 };

void
atexit_func(void)
{
	if (*atexit_msg != '\0');
		syslog(LOG_INFO, atexit_msg);
	return;
}

int
fss_find(char *dev_name)
{
	char fss_dev[MAXPATHLEN];
	struct fss_get fg;
	int i = 0;
	int fd;

	for (i = 0; i < MAX_FSSDEV; i++) {
		(void)snprintf(fss_dev, sizeof(fss_dev), "fss%d", i);
		if ((fd = opendisk(fss_dev, O_RDWR,
				   dev_name, MAXPATHLEN, 0)) == -1)
			return -1;

		/* not in use? */
		if (ioctl(fd, FSSIOCGET, &fg) != 0) {
			if (errno == ENXIO)
				return fd;
		}

		(void)close(fd);
	}

	return -1;
}

void
usage(void)
{
	errx(EX_USAGE, "[-U] filesystem backing-dir [ttl]");
	/* NOTREACHED */
	return;
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	char *argv0 = argv[0];
	char *mount;
	char *backend_dir;
	int fd;
	int tmpfd;
	struct statvfs vfs;
	char fssdev[MAXPATHLEN];
	char backend[MAXPATHLEN];
	char syslogid[MAXPATHLEN];
	struct fss_set fs;
	int fss_ttl;
	int ch;
	int unlink_on_create = FSS_UNLINK_ON_CREATE;

	while ((ch = getopt(argc, argv, "U")) != -1) {
		switch (ch) {
		case 'U':
			unlink_on_create = 0;
			break;
		default:
			usage();
			break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 2 && argc != 3)
		usage();

	if (argc == 3)
		fss_ttl = atoi(argv[2]) * 3600;
	else
		fss_ttl = FSS_TTL * 3600;

	mount = argv[0];
	backend_dir = argv[1];

	if (statvfs(mount, &vfs) != 0)
		err(EX_OSERR, "Cannot get info on %s", mount);

	fd = fss_find(fssdev);

	memset(&fs, 0, sizeof(fs));

	(void)snprintf(backend, sizeof(backend),
		       "%s/fssbackend-XXXXX", backend_dir);
	if ((tmpfd = mkstemp(backend)) == -1)
		err(EX_OSERR, "cannot create backing store in %s", backend_dir);

	if (truncate(backend, vfs.f_blocks * vfs.f_frsize) != 0)
		err(EX_OSERR, "cannot resize %s to %" PRId64,
			      backend, vfs.f_blocks * vfs.f_frsize);

	(void)close(tmpfd);

	if (daemon(0, 0))
		warn("Cannot become a daemon");

	(void)snprintf(syslogid, sizeof(syslogid), "%s[%d]",
		       basename(argv0), getpid());
	openlog(syslogid, 0, LOG_DAEMON);
	syslog(LOG_INFO, "START Snapshot for %s in %s", mount, fssdev);

	(void)snprintf(atexit_msg, sizeof(atexit_msg),
		       "END Snapshot for %s in %s", mount, fssdev);
	(void)atexit(*atexit_func);

	fs.fss_bstore = backend;
	fs.fss_mount = mount;
	fs.fss_flags = FSS_UNCONFIG_ON_CLOSE|unlink_on_create;
	
	if (ioctl(fd, FSSIOCSET, &fs) != 0) {
		syslog(LOG_ERR, "ERROR Snapshot for %s in %s: %s",
		       mount, fssdev, strerror(errno));
		(void)unlink(fs.fss_bstore);
		exit(EX_OSERR);
	}

	sleep(fss_ttl);

	syslog(LOG_INFO, "END Snapshot for %s in %s", mount, fssdev);
	*atexit_msg = '\0';

	(void)close(fd);

	if (unlink_on_create == 0)
		(void)unlink(fs.fss_bstore);

	exit(EX_OK);	
}
