/*
 * ksi_writ.c
 * write
 *
 * Copyright (C) 1997-2010, ivan demakov
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        Ivan Demakov <ksion@users.sourceforge.net>
 * Creation date: Fri Aug  1 04:08:03 1997
 * Last Update:   Sun Apr 25 22:35:39 2010
 *
 */

#include "ksi_int.h"
#include "ksi_comp.h"
#include "ksi_env.h"
#include "ksi_klos.h"
#include "ksi_printf.h"
#include "ksi_buf.h"


static const char*
ksi_imm2str (ksi_obj o)
{
  if (o == ksi_nil)
    return "()";
  if (o == ksi_false)
    return "#f";
  if (o == ksi_true)
    return "#t";
  if (o == ksi_void)
    return "#!void";
  if (o == ksi_eof)
    return "#!eof";
  if (o == ksi_err)
    return "#!error";

  return ksi_aprintf ("#!imm-%x", o);
}

static const char*
ksi_core2str (ksi_core o)
{
    switch (o->core) {
    case KSI_TAG_BEGIN:
        return ksi_aprintf ("#!core-begin");
    case KSI_TAG_AND:
        return ksi_aprintf ("#!core-and");
    case KSI_TAG_OR:
        return ksi_aprintf ("#!core-or");
    case KSI_TAG_IF:
        return ksi_aprintf ("#!core-if");
    case KSI_TAG_LAMBDA:
        return ksi_aprintf ("#!core-lambda");
    case KSI_TAG_DEFINE:
        return ksi_aprintf ("#!core-define");
    case KSI_TAG_DEFSYNTAX:
        return ksi_aprintf ("#!core-defsyntax");
    case KSI_TAG_SET:
        return ksi_aprintf ("#!core-set!");
    case KSI_TAG_QUOTE:
        return ksi_aprintf ("#!core-quote");
    default:
        return ksi_aprintf ("#@{core-%x}", o->core);
    }
}

static void
one_pair (ksi_buffer_t buf, ksi_obj *x, const char* (*print) (ksi_obj))
{
  const char *p = print (KSI_CAR (*x));
  ksi_buffer_append(buf, p, strlen(p));

  *x = KSI_CDR (*x);
  if (*x == ksi_nil)
    return;

  if (!KSI_PAIR_P (*x)) {
    ksi_buffer_put(buf, ' ');
    ksi_buffer_put(buf, '.');
    ksi_buffer_put(buf, ' ');

    p = print (*x);
    ksi_buffer_append(buf, p, strlen(p));
    return;
  }

  ksi_buffer_put(buf, ' ');
}

static const char*
ksi_print_pair (char* prefix, ksi_obj x, char* suffix, const char* (*print) (ksi_obj))
{
    ksi_buffer_t buf = ksi_new_buffer(0, 0);
    ksi_obj z = x;

    if (KSI_PAIR_P(KSI_CDR(x)) && KSI_CDR(KSI_CDR(x)) == ksi_nil) {
        if (KSI_CAR(x) == ksi_data->sym_quote) {
            ksi_buffer_put(buf, '\'');
        quote:
            x = KSI_CDR(x);
            one_pair(buf, &x, print);
            ksi_buffer_put(buf, '\0');
            return ksi_buffer_data(buf);
        }
        if (KSI_CAR(x) == ksi_data->sym_quasiquote) {
            ksi_buffer_put(buf, '`');
            goto quote;
        }
        if (KSI_CAR(x) == ksi_data->sym_unquote) {
            ksi_buffer_put(buf, ',');
            goto quote;
        }
        if (KSI_CAR(x) == ksi_data->sym_unquote_splicing) {
            ksi_buffer_put(buf, ',');
            ksi_buffer_put(buf, '@');
            goto quote;
        }

        if (KSI_CAR(x) == ksi_data->sym_syntax) {
            ksi_buffer_put(buf, '#');
            ksi_buffer_put(buf, '\'');
            goto quote;
        }
        if (KSI_CAR(x) == ksi_data->sym_quasisyntax) {
            ksi_buffer_put(buf, '#');
            ksi_buffer_put(buf, '`');
            goto quote;
        }
        if (KSI_CAR(x) == ksi_data->sym_unsyntax) {
            ksi_buffer_put(buf, '#');
            ksi_buffer_put(buf, ',');
            goto quote;
        }
        if (KSI_CAR(x) == ksi_data->sym_unsyntax_splicing) {
            ksi_buffer_put(buf, '#');
            ksi_buffer_put(buf, ',');
            ksi_buffer_put(buf, '@');
            goto quote;
        }
    }

    ksi_buffer_append(buf, prefix, strlen(prefix));

    for (;;) {
        one_pair(buf, &x, print);
        if (!KSI_PAIR_P(x))
            break;
        one_pair(buf, &x, print);
        if (!KSI_PAIR_P(x))
            break;

        z = KSI_CDR(z);
        if (z == x) {
            /* cycle found */
            ksi_buffer_put(buf, '.');
            ksi_buffer_put(buf, ' ');
            ksi_buffer_put(buf, '.');
            ksi_buffer_put(buf, '.');
            ksi_buffer_put(buf, '.');
            break;
        }
    }

    ksi_buffer_append(buf, suffix, strlen(suffix));
    ksi_buffer_put(buf, '\0');
    return ksi_buffer_data(buf);
}

static const char*
ksi_print_vec (char* beg, ksi_obj x, char* end, const char* (*print) (ksi_obj))
{
  int n;
  ksi_buffer_t buf = ksi_new_buffer(0, 0);

  ksi_buffer_append(buf, beg, strlen(beg));

  if (KSI_VEC_LEN (x) > 0) {
    for (n = 0;;) {
      const char* p = print (KSI_VEC_REF (x, n));
      ksi_buffer_append(buf, p, strlen(p));

      if (++n >= KSI_VEC_LEN (x))
        break;

      ksi_buffer_put(buf, ' ');
    }
  }

  ksi_buffer_append(buf, end, strlen(end));
  ksi_buffer_put(buf, '\0');
  return ksi_buffer_data(buf);
}

static const char*
ksi_print_values (ksi_obj x)
{
  if (KSI_VALUES_VALS (x) == ksi_nil) {
    return "#@<values ()>";
  } else {
    const char *p = ksi_obj2str(KSI_VALUES_VALS (x));
    return ksi_aprintf ("#@<values %s>", p);
  }
}

static const char*
ksi_print_port (ksi_obj o)
{
  return ksi_aprintf ("#@<%s%s%s%s>",
                      ((ksi_port) o) -> closed ? "closed-" : "",
                      ((ksi_port) o) -> input  ? "input-"  : "",
                      ((ksi_port) o) -> output ? "output-" : "",
                      ((ksi_port) o) -> ops -> name ((ksi_port) o));
}

static const char*
ksi_print_prim (ksi_obj x)
{
  if (KSI_PRIM_NAME (x))
    return ksi_aprintf ("#@<procedure %s >", KSI_PRIM_NAME (x));

  return ksi_aprintf ("#@<procedure %p>", x);
}

static const char*
ksi_print_env (ksi_obj x)
{
    if (((ksi_env) x)->name == ksi_void)
        return ksi_aprintf("#@<top-level-env>");
    if (((ksi_env) x)->name == ksi_nil)
        return ksi_aprintf("#@<env %p>", x);
    return ksi_aprintf("#@<env %s>", ksi_obj2str(((ksi_env) x)->name));
}

static const char*
ksi_print_hashtab (ksi_obj x)
{
    return ksi_aprintf ("#@<hashtable %p>", x);
}

static const char*
ksi_print_code (char* beg, ksi_obj x, char* end)
{
    int n;
    ksi_buffer_t buf = ksi_new_buffer(0, 0);

    ksi_buffer_append(buf, beg, strlen(beg));
    for (n = 0;;) {
        const char* p = ksi_obj2str (KSI_CODE_VAL (x, n));
        ksi_buffer_append(buf, p, strlen(p));

        if (++n >= KSI_CODE_NUM (x) + 1)
            break;

        ksi_buffer_put(buf, ' ');
    }

    ksi_buffer_append(buf, end, strlen(end));
    ksi_buffer_put(buf, '\0');
    return ksi_buffer_data(buf);
}

static const char*
ksi_print_closure (ksi_obj x)
{
    const char *name = 0;
    ksi_obj doc = KSI_CLOS_DOC(x);

    if (KSI_SYM_P(doc)) {
        name = KSI_SYM_PTR(doc);
    } else {
        name = ksi_aprintf("%p", x);
    }
    return ksi_aprintf("#@<closure %s %s >", name, ksi_obj2str((ksi_obj) KSI_CLOS_FRM(x)->env));
}

static const char*
ksi_print_varbox (const char* p, ksi_obj x)
{
    return ksi_aprintf("#@{%s %d %d}", p, KSI_VARBOX_LEV(x), KSI_VARBOX_NUM(x));
}

static const char*
ksi_print_freevar (const char* p, ksi_obj x)
{
    return ksi_aprintf("#@{%s %s %s}", p, ksi_obj2str(KSI_FREEVAR_SYM(x)), ksi_obj2str(KSI_FREEVAR_ENV(x)->name));
}

static const char*
ksi_print_event (ksi_obj x)
{
    const char* p = ((ksi_event) x) -> ops -> name ((ksi_event) x);
    return ksi_aprintf ("#@<event %s>", p);
}

const char*
ksi_obj2str (ksi_obj o)
{
  if (!o)
    return "#!null";

  switch (o->o.itag) {
  case KSI_TAG_IMM:
    return ksi_imm2str (o);

  case KSI_TAG_BIGNUM:
  case KSI_TAG_FLONUM:
    return ksi_num2str (o, 10);

  case KSI_TAG_SYMBOL:
    return ksi_symbol2str ((ksi_symbol) o);

  case KSI_TAG_KEYWORD:
    return ksi_key2str ((ksi_keyword) o);

  case KSI_TAG_PAIR:
  case KSI_TAG_CONST_PAIR:
    return ksi_print_pair ("(", o, ")", ksi_obj2str);

  case KSI_TAG_VECTOR:
  case KSI_TAG_CONST_VECTOR:
    return ksi_print_vec ("#(", o, ")", ksi_obj2str);

  case KSI_TAG_STRING:
  case KSI_TAG_CONST_STRING:
    return ksi_string2str (o);

  case KSI_TAG_CHAR:
    return ksi_char2str ((ksi_char) o);

  case KSI_TAG_VAR0:
  case KSI_TAG_VAR1:
  case KSI_TAG_VAR2:
  case KSI_TAG_VARN:
    return ksi_print_varbox("varbox", o);

  case KSI_TAG_FREEVAR:
    return ksi_print_freevar("freevar", o);

  case KSI_TAG_LOCAL:
    return ksi_print_freevar("local", o);

  case KSI_TAG_IMPORTED:
    return ksi_print_freevar("imported", o);

  case KSI_TAG_PRIM:
  case KSI_TAG_PRIM_0:
  case KSI_TAG_PRIM_1:
  case KSI_TAG_PRIM_2:
  case KSI_TAG_PRIM_r:
    return ksi_print_prim(o);

  case KSI_TAG_CONS:
    return "#@cons";

  case KSI_TAG_CAR:
    return "#@car";

  case KSI_TAG_CDR:
    return "#@cdr";

  case KSI_TAG_NOT:
    return "#@not";

  case KSI_TAG_EQP:
    return "#@eq?";

  case KSI_TAG_EQVP:
    return "#@eqv?";

  case KSI_TAG_EQUALP:
    return "#@equal?";

  case KSI_TAG_MEMQ:
    return "#@memq";

  case KSI_TAG_MEMV:
    return "#@memv";

  case KSI_TAG_MEMBER:
    return "#@member";

  case KSI_TAG_NULLP:
    return "#@null?";

  case KSI_TAG_PAIRP:
    return "#@pair?";

  case KSI_TAG_LISTP:
    return "#@list?";

  case KSI_TAG_LIST:
    return "#@list";

  case KSI_TAG_MK_VECTOR:
    return "#@vector";

  case KSI_TAG_LIST2VECTOR:
    return "#@list->vector";

  case KSI_TAG_APPEND:
    return "#@append";

  case KSI_TAG_APPLY:
    return "#@apply";

  case KSI_TAG_CALL_CC:
    return "#@call/cc";

  case KSI_TAG_CALL_WITH_VALUES:
    return "#@call/vs";

  case KSI_TAG_VECTORP:
    return "#@vector?";

  case KSI_TAG_CLOSURE:
    return ksi_print_closure (o);

  case KSI_TAG_PRIM_CLOSURE:
      return ksi_aprintf ("#@<closure %s >", ksi_obj2str(((ksi_prim_closure) o)->proc));

  case KSI_TAG_QUOTE:
    return ksi_print_code ("#@{quote ", o, "}");

  case KSI_TAG_AND:
    return ksi_print_code ("#@{and ", o, "}");

  case KSI_TAG_OR:
    return ksi_print_code ("#@{or ", o, "}");

  case KSI_TAG_BEGIN:
    return ksi_print_code ("#@{begin ", o, "}");

  case KSI_TAG_IF:
    return ksi_print_code ("#@{if ", o, "}");

  case KSI_TAG_DEFINE:
    return ksi_print_code ("#@{define ", o, "}");

  case KSI_TAG_SET:
    return ksi_print_code ("#@{set! ", o, "}");

  case KSI_TAG_CALL:
    return ksi_print_code ("#@{call ", o, "}");

  case KSI_TAG_FRAME:
    return ksi_print_code ("#@{frame ", o, "}");

  case KSI_TAG_LAMBDA:
    return ksi_print_code ("#@{lambda ", o, "}");

  case KSI_TAG_IMPORT:
    return ksi_print_code ("#@{import ", o, "}");

  case KSI_TAG_SYNTAX:
    return ksi_print_code ("#@{syntax ", o, "}");

  case KSI_TAG_INSTANCE:
    return ksi_inst2str ((ksi_instance) o);

  case KSI_TAG_NEXT_METHOD:
    return "#@{next-method}";

  case KSI_TAG_PORT:
    return ksi_print_port (o);

  case KSI_TAG_VALUES:
    return ksi_print_values (o);

  case KSI_TAG_ENVIRON:
    return ksi_print_env (o);

  case KSI_TAG_HASHTAB:
    return ksi_print_hashtab (o);

  case KSI_TAG_EXN:
    return ksi_print_vec ("#@<exception ", o, ">", ksi_obj2str);

  case KSI_TAG_EVENT:
    return ksi_print_event (o);

  case KSI_TAG_CORE:
      return ksi_core2str ((ksi_core) o);

  case KSI_TAG_EXTENDED:
  case KSI_TAG_BROKEN:
    return KSI_EXT_TAG (o) -> print (KSI_EXT_TAG (o), o, 1);
  }

  ksi_exn_error (0, 0, "object->string: invalid object tag: 0x%x", o->o.itag);
  return "";
}

const char*
ksi_obj2name (ksi_obj x)
{
    char *ptr;

    if (!x)
        return "#<null>";

    switch (x->o.itag) {
    case KSI_TAG_SYMBOL:
        return KSI_SYM_PTR(x);

    case KSI_TAG_KEYWORD:
        return KSI_KEY_PTR(x);

    case KSI_TAG_CHAR:
        ptr = ksi_malloc_data(2);
        ptr[0] = KSI_CHAR_CODE(x);
        ptr[1] = '\0';
        return ptr;

    case KSI_TAG_CONST_STRING:
    case KSI_TAG_STRING:
        return KSI_STR_PTR(x);

    case KSI_TAG_PAIR:
    case KSI_TAG_CONST_PAIR:
        return ksi_print_pair("", x, "", ksi_obj2name);

    case KSI_TAG_VECTOR:
    case KSI_TAG_CONST_VECTOR:
        return ksi_print_vec("", x, "", ksi_obj2name);

    default:
        break;
    }

    return ksi_obj2str (x);
}


static ksi_obj
coerce2num (ksi_obj x)
{
  if (KSI_NUM_P (x))
    return x;
  if (KSI_CHAR_P (x))
    return ksi_ulong2num (KSI_CHAR_CODE (x));
  if (x == ksi_false)
    return ksi_long2num(0);
  if (x == ksi_true)
    return ksi_long2num(1);
  return ksi_ulong2num ((unsigned long) x);
}

int
ksi_internal_format (ksi_port port, const char* fmt, int argc, ksi_obj* argv, char* nm)
{
  int nums;
  ksi_obj tmp;
  char *buf;

  for (nums = 0; *fmt; fmt++) {
    if (*fmt == '~' && *(fmt+1) != '\0') {
      fmt++;
      switch (*fmt) {
      case 'a': case 'A':
        if (nums >= argc) {
        too_many:
            ksi_exn_error (0, 0, "%s: too many ~ spec's in format string", nm);
        }
        ksi_display (argv [nums++], (ksi_obj) port);
        break;

      case 's': case 'S':
      case 'c': case 'C':
        if (nums >= argc)
          goto too_many;
        ksi_write (argv [nums++], (ksi_obj) port);
        break;

      case 'd': case 'D':
        if (nums >= argc)
          goto too_many;
        tmp = coerce2num (argv [nums++]);
        buf = ksi_num2str (tmp, 10);
        ksi_port_write ((ksi_obj) port, buf, strlen (buf));
        break;

      case 'b': case 'B':
        if (nums >= argc)
          goto too_many;
        tmp = coerce2num (argv [nums++]);
        buf = ksi_num2str (tmp, 2);
        ksi_port_write ((ksi_obj) port, buf, strlen (buf));
        break;

      case 'o': case 'O':
        if (nums >= argc)
          goto too_many;
        tmp = coerce2num (argv [nums++]);
        buf = ksi_num2str (tmp, 8);
        ksi_port_write ((ksi_obj) port, buf, strlen (buf));
        break;

      case 'x': case 'X':
        if (nums >= argc)
          goto too_many;
        tmp = coerce2num (argv [nums++]);
        buf = ksi_num2str (tmp, 16);
        ksi_port_write ((ksi_obj) port, buf, strlen (buf));
        break;

      case '?':
        if (nums+1 >= argc)
          goto too_many;
        {
          ksi_obj f = argv [nums++];
          ksi_obj x = argv [nums++];
          ksi_obj *av;
          int     ac, i;

          if (!KSI_STR_P (f))
              ksi_exn_error (0, 0, "%s: invalid format string for ~? spec: %s", nm, ksi_obj2str (f));
          if (KSI_LIST_P (x))
            x = ksi_list2vector (x);
          else if (!KSI_VEC_P (x))
              ksi_exn_error (0, 0, "%s: invalid list for ~? spec: %s", nm, ksi_obj2str (x));

          ac = KSI_VEC_LEN (x);
          av = KSI_VEC_ARR (x);
          while (ac > 0) {
            i = ksi_internal_format (port, KSI_STR_PTR (f), ac, av, nm);
            if (i == 0)
              break;
            ac -= i;
            av += i;
          }
        }
        break;

      case '%':
        ksi_port_putc (port, '\n');
        break;

      case '&':
        if (port->last_write_char != '\n')
          ksi_port_putc (port, '\n');
        break;

      case '_':
        ksi_port_putc (port, ' ');
        break;

      case '/':
        ksi_port_putc (port, '\t');
        break;

      case '|':
        ksi_port_putc (port, '\f');
        break;

      case '!':
        ksi_flush_port ((ksi_obj) port);
        break;

      default:
        ksi_port_putc (port, '~');
        ksi_port_putc (port, *fmt);
        break;
      }
    }
    else
      ksi_port_putc (port, *fmt);
  }

  return nums;
}

ksi_obj
ksi_format (ksi_obj port, const char* fmt, int argc, ksi_obj* argv)
{
  ksi_obj retval = ksi_void;

  if (port == ksi_true) {
    port = ksi_current_output_port ();
  } else if (port == ksi_false) {
    retval = ksi_make_string (0, 0);
    port = (ksi_obj) ksi_new_str_port ((ksi_string) retval);
    ((ksi_port) port) -> output = 1;
  } else {
    KSI_CHECK (port, KSI_OUTPUT_PORT_P (port), "format: invalid port in arg1");
  }

  ksi_internal_format ((ksi_port) port, fmt, argc, argv, "format");
  return retval;
}


 /* End of code */
