/*
 * ksi_read.c
 * reader
 *
 * Copyright (C) 1997-2010, ivan demakov
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        Ivan Demakov <ksion@users.sourceforge.net>
 * Creation date: Wed Aug  6 19:36:40 1997
 * Last Update:   Wed Apr 21 01:27:31 2010
 *
 */

#include "ksi_int.h"
#include "ksi_env.h"
#include "ksi_comp.h"
#include "ksi_printf.h"


#define DELIMITERS  "()[]{}\";#'`,"

typedef struct
{
    ksi_port port;
    char *tok_buf;
    unsigned tok_size, tok_len;
    unsigned case_sens : 1;
    unsigned r6rs_mode : 1;
    unsigned kw_prefix : 1;
    unsigned kw_sharp : 1;
} read_state;


static ksi_obj ksi_read_obj (read_state* st, int close_char);

static const char*
src_name (read_state *st, int tok_line, int tok_pos)
{
    if (st->port->is_fd && st->port->good_name)
        return ksi_aprintf ("%s (%d:%d)", ((struct Ksi_FdPort*) st->port) -> name, tok_line, tok_pos);

    return ksi_aprintf ("%s", ksi_obj2str ((ksi_obj) st->port));
}

static void
annotate (ksi_obj x, read_state *st, int tok_line, int tok_pos)
{
    x->o.annotation = src_name(st, tok_line, tok_pos);
}

static void
expand_tok_buf (read_state *st, int sz)
{
    char *p;

    if (sz >= st->tok_size) {
        sz = ((sz + 127) & ~127) + 128;
        p = (char*) ksi_malloc_data (sz);
        if (st->tok_len)
            memcpy (p, st->tok_buf, st->tok_len);

        st->tok_size = sz;
        st->tok_buf  = p;
    }
}

static int
skip_ws (read_state *st)
{
    for (;;) {
        int c = ksi_port_getc (st->port);
        if (c < 0)
            return c;

        if (isspace (c))
            continue;

        if (c < ' ')
            ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: illegal char #\\x%02x", c);

        return c;
    }
}

static ksi_obj
read_not_impl (read_state *st, int ch, int tok_line, int tok_pos)
{
    ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid char `%c' (syntax not implemented)", ch);
    return ksi_void;
}

static int
read_token (read_state *st, int prev_ch, int use_esc)
{
    int c, esc = 0;

    st->tok_len = 0;
    if (prev_ch > 0) {
        expand_tok_buf (st, st->tok_len);
        st->tok_buf [st->tok_len++] = prev_ch;
    }

    for (;;) {
    skip:
        c = ksi_port_getc (st->port);
        if (c < 0 || isspace(c))
            break;
        if (strchr (DELIMITERS, c)) {
            ksi_port_ungetc (st->port, c);
            break;
        }
        if (use_esc && c == '\\') {
            c = ksi_port_getc (st->port);
            if (c < 0)
                ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: unexpected eof after `\\'");

            if (c == 'x') {
                int n = 0;
                for (;;) {
                    c = ksi_port_getc (st->port);
                    if (c < 0)
                        ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: unexpected eof after `\\x'");
                    if ('0' <= c && c <= '9')
                        n = (n * 16) + (c - '0');
                    else if ('a' <= c && c <= 'f')
                        n = (n * 16) + (c - 'a' + 0x10);
                    else if ('A' <= c && c <= 'F')
                        n = (n * 16) + (c - 'A' + 0x10);
                    else if (c == ';' || isspace(c))
                        break;
                    else
                        ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos),  "read: invalid sequence after `\\x'");
                }
                if (st->tok_len == 0)
                    esc = 1;

                c = n;
            } else {
                if (st->r6rs_mode || c < ' ')
                    ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: invalid sequence `\\%c'", c);
            }
        } else if (use_esc && c == '|') {
            if (st->r6rs_mode)
                ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: illegal char `|'");

            if (st->tok_len == 0)
                esc = 1;

            for (;;) {
                c = ksi_port_getc (st->port);
                if (c < 0)
                    ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: unexpected eof after `|'");
                if (c == '|')
                    goto skip;

                expand_tok_buf (st, st->tok_len);
                st->tok_buf[st->tok_len++] = c;
            }
        } else {
            if (c < ' ')
                ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: illegal char #\\x%02x", c);

            if (!st->case_sens)
                c = tolower (c);
        }

        expand_tok_buf (st, st->tok_len);
        st->tok_buf[st->tok_len++] = c;
    }

    expand_tok_buf (st, st->tok_len);
    st->tok_buf [st->tok_len] = 0;

    return esc;
}

static ksi_obj
read_ch (read_state *st, int tok_line, int tok_pos)
{
    ksi_obj x;
    int c;

    st->tok_len = 0;

    c = ksi_port_getc (st->port);
    if (c < 0)
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a `#\' syntax");
    if (c == '\n')
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected linefeed while reading a `#\' syntax");
    if (c < ' ')
        ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: illegal char #\\x%02x", c);

    expand_tok_buf (st, st->tok_len);
    st->tok_buf[st->tok_len++] = c;

    if (!isspace (c) && strchr (DELIMITERS, c) == 0) {
        for (;;) {
            c = ksi_port_getc (st->port);
            if (c < 0 || isspace (c))
                break;
            if (strchr (DELIMITERS, c)) {
                ksi_port_ungetc (st->port, c);
                break;
            }
            if (c < ' ')
                ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: illegal char #\\x%02x", c);

            expand_tok_buf (st, st->tok_len);
            st->tok_buf[st->tok_len++] = c;
        }
    }

    expand_tok_buf (st, st->tok_len);
    st->tok_buf [st->tok_len] = 0;

    if (st->tok_len == 1)
        return ksi_int2char (st->tok_buf[0]);

    x = ksi_str2char (st->tok_buf, st->tok_len);
    if (x != ksi_false)
        return x;

    ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid character name `#\\%s'", st->tok_buf);
    return 0;
}

static char*
read_c_string (read_state *st, int end_char, char* tok_name, int tok_line, int tok_pos)
{
    st->tok_len = 0;
    for (;;) {
        int n, c = ksi_port_getc (st->port);
        if (c < 0) {
        eof_err:
            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a %s", tok_name);
        }

    next_char:
        if (c == end_char)
            break;

        if (c == '\\') {
            if ((c = ksi_port_getc (st->port)) < 0)
                goto eof_err;

            switch (c) {
            case 'a' : c = '\a'; break; /* alarm  */
            case 'b' : c = '\b'; break; /* bs   */
            case 'e' : c = 0x1b; break; /* esc  */
            case 'n' : c = '\n'; break; /* lf   */
            case 'r' : c = '\r'; break; /* cr   */
            case 't' : c = '\t'; break; /* tab  */
            case 'f' : c = '\f'; break; /* ff   */
            case 'v' : c = '\v'; break; /* vtab */
            case '\\': c = '\\'; break;
            case '\"': c = '\"'; break;

            case 'x' :
                n = 0;
                for (;;) {
                    c = ksi_port_getc (st->port);
                    if (c < 0)
                        goto eof_err;
                    if ('0' <= c && c <= '9')
                        n = (n * 16) + (c - '0');
                    else if ('a' <= c && c <= 'f')
                        n = (n * 16) + (c - 'a' + 0x10);
                    else if ('A' <= c && c <= 'F')
                        n = (n * 16) + (c - 'A' + 0x10);
                    else if (c == ';')
                        break;
                    else
                        ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: invalid sequence after `\\x' while reading a %s", tok_name);
                }
                c = n;
                break;

            default:
                if (c < ' ')
                    ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: illegal char #\\x%02x", c);

                if (isspace (c)) {
                    while (c != '\n') {
                        if ((c = ksi_port_getc (st->port)) < 0)
                            goto eof_err;
                        if (!isspace (c))
                            goto esc_err;
                    }
                    while (isspace (c)) {
                        if ((c = ksi_port_getc (st->port)) < 0)
                            goto eof_err;
                    }
                    goto next_char;
                }
            esc_err:
                ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: invalid sequence `\\%c' while reading a %s", c, tok_name);
                break;
            } /* switch */
        } /* if (c == '\\') */

        expand_tok_buf (st, st->tok_len);
        st->tok_buf [st->tok_len++] = c;
    } /* for */

    expand_tok_buf (st, st->tok_len);
    st->tok_buf [st->tok_len] = '\0';
    return st->tok_buf;
}

static ksi_obj
read_list (read_state *st, int close_char, int improper_list, char* tok_name, int tok_line, int tok_pos)
{
    ksi_obj  list = ksi_nil, tmp;
    ksi_obj* last = &list;
    int cur_line, cur_pos;

    for (;;) {
        int c = skip_ws (st);
        if (c < 0) {
        eof_err:
            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a %s", tok_name);
        }
        cur_line = st->port->read_line;
        cur_pos = st->port->read_pos;

        if (c == close_char)
            return list;

        if (c == '.') {
            int c1 = ksi_port_getc (st->port);
            if (c1 < 0)
                goto eof_err;

            if (isspace (c1) || strchr (DELIMITERS, c1)) {
                ksi_port_ungetc (st->port, c1);

                if (!improper_list)
                    ksi_src_error (src_name (st, st->port->read_line+1, st->port->read_pos), "read: unexpected '.' while reading a %s", tok_name);

                *last = ksi_read_obj (st, 0);
                if (*last == ksi_eof)
                    goto eof_err;

                c = skip_ws (st);
                if (c != close_char)
                    ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: missing `%c' while reading %s", close_char, tok_name);

                return list;
            }

            ksi_port_ungetc (st->port, c1);
        }

        ksi_port_ungetc (st->port, c);
        tmp = ksi_read_obj (st, close_char);
        if (!tmp)
            continue;
        if (tmp == ksi_eof)
            goto eof_err;

        *last = ksi_cons(tmp, ksi_nil);
        annotate (*last, st, cur_line, cur_pos);

        last = & KSI_CDR (*last);
    }

    ksi_exn_error (0, 0, "read: internal error");
    return ksi_void;
}

static ksi_obj
read_vector (read_state *st, int close_char, int tok_line, int tok_pos)
{
    ksi_obj x = read_list (st, close_char, 0, "vector", tok_line, tok_pos);
    ksi_obj v = ksi_list2vector (x);

    return v;
}

static ksi_obj
read_sharp (read_state *st, int tok_line, int tok_pos)
{
    int c1, c2, depth;
    ksi_obj x;

    c1 = ksi_port_getc (st->port);
    if (c1 < 0)
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading `#' syntax");
    if (isspace(c1))
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected space while reading `#' syntax");

    switch (c1) {
    case '!':
        if (st->port->read_line == 0 && st->port->read_pos == 2) {
            c1 = ksi_port_getc (st->port);
            if (!isspace(c1) && c1 != '/') {
                ksi_port_ungetc (st->port, c1);
            } else {
                /* ignore first line of a script */
                for (;;) {
                    c1 = ksi_port_getc (st->port);
                    if (c1 < 0)
                        return ksi_eof;
                    if (c1 == '\n')
                        return 0;
                    if (c1 == '\\') {
                        for (;;) {
                            c2 = ksi_port_getc (st->port);
                            if (c2 < 0)
                                return ksi_eof;
                            if (c2 == '\n' || !isspace(c2))
                                break;
                        }
                        if (c2 != '\n')
                            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected char while reading `#!' script header");
                    }
                }
            }
        }
        read_token(st, 0, 0);
        if (strcmp (st->tok_buf, "r6rs") == 0) {
            st->r6rs_mode = 1;
            return 0;
        }
        if (strcmp (st->tok_buf, "ksi") == 0) {
            st->r6rs_mode = 0;
            return 0;
        }
        if (!st->r6rs_mode) {
            if (strcmp (st->tok_buf, "fold-case") == 0) {
                st->case_sens = 0;
                return 0;
            }
            if (strcmp (st->tok_buf, "no-fold-case") == 0) {
                st->case_sens = 1;
                return 0;
            }
            if (strcmp (st->tok_buf, "keyword-prefix") == 0) {
                st->kw_prefix = 1;
                return 0;
            }
            if (strcmp (st->tok_buf, "keyword-postfix") == 0) {
                st->kw_prefix = 0;
                return 0;
            }
            if (strcmp (st->tok_buf, "keyword-sharp") == 0) {
                st->kw_sharp = 1;
                return 0;
            }
            if (strcmp (st->tok_buf, "no-keyword-sharp") == 0) {
                st->kw_sharp = 0;
                return 0;
            }
            if (strcmp (st->tok_buf, "eof") == 0) {
                return ksi_eof;
            }
            if (strcmp (st->tok_buf, "void") == 0) {
                return ksi_void;
            }
        }
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid `#!%s' syntax", st->tok_buf);
        break;

    case ';':
        /* skip datum comment, i.e.  #; <datum> */
        ksi_read_obj (st, 0);
        return 0;

    case '|':
        /* skip balanced comment, i.e.  #| ... |# */
        depth = 1;
        c1 = ' ';
        for (;;) {
            c2 = ksi_port_getc (st->port);
            if (c2 < 0)
                ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading `#|' syntax");

            if (c1 == '#' && c2 == '|') {
                ++depth;
                c1 = ' ';
            } else if (c1 == '|' && c2 == '#') {
                if (--depth == 0)
                    return 0;
                c1 = ' ';
            } else {
                c1 = c2;
            }
        }
        break;

    case 'f': case 'F':
        c2 = ksi_port_getc (st->port);
        if (c2 < 0 || isspace (c2))
            return ksi_false;
        if (strchr (DELIMITERS, c2)) {
            ksi_port_ungetc (st->port, c2);
            return ksi_false;
        }
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid `#%c%c' syntax", c1, c2);

    case 't': case 'T':
        c2 = ksi_port_getc (st->port);
        if (c2 < 0 || isspace (c2))
            return ksi_true;
        if (strchr (DELIMITERS, c2)) {
            ksi_port_ungetc (st->port, c2);
            return ksi_true;
        }
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid `#%c%c' syntax", c1, c2);

    case '\\':
        return read_ch (st, tok_line, tok_pos);

    case ':':
        if (st->kw_sharp) {
            read_token(st, 0, 1);
            return ksi_str2key (st->tok_buf, st->tok_len);
        }
        break;

    case 'b': case 'B': case 'o': case 'O':
    case 'd': case 'D': case 'x': case 'X':
    case 'i': case 'I': case 'e': case 'E':
        ksi_port_ungetc (st->port, c1);
        read_token (st, '#', 0);
        x = ksi_str02num (st->tok_buf, 0);
        if (x == ksi_false)
            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid number syntax `%s'", st->tok_buf);
        return x;

    case '(':
        x = read_vector (st, ')', tok_line, tok_pos);
        return x;

    case '[':
        x = read_vector (st, ']', tok_line, tok_pos);
        return x;

    case '\'':
        x = ksi_read_obj (st, 0);
        if (x == ksi_eof)
            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a syntax");
        x = KSI_LIST2 (ksi_data->sym_syntax, x);
        annotate (x, st, tok_line, tok_pos);
        return x;

    case '`':
        x = ksi_read_obj (st, 0);
        if (x == ksi_eof)
            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a quasisyntax");
        x = KSI_LIST2 (ksi_data->sym_quasisyntax, x);
        annotate (x, st, tok_line, tok_pos);
        return x;

    case ',':
        c2 = ksi_port_getc (st->port);
        if (c2 < 0)
            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a unsyntax");

        if (c2 == '@') {
            x = ksi_read_obj (st, 0);
            if (x == ksi_eof)
                ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a #,@");

            x = KSI_LIST2 (ksi_data->sym_unsyntax_splicing, x);
        } else {
            ksi_port_ungetc (st->port, c2);
            x = ksi_read_obj (st, 0);
            if (x == ksi_eof)
                ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a unsyntax");

            x = KSI_LIST2 (ksi_data->sym_unsyntax, x);
        }
        annotate (x, st, tok_line, tok_pos);
        return x;

    case '@':
        read_token(st, 0, 0);
        if (strcmp (st->tok_buf, "cons") == 0)
            return ksi_data->cons_proc;
        if (strcmp (st->tok_buf, "car") == 0)
            return ksi_data->car_proc;
        if (strcmp (st->tok_buf, "cdr") == 0)
            return ksi_data->cdr_proc;
        if (strcmp (st->tok_buf, "not") == 0)
            return ksi_data->not_proc;
        if (strcmp (st->tok_buf, "eq?") == 0)
            return ksi_data->eq_proc;
        if (strcmp (st->tok_buf, "eqv?") == 0)
            return ksi_data->eqv_proc;
        if (strcmp (st->tok_buf, "equal?") == 0)
            return ksi_data->equal_proc;
        if (strcmp (st->tok_buf, "memq") == 0)
            return ksi_data->memq_proc;
        if (strcmp (st->tok_buf, "memv") == 0)
            return ksi_data->memv_proc;
        if (strcmp (st->tok_buf, "member") == 0)
            return ksi_data->member_proc;
        if (strcmp (st->tok_buf, "null?") == 0)
            return ksi_data->nullp_proc;
        if (strcmp (st->tok_buf, "pair?") == 0)
            return ksi_data->pairp_proc;
        if (strcmp (st->tok_buf, "list?") == 0)
            return ksi_data->listp_proc;
        if (strcmp (st->tok_buf, "vector?") == 0)
            return ksi_data->vectorp_proc;
        if (strcmp (st->tok_buf, "list") == 0)
            return ksi_data->list_proc;
        if (strcmp (st->tok_buf, "apply") == 0)
            return ksi_data->apply_proc;
        if (strcmp (st->tok_buf, "call/cc") == 0)
            return ksi_data->call_cc_proc;
        if (strcmp (st->tok_buf, "call/vs") == 0)
            return ksi_data->call_vs_proc;
        break;
    }

    ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid `#%c' syntax", c1);
    return ksi_void;
}

static ksi_obj
ksi_read_obj (read_state *st, int close_char)
{
  int c, esc, tok_line, tok_pos;

again:
  c = skip_ws(st);
  tok_line = st->port->read_line;
  tok_pos = st->port->read_pos;

//  ksi_debug ("read char=`%c' pos=%d:%d>", c, tok_line+1, tok_pos);

  if (c >= 0) {
    ksi_obj res = ksi_void;
    switch (c) {
    case '{': case '}':
      return read_not_impl (st, c, tok_line, tok_pos);

    case';': /* line comment */
      for (;;) {
        c = ksi_port_getc (st->port);
        if (c < 0)
          return ksi_eof;
        if (c == '\n')
          goto again;
      }
      break;

    case '#':
      res = read_sharp (st, tok_line, tok_pos);
      if (!res)
        goto again;
      break;

    case '"':
      read_c_string (st, '"', "string", tok_line, tok_pos);
      res = ksi_str2string (st->tok_buf, st->tok_len);
      break;

    case '(':
      res = read_list (st, ')', 1, "list", tok_line, tok_pos);
      if (KSI_PAIR_P(res))
          annotate (res, st, tok_line, tok_pos);
      break;

    case '[':
      res = read_list (st, ']', 1, "list", tok_line, tok_pos);
      if (KSI_PAIR_P(res))
          annotate (res, st, tok_line, tok_pos);
      break;

    case ')':
    case ']':
        if (c != close_char)
            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected `%c'", c);
        ksi_port_ungetc (st->port, c);
        return 0;

    case '.':
      c = ksi_port_getc (st->port);
      if (c < 0 || isspace (c) || strchr (DELIMITERS, c))
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected `.'");

      ksi_port_ungetc (st->port, c);
      ksi_port_ungetc (st->port, '.');
      goto token;

    case '\'':
        res = ksi_read_obj (st, 0);
      if (res == ksi_eof)
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a quote");

      res = KSI_LIST2 (ksi_data->sym_quote, res);
      annotate (res, st, tok_line, tok_pos);
      break;

    case '`':
        res = ksi_read_obj (st, 0);
      if (res == ksi_eof)
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a quasiquote");

      res = KSI_LIST2 (ksi_data->sym_quasiquote, res);
      annotate (res, st, tok_line, tok_pos);
      break;

    case ',':
      c = ksi_port_getc (st->port);
      if (c < 0)
        ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a unquote");

      if (c == '@') {
          res = ksi_read_obj (st, 0);
        if (res == ksi_eof)
          ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a ,@");

        res = KSI_LIST2 (ksi_data->sym_unquote_splicing, res);
      } else {
        ksi_port_ungetc (st->port, c);
        res = ksi_read_obj (st, 0);
        if (res == ksi_eof)
          ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: unexpected eof while reading a unquote");

        res = KSI_LIST2 (ksi_data->sym_unquote, res);
      }
      annotate (res, st, tok_line, tok_pos);
      break;

    case '|':
    default:
      ksi_port_ungetc (st->port, c);
    token:
      esc = read_token (st, 0, 1);

      if (esc || st->tok_len == 0)
        return ksi_str2sym (st->tok_buf, st->tok_len);

      if (st->r6rs_mode) {
        if (st->tok_len == 1) {
          if (st->tok_buf[0] == '+')
            return ksi_data->sym_plus;
          else if (st->tok_buf[0] == '-')
            return ksi_data->sym_minus;
        } else if (st->tok_len >= 2) {
          if (st->tok_buf[0] == '-' && st->tok_buf[1] == '>')
            return ksi_str2sym (st->tok_buf, st->tok_len);
          else if (st->tok_len == 2 && st->tok_buf[0] == '=' && st->tok_buf[1] == '>')
            return ksi_data->sym_arrow;
          else if (st->tok_len == 3 && st->tok_buf[0] == '.' && st->tok_buf[1] == '.' && st->tok_buf[2] == '.')
            return ksi_data->sym_dots;
        }
        if (st->tok_buf[0] == '-' || st->tok_buf[0] == '+' || st->tok_buf[0] == '.' || isdigit(st->tok_buf[0])) {
          res = ksi_str02num (st->tok_buf, 0);
          if (res == ksi_false)
            ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid number syntax `%s'", st->tok_buf);
        } else {
          res = ksi_str2sym (st->tok_buf, st->tok_len);
        }
      } else {
        if (st->kw_prefix && st->tok_buf[0] == ':') {
          res = ksi_str2key (st->tok_buf+1, st->tok_len-1);
        } else if (!st->kw_prefix && st->tok_buf[st->tok_len-1] == ':') {
          res = ksi_str2key (st->tok_buf, st->tok_len-1);
        } else {
          char *p = st->tok_buf;
          if (isdigit(p[0])
              || ((p[0] == '-' || p[0] == '+' || p[0] == '.') && isdigit(p[1]))
              || ((p[0] == 'n' || p[0] == 'N') && (p[1] == 'a' || p[1] == 'A') && (p[2] == 'n' || p[2] == 'N') && (p[3] == '.'))
              || ((p[0] == '-' || st->tok_buf[0] == '+') && (p[1] == 'n' || p[1] == 'N') && (p[2] == 'a' || p[2] == 'A') && (p[3] == 'n' || p[3] == 'N') && (p[4] == '.'))
              || ((p[0] == '-' || st->tok_buf[0] == '+') && (p[1] == 'i' || p[1] == 'I') && (p[2] == 'n' || p[2] == 'N') && (p[3] == 'f' || p[3] == 'F') && (p[4] == '.'))
            ) {
            res = ksi_str02num (st->tok_buf, 0);
            if (res == ksi_false)
              ksi_src_error (src_name (st, tok_line+1, tok_pos), "read: invalid number syntax `%s'", st->tok_buf);
          } else {
            res = ksi_str2sym (st->tok_buf, st->tok_len);
          }
        }
      }
      break;
    }

//    ksi_debug ("read res=%s at %d:%d", ksi_obj2str(res), tok_line+1, tok_pos);
    return res;
  }

  return ksi_eof;
}

static ksi_obj
ksi_read_inner (ksi_port p)
{
  read_state st;
  char buf[128];
  ksi_obj res;

  st.port = p;
  st.tok_buf = buf;
  st.tok_size = sizeof buf;
  st.tok_len = 0;
  st.case_sens = 1;
  st.r6rs_mode = 0;
  st.kw_prefix = 0;
  st.kw_sharp = 0;

//  ksi_debug (">>> %s %d:%d", port_name (p), p->read_line+1, p->read_pos);
  res = ksi_read_obj (&st, 0);
//  ksi_debug ("<<< %s %d:%d\n    res=%s", port_name (p), p->read_line+1, p->read_pos, ksi_obj2str(res));

  return res;
}

ksi_obj
ksi_read (ksi_obj p)
{
  if (!p) p = ksi_current_input_port();
  KSI_CHECK(p, KSI_INPUT_PORT_P(p), "read: invalid port");

  return ksi_read_inner((ksi_port) p);
}

ksi_obj
ksi_str2obj (const char *ptr, int len)
{
  ksi_obj port, str;

  if (ptr == 0 || len == 0)
    return ksi_void;

  str = ksi_str2string (ptr, len);
  port = (ksi_obj) ksi_new_str_port ((ksi_string) str);
  ((ksi_port) port) -> input = 1;

  return ksi_read (port);
}

ksi_obj
ksi_str02obj (const char *ptr)
{
  if (ptr == 0)
    return ksi_void;

  return ksi_str2obj (ptr, strlen (ptr));
}

ksi_obj
ksi_scm_annotation_p (ksi_obj x)
{
  if (x->o.annotation)
    return ksi_true;
  return ksi_false;
}

ksi_obj
ksi_scm_annotation_source (ksi_obj x)
{
  if (x->o.annotation)
    return ksi_str02string (x->o.annotation);
  return ksi_false;
}

ksi_obj
ksi_scm_annotation_expression (ksi_obj x)
{
  return x;
}


 /* End of code */
