/*
 * ksi_int.c
 *
 * Copyright (C) 2009-2010, ivan demakov.
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        ivan demakov <ksion@users.sourceforge.net>
 * Creation date: Wed Feb 18 14:35:06 2009
 * Last Update:   Sat Apr 24 22:49:54 2010
 *
 */

#include "ksi_type.h"
#include "ksi_int.h"
#include "ksi_env.h"
#include "ksi_hash.h"
#include "ksi_printf.h"
#include "ksi_util.h"

#include <time.h>

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#endif

#ifdef _MSC_VER
#  include <io.h>
#endif


struct Ksi_Interp *ksi_int_data;

const char *ksi_wna_s = "%s: missing or extra args";
const char *ksi_syntax_s = "syntax";
const char *ksi_assertion_s = "assertion";



struct Errlog_Desc
{
  char *str;
  char *msg;
};

static struct Errlog_Desc errlog[] =
{
    { "emerg",   "EMERG   " },
    { "alert",   "ALERT   " },
    { "error",   "ERROR   " },
    { "warning", "WARNING " },
    { "notice",  "NOTICE  " },
    { "info",    "INFO    " },
    { "debug",   "DEBUG   " }
};


static char*
ksi_errlog_hdr (int pri)
{
    struct tm tm;
    int size = 128;
    char *buf = alloca(size);

    ksi_localtime(time(0), &tm);

    while (strftime (buf, size, "%Y-%m-%d %H:%M:%S %Z", &tm) >= size) {
        size *= 2;
        buf = alloca(size);
    }
    return ksi_aprintf ("%s %s: ", buf, errlog[pri].str);
}

void
ksi_errlog_msg (int pri, const char *msg)
{
    ksi_port log, err;

    if (ksi_int_data && ksi_int_data->errlog_proc) {
        ksi_int_data->errlog_proc(pri, msg);
    } else {
        log = ksi_int_data ? ksi_int_data->errlog_port : 0;
        if (log && KSI_OUTPUT_PORT_P(log)) {
            char *hdr = ksi_errlog_hdr (pri);
            ksi_port_write ((ksi_obj) log, hdr, strlen (hdr));
            ksi_port_write ((ksi_obj) log, msg, strlen (msg));
            ksi_port_putc  (log, '\n');
            ksi_flush_port ((ksi_obj) log);
        }

        err = ksi_int_data ? ksi_int_data->error_port : 0;
        if (err && KSI_OUTPUT_PORT_P(err)) {
            if (log != err && (log == 0 || pri < ERRLOG_WARNING)) {
                ksi_port_putc  (err, ';');
                ksi_port_putc  (err, ' ');
                ksi_port_write ((ksi_obj) err, errlog[pri].msg, strlen (errlog[pri].msg));
                ksi_port_write ((ksi_obj) err, msg, strlen (msg));
                ksi_port_putc  (err, '\n');
                ksi_flush_port ((ksi_obj) err);
            }
        } else {
#ifdef WIN32
            MessageBox (NULL, msg, errlog[pri].msg, MB_TASKMODAL | MB_ICONEXCLAMATION | MB_OK);
#else
            write (2, errlog[pri].msg, strlen (errlog[pri].msg));
            write (2, msg, strlen (msg));
            write (2, "\n", 1);
#endif
        }
    }
}

ksi_obj
ksi_open_errlog (ksi_obj dest)
{
    ksi_port port = 0;

    if (KSI_OUTPUT_PORT_P (dest)) {
        port = (ksi_port) dest;
    } else if (KSI_STR_P (dest)) {
        port = ksi_open_fd_port_int (KSI_STR_PTR (dest), "a", "open-errlog");
    } else {
        ksi_exn_error (ksi_assertion_s, dest, "open-errlog: invalid destination in arg2");
    }

    if (ksi_int_data->errlog_port) {
        ksi_close_port ((ksi_obj) ksi_int_data->errlog_port);
        ksi_int_data->errlog_port = 0;
    }

    ksi_int_data->errlog_port = port;

    return ksi_void;
}

ksi_obj
ksi_errlog_priority (ksi_obj priority, ksi_obj module)
{
    int pri;

    KSI_CHECK (priority, KSI_EINT_P (priority), "errlog-priority: invalid integer in arg2");
    pri = ksi_num2long (priority, "errlog-priority");
    KSI_CHECK (priority, ERRLOG_EMERG <= pri && pri <= ERRLOG_ALL, "errlog-priority: priority out of range");

    if (!module || module == ksi_false) {
        ksi_int_data->errlog_priority = pri;
    } else {
        ksi_int_data->module_priority = ksi_assq_set_x (ksi_int_data->module_priority, module, priority);
    }

    return ksi_void;
}

static inline ksi_obj
mk_exn (ksi_obj type, ksi_obj errobj, ksi_obj msg, ksi_obj src)
{
    ksi_vector exn = ksi_alloc_vector (4, KSI_TAG_EXN);
    KSI_EXN_TYPE (exn) = type;
    KSI_EXN_MSG  (exn) = msg;
    KSI_EXN_VAL  (exn) = errobj;
    KSI_EXN_SRC  (exn) = src;
    return (ksi_obj) exn;
}

ksi_obj
ksi_make_exn (const char *type, ksi_obj errobj, const char *msg, const char *src)
{
    return mk_exn (ksi_str02sym (type ? type : "misc"),
                   errobj ? errobj : ksi_void,
                   ksi_str02string (msg ? msg : "unspecified error"),
                   src ? ksi_str02string(src) : ksi_void);
}

static ksi_obj
ksi_scm_make_exn (ksi_obj type, ksi_obj errobj, ksi_obj msg, ksi_obj src)
{
    KSI_CHECK (type, KSI_SYM_P (type), "make-exn: invalid symbol in arg1");
    KSI_CHECK (msg,  KSI_STR_P (msg),  "make-exn: invalid string in arg3");
    KSI_CHECK (src,  KSI_STR_P (msg),  "make-exn: invalid string in arg4");

    return ksi_make_exn (KSI_SYM_PTR(type), errobj, KSI_STR_PTR(msg), KSI_STR_PTR (src));
}

ksi_obj
ksi_exn_p (ksi_obj x)
{
    return KSI_EXN_P (x) ? ksi_true : ksi_false;
}

ksi_obj
ksi_exn_type (ksi_obj x)
{
    KSI_CHECK (x, KSI_EXN_P (x), "exn-type: invalid exception in arg1");
    return KSI_EXN_TYPE (x);
}

ksi_obj
ksi_exn_message (ksi_obj x)
{
    KSI_CHECK (x, KSI_EXN_P (x), "exn-message: invalid exception in arg1");
    return KSI_EXN_MSG (x);
}

ksi_obj
ksi_exn_value (ksi_obj x)
{
    KSI_CHECK (x, KSI_EXN_P (x), "exn-value: invalid exception in arg1");
    return KSI_EXN_VAL (x);
}

ksi_obj
ksi_exn_source (ksi_obj x)
{
    KSI_CHECK (x, KSI_EXN_P (x), "exn-source: invalid exception in arg1");
    return KSI_EXN_SRC (x);
}

int
ksi_exn_error (const char *type, ksi_obj errobj, const char *fmt, ...)
{
    va_list args;
    char *msg;

    va_start (args, fmt);
    msg = ksi_avprintf (fmt, args);
    va_end (args);

    ksi_throw_error (mk_exn (ksi_str02sym (type ? type : ksi_assertion_s),
                             errobj ? errobj : ksi_void,
                             ksi_str02string(msg),
                             errobj && errobj->o.annotation ? ksi_str02string(errobj->o.annotation) : ksi_void));

    return 0;
}

int
ksi_src_error (const char *src, const char *fmt, ...)
{
    va_list args;
    char *msg;

    va_start (args, fmt);
    msg = ksi_avprintf (fmt, args);
    va_end (args);

    ksi_throw_error (mk_exn (ksi_str02sym ("lexical"),
                             ksi_void,
                             ksi_str02string(msg),
                             ksi_str02string(src)));

    return 0;
}

int
ksi_debug (const char* msg, ...)
{
    char *str;
    va_list args;

    if (!ksi_int_data || ERRLOG_DEBUG > ksi_int_data->errlog_priority)
        return 0;

    va_start (args, msg);
    str = ksi_avprintf(msg, args);
    va_end (args);

    ksi_errlog_msg (ERRLOG_DEBUG, str);
    return 0;
}

int
ksi_warn (const char* msg, ...)
{
    char *str;
    va_list args;

    if (!ksi_int_data || ERRLOG_WARNING > ksi_int_data->errlog_priority)
        return 0;

    va_start (args, msg);
    str = ksi_avprintf(msg, args);
    va_end (args);

    ksi_errlog_msg (ERRLOG_WARNING, str);
    return 0;
}

ksi_obj
ksi_errlog (ksi_obj module, int pri, const char *fmt, ...)
{
    va_list args;
    int log_pri;
    ksi_obj x;
    char *str;

    if (pri > ERRLOG_ALL)
        pri = ERRLOG_ALL;
    else if (pri < ERRLOG_EMERG)
        pri = ERRLOG_EMERG;

    if (ksi_int_data) {
        x = (module ? ksi_assq_ref (ksi_int_data->module_priority, module) : ksi_false);
        if (x == ksi_false)
            log_pri = ksi_int_data->errlog_priority;
        else
            log_pri = ksi_num2long (x, "errlog");
    } else {
        log_pri = ERRLOG_ALL;
    }

    if (pri <= log_pri) {
        va_start (args, fmt);
        str = ksi_avprintf (fmt, args);
        va_end (args);

        ksi_errlog_msg (pri, str);
    }
    return ksi_void;
}

int
ksi_handle_error (ksi_obj tag, ksi_obj exn)
{
    ksi_obj x;

    if (tag != ksi_err)
        exn = ksi_make_exn ("misc", tag, "uncatched thrown object", 0);
    else if (!KSI_EXN_P (exn))
        exn = ksi_make_exn ("misc", exn, "unspecified error", 0);

    if (ksi_int_data && ksi_int_data->error_handlers != ksi_nil) {
        for (x = ksi_int_data->error_handlers; x != ksi_nil; x = KSI_CDR (x))
            ksi_apply_1_with_catch (KSI_CAR (x), exn);
    } else {
        const char *msg;
        if (KSI_EXN_VAL (exn) != ksi_void) {
            if (KSI_EXN_SRC (exn) != ksi_void) {
                msg = ksi_aprintf("%s (errval: %s) at %s", KSI_STR_PTR (KSI_EXN_MSG (exn)), ksi_obj2str(KSI_EXN_VAL (exn)), ksi_obj2name(KSI_EXN_SRC (exn)));
            } else {
                msg = ksi_aprintf("%s (errval: %s)", KSI_STR_PTR (KSI_EXN_MSG (exn)), ksi_obj2str(KSI_EXN_VAL (exn)));
            }
        } else {
            if (KSI_EXN_SRC (exn) != ksi_void) {
                msg = ksi_aprintf("%s at %s", KSI_STR_PTR (KSI_EXN_MSG (exn)), ksi_obj2name(KSI_EXN_SRC (exn)));
            } else {
                msg = KSI_STR_PTR (KSI_EXN_MSG (exn));
            }
        }
        ksi_errlog_msg (ERRLOG_ERROR, msg);
    }

    return 0;
}

ksi_obj
ksi_add_exit_handler (ksi_obj proc)
{
    KSI_ASSERT (ksi_int_data);
    KSI_CHECK (proc, KSI_PROC_P (proc), "add-exit-handler: invalid procedure");

    ksi_int_data->exit_handlers = ksi_cons(proc, ksi_int_data->exit_handlers);
    return ksi_void;
}

ksi_obj
ksi_add_error_handler (ksi_obj proc)
{
    KSI_ASSERT (ksi_int_data);
    KSI_CHECK (proc, KSI_PROC_P (proc), "add-error-handler: invalid procedure");

    ksi_int_data->error_handlers = ksi_cons(proc, ksi_int_data->error_handlers);
    return ksi_void;
}

ksi_obj
ksi_remove_error_handler (ksi_obj proc)
{
    ksi_obj *loc;

    KSI_ASSERT (ksi_int_data);

    loc = &ksi_int_data->error_handlers;
    while (*loc != ksi_nil) {
        if (KSI_CAR (*loc) == proc) {
            *loc = KSI_CDR (*loc);
            break;
        }
        loc = & KSI_CDR (*loc);
    }

    return ksi_void;
}

const char*
ksi_mk_filename (ksi_obj x, char* name)
{
again:
    if (x == ksi_nil)
        return "";
    if (KSI_STR_P(x))
        return KSI_STR_PTR(x);
    if (KSI_SYM_P(x))
        return KSI_SYM_PTR(x);
    if (KSI_EINT_P(x))
        return ksi_num2str(x, 10);

    if (KSI_PAIR_P(x)) {
        if (KSI_CDR(x) == ksi_nil) {
            x = KSI_CAR(x);
            goto again;
        } else {
            return ksi_aprintf("%s%s%s", ksi_mk_filename(KSI_CAR(x), name), DIR_SEP, ksi_mk_filename(KSI_CDR(x), name));
        }
    }

    ksi_exn_error(ksi_assertion_s, x, "%s: invalid file-name", name);
    return 0;
}

void
ksi_init_std_ports (int in, int out, int err)
{
    ksi_port port;
    ksi_env env = ksi_get_lib_env("ksi", "io", "core", 0);

    if (!ksi_int_data) {
        ksi_errlog_msg (ERRLOG_EMERG, "Ksi library is not initialized. (You should call ksi_init() before any other ksi function)\n");
        ksi_quit();
    }

    if (in >= 0) {
        port = (ksi_port) ksi_make_fd_port (in, 0, "r");
        ksi_int_data->input_port = port;
        ksi_defsym ("stdin", (ksi_obj) ksi_int_data->input_port, env);
    }

    if (out >= 0) {
        port = (ksi_port) ksi_make_fd_port (out, 0, "w");
        ksi_int_data->output_port = port;
        ksi_defsym ("stdout", (ksi_obj) ksi_int_data->output_port, env);
    }

    if (err >= 0) {
        port = (ksi_port) ksi_make_fd_port (err, 0, "w");
        ksi_int_data->error_port = port;
        ksi_defsym ("stderr", (ksi_obj) ksi_int_data->error_port, env);
    }
}

static ksi_obj
ksi_copyright ()
{
    static char str [] = "Copyright (C) 1997-2010, Ivan Demakov <ksion@users.sourceforge.net>";
    return ksi_str02string (str);
}

static ksi_obj
ksi_banner ()
{
    static char str [] = "Welcome to the KSi Scheme.";
    return ksi_str02string (str);
}

static ksi_obj
ksi_scm_version ()
{
    return ksi_str02string (ksi_version());
}

static ksi_obj
ksi_scm_major_version ()
{
    return ksi_long2num (ksi_major_version());
}

static ksi_obj
ksi_scm_minor_version ()
{
    return ksi_long2num (ksi_minor_version());
}

static ksi_obj
ksi_scm_patch_level ()
{
    return ksi_long2num (ksi_patch_level());
}

static ksi_obj
ksi_scm_cpu ()
{
    return ksi_str02sym (ksi_cpu());
}

static ksi_obj
ksi_scm_os ()
{
    return ksi_str02sym (ksi_os());
}

static ksi_obj
ksi_scm_host ()
{
    return ksi_str02sym (ksi_host());
}

static ksi_obj
inst_include_dir ()
{
    return ksi_str02string (ksi_instal_include_dir());
}

static ksi_obj
inst_lib_dir ()
{
    return ksi_str02string (ksi_instal_lib_dir());
}

static ksi_obj
inst_bin_dir ()
{
    return ksi_str02string (ksi_instal_bin_dir());
}

static ksi_obj
inst_info_dir ()
{
    return ksi_str02string (ksi_instal_info_dir());
}

static ksi_obj
inst_man_dir ()
{
    return ksi_str02string (ksi_instal_man_dir());
}

static ksi_obj
build_cflags ()
{
    return ksi_str02string (ksi_build_cflags());
}

static ksi_obj
build_libs ()
{
    return ksi_str02string (ksi_build_libs());
}

static ksi_obj
lib_path ()
{
    return ksi_str02string (ksi_scheme_lib_dir());
}

static ksi_obj
gc_expand_heap (ksi_obj x)
{
    KSI_CHECK (x, KSI_EINT_P (x), "gc-expand: invalid integer");
    return ksi_expand_heap (ksi_num2ulong (x, "ksi:gc-expand")) ? ksi_true : ksi_false;
}

static ksi_obj
gc_set_heap_size (ksi_obj x)
{
    KSI_CHECK (x, KSI_EINT_P (x), "gc-set-size!: invalid integer");
    ksi_set_max_heap (ksi_num2ulong (x, "gc-set-size!"));
    return ksi_void;
}

static ksi_obj
gc_heap_size ()
{
    return ksi_ulong2num (ksi_get_heap_size ());
}

static ksi_obj
gc_free_size (void)
{
    return ksi_ulong2num (ksi_get_heap_free ());
}

static ksi_obj
gc_collections (void)
{
    return ksi_ulong2num (ksi_gcollections ());
}

static ksi_obj
gc_collect ()
{
    ksi_gcollect (1);
    return ksi_void;
}

static ksi_obj
scm_format_proc (ksi_obj port, ksi_obj str, int argc, ksi_obj* argv)
{
    KSI_CHECK (str, KSI_STR_P (str), "format: invalid string in arg2");
    return ksi_format (port, KSI_STR_PTR (str), argc, argv);
}

static ksi_obj
scm_error_proc (int argc, ksi_obj* argv)
{
    ksi_obj errobj = 0;
    const char *who = 0, *msg = 0;
    int i;

    if (argc > 0 && KSI_SYM_P(argv[0])) {
        who = KSI_SYM_PTR(argv[0]);
        --argc; ++argv;
    }

    if (argc <= 0)
        ksi_exn_error (0, 0, "unspecified error");

    if (argc > 0 && KSI_STR_P (argv[0])) {
        ksi_string str = (ksi_string) ksi_make_string(0, 0);
        ksi_port port = ksi_new_str_port(str);
        port->output = 1;

        i = ksi_internal_format (port, KSI_STR_PTR(argv[0]), argc-1, argv+1, "error");
        argc -= i + 1;
        argv += i + 1;

        msg = KSI_STR_PTR(str);
    }

    for (i = 0; i < argc; i++) {
        if (!errobj)
            errobj = argv[i];
    }

    ksi_exn_error (who, errobj, msg);
    return ksi_void;
}

static ksi_obj
scm_errlog_proc (ksi_obj mod, ksi_obj priority, ksi_obj fmt, int argc, ksi_obj* argv)
{
    int pri, log_pri;
    ksi_obj str, x;

    KSI_CHECK (priority, KSI_EINT_P (priority), "errlog: invalid priority in arg2");
    KSI_CHECK (fmt, KSI_STR_P (fmt), "errlog: invalid string in arg3");

    pri = ksi_num2long (priority, "errlog");
    KSI_CHECK (priority, ERRLOG_EMERG <= pri && pri <= ERRLOG_ALL, "errlog: priority out of range");

    x = ksi_assq_ref (ksi_int_data->module_priority, mod);

    if (x == ksi_false)
        log_pri = ksi_int_data->errlog_priority;
    else
        log_pri = ksi_num2long (x, "errlog");

    if (pri > log_pri)
        return ksi_void;

    str = ksi_format (ksi_false, KSI_STR_PTR (fmt), argc, argv);
    ksi_errlog_msg (pri, KSI_STR_PTR (str));
    return ksi_void;
}

static ksi_obj
scm_abbrev (ksi_obj str)
{
    KSI_CHECK (str, KSI_STR_P (str), "abbrev: invalid string in arg1");
    return ksi_abbrev (KSI_STR_PTR (str), KSI_STR_LEN (str));
}

static ksi_obj
eval_cxr (char* data, ksi_obj val)
{
    char* ptr;

    ptr = data + strlen (data) - 1;
    for (--ptr; *ptr != 'c'; --ptr) {
        if (!KSI_PAIR_P (val))
            ksi_exn_error (ksi_assertion_s, val, "%s: invalid pair", data);

        val = (*ptr == 'd') ? KSI_CDR (val) : KSI_CAR (val);
    }

    return val;
}

static void
ksi_init_system()
{
    ksi_env env;

    env = ksi_data->syntax_env; /* ksi_get_lib_env("ksi", "core", "syntax", 0); */
    ksi_defsyntax(ksi_data->sym_begin, ksi_new_core(KSI_TAG_BEGIN), env, 1);
    ksi_defsyntax(ksi_data->sym_if, ksi_new_core(KSI_TAG_IF), env, 1);
    ksi_defsyntax(ksi_data->sym_lambda, ksi_new_core(KSI_TAG_LAMBDA), env, 1);
    ksi_defsyntax(ksi_data->sym_define, ksi_new_core(KSI_TAG_DEFINE), env, 1);
    ksi_defsyntax(ksi_data->sym_set, ksi_new_core(KSI_TAG_SET), env, 1);
    ksi_defsyntax(ksi_data->sym_and, ksi_new_core(KSI_TAG_AND), env, 1);
    ksi_defsyntax(ksi_data->sym_or, ksi_new_core(KSI_TAG_OR), env, 1);
    ksi_defsyntax(ksi_data->sym_let, ksi_new_core(KSI_TAG_LET), env, 1);
    ksi_defsyntax(ksi_data->sym_letrec, ksi_new_core(KSI_TAG_LETREC), env, 1);
    ksi_defsyntax(ksi_data->sym_letrec_star, ksi_new_core(KSI_TAG_LETREC_STAR), env, 1);

    ksi_defsyntax(ksi_data->sym_letstar, (ksi_obj) ksi_new_prim("let*", ksi_letstar_macro, KSI_CALL_ARG2, 2), env, 1);
    ksi_defsyntax(ksi_data->sym_case, (ksi_obj) ksi_new_prim("case", ksi_case_macro, KSI_CALL_ARG2, 2), env, 1);
    ksi_defsyntax(ksi_data->sym_cond, (ksi_obj) ksi_new_prim("cond", ksi_cond_macro, KSI_CALL_ARG2, 2), env, 1);
    ksi_defsyntax(ksi_data->sym_else, ksi_data->sym_else, env, 1);
    ksi_defsyntax(ksi_data->sym_arrow, ksi_data->sym_arrow, env, 1);

    ksi_defsyntax(ksi_data->sym_quote, ksi_new_core(KSI_TAG_QUOTE), env, 1);
    ksi_defsyntax(ksi_data->sym_quasiquote, (ksi_obj) ksi_new_prim("quasiquote", ksi_quasiquote_macro, KSI_CALL_ARG2, 2), env, 1);
    ksi_defsyntax(ksi_data->sym_unquote, ksi_data->sym_unquote, env, 1);
    ksi_defsyntax(ksi_data->sym_unquote_splicing, ksi_data->sym_unquote_splicing, env, 1);

    ksi_defsyntax(ksi_data->sym_syntax, ksi_new_core(KSI_TAG_SYNTAX), env, 1);
    ksi_defsyntax(ksi_data->sym_quasisyntax, (ksi_obj) ksi_new_prim("quasisyntax", ksi_quasisyntax_macro, KSI_CALL_ARG2, 2), env, 1);
    ksi_defsyntax(ksi_data->sym_unsyntax, ksi_data->sym_unsyntax, env, 1);
    ksi_defsyntax(ksi_data->sym_unsyntax_splicing, ksi_data->sym_unsyntax_splicing, env, 1);

    ksi_defsyntax(ksi_data->sym_dots, ksi_data->sym_dots, env, 1);


    env = ksi_get_lib_env("ksi", "core", "list", 0);
    ksi_defsym("pair?", ksi_data->pairp_proc, env);
    ksi_defsym("null?", ksi_data->nullp_proc, env);
    ksi_defsym("list?", ksi_data->listp_proc, env);

    ksi_defsym("list", ksi_data->list_proc, env);
    ksi_defsym("cons", ksi_data->cons_proc, env);
    ksi_defun("cons*", ksi_cons_a, KSI_CALL_REST0, 1, env);
    ksi_defun("xcons", ksi_xcons, KSI_CALL_ARG2, 2, env);
    ksi_defun("acons", ksi_acons, KSI_CALL_ARG3, 3, env);
    ksi_defun("make-list", ksi_make_list, KSI_CALL_ARG2, 1, env);

    ksi_defsym("car", ksi_data->car_proc, env);
    ksi_defsym("cdr", ksi_data->cdr_proc, env);
    {
        int l, x;
        /* c(a|d)*r defined separately */
        for (l = 2; l <= 4; ++l) {
            for (x = 0; x < (1 << l); ++x) {
                ksi_obj args[1];
                char* name = (char*) ksi_malloc_data (l+3);
                int i = 0;

                for (name [0] = 'c'; i++ < l; )
                    name[i] = x & (1 << (l-i)) ? 'd' : 'a';
                name[i++] = 'r';
                name[i] = 0;

                args[0] = (ksi_obj) name;
                ksi_defsym (name, ksi_close_proc ((ksi_obj) ksi_new_prim (name, eval_cxr, KSI_CALL_ARG2, 2), 1, args), env);
            }
        }
    }

    ksi_defun("set-car!", ksi_set_car_x, KSI_CALL_ARG2, 2, env);
    ksi_defun("set-cdr!", ksi_set_cdr_x, KSI_CALL_ARG2, 2, env);
    ksi_defun("list-set!", ksi_list_set_x, KSI_CALL_ARG3, 3, env);

    ksi_defun("list-head", ksi_list_head, KSI_CALL_ARG2, 2, env);
    ksi_defun("list-tail", ksi_list_tail, KSI_CALL_ARG2, 2, env);
    ksi_defun("last-pair", ksi_last_pair, KSI_CALL_ARG1, 1, env);
    ksi_defun("list-ref", ksi_list_ref, KSI_CALL_ARG2, 2, env);

    ksi_defun("length", ksi_length, KSI_CALL_ARG1, 1, env);
    ksi_defsym("append", ksi_data->append_proc, env);
    ksi_defun("append!", ksi_append_x, KSI_CALL_REST0, 0, env);
    ksi_defun("reverse", ksi_reverse, KSI_CALL_ARG1, 1, env);
    ksi_defun("reverse!", ksi_reverse_x, KSI_CALL_ARG1, 1, env);
    ksi_defun("copy-list", ksi_copy_list, KSI_CALL_ARG1, 1, env);
    ksi_defun("copy-tree", ksi_copy_tree, KSI_CALL_ARG1, 1, env);

    ksi_defun("map", ksi_map, KSI_CALL_REST1, 2, env);
    ksi_defun("for-each", ksi_for_each, KSI_CALL_REST1, 2, env);
    ksi_defun("fold-left", ksi_fold_left, KSI_CALL_REST2, 3, env);
    ksi_defun("fold-right", ksi_fold_right, KSI_CALL_REST2, 3, env);

    ksi_defun("remq", ksi_remq, KSI_CALL_ARG2, 2, env);
    ksi_defun("remv", ksi_remv, KSI_CALL_ARG2, 2, env);
    ksi_defun("remove", ksi_remove, KSI_CALL_ARG2, 2, env);
    ksi_defun("remp", ksi_remp, KSI_CALL_ARG2, 2, env);

    ksi_defsym("memq", ksi_data->memq_proc, env);
    ksi_defsym("memv", ksi_data->memv_proc, env);
    ksi_defsym("member", ksi_data->member_proc, env);
    ksi_defun("memp", ksi_memp, KSI_CALL_ARG2, 2, env);

    ksi_defun("assq", ksi_assq, KSI_CALL_ARG2, 2, env);
    ksi_defun("assv", ksi_assv, KSI_CALL_ARG2, 2, env);
    ksi_defun("assoc", ksi_assoc, KSI_CALL_ARG2, 2, env);
    ksi_defun("assp", ksi_assp, KSI_CALL_ARG2, 2, env);

    ksi_defun("assq-ref", ksi_assq_ref, KSI_CALL_ARG2, 2, env);
    ksi_defun("assv-ref", ksi_assv_ref, KSI_CALL_ARG2, 2, env);
    ksi_defun("assoc-ref", ksi_assoc_ref, KSI_CALL_ARG3, 2, env);

    ksi_defun("assq-set!", ksi_assq_set_x, KSI_CALL_ARG3, 3, env);
    ksi_defun("assv-set!", ksi_assv_set_x, KSI_CALL_ARG3, 3, env);
    ksi_defun("assoc-set!", ksi_assoc_set_x, KSI_CALL_ARG4, 3, env);

    ksi_defun("assq-remove!", ksi_assq_remove_x, KSI_CALL_ARG2, 2, env);
    ksi_defun("assv-remove!", ksi_assv_remove_x, KSI_CALL_ARG2, 2, env);
    ksi_defun("assoc-remove!", ksi_assoc_remove_x, KSI_CALL_ARG3, 2, env);


    env = ksi_get_lib_env("ksi", "core", "number", 0);
    ksi_defun("number?", ksi_number_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("complex?", ksi_complex_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("real?", ksi_real_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("rational?", ksi_rational_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("integer?", ksi_integer_p, KSI_CALL_ARG1, 1, env);

    ksi_defun("exact?", ksi_exact_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("inexact?", ksi_inexact_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("inexact", ksi_inexact, KSI_CALL_ARG1, 1, env);
    ksi_defun("exact", ksi_exact, KSI_CALL_ARG1, 1, env);

    ksi_defun("=", ksi_num_eq_p, KSI_CALL_REST0, 0, env);
    ksi_defun("<", ksi_num_lt_p, KSI_CALL_REST0, 0, env);
    ksi_defun(">", ksi_num_gt_p, KSI_CALL_REST0, 0, env);
    ksi_defun("<=", ksi_num_le_p, KSI_CALL_REST0, 0, env);
    ksi_defun(">=", ksi_num_ge_p, KSI_CALL_REST0, 0, env);

    ksi_defun("zero?", ksi_zero_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("positive?", ksi_positive_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("negative?", ksi_negative_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("odd?", ksi_odd_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("even?", ksi_even_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("finite?", ksi_finite_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("infinite?", ksi_infinite_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("nan?", ksi_nan_p, KSI_CALL_ARG1, 1, env);

    ksi_defun("max", ksi_max, KSI_CALL_REST0, 1, env);
    ksi_defun("min", ksi_min, KSI_CALL_REST0, 1, env);
    ksi_defun("abs", ksi_abs, KSI_CALL_ARG1,  1, env);

    ksi_defun("+", ksi_plus, KSI_CALL_REST0, 0, env);
    ksi_defun("-", ksi_minus, KSI_CALL_REST0, 1, env);
    ksi_defun("*", ksi_multiply, KSI_CALL_REST0, 0, env);
    ksi_defun("/", ksi_divide, KSI_CALL_REST0, 1, env);

    ksi_defun("exact-div", ksi_exact_div, KSI_CALL_ARG2, 2, env);
    ksi_defun("div", ksi_idiv, KSI_CALL_ARG2, 2, env);
    ksi_defun("mod", ksi_imod, KSI_CALL_ARG2, 2, env);
    ksi_defun("div-and-mod", ksi_idiv_and_mod, KSI_CALL_ARG2, 2, env);
    ksi_defun("div-and-mod*", ksi_idiv_and_mod_who, KSI_CALL_ARG3, 3, env);
//    ksi_defun("quotient", ksi_quot, KSI_CALL_ARG2, 2, env);
//    ksi_defun("remainder", ksi_rem, KSI_CALL_ARG2, 2, env);

    ksi_defun("gcd", ksi_gcd, KSI_CALL_REST0, 0, env);
    ksi_defun("lcm", ksi_lcm, KSI_CALL_REST0, 0, env);

    ksi_defun("numerator", ksi_numerator, KSI_CALL_ARG1, 1, env);
    ksi_defun("denominator", ksi_denominator, KSI_CALL_ARG1, 1, env);

    ksi_defun("floor", ksi_floor, KSI_CALL_ARG1, 1, env);
    ksi_defun("ceiling", ksi_ceiling, KSI_CALL_ARG1, 1, env);
    ksi_defun("truncate", ksi_truncate, KSI_CALL_ARG1, 1, env);
    ksi_defun("round", ksi_round, KSI_CALL_ARG1, 1, env);

    ksi_defun("exp", ksi_exp, KSI_CALL_ARG1, 1, env);
    ksi_defun("log", ksi_log, KSI_CALL_ARG2, 1, env);
    ksi_defun("sin", ksi_sin, KSI_CALL_ARG1, 1, env);
    ksi_defun("cos", ksi_cos, KSI_CALL_ARG1, 1, env);
    ksi_defun("tan", ksi_tan, KSI_CALL_ARG1, 1, env);
    ksi_defun("sinh", ksi_sinh, KSI_CALL_ARG1, 1, env);
    ksi_defun("cosh", ksi_cosh, KSI_CALL_ARG1, 1, env);
    ksi_defun("tanh", ksi_tanh, KSI_CALL_ARG1, 1, env);
    ksi_defun("asin", ksi_asin, KSI_CALL_ARG1, 1, env);
    ksi_defun("acos", ksi_acos, KSI_CALL_ARG1, 1, env);
    ksi_defun("atan", ksi_atan, KSI_CALL_ARG2, 1, env);
    ksi_defun("asinh", ksi_asinh, KSI_CALL_ARG1, 1, env);
    ksi_defun("acosh", ksi_acosh, KSI_CALL_ARG1, 1, env);
    ksi_defun("atanh", ksi_atanh, KSI_CALL_ARG1, 1, env);

    ksi_defun("sqrt", ksi_sqrt, KSI_CALL_ARG1, 1, env);
    ksi_defun("exact-integer-sqrt", ksi_exact_sqrt, KSI_CALL_ARG1, 1, env);
    ksi_defun("expt", ksi_expt, KSI_CALL_ARG2, 2, env);

    ksi_defun("make-rectangular", ksi_make_rectangular, KSI_CALL_ARG2, 2, env);
    ksi_defun("make-polar", ksi_make_polar, KSI_CALL_ARG2, 2, env);
    ksi_defun("real-part", ksi_scm_real_part, KSI_CALL_ARG1, 1, env);
    ksi_defun("imag-part", ksi_scm_imag_part, KSI_CALL_ARG1, 1, env);
    ksi_defun("magnitude", ksi_abs, KSI_CALL_ARG1, 1, env);
    ksi_defun("angle", ksi_scm_angle, KSI_CALL_ARG1, 1, env);

    ksi_defun("number->string", ksi_number2string, KSI_CALL_ARG2, 1, env);
    ksi_defun("string->number", ksi_string2number, KSI_CALL_ARG2, 1, env);

//    ksi_defun("lognot", ksi_lognot, KSI_CALL_ARG1, 1, env);
//    ksi_defun("logtest", ksi_logtest, KSI_CALL_ARG2, 2, env);
//    ksi_defun("logior", ksi_logior, KSI_CALL_ARG2, 2, env);
//    ksi_defun("logxor", ksi_logxor, KSI_CALL_ARG2, 2, env);
//    ksi_defun("logand", ksi_logand, KSI_CALL_ARG2, 2, env);
//    ksi_defun("logbit?", ksi_logbit_p, KSI_CALL_ARG2, 2, env);
//    ksi_defun("ash", ksi_ash, KSI_CALL_ARG2, 2, env);


    env = ksi_get_lib_env("ksi", "core", "char", 0);
    ksi_defun("char?", ksi_char_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("char=?", ksi_char_eq_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char<?", ksi_char_lt_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char>?", ksi_char_gt_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char<=?", ksi_char_le_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char>=?", ksi_char_ge_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char-ci=?", ksi_char_ci_eq_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char-ci<?", ksi_char_ci_lt_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char-ci>?", ksi_char_ci_gt_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char-ci<=?", ksi_char_ci_le_p, KSI_CALL_REST0, 0, env);
    ksi_defun("char-ci>=?", ksi_char_ci_ge_p, KSI_CALL_REST0, 0, env);

    ksi_defun("char-alphabetic?", ksi_char_alpha_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("char-numeric?", ksi_char_digit_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("char-whitespace?", ksi_char_space_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("char-upper-case?", ksi_char_upper_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("char-lower-case?", ksi_char_lower_p, KSI_CALL_ARG1, 1, env);

    ksi_defun("integer->char", ksi_integer2char, KSI_CALL_ARG1, 1, env);
    ksi_defun("char->integer", ksi_char2integer, KSI_CALL_ARG1, 1, env);
    ksi_defun("char-upcase", ksi_char_upcase, KSI_CALL_ARG1, 1, env);
    ksi_defun("char-downcase", ksi_char_downcase, KSI_CALL_ARG1, 1, env);


    env = ksi_get_lib_env("ksi", "core", "string", 0);
    ksi_defun("string?", ksi_string_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("make-string", ksi_scm_make_string, KSI_CALL_ARG2, 1, env);
    ksi_defun("string-length", ksi_string_length, KSI_CALL_ARG1, 1, env);
    ksi_defun("string-ref", ksi_string_ref, KSI_CALL_ARG2, 2, env);
    ksi_defun("string-set!", ksi_string_set_x, KSI_CALL_ARG3, 3, env);
    ksi_defun("substring", ksi_substring, KSI_CALL_ARG3, 2, env);
    ksi_defun("string-copy", ksi_string_copy, KSI_CALL_ARG1, 1, env);
    ksi_defun("string-fill!", ksi_string_fill_x, KSI_CALL_ARG2, 2, env);
    ksi_defun("string", ksi_new_string, KSI_CALL_REST0, 0, env);
    ksi_defun("string-append", ksi_string_append, KSI_CALL_REST0, 0, env);
    ksi_defun("string-for-each", ksi_vector_for_each, KSI_CALL_REST2, 2, env);

    ksi_defun("string->list", ksi_string2list, KSI_CALL_ARG1, 1, env);
    ksi_defun("list->string", ksi_list2string, KSI_CALL_ARG1, 1, env);
    ksi_defun("symbol->string", ksi_symbol2string, KSI_CALL_ARG1, 1, env);
    ksi_defun("string->symbol", ksi_string2symbol, KSI_CALL_ARG1, 1, env);
    ksi_defun("string->keyword", ksi_string2keyword, KSI_CALL_ARG1, 1, env);
    ksi_defun("keyword->string", ksi_keyword2string, KSI_CALL_ARG1, 1, env);

    ksi_defun("string=?", ksi_string_eq_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string-ci=?", ksi_string_ci_eq_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string<?", ksi_string_ls_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string>?", ksi_string_gt_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string<=?", ksi_string_le_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string>=?", ksi_string_ge_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string-ci<?", ksi_string_ci_ls_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string-ci>?", ksi_string_ci_gt_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string-ci<=?", ksi_string_ci_le_p, KSI_CALL_REST0, 0, env);
    ksi_defun("string-ci>=?", ksi_string_ci_ge_p, KSI_CALL_REST0, 0, env);

    // non RNRS string utils
    ksi_defun("string-index", ksi_string_index, KSI_CALL_ARG3, 2, env);
    ksi_defun("string-rindex", ksi_string_rindex, KSI_CALL_ARG3, 2, env);
    ksi_defun("string-upcase!", ksi_string_upcase_x, KSI_CALL_ARG1, 1, env);
    ksi_defun("string-downcase!", ksi_string_downcase_x, KSI_CALL_ARG1, 1, env);
    ksi_defun("string-capitalize!", ksi_string_capitalize_x, KSI_CALL_ARG1, 1, env);


    env = ksi_get_lib_env("ksi", "core", "vector", 0);
    ksi_defsym("vector?", ksi_data->vectorp_proc, env);
    ksi_defsym("vector", ksi_data->vector_proc, env);
    ksi_defsym("list->vector", ksi_data->list2vector_proc, env);
    ksi_defun("vector->list", ksi_vector2list, KSI_CALL_ARG1, 1, env);
    ksi_defun("make-vector", ksi_make_vector, KSI_CALL_ARG2, 1, env);
    ksi_defun("vector-length", ksi_vector_length, KSI_CALL_ARG1, 1, env);
    ksi_defun("vector-ref", ksi_vector_ref, KSI_CALL_ARG2, 2, env);
    ksi_defun("vector-set!", ksi_vector_set_x, KSI_CALL_ARG3, 3, env);
    ksi_defun("vector-fill!", ksi_vector_fill_x, KSI_CALL_ARG2, 2, env);
    ksi_defun("vector-map", ksi_vector_map, KSI_CALL_REST2, 2, env);
    ksi_defun("vector-for-each", ksi_vector_for_each, KSI_CALL_REST2, 2, env);


    env = ksi_get_lib_env("ksi", "core", "base", 0);
    ksi_defun("void?", ksi_void_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("boolean?", ksi_bool_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("boolean=?", ksi_bool_eq_p, KSI_CALL_REST0, 0, env);

    ksi_defsym("not", ksi_data->not_proc, env);
    ksi_defsym("eq?", ksi_data->eq_proc, env);
    ksi_defsym("eqv?", ksi_data->eqv_proc, env);
    ksi_defsym("equal?", ksi_data->equal_proc, env);

    ksi_defun("symbol?", ksi_symbol_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("symbol=?", ksi_symbol_eq_p, KSI_CALL_REST0, 0, env);

    ksi_defun("keyword?", ksi_key_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("make-keyword", ksi_make_keyword, KSI_CALL_ARG1, 1, env);
    ksi_defun("keyword->symbol", ksi_keyword2symbol, KSI_CALL_ARG1, 1, env);
    ksi_defun("symbol->keyword", ksi_symbol2keyword, KSI_CALL_ARG1, 1, env);

    ksi_defun("object->string", ksi_object2string, KSI_CALL_ARG1, 1, env);
    ksi_defun("values", ksi_new_values, KSI_CALL_REST0, 0, env);
    ksi_defun("dynamic-wind", ksi_dynamic_wind, KSI_CALL_ARG3, 3, env);

    ksi_defun("gensym", ksi_gensym, KSI_CALL_ARG2, 0, env);
    ksi_defun("abbrev", scm_abbrev, KSI_CALL_ARG1, 1, env);

    ksi_defun("procedure?", ksi_procedure_p, KSI_CALL_ARG1, 1, env);
    ksi_defsym("void", ksi_data->void_proc, env);
    ksi_defsym("true", ksi_data->true_proc, env);
    ksi_defsym("false", ksi_data->false_proc, env);
    ksi_defsym("apply", ksi_data->apply_proc, env);
    ksi_defsym("call-with-values", ksi_data->call_vs_proc, env);
    ksi_defsym("call-with-current-continuation", ksi_data->call_cc_proc, env);
    ksi_defsym("call/cc", ksi_data->call_cc_proc, env);


    env = ksi_get_lib_env("ksi", "core", "system", 0);
    ksi_defun("copyright", ksi_copyright, KSI_CALL_ARG0, 0, env);
    ksi_defun("banner", ksi_banner, KSI_CALL_ARG0, 0, env);
    ksi_defun("version", ksi_scm_version, KSI_CALL_ARG0, 0, env);
    ksi_defun("major-version", ksi_scm_major_version, KSI_CALL_ARG0, 0, env);
    ksi_defun("minor-version", ksi_scm_minor_version, KSI_CALL_ARG0, 0, env);
    ksi_defun("patch-level", ksi_scm_patch_level, KSI_CALL_ARG0, 0, env);
    ksi_defun("cpu", ksi_scm_cpu, KSI_CALL_ARG0, 0, env);
    ksi_defun("os", ksi_scm_os, KSI_CALL_ARG0, 0, env);
    ksi_defun("host", ksi_scm_host, KSI_CALL_ARG0, 0, env);
    ksi_defun("lib-path", lib_path, KSI_CALL_ARG0, 0, env);


    env = ksi_get_lib_env("ksi", "core", "build", 0);
    ksi_defun("install-include-dir", inst_include_dir, KSI_CALL_ARG0, 0, env);
    ksi_defun("install-lib-dir", inst_lib_dir, KSI_CALL_ARG0, 0, env);
    ksi_defun("install-bin-dir", inst_bin_dir, KSI_CALL_ARG0, 0, env);
    ksi_defun("install-info-dir", inst_info_dir, KSI_CALL_ARG0, 0, env);
    ksi_defun("install-man-dir", inst_man_dir, KSI_CALL_ARG0, 0, env);
    ksi_defun("build-cflags", build_cflags, KSI_CALL_ARG0, 0, env);
    ksi_defun("build-libs", build_libs, KSI_CALL_ARG0, 0, env);


    env = ksi_get_lib_env("ksi", "core", "gc", 0);
    ksi_defun("gc-expand", gc_expand_heap, KSI_CALL_ARG1, 1, env);
    ksi_defun("gc-set-size!", gc_set_heap_size, KSI_CALL_ARG1, 1, env);
    ksi_defun("gc-size", gc_heap_size, KSI_CALL_ARG0, 0, env);
    ksi_defun("gc-free", gc_free_size, KSI_CALL_ARG0, 0, env);
    ksi_defun("gc-collections", gc_collections, KSI_CALL_ARG0, 0, env);
    ksi_defun("gc-collect", gc_collect, KSI_CALL_ARG0, 0, env);


    env = ksi_get_lib_env("ksi", "core", "exn", 0);
    ksi_defun("make-exn", ksi_scm_make_exn, KSI_CALL_ARG4,  3, env);
    ksi_defun("exn?", ksi_exn_p, KSI_CALL_ARG1,  1, env);
    ksi_defun("exn-type", ksi_exn_type, KSI_CALL_ARG1,  1, env);
    ksi_defun("exn-message", ksi_exn_message, KSI_CALL_ARG1,  1, env);
    ksi_defun("exn-value", ksi_exn_value, KSI_CALL_ARG1,  1, env);
    ksi_defun("exn-source", ksi_exn_source, KSI_CALL_ARG1,  1, env);
    ksi_defun("catch", ksi_catch, KSI_CALL_ARG3, 2, env);
    ksi_defun("catch-with-retry", ksi_catch_with_retry, KSI_CALL_ARG3, 2, env);
    ksi_defun("throw", ksi_throw, KSI_CALL_ARG2, 1, env);
    ksi_defun("add-error-handler", ksi_add_error_handler, KSI_CALL_ARG1, 1, env);
    ksi_defun("remove-error-handler", ksi_remove_error_handler, KSI_CALL_ARG1, 1, env);
    ksi_defun("add-exit-handler", ksi_add_exit_handler, KSI_CALL_ARG1, 1, env);
    ksi_defun("exit", ksi_exit, KSI_CALL_ARG1, 0, env);

    ksi_defun("error", scm_error_proc, KSI_CALL_REST0, 0, env);
    ksi_defun("open-errlog", ksi_open_errlog, KSI_CALL_ARG1,  1, env);
    ksi_defun("errlog-priority", ksi_errlog_priority, KSI_CALL_ARG2,  1, env);
    ksi_defun("errlog", scm_errlog_proc, KSI_CALL_REST3, 3, env);
    ksi_defsym ("errlog/emerg", ksi_long2num(ERRLOG_EMERG), env);
    ksi_defsym ("errlog/alert", ksi_long2num(ERRLOG_ALERT), env);
    ksi_defsym ("errlog/error", ksi_long2num(ERRLOG_ERROR), env);
    ksi_defsym ("errlog/warning", ksi_long2num(ERRLOG_WARNING), env);
    ksi_defsym ("errlog/notice", ksi_long2num(ERRLOG_NOTICE), env);
    ksi_defsym ("errlog/info", ksi_long2num(ERRLOG_INFO), env);
    ksi_defsym ("errlog/debug", ksi_long2num(ERRLOG_DEBUG), env);
    ksi_defsym ("errlog/all", ksi_long2num(ERRLOG_ALL), env);


    env = ksi_get_lib_env("ksi", "core", "eval", 0);
    ksi_defsyntax(ksi_data->sym_defmacro, ksi_new_core(KSI_TAG_DEFSYNTAX), env, 1);
    ksi_defun("eval", ksi_eval, KSI_CALL_ARG2, 2, env);
    ksi_defun("environment", ksi_environment, KSI_CALL_REST0, 0, env);
    ksi_defun("compile", ksi_comp, KSI_CALL_ARG2, 2, env);
    ksi_defun("closure-environment", ksi_closure_env, KSI_CALL_ARG1, 1, env);
    ksi_defun("closure-body", ksi_closure_body, KSI_CALL_ARG1, 1, env);
    ksi_defun("closure?", ksi_closure_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("procedure-arity", ksi_procedure_arity, KSI_CALL_ARG1, 1, env);
    ksi_defun("procedure-has-arity?", ksi_procedure_has_arity_p, KSI_CALL_ARG3, 2, env);

    ksi_defun("environment?", ksi_env_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("bound?", ksi_bound_p, KSI_CALL_ARG2, 2, env);
    ksi_defun("exported?", ksi_exported_p, KSI_CALL_ARG3, 2, env);
    ksi_defun("environment-ref", ksi_var_ref, KSI_CALL_ARG2, 2, env);
    ksi_defun("environment-set!", ksi_var_set, KSI_CALL_ARG3, 3, env);
    ksi_defun("environment-for-each", ksi_env_for_each, KSI_CALL_ARG2, 2, env);
    ksi_defun("bound-identifier?", ksi_bound_identifier_p, KSI_CALL_ARG2, 2, env);
    /*ksi_defun("export", ksi_export, KSI_CALL_ARG3, 2, env);*/
    /*ksi_defun("import", ksi_import, KSI_CALL_ARG4, 3, env);*/


    env = ksi_get_lib_env("ksi", "core", "time", 0);
    ksi_defun("current-time", ksi_scm_current_time, KSI_CALL_ARG0, 0, env);
    ksi_defun("cpu-time", ksi_scm_cpu_time, KSI_CALL_ARG0, 0, env);
    ksi_defun("eval-time", ksi_scm_eval_time, KSI_CALL_ARG0, 0, env);
    ksi_defun("real-time", ksi_scm_real_time, KSI_CALL_ARG0, 0, env);
    ksi_defun("gmtime", ksi_scm_gmtime, KSI_CALL_ARG1, 0, env);
    ksi_defun("localtime", ksi_scm_localtime, KSI_CALL_ARG1, 0, env);
    ksi_defun("strftime", ksi_strftime, KSI_CALL_ARG2, 2, env);
    ksi_defun("mktime", ksi_mktime, KSI_CALL_ARG1, 1, env);


    env = ksi_get_lib_env("ksi", "core", "shell", 0);
    ksi_defun("getenv", ksi_getenv, KSI_CALL_ARG1, 0, env);
    ksi_defun("system", ksi_syscall, KSI_CALL_ARG1, 1, env);
    ksi_defun("getcwd", ksi_getcwd, KSI_CALL_ARG0, 0, env);
    ksi_defun("chdir", ksi_chdir, KSI_CALL_ARG1, 1, env);
    ksi_defun("mkdir", ksi_mkdir, KSI_CALL_ARG2, 1, env);
    ksi_defun("rmdir", ksi_rmdir, KSI_CALL_ARG1, 1, env);
    ksi_defun("stat", ksi_stat, KSI_CALL_ARG1, 1, env);
    ksi_defun("file-exists?", ksi_file_exists, KSI_CALL_ARG1, 1, env);
    ksi_defun("delete-file", ksi_delete_file, KSI_CALL_ARG1, 1, env);
    ksi_defun("rename-file", ksi_rename_file, KSI_CALL_ARG2, 2, env);
    ksi_defun("opendir", ksi_opendir, KSI_CALL_ARG1, 1, env);
    ksi_defun("readdir", ksi_readdir, KSI_CALL_ARG1, 1, env);
    ksi_defun("closedir", ksi_closedir, KSI_CALL_ARG1, 1, env);
    ksi_defun("path-list->file-name", ksi_exp_fname, KSI_CALL_ARG2, 1, env);
    ksi_defun("split-file-name", ksi_split_fname, KSI_CALL_ARG1, 1, env);
    ksi_defun("split-path", ksi_split_path, KSI_CALL_ARG1, 1, env);
    ksi_defun("file-name-has-suffix?", ksi_has_suffix_p, KSI_CALL_ARG2, 2, env);


    env = ksi_get_lib_env("ksi", "core", "hashtables", 0);
    ksi_defun("make-eq-hashtable", ksi_make_eq_hashtab, KSI_CALL_ARG1, 0, env);
    ksi_defun("make-eqv-hashtable", ksi_make_eqv_hashtab, KSI_CALL_ARG1, 0, env);
    ksi_defun("make-hashtable", ksi_make_hashtab, KSI_CALL_ARG3, 2, env);
    ksi_defun("hashtable?", ksi_hashtab_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("hashtable-size", ksi_hashtab_size, KSI_CALL_ARG1, 1, env);
    ksi_defun("hashtable-ref", ksi_hash_ref, KSI_CALL_ARG3, 2, env);
    ksi_defun("hashtable-set!", ksi_hash_set_x, KSI_CALL_ARG3, 3, env);
    ksi_defun("hashtable-delete!", ksi_hash_del_x, KSI_CALL_ARG2, 2, env);
    ksi_defun("hashtable-contains?", ksi_hash_has_p, KSI_CALL_ARG2, 2, env);
    ksi_defun("hashtable-for-each", ksi_hash_for_each, KSI_CALL_ARG2, 2, env);
    ksi_defun("hashtable-copy", ksi_hash_copy, KSI_CALL_ARG2, 1, env);
    ksi_defun("hashtable-clear!", ksi_hash_clear, KSI_CALL_ARG1, 1, env);
    ksi_defun("hashtable-equivalence-function", ksi_hash_eq_fun, KSI_CALL_ARG1, 1, env);
    ksi_defun("hashtable-hash-function", ksi_hash_hash_fun, KSI_CALL_ARG1, 1, env);
    ksi_defun("hashtable-mutable?", ksi_hash_mutable_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("equal-hash", ksi_equal_hash, KSI_CALL_ARG1, 1, env);


    env = ksi_get_lib_env("ksi", "core", "io", 0);
    ksi_defun("current-input-port", ksi_current_input_port, KSI_CALL_ARG0, 0, env);
    ksi_defun("current-output-port", ksi_current_output_port, KSI_CALL_ARG0, 0, env);
    ksi_defun("current-error-port", ksi_current_error_port, KSI_CALL_ARG0, 0, env);
    ksi_defun("set-current-input-port", ksi_set_current_input_port, KSI_CALL_ARG1, 1, env);
    ksi_defun("set-current-output-port", ksi_set_current_output_port, KSI_CALL_ARG1, 1, env);
    ksi_defun("set-current-error-port", ksi_set_current_error_port, KSI_CALL_ARG1, 1, env);
    ksi_defun("open-file", ksi_open_file, KSI_CALL_ARG2, 2, env);
    ksi_defun("open-string", ksi_open_string, KSI_CALL_ARG2, 2, env);
    ksi_defun("close-port", ksi_close_port, KSI_CALL_ARG1, 1, env);
    ksi_defun("port?", ksi_port_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("input-port?", ksi_input_port_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("output-port?", ksi_output_port_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("flush-port", ksi_flush_port, KSI_CALL_ARG1, 1, env);
    ksi_defun("port-string", ksi_port_string, KSI_CALL_ARG1, 1, env);
    ksi_defun("read-char", ksi_read_char, KSI_CALL_ARG1, 0, env);
    ksi_defun("peek-char", ksi_peek_char, KSI_CALL_ARG1, 0, env);
    ksi_defun("char-ready?", ksi_char_ready_p, KSI_CALL_ARG1, 0, env);
    ksi_defun("port-ready?", ksi_port_ready_p, KSI_CALL_ARG1, 0, env);
    ksi_defun("eof-object?", ksi_eof_object_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("write", ksi_write, KSI_CALL_ARG2, 1, env);
    ksi_defun("display", ksi_display, KSI_CALL_ARG2, 1, env);
    ksi_defun("newline", ksi_newline, KSI_CALL_ARG1, 0, env);
    ksi_defun("write-char", ksi_write_char, KSI_CALL_ARG2, 1, env);

    ksi_defun("read-block", ksi_read_block, KSI_CALL_ARG2, 1, env);
    ksi_defun("write-block", ksi_write_block, KSI_CALL_ARG2, 2, env);
    ksi_defun("set-async-mode", ksi_set_async_mode, KSI_CALL_ARG2, 2, env);

    ksi_defun("format", scm_format_proc, KSI_CALL_REST2, 2, env);
    ksi_defsym("*null-port*", (ksi_obj) ksi_data->null_port, env);
    ksi_defun("setlocale", ksi_setlocale, KSI_CALL_ARG2, 1, env);

//    ksi_defun("object->dump", ksi_obj2dump, KSI_CALL_ARG1, 1, env);
//    ksi_defun("dump->object", ksi_dump2obj, KSI_CALL_ARG1, 1, env);
//    ksi_defun("read-dump", ksi_read_dump, KSI_CALL_ARG1, 1, env);
//    ksi_defun("write-dump", ksi_write_dump, KSI_CALL_ARG2, 2, env);

    ksi_defun("read", ksi_read, KSI_CALL_ARG1, 0, env);
    ksi_defun("annotation?", ksi_scm_annotation_p, KSI_CALL_ARG1, 1, env);
    ksi_defun("annotation-source", ksi_scm_annotation_source, KSI_CALL_ARG1, 1, env);
    ksi_defun("annotation-expression", ksi_scm_annotation_expression, KSI_CALL_ARG1, 1, env);

    ksi_init_klos();
    ksi_init_events();
    ksi_init_signals();
    ksi_init_dynl();
}

struct Ksi_Interp*
ksi_init (void *top_stack_addr)
{
    if (!ksi_int_data) {
        ksi_init_gc ();

        ksi_int_data = (struct Ksi_Interp*) ksi_malloc_eternal (sizeof *ksi_int_data);
        bzero (ksi_int_data, sizeof *ksi_int_data);

        ksi_internal_data();

        ksi_int_data->stack = top_stack_addr ? top_stack_addr : &top_stack_addr;
        ksi_int_data->user_quit_exn = ksi_make_exn ("quit", ksi_void, "interrupted by user", 0);

        ksi_int_data->exit_handlers = ksi_nil;
        ksi_int_data->error_handlers = ksi_nil;

        ksi_int_data->errlog_priority = ERRLOG_WARNING;
//        ksi_int_data->errlog_priority = ERRLOG_DEBUG;
        ksi_int_data->module_priority = ksi_nil;

        ksi_init_system();

        ksi_int_data->errlog_port = 0;
        ksi_int_data->input_port = ksi_data->null_port;
        ksi_int_data->output_port = ksi_data->null_port;
        ksi_int_data->error_port = ksi_data->null_port;

#if !defined(WIN32) || !defined(KSIDLL)
        atexit (ksi_term);
#endif
    }

    return ksi_int_data;
}

void
ksi_term (void)
{
  if (!ksi_int_data)
    return;

  while (ksi_int_data->exit_handlers != ksi_nil) {
    ksi_obj proc = KSI_CAR (ksi_int_data->exit_handlers);
    ksi_int_data->exit_handlers = KSI_CDR (ksi_int_data->exit_handlers);
    ksi_apply_0 (proc);
  }

  ksi_flush_port ((ksi_obj) ksi_int_data->output_port);
  ksi_flush_port ((ksi_obj) ksi_int_data->error_port);

#ifdef DYNAMIC_LINKING
  ksi_term_dynl ();
#endif
//  ksi_term_ports ();
  ksi_term_events ();
  ksi_term_signals ();

  ksi_free (ksi_int_data);
  ksi_int_data = 0;

  ksi_gcollect (1);
}


#if defined(WIN32) && defined(KSIDLL)

int APIENTRY
DllMain (HINSTANCE hInstance, DWORD fdwReason, PVOID pvReserved)
{
    switch (fdwReason) {
    case DLL_PROCESS_ATTACH:
        ksi_init_gc();
        break;

    case DLL_PROCESS_DETACH:
        ksi_term();
        break;

    case DLL_THREAD_ATTACH:
    case DLL_THREAD_DETACH:
        break;
    }

    return TRUE;
}

#endif

 /* End of code */
