/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method 
 *
 * Copyright (C) 2003 Botond Botyanszki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Based on server.c from nabi by Choe Hwanjin
 *
 */
#include <config.h>

#include <X11/Xlib.h>
#include <X11/keysym.h> 
#include <gtk/gtk.h>
#include <stdio.h>
#include <gdk/gdkx.h>
#include <gconf/gconf-client.h>
#include <locale.h>

#include "IMdkit/IMdkit.h"
#include "IMdkit/Xi18n.h"

#include "xim-server.h"
#include "xim-handler.h"

#include "../error.h"
#include "../conf.h"
#include "../im-ja.h"
#include "../nls.h"

#define DEFAULT_IC_TABLE_SIZE   1024

extern GConfClient *gconf_client;
IMJAConfig cfg; 
gint notify_id;
IMJAXimServer *im_ja_xim_server;
long im_ja_xim_filter_mask = KeyPressMask;

/* Supported Inputstyles */
static XIMStyle im_ja_xim_input_styles[] = {
    XIMPreeditCallbacks | XIMStatusCallbacks,
    XIMPreeditCallbacks | XIMStatusArea,
    XIMPreeditCallbacks | XIMStatusNothing,

		/*
		XIMPreeditPosition  | XIMStatusCallbacks,
		XIMPreeditPosition  | XIMStatusArea,
		*/
		XIMPreeditPosition  | XIMStatusNothing,

		/*
    XIMPreeditArea      | XIMStatusCallbacks,
    XIMPreeditArea      | XIMStatusArea,
    XIMPreeditArea      | XIMStatusNothing,
		*/

    XIMPreeditNothing   | XIMStatusCallbacks,
    XIMPreeditNothing   | XIMStatusArea,
    XIMPreeditNothing   | XIMStatusNothing,
    0
};
  
static XIMEncoding im_ja_xim_encodings[] = {
	"COMPOUND_TEXT",
	NULL
};

/* FIXME */
static XIMTriggerKey im_ja_xim_trigger_keys[] = {
	{XK_space, ShiftMask, ShiftMask},
	{0, 0, 0}
};



IMJAContext *im_ja_xim_server_ic_create(IMChangeICStruct *data) {
	static CARD16 id = 0;
	IMJAContext *ic;

	IM_JA_DEBUG("im_ja_xim_ic_create\n");

	if (im_ja_xim_server->ic_freed == NULL) {
		/* really make ic	 */
		id++;

		/* we do not use ic id 0 */
		if (id == 0) id++;

		if (id >= im_ja_xim_server->ic_table_size) {
	    im_ja_xim_server_ic_table_expand(im_ja_xim_server);
		}
		
		ic = im_ja_context_new();

		if (IS_IM_JA_CONTEXT(ic) == FALSE) IM_JA_DEBUG("**ERROR** INVALID OBJECT: NOT IM_JA_CONTEXT!\n");

		ic->id = id;
		im_ja_xim_server->ic_table[id] = ic;
	} 
	else {
		IM_JA_DEBUG("NEW IC FROM FREED\n");
		/* pick from ic_freed */ 
		ic = im_ja_xim_server->ic_freed;
		im_ja_xim_server->ic_freed = im_ja_xim_server->ic_freed->next;
		im_ja_xim_server->ic_table[ic->id] = ic;
		im_ja_context_init(ic);
	}
    
	/* store ic id */
	data->icid = ic->id;
	ic->connect_id = data->connect_id;

	im_ja_xim_ic_init_values(ic);
	im_ja_xim_ic_set_values(ic, data);

	return ic;
}

static int im_ja_xim_server_x_error_handler(Display *display, XErrorEvent *error) {
	gchar buf[64];

	XGetErrorText(display, error->error_code, buf, 63);
	IM_JA_DEBUG("ERROR in im-ja-xim-server: %s\n", buf);
	fprintf(stderr, "** ERROR **: X error in im-ja-xim-server: %s\n", buf);
	return 0;
}

/* FIXME: use some other glib datatype for this */
void im_ja_xim_server_ic_table_expand() {
	gint i;
	gulong old_size = im_ja_xim_server->ic_table_size;
 
	IM_JA_DEBUG("im_ja_xim_server_ic_table_expand\n");

	im_ja_xim_server->ic_table_size = im_ja_xim_server->ic_table_size * 2;
	im_ja_xim_server->ic_table = g_realloc(im_ja_xim_server->ic_table,
																				 im_ja_xim_server->ic_table_size);
	for (i = old_size; i < im_ja_xim_server->ic_table_size; i++) {
		im_ja_xim_server->ic_table[i] = NULL;
	}
}

void im_ja_xim_server_add_connect(IMJAXimConnect *connect) {
	IMJAXimConnect *list;

	IM_JA_DEBUG("im_ja_xim_server_add_connect\n");

	if (connect == NULL) return;

	list = im_ja_xim_server->connect_list;
	connect->next = list;
	im_ja_xim_server->connect_list = connect;
}

void im_ja_xim_server_remove_connect(IMJAXimConnect *connect) {
	IMJAXimConnect *prev;
	IMJAXimConnect *list;

	if (connect == NULL) return;

	prev = NULL;
	list = im_ja_xim_server->connect_list;
	while (list != NULL) {
		if (list->id == connect->id) {
	    if (prev == NULL)
				im_ja_xim_server->connect_list = list->next;
	    else prev->next = list->next;
	    list->next = NULL;
		}
		prev = list;
		list = list->next;
	}
}


IMJAXimConnect *im_ja_xim_server_get_connect_by_id(CARD16 connect_id) {
	IMJAXimConnect *connect;

	IM_JA_DEBUG("im_ja_xim_server_get_connect_by_id %d\n", (int) connect_id);

	connect = im_ja_xim_server->connect_list;
	while (connect != NULL) {
		if (connect->id == connect_id) return connect;
		connect = connect->next;
	}
	return NULL;
}

IMJAContext *im_ja_xim_server_get_ic(CARD16 icid) {
	IMJAContext *ic;

	IM_JA_DEBUG("im_ja_xim_server_get_ic [icid: %d]\n", icid);
	if (icid > 0 && icid < im_ja_xim_server->ic_table_size) ic = im_ja_xim_server->ic_table[icid];
	else ic = NULL;

	return ic;
}


void im_ja_xim_server_create() {

	IM_JA_DEBUG("im_ja_xim_server_create\n");
	im_ja_xim_server = g_new0(IMJAXimServer, 1);
	im_ja_xim_server->display = NULL;
	im_ja_xim_server->window = 0;
	im_ja_xim_server->xims = 0;
	im_ja_xim_server->filter_mask = KeyPressMask;
	im_ja_xim_server->trigger_keys = im_ja_xim_trigger_keys;

	/* init IC table */
	im_ja_xim_server->ic_table = (IMJAContext **)g_new0(gpointer, DEFAULT_IC_TABLE_SIZE);
	im_ja_xim_server->ic_table_size = DEFAULT_IC_TABLE_SIZE;
	im_ja_xim_server->ic_freed = NULL;

	im_ja_xim_server->dynamic_event_flow = True;

}

void im_ja_xim_server_reload_conf(GConfClient *client, const gchar *key, GConfValue *value) {
	if (strcmp(key, GCONF_NAMESPACE"other/applet_input_method") == 0) return;
	if (strcmp(key, GCONF_NAMESPACE"other/last_input_method") == 0) return;
	IM_JA_DEBUG("im_ja_xim_server_reload_conf: \"%s\" changed.\n", key);

	im_ja_load_conf(&cfg);
}

static void im_ja_check_locale(char *locale) {
	gchar *head, *tail;
	gboolean locale_ok = FALSE;

	head = g_strndup(locale, 5);
	if (strcmp(head, "ja_JP") == 0) locale_ok = TRUE;
	g_free(head);

	if (strlen(locale) > 4) {
		tail = g_strdup(locale + 5);
		if (strcmp(tail, ".UTF-8") == 0) locale_ok = TRUE;
		if (strcmp(tail, ".UTF8") == 0) locale_ok = TRUE;
		if (strcmp(tail, ".utf-8") == 0) locale_ok = TRUE;
		if (strcmp(tail, ".utf8") == 0) locale_ok = TRUE;
		g_free(tail);
	}
	if (locale_ok == FALSE) {
		im_ja_print_error_cmdline("WARNING: Your locale (%s) is non-UTF8, nor Japanese!\n", locale);
		im_ja_print_error_cmdline("You should run im-ja-xim-server from a Japanese (ja_JP) and/or UTF-8 locale, e.g.:\n");
		im_ja_print_error_cmdline("\t \"$ LC_ALL=ja_JP.UTF-8 im-ja-xim-server\"\n");
		im_ja_print_error_cmdline("Otherwise you might get X errors and invisible preedit.\n");
	}
}

gboolean im_ja_xim_server_start(Display *display, Window window) {
	gint i;
	XIMS xims;
	XIMStyles input_styles;
	XIMEncodings encodings;
	XIMTriggerKeys trigger_keys;

	XIMStyles input_styles_ret;
	XIMEncodings encodings_ret;
	int screen_num;

	char *server_locale = NULL;
	char *locale = NULL;

	server_locale = setlocale(LC_ALL, NULL);
	im_ja_check_locale(server_locale);
	locale = g_strdup_printf("%s,ja_JP", server_locale);
	IM_JA_DEBUG("Effective locales: LC_ALL=%s\n", locale);

	screen_num = DefaultScreen(display);
	im_ja_xim_server->preedit_fg = BlackPixel(display, screen_num); 
	im_ja_xim_server->preedit_bg = WhitePixel(display, screen_num); 


	input_styles.count_styles = sizeof(im_ja_xim_input_styles) / sizeof(XIMStyle) - 1;
	input_styles.supported_styles = im_ja_xim_input_styles;

	encodings.count_encodings = sizeof(im_ja_xim_encodings) / sizeof(XIMEncoding) - 1;
	encodings.supported_encodings = im_ja_xim_encodings;

	trigger_keys.count_keys = sizeof(im_ja_xim_trigger_keys) / sizeof(XIMTriggerKey) - 1;
	trigger_keys.keylist = im_ja_xim_server->trigger_keys;

	xims = IMOpenIM(display,
									IMModifiers, "Xi18n",
									IMServerWindow, window,
									IMServerName, "im-ja-xim-server",
									IMLocale, locale, 
									IMServerTransport, "X/",
									IMInputStyles, &input_styles,
									NULL);

	if (xims == NULL) {
		g_error("Can't open Input Method Service.\n");
	}


	if (im_ja_xim_server->dynamic_event_flow) {
		IMSetIMValues(xims,
									IMOnKeysList, &trigger_keys,
									NULL);
	}


 	IMSetIMValues(xims,
								IMEncodingList, &encodings,
								IMProtocolHandler, im_ja_xim_handler,
								IMFilterEventMask, im_ja_xim_filter_mask,
								NULL);

	IMGetIMValues(xims,
								IMInputStyles, &input_styles_ret,
								IMEncodingList, &encodings_ret,
								NULL);

	im_ja_xim_server->xims = xims;
	im_ja_xim_server->display = display;
	im_ja_xim_server->window = window;


	gconf_client = NULL;

	if (im_ja_init_conf_handler() == FALSE) {
		im_ja_print_error_cmdline(_("GConf initialization failed!"));
	}
	im_ja_get_gconf_client();
	cfg.im_ja_version = NULL;
	if (im_ja_load_conf(&cfg) == FALSE) {
		im_ja_print_error_cmdline(_("Couldn't load settings!"));
	}

	if (cfg.im_ja_version == NULL) {
		im_ja_print_error_cmdline(_("You have not yet configured im-ja.\nYou should run 'im-ja-conf'.\n"));
	}
	else if (strcmp(cfg.im_ja_version, VERSION) != 0) {
		im_ja_print_error_cmdline(_("You should update your im-ja settings. Please run 'im-ja-conf'.\n"));
	}
	gconf_client_add_dir(gconf_client, GCONF_NAMESPACE"other", GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_add_dir(gconf_client, GCONF_NAMESPACE"conv_engine", GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_add_dir(gconf_client, GCONF_NAMESPACE"preedit_style", GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_add_dir(gconf_client, GCONF_NAMESPACE"hotkeys", GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_add_dir(gconf_client, GCONF_NAMESPACE"status_window", GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	g_signal_connect(G_OBJECT(gconf_client), "value-changed", G_CALLBACK(im_ja_xim_server_reload_conf), NULL);

	IM_JA_DEBUG("im-ja-xim-server started\n");
	for (i = 0; i < sizeof(im_ja_xim_input_styles) / sizeof(XIMStyle); i++) {
		IM_JA_DEBUG(" set input_style: %d\n", (int) im_ja_xim_input_styles[i]);
	}
	/*
	IM_JA_DEBUG(" got input_style [count:%hu]: %d\n", input_styles_ret.count_styles,
							input_styles_ret.supported_styles);
	*/

	return TRUE;
}

static void on_realize(GtkWidget *widget, gpointer data) {
	im_ja_xim_server_start(GDK_WINDOW_XDISPLAY(widget->window),
												 GDK_WINDOW_XWINDOW(widget->window));
}


int main(int argc, char *argv[]) {
	GtkWidget *window;

#ifdef ENABLE_NLS
	bindtextdomain(GETTEXT_PACKAGE, IM_JA_LOCALE_DIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);
#endif

	gtk_init(&argc, &argv);

	im_ja_xim_server_create();

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_resize(GTK_WINDOW(window), 1, 1);
	gtk_window_set_decorated(GTK_WINDOW(window), FALSE);

	g_signal_connect_after(G_OBJECT(window), "realize",
												 G_CALLBACK(on_realize), NULL);
	gtk_widget_realize(window);
	gtk_widget_hide(window);

	XSetErrorHandler(im_ja_xim_server_x_error_handler);

	gtk_main();
		
	return 0;
}

