/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* IM-JA Japanese Input Method 
 *
 * Copyright (C) 2003 Botond Botyanszki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * Based on ic.c from nabi by Choe Hwanjin
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <glib.h>
#include <gdk/gdkx.h>

#include "../error.h"
#include "../im-ja.h"
#include "../im-ja-impl.h"

#include "xim-ic.h"
#include "xim-server.h"
#include "xim-fontset.h"

extern IMJAXimServer *im_ja_xim_server; 

static void im_ja_xim_ic_init_preedit(IMJAContext *ic);
/*static void debug_preedit(IMJAContext *ic);*/

#define streql(x, y) (strcmp((x), (y)) == 0)


IMJAXimConnect *im_ja_xim_connect_create(CARD16 id) {
	IMJAXimConnect* connect;

	IM_JA_DEBUG("im_ja_xim_connect_create\n");

	connect = g_new0(IMJAXimConnect, 1);
	connect->id = id;
	connect->ic_list = NULL;
	connect->next = NULL;
    
	return connect;
}

void im_ja_xim_connect_destroy(IMJAXimConnect* connect) {
	IMJAContext *ic;
	GSList *list;

	/* remove all input contexts */
	list = connect->ic_list;
	while (list != NULL) {
		ic = (IMJAContext*)(list->data);
		if (ic != NULL) im_ja_context_destroy(ic);
		list = list->next;
	}

	g_slist_free(connect->ic_list);
	g_free(connect);
}

void im_ja_xim_connect_add_ic(IMJAXimConnect* connect, IMJAContext *ic) {
	if (connect == NULL || ic == NULL) return;
	connect->ic_list = g_slist_prepend(connect->ic_list, ic);
}

void im_ja_xim_connect_remove_ic(IMJAXimConnect* connect, IMJAContext *ic) {
	if (connect == NULL || ic == NULL) return;
	connect->ic_list = g_slist_remove(connect->ic_list, ic);
}

void im_ja_xim_ic_init_values(IMJAContext *ic) {
	IM_JA_DEBUG("im_ja_xim_ic_init_values\n");

	ic->connect = im_ja_xim_server_get_connect_by_id(ic->connect_id);

	/* preedit attr */
	ic->preedit.width = 1;	/* minimum window size is 1 x 1 */
	ic->preedit.height = 1;	/* minimum window size is 1 x 1 */
	ic->preedit.foreground = im_ja_xim_server->preedit_fg;
	ic->preedit.background = im_ja_xim_server->preedit_bg;
	ic->preedit.state = XIMPreeditEnable;
	ic->preedit.start = False;

	/* set some sane values for the default colors */
	if ((ic->original_colors[0].red == ic->original_colors[1].red)
			&& (ic->original_colors[0].green == ic->original_colors[1].green)
			&& (ic->original_colors[0].blue == ic->original_colors[1].blue)) {
		IM_JA_DEBUG("**Warning** same fg and bg values\n");
		ic->original_colors[0].red = 0xffff;
		ic->original_colors[0].green = 0xffff;
		ic->original_colors[0].blue = 0xffff;
		ic->original_colors[1].red = 0;
		ic->original_colors[1].green = 0;
		ic->original_colors[1].blue = 0;
	}

	if (ic->prev_preedit_buf == NULL) {
		ic->prev_preedit_buf = g_new0(gchar, BUFFERSIZE);
	}
}


static gboolean im_ja_xim_ic_is_destroyed(IMJAContext *ic) {
	if (ic->id > 0 && ic->id < im_ja_xim_server->ic_table_size) {
		return im_ja_xim_server->ic_table[ic->id] == NULL;
	}
	else return TRUE;
}


void im_ja_context_impl_destroy(IMJAContext *ic) {
	IM_JA_DEBUG("im_ja_context_impl_destroy\n");

	if (im_ja_xim_ic_is_destroyed(ic)) return;

	/* we do not delete, just save it in ic_freed */
	if (im_ja_xim_server->ic_freed == NULL) {
		im_ja_xim_server->ic_freed = ic;
		im_ja_xim_server->ic_freed->next = NULL;
	} 
	else {
		ic->next = im_ja_xim_server->ic_freed;
		im_ja_xim_server->ic_freed = ic;
	}

	im_ja_xim_server->ic_table[ic->id] = NULL;

	ic->connect_id = 0;
	ic->client_window = 0;
	ic->focus_window = 0;
	g_free(ic->resource_name);
	ic->resource_name = NULL;
	g_free(ic->resource_class);
	ic->resource_class = NULL;

	ic->connect = NULL;

	ic->preedit.area.x = 0;
	ic->preedit.area.y = 0;
	ic->preedit.area.width = 0;
	ic->preedit.area.height = 0;
	ic->preedit.area_needed.x = 0;
	ic->preedit.area_needed.y = 0;
	ic->preedit.area_needed.width = 0;
	ic->preedit.area_needed.height = 0;
	ic->preedit.spot.x = 0;
	ic->preedit.spot.y = 0;
	ic->preedit.cmap = 0;

	ic->preedit.width = 1;
	ic->preedit.height = 1;
	ic->preedit.ascent = 0;
	ic->preedit.descent = 0;
	ic->preedit.line_space = 0;
	ic->preedit.cursor = 0;

	ic->preedit.state = XIMPreeditEnable;
	ic->preedit.start = False;
	
	/* destroy preedit window */
	if (ic->preedit.window != 0) {
		XDestroyWindow(im_ja_xim_server->display, ic->preedit.window);
		ic->preedit.window = 0;
		/* XFreeGC ?? */
	}

	/* destroy fontset data */
	if (ic->preedit.font_set) {
		im_ja_xim_fontset_free(im_ja_xim_server->display, ic->preedit.font_set);
		g_free(ic->preedit.base_font);
		ic->preedit.font_set = NULL;
		ic->preedit.base_font = NULL;
	}

	/* status attributes */
	ic->status_attr.area.x = 0;
	ic->status_attr.area.y = 0;
	ic->status_attr.area.width = 0;
	ic->status_attr.area.height = 0;

	ic->status_attr.area_needed.x = 0;
	ic->status_attr.area_needed.y = 0;
	ic->status_attr.area_needed.width = 0;
	ic->status_attr.area_needed.height = 0;

	ic->status_attr.cmap = 0;
	ic->status_attr.foreground = 0;
	ic->status_attr.background = 0;
	ic->status_attr.background = 0;
	ic->status_attr.bg_pixmap = 0;
	ic->status_attr.line_space = 0;
	ic->status_attr.cursor = 0;
	ic->status_attr.base_font = NULL;

}


/*
void im_ja_xim_ic_real_destroy(IMJAContext *ic) {
	if (ic == NULL) return;

	g_free(ic->resource_name);
	g_free(ic->resource_class);
	g_free(ic->preedit.base_font);
	g_free(ic->status_attr.base_font);

	if (ic->preedit.window != 0) {
		XDestroyWindow(im_ja_xim_server->display, ic->preedit.window);
	}
	if (ic->preedit.font_set) {
		im_ja_xim_fontset_free(im_ja_xim_server->display, ic->preedit.font_set);
	}

	if (ic->preedit.gc) XFreeGC(im_ja_xim_server->display, ic->preedit.gc);

	g_free(ic);
}
*/


void im_ja_xim_ic_preedit_show(IMJAContext *ic) {
	IM_JA_DEBUG("im_ja_xim_ic_preedit_show\n");
	im_ja_xim_ic_init_preedit(ic);
	if (ic->preedit.window == 0) {
		IM_JA_DEBUG(" error: no preedit.window!\n");
		return;
	}

	if (strlen(ic->preedit_buf) > 0) {
		XFlush(im_ja_xim_server->display);
		XMapRaised(im_ja_xim_server->display, ic->preedit.window);
		/*debug_preedit(ic);*/
	}
}

/*
static void im_ja_xim_ic_preedit_hide(IMJAContext *ic) {
	if (ic->preedit.window == 0) return;

	XUnmapWindow(im_ja_xim_server->display, ic->preedit.window);
}
*/

/* move and resize preedit window */
static void im_ja_xim_ic_preedit_configure(IMJAContext *ic) {
	IM_JA_DEBUG("im_ja_xim_ic_preedit_configure\n");

	im_ja_xim_ic_init_preedit(ic);
	if (ic->preedit.window == 0) {
		IM_JA_DEBUG(" no preedit window!\n");
		return;
	}

	IM_JA_DEBUG(" move&resize preedit\n");
	XMoveResizeWindow(im_ja_xim_server->display, ic->preedit.window,
										ic->preedit.spot.x - 1, /* -1 is for hiding the cursor of the window (hack ?) */
										ic->preedit.spot.y - ic->preedit.ascent, 
										ic->preedit.width,
										ic->preedit.height);
}

/*
static void debug_preedit(IMJAContext *ic) {
	XWindowAttributes attribs;

	if (ic->preedit.window == 0) {
		IM_JA_DEBUG(" NO PREEDIT WINDOW\n");
		return;
	}

	XGetWindowAttributes(im_ja_xim_server->display, ic->preedit.window, &attribs);

	IM_JA_DEBUG(" PREEDIT WINDOW:\n");
	IM_JA_DEBUG("  X: %d", attribs.x);
	IM_JA_DEBUG("  Y: %d", attribs.y);
	IM_JA_DEBUG("  WIDTH: %d", attribs.width);
	IM_JA_DEBUG("  HEIGHT: %d", attribs.height);
	IM_JA_DEBUG("  MAP STATE: ");
	switch (attribs.map_state) {
		case IsUnmapped: 
			IM_JA_DEBUG("Unmapped");
			break;
		case IsUnviewable: 
			IM_JA_DEBUG("Unviewable");
			break;
		case IsViewable: 
			IM_JA_DEBUG("Viewable");
			break;
	}


	IM_JA_DEBUG("\n PREEDIT AREA:\n");
	IM_JA_DEBUG("  X: %d", ic->preedit.area.x);
	IM_JA_DEBUG("  Y: %d", ic->preedit.area.y);
	IM_JA_DEBUG("  WIDTH: %d", ic->preedit.area.width);
	IM_JA_DEBUG("  HEIGHT: %d\n", ic->preedit.area.height);

	IM_JA_DEBUG(" SPOT:\n");
	IM_JA_DEBUG("  X: %d", ic->preedit.spot.x);
	IM_JA_DEBUG("  Y: %d\n", ic->preedit.spot.y);

	IM_JA_DEBUG(" PREEDIT :\n");
	IM_JA_DEBUG("  WIDTH: %d", ic->preedit.width);
	IM_JA_DEBUG("  HEIGHT: %d\n", ic->preedit.height);
	IM_JA_DEBUG("  ASCENT: %d", ic->preedit.ascent);
	IM_JA_DEBUG("  DESCENT: %d\n", ic->preedit.descent);

}
*/

/*
static void im_ja_preedit_area_update_attribs(IMJAContext *ic) {
	XWindowAttributes attribs;

	if (ic->preedit.window == 0) return;
 
	XGetWindowAttributes(im_ja_xim_server->display, ic->preedit.window, &attribs);
	ic->preedit.area.x = attribs.x;
	ic->preedit.area.y = attribs.y;
	ic->preedit.area.width = attribs.width;
	ic->preedit.area.height = attribs.height;

}
*/

void im_ja_xim_ic_preedit_draw(IMJAContext *ic) {
	int width;
	int cursor_pos;

	/*
		FIXME: GTKify this if possible:
		  - use preeditarea
		  - implement im_ja_xim_get_preedit_string()  and use that
			  instead of directly accessing preedit_buf 
	*/

	/* FIXME2: this cannot draw multi line preedit
	 */

	IM_JA_DEBUG("im_ja_xim_ic_preedit_draw\n");
	if (strlen(ic->preedit_buf) == 0) return;

	if (ic->preedit.window == 0) {
		/* im_ja_xim_ic_preedit_window_new(ic); */
	}
	
	if (ic->preedit.font_set == 0) {
		IM_JA_DEBUG(" NO PREEDIT FONTSET\n");
		return;
	}

	if (ic->preedit.gc == 0) {
		IM_JA_DEBUG(" NO PREEDIT GC\n");
		return;
	}

	width = Xutf8TextEscapement(ic->preedit.font_set, ic->preedit_buf, strlen(ic->preedit_buf));
	cursor_pos = Xutf8TextEscapement(ic->preedit.font_set, ic->preedit_buf, im_ja_get_cursor_pos_bytes(ic));

	if (ic->preedit.width != width) {
		ic->preedit.width = width;
		ic->preedit.height = ic->preedit.ascent + ic->preedit.descent;
		IM_JA_DEBUG(" resize preedit to %d x %d\n", ic->preedit.width, ic->preedit.height);
		XResizeWindow(im_ja_xim_server->display, ic->preedit.window,	ic->preedit.width, ic->preedit.height);
	}

	/* if preedit window is out of focus window 
	 	 we force to put it in focus window (preedit.area) */
	if (ic->preedit.spot.x + ic->preedit.width > ic->preedit.area.width) {
		ic->preedit.spot.x = ic->preedit.area.width - ic->preedit.width;
		XMoveWindow(im_ja_xim_server->display, ic->preedit.window,
								ic->preedit.spot.x, ic->preedit.spot.y - ic->preedit.ascent);
	}

	/* debug_preedit(ic); */

	Xutf8DrawString(im_ja_xim_server->display,
									ic->preedit.window,
									ic->preedit.font_set,
									ic->preedit.gc,
									0,
									ic->preedit.ascent,
									ic->preedit_buf,
									strlen(ic->preedit_buf));

	if (ic->preedit_reverse_start != ic->preedit_reverse_end) {
		int reverse_start_pos;
		int reverse_end_pos;
		int reverse_length;

		reverse_length = ic->preedit_reverse_end - ic->preedit_reverse_start;
		reverse_start_pos = Xutf8TextEscapement(ic->preedit.font_set, ic->preedit_buf, ic->preedit_reverse_start);
		reverse_end_pos = Xutf8TextEscapement(ic->preedit.font_set, ic->preedit_buf, ic->preedit_reverse_end);

		/*
		IM_JA_DEBUG("REVERSE LENGTH: %d\n", reverse_length);
		IM_JA_DEBUG("REVERSE START POS: %d\n", reverse_start_pos);
		IM_JA_DEBUG("REVERSE STRING: %s\n", ic->preedit_buf + ic->preedit_reverse_start);
		*/

		XFillRectangle(im_ja_xim_server->display,
									 ic->preedit.window,
									 ic->preedit.gc,
									 reverse_start_pos,
									 0,
									 reverse_end_pos - reverse_start_pos,
									 ic->preedit.ascent + ic->preedit.descent);

		Xutf8DrawString(im_ja_xim_server->display,
										ic->preedit.window,
										ic->preedit.font_set,
										ic->preedit.reverse_gc,
										reverse_start_pos,
										ic->preedit.ascent,
										ic->preedit_buf + ic->preedit_reverse_start,
										reverse_length);
	}

	if (cursor_pos >= ic->preedit.width) cursor_pos--;

	XFillRectangle(im_ja_xim_server->display,
								 ic->preedit.window,
								 ic->preedit.gc,
								 cursor_pos,
								 0,
								 1,
								 ic->preedit.ascent + ic->preedit.descent);
	
	XDrawLine(im_ja_xim_server->display,
						ic->preedit.window,
						ic->preedit.gc,
						0,
						ic->preedit.ascent + ic->preedit.descent - 1,
						ic->preedit.width - 1,
						ic->preedit.ascent + ic->preedit.descent - 1);

	XFlush(im_ja_xim_server->display);

}

static GdkFilterReturn im_ja_xim_ic_preedit_event_filter(GdkXEvent *xevent, GdkEvent *gevent, gpointer data) {
	IMJAContext *ic = (IMJAContext *)data;
	XEvent *event = (XEvent*)xevent;

	IM_JA_DEBUG("im_ja_xim_preedit_event_filter\n");

	if (ic == NULL) return GDK_FILTER_REMOVE;
	if (ic->preedit.window == 0) return GDK_FILTER_REMOVE;

	if (event->xany.window != ic->preedit.window)	return GDK_FILTER_CONTINUE;

	switch (event->type) {
	case DestroyNotify:
		IM_JA_DEBUG("DESTROYNOTIFY\n");
		if (!im_ja_xim_ic_is_destroyed(ic)) {
	    /* preedit window is destroyed, so we set it 0 */
	    ic->preedit.window = 0;
			im_ja_context_destroy(ic);
		}
		return GDK_FILTER_REMOVE;
		break;
	case Expose:
		IM_JA_DEBUG("Redraw Window\n");
		im_ja_xim_ic_preedit_draw(ic);
		break;
	default:
		IM_JA_DEBUG("event type: %d\n", event->type);
		break;
	}

	return GDK_FILTER_CONTINUE;
}


static void im_ja_xim_ic_preedit_window_new(IMJAContext *ic) {
	GdkWindow *gdk_window;
	Window parent = 0;
	XSetWindowAttributes attr;

	IM_JA_DEBUG("im_ja_xim_ic_preedit_window_new()\n");

	if (ic->focus_window != 0) {
		parent = ic->focus_window;
		IM_JA_DEBUG(" using focus_window as parent\n");
	}
	else if (ic->client_window != 0) {
		parent = ic->client_window;
		IM_JA_DEBUG(" using client_window as parent\n");
	}
	/*
	else {
		parent = DefaultRootWindow(im_ja_xim_server->display);
		IM_JA_DEBUG(" using root_window as parent\n");
	}
	*/

	if (parent == 0) {
		IM_JA_DEBUG(" ERROR: no parent!\n");
		return;
	}
 
	IM_JA_DEBUG(" creating %d x %d\n", ic->preedit.width, ic->preedit.height);

	ic->preedit.window = XCreateSimpleWindow(im_ja_xim_server->display,
																					 parent,
																					 ic->preedit.spot.x,
																					 ic->preedit.spot.y - ic->preedit.ascent,
																					 ic->preedit.width,
																					 ic->preedit.height,
																					 0,
																					 im_ja_xim_server->preedit_fg, /* FIXME */
																					 im_ja_xim_server->preedit_bg); /* FIXME */

	attr.override_redirect = True;
	XChangeWindowAttributes(im_ja_xim_server->display, ic->preedit.window, CWOverrideRedirect,
													&attr);

	XSelectInput(im_ja_xim_server->display, ic->preedit.window,
							 ExposureMask | StructureNotifyMask);

	/* install our preedit window event filter */
	gdk_window = gdk_window_foreign_new(ic->preedit.window);
	if (gdk_window != NULL) {
		gdk_window_add_filter(gdk_window, im_ja_xim_ic_preedit_event_filter, (gpointer)ic);
		g_object_unref(gdk_window);
	}
	IM_JA_DEBUG(" done.\n");
}

static void im_ja_xim_preedit_create_gc(IMJAContext *ic) {
	if (ic->preedit.gc == 0) {
		XGCValues values;

		values.foreground = im_ja_xim_server->preedit_fg;
		values.background = im_ja_xim_server->preedit_bg;
		
		ic->preedit.gc = XCreateGC(im_ja_xim_server->display,
															 im_ja_xim_server->window,
															 GCForeground | GCBackground,
															 &values);
	}

	if (ic->preedit.reverse_gc == 0) {
		XGCValues rev_values;

		rev_values.background = im_ja_xim_server->preedit_fg;
		rev_values.foreground = im_ja_xim_server->preedit_bg;
		
		/*
		//rev_values.background = im_ja_xim_server->preedit_bg;
		//rev_values.foreground = im_ja_xim_server->preedit_fg;
		*/
		ic->preedit.reverse_gc = XCreateGC(im_ja_xim_server->display,
																			 im_ja_xim_server->window,
																			 GCForeground | GCBackground,
																			 &rev_values);
	}

}

static void im_ja_xim_ic_init_preedit(IMJAContext *ic) {
	IM_JA_DEBUG("im_ja_xim_ic_init_preedit()\n");
	if (ic->input_style & XIMPreeditPosition) {
		if ((ic->preedit.gc == 0) || (ic->preedit.reverse_gc == 0)) {
			im_ja_xim_preedit_create_gc(ic);
		}

		/* For Preedit Position aka Over the Spot */
		if (ic->preedit.window == 0) {
			im_ja_xim_ic_preedit_window_new(ic);
		}
	}
}

static void im_ja_xim_ic_set_focus_window(IMJAContext *ic, Window focus_window) {
	XWindowAttributes attribs;
	
	/* FIXME: focus window is client window (in gtkimcontext)?? */
	IM_JA_DEBUG("im_ja_xim_ic_set_focus_window\n");

	
	XGetWindowAttributes(im_ja_xim_server->display, focus_window, &attribs);

	IM_JA_DEBUG(" FOCUS WINDOW: ");
	IM_JA_DEBUG("  X: %d", attribs.x);
	IM_JA_DEBUG("  Y: %d", attribs.y);
	IM_JA_DEBUG("  WIDTH: %d", attribs.width);
	IM_JA_DEBUG("  HEIGHT: %d\n", attribs.height);

	ic->focus_window = focus_window;
	im_ja_xim_ic_init_preedit(ic);
}

static void im_ja_xim_ic_set_preedit_foreground(IMJAContext *ic, unsigned long foreground) { /* FIXME: use conf */
	IM_JA_DEBUG("im_ja_xim_ic_set_preedit_foreground: %d\n", (int) foreground);

	/* foreground = im_ja_xim_server->preedit_fg; */
	im_ja_xim_server->preedit_fg = foreground;
	ic->preedit.foreground = foreground;
	
	if (ic->focus_window == 0) return;

	if ((ic->preedit.gc == 0) || (ic->preedit.reverse_gc == 0)) {
		im_ja_xim_preedit_create_gc(ic);
	}

	XSetForeground(im_ja_xim_server->display, ic->preedit.gc,	ic->preedit.foreground);

	/* XSetForeground(im_ja_xim_server->display, ic->preedit.reverse_gc, ic->preedit.foreground); */
	XSetBackground(im_ja_xim_server->display, ic->preedit.reverse_gc,	ic->preedit.foreground);

}


static void im_ja_xim_ic_set_preedit_background(IMJAContext *ic, unsigned long background) { /* FIXME: use conf */
	IM_JA_DEBUG("im_ja_xim_ic_set_preedit_background: %d\n", (int) background);

	/* background = im_ja_xim_server->preedit_bg; */
	im_ja_xim_server->preedit_bg = background;
	ic->preedit.background = background;

	if (ic->focus_window == 0) return;

	if ((ic->preedit.gc == 0) || (ic->preedit.reverse_gc == 0)) {
		im_ja_xim_preedit_create_gc(ic);
	}

	XSetBackground(im_ja_xim_server->display, ic->preedit.gc, ic->preedit.background);

	/* XSetBackground(im_ja_xim_server->display, ic->preedit.reverse_gc, ic->preedit.background); */
	XSetForeground(im_ja_xim_server->display, ic->preedit.reverse_gc, ic->preedit.background);
	
}


static void im_ja_xim_ic_load_preedit_fontset(IMJAContext *ic, char *font_name) {
	IMJAXimFontSet *fontset;

	IM_JA_DEBUG("im_ja_xim_ic_load_preedit_fontset\n");

	if (ic->preedit.base_font != NULL && strcmp(ic->preedit.base_font, font_name) == 0) {
		/* same font, do not create fontset */
		return;
	}

	if (ic->preedit.base_font != NULL) g_free(ic->preedit.base_font);
	ic->preedit.base_font = g_strdup(font_name);
	if (ic->preedit.font_set)	im_ja_xim_fontset_free(im_ja_xim_server->display, ic->preedit.font_set);

	fontset = im_ja_xim_fontset_create(im_ja_xim_server->display, font_name);
	if (fontset == NULL) {
		IM_JA_DEBUG("WARNING: FONTSET == NULL\n");
		return;
	}

	ic->preedit.font_set = fontset->xfontset;
	ic->preedit.ascent = fontset->ascent;
	ic->preedit.descent = fontset->descent;
	ic->preedit.height = ic->preedit.ascent + ic->preedit.descent;
	ic->preedit.width = 1;
}


static void im_ja_xim_ic_set_spot(IMJAContext *ic, XPoint *point) {

	IM_JA_DEBUG("im_ja_xim_ic_set_spot\n");

	if (point == NULL) return;

	ic->preedit.spot.x = point->x + 1; 
	ic->preedit.spot.y = point->y; 

	IM_JA_DEBUG(" X: %d, Y: %d\n", ic->preedit.spot.x, ic->preedit.spot.y);

	im_ja_cursor_location_changed(ic, ic->preedit.spot.x, ic->preedit.spot.y + 3);


	/* if preedit window is out of focus window 
	 * we force it in focus window (preedit.area) */
	if (ic->preedit.spot.x + ic->preedit.width > ic->preedit.area.width) {
		ic->preedit.spot.x = ic->preedit.area.width - ic->preedit.width;
	}

	im_ja_xim_ic_preedit_configure(ic);

	/* FIXME
	if (im_ja_xim_ic_is_empty(ic)) im_ja_xim_ic_preedit_hide(ic);
	else
	*/
	im_ja_xim_ic_preedit_show(ic);

}
/*
    XIMPreeditCallbacks | XIMStatusCallbacks,
    XIMPreeditCallbacks | XIMStatusArea,
    XIMPreeditCallbacks | XIMStatusNothing,

		XIMPreeditPosition  | XIMStatusCallbacks,
		XIMPreeditPosition  | XIMStatusArea,
		XIMPreeditPosition  | XIMStatusNothing,

    XIMPreeditArea      | XIMStatusCallbacks,
    XIMPreeditArea      | XIMStatusArea,
    XIMPreeditArea      | XIMStatusNothing,

    XIMPreeditNothing   | XIMStatusNothing,
*/

static void im_ja_xim_ic_set_client_window(IMJAContext *ic, Window win) {
	XWindowAttributes attribs;
	GdkWindow *gdk_window;

	IM_JA_DEBUG("im_ja_xim_ic_set_client_window\n");

	ic->client_window = win;

	if (ic->client_window == 0) {
		IM_JA_DEBUG(" NO CLIENT WINDOW\n");
		return;
	}

	XGetWindowAttributes(im_ja_xim_server->display, ic->client_window, &attribs);

	IM_JA_DEBUG(" CLIENT WINDOW: ");
	IM_JA_DEBUG("  X: %d", attribs.x);
	IM_JA_DEBUG("  Y: %d", attribs.y);
	IM_JA_DEBUG("  WIDTH: %d", attribs.width);
	IM_JA_DEBUG("  HEIGHT: %d\n", attribs.height);

	gdk_window = gdk_window_foreign_new(ic->client_window);
	if (gdk_window != NULL) {
		gdk_window_add_filter(gdk_window, im_ja_xim_ic_preedit_event_filter, (gpointer)ic);
		g_object_unref(gdk_window);
	}

}


static void print_xim_styles(XIMStyle style) {

	IM_JA_DEBUG(" Supported styles:\n");
	if (style & XIMPreeditCallbacks) IM_JA_DEBUG("  XIMPreeditCallbacks\n");
	if (style & XIMPreeditPosition) IM_JA_DEBUG("  XIMPreeditPosition\n");
	if (style & XIMPreeditArea) IM_JA_DEBUG("  XIMPreeditArea\n");
	if (style & XIMPreeditNothing) IM_JA_DEBUG("  XIMPreeditNothing\n");

	if (style & XIMStatusCallbacks) IM_JA_DEBUG("  XIMStatusCallbacks\n");
	if (style & XIMStatusArea) IM_JA_DEBUG("  XIMStatusArea\n");
	if (style & XIMStatusNothing) IM_JA_DEBUG("  XIMStatusNothing\n");
}


void im_ja_xim_ic_set_values(IMJAContext *ic, IMChangeICStruct *data) {

	XICAttribute *ic_attr = data->ic_attr;
	XICAttribute *preedit_attr = data->preedit_attr;
	XICAttribute *status_attr = data->status_attr;
	CARD16 i;
	
	IM_JA_DEBUG("im_ja_xim_ic_set_values\n");

	if (ic == NULL) return;

	for (i = 0; i < data->ic_attr_num; i++, ic_attr++) {
		IM_JA_DEBUG(" ic attribute: %s\n", ic_attr->name);
		if (streql(XNInputStyle, ic_attr->name)) {
	    ic->input_style = *(INT32*)ic_attr->value;
			print_xim_styles(ic->input_style);
			if (ic->input_style & XIMPreeditNothing) im_ja_set_use_preedit(ic, TRUE);
			im_ja_xim_ic_init_preedit(ic);

		}
		else if (streql(XNClientWindow, ic_attr->name)) {
			im_ja_xim_ic_set_client_window(ic, *(Window*)ic_attr->value);
		}
		else if (streql(XNFocusWindow, ic_attr->name)) {
	    im_ja_xim_ic_set_focus_window(ic, *(Window*)ic_attr->value);
		}
		else {
	    g_error("im_ja_xim: set unknown ic attribute: %s\n", ic_attr->name);
		}
	}
	
	for (i = 0; i < data->preedit_attr_num; i++, preedit_attr++) {
		IM_JA_DEBUG(" preedit attribute: %s\n", preedit_attr->name);
		if (streql(XNSpotLocation, preedit_attr->name)) {
	    im_ja_xim_ic_set_spot(ic, (XPoint*)preedit_attr->value);
		}
		else if (streql(XNForeground, preedit_attr->name)) {
	    im_ja_xim_ic_set_preedit_foreground(ic, *(unsigned long*)preedit_attr->value);
		}
		else if (streql(XNBackground, preedit_attr->name)) {
	    im_ja_xim_ic_set_preedit_background(ic,	*(unsigned long*)preedit_attr->value);
		}
		else if (streql(XNArea, preedit_attr->name)) {
	    ic->preedit.area = *(XRectangle*)preedit_attr->value;
		}
		else if (streql(XNLineSpace, preedit_attr->name)) {
	    ic->preedit.line_space = *(CARD32*)preedit_attr->value;
		}
		else if (streql(XNPreeditState, preedit_attr->name)) {
	    ic->preedit.state = *(XIMPreeditState*)preedit_attr->value;
		}
		else if (streql(XNFontSet, preedit_attr->name)) {
	    im_ja_xim_ic_load_preedit_fontset(ic, (char*)preedit_attr->value);
		}
		else {
	    g_print("Im_Ja_Xim: set unknown preedit attribute: %s\n",
							preedit_attr->name);
		}
	}
	
	for (i = 0; i < data->status_attr_num; i++, status_attr++) {
		IM_JA_DEBUG(" status attribute: %s\n", status_attr->name);
		if (streql(XNArea, status_attr->name)) {
			ic->status_attr.area = *(XRectangle*)status_attr->value;
		}
		else if (streql(XNAreaNeeded, status_attr->name)) {
			ic->status_attr.area_needed = *(XRectangle*)status_attr->value;
		}
		else if (streql(XNForeground, status_attr->name)) {
			ic->status_attr.foreground = *(unsigned long*)status_attr->value;
		}
		else if (streql(XNBackground, status_attr->name)) {
			ic->status_attr.background = *(unsigned long*)status_attr->value;
		}
		else if (streql(XNLineSpace, status_attr->name)) {
			ic->status_attr.line_space = *(CARD32*)status_attr->value;
		}
		else if (streql(XNFontSet, status_attr->name)) {
			g_free(ic->status_attr.base_font);
			ic->status_attr.base_font = g_strdup((char*)status_attr->value);
		}
		else {
			g_print("Im_Ja_Xim: set unknown status attributes: %s\n",	status_attr->name);
		}
	}
}


void im_ja_xim_ic_get_values(IMJAContext *ic, IMChangeICStruct *data) {
	XICAttribute *ic_attr = data->ic_attr;
	XICAttribute *preedit_attr = data->preedit_attr;
	XICAttribute *status_attr = data->status_attr;
	CARD16 i;

	IM_JA_DEBUG("im_ja_xim_ic_get_values\n");

	if (ic == NULL) return;

	
	for (i = 0; i < data->ic_attr_num; i++, ic_attr++) {
		IM_JA_DEBUG(" ic attribute: %s\n", ic_attr->name);
		if (streql(XNFilterEvents, ic_attr->name)) {
			ic_attr->value = (void *)malloc(sizeof(CARD32));
			*(CARD32*)ic_attr->value = KeyPressMask | KeyReleaseMask;
			ic_attr->value_length = sizeof(CARD32);
		}
		else if (streql(XNInputStyle, ic_attr->name)) {
			ic_attr->value = (void *)malloc(sizeof(INT32));
			*(INT32*)ic_attr->value	= ic->input_style;
			ic_attr->value_length = sizeof(INT32);
		}
		else if (streql(XNSeparatorofNestedList, ic_attr->name)) {
			/* FIXME: what do I do here? */
				;
		}
		else {
			g_error("im_ja_xim: get unknown ic attribute: %s\n", ic_attr->name);
		}
	}
    
	for (i = 0; i < data->preedit_attr_num; i++, preedit_attr++) {
		IM_JA_DEBUG(" preedit attribute: %s\n", preedit_attr->name);
		if (streql(XNArea, preedit_attr->name)) {
			preedit_attr->value = (void *)malloc(sizeof(XRectangle));
			*(XRectangle*)preedit_attr->value = ic->preedit.area;
			preedit_attr->value_length = sizeof(XRectangle);
		}
		else if (streql(XNAreaNeeded, preedit_attr->name)) {
			preedit_attr->value = (void *)malloc(sizeof(XRectangle));
			*(XRectangle*)preedit_attr->value = ic->preedit.area_needed;
			preedit_attr->value_length = sizeof(XRectangle);
		}
		else if (streql(XNSpotLocation, preedit_attr->name)) {
			preedit_attr->value = (void *)malloc(sizeof(XPoint));
			*(XPoint*)preedit_attr->value = ic->preedit.spot;
			preedit_attr->value_length = sizeof(XPoint);
		}
		else if (streql(XNForeground, preedit_attr->name)) {
			preedit_attr->value = (void *)malloc(sizeof(long));
			*(long*)preedit_attr->value	= ic->preedit.foreground;
			preedit_attr->value_length = sizeof(long);
		}
		else if (streql(XNBackground, preedit_attr->name)) {
			preedit_attr->value = (void *)malloc(sizeof(long));
			*(long*)preedit_attr->value	= ic->preedit.background;
			preedit_attr->value_length = sizeof(long);
		}
		else if (streql(XNLineSpace, preedit_attr->name)) {
			preedit_attr->value = (void *)malloc(sizeof(long));
			*(long*)preedit_attr->value	= ic->preedit.line_space;
			preedit_attr->value_length = sizeof(long);
		}
		else if (streql(XNPreeditState, preedit_attr->name)) {
			preedit_attr->value = (void *)malloc(sizeof(XIMPreeditState));
			*(XIMPreeditState*)preedit_attr->value = ic->preedit.state;
			preedit_attr->value_length = sizeof(XIMPreeditState);
		}
		else if (streql(XNFontSet, preedit_attr->name)) {
			CARD16 base_len = (CARD16)strlen(ic->preedit.base_font);
			int total_len = sizeof(CARD16) + (CARD16)base_len;
			char *p;
			
			preedit_attr->value = (void *)malloc(total_len);
			p = (char *)preedit_attr->value;
			memmove(p, &base_len, sizeof(CARD16));
			p += sizeof(CARD16);
			strncpy(p, ic->preedit.base_font, base_len);
			preedit_attr->value_length = total_len;
		}
		else {
			g_print("im_ja_xim: get unknown preedit attribute: %s\n", preedit_attr->name);
		}
	}
	
	for (i = 0; i < data->status_attr_num; i++, status_attr++) {
		IM_JA_DEBUG(" status attribute: %s\n", status_attr->name);
		if (streql(XNArea, status_attr->name)) {
			status_attr->value = (void *)malloc(sizeof(XRectangle));
			*(XRectangle*)status_attr->value = ic->status_attr.area;
			status_attr->value_length = sizeof(XRectangle);
		}
		else if (streql(XNAreaNeeded, status_attr->name)) {
			status_attr->value = (void *)malloc(sizeof(XRectangle));
			*(XRectangle*)status_attr->value = ic->status_attr.area_needed;
			status_attr->value_length = sizeof(XRectangle);
		}
		else if (streql(XNForeground, status_attr->name)) {
			status_attr->value = (void *)malloc(sizeof(long));
			*(long*)status_attr->value = ic->status_attr.foreground;
			status_attr->value_length = sizeof(long);
		}
		else if (streql(XNBackground, status_attr->name)) {
			status_attr->value = (void *)malloc(sizeof(long));
			*(long*)status_attr->value = ic->status_attr.background;
			status_attr->value_length = sizeof(long);
		}
		else if (streql(XNLineSpace, status_attr->name)) {
			status_attr->value = (void *)malloc(sizeof(long));
			*(long*)status_attr->value = ic->status_attr.line_space;
			status_attr->value_length = sizeof(long);
		}
		else if (streql(XNFontSet, status_attr->name)) {
			CARD16 base_len = (CARD16)strlen(ic->status_attr.base_font);
			int total_len = sizeof(CARD16) + (CARD16)base_len;
			char *p;

			status_attr->value = (void *)malloc(total_len);
			p = (char *)status_attr->value;
			memmove(p, &base_len, sizeof(CARD16));
			p += sizeof(CARD16);
			strncpy(p, ic->status_attr.base_font, base_len);
			status_attr->value_length = total_len;
		} 
		else {
			g_print("im_ja_xim: get unknown status attribute: %s\n", status_attr->name);
		}
	}
	
}


/*
void im_ja_xim_ic_reset(IMJAContext *ic) {
	if (!im_ja_xim_ic_is_empty(ic))	im_ja_xim_ic_commit(ic);
}
*/

void im_ja_xim_ic_preedit_clear(IMJAContext *ic) {
	XIMText text;
	XIMFeedback feedback[4] = { XIMUnderline, 0, 0, 0 };
	IMPreeditCBStruct data;
	XTextProperty tp;
	char *list[2];
	IMCommitStruct commit_data;
	int ret;

	IM_JA_DEBUG("im_ja_xim_ic_preedit_clear()\n");

	if (ic->input_style & XIMPreeditCallbacks) {
		data.major_code = XIM_PREEDIT_DRAW;
		data.minor_code = 0;
		data.connect_id = ic->connect_id;
		data.icid = ic->id;
		data.todo.draw.caret = 0;
		data.todo.draw.chg_first = 0;
		data.todo.draw.chg_length = g_utf8_strlen(ic->prev_preedit_buf, -1);
		data.todo.draw.text = &text;

		text.feedback = feedback;
		text.encoding_is_wchar = False;
		text.string.multi_byte = ""; /*NULL;*/
		text.length = 0;
		IMCallCallback(im_ja_xim_server->xims, (XPointer)&data);

		/* this commit is needed for kde/qt for some reason */

		list[0] = "";
		list[1] = 0;
		ret = Xutf8TextListToTextProperty(im_ja_xim_server->display, list, 1,
																			XCompoundTextStyle,
																			&tp);
		commit_data.connect_id = ic->connect_id;
		commit_data.icid = ic->id;
		commit_data.flag = XimLookupChars;
		commit_data.commit_string = (char*) tp.value;
	
		IMCommitString(im_ja_xim_server->xims, (XPointer)&commit_data);
		XFree(tp.value);
	}
	else if (ic->input_style & XIMPreeditPosition) {
		if (ic->preedit.window == 0) return;
		XUnmapWindow(im_ja_xim_server->display, ic->preedit.window);
	}
	memset(ic->prev_preedit_buf, 0, BUFFERSIZE);

}

void im_ja_xim_ic_preedit_start(IMJAContext *ic) {

	IM_JA_DEBUG("im_ja_xim_ic_preedit_start\n");

	if (ic->preedit.start) return;

	if (im_ja_xim_server->dynamic_event_flow) {
		IMPreeditStateStruct preedit_state;

		preedit_state.connect_id = ic->connect_id;
		preedit_state.icid = ic->id;
		IMPreeditStart(im_ja_xim_server->xims, (XPointer)&preedit_state);
	}
	
	if (ic->input_style & XIMPreeditCallbacks) {
		IMPreeditCBStruct preedit_data;

		preedit_data.major_code = XIM_PREEDIT_START;
		preedit_data.minor_code = 0;
		preedit_data.connect_id = ic->connect_id;
		preedit_data.icid = ic->id;
		preedit_data.todo.return_value = 0;
		IMCallCallback(im_ja_xim_server->xims, (XPointer)&preedit_data);
	}
	else if (ic->input_style & XIMPreeditPosition) {
		ic->preedit.start = True;
		im_ja_xim_ic_preedit_clear(ic);

		im_ja_xim_ic_set_preedit_foreground(ic, im_ja_xim_server->preedit_fg);
		im_ja_xim_ic_set_preedit_background(ic, im_ja_xim_server->preedit_bg);
	}

}


/*
void im_ja_xim_ic_preedit_done(IMJAContext *ic) {
	if (!ic->preedit.start) return;
	
	if (ic->input_style & XIMPreeditCallbacks) {
		IMPreeditCBStruct preedit_data;

		preedit_data.major_code = XIM_PREEDIT_DONE;
		preedit_data.minor_code = 0;
		preedit_data.connect_id = ic->connect_id;
		preedit_data.icid = ic->id;
		preedit_data.todo.return_value = 0;
		IMCallCallback(im_ja_xim_server->xims, (XPointer)&preedit_data);
	}
	else if (ic->input_style & XIMPreeditPosition) {
		; // do nothing
	}

	if (im_ja_xim_server->dynamic_event_flow) {
		IMPreeditStateStruct preedit_state;

		preedit_state.connect_id = ic->connect_id;
		preedit_state.icid = ic->id;
		IMPreeditEnd(im_ja_xim_server->xims, (XPointer)&preedit_state);
	}

	ic->preedit.start = False;
}
*/

